#ifndef THERM_H
#define THERM_H

#include <cstdio>
#include <cmath>
#include <QWidget>

#include "Util.h"

// Useful thermodynamic formulas
//
// Documents:
// - Wikipedia
// - Olivier Bouvignies,"L'Émagramme 761 des météorologues", 
//                       Bulletin de l'union des physiciens, vol. 815, no 2, juin 1999
// - skew-t.pdf   Written by Bret D. Whissel, Tallahassee, Florida, November 2011.
//               http://bretwhissel.net/skewt/

#define GRIB_NOTDEF -99999

//-----------------------------------------------------------
class TPoint
{
	public:
		TPoint (double tempC=GRIB_NOTDEF, double hpa=GRIB_NOTDEF)
							{this->tempC=tempC; this->hpa=hpa;}
		bool   ok ()  {return hpa!=GRIB_NOTDEF && tempC!=GRIB_NOTDEF;}
		double hpa;      // altitude
		double tempC;    // °C	
};
//-----------------------------------------------------------
class TPCurve
{
	public:
		QList <TPoint> points;
		
		void clear () {points.clear();}
		void addPoint (const TPoint &pt) {points << pt;}
		void addPoint (double T, double P) {points << TPoint(T,P);}
};

//-----------------------------------------------------------
class Therm
{
	public:
		
		static double hpa2m (double hpa);	// Conversion altitude/pressure
		static double m2hpa (double z);		
		
		static double latentHeatWater (double tempC);
		static double mixingRatio (double tempC, double hpa);
		static double tempFromMixingRatio (double mixr, double hpa);

		static double vaporPressure (double tempC);
		static double tempFromVaporPressure (double hpa);

		static double dryAdiabaticTemperature (double hpa0, double t0, double hpa);
		static double dryAdiabaticPressure (double hpa0, double t0, double tempC);
		
		static double gammaSaturatedAdiabatic (double tempC, double hpa);
		static double saturated_dT_dP (double tempC, double hpa);
		static double saturatedAdiabaticTemperature (double tempC0, double hpa0, double hpa);
		
		static void curveSaturatedAdiabatic (TPCurve *curve, TPoint &start, double hpaLimit, double step);
		static void curveSaturatedAdiabatic (TPCurve *curve, double tempC0, double hpa0, double hpaLimit, double step);
};

//-----------------------------------------------------------
class SoundingPoint
{
	public :
		SoundingPoint (double hpa=1000, double tempC=0, double dewpC=0)
						{this->hpa=hpa; this->tempC=tempC; this->dewpC=dewpC;}
		double hpa;             // altitude
		double tempC, dewpC;    // °C
		bool operator< (const SoundingPoint &other) const
									{return this->hpa < other.hpa;}
};

//-----------------------------------------------------------
class SoundingPointWind
{
	public :
		SoundingPointWind (double hpa=1000, double vx=0, double vy=0)
						{this->hpa=hpa; this->vx=vx; this->vy=vy;}
		double hpa;             // altitude
		double vx, vy;			// wind m/s
};

//-----------------------------------------------------------
class Sounding
{
	public :
		enum ConvBase {
			LV1000, LV975, LV950, LV925, LV900, LV875, LV850, LV825, LV800,
			AVG25, AVG50, AVG75, AVG100, AVG125, AVG150, AVG175, AVG200
		};
		
		Sounding ();
		
		void addSoundingPointC (double hpa, double tempC, double dewpC);
		void addSoundingPointK (double hpa, double tempK, double dewpK);
		
		void addSoundingPointWind (double hpa, double vx, double vy);
		
		QList <SoundingPoint> * getAllSounds() {return &allSounds;}
		QList <SoundingPointWind> * getAllSoundsWind() {return &allSoundsWind;}
		
		double getTempCByAlt (double hpa);
		double getDewpCByAlt (double hpa);

		double getAvgTempCByAlt (double hpa1, double hpa2);
		double getAvgDewpCByAlt (double hpa1, double hpa2);

		double getAltByTempC (double tempC);
		double getAltByDewpC (double dewpC);
		
		double hpaMin ();
		double hpaMax ();
									
		TPoint get_LCL (double hpa0max, double hpa0min);  // Lifted condensation level
		TPoint get_CCL (double hpa0max, double hpa0min);  // Convective condensation level
		TPoint get_LFC (double hpa0max, double hpa0min);  // Level of free convection
		TPoint get_EL  (double hpa0max, double hpa0min);  // Equilibrium level
		
		
	private:
		QList <SoundingPoint> allSounds;
		QList <SoundingPointWind> allSoundsWind;
	
		bool   clvl_ok;			// Convective levels base
		double clvl_hpa0min, clvl_hpa0max;
		ConvBase clvl_base;
		
		TPoint LCL, CCL, LFC, EL;
		void   compute_convective_levels (double hpa0max, double hpa0min);
};

#endif
