/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#include "BoardToolBar.h"


//=====================================================================
void BoardToolBar::showGribPointInfo(const GribPointInfo &pf)
{
    QString s, res;
    
    lbLon.setText(Util::formatLongitude(pf.x));
    lbLat.setText(Util::formatLatitude(pf.y));
    
    if (pf.hasPressure())
        lbPres.setText(s.sprintf("%.1f ",pf.pressure/100.0)+tr("hPa"));
    else
        lbPres.setText("");

    if (pf.hasWind()) {
        float v = sqrt(pf.vx*pf.vx + pf.vy*pf.vy);
        
        float dir = -atan2(-pf.vx, pf.vy) *180.0/M_PI + 180;
        if (dir < 0)
            dir += 360.0;
        if (dir >= 360)
            dir -= 360.0;
        
        lbWindDir.setText(s.sprintf("%.0f ", dir)+tr("°"));
        
        lbWindSpeed.setText(Util::formatSpeed(v));
        
        lbWindBf.setText(s.sprintf("%2d ", Util::kmhToBeaufort(v*3.6))+tr("Bf"));
    }
    else {
        lbWindDir.setText("");
        lbWindSpeed.setText("");
        lbWindBf.setText("");
    }
    
    if (pf.hasRain()) {
        if (pf.rain<10)
            lbRain.setText(s.sprintf("%.2f ",pf.rain)+tr("mm/h"));
        else
            lbRain.setText(s.sprintf("%.1f ",pf.rain)+tr("mm/h"));
    }
    else {
        lbRain.setText("");
    }

    if (pf.hasTemp())
        lbTemp.setText(Util::formatTemperature(pf.temp));
    else
        lbTemp.setText("");
        
    if (pf.hasTempPot())
        lbTempPot.setText(Util::formatTemperature(pf.tempPot));
    else
        lbTempPot.setText("");
        
    if (pf.hasTempMin())
        lbTempMin.setText(Util::formatTemperature(pf.tempMin));
    else
        lbTempMin.setText("");
    
    if (pf.hasTempMax())
        lbTempMax.setText(Util::formatTemperature(pf.tempMax));
    else
        lbTempMax.setText("");

    if (pf.hasCloud())
        lbCloud.setText(Util::formatPercentValue(pf.cloud));
    else
        lbCloud.setText("");

    if (pf.hasHumid())
        lbHumid.setText(Util::formatPercentValue(pf.humid));
    else
        lbHumid.setText("");

    if (pf.hasDewPoint())
        lbDewPoint.setText(Util::formatTemperature(pf.dewPoint));
    else
        lbDewPoint.setText("");

    if (pf.hasIsotherm0HGT())
        lbIsotherm0HGT.setText(Util::formatAltitude(pf.isotherm0HGT));
    else
        lbIsotherm0HGT.setText("");

    if (pf.hasSnowDepth())
        lbSnowDepth.setText(Util::formatSnowDepth(pf.snowDepth));
    else
        lbSnowDepth.setText("");
    
    if (pf.hasCAPEsfc())
        lbCAPEsfc.setText(s.sprintf("%d ", (int)pf.CAPEsfc) +tr("J/kg"));
    else
        lbCAPEsfc.setText("");
}

//===================================================================================
BoardToolBar::BoardToolBar(QWidget *parent)
    : QToolBar(parent)
{
//     this->setFloatable(false);
//     this->setMovable(false);
    this->setOrientation(Qt::Vertical);
    
    QFrame *frm;
    QHBoxLayout *hlay;
    QFontMetrics fmt(qApp->font());
    int style = QFrame::Panel | QFrame::Sunken;
    
    lbLon.setMinimumWidth( fmt.width("X888X88X00X-") );
    lbLon.setFrameStyle(style);
    lbLat.setMinimumWidth( fmt.width("X888X88X00X-") );
    lbLat.setFrameStyle(style);
    
    lbWindDir.setMinimumWidth( fmt.width("888XX-") );
    lbWindDir.setFrameStyle(style);
    lbWindSpeed.setMinimumWidth( fmt.width("888X8Xkm/h-") );
    lbWindSpeed.setFrameStyle(style);
    lbWindBf.setMinimumWidth( fmt.width("8888Bf-") );
    lbWindBf.setFrameStyle(style);
    
    QString strlong = "88888888888";
    lbPres.setMinimumWidth( fmt.width(strlong) );
    lbPres.setFrameStyle(style);
    lbRain.setMinimumWidth( fmt.width(strlong) );
    lbRain.setFrameStyle(style);
    lbTemp.setMinimumWidth( fmt.width(strlong) );
    lbTemp.setFrameStyle(style);
    lbTempPot.setMinimumWidth( fmt.width(strlong) );
    lbTempPot.setFrameStyle(style);
    lbTempMin.setMinimumWidth( fmt.width(strlong) );
    lbTempMin.setFrameStyle(style);
    lbTempMax.setMinimumWidth( fmt.width(strlong) );
    lbTempMax.setFrameStyle(style);
    lbCloud.setMinimumWidth( fmt.width(strlong) );
    lbCloud.setFrameStyle(style);
    lbHumid.setMinimumWidth( fmt.width(strlong) );
    lbHumid.setFrameStyle(style);
    lbDewPoint.setMinimumWidth( fmt.width(strlong) );
    lbDewPoint.setFrameStyle(style);
    lbIsotherm0HGT.setMinimumWidth( fmt.width(strlong) );
    lbIsotherm0HGT.setFrameStyle(style);
    lbSnowDepth.setFrameStyle(style);
    lbSnowDepth.setMinimumWidth( fmt.width(strlong) );
    lbSnowCateg.setFrameStyle(style);
    lbSnowCateg.setMinimumWidth( fmt.width(strlong) );
    lbCAPEsfc.setFrameStyle(style);
    lbCAPEsfc.setMinimumWidth( fmt.width(strlong) );
    
    	// "0" to have definitive widget height on construct
        lbPres.setText("0");
        lbWindDir.setText("0");
        lbWindSpeed.setText("0");
        lbWindBf.setText("0");
        lbRain.setText("0");
        lbTemp.setText("0");
        lbTempPot.setText("0");
        lbTempMin.setText("0");
        lbTempMax.setText("0");
        lbCloud.setText("0");
        lbHumid.setText("0");
        lbDewPoint.setText("0");
        lbIsotherm0HGT.setText("0");
        lbSnowDepth.setText("0");
        lbCAPEsfc.setText("0");
        
    //--------------------------
    this->addSeparator();
    //--------------------------    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
//        hlay->addWidget(new QLabel(tr("Lat:")), 0, Qt::AlignRight);
        hlay->addWidget(&lbLat, 0, Qt::AlignRight);
/*        frm->setLayout(hlay);
    this->addWidget(frm);
    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(new QLabel(tr("Lon:")), 0, Qt::AlignRight);*/
        hlay->addWidget(&lbLon, 0, Qt::AlignRight);
        frm->setLayout(hlay);
    this->addWidget(frm);    
    //--------------------------    
    this->addSeparator();
    //--------------------------        
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(new QLabel(tr("Vent")), 0, Qt::AlignRight);
        hlay->addWidget(&lbWindDir, 0, Qt::AlignRight);
        hlay->addWidget(&lbWindBf, 0, Qt::AlignRight);
/*        frm->setLayout(hlay);
    this->addWidget(frm);
    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);*/
        hlay->addWidget(&lbWindSpeed, 0, Qt::AlignRight);
        frm->setLayout(hlay);
    this->addWidget(frm);

    //--------------------------    
    this->addSeparator();
    //--------------------------    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(new QLabel(tr("Pression")), 0, Qt::AlignCenter);
        hlay->addWidget(&lbPres, 0, Qt::AlignRight);
        frm->setLayout(hlay);
    this->addWidget(frm);

    //--------------------------    
    this->addSeparator();
    //--------------------------    
    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(new QLabel(tr("Température")), 0, Qt::AlignCenter);
        hlay->addWidget(&lbTemp, 0, Qt::AlignRight);
        frm->setLayout(hlay);
    this->addWidget(frm);
    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(new QLabel(tr("Min")), 0, Qt::AlignRight);
        hlay->addWidget(&lbTempMin, 0, Qt::AlignRight);
        hlay->addWidget(new QLabel(tr("Max")), 0, Qt::AlignRight);
        hlay->addWidget(&lbTempMax, 0, Qt::AlignRight);
        frm->setLayout(hlay);
    this->addWidget(frm);
    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(new QLabel(tr("Point de rosée")), 10, Qt::AlignRight);
        hlay->addWidget(&lbDewPoint, 0, Qt::AlignRight);
        frm->setLayout(hlay);
    this->addWidget(frm);
    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(new QLabel(tr("Temp. potentielle")), 10, Qt::AlignRight);
        hlay->addWidget(&lbTempPot, 0, Qt::AlignRight);
        frm->setLayout(hlay);
    this->addWidget(frm);

    //--------------------------    
    this->addSeparator();
    //--------------------------    
    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(new QLabel(tr("Précipitations")), 10, Qt::AlignRight);
        hlay->addWidget(&lbRain, 0, Qt::AlignRight);
        frm->setLayout(hlay);
    this->addWidget(frm);
    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(new QLabel(tr("Nébulosité")), 10, Qt::AlignRight);
        hlay->addWidget(&lbCloud, 0, Qt::AlignRight);
        frm->setLayout(hlay);
    this->addWidget(frm);
    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(new QLabel(tr("Humidité relative")), 10, Qt::AlignRight);
        hlay->addWidget(&lbHumid, 0, Qt::AlignRight);
        frm->setLayout(hlay);
    this->addWidget(frm);
    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(new QLabel(tr("Isotherme 0°C")), 10, Qt::AlignRight);
        hlay->addWidget(&lbIsotherm0HGT, 0, Qt::AlignRight);
        frm->setLayout(hlay);
    this->addWidget(frm);
    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(new QLabel(tr("Epaisseur de neige")), 10, Qt::AlignRight);
        hlay->addWidget(&lbSnowDepth, 0, Qt::AlignRight);
        frm->setLayout(hlay);
    this->addWidget(frm);
    
    frm = new QFrame(this);
    assert(frm);
    hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(new QLabel(tr("CAPE (surface)")), 10, Qt::AlignRight);
        hlay->addWidget(&lbCAPEsfc, 0, Qt::AlignRight);
        frm->setLayout(hlay);
    this->addWidget(frm);
    
    //--------------------------    
    this->addSeparator();
    //--------------------------    
    
    	// Clear "0"
        lbPres.setText("");
        lbWindDir.setText("");
        lbWindSpeed.setText("");
        lbWindBf.setText("");
        lbRain.setText("");
        lbTemp.setText("");
        lbTempPot.setText("");
        lbTempMin.setText("");
        lbTempMax.setText("");
        lbCloud.setText("");
        lbHumid.setText("");
        lbDewPoint.setText("");
        lbIsotherm0HGT.setText("");
        lbSnowDepth.setText("");
        lbCAPEsfc.setText("");
}






