# Copyright (c) 2003-2006 Ingeniweb SAS

# This software is subject to the provisions of the GNU General Public
# License, Version 2.0 (GPL).  A copy of the GPL should accompany this
# distribution.  THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL
# EXPRESS OR IMPLIED WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF TITLE, MERCHANTABILITY,
# AGAINST INFRINGEMENT, AND FITNESS FOR A PARTICULAR PURPOSE

# More details in the ``LICENSE`` file included in this package.

"""
PloneArticle base test

$Id: testPloneArticle.py,v 1.15 2006/09/18 20:52:55 encolpe Exp $
"""

from common import *
from Products.CMFCore.utils import getToolByName

# Fake upload object
class TestFile:
    __allow_access_to_unprotected_subobjects__ = 1
    filename = 'afile.fil'
    def seek(*args): pass
    def tell(*args): return 0
    def read(*args): return 'file_contents'

tests = []

class TestPloneArticle(PloneArticleTestCase):

    def testCreateArticle(self, ):
        """
        """
        self.loginAsPortalMember()
        self.article = self.createEmptyArticle(self.member_folder)

    def testEditArticle(self, ):
        """
        Editing article
        """
        self.testCreateArticle()
        title = "A dummy title"
        text = "<p>My dummy text</p>"
        description = "A dummy description"

        # Edit content
        self.article.edit(text = text, title = title, description = description, text_format = 'html')

        # test if everything is ok
        self.failUnless(self.article.Title() == title, "Value is %s" % self.article.Title())
        self.failUnless(self.article.Description() == description, "Value is %s" % self.article.Description())
        self.failUnless(self.article.getText() == text, "Value is %s" % self.article.getText())


    def addAttachment(self, container=None, title = 'attachment_title', description = 'attachment_description'):
        """
        """
        container = container or self.testCreateArticle()
        self.article.addAttachment(self, title, description, file=TestFile())


    def testAddAttachment(self, ):
        """
        """
        #create article and add attachment
        self.addAttachment()
        contents = self.article.article_contents
        attachments = contents[0].getContents(self.article)
        #the article has only one attachment
        self.assertEqual(1, len(attachments))

        #the attachment added is an ATFile
        self.assertEqual(self.portal.portal_types.getTypeInfo(FILE_PORTAL_TYPE),
                         attachments[0].getTypeInfo())

        """
        from Products.PloneArticle.ArticleCore import ArticleContents
        self.failUnless(isinstance(attachments[0], ArticleContents),
                        'Value is %s' % type(attachments[0]))
        """
        #test content
        self.assertEqual('file_contents', str(attachments[0]))



    def testGetAttachment(self, ):
        """
        """
        self.testAddAttachment()
        contents = self.article.article_contents
        attachments = contents[0].getContents(self.article)

        #attachment uid is non existant
        uid = 'No chance it exists'
        self.assertEqual(self.article.getAttachment(uid), None)

        #get the uid of added attachment
        uid = attachments[0].UID()
        #get attachment by uid
        attachment = self.article.getAttachment(uid)
        self.assertEqual(self.article.getAttachment(uid), attachments[0])
        #the return attachment is an ATFile
        self.assertEqual(self.portal.portal_types.getTypeInfo(FILE_PORTAL_TYPE),
                         attachment.getTypeInfo())

    def testGetAttachmentBrain(self, ):
        """
        """
        self.testAddAttachment()
        contents = self.article.article_contents
        attachments = contents[0].getContents(self.article)
        uid = attachments[0].UID()
        brain = self.article.getAttachmentBrain(uid)

        #getAttachmentBrain return an ArticleReferenceBrain
        from Products.PloneArticle.ArticleCore import ArticleReferenceBrain
        self.failUnless(isinstance(brain, ArticleReferenceBrain),
                        'Value is %s' % type(brain))


    def testGetAttachments(self, ):
        """
        """
        self.testAddAttachment()
        attachments = self.article.getAttachments()
        self.assertEqual(1, len(attachments))
        self.addAttachment(self.article, 'title2', 'desc2')
        attachments = self.article.getAttachments()
        self.assertEqual(2, len(attachments))

    def testGetAttachmentBrains(self,):
        """
        """
        self.testCreateArticle()
        external_file = self.createExternalFile()
        self.article.addAttachmentFromObject(external_file)
        brains = self.article.getAttachmentBrains()

        #getAttachmentBrains return an ArticleReferenceBrain
        from Products.PloneArticle.ArticleCore import ArticleReferenceBrain
        self.failUnless(isinstance(brains[0], ArticleReferenceBrain),
                        'Value is %s' % type(brains[0]))


    def testGetAttachmentUIDs(self, ):
        """
        """
        self.testCreateArticle()
        external_file = self.createExternalFile()
        self.article.addAttachmentFromObject(external_file)
        contents = self.article.article_contents
        attachments = contents[0].getContents(self.article)
        uids = [attach.UID() for attach in attachments]
        # the uid (source) is in getAttachmentUIDS (referenced in article)
        self.failUnless(external_file.UID() in self.article.getAttachmentUIDs())
        # same length
        self.assertEqual(len(uids), len(attachments))

    def testAddAttachmentFromObject(self, ):
        """
        """
        self.testCreateArticle()
        external_file = self.createExternalFile()
        self.article.addAttachmentFromObject(external_file)
        obj = getattr(self.article, external_file.id)
        # file in article is the same as external_file
        self.assertEqual(obj, external_file)
        #added object is return as reference
        ref = self.article.getAttachmentBrain(external_file.UID())
        self.assertEqual(ref.getObject(), obj)

    def testAddAttachmentFromUID(self, ):
        """
        """
        self.testCreateArticle()
        external_file = self.createExternalFile()
        uid = external_file.UID()
        self.article.addAttachmentFromUID(uid)
        attachment = self.article.getAttachment(uid)
        self.failUnless(attachment,
                        'Value is %s' % attachment)

    def testEditAttachment(self, ):
        """
        """
        self.testAddAttachment()
        attachment = self.article.getAttachments()[0]
        uid = attachment.UID()
        title = 'A Dummy Title'
        description = 'A Dummy Description'
        self.article.editAttachment(uid, title, description)
        self.failUnless(attachment.Title() == title, "Value is %s" % attachment.Title())
        self.failUnless(attachment.Description() == description, "Value is %s" % attachment.Description())

    def testEditAttachments(self, ):
        """
        Edit attachments.
        Parameters
            attachment_infos -> List of dictionnaries (UID, title, description)
        """
        self.testAddAttachment()
        #add a second attachment
        self.addAttachment(container=self.article)
        title = 'a title'
        description = 'a description'
        attachments = [
          {'UID': attachment.UID(),
           'title': title,
           'description': description
          } for attachment in self.article.getAttachments()
        ]
        self.article.editAttachments(attachments)
        attach1 = self.article.getAttachments()[0]
        attach2 = self.article.getAttachments()[1]
        self.assertEqual(attach1.Title(), title)
        self.assertEqual(attach1.Description(), description)

    def testRemoveAttachment(self, ):
        """ Remove one attachment
        """
        self.testAddAttachment()
        self.addAttachment(container=self.article)
        attachments = self.article.getAttachments()
        uids = [attachment.UID() for attachment in attachments]
        uid = uids[1]
        self.article.removeAttachment(uid)
        self.assertEqual(None, self.article.getAttachment(uid))

    def testRemoveAttachments(self, ):
        """ Remove all attachments
        """
        self.testAddAttachment()
        self.addAttachment(container=self.article)
        attachments = self.article.getAttachments()
        uids = [attachment.UID() for attachment in attachments]
        self.article.removeAttachments(uids)
        self.assertEqual([], self.article.getAttachments())


    def testMoveAttachmentsByDelta(self, ):
        """
        """
        self.testAddAttachment()
        self.addAttachment(container=self.article)
        attach1 = self.article.getAttachments()[0]
        attach2 = self.article.getAttachments()[1]

    def testCopyAttachmentsToClipboard(self, ):
        """ TODO
        """
        pass

    def testIsAttachmentContained(self, ):
        """ TODO
        """
        pass

    def testCopyArticle(self,):
        """ Copy article
        """

        self.loginAsPortalMember()
        self.createArticle(self.member_folder)
        article_id = self.article.getId()
        cb = self.member_folder.manage_copyObjects(ids=[article_id])
        self.member_folder.manage_pasteObjects(cb_copy_data=cb)
        new_article = getattr(self.member_folder, 'copy_of_%s' % article_id)

        attachment_uids = new_article.getAttachmentUIDs()
        image_uids = new_article.getImageUIDs()
        self.assertEqual(len(attachment_uids), 2)
        self.assertEqual(len(image_uids), 2)
        self.failUnless(self.external_file.UID() in attachment_uids)
        self.failUnless(self.external_image.UID() in image_uids)
        self.failUnless(self.article_file.UID() not in attachment_uids)
        self.failUnless(self.article_image.UID() not in image_uids)
        self.logout()

    def testCutArticle(self,):
        """ Cut article
        """
        return
        self.loginAsPortalMember()
        self.createArticle(self.member_folder)
        article_id = self.article.getId()

        # Create new folder
        new_folder_id = 'test'
        self.member_folder.invokeFactory(type_name='Folder', id=new_folder_id)
        new_folder = getattr(self.member_folder, new_folder_id)

        # make sure we have _p_jar
        get_transaction().commit(1)

        # Cut / Paste
        cb = self.member_folder.manage_cutObjects(ids=[article_id])
        new_folder.manage_pasteObjects(cb_copy_data=cb)
        new_article = getattr(self.member_folder, article_id)

        article_file = getattr(new_article, 'article_file', None)
        self.failUnless(article_file is not None)
        article_image = getattr(new_article, 'article_image', None)
        self.failUnless(article_image is not None)
        attachment_uids = new_article.getAttachmentUIDs()
        image_uids = new_article.getImageUIDs()
        self.assertEqual(len(attachment_uids), 2)
        self.assertEqual(len(image_uids), 2)
        self.failUnless(self.external_file.UID() in attachment_uids)
        self.failUnless(self.external_image.UID() in image_uids)
        self.failUnless(article_file.UID() in attachment_uids)
        self.failUnless(article_image.UID() in image_uids)
        self.logout()

    def testRenameArticle(self,):
        """ Rename article
        """

        self.loginAsPortalMember()
        self.createArticle(self.member_folder)
        article_id = self.article.getId()

        # make sure we have _p_jar
        get_transaction().commit(1)

        # Rename article
        new_article_id = 'new_article'
        self.member_folder.manage_renameObject(article_id, new_article_id)

        # Test renamed article
        new_article = getattr(self.member_folder, new_article_id)
        article_file = getattr(new_article, 'article_file', None)
        self.failUnless(article_file is not None)
        article_image = getattr(new_article, 'article_image', None)
        self.failUnless(article_image is not None)
        attachment_uids = new_article.getAttachmentUIDs()
        image_uids = new_article.getImageUIDs()
        self.assertEqual(len(attachment_uids), 2)
        self.assertEqual(len(image_uids), 2)
        self.failUnless(self.external_file.UID() in attachment_uids)
        self.failUnless(self.external_image.UID() in image_uids)
        self.failUnless(article_file.UID() in attachment_uids)
        self.failUnless(article_image.UID() in image_uids)
        self.logout()

    #
    # TOOL TESTS
    #
    def testAddPloneArticleTool(self, ):
        """
        Test adding of plone_article_tool
        """
        self.failUnless('portal_article' in self.portal.objectIds(), "plone_article tool was not installed")

    def testPloneArticleToolMaxAttachmentSize(self, size=4242):
        """
        Test max attachment size setting in Plone Article Tool
        """

        portal_article = getToolByName(self.portal, 'portal_article')
        # Default should be 0: Unlimited
        self.failUnless(portal_article.getMaxAttachmentSize() == 0)

        # Change to size
        portal_article.max_attachment_size = size
        self.failUnless(portal_article.getMaxAttachmentSize() == size)


    def testPloneArticleToolMaxImageSize(self, size=4242):
        """
        Test max attachment size setting in Plone Article Tool
        """

        portal_article = getToolByName(self.portal, 'portal_article')
        # Default should be 0: Unlimited
        self.failUnless(portal_article.getMaxImageSize() == 0)

        # Change to size
        portal_article.max_image_size = size
        self.failUnless(portal_article.getMaxImageSize()==size)

    def testArticleToolLockSupport(self,):
        """
        """

        portal_article = getToolByName(self.portal, 'portal_article')
        hasLockSupport = portal_article.hasLockSupport()

        #default is 0
        self.assertEqual(hasLockSupport, 0)

    def testArticleToolCMFEditionsSupport(self,):
        """
        """

        portal_article = getToolByName(self.portal, 'portal_article')
        hasCMFEditionsSupport = portal_article.hasCMFEditionsSupport()

        #default is 0
        self.assertEqual(hasCMFEditionsSupport, 0)

    def testPloneArticleToolPreview(self,):
        """
        Test preview setting in Plone Article Tool
        """

        portal_article = getToolByName(self.portal, 'portal_article')

        # Default should be 1: Switched on
        self.failUnless(portal_article.isPreviewActivated() == 1)

        # Change to 0
        portal_article.preview_activated = 0
        self.failUnless(portal_article.isPreviewActivated() == 0)

        # Change to 1
        portal_article.preview_activated = 1
        self.failUnless(portal_article.isPreviewActivated() == 1)



    def TODO_testPloneArticleToolTextFormat(self,):
        """
        Test text format setting in Plone Article Tool
        """
        portal_article = getToolByName(self.portal, 'portal_article')

        # Default should be ...
        #self.assertEqual(portal_article.getSiteTextFormats(), ('html', ))

        # Should be html, plain and structured-text
        self.failUnless(
            portal_article.getAvailableTextFormats() == [
                'html', 'plain', 'structured-text',
                ],
            )

        # Structured Text
        portal_article.setSiteTextFormats(['structured-text',])
        self.assertEqual(portal_article.getSiteTextFormats(), ('structured-text', ))

        # HTML
        portal_article.setSiteTextFormats(['html',])
        self.assertEqual(portal_article.getSiteTextFormats(), ('html', ))

        #self.failUnless('html' in self.portal_article.getSiteTextFormats())
        #self.failIf('plain' in self.portal_article.getSiteTextFormats())
        #self.failIf('structured-text' in self.portal_article.getSiteTextFormats())

        # Plain text
        portal_article.setSiteTextFormats(['plain'])
        self.assertEqual(
            portal_article.getSiteTextFormats(),
            ('plain', )
        )

        #self.failUnless('plain' in self.portal_article.getSiteTextFormats())
        #self.failIf('html' in self.portal_article.getSiteTextFormats())
        #self.failIf('structured-text' in self.portal_article.getSiteTextFormats())

        # Should be plain and structured-text
        portal_article.setSiteTextFormats(['plain','structured-text'])
        self.assertEqual(
            portal_article.getSiteTextFormats(),
            ('plain','structured-text')
        )

        #self.failUnless('plain' in self.portal_article.getSiteTextFormats())
        #self.failUnless('structured-text' in self.portal_article.getSiteTextFormats())
        #self.failIf('html' in self.portal_article.getSiteTextFormats())

        # Should be html, plain and structured-text
        portal_article.setSiteTextFormats(['plain','html','structured-text'])
        self.assertEqual(
            portal_article.getSiteTextFormats(),
            ('plain','html','structured-text', )
        )

        #self.failUnless('plain' in self.portal_article.getSiteTextFormats())
        #self.failUnless('structured-text' in self.portal_article.getSiteTextFormats())
        #self.failUnless('html' in self.portal_article.getSiteTextFormats())

        # Should be html and structured-text
        portal_article.setSiteTextFormats(['plain','structured-text'])
        self.assertEqual(
            portal_article.getSiteTextFormats(),
            ('plain','structured-text', )
        )


    def testGetModel(self,):
        """
        Test if model is available
        """
        # Basic things
        self.testCreateArticle()
        model = self.article.getModel()

        # Check that model exists and is default
        self.failUnless(model is not None)
        self.failUnless(model == "plonearticle_model")

    def testSetModel(self,):
        """
        Test model changing
        """

        portal_article = getToolByName(self.portal, 'portal_article')
        self.testCreateArticle()
        #test the default situation : only 8 model can be chosen
        self.assertEqual(len(portal_article.getSiteModels()), 8)

        # Set a valid model
        site_models = portal_article.getAvailableModels()
        portal_article.site_models = site_models
        self.article.setModel("plonearticle_model2")
        model = self.article.getModel()
        self.failUnless(model == "plonearticle_model2")
        kwargs = {'model' : 'ZORGLUB', }
        # Set an invalid model => must raise ValueError
        self.assertRaises(ValueError, self.article.setModel, **kwargs)
        """
        try:
            article.setModel("ZORGLUB")
        except ValueError:
            pass
        else:
            self.failUnless(0)
        """
        # Force an invalid model => must get back to the default one
        self.article.model = "ZORGLUB"
        model = self.article.getModel()
        self.failUnless(model == "plonearticle_model")


tests.append(TestPloneArticle)

if __name__ == '__main__':
    framework()
else:
    # While framework.py provides its own test_suite()
    # method the testrunner utility does not.
    import unittest
    def test_suite():
        suite = unittest.TestSuite()
        for test in tests:
            suite.addTest(unittest.makeSuite(test))
        return suite

