# Copyright (c) 2003-2006 Ingeniweb SAS

# This software is subject to the provisions of the GNU General Public
# License, Version 2.0 (GPL).  A copy of the GPL should accompany this
# distribution.  THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL
# EXPRESS OR IMPLIED WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF TITLE, MERCHANTABILITY,
# AGAINST INFRINGEMENT, AND FITNESS FOR A PARTICULAR PURPOSE

# More details in the ``LICENSE`` file included in this package.

"""Plone Article with images and attachments and layout models

$id$
"""

__author__ = ''
__docformat__ = 'restructuredtext'

# Zope imports
from AccessControl import ClassSecurityInfo
from ComputedAttribute import ComputedAttribute
from Acquisition import aq_parent

# CMF imports
from Products.CMFCore import CMFCorePermissions
from Products.CMFCore.utils import getToolByName

# Archetypes imports
try:
    from Products.LinguaPlone.public import *
except ImportError:
    # No multilingual support
    from Products.Archetypes.public import *

# plone 2.1 stuff
try:
    from Products.CMFPlone.interfaces.BrowserDefault import IBrowserDefault
    _plone_2_1_implements = (IBrowserDefault,)
except ImportError:
    _plone_2_1_implements = ()

# PloneArticle imports
from Products.PloneArticle.config import *
from Products.PloneArticle import Permissions
from Products.PloneArticle.ATContentType import DocumentContent
from Products.PloneArticle.ArticleCore import registerPloneArticle, ArticleContents
from Products.PloneArticle.ArticleModel import ArticleModel
from Products.PloneArticle.ArticleLock import ArticleLock
from Products.PloneArticle.interfaces.IArticle import IArticleAttachments, IArticleImages, IArticleLinks


# PloneArticle schema definition
PloneArticleSchema = DocumentContent.schema + ArticleLock.schema + ArticleModel.schema


class PloneArticle(ArticleLock, ArticleModel, DocumentContent, BaseFolderMixin):
    """PloneArticle class"""

    schema = PloneArticleSchema.copy()

    article_contents = (
        ArticleContents('attachment'),
        ArticleContents('image'),
        ArticleContents('link'), )

    content_icon   = 'PloneArticle.gif'
    meta_type      = 'PloneArticle'
    archetype_name = 'Article'
    default_view = 'plonearticle_view'
    immediate_view = 'plonearticle_view'
    suppl_views    = ()
    newTypeFor     = ()
    typeDescription= ''
    typeDescMsgId  = ''
    assocMimetypes = ()
    assocFileExt   = ()

    _at_rename_after_creation = True

    use_folder_tabs = 0
    filter_content_types  = 0
    allowed_content_types = ()
    include_default_actions = 0

    actions = (
         {
          'id'          : 'view',
          'name'        : 'View',
          'action'      : 'string:${object_url}/view',
          'permissions' : (CMFCorePermissions.View,),
          'visible'     : 1,
         },
         {
          'id'          : 'edit',
          'name'        : 'Edit',
          'action'      : 'string:${object_url}/base_edit',
          'condition'     : 'python:not object.hasLockSupport() or object.canToggleLock()',
          'permissions' : (Permissions.EditPloneArticle,),
          'visible'     : 1,
         },
         {
          'id'          : 'image',
          'name'        : 'Images',
          'action'      : 'string:${object_url}/plonearticle_image_form',
          'condition'   : 'python:not object.hasLockSupport() or object.canToggleLock()',
          'permissions' : (Permissions.EditPloneArticle,),
          'visible'     : 1,
         },
         {
          'id'          : 'attach',
          'name'        : 'Attachments',
          'action'      : 'string:${object_url}/plonearticle_attachment_form',
          'condition'   : 'python:not object.hasLockSupport() or object.canToggleLock()',
          'permissions' : (Permissions.EditPloneArticle,),
          'visible'     : 1,
         },
         {
          'id'          : 'link',
          'name'        : 'Links',
          'action'      : 'string:${object_url}/plonearticle_link_form',
          'condition'   : 'python:not object.hasLockSupport() or object.canToggleLock()',
          'permissions' : (Permissions.EditPloneArticle,),
          'visible'     : 1,
         },
         {
          'id'          : 'model',
          'name'        : 'Model',
          'action'      : 'string:${object_url}/plonearticle_model_form',
          'condition'   : 'python:not object.hasLockSupport() or object.canToggleLock()',
          'permissions' : (Permissions.EditPloneArticle,),
          'visible'     : 1,
         },
         {
          'id'            : 'lock_unlock',
          'name'          : 'Unlock (Currently locked)',
          'action'        : 'string:${object_url}/plonearticle_toggle_lock',
          'permissions'   : (CMFCorePermissions.View, ),
          'condition'     : 'python:object.isLocked() and object.canToggleLock()',
          'category'      : 'document_actions',
         },
         {
          'id'            : 'unlock_lock',
          'name'          : 'Lock (Currently not locked)',
          'action'        : 'string:${object_url}/plonearticle_toggle_lock',
          'permissions'   : (CMFCorePermissions.View, ),
          'condition'     : 'python:not object.isLocked() and object.canToggleLock()',
          'category'      : 'document_actions',
         },
         {
          'id'          : 'metadata',
          'name'        : 'Properties',
          'action'      : 'string:${object_url}/base_metadata',
          'condition'   : 'python:not object.hasLockSupport() or object.canToggleLock()',
          'permissions' : (Permissions.EditPloneArticle,),
          'visible'     : 1,
         },
         { 'id'         : 'references',
           'name'       : 'References',
           'action'     : 'string:${object_url}/reference_graph',
           'condition'  : 'object/archetype_tool/has_graphviz',
           'permissions': (CMFCorePermissions.ModifyPortalContent,
                           CMFCorePermissions.ReviewPortalContent,),
           'visible'    : True,
           },
         {
           'id'          : 'local_roles',
           'name'        : 'Sharing',
           'action'      : 'string:${object_url}/folder_localrole_form',
           'condition'   : 'python:not object.hasLockSupport() or object.canToggleLock()',
           'permissions' : (CMFCorePermissions.ManageProperties,),
           'visible'     : True,
           },
         )

    __implements__ = ( DocumentContent.__implements__
                     , IArticleAttachments
                     , IArticleImages
                     , IArticleLinks
                     ) + _plone_2_1_implements

    security = ClassSecurityInfo()

    # backward compat
    text_format = 'text/html'

    # Use folder methods for objectValues, objectItems, objectIds
    objectValues = BaseFolderMixin.objectValues
    objectIds = BaseFolderMixin.objectIds
    objectItems = BaseFolderMixin.objectItems

    # Init method
    def __init__(self, oid, **kwargs):
        """__init__(self, oid, **kwargs)"""

        ArticleLock.__init__(self)
        BaseFolderMixin.__init__(self, oid, **kwargs)
        DocumentContent.__init__(self, oid, **kwargs)

    security.declarePrivate('manage_afterAdd')
    def manage_afterAdd(self, item, container):
        """Copy references from source to dest. We must keep attachment references."""

        is_copy = getattr(item, '_v_is_cp', None)
        at_tool = getToolByName(self, 'archetype_tool')

        # Get source and destination objects
        src_UID = self.UID()

        DocumentContent.manage_afterAdd(self, item, container)
        BaseFolderMixin.manage_afterAdd(self, item, container)

        src_obj = at_tool.getObject(src_UID)
        dst_UID = self.UID()
        dst_obj = self

        if is_copy and src_obj is not None and dst_UID != src_UID:
            # Copy references from source Article
            # Compute new references for contents in Article
            # Avoid from renaming : dst_UID != src_UID
            for article_content in self.article_contents:
                for ref in article_content.getContentReferences(src_obj):
                    if hasattr(ref, article_content._content_brain_id):
                        brain = getattr(ref, article_content._content_brain_id)
                        content_uid = ref.targetUID
                        content_title = brain.Title()
                        content_description = brain.Description()

                        if article_content.isContentContained(src_obj, content_uid):
                            # Get content in new article
                            content_obj = at_tool.getObject(content_uid)

                            # Get id and get new content with new article with this ID
                            content_id = content_obj.getId()

                            new_obj = getattr(dst_obj, content_id)
                            content_uid = new_obj.UID()

                        article_content.addContentFromUID(dst_obj, content_uid)
                        article_content.editContent(dst_obj, content_uid, title=content_title, description=content_description)


    security.declareProtected(CMFCorePermissions.View, 'Description')
    def Description(self, **kwargs):
        """We have to override Description here to handle arbitrary
        arguments since PortalFolder defines it."""
        return self.getField('description').get(self, **kwargs)

    security.declareProtected(CMFCorePermissions.ModifyPortalContent, 'setDescription')
    def setDescription(self, value, **kwargs):
        """We have to override setDescription here to handle arbitrary
        arguments since PortalFolder defines it."""
        self.getField('description').set(self, value, **kwargs)

    security.declareProtected(Permissions.EditPloneArticle, 'setText')
    def setText(self, value, **kwargs):
        """Body text mutator
        """
        allowable_content_types = TEXT_ALLOWABLE_CONTENT_TYPES
        mimetype = kwargs.get('mimetype', None)

        if mimetype is None:
            # Select the good mimetype
            if len(allowable_content_types) == 1:
                kwargs['mimetype'] = allowable_content_types[0]
            elif len(allowable_content_types) == 0:
                kwargs['mimetype'] = 'text/html'

        # Set text field
        DocumentContent.setText(self, value, **kwargs)

    security.declareProtected(CMFCorePermissions.View, 'SearchableText')
    def SearchableText(self):
        """Returns a concatination of all searchable text
        """

        attach_index_values = []

        for attach in self.getAttachmentBrains():
            # Append title
            attach_index_values.append(attach.Title())

            # Check if attach use AttachmentField
            attach_obj = attach.getObject()
            attach_field = attach_obj.getField('file')

            if hasattr(attach_field, 'isIndexed') and attach_field.isIndexed(attach_obj):
                # Append attach index value
                attach_index_values.append(attach_field.getIndexableValue(attach_obj))

        ret = "%s %s %s" % (
            DocumentContent.SearchableText(self),
            ' '.join(attach_index_values),
            ' '.join([x.Title() for x in self.getImageBrains()])
            )

        return ret


    #                                                                                   #
    #                               Tool indirections                                   #
    # We use those methods to allow people overloading their behaviour in their own     #
    # article subclasses.                                                               #

    security.declarePublic("getAllowedLinkTypes")
    def getAllowedLinkTypes(self):
        """
        Get allowed link types
        """
        tool = getToolByName(self, 'portal_article')
        return tool.allowed_link_types

    security.declarePublic("getAllowedLinkType")
    def getAllowedLinkType(self):
        """
        Get allowed link type
        """
        tool = getToolByName(self, 'portal_article')
        return tool.allowed_link_type

    security.declarePublic("getAllowedImageTypes")
    def getAllowedImageTypes(self):
        """
        Get allowed image types
        """
        tool = getToolByName(self, 'portal_article')
        return tool.allowed_image_types

    security.declarePublic("getAllowedImageType")
    def getAllowedImageType(self):
        """
        Get allowed image type
        """
        tool = getToolByName(self, 'portal_article')
        return tool.allowed_image_type

    security.declarePublic("getAllowedAttachmentTypes")
    def getAllowedAttachmentTypes(self):
        """
        Get allowed attachment types
        """
        tool = getToolByName(self, 'portal_article')
        return tool.allowed_attachment_types

    security.declarePublic("getAllowedAttachmentType")
    def getAllowedAttachmentType(self):
        """
        Get allowed attachment type
        """
        tool = getToolByName(self, 'portal_article')
        return tool.allowed_attachment_type

    def allowedContentTypes(self):
        """ Call allowedContentTypes in parent context """
        return aq_parent(self).allowedContentTypes()



# Register class
registerPloneArticle(PloneArticle, PROJECTNAME)

