##############################################################################
#
# Copyright (c) 2002 Ingeniweb SARL
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.0 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE
#
##############################################################################

"""
Plone Article core classes for images and attachments and layout models
"""

# Zope imports
from AccessControl import ClassSecurityInfo
import Acquisition
from DateTime import DateTime
from Globals import InitializeClass

# CMF imports
from Products.CMFCore import CMFCorePermissions
from Products.CMFCore.utils import getToolByName

# Archetypes imports
from Products.Archetypes.public import *

class LockMixin:

    security = ClassSecurityInfo()

    LockOwnerRole = 'LockOwner'
    LockOwnerPermission = 'Lock'
    LockEditPermission = CMFCorePermissions.ModifyPortalContent

    schema = Schema((
        # Locked Mixin
        BooleanField('is_locked',
                    accessor='isLocked',
                    mutator='lock',
                    default=False,
                    widget=BooleanWidget(
                        label='lock ?',
                        label_msgid='label_lock',
                        visible={'view' : 'hidden',
                                 'edit' : 'visible'},
            )),
        ))

    # Init method
    security.declarePrivate('__init__')
    def __init__(self):
        """__init__(self)"""
        self._locked_by = None
        self._locked_date = None
        self._unlocked_by = None
        self._unlocked_date = None

    #                                                                           #
    #                             LOCKING SUPPORT                               #
    #                                                                           #
    
    security.declareProtected(CMFCorePermissions.View, 'hasLockSupport')
    def hasLockSupport(self):
        return True

    security.declareProtected(CMFCorePermissions.View, 'isLocked')
    def isLocked(self):
        """Is the content locked ?
        """
        
        # Return lock support
        if not self.hasLockSupport():
            return False
            
        return self.getField('is_locked').get(self)

    security.declareProtected(LockEditPermission, 'lock')
    def lock(self, value, **kwargs):
        """
        lock or unlock
        0 -> unlock
        1 -> lock
        
        Returns islocked
        """
        
        # Return lock support
        if not self.hasLockSupport():
            return False
        
        # Test permission
        mtool = getToolByName(self, 'portal_membership')
        
        # Member should have edit permission
        if not mtool.checkPermission(self.LockEditPermission, self):
            return self.isLocked()
        
        # If content locked member should have lockowner permission
        if self.isLocked() and not mtool.checkPermission(self.LockOwnerPermission, self):
            return True
        
        # Get member information
        member = mtool.getAuthenticatedMember()
        username = member.getUserName()
        userid = member.getMemberId()

        now = DateTime()

        # Remove all LockOwner local roles
        for user, roles in self.get_local_roles():
            if self.LockOwnerRole in roles:
                new_roles = filter(lambda x: x != self.LockOwnerRole, list(roles))
                
                if new_roles:
                    self.manage_setLocalRoles(user, new_roles)
                else:
                    self.manage_delLocalRoles([user,],)
        
        if not value or value == '0' or value == 'False':
            value = False
        else:
            value = True
        
        # Apply locking
        if value:
            self.setLockUser(username)
            self.setLockDate(now)
            self.manage_addLocalRoles(userid, roles = [self.LockOwnerRole,])
        else:
            self.setUnlockUser(username)
            self.setUnlockDate(now)
        
        field = self.getField('is_locked').set(self, value, **kwargs)
        
        # Return status information
        return self.isLocked()
    
    security.declareProtected(LockEditPermission, 'toggleLock')
    def toggleLock(self):
        """
        toggleLock(self)

        Returns is locked
        """
        
        if self.isLocked():
            self.lock(False)
        else:
            self.lock(True)

        return self.isLocked()

    security.declareProtected(CMFCorePermissions.View, 'getLockUser')
    def getLockUser(self):
        """Returns the username of the member which has locked the content"""
        return getattr(self, '_locked_by', None)

    security.declareProtected(CMFCorePermissions.View, 'getLockDate')
    def getLockDate(self):
        """Returns the date when the content was locked"""
        return getattr(self, '_locked_date', None)
        
    security.declareProtected(CMFCorePermissions.View, 'getUnlockUser')
    def getUnlockUser(self):
        """Returns the username of the member which has unlocked the content"""
        return getattr(self, '_unlocked_by', None)

    security.declareProtected(CMFCorePermissions.View, 'getUnlockDate')
    def getUnlockDate(self):
        """Returns the date when the content was unlocked"""
        return getattr(self, '_unlocked_date', None)

    security.declarePrivate('setLockUser')
    def setLockUser(self, value):
        """Set the username of the member which has locked the content"""
        setattr(self, '_locked_by', value)

    security.declarePrivate('setLockDate')
    def setLockDate(self, value):
        """Set the date when the content was locked"""
        setattr(self, '_locked_date', value)
        
    security.declarePrivate('setUnlockUser')
    def setUnlockUser(self, value):
        """Set the username of the member which has unlocked the content"""
        setattr(self, '_unlocked_by', value)

    security.declarePrivate('setUnlockDate')
    def setUnlockDate(self, value):
        """Set the date when the content was unlocked"""
        setattr(self, '_unlocked_date', value)

    security.declareProtected(CMFCorePermissions.View, 'canToggleLock')
    def canToggleLock(self):
        """Convenience method that is used to decide whether the edit form is shown.

           If the content is locked, it can only be edited by the a person that
           can unlock it.
           If the content is not locked, the edit permission is sufficient.
        """
        
        mtool = getToolByName(self, 'portal_membership')
        
        # Return lock support
        if not self.hasLockSupport():
            return False
        
        # Member should have edit permission
        if not mtool.checkPermission(self.LockEditPermission, self):
            return False

        # Check if he has the lock permission
        if not self.isLocked():
            return True

        if mtool.checkPermission(self.LockOwnerPermission, self):
            return True
        else:
            return False

