/* parser.c
 * Main command line parser and parser utilities
 *
 * Yersinia
 * By David Barroso <tomac@wasahero.org> and Alfredo Andres <slay@wasahero.org>
 * Copyright 2005 Alfredo Andres and David Barroso
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef lint
static const char rcsid[] = 
"$Id: parser.c,v 1.10 2005/08/24 21:25:32 slay Exp $";
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _REENTRANT

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>

#ifdef HAVE_NETINET_IN_SYSTM_H
#include <netinet/in_systm.h>
#else
#ifdef HAVE_NETINET_IN_SYSTEM_H
#include <netinet/in_system.h>
#endif
#endif

#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif

#ifdef HAVE_BSTRING_H
#include <bstring.h>
#endif

#ifdef STDC_HEADERS
#include <stdlib.h>
#endif

#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif

#ifdef HAVE_GETOPT_LONG_ONLY
#include <getopt.h>
#else
#include "getopt.h"
#endif

#include <termios.h>

#include <stdarg.h>

#include <limits.h>

#include <ctype.h>

#include "parser.h"

static u_int8_t valid_chars[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890";


/*
 * Initial command line arguments parser.
 * Return -1 on error. Return 0 if Ok.
 * Use global protocol_defs
 */
int8_t
parser_initial(struct term_tty *tty, struct cl_args *cl_args, int argc, char **argv)
{
    int8_t i;

    for (i=1; i<(argc); i++)
    {    
       if (!strcmp(argv[i],"-h") || !strcmp(argv[i],"--help"))
       {
          parser_help();
          return -1;
       }
       else
       if (!strcmp(argv[i],"-V") || !strcmp(argv[i],"--Version"))
       {  
          printf("%s %s\n", PACKAGE, VERSION );
          return -1;
       }
       else
       if (!strcmp(argv[i],"-I") || !strcmp(argv[i],"--Interactive"))
       { 
#ifdef HAS_CURSES
          tty->interactive = 1;
#else
          printf("   Hmmm... it seems that you don't have ncurses support or Yersinia\n");
          printf("   has been compiled with --disable-ncurses...\n");
          printf("   Go and get it!!\n\n");
          return -1;
#endif
       }
       else
       if (!strcmp(argv[i],"-D") || !strcmp(argv[i],"--Daemon"))
       {
          tty->daemonize = 1;
       }
       else
       if (!strcmp(argv[i],"-d") || !strcmp(argv[i],"--debug"))
       {
          tty->debug = 1;
       }
       else
       if (!strcmp(argv[i], "-c") || !strcmp(argv[i], "--conffile"))
       {
           i++;
           strncpy(tty->config_file, argv[i], sizeof(tty->config_file));
       }
       else
       if (!strcmp(argv[i],"-l") || !strcmp(argv[i],"--logfile")) 
       {
           i++;
           if ((tty->log_file = fopen(argv[i], "a+")) == NULL) 
           {
               printf("%s: Error opening logfile %s\n", argv[0], argv[i]);
               /* If the logfile cannot be opened, the default is yersinia.log */
               if ((tty->log_file = fopen("yersinia.log", "a+")) == NULL) 
               {
                   printf("%s: Error opening logfile 'yersinia.log'\n", argv[0]);
                   return -1;
               }
           }
       }
       else
       if (argv[i][0] != '-')/* Ok, it is a protocol...*/
       {
           if (((cl_args->proto_index = protocol_proto2index(argv[i])) < 0)) /* Nonexistant protocol...*/
           {
              printf("%s: Unknown protocol %s!!\n", argv[0], argv[i]);
              return -1;
           }
           cl_args->count = argc-i;
           cl_args->argv_tmp = &argv[i];
           break;
       }
       else
       {
           printf("%s: Unknown option %s!!\n", argv[0], argv[i]);
           return -1;
       } 
    } /* for...*/
    
    if (!tty->log_file) {
        if ((tty->log_file = fopen("yersinia.log", "a+")) == NULL) {
            printf("%s: Error opening logfile 'yersinia.log'\n", argv[0]);
            return -1;
        }
    }

    setvbuf(tty->log_file, NULL, _IONBF, 0);

    return 0;
}


/*
 * Look if *number* of the characters
 * are digits.
 * Return -1 on error. Return 0 if Ok.
 */
int8_t 
parser_are_digits( int8_t *charac, int8_t number )
{
    while (number > 0)
    {
        if ( !isdigit((int)*charac) )
            return -1;
        charac++;
        number--;
    }
    return 0;
}


/*
 * Convert a string to lowercase
 * Return nothing.
 */
void
parser_str_tolower( char *charac )
{
    while (*charac)
    {
        *charac = (char)tolower((int)*charac);
        charac++;
    }
}


/*
 * Verify a MAC address...
 * Return -1 if error
 */
int8_t
parser_vrfy_mac( char *address, u_int8_t *destination )
{
    char *substr, *aux;
    u_int8_t data[]={0,0,0};
    u_int16_t i=0, j;

    substr = address;
	/* trim left spaces (can exist in the configuration file) */
//	while (*substr == ' ')
//		substr++;

	aux = substr;

    while ( (substr = strchr(substr, ':') ))
    {
        if (substr == aux)
            return -1;

        if ( (substr-aux) > (sizeof(data)-1))
            return -1; /* Overflowed? */

        for(j=0; j<(substr-aux);j++)
        {   
            if ( !isxdigit((int)(*(aux+j))) )
                return -1;
            data[j]=*(aux+j);
        }
        data[j]=0;
        if ( strtol((char *)data, (char **)NULL, 16) > 255 || 
                strtol((char *)data, (char **)NULL, 16) < 0 )
            return -1;
        destination[i]=strtol((char *)data, (char **)NULL, 16);
        i++;
        if (i==6)
            return -1;
        substr++;
        aux=substr;
    }

    if (!*aux || (i<5) )
        return -1;

    for(j=0; *aux && !isspace(*aux); j++,aux++)
    {   
        if ( !isxdigit((int)(*aux)) )
            return -1;
        data[j]=*aux;
    }

    data[j]=0;
    if ( strtol((char *)data, (char **)NULL, 16) > 255 || 
            strtol((char *)data, (char **)NULL, 16) < 0 )
        return -1;

    destination[i] = strtol((char *)data, (char **)NULL, 16);

    return 0;
}


/*
 * Verify a bridge id
 * Return 1 if error 
 */
int8_t
parser_vrfy_bridge_id( char *address, u_int8_t *destination )
{
    char *substr, *aux;
    u_int8_t data[]={0,0,0};
    u_int16_t i, j, k=0;

    aux = substr = address;

    for (i=0; i<2; i++)
    {
        for(j=0; j<2;j++)
        {   
            if ( !isxdigit((int)(*(aux+j))) )
                return -1;
            data[j]=*(aux+j);
        }
        data[j]=0;
        if ( strtol((char *)data, (char **)NULL, 16) > 255 || 
                strtol((char *)data, (char **)NULL, 16) < 0 )
            return -1;
        destination[k]=strtol((char *)data, (char **)NULL, 16);
        k++;
        aux+= 2*sizeof(u_char);
    }

    if (*aux != '.')
        return -1;

	aux++;

    for (i=0; i<6; i++)
    {
        for(j=0; j<2;j++)
        {   
            if ( !isxdigit((int)(*(aux+j))) )
                return -1;
            data[j]=*(aux+j);
        }
        data[j]=0;
        if ( strtol((char *)data, (char **)NULL, 16) > 255 || 
                strtol((char *)data, (char **)NULL, 16) < 0 )
            return -1;
        destination[k]=strtol((char *)data, (char **)NULL, 16);
        k++;
        aux+= 2*sizeof(u_char);
    }

    return 0;
}


/*
 * Help function. What do you want to do today?
 */
void
parser_help(void)
{
    printf("%s\n", bin_data);
    printf("\nUsage: %s [-hVIDd] [-l logfile] [-c conffile] protocol [protocol_options]\n", PACKAGE);
    printf("       -V   Program version.\n");
    printf("       -h   This help screen.\n");
    printf("       -I   Interactive mode (ncurses).\n");
    printf("       -D   Daemon mode.\n");
    printf("       -d   Debug.\n");
    printf("       -l logfile   Select logfile.\n");
    printf("       -c conffile  Select config file.\n");
    printf("  protocol   Can be one of the following: cdp, dhcp, 802.1q, dtp, hsrp, stp, vtp, isl.\n\n");         
    printf("Try '%s protocol -h' to see protocol_options help\n\n", PACKAGE);
    printf("Please, see the man page for a full list of options and many examples.\n");
    printf("Send your bugs & suggestions to the Yersinia developers <yersinia@wasahero.org>\n\n");  
}


/*
 *
 */
int8_t
parser_command2index(register const struct attack *lp, register int8_t v)
{
    int i=0;

    while (lp->s != NULL) 
    {
        if (lp->v == v)
            return (i);
        ++lp;
        ++i;
    }
    return (i);
}


int8_t
parser_get_formated_inet_address(u_int32_t in, char *inet, u_int16_t inet_len)
{
    char *p;
    u_int32_t aux_long;

    aux_long = htonl(in);
    p = (char *)&aux_long;
    if (snprintf(inet, inet_len, "%03d.%03d.%03d.%03d",
            (p[0] & 255), (p[1] & 255), (p[2] & 255), (p[3] & 255)) < 0)
        return -1;
    return 0;
}


int8_t
parser_get_inet_aton(char *inet, struct in_addr *in)
{
    char *stripped_ip, *tmp, dots=0;
    char st[4];
    char *save_pointer;
    u_int8_t i, j;

    if ( (strlen(inet) < 7) || (strlen(inet)>15) )
       return -1;
       
    if ( (inet[0] == '.') || (inet[strlen(inet)-1] == '.') ||
          (!isdigit(inet[0])) || (!isdigit(inet[strlen(inet)-1])) )
       return -1;
    
    for (i=1; i< (strlen(inet)-1); i++)
    {
       if (inet[i] == '.')
       {
          dots++;
          if (inet[i+1] == '.')
             return -1;
       }
       else
       {
          if (!isdigit(inet[i]))
             return -1;
       }
    }

    if (dots!=3)
       return -1;

    if ((stripped_ip = calloc(1, 16)) == NULL) {
        write_log(0, "Error in calloc");
        return -1;
    }

    memcpy((void *)stripped_ip, (void *)inet, strlen(inet));

    j = 0;

    tmp = strtok_r(stripped_ip, ".", &save_pointer);

    for (i=0; i < 4; i++) {
        if (tmp) {
            sprintf(st, "%d", atoi(tmp));
            memcpy((void *)(stripped_ip + j), (void *)st, strlen(st));
            
            j += strlen(st);

            if (i < 3) {
                stripped_ip[j] = '.';
                j++;
            } else
                stripped_ip[j] = '\0';
            tmp = strtok_r(NULL, ".", &save_pointer);
        }
    }

    if (inet_aton(stripped_ip, in) <= 0) {
        free(stripped_ip);
        write_log(0, "Error in inet_aton\n");
        return -1;
    }

    free(stripped_ip);

    return 0;
}


int8_t
parser_get_random_string(u_int8_t *string, u_int8_t len)
{
    u_int8_t total, j;
    struct timeval tv;
#ifdef HAVE_RAND_R
    u_int32_t i;
#endif

    total = 0;

    if (!string)
        return -1;

    while (total < len-1) {
	if (gettimeofday(&tv, NULL) < 0) {
		thread_error("Error in gettimeofday", errno);
		return -1;
	}

#ifdef HAVE_RAND_R
       i = (u_int32_t)tv.tv_usec;
       j = rand_r(&i);
#else
       if (pthread_mutex_lock(&terms->mutex_rand) != 0)
       {
           thread_error("get_random_string pthread_mutex_lock()",errno);
           return -1;
       }

       j=rand();

       if (pthread_mutex_unlock(&terms->mutex_rand) != 0)
       {
           thread_error("get_random_string pthread_mutex_unlock()",errno);
           return -1;
       }
#endif
       j = j % (sizeof(valid_chars) - 1);
       string[total] = valid_chars[j];
       total++;
   }
    string[len-1] = '\0';

    return 0;
}


int8_t
parser_get_random_int(u_int8_t max)
{
    u_int8_t j;
    struct timeval tv;
#ifdef HAVE_RAND_R
    u_int32_t i; 
#endif

    if (gettimeofday(&tv, NULL) < 0) {
        thread_error("Error in gettimeofday", errno);
        return -1;
    }

#ifdef HAVE_RAND_R
    i = (u_int32_t)tv.tv_usec;
    j = rand_r(&i);
#else
    if (pthread_mutex_lock(&terms->mutex_rand) != 0)
    {
       thread_error("get_random_int pthread_mutex_lock()",errno);
       return -1;
    }

    j=rand();

    if (pthread_mutex_unlock(&terms->mutex_rand) != 0)
    {
       thread_error("get_random_int pthread_mutex_unlock()",errno);
       return -1;
    }
#endif
    j = j % max;

    return j;
}


int8_t
parser_read_config_file(struct term_tty *tty)
{
	FILE *file;
	char buffer[BUFSIZ], *ptr, *temp, *ptr2, tmp[3];
	u_int8_t state, i, j, len, aux[32], *bytes;
	int16_t proto, field;
	u_int32_t aux_int;
	struct in_addr in_addr;

	state = 0;
	proto = -1;
	field = -1;

	if ((file = fopen(tty->config_file, "r")) == NULL) {
		write_log(1, "Error opening configuration file %s\n", tty->config_file);
		return -1;
    }

	while(fgets(buffer, BUFSIZ,file) != NULL)
	{
		/* trim any initial space */
		ptr = buffer;
		while((*ptr == ' ') || (*ptr == '\t'))
			ptr++;

		if ((*ptr == '\n') || (*ptr == '#')) /* Move to the next line */
			continue;

		switch(state)
		{
			case 0:
			/* State 0. Tokens allowed: global, protocol */
				if (strncmp(ptr, "<global>", 8) == 0)
				{
					/* Global options */
					proto = 666;
					state = 1;
				}
				else if (strncmp(ptr, "<protocol", 9) == 0)
				{
					/* Protocol options */
					proto = -1;
					for (i = 0; i < MAX_PROTOCOLS; i++)
						if (strncasecmp(protocols[i].name, (ptr + 10), strlen(protocols[i].name)) == 0)
							proto = protocols[i].proto;

					if (proto < 0) {
						write_log(1, "Error when parsing file %s, there is no %s protocol!!\n", tty->config_file, (ptr + 10));
						return -1;
					}
					state = 2;
			    }
			break;
			case 1:
			/* State 1: global options */
			    if (strncmp(ptr, "</global>", 9) == 0) {
					state = 0;
				    continue;
				}

				if (strncmp(ptr, "mac_spoofing", 12) == 0)
				{
					ptr += strlen("mac_spoofing");
					if ((ptr = strchr(ptr, '=')) == NULL) {
						write_log(1, "Parse error: missing '=' (%s)\n", buffer);
						return -1;
					}

					tty->mac_spoofing = atoi(ptr + 1);
				}
				else if (strncmp(ptr, "splash", 6) == 0)
				{
					ptr += strlen("splash");
					if ((ptr = strchr(ptr, '=')) == NULL) {
						write_log(1, "Parse error: missing '=' (%s)\n", buffer);
						return -1;
					}

					tty->splash = atoi(ptr + 1);
				}
				else if (strncmp(ptr, "username", 6) == 0)
				{
					ptr += strlen("username");
					if ((ptr = strchr(ptr, '=')) == NULL) {
						write_log(1, "Parse error: missing '=' (%s)\n", buffer);
						return -1;
					}

					ptr++;

     			/* Trim all the spaces */
     				while(*ptr == ' ')
	    				ptr++;

					/* Trim the \n */
					ptr2 = ptr;
					while (*ptr2 != '\0')
						if (*ptr2 == '\n')
							*ptr2 = '\0';
					    else
							ptr2++;

					strncpy(tty->username, ptr, MAX_USERNAME);
				}
				else if (strncmp(ptr, "password", 6) == 0)
				{
					ptr += strlen("password");
					if ((ptr = strchr(ptr, '=')) == NULL) {
						write_log(1, "Parse error: missing '=' (%s)\n", buffer);
						return -1;
					}

					ptr++;
     			/* Trim all the spaces */
     				while(*ptr == ' ')
	    				ptr++;

					/* Trim the \n */
					ptr2 = ptr;
					while (*ptr2 != '\0')
						if (*ptr2 == '\n')
							*ptr2 = '\0';
					    else
							ptr2++;

					strncpy(tty->password, (ptr), MAX_PASSWORD);
				}
				else if (strncmp(ptr, "enable", 6) == 0)
				{
					ptr += strlen("enable");
					if ((ptr = strchr(ptr, '=')) == NULL) {
						write_log(1, "Parse error: missing '=' (%s)\n", buffer);
						return -1;
					}

					ptr++;
                                /* Trim all the spaces */
     				while(*ptr == ' ')
	    				ptr++;

					/* Trim the \n */
					ptr2 = ptr;
					while (*ptr2 != '\0')
						if (*ptr2 == '\n')
							*ptr2 = '\0';
					    else
							ptr2++;

					strncpy(tty->e_password, (ptr), MAX_PASSWORD);
				}
				else if (strncmp(ptr, "hosts", 5) == 0)
				{
				    u_int8_t gotit=0;
                                    ptr += strlen("hosts");
                                    if ((ptr = strchr(ptr, '=')) == NULL) {
                                            write_log(1, "Parse error: missing '=' (%s)\n", buffer);
                                            return -1;
                                    }

                                    ptr++;
                                    while(!gotit)
                                    {
                                        /* Trim all the spaces */
                                        while((*ptr == ' ')||(*ptr == '\t'))
                                            ptr++;

                                        ptr2 = ptr;
                                        while ((*ptr != '\0') && (*ptr != ' ') && (*ptr != '\t'))
                                        {
                                           if (*ptr == '\n')
                                           {
                                              *ptr = '\0';
                                              gotit=1;
                                           }
                                           else
                                              ptr++;
                                        }
                                        if ((*ptr == ' ') || (*ptr == '\t'))
                                        {
                                           *ptr = '\0';
                                           ptr++;
                                        }
                                        if (ptr != ptr2)
                                           if (parser_vrfy_ip2filter(ptr2,tty) < 0 )
                                           {
                                               write_log(1,"Parse error: error parsing IP address '%s'\n",ptr2);
                                               parser_free_ip2filter(tty->ip_filter);
                                               return -1;
                                           }
                                    }
				}
				else 
				{
                                    write_log(1, "Parse error: %s is not a global option\n", ptr);
                                    return -1;
				}
			break;
			
			case 2:
			/* State 2: protocol options */
			    if (strncmp(ptr, "</protocol>", 11) == 0) {
					state = 0;
				    continue;
				}

				/* Now find any possible field */
				i = 0;
				while(protocols[proto].fields[i].name)
				{
					if (strncasecmp(protocols[proto].fields[i].name, ptr, strlen(protocols[proto].fields[i].name)) == 0)
					{
						ptr += strlen(protocols[proto].fields[i].name);
						if ((ptr = strchr(ptr, '=')) == NULL) {
							write_log(1, "Parse error: missing '=' (%s)\n", buffer);
							return -1;
						}
						ptr++;
						/* Trim all the spaces */
						while(*ptr == ' ')
							ptr++;

						/* Trim the \n */
						if (ptr[strlen(ptr) - 1] == '\n')
						    ptr[strlen(ptr) - 1] = '\0';

						switch(protocols[proto].fields[i].type)
						{
							case FIELD_MAC:
								if (parser_vrfy_mac(ptr, aux) < 0) {
									write_log(1, "Parse error: error parsing MAC Address %s\n", ptr);
									return -1;
								}

								if (protocols[proto].update_field)
									if ((*protocols[proto].update_field)(protocols[proto].fields[i].v, NULL, aux) < 0) {
										write_log(0, "Error in update_field\n");
										return -1;
									}
							break;
							case FIELD_BRIDGEID:
								if (parser_vrfy_bridge_id(ptr, aux) < 0) {
									write_log(1, "Parse error: error parsing Bridge ID %s\n", ptr);
									return -1;
								}

								if (protocols[proto].update_field)
									if ((*protocols[proto].update_field)(protocols[proto].fields[i].v, NULL, aux) < 0) {
										write_log(0, "Error in update_field\n");
										return -1;
									}
							break;
							case FIELD_DEC:
							case FIELD_HEX:
							    if (strncmp(ptr, "0x", 2) == 0)
									aux_int = strtol(ptr, NULL, 16);
								else if (isdigit(*ptr))
									aux_int = strtol(ptr, NULL, 10);
								else {
									write_log(1, "Parse error: %s must be a number in hex, octal or decimal\n", ptr);
									return -1;
								}
								if (protocols[proto].update_field)
									if ((*protocols[proto].update_field)(protocols[proto].fields[i].v, NULL, &aux_int) < 0) {
										write_log(0, "Error in update_field\n");
										return -1;
									}
							break;
							case FIELD_IP:
							    if (parser_get_inet_aton(ptr, &in_addr) < 0) {
									write_log(1, "Parse error: error parsing IP %s\n", ptr);
									return -1;
								}
								aux_int = htonl(in_addr.s_addr);
								if (protocols[proto].update_field)
									if ((*protocols[proto].update_field)(protocols[proto].fields[i].v, NULL, &aux_int) < 0) {
										write_log(0, "Error in update_field\n");
										return -1;
									}
							break;
							case FIELD_STR:
								if (protocols[proto].update_field)
									if ((*protocols[proto].update_field)(protocols[proto].fields[i].v, NULL, ptr) < 0) {
										write_log(0, "Error in update_field\n");
										return -1;
									}
                            break;
							case FIELD_BYTES:
								len = strlen(ptr) - 1;
								bytes = (u_int8_t *)calloc(1, (len % 2) ? len/2 + 1 : len/2);
								temp = ptr;
								for (j=0; j < len/2; j++) {
									memcpy((void *)tmp, (void *)temp, 2);
									tmp[2] = '\0';
									bytes[j] = strtol(tmp, NULL, 16);
								    temp+=2;
								}
								if (len % 2)
									bytes[j+1] = strtol(temp, NULL, 16);

								if (protocols[proto].update_field)
									if ((*protocols[proto].update_field)(protocols[proto].fields[i].v, NULL, bytes) < 0) {
										write_log(0, "Error in update_field\n");
										return -1;
									}
								free(bytes);
							break;
							default:
							break;
						}

						/* jump out */
						break;
					}
					i++;
				}

				/* No success */
				if (protocols[proto].fields[i].name == NULL) {
					write_log(1, "Parse error: there is no %s field in %s protocol\n", ptr, protocols[proto].name);
					return -1;
				}
			break;
		}
	}

	if (fclose(file) != 0) {
		write_log(1, "Error closing configuration file %s\n", tty->config_file);
		return -1;
	}

	return 0;
}

int8_t
parser_write_config_file(struct term_tty *tty)
{
    FILE *file;
    u_int8_t i, j;
    char **values;
    struct filter *cursor;
    
    if ((file = fopen(tty->config_file, "w+")) == NULL) {
        write_log(0, "Error opening configuration file %s\n", tty->config_file);
        return -1;
    }

    fputs("# $Id: parser.c,v 1.10 2005/08/24 21:25:32 slay Exp $\n", file);
    fputs("#\n", file);
    fputs("# Yersinia configuration file example\n", file);
    fputs("#\n", file);
    fputs("# Please read the README and the man page before complaining\n", file);
    fputs("\n", file);

    fputs("# Global options\n", file);
    fputs("<global>\n", file);
    fputs("# MAC Spoofing\n", file);

    fputs("mac_spoofing = 1\n", file);
    fputs("# Active interfaces\n", file);
    fputs("#interfaces = eth0, eth1\n", file);

    fputs("# Hosts allowed to connect to the network daemon\n", file);
    fputs("# Examples: www.microsoft.com 192.168.1.0/24 10.31-128.*.13 100.200.*.* 2-20.*.*.10-11\n",file);
    cursor = tty->ip_filter;
    if (!cursor)
        fputs("hosts = localhost", file);
    else
        fputs("hosts =", file);
    while(cursor)
    {
        fputs(" ",file);
        fputs(cursor->expression,file);
        cursor = cursor->next;
    }
    fputs("\n",file);
    fputs("# Propaganda. It's cool, so please, don't disable it!! :-P\n", file);
    fputs("splash = 1\n", file);
    fputs("# Username for the admin mode\n", file);
    fputs("username = ", file);
    fputs(tty->username, file);
    fputs("\n", file);
    fputs("# Password for the admin mode\n", file);
    fputs("password = ", file);
    fputs(tty->password, file);
    fputs("\n", file);
    fputs("# Enable password for the admin mode\n", file);
    fputs("enable = ", file);
    fputs(tty->e_password, file);
    fputs("\n", file);

    fputs("</global>\n\n", file);

    for (i = 0; i < MAX_PROTOCOLS; i++)
    {
        j = 0;
        fputs("<protocol ", file);
        fputs(protocols[i].name, file);
        fputs(">\n", file);

        if (protocols[i].get_printable_store == NULL) {
                write_log(0, "printable_store in protocol %d is NULL\n", i);
        }

        if (protocols[i].get_printable_store) {
            if ((values = (*protocols[i].get_printable_store)(NULL)) == NULL) {
                    write_log(0, "Error in get_printable_store\n");
                    return -1;
            }

            while((protocols[i].fields[j].name) && (protocols[i].fields[j].type != FIELD_TLV))
            {
                fputs(protocols[i].fields[j].name, file);
                fputs(" = ", file);
                switch(protocols[i].fields[j].type)
                {
                        case FIELD_HEX:
                                fputs("0x", file);
                        break;
                        default:
                        break;
                }

                fputs(values[j], file);
                fputs("\n", file);
                j++;
            }
            fputs("</protocol>\n\n", file);

            j = 0;
            if (values)
                    while(values[j]) {
                            free(values[j]);
                            j++;
                    }

            free(values);
        } else
                fputs("</protocol>\n\n", file);
    }

    if (fclose(file) != 0) {
            write_log(1, "Error closing configuration file %s\n", tty->config_file);
            return -1;
    }

    return 0;
}


/* ============================================ */
/* Mask and convert digit to hex representation */
/* Output range is 0..9 and a..f only           */
int hexify(unsigned int value)
{
   int result;

   result = (value & 0x0f) + '0';
   if (result > '9')
      result = result + ('a' - '0' - 10);
   return result;

} /* hexify */

/* ========================================= */
/* convert number to string in various bases */
/* 2 <= base <= 16, controlled by hexify()   */
/* Output string has ls digit first.         */
void parser_basedisplay(u_int8_t number, u_int8_t base,
                 char * string, size_t maxlgh)
{
   /* assert (string[maxlgh]) is valid storage */
   if (!maxlgh) {
      *string = '\0';
      return;
   }
   else {
      *string = hexify(number % base);
      if (!number) *string = '\0';
      else {
         parser_basedisplay(number / base, base, &string[1], maxlgh - 1);
      }
   }

} /* basedisplay */

/* ======================= */
/* reverse string in place */
void revstring(char * string)
{
   char * last, temp;

   last = string + strlen(string); /* points to '\0' */
   while (last-- > string) {
      temp = *string; *string++ = *last; *last = temp;
   }

} /* revstring */ 

/*
 * Verify an expression in order to add it to
 * the linked list of ip filters
 * All data will be stored in Host Byte Order
 * Expressions allowed are:
 * 1- CIDR notation: 10.20.30.0/24
 * 2- Wildcard use: 10.20.30.*
 * 3- Range use: 10.20.30.0-255
 * 4- Range with wildcard: 10.20.20-30.*
 * 5- IP as usual: 10.20.30.40
 * 6- Hostname: www.bloblob.xx
 *
 * Return 0 on success. -1 on error
 */
int8_t
parser_vrfy_ip2filter(char *expr, struct term_tty *tty)
{
   union bla { u_int32_t all;
               u_int8_t byte[4];
   };
   union bla numbegin, numend;  
   char *expr2, *str[4], *ip, *aux, *aux2;
   u_int8_t bits, dots, name, wildcard;
   u_int32_t auxl, beginl, endl,i;
   
   numbegin.all = 0;
   numend.all = 0;
   
   expr2 = strdup(expr);
   if (expr2 == NULL)
   {
      write_log(1,"strdup error!!\n");
      return -1;
   }
   
   ip = expr2;
   aux = strchr(ip, '/');

   if (aux == NULL) /* No CIDR expression */
   {
      /* How many dots? */
      dots = 0;
      aux2 = ip;
      name = wildcard = 0;
      for (i=1; i< (strlen(aux2)-1); i++)
      {
         if (aux2[i] == '.')
         {
            dots++;
            if (aux2[i+1] == '.')
            {
               free(expr2);
               return -1;
            }
         }
         else
         {
            if (!isdigit(aux2[i]))
            {
               if ((aux2[i]=='*') || (aux2[i]=='-'))
                  wildcard = 1;
               else
               {
                  name=1;
                  break;
               }
            }
         }
      }
      
      if (name || (!name && !wildcard)) /* We have a name or a IP address */
      {
         struct hostent *namehost;
         namehost = gethostbyname(aux2);
         if (namehost == NULL)
         {
            write_log(1,"Parse error: Unable to resolve '%s'!! Anyway we'll go on with the rest of the addresses...\n",aux2);         
            free(expr2);
            return 0;
         }
         memcpy((char *)&auxl, namehost->h_addr_list[0], sizeof(struct in_addr));
         auxl = ntohl(auxl);
         beginl = endl = auxl;
      }
      else /* We have an expression with '*' or '-'*/
      {
         char *aux3=NULL, *aux4=NULL;
         if (dots!=3)
         { 
            free(expr2);
            return -1;
         }

         for(i=0; i< 4; i++)
         {
            aux3 = strchr(aux2,'.');
            if (aux3!=NULL)
              *aux3 = '\0';
            str[i] = aux2;
            if ( (strchr(str[i],'*') != NULL) && (strchr(str[i],'-') != NULL) )
            { 
               free(expr2);
               return -1;
            }
            if ( (aux4 = strchr(str[i],'*')) == NULL)
            {
               if ( (aux4 = strchr(str[i],'-')) == NULL)
               {
                  if (strlen(str[i]) > 3)
                  {
                     free(expr2);
                     return -1;
                  }
                  numbegin.byte[i] = atoi(str[i]);
                  numend.byte[i] = numbegin.byte[i];
               }
               else /* We have '-' token... */
               {
                  char *aux5=NULL;
                  if ( strlen(str[i]) > 7)
                  {
                     free(expr2);
                     return -1;
                  }
                  aux5 = str[i];
                  if (aux5 == aux4) /* Value = '-xxx' or '-' */
                  {
                     numbegin.byte[i] = 0;
                     if (strlen(aux5) == 1)
                        numend.byte[i] = 255;
                     else
                     {
                        aux4++;
                        numend.byte[i]=atoi(aux4);
                     }
                  }
                  else /* Value = 'x-' or 'x-x' */
                  {
                     if ( *(aux5+strlen(str[i])-1) == '-') /* Case 'x-' */
                     {
                        *aux4 = '\0';
                        numbegin.byte[i] = atoi(aux5);
                        numend.byte[i] = 255;
                     }
                     else /* Case 'x-x' */
                     {
                        *aux4 = '\0'; aux4++;
                        numbegin.byte[i] = atoi(aux5);
                        numend.byte[i] = atoi(aux4);
                     }
                  }
               }
            }
            else /* We have '*' token */
            {
               if (strlen(str[i]) > 1)
               {
                  free(expr2);
                  return -1;
               }
               numbegin.byte[i] = 0;
               numend.byte[i] = 255;
            }
            aux2 = ++aux3;
         } /* for...*/

         for(i=0;i<4;i++)
         {
            if (numbegin.byte[i] > numend.byte[i])
            {
               free(expr2);
               return -1;
            }
         }
         beginl = ntohl(numbegin.all);
         endl   = ntohl(numend.all);  
      }
   }
   else /* We have a CIDR expression */
   {
      *aux = '\0';
      aux++;
      if ((aux=='\0') || (strlen(aux)>2))
      {
         free(expr2);
         return -1;
      }
      bits = atoi(aux);
      if (bits>32)
      {
         free(expr2);
         return -1;
      }

      if (!inet_aton(ip, (struct in_addr *)&auxl))
      {
         free(expr2);
         return -1;
      }
      auxl   = ntohl(auxl);
      beginl = auxl & (unsigned long) (0 - (1<<(32 - bits)));
      endl   = auxl | (unsigned long) ((1<<(32 - bits)) - 1);
   }

/*   write_log(1,"IP Begin = %08X %s\n",beginl,inet_ntoa( (*((struct in_addr *)&beginl))   )  );
   write_log(1,"IP End   = %08X %s\n",endl,inet_ntoa( (*((struct in_addr *)&endl))   )  );
*/
   free(expr2);
   
   if (beginl > endl)
      return -1;
   
   if (parser_add_ip2filter(beginl,endl,tty,expr) < 0)
      return -1;
   
   return 0;
}

/*
 * Add an IPv4 range to the linked list of ip ranges
 * All data will be stored in Host Byte Order
 *
 * Return 0 on success. -1 on error
 */
int8_t
parser_add_ip2filter(u_int32_t begin, u_int32_t end, struct term_tty *tty, char *expr)
{
   struct filter *new, *cursor=NULL, *last=NULL;
   char *expr2=NULL;

   new = (struct filter *)calloc(1,sizeof(struct filter));
   if (new == NULL)
      return -1;
   
   if ( (expr2=strdup(expr)) == NULL)
   {
      free(new);
      return -1;
   }
   
   new->expression = expr2;
   new->begin = begin;
   new->end = end;
   new->next = NULL;
   
   if (!tty->ip_filter)
   {
      tty->ip_filter = new;
      return 0;
   }

   cursor = tty->ip_filter;
   
   while(cursor)
   {
      last = cursor;
      cursor = cursor->next;
   }

   last->next = new;
 
   return 0;
}

void 
parser_free_ip2filter(struct filter *ipfilter)
{
   struct filter *cursor, *aux;
   cursor = ipfilter;

   while(cursor)
   {
      aux = cursor->next;
      free(cursor->expression);
      free(cursor);
      cursor = aux;
   }
}

