/*
 * Copyright 2004-2006 Luc Verhaegen.
 * Copyright 1998-2003 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2003 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*
 * Crappy panel code i haven't been able to clean up yet.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#endif

#include "via_vgahw.h"
#include "via_id.h"
#include "via_mode.h"

/* Kill me. */
#include "via_panel.h"


/*
 * Output->Private
 */
struct ViaPanelOutputPrivate {
    /* private use */
    Bool Present; /* only used until init is over */

    int  X;
    int  Y;
    
    /* To enforce Native resolution */
    int  HTotalMax;
    int  HTotalMin;
    int  HSyncMax;
    int  HSyncMin;

    int  VTotalMax;
    int  VTotalMin;
    int  VSyncMax;
    int  VSyncMin;

    /* private options */
    Bool  Center;
    CARD8  BusWidth; /* Digital Output Bus Width */
};

/*
 *
 */
static void
ViaPanelPrivateDestroy(struct ViaOutput *Output)
{
    xfree(Output->Private);

    /* So we won't try to destroy this twice */
    Output->PrivateDestroy = NULL;
}

/*
 *
 */
static struct ViaPanelOutputPrivate *
ViaPanelPrivateCreate(struct ViaOutput *Output)
{
    VIAFUNC(Output->scrnIndex);

    Output->PrivSize = sizeof(struct ViaPanelOutputPrivate);
    Output->Private =  xnfcalloc(1, Output->PrivSize);
    memset(Output->Private, 0, Output->PrivSize);

    Output->PrivateDestroy = ViaPanelPrivateDestroy;

    return Output->Private;
}

/*
 *
 * Option handling.
 *
 */
enum ViaPanelOpts {
    OPTION_BUSWIDTH,
    OPTION_CENTER,
    OPTION_FORCEPANEL,
    OPTION_PANELSIZE
};

static OptionInfoRec ViaPanelOptions[] =
{
    {OPTION_BUSWIDTH,	 "BusWidth",  	OPTV_ANYSTR,  {0}, FALSE},
    {OPTION_CENTER,      "Center",      OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_FORCEPANEL,  "ForcePanel",  OPTV_BOOLEAN, {0}, FALSE}, /* last resort - don't doc */
    {OPTION_PANELSIZE,   "PanelSize",   OPTV_ANYSTR,  {0}, FALSE},
    {-1,                 NULL,          OPTV_NONE,    {0}, FALSE}
};

/*
 * Stop the user from telling us lies.
 */
struct {
    int   X;
    int   Y;
    CARD8 Scratch;
} ViaNativePanelSizes[] = {
    { 640,  480, VIA_PANEL6X4  },
    { 800,  600, VIA_PANEL8X6  },
    {1024,  768, VIA_PANEL10X7 },
    {1280,  768, VIA_PANEL12X7 },
    {1280, 1024, VIA_PANEL12X10},
    {1400, 1050, VIA_PANEL14X10},
    {1600, 1200, VIA_PANEL16X12},
    {0, 0, VIA_PANEL_INVALID}
};


/*
 *
 */
static OptionInfoPtr
ViaPanelGetOptions(ScrnInfoPtr pScrn, struct ViaPanelOutputPrivate *Private)
{
    OptionInfoPtr  Options;
    char  *s = NULL;
    
    Options = xnfalloc(sizeof(ViaPanelOptions));
    memcpy(Options, ViaPanelOptions, sizeof(ViaPanelOptions));
    
    xf86ProcessOptions(pScrn->scrnIndex, pScrn->options, Options);

    /* Digital Output Bus Width Option */
    Private->BusWidth = VIA_DI_12BIT;
    if ((s = xf86GetOptValString(Options, OPTION_BUSWIDTH))) {
        if (!xf86NameCmp(s, "12BIT")) {
            Private->BusWidth = VIA_DI_12BIT;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
                       "Digital Output Bus Width is 12BIT\n");
        } else if (!xf86NameCmp(s, "24BIT")) {
            Private->BusWidth = VIA_DI_24BIT;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG,
                       "Digital Output Bus Width is 24BIT\n");
        }
    }

    if (VIAPTR(pScrn)->Chipset == VT3108) {
        /* primary can't do scaling on its own */
        Private->Center = TRUE;
    } else { /* pretty useless */
        /* LCD Center/Expend Option */
        if (xf86ReturnOptValBool(Options, OPTION_CENTER, FALSE)) {
            Private->Center = TRUE;
            xf86DrvMsg(pScrn->scrnIndex, X_CONFIG, "DVI Center is On\n");
        } else
            Private->Center = FALSE;
    }

    /* Force the use of the Panel? */
    if (xf86ReturnOptValBool(Options, OPTION_FORCEPANEL, FALSE)) {
        xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Enabling panel from config.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "This is a last resort option only.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "Report PCI IDs to http://unichrome.sf.net/ ASAP.\n");
        Private->Present = TRUE;
    } else
        Private->Present = FALSE;

    /* Panel Size Option */
    if ((s = xf86GetOptValString(Options, OPTION_PANELSIZE))) {
        int X, Y;
        if (sscanf(s, "%4dx%4d", &X, &Y) == 2) {
            int i;
            
            for (i = 0; ViaNativePanelSizes[i].X; i++) {
                if ((X == ViaNativePanelSizes[i].X) &&
                    (Y == ViaNativePanelSizes[i].Y)) {
                    Private->X = X;
                    Private->Y = Y;
                    break;
                }
            }

            if (!ViaNativePanelSizes[i].X) {
                xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Unknown or Unhandled"
                           " Native Panel Resolution: %dx%d\n", X, Y);
                xf86DrvMsg(pScrn->scrnIndex, X_INFO,
                           "If this is the actual resolution of the panel, "
                           "contact the driver author.\n");
            }
        } else {
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                       "Failed to parse PanelSize Option: %s\n", s);
        }
    }

    return Options;
}


/*
 *
 */
static void
ViaPanelNativeResolution(ScrnInfoPtr pScrn,
                         struct ViaPanelOutputPrivate *Private)
{
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    /* This could have been provided by config */
    if (!Private->X || !Private->Y) {
        int i;

        for (i = 0; ViaNativePanelSizes[i].X; i++) {
            if (ViaNativePanelSizes[i].Scratch == pVia->Scratch->PanelSize) {
                Private->X = ViaNativePanelSizes[i].X;
                Private->Y = ViaNativePanelSizes[i].Y;
                break;
            }
        }

        if (!ViaNativePanelSizes[i].X) {
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Unhandled PanelSize"
                       "(%d): Contact the driver author ASAP.\n",
                       __func__, pVia->Scratch->PanelSize);
            Private->X = 1024;
            Private->Y = 768; 
        }
    }

    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Panel Native Resolution is %dx%d.\n",
               Private->X, Private->Y);
}


/*
 * Depends on Output Name, Modes and Private being initialised already.
 */
static MonPtr
ViaPanelMonitor(struct ViaOutput *Output, DisplayModePtr Modes)
{
    struct ViaPanelOutputPrivate *Private = Output->Private;
    MonPtr Monitor;
    int i;

    VIAFUNC(Output->scrnIndex);

    Monitor = xnfcalloc(1, sizeof(MonRec));
    memset(Monitor, 0, sizeof(MonRec));

    /* Add all modes smaller than native */
    for (i = 0; Modes[i].name; i++)
        if ((Private->X >= Modes[i].HDisplay) &&
            (Private->Y >= Modes[i].VDisplay)) {
            ViaModesCopyAdd(Monitor, &Modes[i]);
        }

    if (!Monitor->Modes) {
        xf86DrvMsg(Output->scrnIndex, X_WARNING,
                   "%s: No matching mode found.\n", __func__);
        xfree(Monitor);
        return NULL;
    }

    Output->ModesExclusive = TRUE;

    /* set up strings */
    Monitor->id = xnfstrdup("Panel");
    Monitor->vendor = xnfstrdup("VIA");
    Monitor->model = xnfstrdup("TTL Panel");

    /* Set up ranges - Very artificial. */
    Monitor->nHsync = 1;
    Monitor->hsync[0].lo = 1024.0;
    Monitor->hsync[0].hi = 0.0;

    Monitor->nVrefresh = 1;
    Monitor->vrefresh[0].lo = 1024.0;
    Monitor->vrefresh[0].hi = 0.0;

    {
        DisplayModePtr Mode;
        float tmp;
        
        Mode = Monitor->Modes;

        while (Mode) {
            tmp = ((float) Mode->Clock) / Mode->HTotal;

            if (tmp < Monitor->hsync[0].lo)
                Monitor->hsync[0].lo = tmp;
            if (tmp > Monitor->hsync[0].hi)
                Monitor->hsync[0].hi = tmp;

            tmp = (Mode->Clock * 1000.0) / Mode->HTotal / Mode->VTotal;
            
            if (tmp < Monitor->vrefresh[0].lo)
                Monitor->vrefresh[0].lo = tmp;
            if (tmp > Monitor->vrefresh[0].hi)
                Monitor->vrefresh[0].hi = tmp;

            Mode = Mode->next;
        }
    }

#ifdef MONREC_HAS_REDUCED
    /* Our code doesn't care about free modes, but what the heck. */
    Monitor->reducedblanking = TRUE;
#endif

    return Monitor;
}

/*
 *
 * Private->Index is the index to lcdTable.
 *
 */
static ModeStatus
ViaPanelModeValid(struct ViaOutput *Output, DisplayModePtr mode)
{
    struct ViaPanelOutputPrivate *Private = Output->Private;
    int i;
    
    VIAFUNC(Output->scrnIndex);

    if ((mode->PrivSize != sizeof(struct ViaModePriv)) ||
	(mode->Private != (void *)&ViaPanelPrivate)){
	xf86DrvMsg(Output->scrnIndex, X_INFO, "%s: Not a mode defined by this "
                   "drivers panel code.\n", __func__);
	return MODE_BAD;
    }

    for (i = 0; i < VIA_PANEL_COUNT; i++)
	if ((lcdTable[i].X == Private->X) &&
            (lcdTable[i].Y == Private->Y)) {
	    int j;

            if ((Private->X == mode->CrtcHDisplay) && 
                (Private->Y == mode->CrtcVDisplay))
                return MODE_OK;
            else if (Private->Center) {
                for (j = 0; j < VIA_PANEL_COUNT_MAIN; j++)
                    if (lcdTable[i].CenterMain[j].HDisplay == mode->CrtcHDisplay)
                        return MODE_OK;
            } else {
                for (j = 0; j < VIA_PANEL_COUNT_MAIN; j++) {
                    if (lcdTable[i].ExpandMain[j].HDisplay == mode->CrtcHDisplay)
                        return MODE_OK;
                }
            }

            xf86DrvMsg(Output->scrnIndex, X_INFO, "%s: Unable to match given "
                       "mode with this PanelSize.\n", __func__);
            return MODE_BAD;
	}

    xf86DrvMsg(Output->scrnIndex, X_INFO, "%s: Unable to match Native"
               " Resolution with an lcdTable entry.\n", __func__);
    return MODE_BAD;
}


/*
 *
 */
static void
ViaPanelMode(struct ViaOutput *Output, DisplayModePtr mode)
{
    ScrnInfoPtr pScrn = xf86Screens[Output->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaPanelOutputPrivate *Private = Output->Private;
    VIALCDModeTableRec Table;
    int  i, HTotal;
    int PLLPrimary = 0, PLLSecondary = 0;
    CARD8 Offset;

    VIAFUNC(Output->scrnIndex);

    for (i = 0; i < VIA_PANEL_COUNT; i++)
	if ((lcdTable[i].X == Private->X) && (lcdTable[i].Y == Private->Y)) {
            Table = lcdTable[i];
            break;
        }

    if (i == VIA_PANEL_COUNT) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: unable to find lcdTable"
                   " index.\n", __func__);
        return;
    }

    if ((Private->X == 1280) && (Private->Y == 1024))
	hwp->writeCrtc(hwp, 0x89, 0x07);

    /* Patch mode for Non-Native. */
    if ((mode->CrtcHDisplay == Private->X) ||
        (mode->CrtcVDisplay == Private->Y)) {

        PLLPrimary = PLLSecondary = Table.PLL;

        /* Set up Secondary and Shadow timing. */
        Offset = 0x50;
        for (i = 0; i < VIA_PANEL_REG_COUNT_MAIN; i++, Offset++) {
            if (Offset == 0x60)
                Offset = 0x65;
            else if (Offset == 0x68)
                Offset = 0x6D;
            hwp->writeCrtc(hwp, Offset, Table.data[i]);
        }

        hwp->writeCrtc(hwp, 0x77, 0x00);
        hwp->writeCrtc(hwp, 0x78, 0x00);
        hwp->writeCrtc(hwp, 0x79, 0x00); 

    } else { /* Patch mode for Non-Native. */
	struct ViaPanelPatchMain *Patch;
        int j;

        if (Private->Center)
	    Patch = Table.CenterMain;
	else
            Patch = Table.ExpandMain;

	/* Set LCD Mode patch registers. */
	for (i = 0; i < VIA_PANEL_COUNT_MAIN; i++, Patch++)
	    if ((Patch->HDisplay == mode->CrtcHDisplay) &&
                (Patch->VDisplay == mode->CrtcVDisplay))
                break;
        
        if (i == VIA_PANEL_COUNT_MAIN) {
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: unable to find matching"
                       " secondary Patch index.\n", __func__);
            return;
        }
            
               
        PLLPrimary = Patch->PLLPrimary;
        PLLSecondary = Patch->PLLSecondary;

        /* Set up Secondary and Shadow timing. */
        Offset = 0x50;
        for (i = 0; i < VIA_PANEL_REG_COUNT_MAIN; i++, Offset++) {
            if (Offset == 0x60)
                Offset = 0x65;
            else if (Offset == 0x68)
                Offset = 0x6D;
            hwp->writeCrtc(hwp, Offset, Patch->data[i]);
        }

        if (Patch->Scaled) {
            int Scale;
            
            if (mode->CrtcHDisplay != Private->X) {
                Scale = ((float) ((mode->CrtcHDisplay - 1) * 1024) /
                         ((float) Private->X)) + 0.5;
                
                hwp->writeCrtc(hwp, 0x77, Scale & 0xFF);
                ViaCrtcMask(hwp, 0x79, (Scale & 0x300) >> 4, 0x30);
                
                ViaCrtcMask(hwp, 0x79, 0x02, 0x02);
            } else {
                hwp->writeCrtc(hwp, 0x77, 0x00);
                ViaCrtcMask(hwp, 0x79, 0x00, 0x30);
            }
            
            if (mode->CrtcVDisplay != Private->Y) {
                Scale = ((float) ((mode->CrtcVDisplay - 1) * 1024) /
                         ((float) Private->Y)) + 0.5;
                
                hwp->writeCrtc(hwp, 0x78, Scale & 0xFF);
                ViaCrtcMask(hwp, 0x79, (Scale & 0x300) >> 2, 0xC0);
                
                ViaCrtcMask(hwp, 0x79, 0x04, 0x04);
            } else {
                hwp->writeCrtc(hwp, 0x78, 0x00);
                ViaCrtcMask(hwp, 0x79, 0x00, 0xC0);
            }
            
            ViaCrtcMask(hwp, 0x79, 0x01, 0x01);
        } else {
            hwp->writeCrtc(hwp, 0x77, 0x00);
            hwp->writeCrtc(hwp, 0x78, 0x00);
            hwp->writeCrtc(hwp, 0x79, 0x00);
        }

	/* Set LCD Mode Patch registers when Primary. */
	if (!pVia->IsSecondary) {
            struct ViaPanelPatchPrimary *Primary;

            if (Private->Center)
                Primary = Table.CenterPrimary;
            else
                Primary = Table.ExpandPrimary;

	    for (i = 0; i < VIA_PANEL_COUNT_PRIMARY; i++, Primary++) {
                if ((Primary->HDisplay == mode->CrtcHDisplay) &&
                    (Primary->VDisplay == mode->CrtcVDisplay)) {
		    for (j = 0; j < Primary->numEntry; j++)
                        hwp->writeCrtc(hwp, Primary->offset[j], Primary->data[j]);
		    break;
		}
	    }
	}
    }

    /* LCD patch 3D5.02 */
    HTotal = hwp->readCrtc(hwp, 0x01);
    hwp->writeCrtc(hwp, 0x02, HTotal);

    /* Set up PLL */
    {
        CARD32 PLL;

        if (Private->BusWidth == VIA_DI_12BIT) {
            PLLPrimary *= 2;
            PLLSecondary *= 2;
        }

        PLL = ViaPLLGenerate(pScrn, PLLSecondary);
        ViaSetSecondaryDotclock(pScrn, PLL);

        if (!pVia->IsSecondary) {
            if (PLLPrimary == PLLSecondary)
                pVia->PanelClock = PLL;
            else
                pVia->PanelClock = ViaPLLGenerate(pScrn, PLLPrimary);
        }
    }

    /* Enable LCD */
    if (!pVia->IsSecondary) {
        /* Enable Simultaneous */
        if (Private->BusWidth == VIA_DI_12BIT) {
	    hwp->writeCrtc(hwp, 0x6B, 0xA8);

            if ((pVia->Chipset == VT3122) && VT3122_REV_IS_AX(pVia->ChipRev))
		hwp->writeCrtc(hwp, 0x93, 0xB1);
            else
		hwp->writeCrtc(hwp, 0x93, 0xAF);
        } else {
	    ViaCrtcMask(hwp, 0x6B, 0x08, 0x08);
	    hwp->writeCrtc(hwp, 0x93, 0x00);
        }
	hwp->writeCrtc(hwp, 0x6A, 0x48);
    } else {
        /* Enable SAMM */
        if (Private->BusWidth == VIA_DI_12BIT) {
	    ViaCrtcMask(hwp, 0x6B, 0x20, 0x20);
            if ((pVia->Chipset == VT3122) && VT3122_REV_IS_AX(pVia->ChipRev))
		hwp->writeCrtc(hwp, 0x93, 0xB1);
            else
		hwp->writeCrtc(hwp, 0x93, 0xAF);
        } else {
	    hwp->writeCrtc(hwp, 0x6B, 0x00);
	    hwp->writeCrtc(hwp, 0x93, 0x00);
        }
	hwp->writeCrtc(hwp, 0x6A, 0xC8);
    }

    Output->Power(Output, TRUE);
}


/*
 *
 */
static void
ViaPanelPower(struct ViaOutput *Output, Bool On)
{
    ScrnInfoPtr pScrn = xf86Screens[Output->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);

    ViaDebug(Output->scrnIndex, "%s: %s.\n", __func__, On ? "On" : "Off");

    /* Enable LCD */
    if (On)
	ViaCrtcMask(hwp, 0x6A, 0x08, 0x08);
    else
	ViaCrtcMask(hwp, 0x6A, 0x00, 0x08);

    usleep(1);

    if (pVia->Chipset == VT3122) {
        if (On) {
            ViaCrtcMask(hwp, 0x91, 0x10, 0x10);
            usleep(0x19);
            ViaCrtcMask(hwp, 0x91, 0x08, 0x08);
            usleep(0x1FE);
            ViaCrtcMask(hwp, 0x91, 0x06, 0x06);
            usleep(0x01);
        } else {
            ViaCrtcMask(hwp, 0x91, 0x00, 0x06);
            usleep(0xD2);
            ViaCrtcMask(hwp, 0x91, 0x00, 0x08);
            usleep(0x19);
            ViaCrtcMask(hwp, 0x91, 0x00, 0x10);
            usleep(0x01);
        }
#if 0
        if (!Private->X || !Private->Y) {  /* old index 0x00 -- never used */
            if (On) {
                ViaCrtcMask(hwp, 0x91, 0x10, 0x10);
                usleep(0x19);
                ViaCrtcMask(hwp, 0x91, 0x08, 0x08);
                usleep(0x64);
            } else {
                ViaCrtcMask(hwp, 0x91, 0x00, 0x18);
                usleep(0x01);
            }
        }
#endif
    } else { /* VT7205 and up */
        if (On) {
            ViaCrtcMask(hwp, 0x91, 0x10, 0x10);
            usleep(0x19);
            ViaCrtcMask(hwp, 0x91, 0x08, 0x08);
            usleep(0x1FE);
            ViaSeqMask(hwp, 0x3D, 0x20, 0x20);
            usleep(0x01);
        } else {
            ViaSeqMask(hwp, 0x3D, 0x00, 0x20);
            usleep(0xD2);
            ViaCrtcMask(hwp, 0x91, 0x00, 0x08);
            usleep(0x19);
            ViaCrtcMask(hwp, 0x91, 0x00, 0x10);
            usleep(0x01);
        }
    }

    usleep(1);
}

/*
 *
 */
static struct ViaOutput *
VT3122PanelInit(ScrnInfoPtr pScrn, struct ViaOutput *Output)
{
    Output->Name = "Panel";

    Output->ClockMaster = FALSE;

    Output->Monitor = ViaPanelMonitor(Output, ViaPanelModes);

    Output->Save = NULL;
    Output->Restore = NULL;
    Output->Sense = NULL;
    Output->ModeValid = ViaPanelModeValid;
    Output->Mode = ViaPanelMode;
    Output->Power = ViaPanelPower;
    Output->PrintRegs = NULL;

    return Output;
}

/*
 *
 * Primary only Panel implementation for the VT3108.
 *
 */
static struct {
    int X;
    int Y;
    char Name[10];
} VT3108PanelModes[] = {
    {1280, 768, "1280x768"},
    {1024, 768, "1024x768"},
    { 800, 600, "800x600"},
    { 640, 480, "640x480"},
    {   0,   0, ""},
};

/*
 * Add centered modes.
 *
 * We use Mode destructively here.
 *
 */
static void
VT3108PanelMonitorCenteredAdd(struct ViaOutput *Output, MonPtr Monitor,
                              DisplayModePtr Mode)
{
    struct ViaPanelOutputPrivate *Private = Output->Private;
    int i;

    for (i = 0; VT3108PanelModes[i].X; i++) {
        if ((VT3108PanelModes[i].X > Private->X) ||
            (VT3108PanelModes[i].Y > Private->Y))
            continue;

        /* We already added native resolution */
        if ((VT3108PanelModes[i].X == Private->X) &&
            (VT3108PanelModes[i].Y == Private->Y))
            continue;

        Mode->HDisplay = VT3108PanelModes[i].X;
        Mode->VDisplay = VT3108PanelModes[i].Y;
        xfree(Mode->name);
        Mode->name = xnfstrdup(VT3108PanelModes[i].Name);
        
        ViaPrintModeline(Output->scrnIndex, Mode);
        ViaModesCopyAdd(Monitor, Mode);
    }
}

/*
 * Here we assume that panels are capable of at least 57.5 and 65Hz CVT timing
 * and CVT reduced blanking timing. We use those timings to impose our limits.
 *
 */
static void
VT3108PanelMonitorCenteredTiming(struct ViaOutput *Output, MonPtr Monitor)
{
    struct ViaPanelOutputPrivate *Private = Output->Private;
    DisplayModePtr Mode, RMode;

    /* Faster normal mode = upper limit for timing*/
    Mode = ViaCVTMode(Private->X, Private->Y, 65.0, FALSE, FALSE);

    /* Reduced blanking mode = lower limit for timing */
    RMode = ViaCVTMode(Private->X, Private->Y, 60.0, TRUE, FALSE); 

    Private->HTotalMax = Mode->HTotal;
    Private->HTotalMin = RMode->HTotal;
    Private->HSyncMax = Mode->HSyncEnd - Mode->HSyncStart;
    Private->HSyncMin = 32; /* CVT -r default */

    Private->VTotalMax = Mode->VTotal;
    Private->VTotalMin = RMode->VTotal;
    Private->VSyncMax = Mode->VSyncEnd - Mode->VSyncStart;
    Private->VSyncMin = 3; /* CVT timing only goes down to 4 */

    ViaModesDestroy(RMode);

    /* Safe bets */
    Monitor->nVrefresh = 1;
    Monitor->vrefresh[0].lo = 57.5;
    Monitor->vrefresh[0].hi = 65.0;

    /* now get slow timing; for hsync */
    RMode = ViaCVTMode(Private->X, Private->Y, 57.5, FALSE, FALSE);

    Monitor->nHsync = 1;
    Monitor->hsync[0].hi = (float) Mode->Clock / Mode->HTotal;
    Monitor->hsync[0].lo = (float) RMode->Clock / RMode->HTotal;

    ViaModesDestroy(Mode);
    ViaModesDestroy(RMode);
}

/*
 *
 */
static MonPtr
VT3108PanelMonitor(struct ViaOutput *Output, DisplayModePtr Modes)
{
    struct ViaPanelOutputPrivate *Private = Output->Private;
    MonPtr Monitor;
    DisplayModePtr Mode;

    VIAFUNC(Output->scrnIndex);

    Monitor = xnfcalloc(1, sizeof(MonRec));
    memset(Monitor, 0, sizeof(MonRec));

#ifdef MONREC_HAS_REDUCED
    /* We need to add both CVT as normal modes when we can. */
    Monitor->reducedblanking = TRUE;

    /* Native mode */
    Mode = ViaCVTMode(Private->X, Private->Y, 60.0, TRUE, FALSE);
    ViaModesCopyAdd(Monitor, Mode);

    /* Add centered Modes */
    VT3108PanelMonitorCenteredAdd(Output, Monitor, Mode);

    ViaModesDestroy(Mode);
#endif

    /* Native mode */
    Mode = ViaCVTMode(Private->X, Private->Y, 60.0, FALSE, FALSE);
    ViaModesCopyAdd(Monitor, Mode);

    /* Add centered Modes */
    VT3108PanelMonitorCenteredAdd(Output, Monitor, Mode);

    ViaModesDestroy(Mode);

    /* set up strings */
    Monitor->id = xnfstrdup("Panel");
    Monitor->vendor = xnfstrdup("VIA");
    Monitor->model = xnfstrdup("TTL Panel");

    VT3108PanelMonitorCenteredTiming(Output, Monitor);

    Output->ModesExclusive = FALSE;

    return Monitor;
}

/*
 *
 */
static ModeStatus
VT3108PanelModeValid(struct ViaOutput *Output, DisplayModePtr mode)
{
    struct ViaPanelOutputPrivate *Private = Output->Private;
    int Sync;

    VIAFUNC(Output->scrnIndex);

    if ((mode->HDisplay > Private->X) || (mode->VDisplay > Private->Y))
        return MODE_PANEL;

    if ((mode->CrtcHTotal > Private->HTotalMax) || 
        (mode->CrtcHTotal < Private->HTotalMin))
        return MODE_H_ILLEGAL;

    if ((mode->CrtcVTotal > Private->VTotalMax) || 
        (mode->CrtcVTotal < Private->VTotalMin))
        return MODE_V_ILLEGAL;

    Sync = mode->CrtcHSyncEnd - mode->CrtcHSyncStart;
    if (Sync > Private->HSyncMax)
        return MODE_HSYNC_WIDE;
    if (Sync < Private->HSyncMin)
        return MODE_HSYNC_NARROW;

    Sync = mode->CrtcVSyncEnd - mode->CrtcVSyncStart;
    if (Sync > Private->VSyncMax)
        return MODE_VSYNC_WIDE;
    if (Sync < Private->VSyncMin)
        return MODE_VSYNC_NARROW;

    if (Private->Center) {
        int Shift;

        /* The timing is that of native resolution but with H/VDisplay smaller
         * than native resolution.
         *
         * We shift blanking away from H/VDisplay as panels are not sync
         * triggered but data-enable triggered.
         *
         * Sync is shifted too so that an attached CRT will be centered as well.
         */

        Shift = (Private->X - mode->HDisplay) / 2;
        if (Shift) {
            mode->CrtcHBlankEnd = mode->CrtcHTotal - Shift;
            mode->CrtcHBlankStart = mode->CrtcHDisplay + Shift;
            mode->CrtcHSyncStart -= Shift / 2;
            mode->CrtcHSyncEnd -= Shift / 2;
            mode->CrtcHAdjusted = TRUE;
        }

        Shift = (Private->Y - mode->VDisplay) / 2;
        if (Shift) {
            mode->CrtcVBlankEnd = mode->CrtcVTotal - Shift;
            mode->CrtcVBlankStart = mode->CrtcVDisplay + Shift;
            mode->CrtcVSyncStart -= Shift / 2;
            mode->CrtcVSyncEnd -= Shift / 2;
            mode->CrtcVAdjusted = TRUE;
        }
    }

    return MODE_OK;
}

/*
 *
 */
static void
VT3108PanelMode(struct ViaOutput *Output, DisplayModePtr mode)
{
    ScrnInfoPtr pScrn = xf86Screens[Output->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn); 
    struct ViaPanelOutputPrivate *Private = Output->Private;

    VIAFUNC(Output->scrnIndex);

    if (Private->Center ||
        ((mode->HDisplay == Private->X) && (mode->VDisplay == Private->Y))) {
        ViaCrtcMask(hwp, 0x79, 0x00, 0x01); /* Disable Scaling */
    } else {
        /* Scaling is not possible from the primary only */
        CARD32 HScale = (mode->HDisplay - 1) * 4096 / (Private->X - 1);
        CARD32 VScale = (mode->VDisplay - 1) * 2048 / (Private->Y - 1);

        ViaCrtcMask(hwp, 0x9F, HScale, 0x03);
        hwp->writeCrtc(hwp, 0x77, HScale >> 2);
        ViaCrtcMask(hwp, 0x79, HScale >> 6, 0x30);

        ViaCrtcMask(hwp, 0x79, VScale << 3, 0x08);
        hwp->writeCrtc(hwp, 0x78, VScale >> 1);
        ViaCrtcMask(hwp, 0x79, VScale >> 3, 0xC0);

        /* Enable Scaling and H/V interpolation. */
        ViaCrtcMask(hwp, 0x79, 0x07, 0x07);
    }
}


/*
 * This implements only superficial power down. Good enough for DPMS.
 */
static void
VT3108PanelPower(struct ViaOutput *Output, Bool On)
{
    ScrnInfoPtr pScrn = xf86Screens[Output->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);

    ViaDebug(Output->scrnIndex, "%s: %s.\n", __func__, On ? "On" : "Off");

    ViaCrtcMask(hwp, 0x91, 0x00, 0x01); /* Disable Software Control */
    usleep(1);
    
    if (On) /* Direct Panel Display and Backlight control */
        ViaCrtcMask(hwp, 0x91, 0x00, 0xC0);
    else
        ViaCrtcMask(hwp, 0x91, 0xC0, 0xC0);
}

/*
 *
 */
static struct ViaOutput *
VT3108PanelInit(ScrnInfoPtr pScrn, struct ViaOutput *Output)
{
    Output->Name = "Panel";

    Output->ClockMaster = FALSE;

    Output->Monitor = VT3108PanelMonitor(Output, ViaPanelModes);

    Output->Save = NULL;
    Output->Restore = NULL;
    Output->Sense = NULL;
    Output->ModeValid = VT3108PanelModeValid;
    Output->Mode = VT3108PanelMode;
    Output->Power = VT3108PanelPower;
    Output->PrintRegs = NULL;

    return Output;
}


/*
 * future: I2CDevPtr for LVDS encoders.
 */
struct ViaOutput *
ViaPanelInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output;
    struct ViaPanelOutputPrivate *Private;

    VIAFUNC(pScrn->scrnIndex);

    if (pDev)
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Ignoring I2C Device"
                   " passed.\n", __func__);
    
    Output = xnfcalloc(1, sizeof(struct ViaOutput));

    Output->Prev = NULL;
    Output->Next = NULL;
    Output->scrnIndex = pScrn->scrnIndex;
    Output->I2CDev = pDev;
    Output->Type = OUTPUT_PANEL; 

    Private = ViaPanelPrivateCreate(Output);

    /* Parse options here ourselves. */
    Output->Options = ViaPanelGetOptions(pScrn, Private);

    /* Is there even a panel present? */
    if (!Private->Present) {
        if (pVia->Id && pVia->Id->HasPanel) {
            xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Enabling panel from"
                       " PCI-Subsystem Id information.\n");
            Private->Present = TRUE;
        } else {
            Output->PrivateDestroy(Output);
            
            xfree(Output->Options);
            xfree(Output);
            return NULL;
        }
    }

    ViaPanelNativeResolution(pScrn, Private);

    switch (pVia->Chipset) {
    case VT3122:
    case VT7205:
        return VT3122PanelInit(pScrn, Output);
    case VT3108:
        return VT3108PanelInit(pScrn, Output);
    default:
	xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Panel on %s is currently "
                   "not supported.\n", pScrn->chipset);
        Output->PrivateDestroy(Output);
        xfree(Output->Options);
        xfree(Output);
        return NULL;
    }
}
