/*
 * Copyright 2004-2006 Luc Verhaegen.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*
 * via_mode.c
 *
 * Everything to do with setting and changing modes.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#endif

#include "via_vgahw.h"
#include "via_id.h"
#include "via_mode.h"

/*
 * Some VGA register debugging.
 */
void
ViaVgaPrintRegs(ScrnInfoPtr pScrn, const char *function)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    int i;

    ViaDebug(pScrn->scrnIndex, "%s: Printing VGA registers:\n", function);
    ViaDebug(pScrn->scrnIndex, "Printing VGA Sequence registers:\n");
    for (i = 0x00; i < 0x80; i++)
	ViaDebug(pScrn->scrnIndex, "SR%02X: 0x%02X\n", i, hwp->readSeq(hwp, i));

    ViaDebug(pScrn->scrnIndex, "Printing VGA CRTM/C registers:\n");
    for (i = 0x00; i < 0x19; i++)
        ViaDebug(pScrn->scrnIndex, "CR%02X: 0x%02X\n", i, hwp->readCrtc(hwp, i));
    for (i = 0x33; i < 0xA3; i++)
	ViaDebug(pScrn->scrnIndex, "CR%02X: 0x%02X\n", i, hwp->readCrtc(hwp, i));

    ViaDebug(pScrn->scrnIndex, "Printing VGA Graphics registers:\n");
    for (i = 0x00; i < 0x08; i++)
	ViaDebug(pScrn->scrnIndex, "GR%02X: 0x%02X\n", i, hwp->readGr(hwp, i));

    ViaDebug(pScrn->scrnIndex, "Printing VGA Attribute registers:\n");
    for (i = 0x00; i < 0x14; i++)
	ViaDebug(pScrn->scrnIndex, "AR%02X: 0x%02X\n", i, hwp->readAttr(hwp, i));
    
    ViaDebug(pScrn->scrnIndex, "Printing VGA Miscellaneous register:\n");
    ViaDebug(pScrn->scrnIndex, "Misc: 0x%02X\n", hwp->readMiscOut(hwp));

    ViaDebug(pScrn->scrnIndex, "End of VGA Registers.\n");
}

/*
 *
 * Dotclock handling.
 *
 */


/*
 * Standard vga call really.
 * Needs to be called to reset the dotclock (after SR40:2/1 reset)
 */
static void
ViaSetUseExternalClock(ScrnInfoPtr pScrn)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    CARD8 data;

    VIAFUNC(pScrn->scrnIndex);

    data = hwp->readMiscOut(hwp);
    hwp->writeMiscOut(hwp, data | 0x0C);
}


/* 
 *
 */
static void
ViaSetPrimaryDotclock(ScrnInfoPtr pScrn, CARD32 clock)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);

    ViaDebug(pScrn->scrnIndex, "%s to 0x%lX\n", __func__, clock);

    if ((pVia->Chipset == VT3122) || (pVia->Chipset == VT7205)) {
	hwp->writeSeq(hwp, 0x46, clock >> 8);
	hwp->writeSeq(hwp, 0x47, clock & 0xFF);
    } else { /* unichrome pro */
	hwp->writeSeq(hwp, 0x44, clock >> 16);
	hwp->writeSeq(hwp, 0x45, (clock >> 8) & 0xFF);
	hwp->writeSeq(hwp, 0x46, clock & 0xFF);
    }

    ViaSeqMask(hwp, 0x40, 0x02, 0x02);
    ViaSeqMask(hwp, 0x40, 0x00, 0x02);
}


/* 
 *
 */
void
ViaSetSecondaryDotclock(ScrnInfoPtr pScrn, CARD32 clock)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);

    ViaDebug(pScrn->scrnIndex, "%s to 0x%lX\n", __func__, clock);

    if ((pVia->Chipset == VT3122) || (pVia->Chipset == VT7205)) {
	hwp->writeSeq(hwp, 0x44, clock >> 8);
	hwp->writeSeq(hwp, 0x45, clock & 0xFF);
    } else { /* unichrome pro */
	hwp->writeSeq(hwp, 0x4A, clock >> 16);
	hwp->writeSeq(hwp, 0x4B, (clock >> 8) & 0xFF);
	hwp->writeSeq(hwp, 0x4C, clock & 0xFF);
    }

    ViaSeqMask(hwp, 0x40, 0x04, 0x04);
    ViaSeqMask(hwp, 0x40, 0x00, 0x04);
}

/*
 *
 */
static CARD32
VT3122PLLGenerateBest(int Clock, int Shift, int MinDiv, int MaxDiv,
                             int *BestDiff)
{
    CARD32 PLL = 0;
    CARD8 PLLShift;
    int Div, Mult, Diff;
    float Ratio = Clock / 14318.0;

    switch (Shift) {
    case 4:
        PLLShift = 0x80;
        break;
    case 2:
        PLLShift = 0x40;
        break;
    default:
        PLLShift = 0x00;
        break;
    }

    for (Div = MinDiv; Div <= MaxDiv; Div++) {
        Mult = Ratio * Div * Shift + 0.5;
        
        if (Mult < 129) {
            Diff = Clock - Mult * 14318 / Div / Shift;

            if (Diff < 0)
                Diff *= -1;

            if (Diff < *BestDiff) {
                *BestDiff = Diff;
                PLL = ((PLLShift | Div) << 8) | Mult;
            }
        }
    }

    return PLL;
}

/*
 * This might seem nasty and ugly, but it's the best solution given the crappy
 * limitations the VT3122 PLL has.
 *
 * The below information has been gathered using nothing but a lot of time and
 * perseverance.
 */
static CARD32
VT3122PLLGenerate(ScrnInfoPtr pScrn, int Clock)
{
    CARD32 PLL;
    int Diff = 300000;

    VIAFUNC(pScrn->scrnIndex);

    if (Clock > 72514)
        PLL = VT3122PLLGenerateBest(Clock, 1, 2, 25, &Diff);
    else if (Clock > 71788)
        PLL = VT3122PLLGenerateBest(Clock, 1, 16, 24, &Diff);
    else if (Clock > 71389) {
        PLL = 0x1050; /* Big singularity. */
        
        Diff = Clock - 71590;
        if (Diff < 0)
            Diff *= -1;
    } else if (Clock > 48833) {
        CARD32 tmpPLL;

        PLL = VT3122PLLGenerateBest(Clock, 2, 7, 18, &Diff);
    
        if (Clock > 69024)
            tmpPLL = VT3122PLLGenerateBest(Clock, 1, 15, 23, &Diff);
        else if (Clock > 63500)
            tmpPLL = VT3122PLLGenerateBest(Clock, 1, 15, 21, &Diff);
        else if (Clock > 52008)
            tmpPLL = VT3122PLLGenerateBest(Clock, 1, 17, 19, &Diff);
        else
            tmpPLL = VT3122PLLGenerateBest(Clock, 1, 17, 17, &Diff);

        if (tmpPLL)
            PLL =  tmpPLL;
    } else if (Clock > 35220)
        PLL = VT3122PLLGenerateBest(Clock, 2, 11, 24, &Diff);
    else if (Clock > 34511)
        PLL = VT3122PLLGenerateBest(Clock, 2, 11, 23, &Diff);
    else if (Clock > 33441)
        PLL = VT3122PLLGenerateBest(Clock, 2, 13, 22, &Diff);
    else if (Clock > 31967)
        PLL = VT3122PLLGenerateBest(Clock, 2, 11, 21, &Diff);
    else
        PLL = VT3122PLLGenerateBest(Clock, 4, 8, 19, &Diff);
    
    ViaDebug(pScrn->scrnIndex, "%s: PLL: 0x%04X (%d off from %d)\n",
             __func__, PLL, Diff, Clock);
    return PLL;
}

/*
 *
 */
static int
VT3108PLLGenerateBest(ScrnInfoPtr pScrn, int Clock, int Shift, int Div,
                      int oldDiff, CARD32 *PLL)
{
    int Mult = ((float) Clock * (Div << Shift)) / 14318.18 + 0.5;
    int Diff;

    if (Mult > 257) /* Don't go over 0xFF + 2; wobbly */
        return oldDiff;

    Diff = Clock - Mult * 14318 / (Div << Shift);
    if (Diff < 0)
        Diff *= -1;
    
    if (Diff < oldDiff) {
        *PLL = (Mult - 2) << 16;
        *PLL |= Div - 2;
        *PLL |= Shift << 10;
        return Diff;
    } else
        return oldDiff;
}

/*
 *
 */
static CARD32
VT3108PLLGenerate(ScrnInfoPtr pScrn, int Clock)
{
    CARD32 PLL;
    int Diff = 300000;
    int i;

    VIAFUNC(pScrn->scrnIndex);

    for (i = 2; i < 15; i++)
        Diff = VT3108PLLGenerateBest(pScrn, Clock, 0, i, Diff, &PLL);

    for (i = 2; i < 15; i++) 
        Diff = VT3108PLLGenerateBest(pScrn, Clock, 1, i, Diff, &PLL);

    for (i = 2; i < 32; i++) 
        Diff = VT3108PLLGenerateBest(pScrn, Clock, 2, i, Diff, &PLL);

    for (i = 2; i < 21; i++) 
        Diff = VT3108PLLGenerateBest(pScrn, Clock, 3, i, Diff, &PLL);

    ViaDebug(pScrn->scrnIndex, "%s: PLL: 0x%04X (%d off from %d)\n",
             __func__, PLL, Diff, Clock);

    return PLL;
}

/*
 *
 */
CARD32
ViaPLLGenerate(ScrnInfoPtr pScrn, int Clock)
{
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    switch (pVia->Chipset) {
    case VT3122:
    case VT7205:
        return VT3122PLLGenerate(pScrn, Clock);
    case VT3108:
        return VT3108PLLGenerate(pScrn, Clock);
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Unhandled Chipset: %s\n",
                   __func__, pScrn->chipset);
        return 0;
    }
}

/*
 * Set the primary CRTC dotclock to act as a slave.
 */
static void
ViaDotclockPrimarySlave(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);

    switch (pVia->Chipset) {
    case VT3122:
        if (VT3122_REV_IS_AX(pVia->ChipRev)) {
            ViaSetPrimaryDotclock(pScrn, 0x471C); /* VT3122Ax use 2x XCLK */
            break;
        }
        /* fall through */
    case VT7205:
        ViaSetPrimaryDotclock(pScrn, 0x871C);
        break;
    case VT3108:
        /* TESTME: Seems to handle external bit without default dotclock */
        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Unhandled Chipset: %s\n",
                   __func__, pScrn->chipset);
        break;
    }
    ViaSetUseExternalClock(pScrn);
}

/*
 *
 * Output Handling.
 *
 */


/*
 *
 */
static struct ViaOutput *
ViaOutputDestroy(struct ViaOutput *Output)
{
    struct ViaOutput *Next;

    Next = Output->Next;

    if (Output->PrivateDestroy)
        Output->PrivateDestroy(Output);

    if (Output->Monitor) {
        xfree(Output->Monitor->id);
        xfree(Output->Monitor->vendor);
        xfree(Output->Monitor->model);
        ViaModesDestroy(Output->Monitor->Modes);
        xfree(Output->Monitor);
    }

    if (Output->I2CDev) 
        xf86DestroyI2CDevRec(Output->I2CDev, TRUE);

    if (Output->Options)
        xfree(Output->Options);
    
    xfree(Output);

    return Next;
}

/*
 *
 */
void
ViaOutputsDestroy(ScrnInfoPtr pScrn)
{
    struct ViaOutput *Output = VIAPTR(pScrn)->Outputs;

    while (Output)
        Output = ViaOutputDestroy(Output);
}

/*
 *
 */
static Bool
ViaOutputAdd(ScrnInfoPtr pScrn, struct ViaOutput *Output)
{
    struct ViaOutput *Last = NULL;

    /* Is this Output sound? */
    if (!Output->Name || !Output->ModeValid || !Output->Mode ||
        !Output->Monitor || !Output->Power) {
        
        if (!Output->Name)
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, 
                       "%s: Nameless output device.\n", __func__);
        
        if (!Output->ModeValid)
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: %s: ModeValid "
                       "callback missing\n", __func__, Output->Name);
        
        if (!Output->Mode)
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: %s: Mode "
                       "callback missing\n", __func__, Output->Name);

        if (!Output->Monitor)
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: %s: Monitor "
                       "uninitialised\n", __func__, Output->Name);
        
        if (!Output->Power)
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: %s: Power "
                       "callback missing\n", __func__, Output->Name);
        
        return FALSE;
    }
    
    /* add Output to list */
    Last = VIAPTR(pScrn)->Outputs;
    if (Last) {
        while (Last->Next)
            Last = Last->Next;
        
        Last->Next = Output;
        Output->Prev = Last;
    } else {
        VIAPTR(pScrn)->Outputs = Output;
        Output->Prev = NULL;
    }

    return TRUE;
}

#ifdef UNUSED
/*
 *
 */
static void
ViaOutputRemove(ScrnInfoPtr pScrn, struct ViaOutput *Output)
{
    if (!Output)
        return;

    /* remove from list */
    if (Output->Prev)
        Output->Prev->Next = Output->Next;
    else
        VIAPTR(pScrn)->Outputs = Output->Next;
    
    if (Output->Next)
        Output->Next->Prev = Output->Prev;

    ViaOutputDestroy(Output);
}
#endif

/*
 *
 * Whine about TODOs.
 *
 */

/*
 * VIAs VT1631 LVDS encoder. (aka VT3191)
 * VIAs VT1632(A) TMDS encoder. (aka VT3192)
 */
static struct ViaOutput *
ViaVT163xInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    CARD8 buf = 0, Read[4];

    VIAFUNC(pScrn->scrnIndex);

    if (!xf86I2CWriteRead(pDev, &buf, 1, Read, 4)) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, 
                   "%s: Unable to read from %s Slave %d.\n",
                   __func__,  pDev->pI2CBus->BusName, pDev->SlaveAddr);
        return NULL;
    }

    if ((((Read[1] << 8) | Read[0]) != 0x1106) || (Read[3] != 0x31)) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Unknown device: %s Slave %d ids: %02X%02X:%02X%02X.\n",
                   __func__, pDev->pI2CBus->BusName, pDev->SlaveAddr,
                   Read[1], Read[0], Read[3], Read[2]);
        return NULL;
    }

    switch (Read[2]) {
    case 0x91:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED,
                   "Detected Via Technologies VT1631 LVDS Transmitter.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "VT1631 is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;
    case 0x92:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, 
                   "Detected Via Technologies VT1632(A) TMDS Transmitter.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "VT1632 is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, 
                   "%s: Unknown VIA device detected: 0x%02X%02X.\n",
                   __func__, Read[3], Read[2]);
        return NULL;
    }

    return NULL;
}

/*
 * Silicon Image TMDS encoders.
 */
static struct ViaOutput *
ViaSiI16xInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    CARD8 buf = 0, Read[4];

    VIAFUNC(pScrn->scrnIndex);

    if (!xf86I2CWriteRead(pDev, &buf, 1, Read, 4)) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, 
                   "%s: Unable to read from %s Slave %d.\n",
                   __func__,  pDev->pI2CBus->BusName, pDev->SlaveAddr);
        return NULL;
    }

    if ((((Read[1] << 8) | Read[0]) != 0x0001) || (Read[3] != 0x00)) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Unknown device: %s Slave %d ids: %02X%02X:%02X%02X.\n",
                   __func__, pDev->pI2CBus->BusName, pDev->SlaveAddr,
                   Read[1], Read[0], Read[3], Read[2]);
        return NULL;
    }

    switch (Read[2]) {
    case 0x06:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED,
                   "Detected Silicon Image SiI164 TMDS Transmitter.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "SiI164 is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, 
                   "%s: Unknown Silicon Image device detected: 0x%02X%02X.\n",
                   __func__, Read[3], Read[2]);
        return NULL;
    }

    return NULL;
}

/*
 * Philips TV encoders.
 * 
 * SAA7108
 * SAA7109
 */
static struct ViaOutput *
ViaSAA710xInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    CARD8 buf;

    VIAFUNC(pScrn->scrnIndex);

    if (!xf86I2CReadByte(pDev, 0x1C, &buf)) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, 
                   "%s: Unable to read from %s Slave %d.\n",
                   __func__,  pDev->pI2CBus->BusName, pDev->SlaveAddr);
        return NULL;
    }

    switch (buf) {
    case 0x02:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, 
                   "Detected Philips Semiconductors SAA7108E TV Encoder.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "SAA7108E is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;
    case 0x03:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, 
                   "Detected Philips Semiconductors SAA7109E TV Encoder.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "SAA7109E is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Unknown device detected: 0x%02X.\n", __func__, buf);
        return NULL;
    };

    return NULL;
}

/*
 * Focus enhancements tv encoders.
 */
static struct ViaOutput *
ViaFS45xInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    CARD8 buf;

    VIAFUNC(pScrn->scrnIndex);

    if (!xf86I2CReadByte(pDev, 0x7F, &buf)) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, 
                   "%s: Unable to read from %s Slave %d.\n",
                   __func__,  pDev->pI2CBus->BusName, pDev->SlaveAddr);
        return NULL;
    }

    switch (buf) {
    case 0x20:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, 
                   "Detected Focus Enhancements 454 TV Encoder.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "FS454 is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Unknown device detected: 0x%02X.\n", __func__, buf);
        return NULL;
    };

    return NULL;
}

/*
 * Aitech AIT2139
 */
static struct ViaOutput *
ViaAIT213xInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    CARD8 buf;

    VIAFUNC(pScrn->scrnIndex);

    if (!xf86I2CReadByte(pDev, 0x3D, &buf)) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, 
                   "%s: Unable to read from %s Slave %d.\n",
                   __func__,  pDev->pI2CBus->BusName, pDev->SlaveAddr);
        return NULL;
    }

    switch (buf) {
    case 0x11:
        xf86DrvMsg(pScrn->scrnIndex, X_PROBED, 
                   "Detected AITech AIT2139 TV Encoder.\n");
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "AIT2139 is not supported yet."
                   " Please contact unichrome-devel@lists.sf.net\n");
        return NULL;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Unknown device detected: 0x%02X.\n", __func__, buf);
        return NULL;

    }
}

static struct {
    CARD8 Address;
    char * Name;
    struct ViaOutput * (*Init) (ScrnInfoPtr pScrn, I2CDevPtr pDev);
} ViaI2CDevices[] = {
    {0x10, "VT163x",  ViaVT163xInit},
    {0x40, "VT162x",  ViaVT162xInit},
    {0x42, "VT162x",  ViaVT162xInit},
    {0x70, "VT163x",  ViaVT163xInit},
    {0x70, "SiI16x",  ViaSiI16xInit},
    {0x88, "SAA710x", ViaSAA710xInit},
    {0x88, "AIT213x", ViaAIT213xInit},
    {0x8A, "AIT213x", ViaAIT213xInit},
    {0xD4, "FS45x",   ViaFS45xInit},
    {0xEA, "CH7xxx",  ViaCH7xxxInit},
    {0xEC, "CH7xxx",  ViaCH7xxxInit},
    {0x00, NULL, NULL}
};

/*
 *
 */
static void
ViaScanBus(ScrnInfoPtr pScrn, I2CBusPtr pI2CBus)
{
    struct ViaOutput *Output;
    CARD8 SkipAddress = 0x00;
    int i;

    VIAFUNC(pScrn->scrnIndex);

    for (i = 0; ViaI2CDevices[i].Init; i++) {
        I2CDevPtr pDev = NULL;

        /* Make sure we don't try to Init a successfully inited device again */
        if (ViaI2CDevices[i].Address == SkipAddress)
            continue;

        if (!xf86I2CProbeAddress(pI2CBus, ViaI2CDevices[i].Address))
            continue;

        pDev = xf86CreateI2CDevRec();
        
        pDev->DevName = ViaI2CDevices[i].Name;
        pDev->SlaveAddr = ViaI2CDevices[i].Address;
        pDev->pI2CBus = pI2CBus;
        
        if (!xf86I2CDevInit(pDev)) {
            xf86DestroyI2CDevRec(pDev, TRUE);
            continue;
        }

        Output = ViaI2CDevices[i].Init(pScrn, pDev);
        
        /* did we actually find anything? */
        if (!Output) {
            xf86DestroyI2CDevRec(pDev,TRUE);
        } else {
            SkipAddress = ViaI2CDevices[i].Address;

            if (!ViaOutputAdd(pScrn, Output)) {
                while (Output) /* could be a whole list */
                    Output = ViaOutputDestroy(Output);
            }
        }
    }
}

/*
 *
 */
static char *
ViaOutputBusName(int Position)
{
    switch(Position) {
    case OUTPUT_BUS_CRT:
        return "CRT";
    case OUTPUT_BUS_DVP0:
        return "DVP0";
    case OUTPUT_BUS_DVP1:
        return "DVP1";
    case OUTPUT_BUS_DFP:
        return "DFP";
    default:
        return "Unknown";
    }
}

/*
 *
 */
void
ViaOutputsDetect(ScrnInfoPtr pScrn)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output;
    CARD8 SR12 = hwp->readSeq(hwp, 0x12);

    VIAFUNC(pScrn->scrnIndex);

    /* CRT */
    Output = ViaCRTInit(pScrn, NULL);
    if (Output && !ViaOutputAdd(pScrn, Output))
        while (Output) /* could be a whole list */
            Output = ViaOutputDestroy(Output);

    /* Panel */
    Output = ViaPanelInit(pScrn, NULL);
    if (Output && !ViaOutputAdd(pScrn, Output))
        while (Output) /* could be a whole list */
            Output = ViaOutputDestroy(Output);

    if (pVia->pI2CBus2)
        ViaScanBus(pScrn, pVia->pI2CBus2);

    if (pVia->pI2CBus3)
        ViaScanBus(pScrn, pVia->pI2CBus3);

    /* Match outputs to Bus */
    Output = pVia->Outputs;
    while (Output) {
        /* check bus position */
        switch (Output->Type) {
        case OUTPUT_CRT:
            Output->Position = OUTPUT_BUS_CRT; /* Duh! */
            break;
        case OUTPUT_TV:
            if (SR12 & 0x20)
                Output->Position = OUTPUT_BUS_DVP0;
            else
                Output->Position = OUTPUT_BUS_DVP1;
            break;
        case OUTPUT_PANEL:
            /* split me! */
            Output->Position = OUTPUT_BUS_DFP;
            break;
        default:
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Unhandled output device"
                       " type.\n", __func__);
            break;
        }

        /* Dump registers as early as possible */
        if (pVia->PrintTVRegs && Output->PrintRegs)
            Output->PrintRegs(Output, __func__);

        Output = Output->Next;
    }
}

/*
 * Save all output devices.
 */
void
ViaOutputsSave(ScrnInfoPtr pScrn)
{
    struct ViaOutput *Output = VIAPTR(pScrn)->Outputs;

    while (Output) {
        if (Output->Save)
            Output->Save(Output);

        Output = Output->Next;
    }
}

/*
 * Restore all output devices.
 */
void
ViaOutputsRestore(ScrnInfoPtr pScrn)
{
    struct ViaOutput *Output = VIAPTR(pScrn)->Outputs;

    while (Output) {
        if (Output->Restore)
            Output->Restore(Output);

        Output = Output->Next;
    }
}

/*
 * Check which output devices are active.
 */
static Bool
ViaOutputsSense(ScrnInfoPtr pScrn)
{
    struct ViaOutput *Output = VIAPTR(pScrn)->Outputs;
    Bool Active = FALSE;

    while (Output) {
        if (Output->Sense) {
            Output->Active = Output->Sense(Output);
            if (Output->Active)
                Active = TRUE;
        } else {
            Output->Active = TRUE;
            Active = TRUE;
        }

        Output = Output->Next;
    }

    return Active;
}

/*
 *
 */
static void
ViaBusPower(ScrnInfoPtr pScrn, struct ViaOutput *Output, Bool On)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);

    ViaDebug(pScrn->scrnIndex, "%s: Bus %s %s.\n", __func__,
             ViaOutputBusName(Output->Position), On ? "On" : "Off");

    if (On) {
        switch (Output->Position) {
        case OUTPUT_BUS_CRT:
            ;
            break;
        case OUTPUT_BUS_DVP0:
            ViaSeqMask(hwp, 0x1E, 0xC0, 0xC0);
            break;
        case OUTPUT_BUS_DVP1:
            ViaSeqMask(hwp, 0x1E, 0x30, 0x30);
            break;
        case OUTPUT_BUS_DFP:
            ViaSeqMask(hwp, 0x2A, 0x0F, 0x0F);
            break;
        default:
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: unhandled bus: %d\n",
                       __func__, Output->Position);
            break;
        }
    } else {
        switch (Output->Position) {
        case OUTPUT_BUS_CRT:
            ;
            break;
        case OUTPUT_BUS_DVP0:
            ViaSeqMask(hwp, 0x1E, 0x00, 0xC0);
            break;
        case OUTPUT_BUS_DVP1:
            ViaSeqMask(hwp, 0x1E, 0x00, 0x30);
            break;
        case OUTPUT_BUS_DFP:
            ViaSeqMask(hwp, 0x2A, 0x00, 0x0F);
            break;
        default:
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: unhandled bus: %d\n",
                       __func__, Output->Position);
            break;
        }
    }
}


/*
 * Power according to On all active devices, power off when inactive.
 *
 * CRT requires owner for secondary.
 *
 */
void
ViaOutputsPower(ScrnInfoPtr pScrn, Bool On)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output = pVia->Outputs;

    ViaDebug(pScrn->scrnIndex, "%s: %s.\n", __func__, On ? "On" : "Off");

    while (Output) {
        if (Output->Power) {
            if (Output->Active && On)
                Output->Power(Output, TRUE);
            else
                Output->Power(Output, FALSE);
        }

        ViaBusPower(pScrn, Output, (Output->Active && On));

        Output = Output->Next;
    }
}

/*
 * Handle Grammar Correction. ;)
 */
static void
ViaOutputsGamma(ScrnInfoPtr pScrn, Bool On)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output = pVia->Outputs;


    if (pVia->Chipset < VT3108) {
        if (On)
            ViaSeqMask(hwp, 0x16, 0x80, 0x80);
        else
            ViaSeqMask(hwp, 0x16, 0x00, 0x80);
    } else {
        if (On)
            ViaCrtcMask(hwp, 0x33, 0x80, 0x80);
        else
            ViaCrtcMask(hwp, 0x33, 0x00, 0x80);
    }

    while (Output) {
        if (Output->Active) {
            switch (Output->Position) {
            case OUTPUT_BUS_CRT:
                break;
            case OUTPUT_BUS_DVP0:
            case OUTPUT_BUS_DVP1:
                if (On)
                    ViaCrtcMask(hwp, 0x32, 0x04, 0x04);
                else
                    ViaCrtcMask(hwp, 0x32, 0x00, 0x04);
                break;
            case OUTPUT_BUS_DFP:
                if (On)
                    ViaCrtcMask(hwp, 0x6A, 0x02, 0x02);
                else
                    ViaCrtcMask(hwp, 0x6A, 0x00, 0x02);
                break;
            default:
                xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: unhandled bus: %d\n",
                           __func__, Output->Position);
                break;
            }
        }
        Output = Output->Next;
    }
}

/*
 *
 */
void
ViaOutputsPrintRegs(ScrnInfoPtr pScrn, const char *function)
{
    struct ViaOutput *Output = VIAPTR(pScrn)->Outputs;

    while (Output) {
        if (Output->PrintRegs)
            Output->PrintRegs(Output, function);

        Output = Output->Next;
    }
}

/*
 * Is the CRTC in control of the dotclock, or is the output device?
 */
static void
ViaOutputsFindDotclockMaster(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput  *Output = pVia->Outputs;
   
    pVia->ClockSlave = FALSE;

    while (Output) {
        if (Output->Active && Output->ClockMaster) {
            if (pVia->ClockSlave)
                xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: multiple Outputs "
                           "want to be the dotclock master.\n", __func__);
            pVia->ClockSlave = TRUE;
        }
        Output = Output->Next;
    }
}

/*
 *
 */
static ModeStatus
ViaOutputsModeValid(ScrnInfoPtr pScrn, DisplayModePtr mode)
{
    struct ViaOutput *Output = VIAPTR(pScrn)->Outputs;

    while (Output) {
        if (Output->Active && Output->ModeValid) {
            ModeStatus Status;
            Status = Output->ModeValid(Output, mode);
            
            if (Status != MODE_OK)
                return Status;
        }
        
        Output = Output->Next;
    }

    return MODE_OK;
}

/*
 * This function is all about handling the output bus surrounding the
 * actual setting of the mode of the output device.
 * 
 */
static void
ViaOutputMode(ScrnInfoPtr pScrn, struct ViaOutput *Output, DisplayModePtr Mode)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    if (!Output->Mode || !Mode)
        return;

    switch (Output->Position) {
    case OUTPUT_BUS_CRT:
        Output->Mode(Output, Mode); /* empty function actually */

        if (pVia->IsSecondary)
            ViaSeqMask(hwp, 0x16, 0x40, 0x40);
        else
            ViaSeqMask(hwp, 0x16, 0x00, 0x40);

        break;
    case OUTPUT_BUS_DVP0:
        if (pVia->Chipset < VT3108) {
            /* Quick 'n dirty workaround for non-primary case where a completely
             * idle bus fails to properly activate the tv encoder. */
            ViaDotclockPrimarySlave(pScrn);
            
            Output->Mode(Output, Mode);
            
            if (pVia->IsSecondary) {
                ViaCrtcMask(hwp, 0x6A, 0x80, 0x80);
                ViaCrtcMask(hwp, 0x6C, 0x80, 0x80);
            
                if ((pVia->Chipset == VT3122) && VT3122_REV_IS_AX(pVia->ChipRev)) {
                    ViaCrtcMask(hwp, 0x6B, 0x20, 0x20);
                    ViaCrtcMask(hwp, 0x6C, 0x10, 0x10);
                }
            
                /* Disable LCD Scaling */
                if (!pVia->SAMM)
                    hwp->writeCrtc(hwp, 0x79, 0x00);
            } else {
                if ((pVia->Chipset == VT3122) && VT3122_REV_IS_AX(pVia->ChipRev))
                    ViaCrtcMask(hwp, 0x6B, 0x80, 0x80);
            }
            ViaCrtcMask(hwp, 0x6A, 0x40, 0x40);
            ViaCrtcMask(hwp, 0x6B, 0x01, 0x01);
            ViaCrtcMask(hwp, 0x6C, 0x01, 0x01);
        } else {
            if (pVia->IsSecondary) {
                ViaCrtcMask(hwp, 0x6C, 0x80, 0x80);
                ViaCrtcMask(hwp, 0x96, 0x10, 0x10);
            } else {
                ViaCrtcMask(hwp, 0x6C, 0x00, 0x80);
                ViaCrtcMask(hwp, 0x96, 0x00, 0x10);
            }
            ViaCrtcMask(hwp, 0x6C, 0x01, 0x01);
            Output->Mode(Output, Mode);
        }

        break;
    case OUTPUT_BUS_DVP1:
        if (pVia->IsSecondary) {
            ViaCrtcMask(hwp, 0x6C, 0x80, 0x80);
            ViaCrtcMask(hwp, 0x9B, 0x10, 0x10);
        } else {
            ViaCrtcMask(hwp, 0x6C, 0x00, 0x80);
            ViaCrtcMask(hwp, 0x9B, 0x00, 0x10);
        }
        ViaCrtcMask(hwp, 0x6C, 0x01, 0x01);

        Output->Mode(Output, Mode);

        break;
    case OUTPUT_BUS_DFP:
        /* If i ever get to clean up the panel crap... */
        Output->Mode(Output, Mode);

        if (pVia->Chipset == VT3108) {
            /* Enable DFP */
            ViaSeqMask(hwp, 0x2A, 0x0F, 0x0F);

            /* Set DFP source */
            if (pVia->IsSecondary) {
                ViaCrtcMask(hwp, 0x97, 0x10, 0x10);
                ViaCrtcMask(hwp, 0x99, 0x10, 0x10);
            } else {
                ViaCrtcMask(hwp, 0x97, 0x00, 0x10);
                ViaCrtcMask(hwp, 0x99, 0x00, 0x10);
            }
            
            /* LCD + LCD Prefetch enable */
            ViaCrtcMask(hwp, 0x6A, 0x09, 0x09);
        }

        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: unhandled bus: %d\n",
                   __func__, Output->Position);
        break;
    }
}


/*
 * Print a list of active output devices.
 */
static void
ViaOutputsPrint(ScrnInfoPtr pScrn)
{
    struct ViaOutput *Output = VIAPTR(pScrn)->Outputs;

    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Listing Outputs:\n");

    while (Output) {
        xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Bus %s: Output %s: %s\n",
                   ViaOutputBusName(Output->Position), Output->Name,
                   Output->Active ? "Active" : "Disabled");

        Output = Output->Next;
    }
}

/*
 *
 */
Bool
ViaOutputsSelect(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output;

    if (pVia->IsSecondary) { /* we just abort for now */
	xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: Not handling secondary.\n",
                   __func__);
	return FALSE;
    }

    VIAFUNC(pScrn->scrnIndex);
    ViaDebug(pScrn->scrnIndex, "X Configuration: 0x%02x\n", pVia->ActiveDevice);

    if (!pVia->ActiveDevice)
        return ViaOutputsSense(pScrn);
    else {
        Bool Active = FALSE, Found;

        if (pVia->ActiveDevice & VIA_DEVICE_LCD) {
            Found = FALSE;

            Output = pVia->Outputs;

            while (Output) {
                if (Output->Type & OUTPUT_PANEL) {
                    Found = TRUE;
                    Output->Active = TRUE;
                    break;
                }

                Output = Output->Next;
            }

            if (!Found)
		xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Unable to activate"
			   " panel: no panel is present.\n");
            else
                Active = TRUE;
	}

        /* way too simplistic */
	if (pVia->ActiveDevice & VIA_DEVICE_TV) {
            Found = FALSE;
            
            Output = pVia->Outputs;
            while (Output) {
                if (Output->Type & OUTPUT_TV) {
                    if (Output->Sense) {
                        if (Output->Sense(Output))
                            Output->Active = TRUE;
                        else {
                            xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                                       "Unable to activate TV encoder: "
                                       "no cable attached.\n");
                            
                            Output->Active = FALSE;
                        }
                    } else
                        Output->Active = TRUE;
                    Found = TRUE;
                }
                
                Output = Output->Next;
            }
            
            if (!Found)
                xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Unable to activate"
                           " TV encoder: no TV encoder present.\n");
            else
                Active = TRUE;
        }

        if (pVia->ActiveDevice & VIA_DEVICE_CRT) {
            Active = TRUE;

            Output = pVia->Outputs;
            while (Output) {
                if (Output->Type & OUTPUT_CRT)
                    Output->Active = TRUE;
                
                Output = Output->Next;
            }
        }

        return Active;
    }
}


/*
 * If mode validation was able to handle multiple monrecs per screen,
 * then it could just cycle down the multiple monrecs when checking
 * ranges. But no, now we have to pull these sort of tricks.
 */
#if MAX_HSYNC != MAX_VREFRESH
#error That idiot libv assumes that MAX_HSYNC equals MAX_VREFRESH.
#endif

/*
 *
 */
static void
ViaMonitorDebug(int scrnIndex, MonPtr Monitor)
{
    int i;
    DisplayModePtr Mode;

    ViaDebug(scrnIndex, "Monitor: %s (%s - %s)\n", Monitor->id,
             Monitor->vendor, Monitor->model);

    for (i = 0; i < Monitor->nHsync; i++)
        ViaDebug(scrnIndex, "Horizontal Range %d: %f - %f\n", i,
                 Monitor->hsync[i].lo, Monitor->hsync[i].hi);

    for (i = 0; i < Monitor->nVrefresh; i++)
        ViaDebug(scrnIndex, "Vertical Range %d: %f - %f\n", i,
                 Monitor->vrefresh[i].lo, Monitor->vrefresh[i].hi);

    Mode = Monitor->Modes;
    while (Mode) {
        ViaPrintModeline(scrnIndex, Mode);
        Mode = Mode->next;
    }
    ViaDebug(scrnIndex, "Physical dimensions: %dmm x %dmm\n",
             Monitor->widthmm, Monitor->heightmm);

    ViaDebug(scrnIndex, "Gamma %f, %f, %f\n", Monitor->gamma.red,
             Monitor->gamma.green, Monitor->gamma.blue);

    if (Monitor->DDC)
        ViaDebug(scrnIndex, "Monitor has DDC\n");

#ifdef MONREC_HAS_REDUCED
    if (Monitor->reducedblanking)
        ViaDebug(scrnIndex, "Monitor allows for reduced blanking\n");
#endif
}

/*
 * Make sure that range is properly sorted and properly reduced.
 */
static int
ViaMonitorRangeSort(range *sortrange, int count)
{
    int i;

    /* fix hi < lo */
    for (i = 0; i < count; i++)
        if (sortrange[i].hi < sortrange[i].lo) {
            float temp = sortrange[i].hi;
            sortrange[i].hi = sortrange[i].lo;
            sortrange[i].lo = temp;
        }

    /* fix order */
    i = 0;
    do {
        for (i = 1; i < count; i++) {
            if (sortrange[i - 1].lo > sortrange[i].lo) {
                range temprange = sortrange[i];
                sortrange[i] = sortrange[i - 1];
                sortrange[i - 1] = temprange;
                break; /* restart */
            }
        }
    } while (i < count);

    /* fix overlaps */
    i = 1;
    while (i < count) {
        if (sortrange[i - 1].hi >= sortrange[i].lo) {
            int j;

            if (sortrange[i].hi > sortrange[i - 1].hi)
                sortrange[i - 1].hi = sortrange[i].hi;

            /* shrink list */
            for (j = i + 1; j < count; j++)
                sortrange[j - 1] = sortrange[j];

            count--;
        } else
            i++;
    }

    return count;
}

/*
 * Nasty business.
 */
static int
ViaMonitorRangesCombine(range *range1, int onecount,
                        range *range2, int twocount,
                        range *new)
{
    range one[MAX_HSYNC], two[MAX_HSYNC];
    float temp;
    int newcount = 0, i = 0, j = 0;

    /* simplest algorithm is destructive */
    memcpy(one, range1, MAX_HSYNC * sizeof(range));
    memcpy(two, range2, MAX_HSYNC * sizeof(range));

    onecount = ViaMonitorRangeSort(one, onecount);
    twocount = ViaMonitorRangeSort(two, twocount);

    while ((i < onecount) && (j < twocount) && (newcount < MAX_HSYNC)) {
        temp = max(one[i].lo, two[j].lo);

        while ((temp > one[i].hi) && (i < onecount))
            i++;

        while ((temp > two[j].hi) && (j < twocount))
            j++;

        if ((i == onecount) || (j == twocount))
            return newcount;

        new[newcount].lo = temp;
        new[newcount].hi = min(one[i].hi, two[j].hi);
        temp = new[newcount].hi;
        newcount++;

        if (one[i].hi == temp)
            i++;
        else
            one[i].lo = temp;

        if (two[j].hi == temp)
            j++;
        else
            two[j].lo = temp;
    }

    return newcount;
}

/*
 * Set pScrn->monitor by using the Outputs MonRecs
 */
void
ViaMonitorInitialise(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output = NULL, *CRT = NULL, *Temp;
    MonPtr New = NULL;

    VIAFUNC(pScrn->scrnIndex);

    /* Make sure only 1 Output with ModesExclusive is active */
    Output = NULL;
    Temp = pVia->Outputs;
    while (Temp) {
        if (Temp->Active && Temp->ModesExclusive) {
            if (!Output)
                Output = Temp;
            else if (Output->Type == OUTPUT_PANEL) {
                xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s can't be driven at "
                           "the same time as %s. Disabling %s\n", Temp->Name,
                           Output->Name, Temp->Name);
                Temp->Active = FALSE;
            } else if (Temp->Type == OUTPUT_PANEL) {
                xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s can't be driven at "
                           "the same time as %s. Disabling %s\n", Output->Name,
                           Temp->Name, Output->Name);
                Output->Active = FALSE;
                Output = Temp;
            }
        }
        Temp = Temp->Next;
    }

    /* From here on we assume there's only either TV or Panel and CRT. Multiple
     * TV encoders or panels are not considered. The algorithm here doesn't
     * handle more than two devices yet. */
    Temp = pVia->Outputs;
    while (Temp) {
        if (Temp->Active) {
            if ((Temp->Type == OUTPUT_TV) || (Temp->Type == OUTPUT_PANEL))
                Output = Temp;
            else if (Temp->Type == OUTPUT_CRT)
                CRT = Temp;
        }
        Temp = Temp->Next;
    }

    /* Check what needs to be done to combine both. */
    if (Output && CRT) {
        ViaMonitorDebug(pScrn->scrnIndex, Output->Monitor);
        ViaMonitorDebug(pScrn->scrnIndex, CRT->Monitor);

        New = xnfcalloc(1, sizeof(MonRec));
        memset(New, 0, sizeof(MonRec));

        /* first of all, check ranges */
        New->nHsync =
            ViaMonitorRangesCombine(CRT->Monitor->hsync, CRT->Monitor->nHsync,
                                    Output->Monitor->hsync,
                                    Output->Monitor->nHsync, New->hsync);

        New->nVrefresh =
            ViaMonitorRangesCombine(CRT->Monitor->vrefresh, CRT->Monitor->nVrefresh,
                                    Output->Monitor->vrefresh,
                                    Output->Monitor->nVrefresh, New->vrefresh);

        /* are the ranges of those two monitors compatible? */
        if (!New->nHsync || !New->nVrefresh) {
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Timing of %s is"
                       " incompatible with %s. Disabling %s\n", CRT->Name,
                       Output->Name, CRT->Name);
            xfree(New);
            New = NULL;

            CRT->Active = FALSE;
        } else { /* set up New further */
            /* name */
            int length = strlen(CRT->Monitor->id) +
                strlen(Output->Monitor->id) + 4;

            New->id = xnfcalloc(1, length);
            snprintf(New->id, length, "%s + %s", Output->Monitor->id,
                     CRT->Monitor->id);

            New->vendor = "";
            New->model = "";

            /* add modes */
            if (!CRT->ModesExclusive)
                ViaModesCopyAdd(New, Output->Monitor->Modes);
            if (!Output->ModesExclusive)
                ViaModesCopyAdd(New, CRT->Monitor->Modes);

            /* gamma */
            if (Output->Monitor->gamma.red || Output->Monitor->gamma.green ||
                Output->Monitor->gamma.blue) {
                New->gamma.red = Output->Monitor->gamma.red;
                New->gamma.green = Output->Monitor->gamma.green;
                New->gamma.blue = Output->Monitor->gamma.blue;
            } else {
                New->gamma.red = CRT->Monitor->gamma.red;
                New->gamma.green = CRT->Monitor->gamma.green;
                New->gamma.blue = CRT->Monitor->gamma.blue;
            }

            /* resolution */
            if (Output->Monitor->widthmm) {
                New->widthmm = Output->Monitor->widthmm;
                New->heightmm = Output->Monitor->heightmm;
            } else {
                New->widthmm = CRT->Monitor->widthmm;
                New->heightmm = CRT->Monitor->heightmm;
            }

#ifdef MONREC_HAS_REDUCED
            if (Output->Monitor->reducedblanking && CRT->Monitor->reducedblanking)
                New->reducedblanking = TRUE;
#endif

            ViaMonitorDebug(pScrn->scrnIndex, New);
        }
    }

    if (New) {
        pVia->Monitor = New; /* so that we can free this */
        pScrn->monitor = New;
    } else if (Output) {
        pScrn->monitor = Output->Monitor;
        /* We don't have a CRT here but we do want to see the Configged modes */
        if (!Output->ModesExclusive)
            ViaModesCopyAdd(pScrn->monitor, pScrn->confScreen->monitor->Modes);
    } else if (CRT)
        pScrn->monitor = CRT->Monitor;
    else { /* should never happen */
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "%s: libv is a git!\n",
                   __func__);
        pScrn->monitor = pScrn->confScreen->monitor;
    }
    ViaMonitorDebug(pScrn->scrnIndex, pScrn->monitor);

    ViaOutputsPrint(pScrn);
    ViaOutputsFindDotclockMaster(pScrn);
}

/*
 * Checks for limitations imposed by the available VGA timing registers.
 *
 */
static ModeStatus
ViaModePrimaryVGAValid(ScrnInfoPtr pScrn, DisplayModePtr mode)
{
    VIAFUNC(pScrn->scrnIndex);

    if (mode->CrtcHTotal > 4100) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcHTotal out of range.\n");
	return MODE_BAD_HVALUE;
    }

    if (mode->CrtcHDisplay > 2048) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcHDisplay out of range.\n");
	return MODE_BAD_HVALUE;
    }

    if (mode->CrtcHBlankStart > 2048) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcHBlankStart out of range.\n");
	return MODE_BAD_HVALUE;
    }

    if ((mode->CrtcHBlankEnd - mode->CrtcHBlankStart) > 1025) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcHBlankEnd out of range.\n");
	return MODE_HBLANK_WIDE;
    }

    if (mode->CrtcHSyncStart > 4095) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcHSyncStart out of range.\n");
	return MODE_BAD_HVALUE;
    }

    if ((mode->CrtcHSyncEnd - mode->CrtcHSyncStart) > 256) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcHSyncEnd out of range.\n");
	return MODE_HSYNC_WIDE;
    }

    if (mode->CrtcVTotal > 2049) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcVTotal out of range.\n");
	return MODE_BAD_VVALUE;
    }

    if (mode->CrtcVDisplay > 2048) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcVDisplay out of range.\n");
	return MODE_BAD_VVALUE;
    }

    if (mode->CrtcVSyncStart > 2047) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcVSyncStart out of range.\n");
	return MODE_BAD_VVALUE;
    }

    if ((mode->CrtcVSyncEnd - mode->CrtcVSyncStart) > 16) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcVSyncEnd out of range.\n");
	return MODE_VSYNC_WIDE;
    }

    if (mode->CrtcVBlankStart > 2048) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcVBlankStart out of range.\n");
	return MODE_BAD_VVALUE;
    }

    if ((mode->CrtcVBlankEnd - mode->CrtcVBlankStart) > 257) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcVBlankEnd out of range.\n");
	return MODE_VBLANK_WIDE;
    }

    return MODE_OK;
}

/*
 *
 */
static ModeStatus
ViaModeSecondaryVGAValid(ScrnInfoPtr pScrn, DisplayModePtr mode)
{
    VIAFUNC(pScrn->scrnIndex);

    if (mode->CrtcHTotal > 4096) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcHTotal out of range.\n");
	return MODE_BAD_HVALUE;
    }

    if (mode->CrtcHDisplay > 2048) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcHDisplay out of range.\n");
	return MODE_BAD_HVALUE;
    }

    if (mode->CrtcHBlankStart > 2048) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcHBlankStart out of range.\n");
	return MODE_BAD_HVALUE;
    }

    if (mode->CrtcHBlankEnd > 4096) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcHBlankEnd out of range.\n");
	return MODE_HBLANK_WIDE;
    }

    if (mode->CrtcHSyncStart > 2047) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcHSyncStart out of range.\n");
	return MODE_BAD_HVALUE;
    }

    if ((mode->CrtcHSyncEnd - mode->CrtcHSyncStart) > 512) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcHSyncEnd out of range.\n");
	return MODE_HSYNC_WIDE;
    }

    if (mode->CrtcVTotal > 2048) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcVTotal out of range.\n");
	return MODE_BAD_VVALUE;
    }

    if (mode->CrtcVDisplay > 2048) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcVDisplay out of range.\n");
	return MODE_BAD_VVALUE;
    }

    if (mode->CrtcVBlankStart > 2048) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcVBlankStart out of range.\n");
	return MODE_BAD_VVALUE;
    }

    if (mode->CrtcVBlankEnd > 2048) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcVBlankEnd out of range.\n");
	return MODE_VBLANK_WIDE;
    }

    if (mode->CrtcVSyncStart > 2047) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcVSyncStart out of range.\n");
	return MODE_BAD_VVALUE;
    }

    if ((mode->CrtcVSyncEnd - mode->CrtcVSyncStart) > 32) {
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "CrtcVSyncEnd out of range.\n");
	return MODE_VSYNC_WIDE;
    }

    return MODE_OK;
}

/*
 *
 */
ModeStatus 
ViaValidMode(int scrnIndex, DisplayModePtr mode, Bool verbose, int flags)
{
    ScrnInfoPtr pScrn = xf86Screens[scrnIndex];
    VIAPtr pVia = VIAPTR(pScrn);
    ModeStatus ret;
    CARD32 temp;

    ViaDebug(scrnIndex, "%s: Validating %s (%d)\n", __func__, mode->name,
             mode->Clock);

    if (pVia->IsSecondary)
	ret = ViaModeSecondaryVGAValid(pScrn, mode);
    else
	ret = ViaModePrimaryVGAValid(pScrn, mode);

    if (ret != MODE_OK)
	return ret;

    ret = ViaOutputsModeValid(pScrn, mode);
    if (ret != MODE_OK) {
        xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Mode \"%s\" not supported by"
                   " output devices.\n", mode->name);
        return ret;
    }

    temp = mode->CrtcHDisplay * mode->CrtcVDisplay *  mode->VRefresh *
	(pScrn->bitsPerPixel >> 3);
    if (pVia->Bandwidth < temp) {
	xf86DrvMsg(scrnIndex, X_INFO, "Required bandwidth is not available. (%u > %u)\n",
		   (unsigned) temp, (unsigned) pVia->Bandwidth);
	return MODE_CLOCK_HIGH; /* since there is no MODE_BANDWIDTH */
    }

    return MODE_OK;
}

/*
 *
 */
static void
ViaModePrimaryVGA(ScrnInfoPtr pScrn, DisplayModePtr mode)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    CARD16 temp;

    VIAFUNC(pScrn->scrnIndex);
    ViaDebug(pScrn->scrnIndex, "Setting up %s\n", mode->name);

    ViaCrtcMask(hwp, 0x11, 0x00, 0x80); /* modify starting address */
    ViaCrtcMask(hwp, 0x03, 0x80, 0x80); /* enable vertical retrace access */
    hwp->writeSeq(hwp, 0x10, 0x01); /* unlock extended registers */
    ViaCrtcMask(hwp, 0x47, 0x00, 0x01); /* unlock CRT registers */

    /* Set Misc Register */
    temp = 0x23;
    if (mode->Flags & V_NHSYNC)
	temp |= 0x40;
    if (mode->Flags & V_NVSYNC)
	temp |= 0x80;
    temp |= 0x0C; /* Undefined/external clock */
    hwp->writeMiscOut(hwp, temp);

    /* Sequence registers */
    hwp->writeSeq(hwp, 0x00, 0x00);

    /* if (mode->Flags & V_CLKDIV2)
	hwp->writeSeq(hwp, 0x01, 0x09);
    else */
    ViaSeqMask(hwp, 0x01, 0x01, 0xDF);

    hwp->writeSeq(hwp, 0x03, 0x00);

    /* 8bit lut / 80 text columns / wrap-around / extended mode */
    ViaSeqMask(hwp, 0x15, 0xA2, 0xE2);

    /* 555/565 -- bpp */
    switch (pScrn->bitsPerPixel) {
    case 8:
	ViaSeqMask(hwp, 0x15, 0x00, 0x1C);
        break;
    case 16:
	ViaSeqMask(hwp, 0x15, 0x14, 0x1C);
        break;
    case 24:
    case 32:
	ViaSeqMask(hwp, 0x15, 0x0C, 0x1C);
        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Unhandled bitdepth: %d\n",
                   pScrn->bitsPerPixel);
        break;
    }

    ViaSeqMask(hwp, 0x16, 0x08, 0xBF);
    ViaSeqMask(hwp, 0x17, 0x1F, 0xFF);
    ViaSeqMask(hwp, 0x18, 0x4E, 0xFF);
    ViaSeqMask(hwp, 0x1A, 0x08, 0xFD);

    /* graphics registers */
    hwp->writeGr(hwp, 0x00, 0x00);
    hwp->writeGr(hwp, 0x01, 0x00);
    hwp->writeGr(hwp, 0x02, 0x00);
    hwp->writeGr(hwp, 0x03, 0x00);
    hwp->writeGr(hwp, 0x04, 0x00);
    hwp->writeGr(hwp, 0x05, 0x40);
    hwp->writeGr(hwp, 0x06, 0x05);
    hwp->writeGr(hwp, 0x07, 0x0F);
    hwp->writeGr(hwp, 0x08, 0xFF);

    ViaGrMask(hwp, 0x20, 0, 0xFF);
    ViaGrMask(hwp, 0x21, 0, 0xFF);
    ViaGrMask(hwp, 0x22, 0, 0xFF);

    /* attribute registers */
    hwp->writeAttr(hwp, 0x00, 0x00);
    hwp->writeAttr(hwp, 0x01, 0x01);
    hwp->writeAttr(hwp, 0x02, 0x02);
    hwp->writeAttr(hwp, 0x03, 0x03);
    hwp->writeAttr(hwp, 0x04, 0x04);
    hwp->writeAttr(hwp, 0x05, 0x05);
    hwp->writeAttr(hwp, 0x06, 0x06);
    hwp->writeAttr(hwp, 0x07, 0x07);
    hwp->writeAttr(hwp, 0x08, 0x08);
    hwp->writeAttr(hwp, 0x09, 0x09);
    hwp->writeAttr(hwp, 0x0A, 0x0A);
    hwp->writeAttr(hwp, 0x0B, 0x0B);
    hwp->writeAttr(hwp, 0x0C, 0x0C);
    hwp->writeAttr(hwp, 0x0D, 0x0D);
    hwp->writeAttr(hwp, 0x0E, 0x0E);
    hwp->writeAttr(hwp, 0x0F, 0x0F);
    hwp->writeAttr(hwp, 0x10, 0x41);
    hwp->writeAttr(hwp, 0x11, 0xFF);
    hwp->writeAttr(hwp, 0x12, 0x0F);
    hwp->writeAttr(hwp, 0x13, 0x00);
    hwp->writeAttr(hwp, 0x14, 0x00);

    /* Crtc registers */
    /* horizontal total : 4100 */
    ViaDebug(pScrn->scrnIndex, "CrtcHTotal: 0x%03X\n", mode->CrtcHTotal);
    temp = (mode->CrtcHTotal >> 3) - 5;
    hwp->writeCrtc(hwp, 0x00, temp & 0xFF);
    ViaCrtcMask(hwp, 0x36, temp >> 5, 0x08);
    
    /* horizontal address : 2048 */
    ViaDebug(pScrn->scrnIndex, "CrtcHDisplay: 0x%03X\n", mode->CrtcHDisplay);
    temp = (mode->CrtcHDisplay >> 3) - 1;
    hwp->writeCrtc(hwp, 0x01, temp & 0xFF);

    /* horizontal blanking start : 2048 */
    ViaDebug(pScrn->scrnIndex, "CrtcHBlankStart: 0x%03X\n", mode->CrtcHBlankStart);
    temp = (mode->CrtcHBlankStart >> 3) - 1;
    hwp->writeCrtc(hwp, 0x02, temp & 0xFF);

    /* horizontal blanking end : start + 1025 */
    ViaDebug(pScrn->scrnIndex, "CrtcHBlankEnd: 0x%03X\n", mode->CrtcHBlankEnd);
    temp = (mode->CrtcHBlankEnd >> 3) - 1;
    ViaCrtcMask(hwp, 0x03, temp, 0x1F);
    ViaCrtcMask(hwp, 0x05, temp << 2, 0x80);
    ViaCrtcMask(hwp, 0x33, temp >> 1, 0x20);

    /* CrtcHSkew ??? */

    /* horizontal sync start : 4095 */
    ViaDebug(pScrn->scrnIndex, "CrtcHSyncStart: 0x%03X\n", mode->CrtcHSyncStart);
    temp = mode->CrtcHSyncStart >> 3;
    hwp->writeCrtc(hwp, 0x04, temp & 0xFF);
    ViaCrtcMask(hwp, 0x33, temp >> 4, 0x10);

    /* horizontal sync end : start + 256 */
    ViaDebug(pScrn->scrnIndex, "CrtcHSyncEnd: 0x%03X\n",  mode->CrtcHSyncEnd);
    temp = mode->CrtcHSyncEnd >> 3;
    ViaCrtcMask(hwp, 0x05, temp, 0x1F);

    /* vertical total : 2049 */
    ViaDebug(pScrn->scrnIndex, "CrtcVTotal: 0x%03X\n", mode->CrtcVTotal);
    temp = mode->CrtcVTotal - 2;
    hwp->writeCrtc(hwp, 0x06, temp & 0xFF);
    ViaCrtcMask(hwp, 0x07, temp >> 8, 0x01);
    ViaCrtcMask(hwp, 0x07, temp >> 4, 0x20);
    ViaCrtcMask(hwp, 0x35, temp >> 10, 0x01);

    /* vertical address : 2048 */
    ViaDebug(pScrn->scrnIndex, "CrtcVDisplay: 0x%03X\n", mode->CrtcVDisplay);
    temp = mode->CrtcVDisplay - 1;
    hwp->writeCrtc(hwp, 0x12, temp & 0xFF);
    ViaCrtcMask(hwp, 0x07, temp >> 7, 0x02);
    ViaCrtcMask(hwp, 0x07, temp >> 3, 0x40);
    ViaCrtcMask(hwp, 0x35, temp >> 8, 0x04);

    /* Primary starting address -> 0x00, adjustframe does the rest */
    hwp->writeCrtc(hwp, 0x0C, 0x00);
    hwp->writeCrtc(hwp, 0x0D, 0x00);
    hwp->writeCrtc(hwp, 0x34, 0x00);
    ViaCrtcMask(hwp, 0x48, 0x00, 0x03);

    /* vertical sync start : 2047 */
    ViaDebug(pScrn->scrnIndex, "CrtcVSyncStart: 0x%03X\n", mode->CrtcVSyncStart);
    temp = mode->CrtcVSyncStart;
    hwp->writeCrtc(hwp, 0x10, temp & 0xFF);
    ViaCrtcMask(hwp, 0x07, temp >> 6, 0x04);
    ViaCrtcMask(hwp, 0x07, temp >> 2, 0x80);
    ViaCrtcMask(hwp, 0x35, temp >> 9, 0x02);

    /* vertical sync end : start + 16 -- other bits someplace? */
    ViaDebug(pScrn->scrnIndex, "CrtcVSyncEnd: 0x%03X\n", mode->CrtcVSyncEnd);
    ViaCrtcMask(hwp, 0x11, mode->CrtcVSyncEnd, 0x0F);

    /* line compare: We are not doing splitscreen so 0x3FFF */
    hwp->writeCrtc(hwp, 0x18, 0xFF);
    ViaCrtcMask(hwp, 0x07, 0x10, 0x10);
    ViaCrtcMask(hwp, 0x09, 0x40, 0x40);
    ViaCrtcMask(hwp, 0x33, 0x07, 0x06);
    ViaCrtcMask(hwp, 0x35, 0x10, 0x10);

    /* zero Maximum scan line */
    ViaCrtcMask(hwp, 0x09, 0x00, 0x1F);
    hwp->writeCrtc(hwp, 0x14, 0x00);

    /* vertical blanking start : 2048 */
    ViaDebug(pScrn->scrnIndex, "CrtcVBlankStart: 0x%03X\n", mode->CrtcVBlankStart);
    temp = mode->CrtcVBlankStart - 1;
    hwp->writeCrtc(hwp, 0x15, temp & 0xFF);
    ViaCrtcMask(hwp, 0x07, temp >> 5, 0x08);
    ViaCrtcMask(hwp, 0x09, temp >> 4, 0x20);
    ViaCrtcMask(hwp, 0x35, temp >> 7, 0x08);

    /* vertical blanking end : start + 257 */
    ViaDebug(pScrn->scrnIndex, "CrtcVBlankEnd: 0x%03X\n", mode->CrtcVBlankEnd);
    temp = mode->CrtcVBlankEnd - 1;
    hwp->writeCrtc(hwp, 0x16, temp & 0xFF);

    /* some leftovers */
    hwp->writeCrtc(hwp, 0x08, 0x00);
    ViaCrtcMask(hwp, 0x32, 0, 0xFF); /* ? */
    ViaCrtcMask(hwp, 0x33, 0, 0xC8);
    
    /* offset */
    temp = (pScrn->displayWidth * (pScrn->bitsPerPixel >> 3)) >> 3;
    /* Make sure that this is 32byte aligned */
    if (temp & 0x03) {
	temp += 0x03;
	temp &= ~0x03;
    }
    ViaDebug(pScrn->scrnIndex, "Offset: 0x%03X\n", temp);
    hwp->writeCrtc(hwp, 0x13, temp & 0xFF);
    ViaCrtcMask(hwp, 0x35, temp >> 3, 0xE0);

    /* fetch count */
    temp = (mode->CrtcHDisplay * (pScrn->bitsPerPixel >> 3)) >> 3;
    /* Make sure that this is 32byte aligned */
    if (temp & 0x03) {
	temp += 0x03;
	temp &= ~0x03;
    }
    ViaDebug(pScrn->scrnIndex, "Fetch Count: 0x%03X\n", temp);
    hwp->writeSeq(hwp, 0x1C, (temp >> 1) & 0xFF);
    ViaSeqMask(hwp, 0x1D, temp >> 9, 0x03);

    /* some leftovers */
    ViaCrtcMask(hwp, 0x32, 0, 0xFF);
    ViaCrtcMask(hwp, 0x33, 0, 0xC8);
}


/*
 *
 */
void
ViaModePrimary(ScrnInfoPtr pScrn, DisplayModePtr mode)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output = pVia->Outputs;

    VIAFUNC(pScrn->scrnIndex);

    ViaCrtcMask(hwp, 0x17, 0x00, 0x80);

    /* Clean Second Path Status */
    ViaCrtcMask(hwp, 0x6A, 0x00, 0xF6);
    hwp->writeCrtc(hwp, 0x6B, 0x00);
    hwp->writeCrtc(hwp, 0x6C, 0x00);
    hwp->writeCrtc(hwp, 0x93, 0x00);

    ViaModePrimaryVGA(pScrn, mode);

    /* Enable MMIO & PCI burst (1 wait state) */
    ViaSeqMask(hwp, 0x1A, 0x06, 0x06);

    ViaSetPrimaryFIFO(pScrn, mode);

    if (pVia->Chipset == VT3108) /* Enable CRTC1 Prefetch */
        ViaCrtcMask(hwp, 0x33, 0x08, 0x08);

    if (pScrn->bitsPerPixel == 8) /* 8bit is palette, not gamma */
        ViaOutputsGamma(pScrn, FALSE);
    else
        ViaOutputsGamma(pScrn, TRUE);

    /* Work around panel blob. */
    pVia->PanelClock = 0x00;

    while (Output) {
        if (Output->Active)
            ViaOutputMode(pScrn, Output, mode);
        else {
            if (Output->Power)
                Output->Power(Output, FALSE);
            ViaBusPower(pScrn, Output, FALSE);
        }

        Output = Output->Next;
    }

    if (pVia->ClockSlave)
        ViaDotclockPrimarySlave(pScrn);
    else {
        if (pVia->PanelClock)
            ViaSetPrimaryDotclock(pScrn, pVia->PanelClock);
        else
            ViaSetPrimaryDotclock(pScrn, ViaPLLGenerate(pScrn, mode->Clock));

	ViaSetUseExternalClock(pScrn);
	ViaCrtcMask(hwp, 0x6B, 0x00, 0x01);
    }

    ViaCrtcMask(hwp, 0x17, 0x80, 0x80);
}

/*
 *
 */
static void
ViaModeSecondaryVGA(ScrnInfoPtr pScrn, DisplayModePtr mode)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    CARD16 temp;

    VIAFUNC(pScrn->scrnIndex);

    /* bpp */
    switch (pScrn->bitsPerPixel) {
    case 8:
        ViaCrtcMask(hwp, 0x67, 0x00, 0xC0);
        break;
    case 16:
        ViaCrtcMask(hwp, 0x67, 0x40, 0xC0);
        break;
    case 24:
    case 32:
        ViaCrtcMask(hwp, 0x67, 0x80, 0xC0);
        break;
    default:
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Unhandled bitdepth: %d\n",
                   pScrn->bitsPerPixel);
        break;
    }

    /* Crtc registers */
    /* horizontal total : 4096 */
    ViaDebug(pScrn->scrnIndex, "CrtcHTotal: 0x%03X\n", mode->CrtcHTotal);
    temp = mode->CrtcHTotal - 1;
    hwp->writeCrtc(hwp, 0x50, temp & 0xFF);
    ViaCrtcMask(hwp, 0x55, temp >> 8, 0x0F);

    /* horizontal address : 2048 */
    ViaDebug(pScrn->scrnIndex, "CrtcHDisplay: 0x%03X\n", mode->CrtcHDisplay);
    temp = mode->CrtcHDisplay - 1;
    hwp->writeCrtc(hwp, 0x51, temp & 0xFF);
    ViaCrtcMask(hwp, 0x55, temp >> 4, 0x70);

    /* horizontal blanking start : 2048 */
    ViaDebug(pScrn->scrnIndex, "CrtcHBlankStart: 0x%03X\n", mode->CrtcHBlankStart);
    if (mode->CrtcHBlankStart != mode->CrtcHDisplay) /* FIX ME */
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Caught X working around an old VGA "
		   "limitation (HBlankStart).\n");
    temp = mode->CrtcHDisplay - 1;
    hwp->writeCrtc(hwp, 0x52, temp & 0xFF);
    ViaCrtcMask(hwp, 0x54, temp >> 8, 0x07);

    /* horizontal blanking end : 4096 */
    ViaDebug(pScrn->scrnIndex, "CrtcHBlankEnd: 0x%03X\n", mode->CrtcHBlankEnd);
    if (mode->CrtcHBlankEnd != mode->CrtcHTotal) /* FIX ME */
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Caught X working around an old VGA "
		   "limitation (HBlankEnd).\n");
    temp = mode->CrtcHTotal - 1;
    hwp->writeCrtc(hwp, 0x53, temp & 0xFF);
    ViaCrtcMask(hwp, 0x54, temp >> 5, 0x38);
    ViaCrtcMask(hwp, 0x5D, temp >> 5, 0x40);

    /* horizontal sync start : 2047 */
    ViaDebug(pScrn->scrnIndex, "CrtcHSyncStart: 0x%03X\n", mode->CrtcHSyncStart);
    temp = mode->CrtcHSyncStart;
    hwp->writeCrtc(hwp, 0x56, temp & 0xFF);
    ViaCrtcMask(hwp, 0x54, temp >> 2, 0xC0);
    ViaCrtcMask(hwp, 0x5C, temp >> 3, 0x80);

    /* horizontal sync end : sync start + 512 */
    ViaDebug(pScrn->scrnIndex, "CrtcHSyncEnd: 0x%03X\n", mode->CrtcHSyncEnd);
    temp = mode->CrtcHSyncEnd;
    hwp->writeCrtc(hwp, 0x57, temp & 0xFF);
    ViaCrtcMask(hwp, 0x5C, temp >> 2, 0x40);
    
    /* vertical total : 2048 */
    ViaDebug(pScrn->scrnIndex, "CrtcVTotal: 0x%03X\n", mode->CrtcVTotal);
    temp = mode->CrtcVTotal - 1;
    hwp->writeCrtc(hwp, 0x58, temp & 0xFF);
    ViaCrtcMask(hwp, 0x5D, temp >> 8, 0x07);

    /* vertical address : 2048 */
    ViaDebug(pScrn->scrnIndex, "CrtcVDisplay: 0x%03X\n", mode->CrtcVDisplay);
    temp = mode->CrtcVDisplay - 1;
    hwp->writeCrtc(hwp, 0x59, temp & 0xFF);
    ViaCrtcMask(hwp, 0x5D, temp >> 5, 0x38);

    /* vertical blanking start : 2048 */
    ViaDebug(pScrn->scrnIndex, "CrtcVBlankStart: 0x%03X\n", mode->CrtcVBlankStart);
    if (mode->CrtcVBlankStart != mode->CrtcVDisplay) /* FIX ME */
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Caught X working around an old VGA "
		   "limitation (VBlankStart).\n");
    temp = mode->CrtcVDisplay - 1;
    hwp->writeCrtc(hwp, 0x5A, temp & 0xFF);
    ViaCrtcMask(hwp, 0x5C, temp >> 8, 0x07);

    /* vertical blanking end : 2048 */
    ViaDebug(pScrn->scrnIndex, "CrtcVBlankEnd: 0x%03X\n", mode->CrtcVBlankEnd);
    if (mode->CrtcVBlankEnd != mode->CrtcVTotal) /* FIX ME */
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Caught X working around an old VGA "
		   "limitation (VBlankEnd).\n");
    temp = mode->CrtcVTotal - 1;
    hwp->writeCrtc(hwp, 0x5B, temp & 0xFF);
    ViaCrtcMask(hwp, 0x5C, temp >> 5, 0x38);

    /* vertical sync start : 2047 */
    ViaDebug(pScrn->scrnIndex, "CrtcVSyncStart: 0x%03X\n", mode->CrtcVSyncStart);
    temp = mode->CrtcVSyncStart;
    hwp->writeCrtc(hwp, 0x5E, temp & 0xFF);
    ViaCrtcMask(hwp, 0x5F, temp >> 3, 0xE0);

    /* vertical sync end : start + 32 */
    ViaDebug(pScrn->scrnIndex, "CrtcVSyncEnd: 0x%03X\n", mode->CrtcVSyncEnd);
    temp = mode->CrtcVSyncEnd;
    ViaCrtcMask(hwp, 0x5F, temp, 0x1F);

    /* offset */
    temp = (pScrn->displayWidth * (pScrn->bitsPerPixel >> 3)) >> 3;
    if (temp & 0x03) { /* Make sure that this is 32byte aligned */
	temp += 0x03;
	temp &= ~0x03;
    }
    ViaDebug(pScrn->scrnIndex, "Offset: 0x%03X\n", temp);
    hwp->writeCrtc(hwp, 0x66, temp & 0xFF);
    ViaCrtcMask(hwp, 0x67, temp >> 8, 0x03);

    /* fetch count */
    temp = (mode->CrtcHDisplay * (pScrn->bitsPerPixel >> 3)) >> 3;
    /* Make sure that this is 32byte aligned */
    if (temp & 0x03) {
	temp += 0x03;
	temp &= ~0x03;
    }
    ViaDebug(pScrn->scrnIndex, "Fetch Count: 0x%03X\n", temp);
    hwp->writeCrtc(hwp, 0x65, (temp >> 1) & 0xFF);
    ViaCrtcMask(hwp, 0x67, temp >> 7, 0x0C);
}

/*
 *
 */
void
ViaModeSecondary(ScrnInfoPtr pScrn, DisplayModePtr mode)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output = pVia->Outputs;

    VIAFUNC(pScrn->scrnIndex);

    /* Turn off Screen */
    ViaCrtcMask(hwp, 0x17, 0x00, 0x80);

    ViaModeSecondaryVGA(pScrn, mode);
    ViaSetSecondaryFIFO(pScrn, mode);

    /* fix me */
    pVia->PanelClock = 0x00;
    while (Output) {
        if (Output->Active && Output->Mode)
            Output->Mode(Output, mode);
        else {
            if (Output->Power)
                Output->Power(Output, FALSE);
            ViaBusPower(pScrn, Output, FALSE);
        }
        Output = Output->Next;
    }

    /* VT3122A2 apparently doesn't like this */
    if ((pVia->Chipset != VT3122) || (pVia->ChipRev != 0x02))
	ViaCrtcMask(hwp, 0x6C, 0x00, 0x1E);

    if (pVia->PanelClock)
        ViaSetSecondaryDotclock(pScrn, pVia->PanelClock);
    else
        ViaSetSecondaryDotclock(pScrn, ViaPLLGenerate(pScrn, mode->Clock));
    ViaSetUseExternalClock(pScrn);

    ViaCrtcMask(hwp, 0x17, 0x80, 0x80);
}
