/*
 * Copyright 2005-2006 Luc Verhaegen.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*
 *
 * Trivial CRT ViaOutput implementation.
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#include <unistd.h>
#endif

#include "via_vgahw.h"
#include "via_id.h"
#include "via_mode.h"

/*
 * Output->Private
 */
struct CRTPrivate {
    Bool Sense; /* used for init only */

    int BandWidth; /* Limit modes - Config or EDID provided. */
};

/*
 *
 */
static void
CRTPrivateDestroy(struct ViaOutput *Output)
{
    xfree(Output->Private);

    /* So we won't try to destroy this twice */
    Output->PrivateDestroy = NULL;
}

/*
 *
 */
static struct CRTPrivate *
CRTPrivateCreate(struct ViaOutput *Output)
{
    VIAFUNC(Output->scrnIndex);

    Output->PrivSize = sizeof(struct CRTPrivate);
    Output->Private =  xnfcalloc(1, Output->PrivSize);
    memset(Output->Private, 0, Output->PrivSize);

    Output->PrivateDestroy = CRTPrivateDestroy;

    return Output->Private;
}

/*
 * Option handling.
 */
enum CRTOpts {
    OPTION_CRTSENSE,
    OPTION_CRTBANDWIDTH
};

static OptionInfoRec CRTOptions[] =
{
    {OPTION_CRTSENSE,     "CRTSense",     OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_CRTBANDWIDTH, "CRTBandWidth", OPTV_INTEGER, {0}, FALSE},
    {-1,  NULL,  OPTV_NONE,  {0},  FALSE }
};

/*
 *
 */
static OptionInfoPtr
CRTGetOptions(ScrnInfoPtr pScrn, struct CRTPrivate *Private)
{
    OptionInfoPtr  Options;
    
    Options = xnfalloc(sizeof(CRTOptions));
    memcpy(Options, CRTOptions, sizeof(CRTOptions));

    xf86ProcessOptions(pScrn->scrnIndex, pScrn->options, Options);

    if (!xf86ReturnOptValBool(Options, OPTION_CRTSENSE, TRUE)) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Disabling CRT Sensing."
                   " CRT is considered always attached.\n");
        Private->Sense = FALSE;
    } else
        Private->Sense = TRUE;

    Private->BandWidth = 0;
    xf86GetOptValInteger(Options, OPTION_CRTBANDWIDTH, &Private->BandWidth);
    if (Private->BandWidth && (Private->BandWidth < 1000)) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Option CRTBandWidth needs "
                   "to be provided in kHz not MHz.\n", __func__);
        Private->BandWidth = 0;
    }

    xfree(Options);
    return NULL;
}

/*
 * Not functional on the VT3122Ax.
 */
static Bool
CRTSense(struct ViaOutput *Output)
{
    ScrnInfoPtr pScrn = xf86Screens[Output->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    CARD8 SR01, CR36;
    Bool Found = FALSE;

    VIAFUNC(pScrn->scrnIndex);

    SR01 = hwp->readSeq(hwp, 0x01);
    CR36 = hwp->readCrtc(hwp, 0x36);

    ViaSeqMask(hwp, 0x01, 0x00, 0x20);
    ViaCrtcMask(hwp, 0x36, 0x00, 0xF0);

    ViaSeqMask(hwp, 0x40, 0x80, 0x80);

    usleep(1);

    if (hwp->readST00(hwp) & 0x10)
        Found = TRUE;

    ViaSeqMask(hwp, 0x40, 0x00, 0x80);

    hwp->writeSeq(hwp, 0x01, SR01);
    hwp->writeCrtc(hwp, 0x36, CR36);

    return Found;
}

/*
 * Sync/Unsync
 */
static void
CRTPower(struct ViaOutput *Output, Bool On)
{
    ScrnInfoPtr pScrn = xf86Screens[Output->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    if (On)
        ViaCrtcMask(hwp, 0x36, 0x00, 0x30);
    else
        ViaCrtcMask(hwp, 0x36, 0x30, 0x30);
}

/*
 * Bandwidth check is missing from X modevalidation.
 */
static ModeStatus
CRTModeValid(struct ViaOutput *Output, DisplayModePtr mode)
{
    struct CRTPrivate *Private = Output->Private;

    VIAFUNC(Output->scrnIndex);

    if (Private->BandWidth) {
        /* In our driver, the real clock is pretty damn close to the requested
         * clock, so ignore SynthClock. */
        if (mode->Clock > Private->BandWidth) {
            char *name;
            
            if (Output->Monitor)
                name = Output->Monitor->id;
            else
                name = Output->Name;

            xf86DrvMsg(Output->scrnIndex, X_INFO, 
                       "%s: \"%s\" dotclock exceeds Monitor bandwidth (%dkHz > %dkHz).\n",
                       name, mode->name, mode->Clock, Private->BandWidth);

            return MODE_CLOCK_HIGH;
        }

    }

    return MODE_OK;
}

/*
 * Empty.
 */
static void
CRTMode(struct ViaOutput *Output, DisplayModePtr mode)
{
    VIAFUNC(Output->scrnIndex);
}

/*
 * Initialise the MonRec for the CRT. Add in some configured
 * values as well.
 */
static MonPtr
CRTMonitor(struct ViaOutput *Output)
{
    ScrnInfoPtr pScrn = xf86Screens[Output->scrnIndex];
    VIAPtr pVia = VIAPTR(pScrn);
    MonPtr Config = pScrn->confScreen->monitor;
    xf86MonPtr DDC = NULL;
    MonPtr Monitor;

    Monitor = xnfcalloc(1, sizeof(MonRec));
    memset(Monitor, 0, sizeof(MonRec));

    if (pVia->pI2CBus1)
        DDC = xf86DoEDID_DDC2(pScrn->scrnIndex, pVia->pI2CBus1);

    if (DDC) {
        xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Printing CRT EDID (DDC) info:\n");
        xf86PrintEDID(DDC);
        ViaDDCMonitorSet(pScrn->scrnIndex, Monitor, DDC);

        /* did the user configure a Monitor? */
        if (strncmp("<default Monitor>", Config->id, 18)) {
            /* keep DDC name, vendor, model */

            /* Use configured H ranges */
            if (Config->nHsync) {
                int i;

                Monitor->nHsync = Config->nHsync;
                for (i = 0; i < Config->nHsync; i++) {
                    Monitor->hsync[i].hi = Config->hsync[i].hi;
                    Monitor->hsync[i].lo = Config->hsync[i].lo;
                }
            }

            /* Use configured V ranges */
            if (Config->nVrefresh) {
                int i;

                Monitor->nVrefresh = Config->nVrefresh;
                for (i = 0; i < Config->nVrefresh; i++) {
                    Monitor->vrefresh[i].hi = Config->vrefresh[i].hi;
                    Monitor->vrefresh[i].lo = Config->vrefresh[i].lo;
                }
            }

            /* add options */
            Monitor->options = Config->options;

            if (Config->gamma.red || Config->gamma.green || Config->gamma.blue) {
                Monitor->gamma.red = Config->gamma.red;
                Monitor->gamma.green = Config->gamma.green;
                Monitor->gamma.blue = Config->gamma.blue;
            }

#ifdef MONREC_HAS_REDUCED
            /* if user wants reduced, he gets reduced */
            if (Config->reducedblanking)
                Monitor->reducedblanking = TRUE;
#endif
        }

        { /* use max dotclock from EDID ranges */
            struct CRTPrivate *Private = Output->Private;

            if (!Private->BandWidth) {
                int i;

                for (i = 0; i < DET_TIMINGS; i++)
                    if (DDC->det_mon[i].type == DS_RANGES) {
                        Private->BandWidth =
                            DDC->det_mon[i].section.ranges.max_clock * 1000;
                        break;
                    }
            }
        }
    } else {
        memcpy(Monitor, Config, sizeof(MonRec));
        if (Config->id)
            Monitor->id = xnfstrdup(Config->id);
        if (Config->vendor)
            Monitor->vendor = xnfstrdup(Config->vendor);
        if (Config->model)
            Monitor->model = xnfstrdup(Config->model);

        /* Only set DURING xf86ValidateModes currently - idiots.
         * Also assume a non-multisync 14"er, instead of a multisync 72Hz one.
         */
        if (!Monitor->nHsync) {
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Adding default "
                       "horizontal timing.\n", Monitor->id);
            Monitor->nHsync = 3;
            Monitor->hsync[0].lo = 31.5;
            Monitor->hsync[0].hi = 31.5;
            Monitor->hsync[1].lo = 35.15;
            Monitor->hsync[1].hi = 35.15;
            Monitor->hsync[2].lo = 35.5;
            Monitor->hsync[2].hi = 35.5;
        }

        if (!Monitor->nVrefresh) {
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Adding default "
                       "vertical timing.\n", Monitor->id);
            Monitor->nVrefresh = 1;
            Monitor->vrefresh[0].lo = 50;
            Monitor->vrefresh[0].hi = 61;
        }

        Monitor->Modes = NULL;
        Monitor->Last = NULL;

#ifdef MONREC_HAS_REDUCED
        Monitor->reducedblanking = FALSE;
#endif
    }

    ViaModesCopyAdd(Monitor, Config->Modes);

    return Monitor;
}

/*
 *
 */
struct ViaOutput *
ViaCRTInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output;
    struct CRTPrivate *Private;

    VIAFUNC(pScrn->scrnIndex);

    if (pDev)
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, 
                   "%s: Ignoring I2C Device.\n", __func__);

    Output = xnfcalloc(1, sizeof(struct ViaOutput));
    memset(Output, 0, sizeof(struct ViaOutput));

    Output->scrnIndex = pScrn->scrnIndex;
    Output->Type = OUTPUT_CRT;
    Output->Name = "CRT";

    Output->Active = TRUE;
    Output->ClockMaster = FALSE;
    
    Private = CRTPrivateCreate(Output);

    Output->Options = CRTGetOptions(pScrn, Private);

    /* Sensing doesn't work on VT3122Ax */
    if (Private->Sense && ((pVia->Chipset > VT3122) || VT3122_REV_IS_CX(pVia->ChipRev)))
        Output->Sense = CRTSense;

    Output->ModeValid = CRTModeValid;
    Output->Mode = CRTMode;
    Output->Power = CRTPower;

    Output->Monitor = CRTMonitor(Output);

    return Output;
}
