/*
 * Copyright 2004-2007 Luc Verhaegen.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 */

/*
 *
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"
#include "via_vgahw.h"
#include "via_id.h"
#include "via_crtc.h"
#include "via_mode.h"


/*
 *
 * Dotclock handling.
 *
 */

/*
 *
 */
static CARD32
VT3122PLLGenerateBest(int Clock, int Shift, int MinDiv, int MaxDiv,
                             int *BestDiff)
{
    CARD32 PLL = 0;
    CARD8 PLLShift;
    int Div, Mult, Diff;
    float Ratio = Clock / 14318.0;

    switch (Shift) {
    case 4:
        PLLShift = 0x80;
        break;
    case 2:
        PLLShift = 0x40;
        break;
    default:
        PLLShift = 0x00;
        break;
    }

    for (Div = MinDiv; Div <= MaxDiv; Div++) {
        Mult = Ratio * Div * Shift + 0.5;

        if (Mult < 129) {
            Diff = Clock - Mult * 14318 / Div / Shift;

            if (Diff < 0)
                Diff *= -1;

            if (Diff < *BestDiff) {
                *BestDiff = Diff;
                PLL = ((PLLShift | Div) << 8) | Mult;
            }
        }
    }

    return PLL;
}

/*
 * This might seem nasty and ugly, but it's the best solution given the crappy
 * limitations the VT3122 PLL has.
 *
 * The below information has been gathered using nothing but a lot of time and
 * perseverance.
 */
static CARD32
VT3122PLLGenerate(int scrnIndex, int Clock)
{
    CARD32 PLL;
    int Diff = 300000;

    VIAFUNC(scrnIndex);

    if (Clock > 72514)
        PLL = VT3122PLLGenerateBest(Clock, 1, 2, 25, &Diff);
    else if (Clock > 71788)
        PLL = VT3122PLLGenerateBest(Clock, 1, 16, 24, &Diff);
    else if (Clock > 71389) {
        PLL = 0x1050; /* Big singularity. */

        Diff = Clock - 71590;
        if (Diff < 0)
            Diff *= -1;
    } else if (Clock > 48833) {
        CARD32 tmpPLL;

        PLL = VT3122PLLGenerateBest(Clock, 2, 7, 18, &Diff);

        if (Clock > 69024)
            tmpPLL = VT3122PLLGenerateBest(Clock, 1, 15, 23, &Diff);
        else if (Clock > 63500)
            tmpPLL = VT3122PLLGenerateBest(Clock, 1, 15, 21, &Diff);
        else if (Clock > 52008)
            tmpPLL = VT3122PLLGenerateBest(Clock, 1, 17, 19, &Diff);
        else
            tmpPLL = VT3122PLLGenerateBest(Clock, 1, 17, 17, &Diff);

        if (tmpPLL)
            PLL =  tmpPLL;
    } else if (Clock > 35220)
        PLL = VT3122PLLGenerateBest(Clock, 2, 11, 24, &Diff);
    else if (Clock > 34511)
        PLL = VT3122PLLGenerateBest(Clock, 2, 11, 23, &Diff);
    else if (Clock > 33441)
        PLL = VT3122PLLGenerateBest(Clock, 2, 13, 22, &Diff);
    else if (Clock > 31967)
        PLL = VT3122PLLGenerateBest(Clock, 2, 11, 21, &Diff);
    else
        PLL = VT3122PLLGenerateBest(Clock, 4, 8, 19, &Diff);

    ViaDebug(scrnIndex, "%s: PLL: 0x%04X (%d off from %d)\n",
             __func__, PLL, Diff, Clock);
    return PLL;
}

/*
 *
 */
static int
VT3108PLLGenerateBest(int Clock, int Shift, int Div, int oldDiff, CARD32 *PLL)
{
    int Mult = ((float) Clock * (Div << Shift)) / 14318.18 + 0.5;
    int Diff;

    if (Mult > 257) /* Don't go over 0xFF + 2; wobbly */
        return oldDiff;

    Diff = Clock - Mult * 14318 / (Div << Shift);
    if (Diff < 0)
        Diff *= -1;

    if (Diff < oldDiff) {
        *PLL = (Mult - 2) << 16;
        *PLL |= Div - 2;
        *PLL |= Shift << 10;
        return Diff;
    } else
        return oldDiff;
}

/*
 *
 */
static CARD32
VT3108PLLGenerate(int scrnIndex, int Clock)
{
    CARD32 PLL;
    int Diff = 300000;
    int i;

    VIAFUNC(scrnIndex);

    for (i = 2; i < 15; i++)
        Diff = VT3108PLLGenerateBest(Clock, 0, i, Diff, &PLL);

    for (i = 2; i < 15; i++)
        Diff = VT3108PLLGenerateBest(Clock, 1, i, Diff, &PLL);

    for (i = 2; i < 32; i++)
        Diff = VT3108PLLGenerateBest(Clock, 2, i, Diff, &PLL);

    for (i = 2; i < 21; i++)
        Diff = VT3108PLLGenerateBest(Clock, 3, i, Diff, &PLL);

    ViaDebug(scrnIndex, "%s: PLL: 0x%04X (%d off from %d)\n",
             __func__, PLL, Diff, Clock);

    return PLL;
}

/*
 *
 */
static void
ViaCrtc1PLLSet(struct ViaCrtc *Crtc, CARD32 Clock, int flags)
{
    ScrnInfoPtr pScrn = xf86Screens[Crtc->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);
    CARD32 PLL;

    switch(pVia->Chipset) {
    case VT3122:
    case VT7205:
        if (flags & PLL_FLAG_EXTERNAL) {
            if ((pVia->Chipset == VT3122) && VT3122_REV_IS_AX(pVia->HostRev))
                PLL = 0x471C;
            else
                PLL = 0x871C;
        } else
            PLL = VT3122PLLGenerate(Crtc->scrnIndex, Clock);

        ViaDebug(Crtc->scrnIndex, "%s: %dkHz -> 0x%04X.\n",
                 __func__, Clock, PLL);

	hwp->writeSeq(hwp, 0x46, PLL >> 8);
	hwp->writeSeq(hwp, 0x47, PLL & 0xFF);

        break;
    case VT3108:
        if (flags & PLL_FLAG_EXTERNAL)
            break; /* seem perfectly happy like this???  */
        else
            PLL = VT3108PLLGenerate(Crtc->scrnIndex, Clock);

        ViaDebug(Crtc->scrnIndex, "%s: %dkHz -> 0x%06X.\n",
                 __func__, Clock, PLL);

        hwp->writeSeq(hwp, 0x44, PLL >> 16);
	hwp->writeSeq(hwp, 0x45, (PLL >> 8) & 0xFF);
	hwp->writeSeq(hwp, 0x46, PLL & 0xFF);

        break;
    default:
        xf86DrvMsg(Crtc->scrnIndex, X_ERROR, "%s: Unhandled Chipset: %s\n",
                   __func__, pScrn->chipset);
        return;
    }

    if (flags & PLL_FLAG_HALVE)
        ViaCrtcMask(hwp, 0x6B, 0x80, 0xC0);
    else if (flags & PLL_FLAG_QUARTER)
        ViaCrtcMask(hwp, 0x6B, 0xC0, 0xC0);
    else
        ViaCrtcMask(hwp, 0x6B, 0x00, 0xC0);

    if (flags & PLL_FLAG_EXTERNAL)
        ViaCrtcMask(hwp, 0x6B, 0x01, 0x01);
    else
        ViaCrtcMask(hwp, 0x6B, 0x00, 0x01);

    ViaSeqMask(hwp, 0x40, 0x02, 0x02);
    ViaSeqMask(hwp, 0x40, 0x00, 0x02);

    /* Set VGA clock to external */
    PLL = hwp->readMiscOut(hwp);
    hwp->writeMiscOut(hwp, PLL | 0x0C);
}


/*
 *
 */
static void
ViaCrtc2PLLSet(struct ViaCrtc *Crtc, CARD32 Clock, int flags)
{
    ScrnInfoPtr pScrn = xf86Screens[Crtc->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);
    CARD32 PLL;

    switch(pVia->Chipset) {
    case VT3122:
    case VT7205:
        if (flags & PLL_FLAG_EXTERNAL) {
            if ((pVia->Chipset == VT3122) && VT3122_REV_IS_AX(pVia->HostRev))
                PLL = 0x471C;
            else
                PLL = 0x871C;
        } else
            PLL = VT3122PLLGenerate(Crtc->scrnIndex, Clock);

        ViaDebug(Crtc->scrnIndex, "%s: %dkHz -> 0x%04X.\n",
                 __func__, Clock, PLL);

	hwp->writeSeq(hwp, 0x44, PLL >> 8);
	hwp->writeSeq(hwp, 0x45, PLL & 0xFF);

        break;
    case VT3108:
        if (flags & PLL_FLAG_EXTERNAL)
            break; /* seem perfectly happy like this???  */
        else
            PLL = VT3108PLLGenerate(Crtc->scrnIndex, Clock);

        ViaDebug(Crtc->scrnIndex, "%s: %dkHz -> 0x%06X.\n",
                 __func__, Clock, PLL);

        hwp->writeSeq(hwp, 0x4A, PLL >> 16);
	hwp->writeSeq(hwp, 0x4B, (PLL >> 8) & 0xFF);
	hwp->writeSeq(hwp, 0x4C, PLL & 0xFF);

        break;
    default:
        xf86DrvMsg(Crtc->scrnIndex, X_ERROR, "%s: Unhandled Chipset: %s\n",
                   __func__, pScrn->chipset);
        return;
    }

    if (flags & PLL_FLAG_HALVE)
        ViaCrtcMask(hwp, 0x6B, 0x20, 0x30);
    else if (flags & PLL_FLAG_QUARTER)
        ViaCrtcMask(hwp, 0x6B, 0x30, 0x30);
    else
        ViaCrtcMask(hwp, 0x6B, 0x00, 0x30);

    if (flags & PLL_FLAG_EXTERNAL)
        ViaCrtcMask(hwp, 0x6B, 0x01, 0x01);
    else
        ViaCrtcMask(hwp, 0x6B, 0x00, 0x01);

    ViaSeqMask(hwp, 0x40, 0x04, 0x04);
    ViaSeqMask(hwp, 0x40, 0x00, 0x04);
}

/*
 * Check whether our CRTC1 supports this mode.
 */
static ModeStatus
ViaCrtc1ModeValidate(struct ViaCrtc *Crtc, DisplayModePtr Mode)
{
    ScrnInfoPtr pScrn = xf86Screens[Crtc->scrnIndex];
    VIAPtr pVia = VIAPTR(pScrn);
    unsigned int temp;

    if (Mode->SynthClock < 20000)
        return MODE_CLOCK_LOW;

    if (Mode->SynthClock > 200000)
        return MODE_CLOCK_HIGH;

    temp = Mode->CrtcHDisplay * Mode->CrtcVDisplay * Mode->VRefresh * (Crtc->bpp >> 3);
    if (pVia->Bandwidth < temp)
	return MODE_MEM_BW;

    temp = (Mode->CrtcHDisplay + 0x1F) & ~0x1F;
    if ((temp < Crtc->MinPitch) || (temp > Crtc->MaxPitch))
        return MODE_PITCH;

    if (Mode->CrtcVDisplay < 128)
        return MODE_MINHEIGHT;

    if ((temp * Mode->CrtcVDisplay * Crtc->bpp / 8) >
        (Crtc->MaxOffset - Crtc->Offset))
        return MODE_OFFSET;

    if (Mode->CrtcHTotal > 4128)
	return MODE_HTOTAL_WIDE;

    if (Mode->CrtcHDisplay > 2048)
	return MODE_HDISPLAY_WIDE;

    /* First check Sync */
    if (Mode->CrtcHSyncStart > 4088)
	return MODE_HSYNC_RANGE;

    if ((Mode->CrtcHSyncEnd - Mode->CrtcHSyncStart) > 256)
	return MODE_HSYNC_WIDE;

    /* Silently increase overscan */
    if (Mode->CrtcHBlankStart > 2048) {
        Mode->CrtcHBlankStart = 2048;
        Mode->CrtcHAdjusted = TRUE;
    }

    /* Now check the width of our blanking, as we can correct this through
       overscan. */
    if ((Mode->CrtcHBlankEnd - Mode->CrtcHBlankStart) > 1032) {
        /* what we need to fill out with overscan */
        int gap = (Mode->CrtcHBlankEnd - Mode->CrtcHBlankStart) - 1032;
        /* what we can play with _before_ we go into sync */
        int tmp = Mode->CrtcHSyncStart - Mode->CrtcHBlankStart - 8;

        /* mind the HBlankStart */
        if ((Mode->CrtcHBlankStart + tmp) > 2048)
            tmp = 2048 - Mode->CrtcHBlankStart;

        if (tmp >= gap)
            Mode->CrtcHBlankStart += gap;
        else {
            Mode->CrtcHBlankStart += tmp;
            gap -= tmp;

            /* now, adjust the blanking at the other side */
            tmp = Mode->CrtcHBlankEnd - Mode->CrtcHSyncEnd - 8;
            if (tmp >= gap)
                Mode->CrtcHBlankEnd -= gap;
            else /* Sorry, we can't do this: HSyncEnd > 3072 */
                return MODE_HBLANK_RANGE;
        }

        Mode->CrtcHAdjusted = TRUE;
    }

    if (Mode->CrtcVTotal > 2049)
	return MODE_VTOTAL_WIDE;

    if (Mode->CrtcVDisplay > 2048)
	return MODE_VDISPLAY_WIDE;

    if (Mode->CrtcVSyncStart > 2047)
	return MODE_VSYNC_RANGE;

    if ((Mode->CrtcVSyncEnd - Mode->CrtcVSyncStart) > 16)
	return MODE_VSYNC_WIDE;

    /* Silently increase overscan */
    if (Mode->CrtcVBlankStart > 2048) {
        Mode->CrtcVBlankStart = 2048;
        Mode->CrtcVAdjusted = TRUE;
    }

    /* Now check the width of our blanking, as we can correct this through
       overscan. */
    if ((Mode->CrtcVBlankEnd - Mode->CrtcVBlankStart) > 257) {
        /* what we need to fill out with overscan */
        int gap = (Mode->CrtcVBlankEnd - Mode->CrtcVBlankStart) - 257;
        /* what we can play with _before_ we go into sync */
        int tmp = Mode->CrtcVSyncStart - Mode->CrtcVBlankStart - 1;

        /* mind the VBlankStart */
        if ((Mode->CrtcVBlankStart + tmp) > 2048)
            tmp = 2048 - Mode->CrtcVBlankStart;

        if (tmp >= gap)
            Mode->CrtcVBlankStart += gap;
        else {
            Mode->CrtcVBlankStart += tmp;
            gap -= tmp;

            /* now, adjust the blanking at the other side */
            tmp = Mode->CrtcVBlankEnd - Mode->CrtcVSyncEnd - 1;
            if (tmp >= gap)
                Mode->CrtcVBlankEnd -= gap;
            else /* Sorry, we can't do this: VSyncEnd > 3072 */
                return MODE_VBLANK_RANGE;
        }

        Mode->CrtcVAdjusted = TRUE;
    }

    return MODE_OK;
}

/*
 *
 */
static ModeStatus
ViaCrtc2ModeValidate(struct ViaCrtc *Crtc, DisplayModePtr Mode)
{
    VIAPtr pVia = VIAPTR(xf86Screens[Crtc->scrnIndex]);
    unsigned int temp;

    if (Mode->SynthClock < 20000)
        return MODE_CLOCK_LOW;

    if (Mode->SynthClock > 200000)
        return MODE_CLOCK_HIGH;

    temp = Mode->CrtcHDisplay * Mode->CrtcVDisplay * Mode->VRefresh * (Crtc->bpp >> 3);
    if (pVia->Bandwidth < temp)
	return MODE_MEM_BW;

    temp = (Mode->CrtcHDisplay + 0x1F) & ~0x1F;
    if ((temp < Crtc->MinPitch) || (temp > Crtc->MaxPitch))
        return MODE_PITCH;

    if (Mode->CrtcVDisplay < 128)
        return MODE_MINHEIGHT;

    if ((temp * Mode->CrtcVDisplay * Crtc->bpp / 8) >
        (Crtc->MaxOffset - Crtc->Offset))
        return MODE_OFFSET;

    if (Mode->CrtcHTotal > 4096)
	return MODE_HTOTAL_WIDE;

    /* HDisplay is max 2048, but offset is max 2040. */
    if (Mode->CrtcHDisplay > 2040)
	return MODE_HDISPLAY_WIDE;

    /* First check Sync */
    if (Mode->CrtcHSyncStart > 2047)
	return MODE_HSYNC_RANGE;

    if ((Mode->CrtcHSyncEnd - Mode->CrtcHSyncStart) > 512)
	return MODE_HSYNC_WIDE;

    /* Silently increase overscan */
    if (Mode->CrtcHBlankStart > 2048) {
        Mode->CrtcHBlankStart = 2048;
        Mode->CrtcHAdjusted = TRUE;
    }

    /* Ah, nice */
    if (Mode->CrtcHBlankEnd > 4096) {
        Mode->CrtcHBlankEnd = 4096;
        Mode->CrtcHAdjusted = TRUE;
    }

    if (Mode->CrtcVTotal > 2048)
	return MODE_VTOTAL_WIDE;

    if (Mode->CrtcVDisplay > 2048)
	return MODE_VDISPLAY_WIDE;

    if (Mode->CrtcVSyncStart > 2047)
	return MODE_VSYNC_RANGE;

    if ((Mode->CrtcVSyncEnd - Mode->CrtcVSyncStart) > 32)
	return MODE_VSYNC_WIDE;

    /* Silently increase overscan */
    if (Mode->CrtcVBlankStart > 2048) {
        Mode->CrtcVBlankStart = 2048;
        Mode->CrtcVAdjusted = TRUE;
    }

    if (Mode->CrtcHBlankEnd > 2048) {
        Mode->CrtcHBlankEnd = 2048;
        Mode->CrtcHAdjusted = TRUE;
    }

    return MODE_OK;
}

/*
 *
 */
static void
ViaCrtc1ModeSet(struct ViaCrtc *Crtc, DisplayModePtr mode)
{
    vgaHWPtr hwp = VGAHWPTR(xf86Screens[Crtc->scrnIndex]);
    CARD16 temp;

    ViaDebug(Crtc->scrnIndex, "%s: Setting up %s\n", __func__, mode->name);

    /* Make this 32byte aligned as well... for reasons unknown and untested */
    Crtc->VisiblePitch = (mode->HDisplay * (Crtc->bpp >> 3) + 0x1F) & ~0x1F;

    /* Set sync polarity */
    temp = hwp->readMiscOut(hwp) & 0x3F;
    if (mode->Flags & V_NHSYNC)
	temp |= 0x40;
    if (mode->Flags & V_NVSYNC)
	temp |= 0x80;
    hwp->writeMiscOut(hwp, temp);

    /* 555/565 -- bpp */
    switch (Crtc->bpp) {
    case 8:
	ViaSeqMask(hwp, 0x15, 0x00, 0x1C);
        break;
    case 16:
	ViaSeqMask(hwp, 0x15, 0x14, 0x1C);
        break;
    case 24:
    case 32:
	ViaSeqMask(hwp, 0x15, 0x0C, 0x1C);
        break;
    default:
        xf86DrvMsg(Crtc->scrnIndex, X_ERROR, "Unhandled bitdepth: %d\n",
                   Crtc->bpp);
        break;
    }

    /* Crtc registers */
    /* horizontal total : 4128 */
    ViaDebug(Crtc->scrnIndex, "CrtcHTotal: 0x%03X\n", mode->CrtcHTotal);
    temp = (mode->CrtcHTotal >> 3) - 5;
    hwp->writeCrtc(hwp, 0x00, temp & 0xFF);
    ViaCrtcMask(hwp, 0x36, temp >> 5, 0x08);

    /* horizontal address : 2048 */
    ViaDebug(Crtc->scrnIndex, "CrtcHDisplay: 0x%03X\n", mode->CrtcHDisplay);
    temp = (mode->CrtcHDisplay >> 3) - 1;
    hwp->writeCrtc(hwp, 0x01, temp & 0xFF);

    /* horizontal blanking start : 2048 */
    ViaDebug(Crtc->scrnIndex, "CrtcHBlankStart: 0x%03X\n", mode->CrtcHBlankStart);
    temp = (mode->CrtcHBlankStart >> 3) - 1;
    hwp->writeCrtc(hwp, 0x02, temp & 0xFF);

    /* horizontal blanking end : start + 1032 */
    ViaDebug(Crtc->scrnIndex, "CrtcHBlankEnd: 0x%03X\n", mode->CrtcHBlankEnd);
    temp = (mode->CrtcHBlankEnd >> 3) - 1;
    ViaCrtcMask(hwp, 0x03, temp, 0x1F);
    ViaCrtcMask(hwp, 0x05, temp << 2, 0x80);
    ViaCrtcMask(hwp, 0x33, temp >> 1, 0x20);

    /* CrtcHSkew ??? */

    /* horizontal sync start : 4088 */
    ViaDebug(Crtc->scrnIndex, "CrtcHSyncStart: 0x%03X\n", mode->CrtcHSyncStart);
    temp = mode->CrtcHSyncStart >> 3;
    hwp->writeCrtc(hwp, 0x04, temp & 0xFF);
    ViaCrtcMask(hwp, 0x33, temp >> 4, 0x10);

    /* horizontal sync end : start + 256 */
    ViaDebug(Crtc->scrnIndex, "CrtcHSyncEnd: 0x%03X\n",  mode->CrtcHSyncEnd);
    temp = mode->CrtcHSyncEnd >> 3;
    ViaCrtcMask(hwp, 0x05, temp, 0x1F);

    /* vertical total : 2049 */
    ViaDebug(Crtc->scrnIndex, "CrtcVTotal: 0x%03X\n", mode->CrtcVTotal);
    temp = mode->CrtcVTotal - 2;
    hwp->writeCrtc(hwp, 0x06, temp & 0xFF);
    ViaCrtcMask(hwp, 0x07, temp >> 8, 0x01);
    ViaCrtcMask(hwp, 0x07, temp >> 4, 0x20);
    ViaCrtcMask(hwp, 0x35, temp >> 10, 0x01);

    /* vertical address : 2048 */
    ViaDebug(Crtc->scrnIndex, "CrtcVDisplay: 0x%03X\n", mode->CrtcVDisplay);
    temp = mode->CrtcVDisplay - 1;
    hwp->writeCrtc(hwp, 0x12, temp & 0xFF);
    ViaCrtcMask(hwp, 0x07, temp >> 7, 0x02);
    ViaCrtcMask(hwp, 0x07, temp >> 3, 0x40);
    ViaCrtcMask(hwp, 0x35, temp >> 8, 0x04);

    /* Primary starting address -> 0x00, adjustframe does the rest */
    hwp->writeCrtc(hwp, 0x0C, 0x00);
    hwp->writeCrtc(hwp, 0x0D, 0x00);
    hwp->writeCrtc(hwp, 0x34, 0x00);
    ViaCrtcMask(hwp, 0x48, 0x00, 0x03);

    /* vertical sync start : 2047 */
    ViaDebug(Crtc->scrnIndex, "CrtcVSyncStart: 0x%03X\n", mode->CrtcVSyncStart);
    temp = mode->CrtcVSyncStart;
    hwp->writeCrtc(hwp, 0x10, temp & 0xFF);
    ViaCrtcMask(hwp, 0x07, temp >> 6, 0x04);
    ViaCrtcMask(hwp, 0x07, temp >> 2, 0x80);
    ViaCrtcMask(hwp, 0x35, temp >> 9, 0x02);

    /* vertical sync end : start + 16 -- other bits someplace? */
    ViaDebug(Crtc->scrnIndex, "CrtcVSyncEnd: 0x%03X\n", mode->CrtcVSyncEnd);
    ViaCrtcMask(hwp, 0x11, mode->CrtcVSyncEnd, 0x0F);

    /* line compare: We are not doing splitscreen so 0x3FFF */
    hwp->writeCrtc(hwp, 0x18, 0xFF);
    ViaCrtcMask(hwp, 0x07, 0x10, 0x10);
    ViaCrtcMask(hwp, 0x09, 0x40, 0x40);
    ViaCrtcMask(hwp, 0x33, 0x07, 0x06);
    ViaCrtcMask(hwp, 0x35, 0x10, 0x10);

    /* zero Maximum scan line */
    ViaCrtcMask(hwp, 0x09, 0x00, 0x1F);
    hwp->writeCrtc(hwp, 0x14, 0x00);

    /* vertical blanking start : 2048 */
    ViaDebug(Crtc->scrnIndex, "CrtcVBlankStart: 0x%03X\n", mode->CrtcVBlankStart);
    temp = mode->CrtcVBlankStart - 1;
    hwp->writeCrtc(hwp, 0x15, temp & 0xFF);
    ViaCrtcMask(hwp, 0x07, temp >> 5, 0x08);
    ViaCrtcMask(hwp, 0x09, temp >> 4, 0x20);
    ViaCrtcMask(hwp, 0x35, temp >> 7, 0x08);

    /* vertical blanking end : start + 257 */
    ViaDebug(Crtc->scrnIndex, "CrtcVBlankEnd: 0x%03X\n", mode->CrtcVBlankEnd);
    temp = mode->CrtcVBlankEnd - 1;
    hwp->writeCrtc(hwp, 0x16, temp & 0xFF);

    /* some leftovers */
    hwp->writeCrtc(hwp, 0x08, 0x00);

    /* offset */
    temp = Crtc->Pitch >> 3;
    ViaDebug(Crtc->scrnIndex, "Offset: 0x%03X\n", temp);
    hwp->writeCrtc(hwp, 0x13, temp & 0xFF);
    ViaCrtcMask(hwp, 0x35, temp >> 3, 0xE0);

    /* fetch count */
    temp = Crtc->VisiblePitch >> 3;
    ViaDebug(Crtc->scrnIndex, "Fetch Count: 0x%03X\n", temp);
    hwp->writeSeq(hwp, 0x1C, (temp >> 1) & 0xFF);
    ViaSeqMask(hwp, 0x1D, temp >> 9, 0x03);

    /* some leftovers */
    ViaCrtcMask(hwp, 0x32, 0, 0xFF);
    ViaCrtcMask(hwp, 0x33, 0, 0xC8);
}

/*
 *
 */
static void
ViaCrtc2ModeSet(struct ViaCrtc *Crtc, DisplayModePtr mode)
{
    vgaHWPtr hwp = VGAHWPTR(xf86Screens[Crtc->scrnIndex]);
    CARD16 temp;

    ViaDebug(Crtc->scrnIndex, "%s: Setting up %s\n", __func__, mode->name);

    /* Make this 32byte aligned as well... for reasons unknown and untested */
    Crtc->VisiblePitch = (mode->HDisplay * (Crtc->bpp >> 3) + 0x1F) & ~0x1F;

    /* bpp */
    switch (Crtc->bpp) {
    case 8:
        ViaCrtcMask(hwp, 0x67, 0x00, 0xC0);
        break;
    case 16:
        ViaCrtcMask(hwp, 0x67, 0x40, 0xC0);
        break;
    case 24:
    case 32:
        ViaCrtcMask(hwp, 0x67, 0x80, 0xC0);
        break;
    default:
        xf86DrvMsg(Crtc->scrnIndex, X_ERROR, "Unhandled bitdepth: %d\n",
                   Crtc->bpp);
        break;
    }

    /* Crtc registers */
    /* horizontal total : 4096 */
    ViaDebug(Crtc->scrnIndex, "CrtcHTotal: 0x%03X\n", mode->CrtcHTotal);
    temp = mode->CrtcHTotal - 1;
    hwp->writeCrtc(hwp, 0x50, temp & 0xFF);
    ViaCrtcMask(hwp, 0x55, temp >> 8, 0x0F);

    /* horizontal address : 2048 */
    ViaDebug(Crtc->scrnIndex, "CrtcHDisplay: 0x%03X\n", mode->CrtcHDisplay);
    temp = mode->CrtcHDisplay - 1;
    hwp->writeCrtc(hwp, 0x51, temp & 0xFF);
    ViaCrtcMask(hwp, 0x55, temp >> 4, 0x70);

    /* horizontal blanking start : 2048 */
    ViaDebug(Crtc->scrnIndex, "CrtcHBlankStart: 0x%03X\n", mode->CrtcHBlankStart);
    temp = mode->CrtcHBlankStart - 1;
    hwp->writeCrtc(hwp, 0x52, temp & 0xFF);
    ViaCrtcMask(hwp, 0x54, temp >> 8, 0x07);

    /* horizontal blanking end : 4096 */
    ViaDebug(Crtc->scrnIndex, "CrtcHBlankEnd: 0x%03X\n", mode->CrtcHBlankEnd);
    temp = mode->CrtcHBlankEnd - 1;
    hwp->writeCrtc(hwp, 0x53, temp & 0xFF);
    ViaCrtcMask(hwp, 0x54, temp >> 5, 0x38);
    ViaCrtcMask(hwp, 0x5D, temp >> 5, 0x40);

    /* horizontal sync start : 2047 */
    ViaDebug(Crtc->scrnIndex, "CrtcHSyncStart: 0x%03X\n", mode->CrtcHSyncStart);
    temp = mode->CrtcHSyncStart;
    hwp->writeCrtc(hwp, 0x56, temp & 0xFF);
    ViaCrtcMask(hwp, 0x54, temp >> 2, 0xC0);
    ViaCrtcMask(hwp, 0x5C, temp >> 3, 0x80);

    /* horizontal sync end : sync start + 512 */
    ViaDebug(Crtc->scrnIndex, "CrtcHSyncEnd: 0x%03X\n", mode->CrtcHSyncEnd);
    temp = mode->CrtcHSyncEnd;
    hwp->writeCrtc(hwp, 0x57, temp & 0xFF);
    ViaCrtcMask(hwp, 0x5C, temp >> 2, 0x40);

    /* vertical total : 2048 */
    ViaDebug(Crtc->scrnIndex, "CrtcVTotal: 0x%03X\n", mode->CrtcVTotal);
    temp = mode->CrtcVTotal - 1;
    hwp->writeCrtc(hwp, 0x58, temp & 0xFF);
    ViaCrtcMask(hwp, 0x5D, temp >> 8, 0x07);

    /* vertical address : 2048 */
    ViaDebug(Crtc->scrnIndex, "CrtcVDisplay: 0x%03X\n", mode->CrtcVDisplay);
    temp = mode->CrtcVDisplay - 1;
    hwp->writeCrtc(hwp, 0x59, temp & 0xFF);
    ViaCrtcMask(hwp, 0x5D, temp >> 5, 0x38);

    /* vertical blanking start : 2048 */
    ViaDebug(Crtc->scrnIndex, "CrtcVBlankStart: 0x%03X\n", mode->CrtcVBlankStart);
    temp = mode->CrtcVBlankStart - 1;
    hwp->writeCrtc(hwp, 0x5A, temp & 0xFF);
    ViaCrtcMask(hwp, 0x5C, temp >> 8, 0x07);

    /* vertical blanking end : 2048 */
    ViaDebug(Crtc->scrnIndex, "CrtcVBlankEnd: 0x%03X\n", mode->CrtcVBlankEnd);
    temp = mode->CrtcVBlankEnd - 1;
    hwp->writeCrtc(hwp, 0x5B, temp & 0xFF);
    ViaCrtcMask(hwp, 0x5C, temp >> 5, 0x38);

    /* vertical sync start : 2047 */
    ViaDebug(Crtc->scrnIndex, "CrtcVSyncStart: 0x%03X\n", mode->CrtcVSyncStart);
    temp = mode->CrtcVSyncStart;
    hwp->writeCrtc(hwp, 0x5E, temp & 0xFF);
    ViaCrtcMask(hwp, 0x5F, temp >> 3, 0xE0);

    /* vertical sync end : start + 32 */
    ViaDebug(Crtc->scrnIndex, "CrtcVSyncEnd: 0x%03X\n", mode->CrtcVSyncEnd);
    temp = mode->CrtcVSyncEnd;
    ViaCrtcMask(hwp, 0x5F, temp, 0x1F);

    /* offset */
    temp = Crtc->Pitch >> 3;
    ViaDebug(Crtc->scrnIndex, "Offset: 0x%03X\n", temp);
    hwp->writeCrtc(hwp, 0x66, temp & 0xFF);
    ViaCrtcMask(hwp, 0x67, temp >> 8, 0x03);

    /* fetch count */
    temp = Crtc->VisiblePitch >> 3;
    ViaDebug(Crtc->scrnIndex, "Fetch Count: 0x%03X\n", temp);
    hwp->writeCrtc(hwp, 0x65, (temp >> 1) & 0xFF);
    ViaCrtcMask(hwp, 0x67, temp >> 7, 0x0C);

    /* always disable scaling for now - will be handled properly soon */
    ViaCrtcMask(hwp, 0x79, 0x00, 0x01);
}

/*
 *
 */
typedef struct {
    CARD16   X;
    CARD16   Y;
    CARD16   Bpp;
    CARD8    bRamClock;
    CARD8    bTuningValue;
} ViaExpireNumberTable;

static ViaExpireNumberTable VT3122AExpireNumber[] = {
    {1280,  768, 32, VIA_MEM_DDR200,  3},
    {1280, 1024, 32, VIA_MEM_DDR200,  4},
    {1600, 1200, 16, VIA_MEM_DDR200,  4},
    {1024,  768, 32, VIA_MEM_DDR200, 10},
    {1400, 1050, 16, VIA_MEM_DDR200,  3},
    {1400, 1050, 32, VIA_MEM_DDR200,  4},
    { 800,  600, 32, VIA_MEM_DDR200, 10},
    {1280, 1024, 32, VIA_MEM_DDR266,  3},
    {1600, 1200, 32, VIA_MEM_DDR266,  4},
    {1400, 1050, 32, VIA_MEM_DDR266,  4},
    { 0, 0, 0, 0, 0}
};

static ViaExpireNumberTable VT3122CExpireNumber[] = {
    {1280,  768, 32, VIA_MEM_DDR200,  3},
    {1280, 1024, 32, VIA_MEM_DDR200,  4},
    {1600, 1200, 32, VIA_MEM_DDR200,  3},
    {1024,  768, 32, VIA_MEM_DDR200, 10},
    {1400, 1050, 32, VIA_MEM_DDR200,  4},
    { 800,  600, 32, VIA_MEM_DDR200, 10},
    {1280, 1024, 32, VIA_MEM_DDR266,  4},
    {1600, 1200, 32, VIA_MEM_DDR266,  4},
    {1400, 1050, 32, VIA_MEM_DDR266,  4},
    { 0, 0, 0, 0, 0}
};

static ViaExpireNumberTable VT7205ExpireNumber[]={
    {1280, 1024, 32, VIA_MEM_DDR200,  3},
    {1280,  768, 32, VIA_MEM_DDR200,  3},
    {1400, 1050, 32, VIA_MEM_DDR200,  3},
    {1600, 1200, 32, VIA_MEM_DDR200,  4},
    {1280, 1024, 32, VIA_MEM_DDR266,  9},
    {1280,  768, 32, VIA_MEM_DDR266,  9},
    {1400, 1050, 32, VIA_MEM_DDR266,  9},
    {1600, 1200, 32, VIA_MEM_DDR266, 10},
    { 0, 0, 0, 0,0}
};

/*
 *
 */
static CARD8
ViaCrtc1ExpireGet(struct ViaCrtc *Crtc, DisplayModePtr mode, ViaExpireNumberTable *Expire)
{
    ScrnInfoPtr pScrn = xf86Screens[Crtc->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(Crtc->scrnIndex);

    for (; Expire->X; Expire++)
        if ((Expire->X == mode->CrtcHDisplay) &&
            (Expire->Y == mode->CrtcVDisplay) &&
            (Expire->Bpp == Crtc->bpp) &&
            (Expire->bRamClock == pVia->MemType))
	    return Expire->bTuningValue;

    return (hwp->readSeq(hwp, 0x22) & 0x1F);
}

/*
 * Priority being PriorityThreshold.
 */
static void
ViaCrtc1FIFOHelper(vgaHWPtr hwp, CARD8 Depth, CARD8 Threshold, CARD8 Priority,
                   CARD8 Expire, Bool PriorityAboveThreshold)
{
    CARD8 tmp;

    /* Fifo Depth */
    hwp->writeSeq(hwp, 0x17, Depth & 0xFF);

    /* Fifo Threshold */
    tmp = Threshold & 0x3F;
    tmp |= (Threshold << 1) & 0x80;
    ViaSeqMask(hwp, 0x16, tmp, 0xBF);

    /* Fifo Priority Threshold */
    tmp = Priority & 0x3F;
    tmp |= (Priority << 1) & 0x80;
    if (PriorityAboveThreshold)
        tmp |= 0x40;
    hwp->writeSeq(hwp, 0x18, tmp);

    /* Fifo Expire Number */
    ViaSeqMask(hwp, 0x22, Expire, 0x1F);
}

/*
 *
 */
static void
ViaCrtc1FIFOSet(struct ViaCrtc *Crtc, DisplayModePtr mode)
{
    ScrnInfoPtr pScrn = xf86Screens[Crtc->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);
    CARD8 tmp;

    VIAFUNC(Crtc->scrnIndex);

    switch(pVia->Chipset) {
    case VT3122:
	if (VT3122_REV_IS_CX(pVia->HostRev)) {
            tmp = ViaCrtc1ExpireGet(Crtc, mode, VT3122CExpireNumber);

	    if (pVia->HasSecondary) {
                if (Crtc->VisiblePitch >= 4096)
                    ViaCrtc1FIFOHelper(hwp, 63, 28, 23, tmp, TRUE);
                else
                    ViaCrtc1FIFOHelper(hwp, 31, 8, 23, tmp, TRUE);
	    } else {
		if (Crtc->VisiblePitch >= 4096)
                    ViaCrtc1FIFOHelper(hwp, 47, 23, 23, tmp, TRUE);
                else
                    ViaCrtc1FIFOHelper(hwp, 31, 8, 23, tmp, TRUE);
	    }
	} else {
            tmp = ViaCrtc1ExpireGet(Crtc, mode, VT3122AExpireNumber);

            if (pVia->HasSecondary) {
                if (Crtc->VisiblePitch >= 4096)
                    ViaCrtc1FIFOHelper(hwp, 31, 23, 23, tmp, TRUE);
                else
                    ViaCrtc1FIFOHelper(hwp, 31, 8, 14, tmp, TRUE);
            } else {
                if (Crtc->VisiblePitch >= 6400)
                    ViaCrtc1FIFOHelper(hwp, 31, 15, 15, tmp, TRUE);
                else if (Crtc->VisiblePitch >= 4096)
                    ViaCrtc1FIFOHelper(hwp, 31, 12, 12, tmp, TRUE);
                else
                    ViaCrtc1FIFOHelper(hwp, 31, 8, 14, tmp, TRUE);
            }
	}
	break;
    case VT7205:
        tmp = ViaCrtc1ExpireGet(Crtc, mode, VT7205ExpireNumber);

        if (pVia->HasSecondary) {
            if ((pVia->MemType <= VIA_MEM_DDR200) && (Crtc->VisiblePitch >= 6400))
                ViaCrtc1FIFOHelper(hwp, 28, 9, 23, tmp, TRUE);
            else
                ViaCrtc1FIFOHelper(hwp, 63, 28, 23, tmp, TRUE);
        } else {
            if (Crtc->VisiblePitch >= 5120) /* 1024@32bpp */
                ViaCrtc1FIFOHelper(hwp, 63, 28, 23, tmp, TRUE);
            else if (Crtc->VisiblePitch >= 4096) /* 1024@32bpp */
                ViaCrtc1FIFOHelper(hwp, 63, 23, 23, tmp, TRUE);
            else
                ViaCrtc1FIFOHelper(hwp, 63, 16, 23, tmp, TRUE);
        }
	break;
    case VT3108:
        /* Enable Prefetch */
        ViaCrtcMask(hwp, 0x33, 0x08, 0x08);

	if (Crtc->VisiblePitch >= 5600) /* 1400@32bpp */
            ViaCrtc1FIFOHelper(hwp, 191, 82, 74, 16, FALSE);
	else
            ViaCrtc1FIFOHelper(hwp, 191, 82, 74, 0, FALSE);
	break;
    case VT3118:
	if (Crtc->VisiblePitch >= 5600) /* 1400@32bpp */
            ViaCrtc1FIFOHelper(hwp, 95, 32, 16, 16, TRUE);
	else
            ViaCrtc1FIFOHelper(hwp, 95, 32, 16, 31, TRUE);
	break;
    default:
	xf86DrvMsg(Crtc->scrnIndex, X_ERROR,
                   "%s: Chipset %d not implemented\n", __func__, pVia->Chipset);
	break;

    }
}

/*
 *
 */
static void
VT3122Crtc2FIFOHelper(vgaHWPtr hwp, Bool Enable, CARD8 Depth, CARD8 Threshold)
{
    if (Enable) {
        ViaCrtcMask(hwp, 0x6A, 0x20, 0x20);
        hwp->writeCrtc(hwp, 0x68, ((Depth & 0x0F) << 4) | (Threshold & 0x0F));
    } else
        ViaCrtcMask(hwp, 0x6A, 0x00, 0x20);
}

/*
 *
 */
static void
VT3108Crtc2FIFOHelper(vgaHWPtr hwp, CARD8 Depth, CARD8 Threshold,
                      CARD8 Priority, CARD8 Expire)
{
    /* Depth */
    ViaCrtcMask(hwp, 0x68, Depth << 4, 0xF0);
    ViaCrtcMask(hwp, 0x94, Depth << 3, 0x80);
    ViaCrtcMask(hwp, 0x95, Depth << 2, 0x80);

    /* Read Threshold */
    ViaCrtcMask(hwp, 0x68, Threshold, 0x0F);
    ViaCrtcMask(hwp, 0x95, Threshold, 0x70);

    /* Priority threshold */
    ViaCrtcMask(hwp, 0x92, Priority, 0x0F);
    ViaCrtcMask(hwp, 0x95, Priority, 0x07);

    /* Expire Number */
    ViaCrtcMask(hwp, 0x94, Expire, 0x7F);
}

/*
 *
 */
static void
ViaCrtc2FIFOSet(struct ViaCrtc *Crtc, DisplayModePtr mode)
{
    ScrnInfoPtr pScrn = xf86Screens[Crtc->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    VIAPtr pVia = VIAPTR(pScrn);

    VIAFUNC(Crtc->scrnIndex);

    switch (pVia->Chipset) {
    case VT3122:
        if (VT3122_REV_IS_CX(pVia->HostRev)) {
            if (Crtc->VisiblePitch >= 4096)
		VT3122Crtc2FIFOHelper(hwp, TRUE, 10, 11);
            else
                VT3122Crtc2FIFOHelper(hwp, FALSE, 6, 7);
        } else
            /* don't go over 7, high bits trigger a shift left */
            VT3122Crtc2FIFOHelper(hwp, FALSE, 7, 7);
	break;
    case VT7205:
	if ((pVia->MemType <= VIA_MEM_DDR200) && (Crtc->VisiblePitch >= 6400))
            VT3122Crtc2FIFOHelper(hwp, TRUE, 14, 11);
        else if (((pVia->MemType <= VIA_MEM_DDR200) && (Crtc->VisiblePitch >= 4096)) ||
                 ((pVia->MemType <= VIA_MEM_DDR333) && (Crtc->VisiblePitch >= 5120)))
            VT3122Crtc2FIFOHelper(hwp, TRUE, 12, 11);
        else if (((pVia->MemType <= VIA_MEM_DDR200) && (Crtc->VisiblePitch >= 2048)) ||
                 ((pVia->MemType <= VIA_MEM_DDR333) && (Crtc->VisiblePitch >= 2560)))
            VT3122Crtc2FIFOHelper(hwp, TRUE, 10, 11);
        else
            VT3122Crtc2FIFOHelper(hwp, FALSE, 6, 7);

	break;
    case VT3108:
        ViaCrtcMask(hwp, 0x6A, 0x01, 0x01); /* Enable prefetch */

	if (Crtc->VisiblePitch >= 5600)
            VT3108Crtc2FIFOHelper(hwp, 46, 82, 74, 16);
	else
            VT3108Crtc2FIFOHelper(hwp, 46, 82, 74, 32);
	break;
    case VT3118:
        if (Crtc->VisiblePitch >= 5600)
            VT3108Crtc2FIFOHelper(hwp, 11, 16, 8, 16);
	else
            VT3108Crtc2FIFOHelper(hwp, 11, 16, 8, 32);
	break;
    default:
	xf86DrvMsg(Crtc->scrnIndex, X_ERROR,
                   "%s: Chipset %d not implemented\n", __func__, pVia->Chipset);
	break;
    }
}

/*
 *
 */
static void
ViaCrtc1FrameSet(struct ViaCrtc *Crtc, int X, int Y)
{
    vgaHWPtr  hwp = VGAHWPTR(xf86Screens[Crtc->scrnIndex]);
    CARD32  Base;

    Base = Crtc->Offset + Y * Crtc->Pitch + ((X * Crtc->bpp) >> 3);
    Base >>= 1;

    Base += 7;
    Base &= 0xFFFFFFF8;

    hwp->writeCrtc(hwp, 0x0C, (Base & 0xFF00) >> 8);
    hwp->writeCrtc(hwp, 0x0D, Base & 0xFF);
    hwp->writeCrtc(hwp, 0x34, (Base & 0xFF0000) >> 16);
    /* VT3122A doesn't implement this, but luckily it simply doesn't care.
     * Value is properly limited in PreInit anyway. */
    ViaCrtcMask(hwp, 0x48, Base >> 24, 0x03);
}

/*
 *
 */
static void
ViaCrtc2FrameSet(struct ViaCrtc *Crtc, int X, int Y)
{
    vgaHWPtr  hwp = VGAHWPTR(xf86Screens[Crtc->scrnIndex]);
    CARD32  Base;

    Base = Crtc->Offset + Y * Crtc->Pitch + ((X * Crtc->bpp) >> 3);
    Base >>= 3;

    /* Round, jumps some more than before, but avoids the nasty border. */
    Base += 3;
    Base &= ~3;

    ViaCrtcMask(hwp, 0x62, (Base & 0x7F) << 1 , 0xFE);
    hwp->writeCrtc(hwp, 0x63, (Base & 0x7F80) >>  7);
    hwp->writeCrtc(hwp, 0x64, (Base & 0x7F8000) >>  15);
}

/*
 *
 */
static void
ViaCrtcGammaSetHelper(struct ViaCrtc *Crtc, int numColors, int *indices, LOCO *colors)
{
    vgaHWPtr hwp = VGAHWPTR(xf86Screens[Crtc->scrnIndex]);
    int i, j, index;

    hwp->enablePalette(hwp);
    hwp->writeDacMask(hwp, 0xFF);

    /* We need the same palette contents for 16 as 24bit, but X doesn't
     * play. X cmap handling is hopelessly intertwined with just about any X
     * subsystem you can care to name. So we just space out RGB values over the
     * 256*3 lut. This is quite unlike me, but even i draw a line somewhere.
     * Digging out the DIX is somewhere. */
    switch (Crtc->bpp) {
    case 16:
        for (i = 0; i < numColors; i++) {
            index = indices[i];
            hwp->writeDacWriteAddr(hwp, index * 4);

            for (j = 0; j < 4; j++) {
                hwp->writeDacData(hwp, colors[index/2].red);
                hwp->writeDacData(hwp, colors[index].green);
                hwp->writeDacData(hwp, colors[index/2].blue);
            }
        }
        break;
    case 8:
    case 24:
    case 32:
        for (i = 0; i < numColors; i++) {
            index = indices[i];
            hwp->writeDacWriteAddr(hwp, index);

            hwp->writeDacData(hwp, colors[index].red);
            hwp->writeDacData(hwp, colors[index].green);
            hwp->writeDacData(hwp, colors[index].blue);
        }
        break;
    default:
        xf86DrvMsg(Crtc->scrnIndex, X_ERROR, "%s: Unsupported bitdepth: %d\n",
                   __func__, Crtc->bpp);
        break;
    }

    hwp->disablePalette(hwp);
}

/*
 *
 */
static void
ViaCrtc1GammaSet(struct ViaCrtc *Crtc, int numColors, int *indices, LOCO *colors)
{
    vgaHWPtr hwp = VGAHWPTR(xf86Screens[Crtc->scrnIndex]);

    VIAFUNC(Crtc->scrnIndex);

    ViaSeqMask(hwp, 0x1A, 0x00, 0x01);

    ViaCrtcGammaSetHelper(Crtc, numColors, indices, colors);
}

/*
 *
 */
static void
ViaCrtc2GammaSet(struct ViaCrtc *Crtc, int numColors, int *indices, LOCO *colors)
{
    vgaHWPtr hwp = VGAHWPTR(xf86Screens[Crtc->scrnIndex]);

    VIAFUNC(Crtc->scrnIndex);

    ViaSeqMask(hwp, 0x1A, 0x01, 0x01);

    ViaCrtcGammaSetHelper(Crtc, numColors, indices, colors);
}

/*
 * Can't really enable/disable this one.
 */
static void
ViaCrtc1Enable(struct ViaCrtc *Crtc, Bool Enable)
{
    VIAFUNC(Crtc->scrnIndex);
}

/*
 *
 */
static void
ViaCrtc2Enable(struct ViaCrtc *Crtc, Bool Enable)
{
    vgaHWPtr hwp = VGAHWPTR(xf86Screens[Crtc->scrnIndex]);

    VIAFUNC(Crtc->scrnIndex);

    if (Enable)
        ViaCrtcMask(hwp, 0x6A, 0x80, 0x80);
    else
        ViaCrtcMask(hwp, 0x6A, 0x00, 0x80);
}

/*
 * Asynch reset doesn't respond in extended mode.
 * So just disable the clock to have the same effect.
 */
static void
ViaCrtc1Reset(struct ViaCrtc *Crtc, Bool Reset)
{
    vgaHWPtr hwp = VGAHWPTR(xf86Screens[Crtc->scrnIndex]);

    VIAFUNC(Crtc->scrnIndex);

    if (Reset)
        ViaSeqMask(hwp, 0x1B, 0x00, 0x30);
    else
        ViaSeqMask(hwp, 0x1B, 0x30, 0x30);
}

/*
 *
 */
static void
ViaCrtc2Reset(struct ViaCrtc *Crtc, Bool Reset)
{
    vgaHWPtr hwp = VGAHWPTR(xf86Screens[Crtc->scrnIndex]);

    VIAFUNC(Crtc->scrnIndex);

    if (Reset)
        ViaCrtcMask(hwp, 0x6A, 0x00, 0x40);
    else
        ViaCrtcMask(hwp, 0x6A, 0x40, 0x40);
}

/*
 *
 * Sets up our CRTCs. Only call this from the primary.
 *
 */
void
ViaCrtcInit(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaCrtc *Crtc;

    if (!pVia->Crtc[0]) {
        Crtc = xnfcalloc(1, sizeof(struct ViaCrtc));

        Crtc->scrnIndex = pScrn->scrnIndex;

        Crtc->ID = VIA_CRTC_PRIMARY;
        Crtc->Name = xnfstrdup("Primary");

        Crtc->ModeValidate = ViaCrtc1ModeValidate;
        Crtc->Enable = ViaCrtc1Enable;
        Crtc->Reset = ViaCrtc1Reset;
        Crtc->ModeSet = ViaCrtc1ModeSet;
        Crtc->PLLSet = ViaCrtc1PLLSet;
        Crtc->FIFOSet = ViaCrtc1FIFOSet;
        Crtc->FrameSet = ViaCrtc1FrameSet;
        Crtc->GammaSet = ViaCrtc1GammaSet;

        pVia->Crtc[0] = Crtc;

        /* already set */
        Crtc->bpp = pScrn->bitsPerPixel;

        if (pVia->Chipset == VT3122)
            Crtc->MaxOffset = 32 << 20;
        else
            Crtc->MaxOffset = 128 << 20;
        Crtc->MinPitch = 256; /* be reasonable */
        Crtc->MaxPitch = (64 * 0x3FF) / Crtc->bpp; /* dropped *8/8 */
    }

    if (!pVia->Crtc[1]) {
        Crtc = xnfcalloc(1, sizeof(struct ViaCrtc));

        Crtc->scrnIndex = pScrn->scrnIndex;

        Crtc->ID = VIA_CRTC_SECONDARY;
        Crtc->Name = xnfstrdup("Secondary");

        Crtc->ModeValidate = ViaCrtc2ModeValidate;
        Crtc->Enable = ViaCrtc2Enable;
        Crtc->Reset = ViaCrtc2Reset;
        Crtc->ModeSet = ViaCrtc2ModeSet;
        Crtc->PLLSet = ViaCrtc2PLLSet;
        Crtc->FIFOSet = ViaCrtc2FIFOSet;
        Crtc->FrameSet = ViaCrtc2FrameSet;
        Crtc->GammaSet = ViaCrtc2GammaSet;

        pVia->Crtc[1] = Crtc;

        /* already set */
        Crtc->bpp = pScrn->bitsPerPixel;

        Crtc->MaxOffset = 128 << 20;
        Crtc->MinPitch = 256; /* be reasonable */
        Crtc->MaxPitch = (64 * 0x7FF) / Crtc->bpp; /* dropped *8/8 */
    }
}

/*
 *
 */
void
ViaCrtcInitForFB(ScrnInfoPtr pScrn, struct ViaCrtc *Crtc)
{
    /* we might have gotten hooked to a new Screen */
    Crtc->scrnIndex = pScrn->scrnIndex;

    Crtc->Offset = 0; /* for now */

    /* displayWidth is already 32bytes aligned */
    Crtc->Pitch = pScrn->displayWidth * (Crtc->bpp >> 3);

    Crtc->Active = TRUE;
}

/*
 *
 */
void
ViaCrtcModeSetInitial(ScrnInfoPtr pScrn)
{
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    CARD8 i;

    VIAFUNC(pScrn->scrnIndex);

    i = hwp->readMiscOut(hwp) | 0x22;
    hwp->writeMiscOut(hwp, i);

    /* Sequence registers */
    hwp->writeSeq(hwp, 0x00, 0x00);
    ViaSeqMask(hwp, 0x01, 0x01, 0xDF);
    hwp->writeSeq(hwp, 0x03, 0x00);

    /* 8bits palette, 80 columns, wrap around enable */
    ViaSeqMask(hwp, 0x15, 0xA0, 0xE0);

    /* graphics registers */
    hwp->writeGr(hwp, 0x00, 0x00);
    hwp->writeGr(hwp, 0x01, 0x00);
    hwp->writeGr(hwp, 0x02, 0x00);
    hwp->writeGr(hwp, 0x03, 0x00);
    hwp->writeGr(hwp, 0x04, 0x00);
    hwp->writeGr(hwp, 0x05, 0x40);
    hwp->writeGr(hwp, 0x06, 0x05);
    hwp->writeGr(hwp, 0x07, 0x0F);
    hwp->writeGr(hwp, 0x08, 0xFF);

    /* null VGA FB offsets */
    ViaGrMask(hwp, 0x20, 0, 0xFF);
    ViaGrMask(hwp, 0x21, 0, 0xFF);
    ViaGrMask(hwp, 0x22, 0, 0xFF);

    /* attribute registers */
    for (i = 0; i < 0x10; i++)
        hwp->writeAttr(hwp, i, i);

    hwp->writeAttr(hwp, 0x10, 0x41);
    hwp->writeAttr(hwp, 0x11, 0xFF);
    hwp->writeAttr(hwp, 0x12, 0x0F);
    hwp->writeAttr(hwp, 0x13, 0x00);
    hwp->writeAttr(hwp, 0x14, 0x00);

    /* Enable extended display mode */
    ViaSeqMask(hwp, 0x15, 0x02, 0x02);
}
