/*-
# MOTIF/X-BASED TRIANGLES
#
#  xtriangles.c
#
###
#
#  Copyright (c) 1993 - 2004	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/11/15 X/Windows
  Version 5: 95/10/01 Xt/Motif
  Version 4: 94/06/07 Xt
  Version 3: 93/04/01 Motif
  Version 2: 92/01/06 XView
  Version 1: 91/09/05 SunView
*/

#ifdef WINVER
#include "TrianglesP.h"
#include "wtriangles.h"
#define TITLE "wtriangles"

static TrianglesRec widget;
static HWND Wnd;

#ifndef SCOREPATH
#define SCOREPATH "c:\\Windows"
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(Wnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->triangles.started = b
#else
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#ifdef VMS
#include <unixlib.h>
#define getlogin() cuserid(NULL)
#else
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif
#if HAVE_FCNTL_H
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtSetArg(arg[0], XtNstart, b); XtSetValues(w, arg, 1)
#include "Triangles.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/triangles.xpm"
#endif
#include "icons/triangles.xbm"
#ifndef SCOREPATH
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif

#ifdef HAVE_MOTIF
static const char aboutHelp[] = {
	"Version 7.1.1\n"
	"Send bugs (reports or fixes) to the author: "
	"David Bagley <bagleyd@tux.org>\n"
	"The latest version is at: "
	"http://www.tux.org/~bagleyd/puzzles.html\n"
};
static const char optionsHelp[] = {
	"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]] "
	"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
	"[-{foreground|fg} {color}] [-{background|bg} {color}] "
	"[-{border|bd} {color}] [-tile {color}] [-size {int}]\n"
	"[-[no]install] [-picture {filename}] [-delay msecs] "
	"[-[no]sound] [-bumpsound {filename}] [-base {int}]\n"
	"[-{font|fn} {fontname}] [-username {string}]\n"
};
#endif
#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
	"A sliding tile puzzle.  A triangular version of the "
	"15 puzzle which was originally made around 1879 (the 14-15\n"
	"puzzle of Sam Loyd was made soon after).   T. R. Dawson "
	"in Original Puzzles has a similar puzzle with 4 diamond-\n"
	"and 6 triangular-shaped pieces.  This is easier to solve "
	"than 15 puzzle.\n"
};
static const char featuresHelp[] = {
	"Click \"mouse-left\" button to move a tile.  Clicks on "
	"spaces, or clicks on tiles that are not in line with\n"
	"two spaces will not move tiles.  Notice, one can move "
	"more than  one tile at a time  by clicking on\n"
	"the tile farthest from the spaces but still in line with the "
	"spaces.\n"
	"\n"
	"Click \"mouse-right\" button, or press \"R\" or \"r\" "
	"keys, to randomize the puzzle.\n"
	"\n"
	"Press \"I\" or \"i\" keys to increase the number of tiles.\n"
	"\n"
	"Press \"O\" or \"o\" keys to decrease the number of tiles.\n"
	"\n"
	"\"S\" or \"s\" keys reserved for the auto-solver "
	"(not implemented).\n"
	"\n"
	"Press \"U\" or \"u\" keys to undo a move.\n"
	"\n"
	"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
	"\n"
	"Press \"W\" or \"w\" keys to save a puzzle.\n"
	"\n"
	"Press \">\" or \".\" keys to speed up the movement of tiles.\n"
	"\n"
	"Press \"<\" or \",\" keys to slow down the movement of tiles.\n"
	"\n"
	"Press \"@\" or \"2\" keys to toggle the sound.\n"
	"\n"
	"Press \"Esc\" key to hide program.\n"
	"\n"
	"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
	"\n"
	"Use the key pad, \"R\" keys, or arrow keys to move without "
	"the mouse.\n"
	"Key pad is defined for Triangles as:\n"
	"7   9  Upper Left, Upper Right\n"
	"  ^\n"
	"4< >6  Left, Right\n"
	"  v\n"
	"1   3  Lower Left, Lower Right\n"
};
static const char referencesHelp[] = {
	"L. E. Horden, Sliding Piece Puzzles  (Recreations in "
	"Mathematics Series), Oxford University Press  1986,\n"
	"pp 1, 148, 149.\n"
	"Jerry Slocum & Jack Botermans, Puzzles Old & New (How to Make "
	"and Solve Them),  University of Washington\n"
	"Press, Seattle 1987, p 126, 127.\n"
};
#endif
static const char solveHelp[] = {
	"Auto-solver: sorry, not implemented.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "triangles.scores"
#endif

#define MAXTILES 16
#define NEVER -1
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} GameRecord;

static GameRecord trianglesRecord[MAXTILES - MINTILES + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";

#ifdef HAVE_MOTIF
#define MINSPEED 1
#define MAXSPEED 50
#ifdef ANIMATE
static Widget speed;
#endif
static Widget moves, record, message, tile;
static char buff[21];
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, randomizeDialog;
#else
static char titleDsp[TITLELEN] = "";
#endif
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char usernameDsp[USERNAMELEN] = "Guest";
#else
static Pixmap trianglesIcon = None;
static Widget topLevel, triangles;
static Arg arg[3];
static char *progDsp;
static char usernameDsp[USERNAMELEN] = "";
#endif

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(arg[0], XmNlabelString, xmstr);
	XtSetValues(w, arg, 1);
}
#else
static void
PrintState(
#ifndef WINVER
Widget w,
#endif
int size, int moves, char *msg)
{
	(void) sprintf(titleDsp, "%s: %d@ (%d/%s) - %s",
		progDsp, size, moves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(Wnd, (LPSTR) titleDsp);
#else
	XtSetArg(arg[0], XtNtitle, titleDsp);
	XtSetValues(w, arg, 1);
#endif
}
#endif

static void
InitRecords(void)
{
	int i;

	for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
		trianglesRecord[i].score = NEVER;
		(void) strncpy(trianglesRecord[i].name, NOACCESS, USERNAMELEN);
	}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int i, n;
	char username[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "r")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (i = 0; i < MAXTILES - MINTILES + 1; i++) {
		(void) fscanf(fp, "%d %s\n", &n, username);
		if (n <= trianglesRecord[i].score ||
				trianglesRecord[i].score <= NEVER) {
			trianglesRecord[i].score = n;
			(void) strncpy(trianglesRecord[i].name, username,
				USERNAMELEN);
		}
	}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int i;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "w")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockfile[FILENAMELEN];

		(void) strncpy(lockfile, name, FILENAMELEN - 6);
		(void) strcat(lockfile, ".lock");
		while (((lfd = open(lockfile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
			  "Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
			  "Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (i = 0; i < MAXTILES - MINTILES + 1; i++)
			(void) fprintf(fp, "%d %s\n",
				trianglesRecord[i].score,
				trianglesRecord[i].name);
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockfile);
#endif
		(void) fclose(fp);
	}
	free(lname);
	free(fname);
}

static void
PrintRecord(int size)
{
	int i = size - MINTILES;

	if (size > MAXTILES) {
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGELEN);
	} else if (trianglesRecord[i].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			trianglesRecord[i].score, trianglesRecord[i].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static Boolean
HandleSolved(int counter, int size)
{
	int i = size - MINTILES;

	if (size <= MAXTILES && (counter < trianglesRecord[i].score ||
			trianglesRecord[i].score <= NEVER)) {
		ReadRecords();	/* Maybe its been updated by another */
		trianglesRecord[i].score = counter;
		(void) strncpy(trianglesRecord[i].name, usernameDsp,
			USERNAMELEN);
		WriteRecords();
		PrintRecord(size);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
TrianglesWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	int size;
	char *username;

#ifdef WINVER
	InitializeTriangles(w, brush);

	size = w->triangles.size;
	username = w->triangles.username;
#else

	XtVaGetValues(w,
		XtNuserName, &username,
		XtNsize, &size, NULL);
#ifdef HAVE_MOTIF
	if (size > MAXTILES)
		XtVaSetValues(tile, XmNmaximum, size, NULL);
	XmScaleSetValue(tile, size);
#ifdef ANIMATE
	{
		int delay;

		XtVaGetValues(w, XtNdelay, &delay, NULL);
		XmScaleSetValue(speed, MAXSPEED + MINSPEED - delay - 1);
	}
#endif
#endif
#endif
	SET_STARTED(w, False);
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(usernameDsp, username, USERNAMELEN);
#endif
	if (!strcmp(username, "") || !strcmp(username, "(null)") ||
			!strcmp(username, NOACCESS) ||
			!strcmp(username, NOBODY)) {
		/* The NOACCESS is not necessary, but it may stop silliness. */
#ifdef WINVER
		(void) strncpy(usernameDsp, username, USERNAMELEN);
#else
		(void) sprintf(usernameDsp, "%s", getlogin());
		if (!strcmp(usernameDsp, "") ||
				!strcmp(usernameDsp, "(null)") ||
				!strcmp(usernameDsp, NOACCESS) ||
				!strcmp(usernameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(usernameDsp, "%s", "guest");
#endif
	}
	PrintRecord(size);
#ifndef HAVE_MOTIF
	PrintState(
#ifndef WINVER
		XtParent(w),
#endif
		size, movesDsp, messageDsp);
#endif
}

#ifdef WINVER
void
SetTriangles(TrianglesWidget w, int reason)
#else
static void
CallbackTriangles(Widget w, caddr_t clientData,
		trianglesCallbackStruct * callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	int size;
	Boolean cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	size = w->triangles.size;
	cheat = w->triangles.cheat;
#else
	XtVaGetValues(w,
		XtNsize, &size,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case TRIANGLES_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
			break;
#ifndef WINVER
		case TRIANGLES_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(randomizeDialog);
#else
			XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
			XtSetValues(triangles, arg, 1);
#endif
			break;
#endif
		case TRIANGLES_SOLVE_MESSAGE:
#ifdef WINVER
			(void) MessageBox(w->core.hWnd, solveHelp,
				"Auto-Solve", MB_OK);
#else
#ifdef HAVE_MOTIF
			XtManageChild(solveDialog);
#else
			(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
			break;
		case TRIANGLES_RESTORE:
		case TRIANGLES_RESET:
			movesDsp = 0;
			break;
		case TRIANGLES_BLOCKED:
			(void) strncpy(messageDsp, "Blocked", MESSAGELEN);
			break;
		case TRIANGLES_SPACE:
#if 0
			/* Too annoying */
			(void) strncpy(messageDsp, "Spaces can not slide",
				MESSAGELEN);
#endif
			break;
		case TRIANGLES_IGNORE:
			(void) strncpy(messageDsp, "Randomize to start",
				MESSAGELEN);
			break;
		case TRIANGLES_MOVED:
			movesDsp++;
			SET_STARTED(w, True);
			break;
		case TRIANGLES_SOLVED:
			if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", usernameDsp);
			else if (HandleSolved(movesDsp, size))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", usernameDsp);
			else
				(void) strncpy(messageDsp, "Solved!",
					MESSAGELEN);
			SET_STARTED(w, False);
			break;
		case TRIANGLES_RANDOMIZE:
			movesDsp = 0;
			SET_STARTED(w, False);
			break;
		case TRIANGLES_DEC:
			movesDsp = 0;
			size--;
			PrintRecord(size);
#ifdef WINVER
			w->triangles.size = size;
#else
			XtSetArg(arg[0], XtNsize, size);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmScaleSetValue(tile, size);
			if (size >= MAXTILES)
				XtVaSetValues(tile, XmNmaximum, size, NULL);
#endif
#endif
			break;
		case TRIANGLES_INC:
			movesDsp = 0;
			size++;
			PrintRecord(size);
#ifdef WINVER
			w->triangles.size = size;
#else
			XtSetArg(arg[0], XtNsize, size);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			if (size > MAXTILES)
				XtVaSetValues(tile, XmNmaximum, size, NULL);
			XmScaleSetValue(tile, size);
#endif
#endif
			break;
		case TRIANGLES_COMPUTED:
			SET_STARTED(w, False);
			break;
		case TRIANGLES_UNDO:
			movesDsp--;
			SET_STARTED(w, True);
			break;
#if defined(HAVE_MOTIF) && defined(ANIMATE)
		case TRIANGLES_SPEED:
			{
				int oldDelay, delay;

				XtVaGetValues(triangles,
					XtNdelay, &delay, NULL);
				oldDelay = delay;
				if (delay > MAXSPEED - MINSPEED)
					delay = MAXSPEED - MINSPEED;
				if (delay < 0)
					delay = 0;
				if (delay != oldDelay) {
					XtSetArg(arg[0], XtNdelay, delay);
					XtSetValues(w, arg, 1);
				}
				XmScaleSetValue(speed,
					MAXSPEED + MINSPEED - delay - 1);
			}
			return;
#endif
	}
#ifdef HAVE_MOTIF
	PrintState(message, messageDsp);
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
#else
	PrintState(
#ifndef WINVER
		XtParent(w),
#endif
		size, movesDsp, messageDsp);
#endif
}

#ifdef WINVER
static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	Wnd = widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.triangles.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			EnterTriangles(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.triangles.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			LeaveTriangles(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyTriangles(brush);
			break;
		case WM_SIZE:
			ResizeTriangles(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ExposeTriangles(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			RandomizeTriangles(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			SelectTriangles(&widget, LOWORD(lParam), HIWORD(lParam));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ReleaseTriangles(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDM_GET:
					GetTriangles(&widget);
					ResizeTriangles(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WriteTriangles(&widget);
					break;
				case IDM_EXIT:
					DestroyTriangles(brush);
					break;
				case IDM_HIDE:
					HideTriangles(&widget);
					break;
				case IDM_CLEAR:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					ClearTriangles(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					UndoTriangles(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_RANDOMIZE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					RandomizeTriangles(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					SolveTriangles(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_SPEED:
					SpeedTriangles(&widget);
					break;
				case IDM_SLOW:
					SlowTriangles(&widget);
					break;
				case IDM_SOUND:
					SoundTriangles(&widget);
					break;
				case IDM_TR:
				case IDM_RIGHT:
				case IDM_BR:
				case IDM_BL:
				case IDM_LEFT:
				case IDM_TL:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					(void) MoveTriangles(&widget, (int) LOWORD(wParam) - IDM_TR);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_DEC:
					DecrementTriangles(&widget);
					SizeTriangles(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_INC:
					IncrementTriangles(&widget);
					SizeTriangles(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
					 	"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
					 	"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
					 	"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
Usage(char * programName)
{
	(void) fprintf(stderr, "usage: %s\n", programName);
	(void) fprintf(stderr,
		"\t[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n");
	(void) fprintf(stderr,
		"\t[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n");
	(void) fprintf(stderr,
		"\t[-{foreground|fg} {color}] [-{background|bg} {color}]\n");
	(void) fprintf(stderr,
		"\t[-{border|bd} {color}] [-tile {color}] [-size {int}]\n");
	(void) fprintf(stderr,
		"\t[-[no]install] [-picture {filename}] [-delay msecs]\n");
	(void) fprintf(stderr,
		"\t[-[no]sound] [-bumpsound {filename}] [-base {int}]\n");
	(void) fprintf(stderr,
		"\t[-{font|fn} {fontname}] [-username {string}]\n");
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*triangles.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*triangles.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*triangles.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*triangles.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*triangles.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*triangles.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*triangles.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*triangles.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*triangles.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*triangles.tileBorder", XrmoptionSepArg, NULL},
	{(char *) "-tile", (char *) "*triangles.tileColor", XrmoptionSepArg, NULL},
	{(char *) "-size", (char *) "*triangles.size", XrmoptionSepArg, NULL},
	{(char *) "-install", (char *) "*triangles.install", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noinstall", (char *) "*triangles.install", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-picture", (char *) "*triangles.picture", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*triangles.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*triangles.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*triangles.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-bumpsound", (char *) "*triangles.bumpSound", XrmoptionSepArg, NULL},
	{(char *) "-base", (char *) "*triangles.base", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*triangles.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*triangles.font", XrmoptionSepArg, NULL},
	{(char *) "-username", (char *) "*triangles.userName", XrmoptionSepArg, NULL}
};

#ifdef HAVE_MOTIF
static void
CallbackTrianglesRandomize(Widget w, XtPointer clientData,
		XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
		XtSetValues(triangles, arg, 1);
	}
}

static void
TileSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int size = cbs->value, old;

	XtVaGetValues(triangles,
		XtNsize, &old, NULL);
	if (old == size)
		return;
	XtVaSetValues(triangles,
		XtNsize, size, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(size);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

#ifdef ANIMATE
static void
SpeedSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int delay = MAXSPEED + MINSPEED - cbs->value - 1, oldDelay;

	XtVaGetValues(triangles,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
		XtVaSetValues(triangles, XtNdelay, delay, NULL);
	}
}
#endif

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == 2)
		exit(0);
	XtSetArg(arg[0], XtNmenu, val);
	XtSetValues(triangles, arg, 1);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;
	XtSetArg(arg[0], XtNmenu, val + 3); /* GWQ */
	XtSetValues(triangles, arg, 1);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, tilesRowCol, messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString clearString, undoString, randomizeString, solveString;
	XmString incrementString, decrementString;
	XmString speedString, slowString, soundString;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Triangles",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#if HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) triangles_xpm, &trianglesIcon, NULL,
			&xpmAttributes);
	}
	if (trianglesIcon == (Pixmap) NULL)
#endif
		trianglesIcon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) triangles_bits,
			triangles_width, triangles_height);
	XtSetArg(arg[0], XtNiconPixmap, trianglesIcon);
#ifdef HAVE_MOTIF
	/* not XmEXPLICIT */
	XtSetArg(arg[1], XmNkeyboardFocusPolicy, XmPOINTER);
	XtSetValues(topLevel, arg, 2);
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	clearString = XmStringCreateSimple((char *) "Clear");
	undoString = XmStringCreateSimple((char *) "Undo");
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "(Solve)");
	incrementString = XmStringCreateSimple((char *) "Increment");
	decrementString = XmStringCreateSimple((char *) "Decrement");
	speedString = XmStringCreateSimple((char *) ">Speed");
	slowString = XmStringCreateSimple((char *) "<Slow");
	soundString = XmStringCreateSimple((char *) "@Sound");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, incrementString, 'I', NULL, NULL,
		XmVaPUSHBUTTON, decrementString, 'D', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
		XmVaPUSHBUTTON, soundString, '@', NULL, NULL,
		NULL);
	XmStringFree(clearString);
	XmStringFree(undoString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(incrementString);
	XmStringFree(decrementString);
	XmStringFree(speedString);
	XmStringFree(slowString);
	XmStringFree(soundString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) CallbackTrianglesRandomize);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move tile", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

	tilesRowCol = XtVaCreateManagedWidget("tilesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	tile = XtVaCreateManagedWidget("tile",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size", 5,
		XmNminimum, MINTILES,
		XmNmaximum, MAXTILES,
		XmNvalue, DEFAULTTILES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(tile, XmNvalueChangedCallback,
		(XtCallbackProc) TileSlider, (XtPointer) NULL);
#ifdef ANIMATE
	speed = XtVaCreateManagedWidget("Animation Speed",
		xmScaleWidgetClass, tilesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MINSPEED,
		XmNmaximum, MAXSPEED,
		XmNvalue, MAXSPEED - DEFAULTDELAY,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speed, XmNvalueChangedCallback,
		(XtCallbackProc) SpeedSlider, (XtPointer) NULL);
#endif
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Triangles! (use mouse or keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);
	triangles = XtCreateManagedWidget("triangles",
		trianglesWidgetClass, mainPanel, NULL, 0);
#else
	XtSetArg(arg[1], XtNinput, True);
	XtSetValues(topLevel, arg, 2);
	triangles = XtCreateManagedWidget("triangles",
		trianglesWidgetClass, topLevel, NULL, 0);
#endif
	XtAddCallback(triangles, XtNselectCallback,
		(XtCallbackProc) CallbackTriangles, (XtPointer) NULL);
	Initialize(triangles);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(triangles), (unsigned int) AnyButton, AnyModifier,
		XtWindow(triangles), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(triangles),
		XCreateFontCursor(XtDisplay(triangles), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
