/*
# X-BASED THREED
#
#  Threed.c
#
###
#
#  Copyright (c) 1994 - 2004	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "useful",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Threed */

#include "ThreedP.h"

#ifdef WINVER
#ifndef INIFILE
#define INIFILE "wthreed.ini"
#endif
#ifndef DATAFILE
#define DATAFILE "c:\\Windows\threed.dat"
#endif

#define SECTION "setup"
#else
#include <math.h>
#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

#ifndef DATAFILE
#define DATAFILE "/usr/games/threed.dat"
#endif

static void InitializeThreeD(Widget request, Widget renew);
static void ExposeThreeD(Widget renew, XEvent * event, Region region);
static void ResizeThreeD(ThreeDWidget w);
static void DestroyThreeD(Widget old);
static Boolean SetValuesThreeD(Widget current, Widget request, Widget renew);
static void QuitThreeD(ThreeDWidget w, XEvent * event, char **args,
	int n_args);
static void HideThreeD(ThreeDWidget w, XEvent * event, char **args,
	int n_args);
static void SurfaceThreeD(ThreeDWidget w, XEvent * event, char **args,
	int n_args);
static void ObjectThreeD(ThreeDWidget w, XEvent * event, char **args,
	int n_args);
static void MoveOutThreeD(ThreeDWidget w, XEvent * event, char **args,
	int n_args);
static void MoveLowerPhiThreeD(ThreeDWidget w, XEvent * event, char **args,
	int n_args);
static void MoveLowerThetaThreeD(ThreeDWidget w, XEvent * event, char **args,
	int n_args);
static void MoveInThreeD(ThreeDWidget w, XEvent * event, char **args,
	int n_args);
static void MoveRaiseThetaThreeD(ThreeDWidget w, XEvent * event, char **args,
	int n_args);
static void MoveRaisePhiThreeD(ThreeDWidget w, XEvent * event, char **args,
	int n_args);

static char defaultTranslationsThreeD[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>o: MoveOut()\n\
 <KeyPress>KP_Divide: MoveOut()\n\
 <KeyPress>R5: MoveOut()\n\
 <KeyPress>Up: MoveRaisePhi()\n\
 <KeyPress>osfUp: MoveRaisePhi()\n\
 <KeyPress>KP_Up: MoveRaisePhi()\n\
 <KeyPress>KP_8: MoveRaisePhi()\n\
 <KeyPress>R8: MoveRaisePhi()\n\
 <Btn4Down>: MoveRaisePhi()\n\
 <KeyPress>Left: MoveRaiseTheta()\n\
 <KeyPress>osfLeft: MoveRaiseTheta()\n\
 <KeyPress>KP_Left: MoveRaiseTheta()\n\
 <KeyPress>KP_4: MoveRaiseTheta()\n\
 <KeyPress>R10: MoveRaiseTheta()\n\
 <KeyPress>i: MoveIn()\n\
 <KeyPress>Begin: MoveIn()\n\
 <KeyPress>KP_5: MoveIn()\n\
 <KeyPress>R11: MoveIn()\n\
 <KeyPress>Right: MoveLowerTheta()\n\
 <KeyPress>osfRight: MoveLowerTheta()\n\
 <KeyPress>KP_Right: MoveLowerTheta()\n\
 <KeyPress>KP_6: MoveLowerTheta()\n\
 <KeyPress>R12: MoveLowerTheta()\n\
 <KeyPress>Down: MoveLowerPhi()\n\
 <KeyPress>osfDown: MoveLowerPhi()\n\
 <KeyPress>KP_Down: MoveLowerPhi()\n\
 <KeyPress>KP_2: MoveLowerPhi()\n\
 <KeyPress>R14: MoveLowerPhi()\n\
 <Btn5Down>: MoveLowerPhi()\n\
 <KeyPress>s: Surface()\n\
 <KeyPress>b: Object()";

static XtActionsRec actionsListThreeD[] = {
	{(char *) "Quit", (XtActionProc) QuitThreeD},
	{(char *) "Hide", (XtActionProc) HideThreeD},
	{(char *) "MoveOut", (XtActionProc) MoveOutThreeD},
	{(char *) "MoveLowerPhi", (XtActionProc) MoveLowerPhiThreeD},
	{(char *) "MoveLowerTheta", (XtActionProc) MoveLowerThetaThreeD},
	{(char *) "MoveIn", (XtActionProc) MoveInThreeD},
	{(char *) "MoveRaiseTheta", (XtActionProc) MoveRaiseThetaThreeD},
	{(char *) "MoveRaisePhi", (XtActionProc) MoveRaisePhiThreeD},
	{(char *) "Surface", (XtActionProc) SurfaceThreeD},
	{(char *) "Object", (XtActionProc) ObjectThreeD},
};

static XtResource resourcesThreeD[] = {
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(ThreeDWidget, core.width), XtRString, (caddr_t) "400"}
	,
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(ThreeDWidget, core.height), XtRString, (caddr_t) "200"}
	,
	{XtNdistance, XtCDistance, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.distance), XtRString, (caddr_t) "1"},
	{XtNthetaDegrees, XtCThetaDegrees, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.thetaDegrees), XtRString, (caddr_t) "0"},
	{XtNphiDegrees, XtCPhiDegrees, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.phiDegrees), XtRString, (caddr_t) "0"},
	{XtNsurface, XtCSurface, XtRBoolean, sizeof (Boolean),
	 XtOffset(ThreeDWidget, threed.surface), XtRString, (caddr_t) "TRUE"}
	,
	{XtNobject, XtCObject, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.object), XtRString, (caddr_t) "0"},
	{XtNobjectName, XtCObjectName, XtRString, sizeof (String),
	 XtOffset(ThreeDWidget, threed.name), XtRString, (caddr_t) "None"}
	,
	{XtNobjectNumber, XtCObjectNumber, XtRInt, sizeof (int),
	 XtOffset(ThreeDWidget, threed.numObjects), XtRString, (caddr_t) "0"},
	{XtNobjectList, XtCObjectList, XtRString, sizeof (String),
	 XtOffset(ThreeDWidget, threed.list), XtRString, (caddr_t) "None"}
	,
	{XtNwhiteBrush, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.shadeColor[WHITE_BRUSH]), XtRString,
	 (caddr_t) "Yellow"}
	,
	{XtNltgrayBrush, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.shadeColor[LTGRAY_BRUSH]), XtRString,
	 (caddr_t) "Green"}
	,
	{XtNgrayBrush, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.shadeColor[GRAY_BRUSH]), XtRString,
	 (caddr_t) "Red"}
	,
	{XtNdkgrayBrush, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.shadeColor[DKGRAY_BRUSH]), XtRString,
	 (caddr_t) "Blue"}
	,
	{XtNblackBrush, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.shadeColor[BLACK_BRUSH]), XtRString,
	 (caddr_t) "Black"}
	,
	{XtNanotherBrush, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.shadeColor[ANOTHER_BRUSH]), XtRString,
	 (caddr_t) "Brown"}
	,
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(ThreeDWidget, threed.foreground), XtRString, (caddr_t) "Black"}
	,
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(ThreeDWidget, threed.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverse, XtCReverse, XtRBoolean, sizeof (Boolean),
	 XtOffset(ThreeDWidget, threed.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(ThreeDWidget, threed.select), XtRCallback, NULL}
};

ThreeDClassRec threedClassRec = {
	{
	 (WidgetClass) & widgetClassRec,	/* superclass */
	 (char *) "ThreeD",		/* class name */
	 sizeof (ThreeDRec),		/* widget size */
	 NULL,			/* class initialize */
	 NULL,			/* class part initialize */
	 FALSE,			/* class inited */
	 (XtInitProc) InitializeThreeD,	/* initialize */
	 NULL,			/* initialize hook */
	 XtInheritRealize,		/* realize */
	 actionsListThreeD,		/* actions */
	 XtNumber(actionsListThreeD),	/* num actions */
	 resourcesThreeD,		/* resources */
	 XtNumber(resourcesThreeD),	/* num resources */
	 NULLQUARK,			/* xrm class */
	 TRUE,			/* compress motion */
	 TRUE,			/* compress exposure */
	 TRUE,			/* compress enterleave */
	 TRUE,			/* visible interest */
	 (XtWidgetProc) DestroyThreeD,	/* destroy */
	 (XtWidgetProc) ResizeThreeD,	/* resize */
	 (XtExposeProc) ExposeThreeD,	/* expose */
	 (XtSetValuesFunc) SetValuesThreeD,	/* set values */
	 NULL,			/* set values hook */
	 XtInheritSetValuesAlmost,	/* set values almost */
	 NULL,			/* get values hook */
	 NULL,			/* accept focus */
	 XtVersion,			/* version */
	 NULL,			/* callback private */
	 defaultTranslationsThreeD,	/* tm table */
	 NULL,			/* query geometry */
	 NULL,			/* display accelerator */
	 NULL				/* extension */
	 }
	,
	{
	 0				/* ignore */
	 }
};

WidgetClass threedWidgetClass = (WidgetClass) & threedClassRec;

static void
SetThreeD(ThreeDWidget w, int reason)
{
	threedCallbackStruct cb;

	cb.reason = reason;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}

#endif

#define RT_ANGLE 90.0
#define ST_ANGLE 180.0
#define RADIANS(x) (M_PI*(x)/ST_ANGLE)
#define DEGREES(x) ((x)/M_PI*ST_ANGLE)

static ThreeDWidget *W;
static int O;

void
intCat(char ** string, const char * var1, const int var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + 21))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	(void) sprintf(*string, "%s%d", var1, var2);
}

static void
stringCat(char ** string, const char * var1, const char * var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + strlen(var2) + 1))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	(void) sprintf(*string, "%s%s", var1, var2);
}

static void
CheckGraphicsThreeD(ThreeDWidget w)
{
	char *buf1 = NULL, *buf2 = NULL;

	if (w->threed.distance < MINDISTANCE ||
			w->threed.distance > MAXDISTANCE) {
		w->threed.distance = (MAXDISTANCE + MINDISTANCE) / 2;
		intCat(&buf1,
			"Distance out of bounds, use ",
			MINDISTANCE);
		stringCat(&buf2, buf1, "..");
		free(buf1);
		intCat(&buf1, buf2, MAXDISTANCE);
		free(buf2);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, w->threed.distance);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	if (w->threed.thetaDegrees < MINDEGREES ||
			w->threed.thetaDegrees > MAXDEGREES)
		w->threed.thetaDegrees = w->threed.thetaDegrees % NUMDEGREES;
	if (w->threed.phiDegrees < MINDEGREES ||
			w->threed.phiDegrees > MAXDEGREES)
		w->threed.phiDegrees = w->threed.phiDegrees % NUMDEGREES;
	if (w->threed.object < 0 || w->threed.object >= w->threed.numObjects) {
		w->threed.object = 0;
		intCat(&buf1, "Object type number out of bounds, use 0..",
			w->threed.numObjects - 1);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, w->threed.object);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}

}

static int
#ifdef WINVER
__cdecl
#endif
CompareObject(const void *elem1, const void *elem2)
{
	if ((*W)->threed.objects[O].vertex[((Surface *)
			elem1)->depthIndex].eye.z >
	    (*W)->threed.objects[O].vertex[((Surface *)
			elem2)->depthIndex].eye.z)
		return -1;
	if ((*W)->threed.objects[O].vertex[((Surface *)
			elem1)->depthIndex].eye.z <
	    (*W)->threed.objects[O].vertex[((Surface *)
			elem2)->depthIndex].eye.z)
		return 1;
	return 0;
}

#ifdef WINVER
static void
POLYLINE(ThreeDWidget w, GC color, const Point * poly, int n,
		Boolean origin)
{
	/* CoordModePrevious -> CoordModeOrigin */
	Point *temp = NULL;
	int pt;

	if (!origin) {
		if (!(temp = (Point *) malloc(sizeof (Point) * n))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		temp[0] = poly[0];
		for (pt = 1; pt < n; pt++) {
			temp[pt].x = temp[pt - 1].x + poly[pt].x,
			temp[pt].y = temp[pt - 1].y + poly[pt].y;
		}
	}
	w->threed.hPen = CreatePen(PS_SOLID, 1, color);
	w->threed.hOldPen = (HPEN) SelectObject(w->core.hDC, w->threed.hPen);
	(void) Polyline(w->core.hDC, (origin) ? poly : temp, n);
	(void) SelectObject(w->core.hDC, w->threed.hOldPen);
	(void) DeleteObject(w->threed.hPen);
	if (!origin) {
		free(temp);
	}
}
#else
#define POLYLINE(w,c,l,n,o) XDrawLines(XtDisplay(w), XtWindow(w), c, \
  l, n,(o) ? CoordModeOrigin : CoordModePrevious)
#endif

#ifdef WINVER
static void
POLYGON(ThreeDWidget w, GC color, GC lineColor, const Point * poly, int n,
		Boolean origin)
{
	/* CoordModePrevious -> CoordModeOrigin */
	Point *temp = NULL;
	int pt;

	if (!origin) {
		if (!(temp = (Point *) malloc(sizeof (Point) * n))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		temp[0] = poly[0];
		for (pt = 1; pt < n; pt++) {
			temp[pt].x = temp[pt - 1].x + poly[pt].x,
			temp[pt].y = temp[pt - 1].y + poly[pt].y;
		}
	}
	w->threed.hPen = CreatePen(PS_SOLID, 1, lineColor);
	w->threed.hOldPen = (HPEN) SelectObject(w->core.hDC, w->threed.hPen);
	w->threed.hBrush = CreateSolidBrush(color);
	w->threed.hOldBrush = (HBRUSH) SelectObject(w->core.hDC,
		w->threed.hBrush);
	(void) Polygon(w->core.hDC, (origin) ? poly : temp, n);
	(void) SelectObject(w->core.hDC, w->threed.hOldBrush);
	(void) DeleteObject(w->threed.hBrush);
	(void) SelectObject(w->core.hDC, w->threed.hOldPen);
	(void) DeleteObject(w->threed.hPen);
	if (!origin) {
		free(temp);
	}
}
#else
  /* Complex/Convex: F16 has some complex planes :) */
#define POLYGON(w,c,lc,l,n,o) XFillPolygon(XtDisplay(w), XtWindow(w), c, \
  l, n, Complex, (o) ? CoordModeOrigin : CoordModePrevious); \
  XDrawLines(XtDisplay(w), XtWindow(w), lc, \
  l, (n)+1,(o) ? CoordModeOrigin : CoordModePrevious)
#endif

static void
DrawObject(ThreeDWidget w, int o)
{
	double sinTheta, cosTheta, sinPhi, cosPhi;
	double s1 = 0.0, s2 = 0.0, s3 = 0.0;
	Point3D *v1, *v2, *v3;
	Point points[10];
	int surface, vertex, mapIndex, vertexIndex, loop;

	if (!w->threed.objects)
		return;
	cosTheta = cos(RADIANS((double) w->threed.thetaDegrees));
	sinTheta = sin(RADIANS((double) w->threed.thetaDegrees));
	cosPhi = cos(RADIANS((double) w->threed.phiDegrees));
	sinPhi = sin(RADIANS((double) w->threed.phiDegrees));
	for (loop = 0; loop < w->threed.objects[o].numVertices; loop++) {
	    w->threed.objects[o].vertex[loop].eye.x = -
		(w->threed.objects[o].world[loop].x * sinTheta) +
		(w->threed.objects[o].world[loop].y * cosTheta);
	    w->threed.objects[o].vertex[loop].eye.y = -
		(w->threed.objects[o].world[loop].x * cosTheta * cosPhi) -
		(w->threed.objects[o].world[loop].y * sinTheta * cosPhi) +
		(w->threed.objects[o].world[loop].z * sinPhi);
	    w->threed.objects[o].vertex[loop].eye.z = -
		(w->threed.objects[o].world[loop].x * cosTheta * sinPhi) -
		(w->threed.objects[o].world[loop].y * sinTheta * sinPhi) -
		(w->threed.objects[o].world[loop].z * cosPhi) +
		(double) w->threed.distance;
	    if (w->threed.objects[o].vertex[loop].eye.z < 0.000001 &&
	      w->threed.objects[o].vertex[loop].eye.z >= 0.0)
		w->threed.objects[o].vertex[loop].eye.z = 0.000001;
	    if (w->threed.objects[o].vertex[loop].eye.z < 0.0 &&
	      w->threed.objects[o].vertex[loop].eye.z > -0.000001)
		w->threed.objects[o].vertex[loop].eye.z = -0.000001;
	    w->threed.objects[o].vertex[loop].screen.x = (int) (VIEWRATIO *
		(w->threed.objects[o].vertex[loop].eye.x /
		w->threed.objects[o].vertex[loop].eye.z) *
		w->threed.size + w->threed.center.x);
	    w->threed.objects[o].vertex[loop].screen.y =
		(int) (-VIEWRATIO *
		(w->threed.objects[o].vertex[loop].eye.y /
		w->threed.objects[o].vertex[loop].eye.z) * w->threed.size +
		w->threed.center.y);
	}
	/* Draw graphical object */
	if (w->threed.surface) {
		W = &w;
		O = o;
		(void) qsort((void *) w->threed.objects[o].info,
			(unsigned int) w->threed.objects[o].numSurfaces,
			sizeof(Surface), CompareObject);
	}
	for (surface = 0; surface < w->threed.objects[o].numSurfaces; surface++)
	{
		mapIndex = w->threed.objects[o].info[surface].mapIndex;
		if (w->threed.surface) {
		  v1 = &w->threed.objects[o].vertex
		    [w->threed.objects[o].map[mapIndex]].eye;
		  v2 = &w->threed.objects[o].vertex
		    [w->threed.objects[o].map[mapIndex + 1]].eye;
		  v3 = &w->threed.objects[o].vertex
		    [w->threed.objects[o].map[mapIndex + 2]].eye;
		  s1 = v1->x * (v2->y * v3->z - v3->y * v2->z);
		  s2 = v2->x * (v3->y * v1->z - v1->y * v3->z);
		  s3 = v3->x * (v1->y * v2->z - v2->y * v1->z);
		}
		if (!w->threed.surface || -s1 - s2 - s3 <= 0.0) {
		  for (vertex = 0;
		       vertex < w->threed.objects[o].info[surface].numVertices;
		       vertex++, mapIndex++) {
		      vertexIndex = w->threed.objects[o].map[mapIndex];
		      points[vertex].x =
		       w->threed.objects[o].vertex[vertexIndex].screen.x;
		      points[vertex].y =
		       w->threed.objects[o].vertex[vertexIndex].screen.y;
		  }
		  points[w->threed.objects[o].info[surface].numVertices].x =
		    points[0].x;
		  points[w->threed.objects[o].info[surface].numVertices].y =
		    points[0].y;

		  if (!w->threed.surface || s1 + s2 + s3 == 0.0) {
		      POLYLINE(w, w->threed.shadeGC[w->threed.objects[o].
			info[surface].brushColor],
			&points[0], vertex + 1, True);
		   } else
		      POLYGON(w, w->threed.shadeGC[w->threed.objects[o].
			info[surface].brushColor],
			w->threed.shadeGC[w->threed.objects[o].
			info[surface].brushColor],
			&points[0], vertex, True);
		}
	}
}

static void
RepaintGraphicsThreeD(ThreeDWidget w)
{
#ifndef WINVER
	XClearWindow(XtDisplay(w), XtWindow(w));
#endif
	DrawObject(w, w->threed.object);
}

#ifdef WINVER
static FILE *
OpenIni(const HWND hWnd, const char *filename)
{
	unsigned short wReturn;
	char buf[256], szBuf[144];
	FILE *fp;

#ifdef DEBUG
	(void) fprintf(stderr, "INI=%s\n", filename);
#endif
	if ((fp = fopen(filename, "r")) == NULL) {
		wReturn = GetWindowsDirectory((LPSTR) szBuf, sizeof (szBuf));
		if (!wReturn || wReturn > sizeof (szBuf)) {
			(void) MessageBox(hWnd, (!wReturn) ? "threed: function failed" :
			"threed: buffer is too small", "GetWindowsDirectory",
					MB_ICONEXCLAMATION);
#ifdef DEBUG
			fclose(fp);
#endif
			exit(1);
		}
		(void) sprintf(buf, "%s\\%s", szBuf, filename);
#ifdef DEBUG
		(void) fprintf(stderr, "INI=%s\n", buf);
#endif
		if ((fp = fopen(buf, "r")) == NULL) {
			(void) sprintf(buf,
				"%s: does not exist in \".\" or windows directory", filename);
			(void) MessageBox(hWnd, buf, "OpenIni", MB_ICONEXCLAMATION);
#ifdef DEBUG
			fclose(fp);
#endif
			exit(1);
		}
	}
#ifdef DEBUG
	(void) fprintf(stderr, "Exiting OpenIni\n");
#endif

	return fp;
}
/*-
   Not using GetProfile[String,Int] because most of the data is on
   multiple lines, repeated, and dynamic.  Data must be in proper order
   i.e. information left of "=" is not read.
 */
static void
ReadThreeD(ThreeDWidget w)
{
	FILE *fp;
	char c;
	int i, j, k, l;
	double ox, oy, oz;	/* DUMMY variables */
	struct tagColor {
		int red, green, blue;
	} color;

	fp = OpenIni(w->core.hWnd, DATAFILE);
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &(w->threed.surface));
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &(w->threed.object));
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &(w->threed.distance));
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &(w->threed.thetaDegrees));
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &(w->threed.phiDegrees));
	for (i = 0; i <= COLORS; i++) {
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d %d %d",
			&(color.red), &(color.green), &(color.blue));
		w->threed.shadeGC[i] = RGB(color.red, color.green, color.blue);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%lg %lg %lg", &ox, &oy, &oz);
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &(w->threed.numObjects));
	w->threed.objects = (Object3D *) malloc(w->threed.numObjects *
		sizeof (Object3D));
#ifdef DEBUG
	(void) fprintf(stderr, "Surface%c%d\n", SYMBOL, w->threed.surface);
	(void) fprintf(stderr, "Object%c%d\n", SYMBOL, w->threed.object);
	(void) fprintf(stderr, "Distance%c%d\n", SYMBOL, w->threed.distance);
	(void) fprintf(stderr, "Theta%c%d\n", SYMBOL, w->threed.thetaDegrees);
	(void) fprintf(stderr, "Phi%c%d\n", SYMBOL, w->threed.phiDegrees);
	for (i = 0; i <= COLORS; i++)
		(void) fprintf(stderr, "%c%d %d %d\n", SYMBOL,
			GetRValue(w->threed.shadeGC[i]),
			GetGValue(w->threed.shadeGC[i]),
			GetBValue(w->threed.shadeGC[i]));
	(void) fprintf(stderr, "Origin%c%g %g %g\n", SYMBOL, ox, oy, oz);
	(void) fprintf(stderr, "OBJECTS%c%d\n", SYMBOL, w->threed.numObjects);
#endif
	for (i = 0; i < w->threed.numObjects; i++) {
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%s", w->threed.objects[i].name);
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d", &(w->threed.objects[i].numVertices));
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d", &(w->threed.objects[i].numSurfaces));
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d",
			&(w->threed.objects[i].numSurfaceVertices));
		w->threed.objects[i].world =
			(Point3D *) malloc(w->threed.objects[i].numVertices *
			sizeof (Point3D));
		w->threed.objects[i].vertex =
			(Vertex *) malloc(w->threed.objects[i].numVertices *
			sizeof (Vertex));
		w->threed.objects[i].info =
			(Surface *) malloc(w->threed.objects[i].numSurfaces *
			sizeof (Surface));
		w->threed.objects[i].map =
			(int *) malloc(w->threed.objects[i].numSurfaceVertices
			* sizeof (int));

#ifdef DEBUG
		(void) fprintf(stderr, "\nNAME%c%s\n", SYMBOL,
			w->threed.objects[i].name);
		(void) fprintf(stderr, "Vertices%c%d\n", SYMBOL,
			w->threed.objects[i].numVertices);
		(void) fprintf(stderr, "Surfaces%c%d\n", SYMBOL,
			w->threed.objects[i].numSurfaces);
		(void) fprintf(stderr, "SurfacesVertices%c%d\n", SYMBOL,
			w->threed.objects[i].numSurfaceVertices);
#endif

		while ((c = getc(fp)) != EOF && c != SYMBOL);
		for (j = 0; j < w->threed.objects[i].numVertices; j++)
			(void) fscanf(fp, "%lg %lg %lg",
				&(w->threed.objects[i].world[j].x),
				&(w->threed.objects[i].world[j].y),
				&(w->threed.objects[i].world[j].z));
#ifdef DEBUG
		(void) fprintf(stderr, "\nVERTICES%c\n", SYMBOL);
		for (j = 0; j < w->threed.objects[i].numVertices; j++)
			(void) fprintf(stderr, "%g\t%g\t%g\n",
				w->threed.objects[i].world[j].x,
				w->threed.objects[i].world[j].y,
				w->threed.objects[i].world[j].z);
#endif
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		l = 0;
		for (j = 0; j < w->threed.objects[i].numSurfaces; j++) {
			(void) fscanf(fp, "%d",
				&(w->threed.objects[i].info[j].numVertices));
			for (k = 0; k < w->threed.objects[i].info[j].numVertices; k++) {
				(void) fscanf(fp, "%d",
					&(w->threed.objects[i].map[l]));
				w->threed.objects[i].map[l++]--;
			}
		}
#ifdef DEBUG
		(void) fprintf(stderr, "\nSURFACESVERTICES%c\n", SYMBOL);
		l = 0;
		for (j = 0; j < w->threed.objects[i].numSurfaces; j++) {
			(void) fprintf(stderr, "%d\t", w->threed.objects[i].info[j].numVertices);
			for (k = 0; k < w->threed.objects[i].info[j].numVertices; k++)
				(void) fprintf(stderr, "%d\t",
					w->threed.objects[i].map[l++] + 1);
			(void) fprintf(stderr, "\n");
		}
#endif
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		for (j = 0; j < w->threed.objects[i].numSurfaces; j++)
			(void) fscanf(fp, "%d %d %d",
				&(w->threed.objects[i].info[j].mapIndex),
				&(w->threed.objects[i].info[j].depthIndex),
				&(w->threed.objects[i].info[j].brushColor));
#ifdef DEBUG
		(void) fprintf(stderr, "\nSURFACES%c\n", SYMBOL);
		for (j = 0; j < w->threed.objects[i].numSurfaces; j++)
			(void) fprintf(stderr, "%d\t%d\t%d\n",
				w->threed.objects[i].info[j].mapIndex,
				w->threed.objects[i].info[j].depthIndex,
				w->threed.objects[i].info[j].brushColor);
#endif
	}
	(void) fclose(fp);
}

void
DestroyThreeD(HBRUSH brush)
{
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

#else

static void
ReadThreeD(ThreeDWidget w)
{
	FILE *fp;
	char c;
	int i, j, k, l;
	double ox, oy, oz;		/* DUMMY variables */

	if ((fp = fopen(DATAFILE, "r")) == NULL) {
		char *buf1;

		stringCat(&buf1, "Can not read ", DATAFILE);
		XtWarning(buf1);
		free(buf1);
	} else {
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d", &(w->threed.numObjects));
		if (!(w->threed.objects = (Object3D *)
		    malloc(w->threed.numObjects * sizeof (Object3D)))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		while ((c = getc (fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d %d %d", &(w->threed.distance),
		     &(w->threed.thetaDegrees), &(w->threed.phiDegrees));
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%lg %lg %lg", &ox, &oy, &oz);
#ifdef DEBUG
		(void) printf("OBJECTS%c %d\n", SYMBOL, w->threed.numObjects);
		(void) printf("Distance Theta Edges%c\n %d %d %d\n", SYMBOL,
		     w->threed.distance, w->threed.thetaDegrees,
		     w->threed.phiDegrees);
		(void) printf("Origin%c  %g %g %g\n", SYMBOL, ox, oy, oz);
#endif
		for (i = 0; i < w->threed.numObjects; i++) {
		  while ((c = getc(fp)) != EOF && c != SYMBOL);
		  (void) fscanf(fp, "\t%s", w->threed.objects[i].name);
		  while ((c = getc(fp)) != EOF && c != SYMBOL);
		  (void) fscanf(fp, "%d %d %d",
			&(w->threed.objects[i].numVertices),
			&(w->threed.objects[i].numSurfaces),
			&(w->threed.objects[i].numSurfaceVertices));
		  if (!(w->threed.objects[i].world = (Point3D *)
		    malloc(w->threed.objects[i].numVertices *
		    sizeof (Point3D)))) {
		      DISPLAY_ERROR("Not enough memory, exiting.");
		  }
		  if (!(w->threed.objects[i].vertex = (Vertex *)
		    malloc(w->threed.objects[i].numVertices *
		    sizeof (Vertex)))) {
		      DISPLAY_ERROR("Not enough memory, exiting.");
		  }
		  if (!(w->threed.objects[i].info = (Surface *)
		    malloc(w->threed.objects[i].numSurfaces *
		    sizeof (Surface)))) {
		      DISPLAY_ERROR("Not enough memory, exiting.");
		  }
		  if (!(w->threed.objects[i].map = (int *)
		    malloc(w->threed.objects[i].numSurfaceVertices *
		    sizeof (int)))) {
		      DISPLAY_ERROR("Not enough memory, exiting.");
		  }
#ifdef DEBUG
		  (void) printf("\nNAME%c %s\n", SYMBOL,
		    w->threed.objects[i].name);
		  (void) printf("Vertices Surfaces SurfaceVertices%c\n%d\t%d\t%d\n",
		    SYMBOL, w->threed.objects[i].numVertices,
		    w->threed.objects[i].numSurfaces,
		    w->threed.objects[i].numSurfaceVertices);
#endif

		  while ((c = getc(fp)) != EOF && c != SYMBOL);
		  for (j = 0; j < w->threed.objects[i].numVertices; j++)
		    (void) fscanf(fp, "%lg %lg %lg",
			   &(w->threed.objects[i].world[j].x),
			   &(w->threed.objects[i].world[j].y),
			   &(w->threed.objects[i].world[j].z));
#ifdef DEBUG
		  (void) printf("\nVERTICES%c\n", SYMBOL);
		  for (j = 0; j < w->threed.objects[i].numVertices; j++)
		    (void) printf("%g\t%g\t%g\n",
			   w->threed.objects[i].world[j].x,
			   w->threed.objects[i].world[j].y,
			   w->threed.objects[i].world[j].z);
#endif
		  while ((c = getc(fp)) != EOF && c != SYMBOL);
		  l = 0;
		  for (j = 0; j < w->threed.objects[i].numSurfaces; j++) {
		      (void) fscanf(fp, "%d",
			     &(w->threed.objects[i].info[j].numVertices));
		      for (k = 0; k < w->threed.objects[i].info[j].numVertices;
			      k++) {
		  (void) fscanf(fp, "%d", &(w->threed.objects[i].map[l]));
		  w->threed.objects[i].map[l++]--;
		}
		    }
#ifdef DEBUG
		  (void) printf("\nSURFACESVERTICES%c\n", SYMBOL);
		  l = 0;
		  for (j = 0; j < w->threed.objects[i].numSurfaces; j++) {
		      (void) printf("%d\t",
			w->threed.objects[i].info[j].numVertices);
		      for (k = 0; k < w->threed.objects[i].info[j].numVertices;
			      k++)
			(void) printf("%d\t",
				w->threed.objects[i].map[l++] + 1);
		      (void) printf("\n");
		    }
		  (void) printf("\n");
#endif
		  while ((c = getc(fp)) != EOF && c != SYMBOL);
		  for (j = 0; j < w->threed.objects[i].numSurfaces; j++)
		    (void) fscanf(fp, "%d %d %d",
			   &(w->threed.objects[i].info[j].mapIndex),
			   &(w->threed.objects[i].info[j].depthIndex),
			   &(w->threed.objects[i].info[j].brushColor));
#ifdef DEBUG
		  (void) printf("\nSURFACES%c\n", SYMBOL);
		  for (j = 0; j < w->threed.objects[i].numSurfaces; j++)
		    (void) printf("%d\t%d\t%d\n",
			   w->threed.objects[i].info[j].mapIndex,
			   w->threed.objects[i].info[j].depthIndex,
			   w->threed.objects[i].info[j].brushColor);
#endif
		}
	}
}

static Boolean
SetValuesThreeD(Widget current, Widget request, Widget renew)
{
	ThreeDWidget c = (ThreeDWidget) current, w = (ThreeDWidget) renew;
	XGCValues values;
	XtGCMask valueMask;
	Boolean redraw = FALSE;

	CheckGraphicsThreeD(w);
	if (w->threed.foreground != c->threed.foreground) {
		valueMask = GCForeground | GCBackground;
		values.foreground = w->threed.foreground;
		values.background = w->core.background_pixel;
		XtReleaseGC(renew, w->threed.graphicsGC);
		w->threed.graphicsGC = XtGetGC(renew, valueMask, &values);
		redraw = TRUE;
	}
	if (w->threed.distance != c->threed.distance ||
	    w->threed.thetaDegrees != c->threed.thetaDegrees ||
	    w->threed.phiDegrees != c->threed.phiDegrees ||
	    w->threed.surface != c->threed.surface ||
	    w->threed.object != c->threed.object) {
		w->threed.name = w->threed.objects[w->threed.object].name;
#ifdef DEBUG
		(void) printf("%s\n", w->threed.name);
#endif
		ResizeThreeD(w);
		redraw = TRUE;
	}
	return (redraw);
}

static void
SetAllColors(ThreeDWidget w)
{
	XGCValues values;
	XtGCMask valueMask;
	int color;

	valueMask = GCForeground | GCBackground;
	if (w->threed.reverse) {
		values.foreground = w->core.background_pixel;
		values.background = w->threed.foreground;
	} else {
		values.foreground = w->threed.foreground;
		values.background = w->core.background_pixel;
	}
	if (w->threed.graphicsGC)
		XtReleaseGC((Widget) w, w->threed.graphicsGC);
	w->threed.graphicsGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->threed.reverse) {
		values.foreground = w->threed.foreground;
		values.background = w->core.background_pixel;
	} else {
		values.foreground = w->core.background_pixel;
		values.background = w->threed.foreground;
	}
	if (w->threed.inverseGC)
		XtReleaseGC((Widget) w, w->threed.inverseGC);
	w->threed.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	for (color = 0; color < COLORS; color++) {
		if (!w->threed.mono) {
			values.foreground = w->threed.shadeColor[color];
		}
		if (w->threed.shadeGC[color])
			XtReleaseGC((Widget) w, w->threed.shadeGC[color]);
		w->threed.shadeGC[color] = XtGetGC((Widget) w, valueMask,
			&values);
	}
}

static void
QuitThreeD(ThreeDWidget w, XEvent * event, char **args, int n_args)
{
	XtCloseDisplay(XtDisplay(w));
	exit(0);
}

static void
DestroyThreeD(Widget old)
{
	ThreeDWidget w = (ThreeDWidget) old;
	int color;

	for (color = 0; color < COLORS; color++)
		XtReleaseGC(old, w->threed.shadeGC[color]);
	XtReleaseGC(old, w->threed.graphicsGC);
	XtReleaseGC(old, w->threed.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->threed.select);
}
#endif

static void
ResizeThreeD(ThreeDWidget w)
{
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	w->threed.center.x = w->core.width / 2;
	w->threed.center.y = w->core.height / 2;
	w->threed.size = MIN(w->threed.center.x, w->threed.center.y);
}

#ifndef WINVER
static
#endif
void
InitializeThreeD(
#ifdef WINVER
ThreeDWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
	int i;

#ifdef WINVER
	w->threed.mono = DEFAULTMONO;
	w->threed.reverse = DEFAULTREVERSE;
#else
	ThreeDWidget w = (ThreeDWidget) renew;
	int color;

	w->threed.mono = (DefaultDepthOfScreen(XtScreen(w)) < 2 ||
		w->threed.mono);
	for (color = 0; color < COLORS; color++)
		w->threed.shadeGC[color] = NULL;
	w->threed.graphicsGC = NULL;
	w->threed.inverseGC = NULL;
#endif
	ReadThreeD(w);
	CheckGraphicsThreeD(w);
	w->threed.object = 0;
	w->threed.name = w->threed.objects[w->threed.object].name;
	if (!(w->threed.list = (char **) calloc((unsigned int)
			w->threed.numObjects, sizeof (char **)))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	for (i = 0; i < w->threed.numObjects; i++)
		w->threed.list[i] = w->threed.objects[i].name;
	ResizeThreeD(w);
#ifdef DEBUG
		(void) printf("%s\n", w->threed.name);
#endif
#ifdef WINVER
	brush = CreateSolidBrush(w->threed.inverseGC);
	SETBACK(w->core.hWnd, brush);
#else
	SetAllColors(w);
#endif
}

#ifndef WINVER
static
#endif
void
ExposeThreeD(
#ifdef WINVER
ThreeDWidget w
#else
Widget renew, XEvent * event, Region region
#endif
)
{
#ifndef WINVER
	ThreeDWidget w = (ThreeDWidget) renew;

	if (!w->core.visible)
		return;
#endif
	RepaintGraphicsThreeD(w);
}

#ifndef WINVER
static
#endif
void
HideThreeD(ThreeDWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SetThreeD(w, THREED_HIDE);
}

#ifdef WINVER
void
SetSurfaceThreeD(ThreeDWidget w , Boolean surface)
{
	w->threed.surface = surface;
}

void
SetObjectThreeD(ThreeDWidget w, int object3D)
{
	w->threed.object = object3D;
}

#else

static
void
SurfaceThreeD(ThreeDWidget w
, XEvent * event, char **args, int n_args
)
{
	SetThreeD(w, THREED_SURFACE);
}

static
void
ObjectThreeD(ThreeDWidget w
, XEvent * event, char **args, int n_args
)
{
	SetThreeD(w, THREED_OBJECT);
}

static void
MoveOutThreeD(ThreeDWidget w, XEvent * event, char **args, int n_args)
{
	if (w->threed.distance < MAXDISTANCE)
		w->threed.distance += DELTADISTANCE;
	SetThreeD(w, THREED_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveLowerPhiThreeD(ThreeDWidget w, XEvent * event, char **args, int n_args)
{
	w->threed.phiDegrees -= DELTADISTANCE;
	if (w->threed.phiDegrees < MINDEGREES)
		w->threed.phiDegrees += NUMDEGREES;
	SetThreeD(w, THREED_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveLowerThetaThreeD(ThreeDWidget w, XEvent * event, char **args, int n_args)
{
	w->threed.thetaDegrees -= DELTADISTANCE;
	if (w->threed.thetaDegrees < MINDEGREES)
		w->threed.thetaDegrees += NUMDEGREES;
	SetThreeD(w, THREED_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveRaiseThetaThreeD(ThreeDWidget w, XEvent * event, char **args, int n_args)
{
	w->threed.thetaDegrees += DELTADEGREES;
	if (w->threed.thetaDegrees > MAXDEGREES)
		w->threed.thetaDegrees -= NUMDEGREES;
	SetThreeD(w, THREED_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveRaisePhiThreeD(ThreeDWidget w, XEvent * event, char **args, int n_args)
{
	w->threed.phiDegrees += DELTADEGREES;
	if (w->threed.phiDegrees > MAXDEGREES)
		w->threed.phiDegrees -= NUMDEGREES;
	SetThreeD(w, THREED_SET);
	RepaintGraphicsThreeD(w);
}

static void
MoveInThreeD(ThreeDWidget w, XEvent * event, char **args, int n_args)
{
	if (w->threed.distance > MINDISTANCE)
		w->threed.distance -= DELTADEGREES;
	SetThreeD(w, THREED_SET);
	RepaintGraphicsThreeD(w);
}
#endif
