/*-
# MOTIF/X-BASED SKEWB
#
#  xskewb.c
#
###
#
#  Copyright (c) 1993 - 2004	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/12/15 Xt/Motif
  Version 5: 95/10/06 Xt/Motif
  Version 4: 94/05/30 Xt
  Version 3: 93/10/03 Motif
  Version 2: 92/01/22 XView
  Version 1: 91/03/19 SunView
*/

#ifdef WINVER
#include "SkewbP.h"
#include "wskewb.h"
#define TITLE "wskewb"

static SkewbRec widget;
static HWND Wnd;

#ifndef SCOREPATH
#define SCOREPATH "c:\\Windows"
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(Wnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->skewb.started = b
#else
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#ifdef VMS
#include <unixlib.h>
#define getlogin() cuserid(NULL)
#else
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif
#if HAVE_FCNTL_H
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/ToggleB.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtSetArg(arg[0], XtNstart, b); XtSetValues(w, arg, 1)
#include "Skewb.h"
#include "Skewb2d.h"
#include "Skewb3d.h"
#ifdef HAVE_OPENGL
#include "SkewbGL.h"
#endif
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/skewb.xpm"
#endif
#include "icons/skewb.xbm"
#ifndef SCOREPATH
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif

#ifdef HAVE_MOTIF
static const char aboutHelp[] = {
	"Version 7.1\n"
	"Send bugs (reports or fixes) to the author: "
	"David Bagley <bagleyd@tux.org>\n"
	"The latest version is at: "
	"http://www.tux.org/~bagleyd/puzzles.html\n"
};
static const char optionsHelp[] = {
	"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]] "
	"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
	"[-{foreground|fg} {color}] [-{background|bg} {color}] "
	"[-{border|bd} {color}] [-face{0|1|2|3|4|5} {color}] \n"
	"[-[no]orient] [-[no]practice] [-{font|fn} {fontname}] "
	"[-username {string}]\n"
};
#endif
#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
	"The original puzzle has each face cut by a diamond, so "
	"that there are 5 pieces,  4 corner pieces and\n"
	"one diamond piece in the center.  This was designed by "
	"Uwe Meffert and called the Pyraminx Cube.  Douglas\n"
	"Hofstadter later coined it a Skewb and it stuck.  The "
	"puzzle has period 3 turning (i.e. each half turns\n"
	"with 120 degree intervals).  The Skewb has  2^5*3^8*6!/2^6 "
	"or 3,149,280  different combinations (with\n"
	"centers oriented 2^5*3^8*6!/2 or 100,766,960 different "
	"combinations).\n"
	"More recently, Disney released Mickey's Challenge, its "
	"a spherical skewb with a pretty good internal\n"
	"mechanism.  Mickey's challenge has 2^5*3^8*6!/36 "
	"5,598,720 visually different combinations).  It also\n"
	"comes with a pretty neat book.  Also released "
	"is the Creative Puzzle Ball or Meffert's Challenge which\n"
	"has 4 rings in different colors.\n"
	"Mach Balls of the Hungarian Gyula Mach are similar "
	"but they do not have a ratchet mechanism and do not\n"
	"turn as easily or smoothly.  One must match the 12 "
	"different symbols of 4 each  at the 12 intersection\n"
	"points.\n"
};
static const char featuresHelp[] = {
	"Press  \"mouse-left\"  button  to move a piece.  Release "
	"\"mouse-left\"  button on a piece  on the same face.\n"
	"(Clicks on diamonds are ignored).  The pieces will then "
	"turn towards where the mouse button was released.\n"
	"Click \"mouse-center\" button, or press \"P\" or \"p\" "
	"keys, to toggle  the practice mode (in practice mode the\n"
	"record should say \"practice\").  This is good for learning "
	"moves and experimenting.\n"
	"Click \"mouse-right\" button,  or press \"R\" or \"r\" "
	"keys,  to randomize the puzzle  (this must be done first\n"
	"to set a new record).\n"
	"Press \"O\" or \"o\" keys to toggle  the orient mode.  One "
	"has  to orient the faces  in orient mode,  besides\n"
	"getting  all the faces to be the  same color.  To do this "
	"one has to get the lines to be  oriented in the\n"
	"same direction,  this only matters with  center diamond "
	"piece.  This does add complexity  so there are 2\n"
	"sets of records.\n"
	"\"S\" or \"s\" keys reserved for the auto-solver "
	"(not implemented).\n"
	"Press \"U\" or \"u\" keys to undo a move.\n"
	"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
	"Press \"W\" or \"w\" keys to save a puzzle.\n"
	"Press \"Esc\" key to hide program.\n"
	"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
	"Use the key pad, \"R\" keys, or arrow keys to move without "
	"mouse clicks.\n"
	"Key pad is defined for Skewb2d as:\n"
	"  /    Counterclockwise\n"
	"7 8 9  Upper Left, Up, Upper Right\n"
	"  ^\n"
	"4<5>6  Left, Clockwise, Right\n"
	"  v\n"
	"1 2 3  Lower Left, Down, Lower Right\n"
	"Note: Top, Left, Right, and Down only work when the "
	"control key is pressed and there is no analog for\n"
	"Skewb3d.\n"
	"If the mouse is on a diamond, the above keys will not "
	"move cube because the move is ambiguous.  Also\n"
	"if the mouse is on a triangle, not all the keys will "
	"function because the puzzle will only rotate on the\n"
	"cuts, i.e. a triangle with a Upper Left - Lower Right "
	"cut will rotate only Upper Left & Lower Right, a\n"
	"triangle with a Upper Right - Lower Left cut will rotate "
	"only Upper Right & Lower Left.  Therefore, a\n"
	"triangle can only move tangential to the center of "
	"the face. No doubt this is confusing, but the physical\n"
	"skewb is the same way. In fact, that is part of its appeal.\n"
	"Key pad for Skewb3d, use must use your intuition (is this "
	"a cop out or what?).  The key pad is defined\n"
	"differently depending on which side of the cube your mouse "
	"is pointing at.  One thing that stays the\n"
	"same is \"5\" is Clockwise and \"/\" is Counterclockwise.\n"
	"Use the control key and the left mouse button, keypad, or "
	"arrow keys to move the whole cube.  This is\n"
	"not recorded as a turn.\n"
};
static const char referencesHelp[] = {
	"Beyond Rubik's Cube: spheres, pyramids, dodecahedrons and "
	"God knows what else by Douglas\n"
	"R. Hofstadter, Scientific American, July 1982, pp 16-31.\n"
	"Mickey's Challenge by Christoph Bandelow.\n"
	"Magic Cubes 1996 Catalog of Dr. Christoph Bandelow.\n"
};
#endif
static const char solveHelp[] = {
	"Auto-solver: sorry, not implemented.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "skewb.scores"
#endif

#define NEVER -1
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} GameRecord;

static GameRecord skewbRecord[2];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";
#ifndef HAVE_MOTIF
static char titleDsp[TITLELEN] = "";
#endif
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char usernameDsp[USERNAMELEN] = "Guest";
#else
#ifdef HAVE_MOTIF
static Widget moves, record, message, orientSwitch, practiceSwitch;
static char buff[21];
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, practiceDialog, randomizeDialog;
#else
Widget shell;
#ifdef HAVE_OPENGL
Widget shellGL;
#endif
#endif
static Pixmap skewbIcon = None;
static Widget topLevel, skewb2d, skewb3d;
#ifdef HAVE_OPENGL
static Widget skewbGL;
#endif
static Arg arg[4];
static char *progDsp;
static char usernameDsp[USERNAMELEN] = "";

/* There's probably a better way to assure that they are the same but I don't
   know it off hand. */
static void
MakeEquivalent(String * username, Boolean * orient, Boolean * practice)
{
	Boolean mono, reverse;
	Pixel foreground, background, pieceBorder;
	String faceColor[MAXFACES];

	XtVaGetValues(skewb2d,
		XtNuserName, username,
		XtNorient, orient,
		XtNpractice, practice,
		XtNmono, &mono,
		XtNreverse, &reverse,
		XtNforeground, &foreground,
		XtNbackground, &background,
		XtNpieceBorder, &pieceBorder,
		XtNfaceColor0, &(faceColor[0]),
		XtNfaceColor1, &(faceColor[1]),
		XtNfaceColor2, &(faceColor[2]),
		XtNfaceColor3, &(faceColor[3]),
		XtNfaceColor4, &(faceColor[4]),
		XtNfaceColor5, &(faceColor[5]), NULL);
	XtVaSetValues(skewb2d,
		XtNdirection, SKEWB_IGNORE,
		XtNstart, False, NULL);
	XtVaSetValues(skewb3d,
		XtNuserName, *username,
		XtNorient, *orient,
		XtNmono, mono,
		XtNreverse, reverse,
		XtNdirection, SKEWB_IGNORE,
		XtNpractice, *practice,
		XtNstart, False,
		XtNforeground, foreground,
		XtNbackground, background,
		XtNpieceBorder, pieceBorder,
		XtNfaceColor0, faceColor[0],
		XtNfaceColor1, faceColor[1],
		XtNfaceColor2, faceColor[2],
		XtNfaceColor3, faceColor[3],
		XtNfaceColor4, faceColor[4],
		XtNfaceColor5, faceColor[5], NULL);
#ifdef HAVE_OPENGL
	XtVaSetValues(skewbGL,
		XtNuserName, *username,
		XtNorient, *orient,
		XtNmono, mono,
		XtNreverse, reverse,
		XtNdirection, SKEWB_IGNORE,
		XtNpractice, *practice,
		XtNstart, False,
		XtNforeground, foreground,
		XtNbackground, background,
		XtNpieceBorder, pieceBorder,
		XtNfaceColor0, faceColor[0],
		XtNfaceColor1, faceColor[1],
		XtNfaceColor2, faceColor[2],
		XtNfaceColor3, faceColor[3],
		XtNfaceColor4, faceColor[4],
		XtNfaceColor5, faceColor[5], NULL);
#endif
}
#endif

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString    xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(arg[0], XmNlabelString, xmstr);
	XtSetValues(w, arg, 1);
}
#else
static void
PrintState(
#ifndef WINVER
Widget w,
#endif
int dim, int moves, char *msg)
{
#ifdef HAVE_OPENGL
	if (dim == 4)
		(void) sprintf(titleDsp, "%sGL: (%d/%s) - %s",
			progDsp, moves, recordDsp, msg);
	else
#endif
		(void) sprintf(titleDsp, "%s%dd: (%d/%s) - %s",
			progDsp, dim, moves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(Wnd, (LPSTR) titleDsp);
#else
	XtSetArg(arg[0], XtNtitle, titleDsp);
	XtSetValues(w, arg, 1);
#endif
}
#endif

static void
InitRecords(void)
{
	int orient;

	for (orient = 0; orient < 2; orient++) {
		skewbRecord[orient].score = NEVER;
		(void) strncpy(skewbRecord[orient].name, NOACCESS, USERNAMELEN);
	}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int n, orient;
	char username[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "r")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (orient = 0; orient < 2; orient++) {
		(void) fscanf(fp, "%d %s\n", &n, username);
		if (n <= skewbRecord[orient].score ||
		    skewbRecord[orient].score <= NEVER) {
			skewbRecord[orient].score = n;
			(void) strncpy(skewbRecord[orient].name, username,
				USERNAMELEN);
		}
	}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int orient;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "w")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockfile[FILENAMELEN];

		(void) strncpy(lockfile, name, FILENAMELEN - 6);
		(void) strcat(lockfile, ".lock");
		while (((lfd = open(lockfile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
			  "Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
			  "Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (orient = 0; orient < 2; orient++)
			(void) fprintf(fp, "%d %s\n",
			skewbRecord[orient].score, skewbRecord[orient].name);
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockfile);
#endif
		(void) fclose(fp);
	}
	free(lname);
	free(fname);
}

static void
PrintRecord(Boolean orient, Boolean practice)
{
	int i = (orient) ? 1 : 0;

	if (practice)
		(void) strncpy(recordDsp, "practice", MESSAGELEN);
	else if (skewbRecord[i].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			skewbRecord[i].score, skewbRecord[i].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static Boolean
HandleSolved(int counter, Boolean orient)
{
	int i = (orient) ? 1 : 0;

	if (counter < skewbRecord[i].score || skewbRecord[i].score <= NEVER) {
		ReadRecords();	/* Maybe its been updated by another */
		skewbRecord[i].score = counter;
		(void) strncpy(skewbRecord[i].name, usernameDsp, USERNAMELEN);
		if (orient && (counter < skewbRecord[!i].score ||
				skewbRecord[!i].score <= NEVER)) {
			skewbRecord[!i].score = counter;
			(void) strncpy(skewbRecord[!i].name, usernameDsp,
				USERNAMELEN);
		}
		WriteRecords();
		PrintRecord(orient, False);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
SkewbWidget w, HBRUSH brush
#else
void
#endif
)
{
	Boolean orient, practice;
	char *username;
#ifdef WINVER
	int dim;

	InitializeSkewb(w, brush);

	orient = w->skewb.orient;
	practice = w->skewb.practice;
	username = w->skewb.username;
	dim = w->skewb.dim;
	SET_STARTED(w, False);
#else
	MakeEquivalent(&username, &orient, &practice);
#ifdef HAVE_MOTIF
	XmToggleButtonSetState(orientSwitch, orient, True);
	XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
#endif
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(usernameDsp, username, USERNAMELEN);
#endif
	if (!strcmp(username, "") || !strcmp(username, "(null)") ||
	    !strcmp(username, NOACCESS) || !strcmp(username, NOBODY)) {
#ifdef WINVER
		(void) strncpy(usernameDsp, username, USERNAMELEN);
#else
		/* The NOACCESS is not necessary, but it may stop silliness. */
		(void) sprintf(usernameDsp, "%s", getlogin());
		if (!strcmp(usernameDsp, "") ||
				!strcmp(usernameDsp, "(null)") ||
				!strcmp(usernameDsp, NOACCESS) ||
				!strcmp(usernameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(usernameDsp, "%s", NOBODY);
#endif
	}
	PrintRecord(orient, practice);
#ifdef WINVER
	PrintState(dim, movesDsp, messageDsp);
#else
#ifndef HAVE_MOTIF
	PrintState(XtParent(skewb2d), 2, movesDsp, messageDsp);
	PrintState(XtParent(skewb3d), 3, movesDsp, messageDsp);
#ifdef HAVE_OPENGL
	PrintState(XtParent(skewbGL), 4, movesDsp, messageDsp);
#endif
#endif
#endif
}

#ifdef WINVER
void
SetSkewb(SkewbWidget w, int reason)
#else
static void
CallbackSkewb(Widget w, caddr_t clientData, skewbCallbackStruct * callData)
#endif
{
	Boolean orient, practice, start, cheat;
#ifdef WINVER
	int dim = 0;
#else
	int reason = callData->reason;
	Widget otherw1 = (Widget) NULL;
#ifdef HAVE_OPENGL
	Widget otherw2 = (Widget) NULL;
#endif
#ifndef HAVE_MOTIF
	int dim = 0, otherdim1 = 0;
#ifdef HAVE_OPENGL
	int otherdim2 = 0;
#endif
#endif

	if (w == skewb2d) {
		otherw1 = skewb3d;
#ifdef HAVE_OPENGL
		otherw2 = skewbGL;
#endif
#ifndef HAVE_MOTIF
		dim = 2;
		otherdim1 = 3;
#ifdef HAVE_OPENGL
		otherdim2 = 4;
#endif
#endif
	} else if (w == skewb3d) {
		otherw1 = skewb2d;
#ifdef HAVE_OPENGL
		otherw2 = skewbGL;
#endif
#ifndef HAVE_MOTIF
		dim = 3;
		otherdim1 = 2;
#ifdef HAVE_OPENGL
		otherdim2 = 4;
#endif
#endif
	}
#ifdef HAVE_OPENGL
	else if (w == skewbGL) {
		otherw1 = skewb2d;
		otherw2 = skewb3d;
#ifndef HAVE_MOTIF
		dim = 4;
		otherdim1 = 2;
		otherdim2 = 3;
#endif
	}
#endif
#endif
	(void) strcpy(messageDsp, "");
#ifdef WINVER
	orient = w->skewb.orient;
	practice = w->skewb.practice;
	cheat = w->skewb.cheat;
	start = w->skewb.started;
	dim = w->skewb.dim;
#else
	XtVaGetValues(w,
		XtNorient, &orient,
		XtNpractice, &practice,
		XtNstart, &start,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case SKEWB_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#ifndef HAVE_MOTIF
			(void) XIconifyWindow(XtDisplay(shell),
				XtWindow(shell),
				XScreenNumberOfScreen(XtScreen(shell)));
#ifdef HAVE_OPENGL
			(void) XIconifyWindow(XtDisplay(shellGL),
				XtWindow(shellGL),
				XScreenNumberOfScreen(XtScreen(shellGL)));
#endif
#endif
#endif
			break;
#ifndef WINVER
		case SKEWB_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(practiceDialog);
#else
			XtSetArg(arg[0], XtNmenu, 8); /* menu choice */
			XtSetValues(skewb2d, arg, 1);
#endif
			break;
		case SKEWB_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(randomizeDialog);
#else
			XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
			XtSetValues(skewb2d, arg, 1);
#endif
			break;
#endif
		case SKEWB_SOLVE_MESSAGE:
#ifdef WINVER
			(void) MessageBox(w->core.hWnd, solveHelp,
				"Auto-Solve", MB_OK);
#else
#ifdef HAVE_MOTIF
			XtManageChild(solveDialog);
#else
			(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
			break;
		case SKEWB_RESTORE:
			if (practice) {
				(void) strncpy(recordDsp, "practice", MESSAGELEN);
#ifdef HAVE_MOTIF
				PrintState(record, recordDsp);
#endif
			}
			movesDsp = 0;
#ifndef WINVER
			XtSetArg(arg[0], XtNdirection, SKEWB_RESTORE);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
			XtSetValues(w, arg, 1);
#endif
			break;
		case SKEWB_CLEAR:
			movesDsp = 0;
#ifndef WINVER
			XtSetArg(arg[0], XtNdirection, SKEWB_CLEAR);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
			XtSetValues(w, arg, 1);
#endif
			break;
		case SKEWB_RESET:
			movesDsp = 0;
			break;
		case SKEWB_ILLEGAL:
			if (practice || start)
				(void) strncpy(messageDsp, "Illegal move",
					MESSAGELEN);
			else
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			break;
#ifndef WINVER
		case SKEWB_MOVED:
			movesDsp++;
			XtSetArg(arg[0], XtNstart, True);
			XtSetArg(arg[1], XtNface, callData->face);
			XtSetArg(arg[2], XtNpos, callData->position);
			XtSetArg(arg[3], XtNdirection, callData->direction);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 4);
#endif
			XtSetValues(otherw1, arg, 4);
			XtSetValues(w, arg, 1);
			break;
		case SKEWB_CONTROL:
			XtSetArg(arg[0], XtNface, callData->face);
			XtSetArg(arg[1], XtNpos, callData->position);
			XtSetArg(arg[2], XtNdirection, callData->direction);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 3);
#endif
			XtSetValues(otherw1, arg, 3);
			return;
#endif
		case SKEWB_SOLVED:
			if (practice)
				movesDsp = 0;
			else if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", usernameDsp);
			else if (HandleSolved(movesDsp, orient))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", usernameDsp);
			else
				(void) strncpy(messageDsp, "Solved!",
					MESSAGELEN);
#ifdef WINVER
			SET_STARTED(w, False);
#else
			XtSetArg(arg[0], XtNstart, False);
			XtSetValues(w, arg, 1);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
#endif
			break;
		case SKEWB_PRACTICE:
			movesDsp = 0;
			practice = !practice;
			if (!practice)
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			PrintRecord(orient, practice);
#ifdef WINVER
			w->skewb.practice = practice;
			w->skewb.started = False;
#else

			XtSetArg(arg[0], XtNpractice, practice);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 2);
#endif
			XtSetValues(otherw1, arg, 2);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
#endif
			break;
		case SKEWB_RANDOMIZE:
			movesDsp = 0;
#ifdef WINVER
			w->skewb.practice = practice;
			w->skewb.started = False;
#else
			XtSetArg(arg[0], XtNpractice, False);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 2);
#endif
			XtSetValues(otherw1, arg, 2);
#endif
			break;
		case SKEWB_ORIENT:
			movesDsp = 0;
			orient = !orient;
			PrintRecord(orient, practice);
#ifdef WINVER
			w->skewb.orient = orient;
#else
			XtSetArg(arg[0], XtNorient, orient);
			XtSetValues(w, arg, 1);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(orientSwitch, orient, True);
#endif
#endif
			break;
		case SKEWB_COMPUTED:
#ifdef WINVER
			SET_STARTED(w, False);
#else
			XtSetArg(arg[0], XtNstart, False);
			XtSetValues(w, arg, 1);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 1);
#endif
			XtSetValues(otherw1, arg, 1);
#endif
			break;
#ifndef WINVER
		case SKEWB_UNDO:
			movesDsp--;
			XtSetArg(arg[0], XtNstart, True);
			XtSetArg(arg[1], XtNface, callData->face);
			XtSetArg(arg[2], XtNpos, callData->position);
			XtSetArg(arg[3], XtNdirection, callData->direction);
#ifdef HAVE_OPENGL
			XtSetValues(otherw2, arg, 4);
#endif
			XtSetValues(otherw1, arg, 4);
			XtSetValues(w, arg, 1);
			break;
#endif
#ifdef WINVER
		case SKEWB_DIM:
			dim++;
#ifdef HAVE_OPENGL
			if (dim > 4)
#else
			if (dim > 3)
#endif
				dim = 2;
			w->skewb.dim = dim;
			break;
#endif

	}
#ifdef WINVER
	PrintState(dim, movesDsp, messageDsp);
#else
#ifdef HAVE_MOTIF
	PrintState(message, messageDsp);
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
#else
	PrintState(XtParent(w), dim, movesDsp, messageDsp);
	PrintState(XtParent(otherw1), otherdim1, movesDsp, messageDsp);
#ifdef HAVE_OPENGL
	PrintState(XtParent(otherw2), otherdim2, movesDsp, messageDsp);
#endif
#endif
#endif
}

#ifdef WINVER
void
SetSkewbMove(SkewbWidget w, int reason, int face, int position, int direction)
{
	/*Boolean orient, practice, cheat; */
	int dim;

	(void) strcpy(messageDsp, "");
#if 0
	orient = w->skewb.orient;
	practice = w->skewb.practice;
	cheat = w->skewb.cheat;
#endif
	dim = w->skewb.dim;
	switch (reason) {
		case SKEWB_MOVED:
			movesDsp++;
			SET_STARTED(w, True);
			break;
		case SKEWB_CONTROL:
			break;
		case SKEWB_UNDO:
			movesDsp--;
			SET_STARTED(w, True);
			break;
	}
	PrintState(dim, movesDsp, messageDsp);
}

static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	Wnd = widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.skewb.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			EnterSkewb(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.skewb.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			LeaveSkewb(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroySkewb(brush);
			break;
		case WM_SIZE:
			ResizeSkewb(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ExposeSkewb(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			RandomizeSkewb(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			SelectSkewb(&widget, LOWORD(lParam), HIWORD(lParam),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0, FALSE);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ReleaseSkewb(&widget, LOWORD(lParam), HIWORD(lParam),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0, FALSE);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDM_GET:
					GetSkewb(&widget);
					ResizeSkewb(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WriteSkewb(&widget);
					break;
				case IDM_EXIT:
					DestroySkewb(brush);
					break;
				case IDM_HIDE:
					HideSkewb(&widget);
					break;
				case IDM_CLEAR:
					ClearSkewb(&widget);
					SizeSkewb(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					UndoSkewb(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_RANDOMIZE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					RandomizeSkewb(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_PRACTICE:
					PracticeSkewb(&widget);
					SizeSkewb(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					SolveSkewb(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_ORIENT:
					OrientizeSkewb(&widget);
					SizeSkewb(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DIM:
					(void) DimSkewb(&widget);
					ResizeSkewb(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_TR:
				case IDM_BR:
				case IDM_BL:
				case IDM_TL:
				case IDM_CW:
				case IDM_CCW:
				case IDM_TOP:
				case IDM_RIGHT:
				case IDM_BOTTOM:
				case IDM_LEFT:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MoveSkewbInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - IDM_TR,
							FALSE, FALSE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}

					break;
				case IDM_CONTROL_TR:
				case IDM_CONTROL_BR:
				case IDM_CONTROL_BL:
				case IDM_CONTROL_TL:
				case IDM_CONTROL_CW:
				case IDM_CONTROL_CCW:
				case IDM_CONTROL_TOP:
				case IDM_CONTROL_RIGHT:
				case IDM_CONTROL_BOTTOM:
				case IDM_CONTROL_LEFT:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MoveSkewbInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - IDM_CONTROL_TR,
							TRUE, FALSE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}

					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
					 	"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
					 	"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
					 	"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else
static void
Usage(char * programName)
{
	(void) fprintf(stderr, "usage: %s\n", programName);
	(void) fprintf(stderr,
		"\t[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n");
	(void) fprintf(stderr,
		"\t[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n");
	(void) fprintf(stderr,
		"\t[-{foreground|fg} {color}] [-{background|bg} {color}]\n");
	(void) fprintf(stderr,
		"\t[-{border|bd} {color}] [-face{0|1|2|3|4|5} {color}]\n");
	(void) fprintf(stderr,
		"\t[-[no]orient] [-[no]practice] [-{font|fn} {fontname}]\n");
	(void) fprintf(stderr,
		"\t[-username {string}]\n");
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*skewb.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*skewb.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*skewb.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*skewb.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*skewb.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*skewb.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*skewb.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*skewb.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*skewb.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*skewb.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-face0", (char *) "*skewb.faceColor0", XrmoptionSepArg, NULL},
	{(char *) "-face1", (char *) "*skewb.faceColor1", XrmoptionSepArg, NULL},
	{(char *) "-face2", (char *) "*skewb.faceColor2", XrmoptionSepArg, NULL},
	{(char *) "-face3", (char *) "*skewb.faceColor3", XrmoptionSepArg, NULL},
	{(char *) "-face4", (char *) "*skewb.faceColor4", XrmoptionSepArg, NULL},
	{(char *) "-face5", (char *) "*skewb.faceColor5", XrmoptionSepArg, NULL},
	{(char *) "-orient", (char *) "*skewb.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*skewb.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-practice", (char *) "*skewb.practice", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopractice", (char *) "*skewb.practice", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fn", (char *) "*skewb.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*skewb.font", XrmoptionSepArg, NULL},
	{(char *) "-username", (char *) "*skewb.userName", XrmoptionSepArg, NULL}
};

#ifdef HAVE_MOTIF
static void
CallbackSkewbPractice(Widget w, XtPointer clientData, XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 8); /* menu choice */
		XtSetValues(skewb2d, arg, 1);
	}
}

static void
CallbackSkewbRandomize(Widget w, XtPointer clientData, XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
		XtSetValues(skewb2d, arg, 1);
	}
}

static void
OrientToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	Boolean orient = cbs->set, practice;

	XtVaGetValues(skewb2d,
		XtNpractice, &practice, NULL);
	XtVaSetValues(skewb2d,
		XtNorient, orient, NULL);
	XtVaSetValues(skewb3d,
		XtNorient, orient, NULL);
#ifdef HAVE_OPENGL
	XtVaSetValues(skewbGL,
		XtNorient, orient, NULL);
#endif
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(orient, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
PracticeToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	Boolean orient, practice = cbs->set;

	XtVaSetValues(skewb2d,
		XtNpractice, practice,
		XtNstart, False, NULL);
	XtVaSetValues(skewb3d,
		XtNpractice, practice,
		XtNstart, False, NULL);
#ifdef HAVE_OPENGL
	XtVaSetValues(skewbGL,
		XtNpractice, practice,
		XtNstart, False, NULL);
#endif
	XtVaGetValues(skewb2d,
		XtNpractice, &orient, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(orient, practice);
	if (practice)
		(void) strcpy(messageDsp, "");
	else
		(void) strncpy(messageDsp, "Randomize to start", MESSAGELEN);
	PrintState(message, messageDsp);
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == 2)
		exit(0);
	XtSetArg(arg[0], XtNmenu, val);
	XtSetValues(skewb2d, arg, 1);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;
	XtSetArg(arg[0], XtNmenu, val + 3); /* GWQ */
	XtSetValues(skewb2d, arg, 1);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d\n", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, switchRowCol, messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString clearString, undoString, randomizeString, solveString;
	XmString orientString, practiceString;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Skewb",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#if HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) skewb_xpm, &skewbIcon, NULL,
			&xpmAttributes);
	}
	if (skewbIcon == (Pixmap) NULL)
#endif
		skewbIcon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) skewb_bits,
			skewb_width, skewb_height);
	XtSetArg(arg[0], XtNiconPixmap, skewbIcon);
	/* not XmEXPLICIT */
#ifdef HAVE_MOTIF
	XtSetArg(arg[1], XmNkeyboardFocusPolicy, XmPOINTER);
	XtSetValues(topLevel, arg, 2);
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	clearString = XmStringCreateSimple((char *) "Clear");
	undoString = XmStringCreateSimple((char *) "Undo");
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "(Solve)");
	orientString = XmStringCreateSimple((char *) "Orientize");
	practiceString = XmStringCreateSimple((char *) "Practice");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, orientString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
		NULL);
	XmStringFree(clearString);
	XmStringFree(undoString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(orientString);
	XmStringFree(practiceString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	practiceDialog = createQuery(topLevel,
		(char *) "Are you sure you want to toggle the practice mode?",
		(char *) "Practice Query",
		(XtCallbackProc) CallbackSkewbPractice);
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) CallbackSkewbRandomize);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("Rowcol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move", 5, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

	switchRowCol = XtVaCreateManagedWidget("switchRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	orientSwitch = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, switchRowCol, NULL);
	XtAddCallback(orientSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) OrientToggle, (XtPointer) NULL);
	practiceSwitch = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, switchRowCol, NULL);
	XtAddCallback(practiceSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) PracticeToggle, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Skewb! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);

	skewb2d = XtCreateManagedWidget("skewb",
		skewb2dWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(skewb2d,
		XtNheight, 200, NULL);
	XtAddCallback(skewb2d, XtNselectCallback,
		(XtCallbackProc) CallbackSkewb, (XtPointer) NULL);
	skewb3d = XtCreateManagedWidget("skewb",
		skewb3dWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(skewb3d,
		XtNheight, 200, NULL);
#ifdef HAVE_OPENGL
	skewbGL = XtCreateManagedWidget("skewb",
		skewbGLWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(skewbGL,
		XtNheight, 200, NULL);
#endif
#else
	XtSetArg(arg[1], XtNinput, True);
	XtSetValues(topLevel, arg, 2);
	shell = XtCreateApplicationShell(argv[0],
		topLevelShellWidgetClass, NULL, 0);
	XtSetValues(shell, arg, 2);
#ifdef HAVE_OPENGL
	shellGL = XtCreateApplicationShell(argv[0],
		topLevelShellWidgetClass, NULL, 0);
	XtSetValues(shellGL, arg, 2);
#endif
	skewb2d = XtCreateManagedWidget("skewb",
		skewb2dWidgetClass, topLevel, NULL, 0);
	XtAddCallback(skewb2d, XtNselectCallback,
		(XtCallbackProc) CallbackSkewb, (XtPointer) NULL);
	skewb3d = XtCreateManagedWidget("skewb",
		skewb3dWidgetClass, shell, NULL, 0);
#ifdef HAVE_OPENGL
	skewbGL = XtCreateManagedWidget("skewb",
		skewbGLWidgetClass, shellGL, NULL, 0);
#endif
#endif
	XtAddCallback(skewb3d, XtNselectCallback,
		(XtCallbackProc) CallbackSkewb, (XtPointer) NULL);
#ifdef HAVE_OPENGL
	XtAddCallback(skewbGL, XtNselectCallback,
		(XtCallbackProc) CallbackSkewb, (XtPointer) NULL);
#endif
	Initialize();
	XtRealizeWidget(topLevel);
#ifndef HAVE_MOTIF
	XtRealizeWidget(shell);
#ifdef HAVE_OPENGL
	XtRealizeWidget(shellGL);
#endif
#endif
	XGrabButton(XtDisplay(skewb2d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(skewb2d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(skewb2d),
		XCreateFontCursor(XtDisplay(skewb2d), XC_crosshair));
	XGrabButton(XtDisplay(skewb3d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(skewb3d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(skewb3d),
		XCreateFontCursor(XtDisplay(skewb3d), XC_crosshair));
#ifdef HAVE_OPENGL
	XGrabButton(XtDisplay(skewbGL), (unsigned int) AnyButton, AnyModifier,
		XtWindow(skewbGL), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(skewbGL),
		XCreateFontCursor(XtDisplay(skewbGL), XC_crosshair));
#endif
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
