/*-
# MOTIF/X-BASED MASTERBALL(tm)
#
#  xmball.c
#
###
#
#  Copyright (c) 1994 - 2004	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/11/15 X/Windows
  Version 5: 95/10/02 Xt/Motif
  Version 4: 94/09/15 Xt
*/

#ifdef WINVER
#include "MballP.h"
#include "wmball.h"
#define TITLE "wmball"

static MballRec widget;
static HWND Wnd;

#ifndef SCOREPATH
#define SCOREPATH "c:\\Windows"
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(Wnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->mball.started = b
#else
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#ifdef VMS
#include <unixlib.h>
#define getlogin() cuserid(NULL)
#else
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif
#if HAVE_FCNTL_H
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#include <Xm/ToggleBG.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtSetArg(arg[0], XtNstart, b); XtSetValues(w, arg, 1)
#include "Mball.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/mball.xpm"
#endif
#include "icons/mball.xbm"
#ifndef SCOREPATH
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif

#ifdef HAVE_MOTIF
static const char aboutHelp[] = {
	"Version 7.1\n"
	"Send bugs (reports or fixes) to the author: "
	"David Bagley <bagleyd@tux.org>\n"
	"The latest version is at: "
	"http://www.tux.org/~bagleyd/puzzles.html\n"
};
static const char optionsHelp[] = {
	"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]] "
	"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
	"[-{foreground|fg} {color}] [-{background|bg} {color}] "
	"[-{border|bd} {color}] [-wedge{0|1|2|3|4|5|6|7} {color}]\n"
	"[-wedges {int}] [-ring {int}] [-[no]orient] [-[no]practice] "
	"[-base {int}] [-{font|fn} {fontname}] [-username {string}]\n"
};
#endif
#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
	"The original puzzle has 8 sectors on a sphere "
	"(longitudinal cuts), with each sector divided into 4 segments\n"
	"(latitudinal cuts).  There are essentially 3 varieties: "
	"Geomaster 8 colors (beachball, default colors in this\n"
	"puzzle), Duomaster 2 colors (black and white beachball), "
	"and a variety of picture Masterballs.  For some reason,\n"
	"they say the Geomaster is easier than the Duomaster.  The "
	"picture Masterballs are the hardest since all the pieces\n"
	"have a set solved postion.  On the Duo and Geo Masterballs "
	"pieces could be swapped or in a different order and still\n"
	"be in a solved postion.  Zurick University's Mathematics "
	"faculty has calculated that Masterball's 32 segments can\n"
	"be arranged in 355,682,548,566,633,480,192,000,000 different "
	"possible combinations.  Masterball was invented by Dr.\n"
	"Geza Gyovai Hungarian Engineer, manufactured by Whole "
	"Systems Design, Inc..\n"
};
static const char featuresHelp[] = {
	"Press \"mouse-left\" button to move a piece.  Release "
	"\"mouse-left\" button on a piece and the pieces will\n"
	"then turn towards where the mouse button was released.  "
	"Usually, a click and release on the same wedge is\n"
	"ambiguous and the puzzle will not turn.\n"
	"Click \"mouse-center\" button, or press \"P\" or \"p\" "
	"keys, to toggle the practice mode (in practice mode the\n"
	"record should say \"practice\").  This is good for learning "
	"moves and experimenting.\n"
	"Click \"mouse-right\" button, or press \"R\" or \"r\" "
	"keys, to randomize the puzzle (this must be done first\n"
	"to set a new record).\n"
	"Press \"I\" or \"i\" keys to increase the number of rings.\n"
	"Press \"D\" or \"d\" keys to decrease the number of rings.\n"
	"Press \"O\" or \"o\" keys to toggle the orient mode.  One "
	"has to orient the wedges in orient mode, besides\n"
	"getting all the wedges to be the same color.  To do this "
	"one has to get the numbers to be on the same side\n"
	"of the ball in clockwise order.\n"
	"Press \"2\", \"4\", \"6\", \"8\", \"0\", or \"=\" keys "
	"(not the keypad 2, 4, 6, 8, 0) to change to 2, 4, 6, 8, 10,\n"
	"or 12 wedges, respectively.  Note: if there were odd "
	"number of wedges, there would be no 180 degree turn and\n"
	"therefore the puzzle would be inoperable.\n"
	"\"S\" or \"s\" keys reserved for the auto-solver "
	"(not implemented).\n"
	"1x1x1, 2x2x2, and 3x3x3 pyrmaminxs in Period 3 mode.\n"
	"Press \"U\" or \"u\" keys to undo a move.\n"
	"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
	"Press \"W\" or \"w\" keys to save a puzzle.\n"
	"Press \"Esc\" key to hide program.\n"
	"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
	"Use the key pad, \"R\" keys, or arrow keys to move without "
	"mouse clicks.\n"
	"Key pad is defined for Master Ball as:\n"
	"N / *  Upper Upper Left, Counterclockwise, Upper Upper Right\n"
	"7 8 9  Up, Upper Right\n"
	"  ^\n"
	"4<5>6  Left, Clockwise, Right\n"
	"  v\n"
	"1 2 3  Lower Left, Down, Lower Right\n"
	" 0  .  Lower Lower Left, Lower Lower Right\n"
	"Use the control key and the left mouse button, keypad, or "
	"arrow keys to move the whole Masterball.  This is not\n"
	"recorded as a turn.\n"
};
static const char referencesHelp[] = {
	"http://wsd.com/masterball\n"
};
#endif
static const char solveHelp[] = {
	"Auto-solver: sorry, not implemented.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "mball.scores"
#endif

#define MAXRINGS 6
#define NEVER -1
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} GameRecord;

static GameRecord mballRecord[2][(MAXWEDGES - MINWEDGES) / 2 + 1]
[MAXRINGS - MINRINGS + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";

#ifdef HAVE_MOTIF
static Widget moves, record, message, orientSwitch, practiceSwitch;
static Widget wedge[(MAXWEDGES - MINWEDGES) / 2 + 1], ring;
static char buff[21];
static const char *wedgeString[] =
{
	"Two", "Four", "Six", "Eight", "Ten", "Twelve"
};
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, practiceDialog, randomizeDialog;
#else
static char titleDsp[TITLELEN] = "";
#endif
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char usernameDsp[USERNAMELEN] = "Guest";
#else
static Pixmap mballIcon = None;
static Widget topLevel, mball;
static Arg arg[5];
static char *progDsp;
static char usernameDsp[USERNAMELEN] = "";
#endif

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(arg[0], XmNlabelString, xmstr);
	XtSetValues(w, arg, 1);
}
#else
static void
PrintState(
#ifndef WINVER
Widget w,
#endif
int wedges, int rings, int moves, char *msg)
{
	(void) sprintf(titleDsp, "%s.%d: %d@ (%d/%s) - %s",
		progDsp, wedges, rings, moves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(Wnd, (LPSTR) titleDsp);
#else
	XtSetArg(arg[0], XtNtitle, titleDsp);
	XtSetValues(w, arg, 1);
#endif
}
#endif

static void
InitRecords(void)
{
	int i, j, orient;

	for (orient = 0; orient < 2; orient++)
		for (i = 0; i < (MAXWEDGES - MINWEDGES) / 2 + 1; i++)
			for (j = 0; j < MAXRINGS - MINRINGS + 1; j++) {
				mballRecord[orient][i][j].score = NEVER;
				(void) strncpy(mballRecord[orient][i][j].name,
					NOACCESS, USERNAMELEN);
			}
}


static void
ReadRecords(void)
{
	FILE *fp;
	int i, j, n, orient;
	char username[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "r")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (orient = 0; orient < 2; orient++)
		for (i = 0; i < (MAXWEDGES - MINWEDGES) / 2 + 1; i++)
			for (j = 0; j < MAXRINGS - MINRINGS + 1; j++) {
				(void) fscanf(fp, "%d %s\n", &n, username);
				if (n <= mballRecord[orient][i][j].score ||
				    mballRecord[orient][i][j].score <= NEVER) {
					mballRecord[orient][i][j].score = n;
					(void) strncpy(mballRecord[orient][i][j].name,
						username, USERNAMELEN);
				}
			}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int i, j, orient;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "w")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockfile[FILENAMELEN];

		(void) strncpy(lockfile, name, FILENAMELEN - 6);
		(void) strcat(lockfile, ".lock");
		while (((lfd = open(lockfile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
			  "Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
			  "Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (orient = 0; orient < 2; orient++) {
			for (i = 0; i < (MAXWEDGES - MINWEDGES) / 2 + 1; i++) {
				for (j = 0; j < MAXRINGS - MINRINGS + 1; j++)
					(void) fprintf(fp, "%d %s\n",
						mballRecord[orient][i][j].score, mballRecord[orient][i][j].name);
				(void) fprintf(fp, "\n");
			}
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockfile);
#endif
		(void) fclose(fp);
	}
	free(lname);
	free(fname);
}

static void
PrintRecord(int wedges, int rings, Boolean orient, Boolean practice)
{
	int i = (orient) ? 1 : 0;
	int w = (wedges - MINWEDGES) / 2;
	int r = rings - MINRINGS;

	if (practice)
		(void) strncpy(recordDsp, "practice", MESSAGELEN);
	else if (rings > MAXRINGS)
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGELEN);
	else if (mballRecord[i][w][r].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			mballRecord[i][w][r].score, mballRecord[i][w][r].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static Boolean
HandleSolved(int counter, int wedges, int rings, Boolean orient)
{
	int i = (orient) ? 1 : 0;
	int w = (wedges - MINWEDGES) / 2;
	int r = rings - MINRINGS;

	if (rings <= MAXRINGS && (counter < mballRecord[i][w][r].score ||
			mballRecord[i][w][r].score <= NEVER)) {
		ReadRecords();	/* Maybe its been updated by another */
		mballRecord[i][w][r].score = counter;
		(void) strncpy(mballRecord[i][w][r].name, usernameDsp,
			USERNAMELEN);
		if (orient && (counter < mballRecord[!i][w][r].score ||
				mballRecord[!i][w][r].score <= NEVER)) {
			mballRecord[!i][w][r].score = counter;
			(void) strncpy(mballRecord[!i][w][r].name, usernameDsp,
				USERNAMELEN);
		}
		WriteRecords();
		PrintRecord(wedges, rings, orient, False);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
MballWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	int wedges, rings;
	Boolean orient, practice;
	char *username;

#ifdef WINVER
	InitializeMball(w, brush);

	wedges = w->mball.wedges;
	rings = w->mball.rings;
	orient = w->mball.orient;
	practice = w->mball.practice;
	username = w->mball.username;
#else
	XtVaGetValues(w,
		XtNuserName, &username,
		XtNwedges, &wedges,
		XtNrings, &rings,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
#ifdef HAVE_MOTIF
	XmToggleButtonSetState(wedge[(wedges - MINWEDGES) / 2], True, False);
	if (rings > MAXRINGS)
		XtVaSetValues(ring, XmNmaximum, rings, NULL);
	XmScaleSetValue(ring, rings);
	XmToggleButtonSetState(orientSwitch, orient, True);
	XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
#endif
	SET_STARTED(w, False);
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(usernameDsp, username, USERNAMELEN);
#endif
	if (!strcmp(username, "") || !strcmp(username, "(null)") ||
	    !strcmp(username, NOACCESS) || !strcmp(username, NOBODY)) {
#ifdef WINVER
		(void) strncpy(usernameDsp, username, USERNAMELEN);
#else
		/* The NOACCESS is not necessary, but it may stop silliness. */
		(void) sprintf(usernameDsp, "%s", getlogin());
		if (!strcmp(usernameDsp, "") ||
				!strcmp(usernameDsp, "(null)") ||
				!strcmp(usernameDsp, NOBODY) ||
				!strcmp(usernameDsp, NOACCESS))
			/* It really IS nobody */
			(void) sprintf(usernameDsp, "%s", "guest");
#endif
	}
	PrintRecord(wedges, rings, orient, practice);
#ifndef HAVE_MOTIF
	PrintState(
#ifndef WINVER
		XtParent(w),
#endif
		wedges, rings, movesDsp, messageDsp);
#endif
}

#ifdef WINVER
void
SetMball(MballWidget w, int reason)
#else
static void
CallbackMball(Widget w, caddr_t clientData, mballCallbackStruct * callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	int wedges, rings;
	Boolean orient, practice, start, cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	wedges = w->mball.wedges;
	rings = w->mball.rings;
	orient = w->mball.orient;
	practice = w->mball.practice;
	cheat = w->mball.cheat;
	start = w->mball.started;
#else
	XtVaGetValues(w,
		XtNwedges, &wedges,
		XtNrings, &rings,
		XtNorient, &orient,
		XtNpractice, &practice,
		XtNstart, &start,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case MBALL_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
			break;
#ifndef WINVER
		case MBALL_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(practiceDialog);
#else
			XtSetArg(arg[0], XtNmenu, 10); /* menu choice */
			XtSetValues(mball, arg, 1);
#endif
			break;
		case MBALL_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(randomizeDialog);
#else
			XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
			XtSetValues(mball, arg, 1);
#endif
			break;
#endif
		case MBALL_SOLVE_MESSAGE:
#ifdef WINVER
			(void) MessageBox(w->core.hWnd, solveHelp,
				"Auto-Solve", MB_OK);
#else
#ifdef HAVE_MOTIF
			XtManageChild(solveDialog);
#else
			(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
			break;
		case MBALL_RESTORE:
			if (practice) {
				(void) strncpy(recordDsp, "practice", MESSAGELEN);
#ifdef HAVE_MOTIF
				PrintState(record, recordDsp);
#endif
			}
			movesDsp = 0;
			break;
		case MBALL_RESET:
			movesDsp = 0;
			break;
		case MBALL_AMBIGUOUS:
			(void) strncpy(messageDsp, "Ambiguous move", MESSAGELEN);
			break;
		case MBALL_ILLEGAL:
			if (practice || start)
				(void) strncpy(messageDsp, "Illegal move",
					MESSAGELEN);
			else
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			break;
		case MBALL_MOVED:
			movesDsp++;
			SET_STARTED(w, True);
			break;
		case MBALL_CONTROL:
			return;
		case MBALL_SOLVED:
			if (practice)
				movesDsp = 0;
			else if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", usernameDsp);
			else if (HandleSolved(movesDsp, wedges, rings, orient))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", usernameDsp);
			else
				(void) strncpy(messageDsp, "Solved!",
					MESSAGELEN);
			SET_STARTED(w, False);
			break;
		case MBALL_PRACTICE:
			movesDsp = 0;
			practice = !practice;
			if (!practice)
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			PrintRecord(wedges, rings, orient, practice);
#ifdef WINVER
			w->mball.practice = practice;
			w->mball.started = False;
#else
			XtSetArg(arg[0], XtNpractice, practice);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
#endif
			break;
		case MBALL_RANDOMIZE:
			movesDsp = 0;
#ifdef WINVER
			w->mball.practice = False;
			w->mball.started = False;
#else
			XtSetArg(arg[0], XtNpractice, False);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
#endif
			break;
		case MBALL_DEC:
			movesDsp = 0;
			rings--;
			PrintRecord(wedges, rings, orient, practice);
#ifdef WINVER
			w->mball.rings = rings;
#else
			XtSetArg(arg[0], XtNrings, rings);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmScaleSetValue(ring, rings);
			if (rings >= MAXRINGS)
				XtVaSetValues(ring, XmNmaximum, rings, NULL);
#endif
#endif
			break;
		case MBALL_ORIENT:
			movesDsp = 0;
			orient = !orient;
			PrintRecord(wedges, rings, orient, practice);
#ifdef WINVER
			w->mball.orient = orient;
#else
			XtSetArg(arg[0], XtNorient, orient);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(orientSwitch, orient, True);
#endif
#endif
			break;
		case MBALL_INC:
			movesDsp = 0;
			rings++;
			PrintRecord(wedges, rings, orient, practice);
#ifdef WINVER
			w->mball.rings = rings;
#else
			XtSetArg(arg[0], XtNrings, rings);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			if (rings > MAXRINGS)
				XtVaSetValues(ring, XmNmaximum, rings, NULL);
			XmScaleSetValue(ring, rings);
#endif
#endif
			break;
		case MBALL_WEDGE2:
		case MBALL_WEDGE4:
		case MBALL_WEDGE6:
		case MBALL_WEDGE8:
		case MBALL_WEDGE10:
		case MBALL_WEDGE12:
			movesDsp = 0;
			wedges = 2 * (reason - MBALL_WEDGE2) +
				MINWEDGES;
			PrintRecord(wedges, rings, orient, practice);
#ifdef WINVER
			w->mball.wedges = wedges;
#else
			XtSetArg(arg[0], XtNwedges, wedges);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(wedge[(wedges - MINWEDGES) / 2],
				 True, True);
#endif
#endif
			break;
		case MBALL_COMPUTED:
			SET_STARTED(w, False);
			break;
		case MBALL_UNDO:
			movesDsp--;
			SET_STARTED(w, True);
			break;
	}
#ifdef HAVE_MOTIF
	PrintState(message, messageDsp);
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
#else
	PrintState(
#ifndef WINVER
		XtParent(w),
#endif
		wedges, rings, movesDsp, messageDsp);
#endif

}

#ifdef WINVER
static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	Wnd = widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.mball.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			EnterMball(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.mball.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			LeaveMball(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyMball(brush);
			break;
		case WM_SIZE:
			ResizeMball(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ExposeMball(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			RandomizeMball(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			SelectMball(&widget, LOWORD(lParam), HIWORD(lParam),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ReleaseMball(&widget, LOWORD(lParam), HIWORD(lParam),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDM_GET:
					GetMball(&widget);
					ResizeMball(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WriteMball(&widget);
					break;
				case IDM_EXIT:
					DestroyMball(brush);
					break;
				case IDM_HIDE:
					HideMball(&widget);
					break;
				case IDM_CLEAR:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					ClearMball(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					UndoMball(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_RANDOMIZE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					RandomizeMball(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_PRACTICE:
					PracticeMball(&widget);
					SizeMball(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					SolveMball(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_ORIENT:
					OrientizeMball(&widget);
					SizeMball(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_TOP:
				case IDM_TTR:
				case IDM_TR:
				case IDM_RIGHT:
				case IDM_BR:
				case IDM_BBR:
				case IDM_BOTTOM:
				case IDM_BBL:
				case IDM_BL:
				case IDM_LEFT:
				case IDM_TL:
				case IDM_TTL:
				case IDM_CW:
				case IDM_CCW:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MoveMballInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - IDM_TOP, FALSE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}

					break;
				case IDM_CONTROL_TOP:
				case IDM_CONTROL_TTR:
				case IDM_CONTROL_TR:
				case IDM_CONTROL_RIGHT:
				case IDM_CONTROL_BR:
				case IDM_CONTROL_BBR:
				case IDM_CONTROL_BOTTOM:
				case IDM_CONTROL_BBL:
				case IDM_CONTROL_BL:
				case IDM_CONTROL_LEFT:
				case IDM_CONTROL_TL:
				case IDM_CONTROL_TTL:
				case IDM_CONTROL_CW:
				case IDM_CONTROL_CCW:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MoveMballInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - IDM_CONTROL_TOP, TRUE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}

					break;
				case IDM_DEC:
					if (DecrementMball(&widget)) {
						SizeMball(&widget);
						(void) InvalidateRect(hWnd, NULL, TRUE);
					}
					break;
				case IDM_INC:
					IncrementMball(&widget);
					SizeMball(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WEDGE2:
				case IDM_WEDGE4:
				case IDM_WEDGE6:
				case IDM_WEDGE8:
				case IDM_WEDGE10:
				case IDM_WEDGE12:
					WedgeMball(&widget, (int) LOWORD(wParam) - IDM_WEDGE2);
					SizeMball(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
					 	"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
					 	"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
					 	"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
Usage(char * programName)
{
	(void) fprintf(stderr, "usage: %s\n", programName);
	(void) fprintf(stderr,
		"\t[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n");
	(void) fprintf(stderr,
		"\t[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n");
	(void) fprintf(stderr,
		"\t[-{foreground|fg} {color}] [-{background|bg} {color}]\n");
	(void) fprintf(stderr,
		"\t[-{border|bd} {color}] [-wedge{0|1|2|3|4|5|6|7} {color}]\n");
	(void) fprintf(stderr,
		"\t[-{wedges {int}}] [-{rings {int}}] [-[no]orient] [-[no]practice]\n");
	(void) fprintf(stderr,
		"\t[-base {int}] [-{font|fn} {fontname}] [-username {string}]\n");
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*mball.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*mball.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*mball.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*mball.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*mball.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*mball.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*mball.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*mball.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*mball.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*mball.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-wedge0", (char *) "*mball.wedgeColor0", XrmoptionSepArg, NULL},
	{(char *) "-wedge1", (char *) "*mball.wedgeColor1", XrmoptionSepArg, NULL},
	{(char *) "-wedge2", (char *) "*mball.wedgeColor2", XrmoptionSepArg, NULL},
	{(char *) "-wedge3", (char *) "*mball.wedgeColor3", XrmoptionSepArg, NULL},
	{(char *) "-wedge4", (char *) "*mball.wedgeColor4", XrmoptionSepArg, NULL},
	{(char *) "-wedge5", (char *) "*mball.wedgeColor5", XrmoptionSepArg, NULL},
	{(char *) "-wedge6", (char *) "*mball.wedgeColor6", XrmoptionSepArg, NULL},
	{(char *) "-wedge7", (char *) "*mball.wedgeColor7", XrmoptionSepArg, NULL},
	{(char *) "-wedges", (char *) "*mball.wedges", XrmoptionSepArg, NULL},
	{(char *) "-rings", (char *) "*mball.rings", XrmoptionSepArg, NULL},
	{(char *) "-orient", (char *) "*mball.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*mball.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-practice", (char *) "*mball.practice", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopractice", (char *) "*mball.practice", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-base", (char *) "*mball.base", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*mball.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*mball.font", XrmoptionSepArg, NULL},
	{(char *) "-username", (char *) "*mball.userName", XrmoptionSepArg, NULL}
};

#ifdef HAVE_MOTIF
static void
CallbackMballPractice(Widget w, XtPointer clientData, XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 10); /* menu choice */
		XtSetValues(mball, arg, 1);
	}
}

static void
CallbackMballRandomize(Widget w, XtPointer clientData, XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
		XtSetValues(mball, arg, 1);
	}
}

static void
WedgeToggle(Widget w, int bit, XmToggleButtonCallbackStruct * cbs)
{
	int wedges, rings;
	Boolean orient, practice;

	if (cbs->set) {
		XtVaGetValues(mball,
			XtNrings, &rings,
			XtNorient, &orient,
			XtNpractice, &practice, NULL);
		wedges = bit * 2 + MINWEDGES;
		XtVaSetValues(mball,
			XtNwedges, wedges, NULL);
		movesDsp = 0;
		(void) sprintf(buff, "%d", movesDsp);
		PrintState(moves, buff);
		PrintRecord(wedges, rings, orient, practice);
		(void) strcpy(messageDsp, "");
		PrintState(message, messageDsp);
	}
}

static void
RingSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int wedges, rings = cbs->value, old;
	Boolean orient, practice;

	XtVaGetValues(mball,
		XtNwedges, &wedges,
		XtNrings, &old,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	if (old == rings)
		return;
	XtVaSetValues(mball,
		XtNrings, rings, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(wedges, rings, orient, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
OrientToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	int wedges, rings;
	Boolean orient = cbs->set, practice, oldOrient;

	XtVaGetValues(mball,
		XtNwedges, &wedges,
		XtNrings, &rings,
		XtNorient, &oldOrient,
		XtNpractice, &practice, NULL);
	if (oldOrient == orient)
		return;
	XtVaSetValues(mball,
		XtNorient, orient, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(wedges, rings, orient, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
PracticeToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	int wedges, rings;
	Boolean orient, practice = cbs->set, oldPract;

	XtVaGetValues(mball,
		XtNwedges, &wedges,
		XtNrings, &rings,
		XtNorient, &orient,
		XtNpractice, &oldPract, NULL);
	if (oldPract == practice)
		return;
	XtVaSetValues(mball,
		XtNpractice, practice,
		XtNstart, False, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(wedges, rings, orient, practice);
	if (practice)
		(void) strcpy(messageDsp, "");
	else
		(void) strncpy(messageDsp, "Randomize to start", MESSAGELEN);
	PrintState(message, messageDsp);
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == 2)
		exit(0);
	XtSetArg(arg[0], XtNmenu, val);
	XtSetValues(mball, arg, 1);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;
	XtSetArg(arg[0], XtNmenu, val + 3); /* GWQ */
	XtSetValues(mball, arg, 1);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d\n", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, labelRowCol, wedgesRowCol, ringsRowCol;
	Widget messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString clearString, undoString, randomizeString, solveString;
	XmString incrementString, decrementString;
	XmString orientString, practiceString;
	unsigned int i;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Mball",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#if HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) mball_xpm, &mballIcon, NULL,
			&xpmAttributes);
	}
	if (mballIcon == (Pixmap) NULL)
#endif
		mballIcon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) mball_bits,
			mball_width, mball_height);
	XtSetArg(arg[0], XtNiconPixmap, mballIcon);
#ifdef HAVE_MOTIF
	/* not XmEXPLICIT */
	XtSetArg(arg[1], XmNkeyboardFocusPolicy, XmPOINTER);
	XtSetValues(topLevel, arg, 2);
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	clearString = XmStringCreateSimple((char *) "Clear");
	undoString = XmStringCreateSimple((char *) "Undo");
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "(Solve)");
	incrementString = XmStringCreateSimple((char *) "Increment");
	decrementString = XmStringCreateSimple((char *) "Decrement");
	orientString = XmStringCreateSimple((char *) "Orientize");
	practiceString = XmStringCreateSimple((char *) "Practice");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, incrementString, 'I', NULL, NULL,
		XmVaPUSHBUTTON, decrementString, 'D', NULL, NULL,
		XmVaPUSHBUTTON, orientString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
		NULL);
	XmStringFree(clearString);
	XmStringFree(undoString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(incrementString);
	XmStringFree(decrementString);
	XmStringFree(orientString);
	XmStringFree(practiceString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	practiceDialog = createQuery(topLevel,
		(char *) "Are you sure you want to toggle the practice mode?",
		(char *) "Practice Query",
		(XtCallbackProc) CallbackMballPractice);
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) CallbackMballRandomize);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move", 5, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	labelRowCol = XtVaCreateManagedWidget("labelRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	(void) XtVaCreateManagedWidget("WedgeText",
		xmLabelGadgetClass, labelRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Wedges:", 8, NULL);
	wedgesRowCol = XtVaCreateManagedWidget("wedgesRowCol",
		xmRowColumnWidgetClass, labelRowCol,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN,
		XmNradioBehavior, True, NULL);
	for (i = 0; i < XtNumber(wedgeString); i++) {
		wedge[i] = XtVaCreateManagedWidget(wedgeString[i],
			xmToggleButtonGadgetClass, wedgesRowCol,
			XmNradioBehavior, True, NULL);
		XtAddCallback(wedge[i], XmNvalueChangedCallback,
			(XtCallbackProc) WedgeToggle, (XtPointer) i);
	}
	ringsRowCol = XtVaCreateManagedWidget("ringsRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	ring = XtVaCreateManagedWidget("ring",
		xmScaleWidgetClass, ringsRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Rings", 6,
		XmNminimum, MINRINGS,
		XmNmaximum, MAXRINGS,
		XmNvalue, DEFAULTRINGS,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(ring, XmNvalueChangedCallback,
		(XtCallbackProc) RingSlider, (XtPointer) NULL);

	orientSwitch = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, ringsRowCol,
		XmNset, DEFAULTORIENT, NULL);
	XtAddCallback(orientSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) OrientToggle, (XtPointer) NULL);
	practiceSwitch = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, ringsRowCol,
		XmNset, DEFAULTPRACTICE, NULL);
	XtAddCallback(practiceSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) PracticeToggle, (XtPointer) NULL);

	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Masterball! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);
	mball = XtCreateManagedWidget("mball",
		mballWidgetClass, mainPanel, NULL, 0);
#else
	XtSetArg(arg[1], XtNinput, True);
	XtSetValues(topLevel, arg, 2);
	mball = XtCreateManagedWidget("mball",
		mballWidgetClass, topLevel, NULL, 0);
#endif
	XtAddCallback(mball, XtNselectCallback,
		(XtCallbackProc) CallbackMball, (XtPointer) NULL);
	Initialize(mball);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(mball), (unsigned int) AnyButton, AnyModifier,
		XtWindow(mball), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(mball),
		XCreateFontCursor(XtDisplay(mball), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
