/*-
# X-BASED MASTERBALL(tm)
#
#  Mball.c
#
###
#
#  Copyright (c) 1994 - 2004	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Mball */

#include "MballP.h"

#ifdef WINVER
#ifndef LOGPATH
#define LOGPATH "/usr/tmp"
#endif
#ifndef INIFILE
#define INIFILE "wmball.ini"
#endif

#define SECTION "setup"

static const char *wedgeColorString[MAXWEDGES] =
{
	"255 255 0",
	"0 0 255",
	"255 0 0",
	"255 0 255",
	"0 255 0",
	"255 165 0",
	"1 255 255",
	"0 100 0",
	"255 192 203",
	"139 69 19",
	"176 196 222",
	"205 92 92"
};

static char wedgeColorChar[MAXWEDGES] =
{'Y', 'B', 'R', 'M', 'G', 'O', 'C', 'D', 'P', 'L', 'A', 'T'};
#else

#include <math.h>
#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

#ifndef LOGPATH
#ifdef VMS
#define LOGPATH "SYS$SCRATCH:"
#else
#define LOGPATH "/usr/tmp"
#endif
#endif

static void ResizeMball(MballWidget w);
static void SizeMball(MballWidget w);
static Boolean SetValuesMball(Widget current, Widget request, Widget renew);
static void DestroyMball(Widget old);
static void QuitMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void InitializeMball(Widget request, Widget renew);
static void ExposeMball(Widget renew, XEvent * event, Region region);
static void HideMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void PracticeMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void PracticeMballMaybe(MballWidget w, XEvent * event, char **args, int nArgs);
static void PracticeMball2(MballWidget w, XEvent * event, char **args, int nArgs);
static void RandomizeMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void RandomizeMballMaybe(MballWidget w, XEvent * event, char **args, int nArgs);
static void RandomizeMball2(MballWidget w, XEvent * event, char **args, int nArgs);
static void GetMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void WriteMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void ClearMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void UndoMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void SolveMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void IncrementMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void DecrementMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void OrientizeMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void Wedge2ModeMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void Wedge4ModeMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void Wedge6ModeMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void Wedge8ModeMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void Wedge10ModeMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void Wedge12ModeMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void EnterMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void LeaveMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballTtl(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballTl(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballTop(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballTtr(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballTr(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballLeft(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballCw(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballRight(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballBl(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballBbl(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballBottom(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballBbr(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballBr(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballCcw(MballWidget w, XEvent * event, char **args, int nArgs);
static void MoveMballInput(MballWidget w, int x, int y, int direction, int control);
static void SelectMball(MballWidget w, XEvent * event, char **args, int nArgs);
static void ReleaseMball(MballWidget w, XEvent * event, char **args, int nArgs);

static char defaultTranslationsMball[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>KP_Divide: MoveCcw()\n\
 <KeyPress>R5: MoveCcw()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Num_Lock: MoveTtl()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>KP_Multiply: MoveTtr()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>Begin: MoveCw()\n\
 <KeyPress>KP_5: MoveCw()\n\
 <KeyPress>R11: MoveCw()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>KP_0: MoveBbl()\n\
 <KeyPress>KP_Insert: MoveBbl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <KeyPress>KP_Decimal: MoveBbr()\n\
 <KeyPress>KP_Delete: MoveBbr()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R15: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>r: Randomize()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>i: Increment()\n\
 <KeyPress>d: Decrement()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>2: Wedge2()\n\
 <KeyPress>4: Wedge4()\n\
 <KeyPress>6: Wedge6()\n\
 <KeyPress>8: Wedge8()\n\
 <KeyPress>0: Wedge10()\n\
 <KeyPress>=: Wedge12()\n\
 <KeyPress>.: Wedge12()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

/* '=' is the 2nd key to the right of '0' on keyboard
   '.' follows '0' on keypad
 */

static XtActionsRec actionsListMball[] =
{
	{(char *) "Quit", (XtActionProc) QuitMball},
	{(char *) "Hide", (XtActionProc) HideMball},
	{(char *) "MoveCcw", (XtActionProc) MoveMballCcw},
	{(char *) "MoveTl", (XtActionProc) MoveMballTl},
	{(char *) "MoveTtl", (XtActionProc) MoveMballTtl},
	{(char *) "MoveTop", (XtActionProc) MoveMballTop},
	{(char *) "MoveTtr", (XtActionProc) MoveMballTtr},
	{(char *) "MoveTr", (XtActionProc) MoveMballTr},
	{(char *) "MoveLeft", (XtActionProc) MoveMballLeft},
	{(char *) "MoveCw", (XtActionProc) MoveMballCw},
	{(char *) "MoveRight", (XtActionProc) MoveMballRight},
	{(char *) "MoveBl", (XtActionProc) MoveMballBl},
	{(char *) "MoveBbl", (XtActionProc) MoveMballBbl},
	{(char *) "MoveBottom", (XtActionProc) MoveMballBottom},
	{(char *) "MoveBbr", (XtActionProc) MoveMballBbr},
	{(char *) "MoveBr", (XtActionProc) MoveMballBr},
	{(char *) "Select", (XtActionProc) SelectMball},
	{(char *) "Release", (XtActionProc) ReleaseMball},
	{(char *) "Practice", (XtActionProc) PracticeMball},
	{(char *) "PracticeMaybe", (XtActionProc) PracticeMballMaybe},
	{(char *) "Practice2", (XtActionProc) PracticeMball2},
	{(char *) "Randomize", (XtActionProc) RandomizeMball},
	{(char *) "RandomizeMaybe", (XtActionProc) RandomizeMballMaybe},
	{(char *) "Randomize2", (XtActionProc) RandomizeMball2},
	{(char *) "Get", (XtActionProc) GetMball},
	{(char *) "Write", (XtActionProc) WriteMball},
	{(char *) "Clear", (XtActionProc) ClearMball},
	{(char *) "Undo", (XtActionProc) UndoMball},
	{(char *) "Solve", (XtActionProc) SolveMball},
	{(char *) "Increment", (XtActionProc) IncrementMball},
	{(char *) "Decrement", (XtActionProc) DecrementMball},
	{(char *) "Orientize", (XtActionProc) OrientizeMball},
	{(char *) "Wedge2", (XtActionProc) Wedge2ModeMball},
	{(char *) "Wedge4", (XtActionProc) Wedge4ModeMball},
	{(char *) "Wedge6", (XtActionProc) Wedge6ModeMball},
	{(char *) "Wedge8", (XtActionProc) Wedge8ModeMball},
	{(char *) "Wedge10", (XtActionProc) Wedge10ModeMball},
	{(char *) "Wedge12", (XtActionProc) Wedge12ModeMball},
	{(char *) "Enter", (XtActionProc) EnterMball},
	{(char *) "Leave", (XtActionProc) LeaveMball}
};

static XtResource resourcesMball[] =
{
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.username),
	 XtRString, (caddr_t) "nobody"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(MballWidget, mball.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(MballWidget, mball.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(MballWidget, mball.frameColor),
	 XtRString, (caddr_t) "cyan" /*XtDefaultForeground*/},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(MballWidget, mball.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNwedgeColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.wedgeName[0]),
	 XtRString, (caddr_t) "yellow"},
	{XtNwedgeColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.wedgeName[1]),
	 XtRString, (caddr_t) "blue"},
	{XtNwedgeColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.wedgeName[2]),
	 XtRString, (caddr_t) "red"},
	{XtNwedgeColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.wedgeName[3]),
	 XtRString, (caddr_t) "magenta"},
	{XtNwedgeColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.wedgeName[4]),
	 XtRString, (caddr_t) "green"},
	{XtNwedgeColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.wedgeName[5]),
	 XtRString, (caddr_t) "orange"},
	{XtNwedgeColor6, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.wedgeName[6]),
	 XtRString, (caddr_t) "cyan"},
	{XtNwedgeColor7, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.wedgeName[7]),
	 XtRString, (caddr_t) "Dark Green"},
	{XtNwedgeColor8, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.wedgeName[8]),
	 XtRString, (caddr_t) "pink"},
	{XtNwedgeColor9, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.wedgeName[9]),
	 XtRString, (caddr_t) "SaddleBrown"},
	{XtNwedgeColor10, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.wedgeName[10]),
	 XtRString, (caddr_t) "LightSteelBlue"},
	{XtNwedgeColor11, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.wedgeName[11]),
	 XtRString, (caddr_t) "IndianRed"},
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(MballWidget, core.width),
	 XtRString, (caddr_t) "200"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(MballWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(MballWidget, mball.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverse, XtCReverse, XtRBoolean, sizeof (Boolean),
	 XtOffset(MballWidget, mball.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNwedges, XtCWedges, XtRInt, sizeof (int),
	 XtOffset(MballWidget, mball.wedges),
	 XtRString, (caddr_t) "8"}, /* DEFAULTWEDGES */
	{XtNrings, XtCRings, XtRInt, sizeof (int),
	 XtOffset(MballWidget, mball.rings),
	 XtRString, (caddr_t) "4"}, /* DEFAULTRINGS */
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(MballWidget, mball.orient),
	 XtRString, (caddr_t) "FALSE"}, /* DEFAULTORIENT */
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(MballWidget, mball.practice),
	 XtRString, (caddr_t) "FALSE"}, /* DEFAULTPRACTICE */
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(MballWidget, mball.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNbase, XtCBase, XtRInt, sizeof (int),
	 XtOffset(MballWidget, mball.base),
	 XtRString, (caddr_t) "16"}, /* DEFAULTBASE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(MballWidget, mball.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(MballWidget, mball.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(MballWidget, mball.menu),
	 XtRString, (caddr_t) "-1"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(MballWidget, mball.select),
	 XtRCallback, (caddr_t) NULL}
};

MballClassRec mballClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "Mball",	/* class name */
		sizeof (MballRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializeMball,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListMball,	/* actions */
		XtNumber(actionsListMball),	/* num actions */
		resourcesMball,	/* resources */
		XtNumber(resourcesMball),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyMball,	/* destroy */
		(XtWidgetProc) ResizeMball,	/* resize */
		(XtExposeProc) ExposeMball,	/* expose */
		(XtSetValuesFunc) SetValuesMball,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsMball,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass mballWidgetClass = (WidgetClass) & mballClassRec;

void
SetMball(MballWidget w, int reason)
{
	mballCallbackStruct cb;

	cb.reason = reason;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}
#endif

static void
loadFont(MballWidget w)
{
#ifndef WINVER
	Display *display = XtDisplay(w);
	const char *altfontname = "-*-times-*-r-*-*-*-180-*";
	char buf[512];

	if (w->mball.fontInfo) {
		XUnloadFont(XtDisplay(w), w->mball.fontInfo->fid);
		XFreeFont(XtDisplay(w), w->mball.fontInfo);
	}
	if ((w->mball.fontInfo = XLoadQueryFont(display,
			w->mball.font)) == NULL) {
		(void) sprintf(buf,
			"Cannot open %s font.\nAttempting %s font as alternate\n",
			w->mball.font, altfontname);
		DISPLAY_WARNING(buf);
		if ((w->mball.fontInfo = XLoadQueryFont(display,
				altfontname)) == NULL) {
			(void) sprintf(buf,
				"Cannot open %s alternate font.\nUse the -font option to specify a font to use.\n",
				altfontname);
			DISPLAY_WARNING(buf);
		}
	}
	if (w->mball.fontInfo) {
		w->mball.letterOffset.x = XTextWidth(w->mball.fontInfo, "8", 1)
			/ 2;
		w->mball.letterOffset.y = w->mball.fontInfo->max_bounds.ascent
			/ 2;
	} else
#endif
	{
		w->mball.letterOffset.x = 3;
		w->mball.letterOffset.y = 4;
	}
}

#ifndef LOGFILE
#define LOGFILE "mball.log"
#endif

static int mapDirToWedge[(MAXWEDGES - MINWEDGES) / 2 + 1][COORD] =
{
	{
		0, 6, 6, 6, 6, 6, 0, 6, 6, 6, 6, 6
	},
	{
		0, 6, 6, 1, 6, 6, 0, 6, 6, 1, 6, 6
	},
	{
		0, 6, 1, 6, 2, 6, 0, 6, 1, 6, 2, 6
	},
	{
		0, 6, 1, 2, 3, 6, 0, 6, 1, 2, 3, 6
	},
	{
		0, 1, 2, 6, 3, 4, 0, 1, 2, 6, 3, 4
	},
	{
		0, 1, 2, 3, 4, 5, 0, 1, 2, 3, 4, 5
	}
};

static int mapWedgeToDir[(MAXWEDGES - MINWEDGES) / 2 + 1][MAXWEDGES] =
{
	{
		0,  6, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12
	},
	{
		0,  3,  6,  9, 12, 12, 12, 12, 12, 12, 12, 12
	},
	{
		0,  2,  4,  6,  8, 10, 12, 12, 12, 12, 12, 12
	},
	{
		0,  2,  3,  4,  6,  8,  9, 10, 12, 12, 12, 12
	},
	{
		0,  1,  2,  4,  5,  6,  7,  8, 10, 11, 12, 12
	},
	{
		0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11
	}
};


void
intCat(char ** string, const char * var1, const int var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + 21))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	(void) sprintf(*string, "%s%d", var1, var2);
}

void
stringCat(char ** string, const char * var1, const char * var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + strlen(var2) + 1))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	(void) sprintf(*string, "%s%s", var1, var2);
}

static void
CheckWedges(MballWidget w)
{
	char *buf1 = NULL, *buf2 = NULL;

	if (w->mball.wedges < MINWEDGES || w->mball.wedges > MAXWEDGES) {
		intCat(&buf1,
			"Number of wedges out of bounds, use even ",
			MINWEDGES);
		stringCat(&buf2, buf1, "..");
		free(buf1);
		intCat(&buf1, buf2, MAXWEDGES);
		free(buf2);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTWEDGES);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->mball.wedges = DEFAULTWEDGES;
	}
	if (w->mball.rings < MINRINGS) {
		intCat(&buf1,
			"Number of rings out of bounds, use at least ",
			MINRINGS);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTRINGS);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->mball.rings = DEFAULTRINGS;
	}
#if 0
	if (w->mball.delay < 0) {
		intCat(&buf1, "Delay can not be negative (",
			w->mball.delay);
		stringCat(&buf2, buf1, "), taking absolute value");
		free(buf1);
		DISPLAY_WARNING(buf2);
		free(buf2);
		w->mball.delay = -w->mball.delay;
	}
#endif
	if (w->mball.base < MINBASE || w->mball.base > MAXBASE) {
		intCat(&buf1, "Base out of bounds, use ", MINBASE);
		stringCat(&buf2, buf1, "..");
		free(buf1);
		intCat(&buf1, buf2, MAXBASE);
		free(buf2);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTBASE);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->mball.base = DEFAULTBASE;
	}
}

Boolean
CheckSolved(MballWidget w)
{
	int wedge, ring;
	MballLoc test;

	if (w->mball.orient)
		for (wedge = 0; wedge < w->mball.wedges; wedge++) {
			if (wedge == 0) {
				test.wedge = w->mball.mballLoc[wedge][0].wedge;
				test.direction = w->mball.mballLoc[wedge][0].direction;
			}
			for (ring = 0; ring < w->mball.rings; ring++) {
				if (test.direction != w->mball.mballLoc[wedge][ring].direction)
					return False;
				if (test.direction) {
					if ((w->mball.wedges - w->mball.mballLoc[wedge][ring].wedge +
					     test.wedge) % w->mball.wedges != wedge)
						return False;
				} else {
					if ((w->mball.wedges + w->mball.mballLoc[wedge][ring].wedge -
					     test.wedge) % w->mball.wedges != wedge)
						return False;
				}
			}
	} else
		for (wedge = 0; wedge < w->mball.wedges; wedge++)
			for (ring = 0; ring < w->mball.rings; ring++)
				if (ring == 0) {
					test.wedge = w->mball.mballLoc[wedge][ring].wedge;
					test.direction = w->mball.mballLoc[wedge][ring].direction;
				} else if (test.wedge != w->mball.mballLoc[wedge][ring].wedge)
					return False;
	return True;
}

static int
int2String(MballWidget w, char *buf, int number, int base, Boolean capital)
{
	int digit, mult = base, last, position;
	int a, i, j, s, r;

	if (capital) {
		a = 'A', i = 'I', j = 'J', s = 'S', r = 'R';
	} else {
		a = 'a', i = 'i', j = 'j', s = 's', r = 'r';
	}
	if (number < 0) {
		char *buff;

		intCat(&buff, "int2String: 0 > ", number);
		DISPLAY_WARNING(buff);
		free(buff);
		return 0;
	}
	last = 1;
	while (number >= mult) {
		last++;
		mult *= base;
	}
	for (position = 0; position < last; position++) {
		mult /= base;
		digit = number / mult;
		number -= digit * mult;
		buf[position] = digit + '0';
		if (buf[position] > '9') {	/* ASCII */
			buf[position] += (a - '9' - 1);
		} else if (buf[position] < '0') {	/* EBCDIC */
			buf[position] += (a - '9' - 1);
			if (buf[position] > i)
				buf[position] += (j - i - 1);
			if (buf[position] > r)
				buf[position] += (s - r - 1);
		}
	}
	buf[last] = '\0';
	return last;
}

#ifdef DEBUG
static void
PrintMball(MballWidget w)
{
	int wedge, ring;

	for (wedge = 0; wedge < w->mball.wedges; wedge++) {
		for (ring = 0; ring < w->mball.rings; ring++) {
			(void) printf("%d %d  ", w->mball.mballLoc[wedge][ring].wedge,
				w->mball.mballLoc[wedge][ring].direction);
		}
		(void) printf("\n");
	}
	(void) printf("\n");
}

#endif

static void
LetterPosition(MballWidget w, int wedge, int ring, int lengthx, int lengthy,
		int *dx, int *dy)
{
	double angle, radius;

	angle = (double) (2 * wedge + 1) * M_PI / w->mball.wedges;
	if (w->mball.rings % 2 && ring == w->mball.rings / 2)
		radius = ((double) 4.0 * ring + 1.0) / ((double) 4.0 * w->mball.rings);
	else
		radius = ((double) 2.0 * ring + 1.0) / ((double) 2.0 * w->mball.rings);
	*dx = lengthx / 2 + (int) ((double) lengthx * radius * cos(angle - M_PI / 2));
	*dy = lengthy / 2 + (int) ((double) lengthy * radius * sin(angle - M_PI / 2));
}

#if defined(WINVER) && (WINVER <= 0x030a)	/* if WINDOWS 3.1 or less */
#define FLOODFILL 1
#endif

#ifndef FLOODFILL
static void
OffsetSect(const MballWidget w, const int wedge, int *dx, int *dy)
{
	double angle = (double) (2 * wedge + 1) * M_PI / w->mball.wedges
			- M_PI / 2.0;

	*dx = (int) ((double) w->mball.dr * cos(angle));
	*dy = (int) ((double) w->mball.dr * sin(angle));
}
#endif

#ifdef WINVER
#ifdef FLOODFILL
/* Using a dangerous flood fill.  Probably faster then filling a sector by
   drawing many arcs of width 1 */
static void
SECTOR(MballWidget w, const GC pieceGC, const GC borderGC,
		const int letterx, const int lettery)
{
	w->mball.hBrush = CreateSolidBrush(pieceGC);
	w->mball.hOldBrush = (HBRUSH) SelectObject(w->core.hDC,
		w->mball.hBrush);
	(void) FloodFill(w->core.hDC, letterx, lettery, borderGC);
	(void) SelectObject(w->core.hDC, w->mball.hOldBrush);
	(void) DeleteObject(w->mball.hBrush);
}

#else
static void
SECTOR(MballWidget w, const GC pieceGC, const GC borderGC,
		const int xo, const int yo,
		const int width1, const int height1,
		const int width2, const int height2,
		const int angle1, const int angle2)
{
	LOGBRUSH lb;
	int d, r1 = MIN(width1, height1) / 2;
	int r2 = MIN(width2, height2) / 2;
	int t = MAX(r2 - r1 - 8, 1);

	if (r1 > r2) {
		d = r1;
		r1 = r2;
		r2 = d;
	} if (r1 < 0)
		r1 = -3;
	if (r1 <= 3) {
		d = MAX(2 * (r2 - 1), 3);
		DRAWPIE(w, pieceGC, xo - d / 2, yo - d / 2, d, d, angle1, angle2);
	} else {
		d = MAX(r1 + r2 + 2, 2);
		DRAWARC2(w, pieceGC, t, xo - d / 2, yo - d / 2, d, d, angle1, angle2);
		d = MAX(2 * (r1 + 3), 1);
		DRAWARC(w, borderGC, 1, xo - d / 2, yo - d / 2, d, d, angle1, angle2);
	}
	DRAWARC(w, borderGC, 1, xo - d / 2, yo - d / 2, d, d, angle1, angle2);
#if 0
	{
		double ang, x, y;
	if (r1 <= 0)
		r1 = 0;
	ang = RADIANS((double) angle1);
	x = cos(ang);
	y = -sin(ang);
	DRAWLINE(w, pieceGC,
		(int) ((double) r1 * x) + xo, (int) ((double) r1 * y) + yo,
		(int) ((double) r2 * x) + xo, (int) ((double) r2 * y) + yo);
	ang = RADIANS((double) angle1+angle2);
	x = cos(ang);
	y = -sin(ang);
	DRAWLINE(w, pieceGC,
		(int) ((double) r1 * x) + xo, (int) ((double) r1 * y) + yo,
		(int) ((double) r2 * x) + xo, (int) ((double) r2 * y) + yo);
	}
#endif
}
#endif

static void
DrawSect(const MballWidget w, const GC pieceGC, const GC borderGC,
		const int r, const int wedge,
		const int startx, const int starty,
		const int lengthx, const int lengthy)
{
	int dx, dy;

#ifdef FLOODFILL
	LetterPosition(w, wedge, r, lengthx, lengthy, &dx, &dy);
#ifdef DEBUG
	SetPixel(w->core.hDC, dx + startx, dy + starty, piece);
#else
	SECTOR(w, pieceGC, borderGC, dx + startx, dy + starty);
#endif
#else
	OffsetSect(w, wedge, &dx, &dy);
	if (w->mball.rings % 2) {
		if (r == w->mball.rings / 2)
			SECTOR(w, pieceGC, borderGC,
			startx + lengthx / 2 + dx, starty + lengthy / 2 + dy,
			  r * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
			  r * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
			       (r + 1) * 2 * lengthx / (w->mball.rings + 1) - 2 * w->mball.dr,
			       (r + 1) * 2 * lengthy / (w->mball.rings + 1) - 2 * w->mball.dr,
			       CIRCLE_4 - FULLCIRCLE * wedge / w->mball.wedges,
			       -FULLCIRCLE / w->mball.wedges);
		else
			SECTOR(w, pieceGC, borderGC,
			startx + lengthx / 2 + dx, starty + lengthy / 2 + dy,
			  r * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
			  r * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
			       (r + 1) * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
			       (r + 1) * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
			       CIRCLE_4 - FULLCIRCLE * wedge / w->mball.wedges,
			       -FULLCIRCLE / w->mball.wedges);
	} else
		SECTOR(w, pieceGC, borderGC,
		       startx + lengthx / 2 + dx, starty + lengthy / 2 + dy,
		       r * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
		       r * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
		    (r + 1) * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
		    (r + 1) * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
		       CIRCLE_4 - FULLCIRCLE * wedge / w->mball.wedges,
		       -FULLCIRCLE / w->mball.wedges);
#endif
}

static void
DrawSector(const MballWidget w,
		const int wedge, const int ring, const int offset)
{
	int startx, starty, lengthx, lengthy, i, l;
	GC pieceGC, borderGC;

	startx = 1 + w->mball.puzzleOffset.x;
	starty = 1 + w->mball.puzzleOffset.y;
	lengthx = w->mball.viewLength - w->mball.delta + w->mball.puzzleOffset.x;
	lengthy = w->mball.viewLength - w->mball.delta + w->mball.puzzleOffset.y;
	if (offset) {
		pieceGC = w->mball.borderGC;
#ifdef FLOODFILL
		borderGC = (w->mball.focus) ? w->mball.frameGC :
			w->mball.borderGC;
#else
		borderGC = w->mball.wedgeGC[w->mball.mballLoc[wedge][ring].wedge];
#endif
	} else {
		pieceGC = w->mball.wedgeGC[w->mball.mballLoc[wedge][ring].wedge];
#ifdef FLOODFILL
		borderGC = (w->mball.focus) ? w->mball.frameGC :
			w->mball.borderGC;
#else
		borderGC = w->mball.borderGC;
#endif
	}
	if (ring < (w->mball.rings + 1) / 2) {
		DrawSect(w, pieceGC, borderGC, ring, wedge,
			startx, starty, lengthx - startx, lengthy - starty);
	}
	if (ring + 1 > w->mball.rings / 2) {
		if (w->mball.vertical) {
			DrawSect(w, pieceGC, borderGC,
				 w->mball.rings - 1 - ring,
				 (3 * w->mball.wedges / 2 - 1 - wedge) % w->mball.wedges,
				 startx, lengthy + 3, lengthx - startx, lengthy - starty);
		} else {
			DrawSect(w, pieceGC, borderGC,
				 w->mball.rings - 1 - ring, w->mball.wedges - 1 - wedge,
				 lengthx + 3, starty, lengthx - startx, lengthy - starty);
		}
	}
	if (w->mball.mono) {
		int letterX, letterY;
		char buf[3];

		if (ring < (w->mball.rings + 1) / 2) {
			LetterPosition(w, wedge, ring, lengthx - startx, lengthy - starty,
				&letterX, &letterY);
			letterX += startx + 6 + w->mball.letterOffset.x;
			letterY += starty + w->mball.letterOffset.y;
			if (w->mball.orient && !w->mball.mballLoc[wedge][ring].direction) {
				int last;

				l = w->mball.mballLoc[wedge][ring].wedge;
				last = int2String(w, buf, l, w->mball.base, FALSE);
				buf[last] = w->mball.wedgeChar[l];
				buf[last + 1] = '\0';
				i = 0;
				if (l == 0)
					l = 1;
				l *= w->mball.base;
				while (l >= 1) {
					l /= w->mball.base;
					letterX += w->mball.letterOffset.x;
					i++;
				}
			} else {
				buf[0] = w->mball.wedgeChar[w->mball.mballLoc[wedge][ring].wedge];
				buf[1] = '\0';
				i = 1;
			}
			DRAWTEXT(w, borderGC, letterX, letterY, buf, i);
		}
		if (ring + 1 > w->mball.rings / 2) {
			if (w->mball.vertical) {
				LetterPosition(w,
					(3 * w->mball.wedges / 2 - 1 - wedge) % w->mball.wedges,
					w->mball.rings - 1 - ring,
					lengthx - startx, lengthy - starty,
					&letterX, &letterY);
				letterX += startx + 5 + w->mball.letterOffset.x;
				letterY += lengthy + 3 + w->mball.letterOffset.y;
			} else {
				LetterPosition(w,
					w->mball.wedges - 1 - wedge,
					w->mball.rings - 1 - ring,
					lengthx - startx, lengthy - starty,
					&letterX, &letterY);
				letterX += lengthx + 8 + w->mball.letterOffset.x;
				letterY += starty + w->mball.letterOffset.y;
			}
			if (w->mball.orient && w->mball.mballLoc[wedge][ring].direction) {
				int last;

				l = w->mball.mballLoc[wedge][ring].wedge;
				last = int2String(w, buf, l, w->mball.base, FALSE);
				buf[last] = w->mball.wedgeChar[l];
				buf[last + 1] = '\0';
				i = 0;
				if (l == 0)
					l = 1;
				l *= w->mball.base;
				while (l >= 1) {
					l /= w->mball.base;
					letterX += w->mball.letterOffset.x;
					i++;
				}
			} else {
				buf[0] = w->mball.wedgeChar[w->mball.mballLoc[wedge][ring].wedge];
				buf[1] = '\0';
				i = 1;
			}
			DRAWTEXT(w, borderGC, letterX, letterY, buf, i);
		}
	} else if (w->mball.orient) {
		int letterX, letterY;
		char buf[2];

		if (ring < (w->mball.rings + 1) / 2 &&
		    !w->mball.mballLoc[wedge][ring].direction) {
			LetterPosition(w, wedge, ring, lengthx - startx, lengthy - starty,
				&letterX, &letterY);
			letterX += startx + 5 + w->mball.letterOffset.x;
			letterY += starty + w->mball.letterOffset.y;
			l = w->mball.mballLoc[wedge][ring].wedge;
			(void) int2String(w, buf, l, w->mball.base, TRUE);
			i = 0;
			if (l == 0)
				l = 1;
			while (l >= 1) {
				l /= w->mball.base;
				letterX += w->mball.letterOffset.x;
				i++;
			}
			DRAWTEXT(w, borderGC, letterX, letterY, buf, i);
		}
		if (ring + 1 > w->mball.rings / 2 &&
		    w->mball.mballLoc[wedge][ring].direction) {
			if (w->mball.vertical) {
				LetterPosition(w,
					(3 * w->mball.wedges / 2 - 1 - wedge) % w->mball.wedges,
					w->mball.rings - 1 - ring,
					lengthx - startx, lengthy - starty,
					&letterX, &letterY);
				letterX += startx + 5 + w->mball.letterOffset.x;
				letterY += lengthy + 3 + w->mball.letterOffset.y;
			} else {
				LetterPosition(w,
					w->mball.wedges - 1 - wedge,
					w->mball.rings - 1 - ring,
					lengthx - startx, lengthy - starty,
					&letterX, &letterY);
				letterX += lengthx + 8 + w->mball.letterOffset.x;
				letterY += starty + w->mball.letterOffset.y;
			}
			l = w->mball.mballLoc[wedge][ring].wedge;
			(void) int2String(w, buf, l, w->mball.base, TRUE);
			i = 0;
			if (l == 0)
				l = 1;
			while (l >= 1) {
				l /= w->mball.base;
				letterX += w->mball.letterOffset.x;
				i++;
			}
			DRAWTEXT(w, borderGC, letterX, letterY, buf, i);
		}
	}
}

#else

static void
XFillSector(Display * display, Drawable drawable, GC gc, int xo, int yo,
		int width1, int height1, int width2, int height2,
		int angle1, int angle2)
{
	int d, r1 = MIN(width1, height1) / 2, r2 = MIN(width2, height2) / 2;
	int w = MAX(r2 - r1 - 8, 1);

	if (r1 > r2) {
		d = r1;
		r1 = r2;
		r2 = d;
	}
	if (r1 < 0)
		r1 = -3;
	d = MAX(r1 + r2 + 2, 2);
	XSetLineAttributes(display, gc, w, LineSolid, CapNotLast, JoinRound);
	XDrawArc(display, drawable, gc, xo - d / 2, yo - d / 2, d, d,
		 angle1, angle2);
	XSetLineAttributes(display, gc, 1, LineSolid, CapNotLast, JoinRound);
}

static void
XDrawSector(Display * display, Drawable drawable, GC gc,
		const int xo, const int yo,
		const int width1, const int height1,
		const int width2, const int height2,
		const int angle1, const int angle2)
{
	int d, r1 = MIN(width1, height1) / 2, r2 = MIN(width2, height2) / 2;

	/*double ang, x, y; */

	if (r1 > r2) {
		d = r1;
		r1 = r2;
		r2 = d;
	}
	if (r1 < 0)
		r1 = -3;
	d = MAX(2 * (r1 + 3), 1);
	XDrawArc(display, drawable, gc, xo - d / 2, yo - d / 2, d, d,
		 angle1, angle2);
	d = MAX(2 * (r2 - 1), 3);
	XDrawArc(display, drawable, gc, xo - d / 2, yo - d / 2, d, d,
		 angle1, angle2);
#if 0
	ang = RADIANS((double) angle1 / MULT);
	x = cos(ang);
	y = sin(ang);
	XDrawLine(display, drawable, gc,
	   (int) ((double) r1 * x) + xo, (int) ((double) r1 * y) + yo,
	   (int) ((double) r2 * x) + xo, (int) ((double) r2 * y) + yo);
	ang = RADIANS((double) angle2 / MULT);
	x = cos(ang);
	y = sin(ang);
	XDrawLine(display, drawable, gc,
	   (int) ((double) r1 * x) + xo, (int) ((double) r1 * y) + yo,
	   (int) ((double) r2 * x) + xo, (int) ((double) r2 * y) + yo);
#endif
}

#if 0
static void
XFillSector(Display *display, Drawable drawable, GC gc,
		const int xo, const int yo,
		const int width1, const int height1,
		const int width2, const int height2,
		const int angle1, const int angle2)
{
	int d, r1 = MIN(width1, height1) / 2, r2 = MIN(width2, height2) / 2;

	if (r1 > r2) {
		d = r1;
		r1 = r2;
		r2 = d;
	}
	if (r1 < 0)
		r1 = -3;
	for (d = 2 * (r1 + 3); d < 2 * (r2 - 1); d++)
		XDrawArc(display, drawable, gc, xo - d / 2, yo - d / 2, d, d,
			 angle1, angle2);
}
#endif

static void
DrawSect(const MballWidget w, GC wedgeGC, GC borderGC,
		const int r, const int wedge,
		const int startx, const int starty,
		const int lengthx, const int lengthy)
{
	int dx, dy;

	OffsetSect(w, wedge, &dx, &dy);
	if (w->mball.rings % 2) {
		if (r == w->mball.rings / 2) {
			XFillSector(XtDisplay(w), XtWindow(w), wedgeGC,
			  startx + lengthx / 2 + dx, starty + lengthy / 2 + dy,
			  r * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
			  r * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
				    (r + 1) * 2 * lengthx / (w->mball.rings + 1) - 2 * w->mball.dr,
				    (r + 1) * 2 * lengthy / (w->mball.rings + 1) - 2 * w->mball.dr,
				 CIRCLE_4 - FULLCIRCLE * wedge / w->mball.wedges,
				    -FULLCIRCLE / w->mball.wedges);
			XDrawSector(XtDisplay(w), XtWindow(w), borderGC,
			startx + lengthx / 2 + dx, starty + lengthy / 2 + dy,
			  r * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
			  r * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
				    (r + 1) * 2 * lengthx / (w->mball.rings + 1) - 2 * w->mball.dr,
				    (r + 1) * 2 * lengthy / (w->mball.rings + 1) - 2 * w->mball.dr,
				 CIRCLE_4 - FULLCIRCLE * wedge / w->mball.wedges,
				    -FULLCIRCLE / w->mball.wedges);
		} else {
			XFillSector(XtDisplay(w), XtWindow(w), wedgeGC,
			startx + lengthx / 2 + dx, starty + lengthy / 2 + dy,
			  r * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
			  r * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
				    (r + 1) * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
				    (r + 1) * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
				 CIRCLE_4 - FULLCIRCLE * wedge / w->mball.wedges,
				    -FULLCIRCLE / w->mball.wedges);
			XDrawSector(XtDisplay(w), XtWindow(w), borderGC,
			startx + lengthx / 2 + dx, starty + lengthy / 2 + dy,
			  r * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
			  r * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
				    (r + 1) * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
				    (r + 1) * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
				 CIRCLE_4 - FULLCIRCLE * wedge / w->mball.wedges,
				    -FULLCIRCLE / w->mball.wedges);
		}
	} else {
		XFillSector(XtDisplay(w), XtWindow(w), wedgeGC,
			startx + lengthx / 2 + dx, starty + lengthy / 2 + dy,
			  r * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
			  r * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
		    (r + 1) * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
		    (r + 1) * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
			    CIRCLE_4 - FULLCIRCLE * wedge / w->mball.wedges,
			    -FULLCIRCLE / w->mball.wedges);
		XDrawSector(XtDisplay(w), XtWindow(w), borderGC,
			startx + lengthx / 2 + dx, starty + lengthy / 2 + dy,
			  r * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
			  r * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
		    (r + 1) * 2 * lengthx / w->mball.rings - 2 * w->mball.dr,
		    (r + 1) * 2 * lengthy / w->mball.rings - 2 * w->mball.dr,
			    CIRCLE_4 - FULLCIRCLE * wedge / w->mball.wedges,
			    -FULLCIRCLE / w->mball.wedges);
	}
}

static void
DrawSector(const MballWidget w, const int wedge, const int ring,
		const int offset)
{
	GC wedgeGC, borderGC;
	int startx, starty, lengthx, lengthy, i, l;

	startx = 1 + w->mball.puzzleOffset.x;
	starty = 1 + w->mball.puzzleOffset.y;
	lengthx = w->mball.viewLength - w->mball.delta + w->mball.puzzleOffset.x;
	lengthy = w->mball.viewLength - w->mball.delta + w->mball.puzzleOffset.y;
	if (offset) {
		borderGC = w->mball.wedgeGC[w->mball.mballLoc[wedge][ring].wedge];
		if (w->mball.mono) {
			wedgeGC = w->mball.inverseGC;
		} else {
			wedgeGC = w->mball.borderGC;
		}
	} else {
		wedgeGC = w->mball.wedgeGC[w->mball.mballLoc[wedge][ring].wedge];
		borderGC = w->mball.borderGC;
	}
	/* if the number of rings is odd
	 * the ring can straddle both hemispheres */
	if (ring < (w->mball.rings + 1) / 2)
		DrawSect(w, wedgeGC, borderGC, ring, wedge,
			 startx, starty, lengthx - startx, lengthy - starty);
	if (ring + 1 > w->mball.rings / 2) {
		if (w->mball.vertical)
			DrawSect(w, wedgeGC, borderGC,
				 w->mball.rings - 1 - ring,
				 (3 * w->mball.wedges / 2 - 1 - wedge) % w->mball.wedges,
				 startx, lengthy + 3, lengthx - startx, lengthy - starty);
		else
			DrawSect(w, wedgeGC, borderGC,
				 w->mball.rings - 1 - ring,
				 w->mball.wedges - 1 - wedge,
				 lengthx + 3, starty, lengthx - startx, lengthy - starty);
	}
	if (w->mball.mono) {
		int letterX, letterY;
		char buf[66];

		if (offset) {
			borderGC = w->mball.borderGC;
		} else {
			borderGC = w->mball.inverseGC;
		}
		if (ring < (w->mball.rings + 1) / 2) {
			LetterPosition(w, wedge, ring, lengthx - startx, lengthy - starty,
				&letterX, &letterY);
			letterX += startx + w->mball.letterOffset.x;
			letterY += starty + w->mball.letterOffset.y;
			if (w->mball.orient && !w->mball.mballLoc[wedge][ring].direction) {
				int last;

				l = w->mball.mballLoc[wedge][ring].wedge;
				last = int2String(w, buf, l, w->mball.base, False);
				buf[last] = w->mball.wedgeName[l][0];
				buf[last + 1] = '\0';
				i = 0;
				if (l == 0)
					l = 1;
				l *= w->mball.base;
				while (l >= 1) {
					l /= w->mball.base;
					letterX += w->mball.letterOffset.x;
					i++;
				}
			} else {
				buf[0] = w->mball.wedgeName[w->mball.mballLoc[wedge][ring].wedge][0];
				buf[1] = '\0';
				i = 1;
			}
			DRAWTEXT(w, borderGC, letterX, letterY, buf, i);
		}
		if (ring + 1 > w->mball.rings / 2) {
			if (w->mball.vertical) {
				LetterPosition(w,
					(3 * w->mball.wedges / 2 - 1 - wedge) % w->mball.wedges,
					w->mball.rings - 1 - ring,
					lengthx - startx, lengthy - starty,
					&letterX, &letterY);
				letterX += startx + w->mball.letterOffset.x;
				letterY += lengthy + 3 + w->mball.letterOffset.y;
			} else {
				LetterPosition(w,
					w->mball.wedges - 1 - wedge,
					w->mball.rings - 1 - ring,
					lengthx - startx, lengthy - starty,
					&letterX, &letterY);
				letterX += lengthx + 3 + w->mball.letterOffset.x;
				letterY += starty + w->mball.letterOffset.y;
			}
			if (w->mball.orient && w->mball.mballLoc[wedge][ring].direction) {
				int last;

				l = w->mball.mballLoc[wedge][ring].wedge;
				last = int2String(w, buf, l, w->mball.base, False);
				buf[last] = w->mball.wedgeName[l][0];
				buf[last + 1] = '\0';
				i = 0;
				if (l == 0)
					l = 1;
				l *= w->mball.base;
				while (l >= 1) {
					l /= w->mball.base;
					letterX += w->mball.letterOffset.x;
					i++;
				}
			} else {
				buf[0] = w->mball.wedgeName[w->mball.mballLoc[wedge][ring].wedge][0];
				buf[1] = '\0';
				i = 1;
			}
			DRAWTEXT(w, borderGC, letterX, letterY, buf, i);
		}
	} else if (w->mball.orient) {
		int letterX, letterY;
		char buf[65];

		if (ring < (w->mball.rings + 1) / 2 &&
		    !w->mball.mballLoc[wedge][ring].direction) {
			LetterPosition(w, wedge, ring, lengthx - startx, lengthy - starty,
				&letterX, &letterY);
			letterX += startx + w->mball.letterOffset.x;
			letterY += starty + w->mball.letterOffset.y;
			l = w->mball.mballLoc[wedge][ring].wedge;
			(void) int2String(w, buf, l, w->mball.base, True);
			i = 0;
			if (l == 0)
				l = 1;
			while (l >= 1) {
				l /= w->mball.base;
				letterX += w->mball.letterOffset.x;
				i++;
			}
			DRAWTEXT(w, borderGC, letterX, letterY, buf, i);
		}
		if (ring + 1 > w->mball.rings / 2 &&
		    w->mball.mballLoc[wedge][ring].direction) {
			if (w->mball.vertical) {
				LetterPosition(w,
					(3 * w->mball.wedges / 2 - 1 - wedge) % w->mball.wedges,
					w->mball.rings - 1 - ring,
					lengthx - startx, lengthy - starty,
					&letterX, &letterY);
				letterX += startx + w->mball.letterOffset.x;
				letterY += lengthy + 3 + w->mball.letterOffset.y;
			} else {
				LetterPosition(w,
					w->mball.wedges - 1 - wedge,
					w->mball.rings - 1 - ring,
					lengthx - startx, lengthy - starty,
					&letterX, &letterY);
				letterX += lengthx + 3 + w->mball.letterOffset.x;
				letterY += starty + w->mball.letterOffset.y;
			}
			l = w->mball.mballLoc[wedge][ring].wedge;
			(void) int2String(w, buf, l, w->mball.base, True);
			i = 0;
			if (l == 0)
				l = 1;
			while (l >= 1) {
				l /= w->mball.base;
				letterX += w->mball.letterOffset.x;
				i++;
			}
			DRAWTEXT(w, borderGC, letterX, letterY, buf, i);
		}
	}
}
#endif

static void
DrawWedge(MballWidget w, int wedge)
{
	int ring;

	for (ring = 0; ring < w->mball.rings; ring++)
		DrawSector(w, wedge, ring, FALSE);
}

void
DrawAllWedges(MballWidget w)
{
	int wedge;

	for (wedge = 0; wedge < w->mball.wedges; wedge++)
		DrawWedge(w, wedge);
}

static void
DrawRadar(const MballWidget w, GC gc, const int startx, const int starty,
		const int lengthx, const int lengthy)
{
	int r, i, stx, sty, lenx, leny;
	double angle, increment;

	DRAWARC(w, gc, 1, startx, starty,
		 lengthx, lengthy, 0, FULLCIRCLE);
	if (w->mball.rings % 2) {
		stx = startx - lengthx / (2 * w->mball.rings);
		sty = starty - lengthy / (2 * w->mball.rings);
		for (r = 1; r < w->mball.rings / 2 + 1; r++) {
			DRAWARC(w, gc, 1,
				 stx + (w->mball.rings / 2 + 1 - r) * lengthx /
				 w->mball.rings,
				 sty + (w->mball.rings / 2 + 1 - r) * lengthy /
				 w->mball.rings,
				 r * 2 * lengthx / w->mball.rings,
				 r * 2 * lengthy / w->mball.rings,
				 0, FULLCIRCLE);
		}
	} else {
		for (r = 1; r < w->mball.rings / 2; r++) {
			DRAWARC(w, gc, 1,
				 startx + (w->mball.rings / 2 - r) * lengthx /
				 w->mball.rings,
				 starty + (w->mball.rings / 2 - r) * lengthy /
				 w->mball.rings,
				 r * 2 * lengthx / w->mball.rings,
				 r * 2 * lengthy / w->mball.rings,
				 0, FULLCIRCLE);
		}
	}
	increment = RADIANS(NUM_DEGREES) / (double) w->mball.wedges;
	angle = RADIANS(RT_ANG);
	stx = startx + lengthx / 2;
	sty = starty + lengthy / 2;
	lenx = lengthx;
	leny = lengthy;
#ifdef WINVER
	lenx += 2;
	leny += 2;
#endif
	for (i = 0; i < w->mball.wedges; i++) {
		DRAWLINE(w, gc,
			stx, sty,
			stx + (int) ((double) (lenx) * cos(angle) / 2.0),
			sty + (int) ((double) (leny) * sin(angle) / 2.0));
		angle += increment;
	}
}

static void
EraseFrame(const MballWidget w)
{
	FILLRECTANGLE(w, w->mball.inverseGC,
		0, 0, w->core.width, w->core.height);
}

static void
DrawFrame(MballWidget w, Boolean focus)
{
	int startx, starty, lengthx, lengthy;
	GC gc = (focus) ? w->mball.frameGC : w->mball.borderGC;

	startx = 1 + w->mball.puzzleOffset.x;
	starty = 1 + w->mball.puzzleOffset.y;
	lengthx = w->mball.viewLength - w->mball.delta + w->mball.puzzleOffset.x;
	lengthy = w->mball.viewLength - w->mball.delta + w->mball.puzzleOffset.y;
	DrawRadar(w, gc, startx, starty, lengthx - startx, lengthy - starty);
	if (w->mball.vertical) {
		DRAWLINE(w, w->mball.frameGC, 0, lengthy + 1,
			(int) w->core.width, lengthy + 1);
		DrawRadar(w, gc, startx, lengthy + 3, lengthx - startx, lengthy - starty);
	} else {
		DRAWLINE(w, w->mball.frameGC, lengthx + 1, 0,
			lengthx + 1, (int) w->core.height);
		DrawRadar(w, gc, lengthx + 3, starty, lengthx - startx, lengthy - starty);
	}
}

static void
MoveNoWedges(MballWidget w)
{
	SetMball(w, MBALL_ILLEGAL);
}

static void
SwapWedges(MballWidget w, int wedge1, int wedge2)
{
	MballLoc temp;
	int ring;

	if (wedge1 == wedge2) {
		for (ring = 0; ring < w->mball.rings / 2; ring++) {
			temp = w->mball.mballLoc[wedge1][ring];
			w->mball.mballLoc[wedge1][ring] =
				w->mball.mballLoc[wedge1][w->mball.rings - 1 - ring];
			w->mball.mballLoc[wedge1][w->mball.rings - 1 - ring] = temp;
		}
		for (ring = 0; ring < w->mball.rings; ring++)
			w->mball.mballLoc[wedge1][ring].direction =
				!w->mball.mballLoc[wedge1][ring].direction;
		DrawWedge(w, wedge1);
	} else {
		for (ring = 0; ring < w->mball.rings; ring++) {
			temp = w->mball.mballLoc[wedge1][ring];
			w->mball.mballLoc[wedge1][ring] =
				w->mball.mballLoc[wedge2][w->mball.rings - 1 - ring];
			w->mball.mballLoc[wedge2][w->mball.rings - 1 - ring] = temp;
			w->mball.mballLoc[wedge1][ring].direction =
				!w->mball.mballLoc[wedge1][ring].direction;
			w->mball.mballLoc[wedge2][w->mball.rings - 1 - ring].direction =
				!w->mball.mballLoc[wedge2][w->mball.rings - 1 - ring].direction;
		}
		DrawWedge(w, wedge1);
		DrawWedge(w, wedge2);
	}
}

static void
MoveWedges(MballWidget w, int wedge, int ring, int direction)
{
	int i;

	if (direction == CW || direction == CCW) {	/* rotate ring */
		int newI;
		MballLoc temp1, temp2;

		for (i = 0; i < w->mball.wedges; i++) {
			newI = (direction == CW) ? i : w->mball.wedges - 1 - i;
			if (newI == ((direction == CW) ? 0 : w->mball.wedges - 1)) {
				temp1 = w->mball.mballLoc[newI][ring];
				w->mball.mballLoc[newI][ring] = w->mball.mballLoc
					[((direction == CW) ? w->mball.wedges - 1 : 0)][ring];
			} else {
				temp2 = temp1;
				temp1 = w->mball.mballLoc[newI][ring];
				w->mball.mballLoc[newI][ring] = temp2;
			}
			DrawSector(w, newI, ring, FALSE);
		}
	} else {		/* flip */
		int sphereDir = mapDirToWedge[(w->mball.wedges - MINWEDGES) / 2][direction];
		int offset = w->mball.wedges / 2;
		int wedge1, wedge2;

		for (i = 0; i < w->mball.wedges / 2; i++)
			if (wedge == i + sphereDir)
				offset = 0;
		for (i = 0; i < (w->mball.wedges + 2) / 4; i++) {
			wedge1 = (i + sphereDir + offset) % w->mball.wedges;
			wedge2 = (w->mball.wedges / 2 - 1 - i + sphereDir + offset) %
				w->mball.wedges;
			SwapWedges(w, wedge1, wedge2);
		}
	}
}

static void
MoveControlCb(MballWidget w, int wedge, int direction)
{
	int ring;

	if (direction > COORD)
		for (ring = 0; ring < w->mball.rings; ring++) {
			MoveWedges(w, wedge, ring, direction);
			SetMball(w, MBALL_CONTROL);
	} else {
		MoveWedges(w, 0, 0, direction);
		SetMball(w, MBALL_CONTROL);
		MoveWedges(w, w->mball.wedges / 2, 0, direction);
		SetMball(w, MBALL_CONTROL);
	}
}

void
MoveMball(MballWidget w, const int wedge, const int ring, const int direction,
		const int control)
{
	if (control)
		MoveControlCb(w, wedge, direction);
	else {
		MoveWedges(w, wedge, ring, direction);
		SetMball(w, MBALL_MOVED);
	}
	PutMove(wedge, ring, direction, control);
}

static Boolean
SelectWedges(MballWidget w, int x, int y, int *wedge, int *ring, int *view)
{
	double angle, radius;

	x -= w->mball.puzzleOffset.x;
	y -= w->mball.puzzleOffset.y;
	if (w->mball.vertical && y > w->mball.viewLength - 1) {
		y -= (w->mball.viewLength - 1);
		*view = DOWN;
	} else if (!w->mball.vertical && x > w->mball.viewLength - 1) {
		x -= (w->mball.viewLength - 1);
		*view = DOWN;
	} else
		*view = UP;
	x -= (w->mball.wedgeLength + 1) / 2;
	y -= (w->mball.wedgeLength + 1) / 2;
	radius = sqrt((double) x * x + y * y);
	if (y >= 0)
		angle = atan2((double) -x, (double) y) + M_PI;
	else if (x < 0)
		angle = 2 * M_PI - atan2((double) -x, (double) -y);
	else
		angle = -atan2((double) -x, (double) -y);
	*ring = (int) (radius * (double) w->mball.rings /
		((double) w->mball.wedgeLength));
	*wedge = (int) (angle * (double) w->mball.wedges / (2.0 * M_PI));
	if (*view == DOWN) {
		if (w->mball.vertical)
			*wedge = (3 * w->mball.wedges / 2 - 1 - *wedge) % w->mball.wedges;
		else
			*wedge = (w->mball.wedges - 1 - *wedge) % w->mball.wedges;
		*ring = w->mball.rings - 1 - *ring;
	}
	if (radius > w->mball.wedgeLength / 2 + w->mball.delta)
		return False;
	return True;
}

static Boolean
PositionWedges(MballWidget w, int x, int y, int *wedge, int *ring, int *direction)
{
	int view, inside;

	inside = SelectWedges(w, x, y, wedge, ring, &view);
	if ((*direction == CW || *direction == CCW) && !inside)
		return False;
	if (view == DOWN) {
		if (*direction == CCW)
			*direction = CW;
		else if (*direction == CW)
			*direction = CCW;
		else if (*direction < COORD)
			*direction = (COORD - *direction) % COORD;
	}
	if (*direction != CW && *direction != CCW &&
	    mapDirToWedge[(w->mball.wedges - MINWEDGES) / 2][*direction] ==
	      CUTS) {
		return False;
	}
	return True;
}

#ifndef WINVER
static
#endif
void
MoveMballInput(MballWidget w, int x, int y, int direction, int control)
{
	int wedge, ring;

	if (CheckSolved(w) && !w->mball.practice && !control) {
		MoveNoWedges(w);
		return;
	}
	if (!PositionWedges(w, x, y, &wedge, &ring, &direction))
		return;
	control = (control) ? 1 : 0;
	MoveMball(w, wedge, ring, direction, control);
	if (!control && CheckSolved(w)) {
		SetMball(w, MBALL_SOLVED);
	}
}

static void
ResetWedges(MballWidget w)
{
	int wedge, ring;

	for (wedge = 0; wedge < MAXWEDGES; wedge++) {
		if (w->mball.mballLoc[wedge])
			(void) free((void *) w->mball.mballLoc[wedge]);
		if (!(w->mball.mballLoc[wedge] = (MballLoc *)
		      malloc(sizeof (MballLoc) * w->mball.rings))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		if (startLoc[wedge])
			(void) free((void *) startLoc[wedge]);
		if (!(startLoc[wedge] = (MballLoc *)
		      malloc(sizeof (MballLoc) * w->mball.rings))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (wedge = 0; wedge < w->mball.wedges; wedge++)
		for (ring = 0; ring < w->mball.rings; ring++) {
			w->mball.mballLoc[wedge][ring].wedge = wedge;
			w->mball.mballLoc[wedge][ring].direction = DOWN;
		}
	FlushMoves(w);
	w->mball.currentWedge = -1;
	w->mball.started = False;
}

static void
ResizeWedges(MballWidget w)
{
	w->mball.mballLength = w->mball.wedgeLength / (2 * w->mball.wedges) -
		w->mball.delta - 1;
	w->mball.letterOffset.x = -2;
	w->mball.letterOffset.y = 4;
	w->mball.dr = w->mball.wedges;
}

static void
GetWedges(MballWidget w)
{
	FILE *fp;
	int c, i, wedge, ring, orient, practice, moves;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "r")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	FlushMoves(w);
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &wedge);
	if (wedge >= MINWEDGES && wedge <= MAXWEDGES && !(wedge % 2)) {
		if (w->mball.wedges != wedge) {
			SetMball(w, (wedge - MINWEDGES) / 2 + MBALL_WEDGE2);
		}
	} else {
		stringCat(&buf1, name, " corrupted: wedge ");
		intCat(&buf2, buf1, wedge);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MINWEDGES);
		free(buf1);
		stringCat(&buf1, buf2, " and ");
		free(buf2);
		intCat(&buf2, buf1, MAXWEDGES);
		free(buf1);
			DISPLAY_WARNING(buf2);
		free(buf2);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &ring);
	if (ring >= MINRINGS) {
		for (i = w->mball.rings; i < ring; i++) {
			SetMball(w, MBALL_INC);
		}
		for (i = w->mball.rings; i > ring; i--) {
			SetMball(w, MBALL_DEC);
		}
	} else {
		stringCat(&buf1, name, " corrupted: ring ");
		intCat(&buf2, buf1, ring);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MINRINGS);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &orient);
	if (w->mball.orient != (Boolean) orient) {
		SetMball(w, MBALL_ORIENT);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &practice);
	if (w->mball.practice != (Boolean) practice) {
		SetMball(w, MBALL_PRACTICE);
	}
#ifdef WINVER
	ResetWedges(w);
#endif
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &moves);
	ScanStartPosition(fp, w);
	SetMball(w, MBALL_RESTORE);
	SetStartPosition(w);
	ScanMoves(fp, w, moves);
	(void) fclose(fp);
	(void) printf("%s: wedge %d, ring %d, orient %d, ",
		name, wedge, ring, orient);
	(void) printf("practice %d, moves %d\n", practice, moves);
	free(lname);
	free(fname);
	w->mball.cheat = True; /* Assume the worst. */
}

static void
WriteWedges(MballWidget w)
{
	FILE *fp;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "w")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	(void) fprintf(fp, "wedge%c %d\n", SYMBOL, w->mball.wedges);
	(void) fprintf(fp, "ring%c %d\n", SYMBOL, w->mball.rings);
	(void) fprintf(fp, "orient%c %d\n", SYMBOL, (w->mball.orient) ? 1 : 0);
	(void) fprintf(fp, "practice%c %d\n", SYMBOL,
		(w->mball.practice) ? 1 : 0);
	(void) fprintf(fp, "moves%c %d\n", SYMBOL, NumMoves());
	PrintStartPosition(fp, w);
	PrintMoves(fp);
	(void) fclose(fp);
	(void) printf("Saved to %s.\n", name);
	free(lname);
	free(fname);
}

static void
ClearWedges(MballWidget w)
{
	if (w->mball.currentWedge >= 0)
		return;
	ResetWedges(w);
	DrawAllWedges(w);
	SetMball(w, MBALL_RESET);
}

static void
UndoWedges(MballWidget w)
{
	if (MadeMoves() && w->mball.currentWedge < 0) {
		int wedge, ring, direction, control;

		GetMove(&wedge, &ring, &direction, &control);
		direction = (direction < COORD) ? direction : 3 * COORD - direction;
		if (control)
			MoveControlCb(w, wedge, direction);
		else {
			MoveWedges(w, wedge, ring, direction);
			SetMball(w, MBALL_UNDO);
		}
	}
}

static void
PracticeWedges(MballWidget w)
{
	SetMball(w, MBALL_PRACTICE);
}

static void
RandomizeWedges(MballWidget w)
{
	int randomDirection, wedge, ring;
	int big = w->mball.wedges * (w->mball.rings + 1) + NRAND(2);

	if (w->mball.currentWedge >= 0)
		return;
	if (w->mball.practice)
		PracticeWedges(w);
	SetMball(w, MBALL_RESET);
	w->mball.cheat = False;
	if (big > 100)
		big = 100;
#ifdef DEBUG
	big = 3;
#endif
	while (big--) {
		wedge = NRAND(w->mball.wedges);
		ring = NRAND(w->mball.rings);
		do
			randomDirection = NRAND(2 * COORD);
		while (randomDirection < COORD &&
		       mapDirToWedge[(w->mball.wedges - MINWEDGES) / 2][randomDirection] ==
		       CUTS);
		if (randomDirection >= COORD) {
			if (randomDirection - COORD < CUTS)
				randomDirection = CW;
			else
				randomDirection = CCW;
		}
		MoveMball(w, wedge, ring, randomDirection, FALSE);
	}
	FlushMoves(w);
	SetMball(w, MBALL_RANDOMIZE);
	if (CheckSolved(w)) {
		SetMball(w, MBALL_SOLVED);
	}
}

static void
SolveWedges(MballWidget w)
{
	if (CheckSolved(w) || w->mball.currentWedge >= 0)
		return;
	{
		SetMball(w, MBALL_SOLVE_MESSAGE);
	}
}

static void
IncrementWedges(MballWidget w)
{
	SetMball(w, MBALL_INC);
}

static Boolean
DecrementWedges(MballWidget w)
{
	if (w->mball.rings <= MINRINGS)
		return False;
	SetMball(w, MBALL_DEC);
	return True;
}

static void
OrientizeWedges(MballWidget w)
{
	SetMball(w, MBALL_ORIENT);
}

#ifdef WINVER
static void
SetValuesMball(MballWidget w)
{
	struct tagColor {
		int red, green, blue;
	} color;
	char szBuf[80], buf[20], charbuf[2];
	int wedge;

	w->mball.wedges = GetPrivateProfileInt(SECTION, "wedges",
		DEFAULTWEDGES, INIFILE);
	w->mball.rings = GetPrivateProfileInt(SECTION, "rings",
		DEFAULTRINGS, INIFILE);
	w->mball.orient = (BOOL) GetPrivateProfileInt(SECTION, "orient",
		DEFAULTORIENT, INIFILE);
	w->mball.practice = (BOOL) GetPrivateProfileInt(SECTION, "practice",
		DEFAULTPRACTICE, INIFILE);
	w->mball.base = GetPrivateProfileInt(SECTION, "base",
		DEFAULTBASE, INIFILE);
	w->mball.mono = (BOOL) GetPrivateProfileInt(SECTION, "mono",
		DEFAULTMONO, INIFILE);
	w->mball.reverse = (BOOL) GetPrivateProfileInt(SECTION, "reverse",
		DEFAULTREVERSE, INIFILE);
	/* cyan */
	(void) GetPrivateProfileString(SECTION, "frameColor", "0 255 254",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->mball.frameGC = RGB(color.red, color.green, color.blue);
	/* gray25 */
	(void) GetPrivateProfileString(SECTION, "pieceBorder", "64 64 64",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->mball.borderGC = RGB(color.red, color.green, color.blue);
	/* #AEB2C3 */
	(void) GetPrivateProfileString(SECTION, "background", "174 178 195",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->mball.inverseGC = RGB(color.red, color.green, color.blue);
	for (wedge = 0; wedge < MAXWEDGES; wedge++) {
		(void) sprintf(buf, "wedgeColor%d", wedge);
		(void) GetPrivateProfileString(SECTION,
			buf, wedgeColorString[wedge],
			szBuf, sizeof (szBuf), INIFILE);
		(void) sscanf(szBuf, "%d %d %d",
			&(color.red), &(color.green), &(color.blue));
		w->mball.wedgeGC[wedge] =
			RGB(color.red, color.green, color.blue);
		(void) sprintf(buf, "wedgeChar%d", wedge);
		charbuf[0] = wedgeColorChar[wedge];
		charbuf[1] = '\0';
		(void) GetPrivateProfileString(SECTION, buf, charbuf,
			szBuf, sizeof (szBuf), INIFILE);
		w->mball.wedgeChar[wedge] = szBuf[0];
	}
	(void) GetPrivateProfileString(SECTION, "name", "Guest",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->mball.username, szBuf);
	w->mball.username[80] = 0;
}

void
DestroyMball(HBRUSH brush)
{
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

#else
static void
GetColor(MballWidget w, int wedge)
{
	XGCValues values;
	XtGCMask valueMask;
	XColor colorCell, rgb;

	valueMask = GCForeground | GCBackground;
	if (w->mball.reverse) {
		values.background = w->mball.foreground;
	} else {
		values.background = w->mball.background;
	}
	if (!w->mball.mono) {
		if (XAllocNamedColor(XtDisplay(w),
				DefaultColormapOfScreen(XtScreen(w)),
			    w->mball.wedgeName[wedge], &colorCell, &rgb)) {
			values.foreground = w->mball.wedgeColor[wedge] = colorCell.pixel;
			if (w->mball.wedgeGC[wedge])
				XtReleaseGC((Widget) w, w->mball.wedgeGC[wedge]);
			w->mball.wedgeGC[wedge] = XtGetGC((Widget) w, valueMask, &values);
			if (w->mball.fontInfo)
				XSetFont(XtDisplay(w), w->mball.wedgeGC[wedge],
					w->mball.fontInfo->fid);
			return;
		} else {
			char *buf1, *buf2;

			stringCat(&buf1, "Color name \"",
				w->mball.wedgeName[wedge]);
			stringCat(&buf2, buf1, "\" is not defined for wedge ");
			free(buf1);
			intCat(&buf1, buf2, wedge);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf2);
		}
	}
	if (w->mball.reverse) {
		values.background = w->mball.foreground;
		values.foreground = w->mball.background;
	} else {
		values.background = w->mball.background;
		values.foreground = w->mball.foreground;
	}
	if (w->mball.wedgeGC[wedge])
		XtReleaseGC((Widget) w, w->mball.wedgeGC[wedge]);
	w->mball.wedgeGC[wedge] = XtGetGC((Widget) w, valueMask, &values);
	if (w->mball.fontInfo)
		XSetFont(XtDisplay(w), w->mball.wedgeGC[wedge],
			w->mball.fontInfo->fid);
}

static void
SetAllColors(MballWidget w)
{
	XGCValues values;
	XtGCMask valueMask;
	int wedge;

	valueMask = GCForeground | GCBackground;

	if (w->mball.reverse) {
		values.background = w->mball.background;
		values.foreground = w->mball.foreground;
	} else {
		values.foreground = w->mball.background;
		values.background = w->mball.foreground;
	}
	if (w->mball.inverseGC)
		XtReleaseGC((Widget) w, w->mball.inverseGC);
	w->mball.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->mball.mono) {
		if (w->mball.reverse) {
			values.background = w->mball.foreground;
			values.foreground = w->mball.background;
		} else {
			values.foreground = w->mball.foreground;
			values.background = w->mball.background;
		}
	} else {
		values.foreground = w->mball.frameColor;
		values.background = w->mball.background;
	}
	if (w->mball.frameGC)
		XtReleaseGC((Widget) w, w->mball.frameGC);
	w->mball.frameGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->mball.mono) {
		if (w->mball.reverse) {
			values.background = w->mball.foreground;
			values.foreground = w->mball.background;
		} else {
			values.foreground = w->mball.foreground;
			values.background = w->mball.background;
		}
	} else {
		values.foreground = w->mball.borderColor;
		values.background = w->mball.background;
	}
	if (w->mball.borderGC)
		XtReleaseGC((Widget) w, w->mball.borderGC);
	w->mball.borderGC = XtGetGC((Widget) w, valueMask, &values);
	for (wedge = 0; wedge < MAXWEDGES; wedge++)
		GetColor(w, wedge);
	if (w->mball.fontInfo)
		XSetFont(XtDisplay(w), w->mball.borderGC,
			w->mball.fontInfo->fid);
}

static Boolean
SetValuesMball(Widget current, Widget request, Widget renew)
{
	MballWidget c = (MballWidget) current, w = (MballWidget) renew;
	Boolean redraw = False, setColors = False;
	int wedge;

	CheckWedges(w);
	for (wedge = 0; wedge < MAXWEDGES; wedge++) {
		if (strcmp(w->mball.wedgeName[wedge], c->mball.wedgeName[wedge])) {
			setColors = True;
			break;
		}
	}
	if (w->mball.font != c->mball.font ||
			w->mball.borderColor != c->mball.borderColor ||
			w->mball.reverse != c->mball.reverse ||
			w->mball.mono != c->mball.mono ||
			setColors) {
		loadFont(w);
		SetAllColors(w);
		redraw = True;
	} else if (w->mball.background != c->mball.background ||
			w->mball.foreground != c->mball.foreground) {
		SetAllColors(w);
		redraw = True;
	}
	if (w->mball.orient != c->mball.orient ||
			w->mball.base != c->mball.base ||
			w->mball.practice != c->mball.practice) {
		ResetWedges(w);
		redraw = True;
	}
	if (w->mball.wedges != c->mball.wedges ||
			w->mball.rings != c->mball.rings) {
		SizeMball(w);
		redraw = True;
	}
	if (w->mball.mballLength != c->mball.mballLength) {
		ResizeMball(w);
		redraw = True;
	}
	if (w->mball.menu != -1) {
		int menu = w->mball.menu;

		w->mball.menu = -1;
		switch (menu) {
		case 0:
			GetWedges(w);
			break;
		case 1:
			WriteWedges(w);
			break;
		case 3:
			ClearWedges(w);
			break;
		case 4:
			UndoWedges(w);
			break;
		case 5:
			RandomizeWedges(w);
			break;
		case 6:
			SolveWedges(w);
			break;
		case 7:
			IncrementWedges(w);
			break;
		case 8:
			(void) DecrementWedges(w);
			break;
		case 9:
			OrientizeWedges(w);
			break;
		case 10:
			PracticeWedges(w);
			break;
		default:
			break;
		}
	}
	return (redraw);
}

static void
QuitMball(MballWidget w, XEvent * event, char **args, int nArgs)
{
	Display *display = XtDisplay(w);

	if (w->mball.fontInfo) {
		XUnloadFont(display, w->mball.fontInfo->fid);
		XFreeFont(display, w->mball.fontInfo);
	}
	XtCloseDisplay(display);
	exit(0);
}

static void
DestroyMball(Widget old)
{
	MballWidget w = (MballWidget) old;
	int wedge;

	for (wedge = 0; wedge < MAXWEDGES; wedge++)
		XtReleaseGC(old, w->mball.wedgeGC[wedge]);
	XtReleaseGC(old, w->mball.borderGC);
	XtReleaseGC(old, w->mball.frameGC);
	XtReleaseGC(old, w->mball.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->mball.select);
}
#endif

#ifndef WINVER
static
#endif
void
ResizeMball(MballWidget w)
{
	int tempLength;
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif

	w->mball.delta = 4;
	w->mball.vertical = (w->core.height >= w->core.width);
	if (w->mball.vertical)
		tempLength = MIN(w->core.height / 2, w->core.width);
	else
		tempLength = MIN(w->core.height, w->core.width / 2);
	w->mball.mballLength = MAX((tempLength - w->mball.delta + 1) /
				w->mball.wedges, 0);
	w->mball.wedgeLength = w->mball.wedges * w->mball.mballLength;
	w->mball.viewLength = w->mball.wedgeLength + w->mball.delta;
	w->mball.viewMiddle = w->mball.viewLength / 2;
	if (w->mball.vertical) {
		w->mball.puzzleSize.x = w->mball.viewLength - 1;
		w->mball.puzzleSize.y = 2 * w->mball.viewLength - w->mball.delta - 2;
	} else {
		w->mball.puzzleSize.x = 2 * w->mball.viewLength - w->mball.delta - 2;
		w->mball.puzzleSize.y = w->mball.viewLength - 1;
	}
	w->mball.puzzleOffset.x = ((int) w->core.width - w->mball.puzzleSize.x) / 2;
	w->mball.puzzleOffset.y = ((int) w->core.height - w->mball.puzzleSize.y) / 2;
	ResizeWedges(w);
}

#ifndef WINVER
static
#endif
void
SizeMball(MballWidget w)
{
	ResetWedges(w);
	ResizeMball(w);
}

#ifndef WINVER
static
#endif
void
InitializeMball(
#ifdef WINVER
MballWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
	int wedge;
#ifdef WINVER
	SetValuesMball(w);
#else
	MballWidget w = (MballWidget) renew;

	w->mball.mono = (DefaultDepthOfScreen(XtScreen(w)) < 2 ||
		w->mball.mono);
	w->mball.fontInfo = NULL;
	for (wedge = 0; wedge < MAXWEDGES; wedge++)
		 w->mball.wedgeGC[wedge] = NULL;
	w->mball.borderGC = NULL;
	w->mball.frameGC = NULL;
	w->mball.inverseGC = NULL;
#endif
	w->mball.focus = False;
	loadFont(w);
	for (wedge = 0; wedge < MAXWEDGES; wedge++)
		w->mball.mballLoc[wedge] = NULL;
	CheckWedges(w);
	InitMoves();
	w->mball.cheat = False;
	SizeMball(w);
#ifdef WINVER
	brush = CreateSolidBrush(w->mball.inverseGC);
	SETBACK(w->core.hWnd, brush);
	(void) SRAND(time(NULL));
#else
	(void) SRAND(getpid());
	SetAllColors(w);
#endif
}

#ifndef WINVER
static
#endif
void
ExposeMball(
#ifdef WINVER
MballWidget w
#else
Widget renew, XEvent * event, Region region
#endif
)
{
#ifndef WINVER
	MballWidget w = (MballWidget) renew;

	if (!w->core.visible)
		return;
#endif
	EraseFrame(w);
	DrawFrame(w, w->mball.focus);
	DrawAllWedges(w);
}

#ifndef WINVER
static
#endif
void
HideMball(MballWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SetMball(w, MBALL_HIDE);
}

#ifndef WINVER
static
#endif
void
SelectMball(MballWidget w
#ifdef WINVER
, const int x, const int y, const int control
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
	int view;
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
	int control = (int) (event->xkey.state & ControlMask);
#endif

	if (SelectWedges(w, x, y,
		 &(w->mball.currentWedge), &(w->mball.currentRing), &view)) {
		if (control || w->mball.practice || !CheckSolved(w))
			DrawSector(w, w->mball.currentWedge, w->mball.currentRing,
				TRUE);
	} else
		w->mball.currentWedge = -1;
}

#ifndef WINVER
static
#endif
void
ReleaseMball(MballWidget w
#ifdef WINVER
, const int x, const int y, const int control
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
	int wedge, ring, view, i, diff, opp;
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
	int control = (int) (event->xkey.state & ControlMask);
#endif

	if (w->mball.currentWedge < 0)
		return;
	DrawSector(w, w->mball.currentWedge, w->mball.currentRing, FALSE);
	if (!control && !w->mball.practice && CheckSolved(w))
		MoveNoWedges(w);
	else if (SelectWedges(w, x, y, &wedge, &ring, &view)) {
		opp = (w->mball.currentWedge + w->mball.wedges / 2) %
			w->mball.wedges;
		if (ring == w->mball.currentRing) {
			if (wedge == w->mball.currentWedge) {
				w->mball.currentWedge = -1;
				return;
			}
			if (opp == (wedge + 1) % w->mball.wedges ||
					wedge == (opp + 1) % w->mball.wedges) {
				SetMball(w, MBALL_AMBIGUOUS);
			} else {
				diff = (w->mball.currentWedge - wedge +
					w->mball.wedges) % w->mball.wedges;
				if (diff > w->mball.wedges / 2)
					for (i = 0; i < w->mball.wedges - diff; i++)
						MoveMball(w, wedge, ring, CW,
							control);
				else
					for (i = 0; i < diff; i++)
						MoveMball(w, wedge, ring, CCW,
							control);
				if (!control && CheckSolved(w)) {
					SetMball(w, MBALL_SOLVED);
				}
			}
		} else if (wedge == w->mball.currentWedge && w->mball.wedges > 2) {
			SetMball(w, MBALL_AMBIGUOUS);
		} else if (opp == (wedge + 1) % w->mball.wedges)
			MoveMball(w, wedge, ring,
			    mapWedgeToDir[(w->mball.wedges - MINWEDGES) / 2]
				  [w->mball.currentWedge], control);
		else if (wedge == (opp + 1) % w->mball.wedges)
			MoveMball(w, wedge, ring,
				  mapWedgeToDir[(w->mball.wedges - MINWEDGES) /
				  2][wedge], control);
		else
			MoveNoWedges(w);
	}
	w->mball.currentWedge = -1;
}

#ifndef WINVER
static
#endif
void
PracticeMball(MballWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	PracticeWedges(w);
}

#ifndef WINVER
static void
PracticeMballMaybe(MballWidget w
, XEvent * event, char **args, int nArgs
)
{
	if (!w->mball.started)
		PracticeWedges(w);
#ifdef HAVE_MOTIF
	else {
		SetMball(w, MBALL_PRACTICE_QUERY);
	}
#endif
}

static void
PracticeMball2(MballWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->mball.started)
#endif
		PracticeWedges(w);
}
#endif

#ifndef WINVER
static
#endif
void
RandomizeMball(MballWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	RandomizeWedges(w);
}

#ifndef WINVER
static void
RandomizeMballMaybe(MballWidget w
, XEvent * event, char **args, int nArgs
)
{
	if (!w->mball.started)
		RandomizeWedges(w);
#ifdef HAVE_MOTIF
	else {
		SetMball(w, MBALL_RANDOMIZE_QUERY);
	}
#endif
}

static void
RandomizeMball2(MballWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->mball.started)
#endif
		RandomizeWedges(w);
}
#endif

#ifndef WINVER
static
#endif
void
GetMball(MballWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	GetWedges(w);
}

#ifndef WINVER
static
#endif
void
WriteMball(MballWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	WriteWedges(w);
}

#ifndef WINVER
static
#endif
void
ClearMball(MballWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	ClearWedges(w);
}

#ifndef WINVER
static
#endif
void
UndoMball(MballWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	UndoWedges(w);
}

#ifndef WINVER
static
#endif
void
SolveMball(MballWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SolveWedges(w);
}

#ifndef WINVER
static void
Wedge2ModeMball(MballWidget w
, XEvent * event, char **args, int nArgs
)
{
	SetMball(w, MBALL_WEDGE2);
}

static void
Wedge4ModeMball(MballWidget w
, XEvent * event, char **args, int nArgs
)
{
	SetMball(w, MBALL_WEDGE4);
}

static void
Wedge6ModeMball(MballWidget w
, XEvent * event, char **args, int nArgs
)
{
	SetMball(w, MBALL_WEDGE6);
}

static void
Wedge8ModeMball(MballWidget w
, XEvent * event, char **args, int nArgs
)
{
	SetMball(w, MBALL_WEDGE8);
}

static void
Wedge10ModeMball(MballWidget w
, XEvent * event, char **args, int nArgs
)
{
	SetMball(w, MBALL_WEDGE10);
}

static void
Wedge12ModeMball(MballWidget w
, XEvent * event, char **args, int nArgs
)
{
	SetMball(w, MBALL_WEDGE12);
}
#endif

#ifndef WINVER
static
#endif
void
IncrementMball(MballWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	IncrementWedges(w);
}

#ifdef WINVER
Boolean
#else
static void
#endif
DecrementMball(MballWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
#ifdef WINVER
	return
#else
	(void)
#endif
	DecrementWedges(w);
}

#ifndef WINVER
static
#endif
void
OrientizeMball(MballWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	OrientizeWedges(w);
}

#ifndef WINVER
static
#endif
void
EnterMball(MballWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->mball.focus = True;
	DrawFrame(w, w->mball.focus);
}

#ifndef WINVER
static
#endif
void
LeaveMball(MballWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->mball.focus = False;
	DrawFrame(w, w->mball.focus);
}

#ifdef WINVER
void
WedgeMball(MballWidget w, const int mode)
{
	SetMball(w, mode + MBALL_WEDGE2);
}

#else

static void
MoveMballCcw(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, CCW,
		(int) (event->xbutton.state & ControlMask));
}

static void
MoveMballTl(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, TL,
		(int) (event->xkey.state & ControlMask));
}

static void
MoveMballTtl(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, TTL,
		(int) (event->xkey.state & ControlMask));
}

static void
MoveMballTop(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, TOP,
		(int) (event->xkey.state & ControlMask));
}

static void
MoveMballTtr(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, TTR,
		(int) (event->xkey.state & ControlMask));
}

static void
MoveMballTr(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, TR,
		(int) (event->xkey.state & ControlMask));
}

static void
MoveMballLeft(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, LEFT,
		(int) (event->xkey.state & ControlMask));
}

static void
MoveMballCw(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, CW,
		(int) (event->xkey.state & ControlMask));
}

static void
MoveMballRight(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, RIGHT,
		(int) (event->xkey.state & ControlMask));
}

static void
MoveMballBl(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, BL,
		(int) (event->xkey.state & ControlMask));
}

static void
MoveMballBbl(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, BBL,
		(int) (event->xkey.state & ControlMask));
}

static void
MoveMballBottom(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, BOTTOM,
		(int) (event->xkey.state & ControlMask));
}

static void
MoveMballBbr(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, BBR,
		(int) (event->xkey.state & ControlMask));
}

static void
MoveMballBr(MballWidget w, XEvent * event, char **args, int nArgs)
{
	MoveMballInput(w, event->xbutton.x, event->xbutton.y, BR,
		(int) (event->xkey.state & ControlMask));
}
#endif
