/*-
# X-BASED CUBES
#
#  Cubes.c
#
###
#
#  Copyright (c) 1994 - 2004	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Cubes */

#if defined(USE_RPLAY) || defined(USE_NAS) || defined(USE_VMSPLAY) || defined(USE_ESOUND) || defined(WINVER) || defined(DEF_PLAY)
#define USE_SOUND
extern void playSound(char * filename);
#endif
#include "CubesP.h"

#ifndef PICTURE
#if 1
#define PICTURE ""
#else
#ifdef WINVER
#define PICTURE "picture"
#else
#ifdef HAVE_XPM
#define PICTURE "./mandrill.xpm"
#else
#define PICTURE "./mandrill.xbm"
#endif
#endif
#endif
#endif

#ifdef WINVER
#ifndef LOGPATH
#define LOGPATH "/usr/tmp"
#endif
#ifndef INIFILE
#define INIFILE "wcubes.ini"
#endif
#define SECTION "setup"
#else
#include "picture.h"

#ifndef LOGPATH
#ifdef VMS
#define LOGPATH "SYS$SCRATCH:"
#else
#define LOGPATH "/usr/tmp"
#endif
#endif

static Boolean SetValuesCubes(Widget current, Widget request, Widget renew);
static void QuitCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void DestroyCubes(Widget old);
static void ResizeCubes(CubesWidget w);
static void SizeCubes(CubesWidget w);
static void InitializeCubes(Widget request, Widget renew);
static void ExposeCubes(Widget renew, XEvent * event, Region region);
static void HideCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void SelectCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void ReleaseCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void RandomizeCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void RandomizeCubesMaybe(CubesWidget w, XEvent * event, char **args, int nArgs);
static void RandomizeCubes2(CubesWidget w, XEvent * event, char **args, int nArgs);
static void GetCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void WriteCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void ClearCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void UndoCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void SolveCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void SpeedCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void SlowCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void SoundCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void EnterCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void LeaveCubes(CubesWidget w, XEvent * event, char **args, int nArgs);
static void MoveCubesOut(CubesWidget w, XEvent * event, char **args, int nArgs);
static void MoveCubesTop(CubesWidget w, XEvent * event, char **args, int nArgs);
static void MoveCubesLeft(CubesWidget w, XEvent * event, char **args, int nArgs);
static void MoveCubesIn(CubesWidget w, XEvent * event, char **args, int nArgs);
static void MoveCubesRight(CubesWidget w, XEvent * event, char **args, int nArgs);
static void MoveCubesBottom(CubesWidget w, XEvent * event, char **args, int nArgs);

static char defaultTranslationsCubes[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>o: MoveOut()\n\
 <KeyPress>KP_Divide: MoveOut()\n\
 <KeyPress>R5: MoveOut()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>i: MoveIn()\n\
 <KeyPress>Begin: MoveIn()\n\
 <KeyPress>KP_5: MoveIn()\n\
 <KeyPress>R11: MoveIn()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <KeyPress>r: Randomize()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 <KeyPress>2: Sound()\n\
 <KeyPress>@: Sound()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";


static XtActionsRec actionsListCubes[] =
{
	{(char *) "Quit", (XtActionProc) QuitCubes},
	{(char *) "Hide", (XtActionProc) HideCubes},
	{(char *) "MoveOut", (XtActionProc) MoveCubesOut},
	{(char *) "MoveTop", (XtActionProc) MoveCubesTop},
	{(char *) "MoveLeft", (XtActionProc) MoveCubesLeft},
	{(char *) "MoveIn", (XtActionProc) MoveCubesIn},
	{(char *) "MoveRight", (XtActionProc) MoveCubesRight},
	{(char *) "MoveBottom", (XtActionProc) MoveCubesBottom},
	{(char *) "Select", (XtActionProc) SelectCubes},
	{(char *) "Release", (XtActionProc) ReleaseCubes},
	{(char *) "Randomize", (XtActionProc) RandomizeCubes},
	{(char *) "RandomizeMaybe", (XtActionProc) RandomizeCubesMaybe},
	{(char *) "Randomize2", (XtActionProc) RandomizeCubes2},
	{(char *) "Get", (XtActionProc) GetCubes},
	{(char *) "Write", (XtActionProc) WriteCubes},
	{(char *) "Clear", (XtActionProc) ClearCubes},
	{(char *) "Undo", (XtActionProc) UndoCubes},
	{(char *) "Solve", (XtActionProc) SolveCubes},
	{(char *) "Speed", (XtActionProc) SpeedCubes},
	{(char *) "Slow", (XtActionProc) SlowCubes},
	{(char *) "Sound", (XtActionProc) SoundCubes},
	{(char *) "Enter", (XtActionProc) EnterCubes},
	{(char *) "Leave", (XtActionProc) LeaveCubes}
};

static XtResource resourcesCubes[] =
{
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(CubesWidget, cubes.username),
	 XtRString, (caddr_t) "guest"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(CubesWidget, cubes.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(CubesWidget, cubes.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(CubesWidget, cubes.frameColor),
	 XtRString, (caddr_t) "cyan" /* XtDefaultForeground*/},
	{XtNblockColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(CubesWidget, cubes.blockColor),
	 XtRString, (caddr_t) "gray75" /*XtDefaultForeground*/},
	{XtNblockBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(CubesWidget, cubes.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultBackground*/},
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(CubesWidget, core.width),
	 XtRString, (caddr_t) "512"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(CubesWidget, core.height),
	 XtRString, (caddr_t) "512"},
	{XtNsizeX, XtCSizeX, XtRInt, sizeof (int),
	 XtOffset(CubesWidget, cubes.sizeX),
	 XtRString, (caddr_t) "4"}, /* DEFAULTCUBESX */
	{XtNsizeY, XtCSizeY, XtRInt, sizeof (int),
	 XtOffset(CubesWidget, cubes.sizeY),
	 XtRString, (caddr_t) "4"}, /* DEFAULTCUBESY */
	{XtNsizeZ, XtCSizeZ, XtRInt, sizeof (int),
	 XtOffset(CubesWidget, cubes.sizeZ),
	 XtRString, (caddr_t) "1"}, /* DEFAULTCUBESZ */
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(CubesWidget, cubes.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverse, XtCReverse, XtRBoolean, sizeof (Boolean),
	 XtOffset(CubesWidget, cubes.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNinstall, XtCInstall, XtRBoolean, sizeof (Boolean),
	 XtOffset(CubesWidget, cubes.install),
	 XtRString, (caddr_t) "FALSE"},
	{XtNpicture, XtCPicture, XtRString, sizeof (String),
	 XtOffset(CubesWidget, cubes.picture),
	 XtRString, (caddr_t) PICTURE},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(CubesWidget, cubes.delay),
	 XtRString, (caddr_t) "10"}, /* DEFAULTDELAY */
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(CubesWidget, cubes.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNbumpSound, XtCBumpSound, XtRString, sizeof (String),
	 XtOffset(CubesWidget, cubes.bumpSound),
	 XtRString, (caddr_t) BUMPSOUND},
	{XtNdripSound, XtCDripSound, XtRString, sizeof (String),
	 XtOffset(CubesWidget, cubes.dripSound),
	 XtRString, (caddr_t) DRIPSOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(CubesWidget, cubes.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNbase, XtCBase, XtRInt, sizeof (int),
	 XtOffset(CubesWidget, cubes.base),
	 XtRString, (caddr_t) "10"}, /* DEFAULTBASE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(CubesWidget, cubes.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(CubesWidget, cubes.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(CubesWidget, cubes.menu),
	 XtRString, (caddr_t) "-1"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(CubesWidget, cubes.select),
	 XtRCallback, (caddr_t) NULL}
};

CubesClassRec cubesClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "Cubes",	/* class name */
		sizeof (CubesRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializeCubes,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListCubes,	/* actions */
		XtNumber(actionsListCubes),	/* num actions */
		resourcesCubes,	/* resources */
		XtNumber(resourcesCubes),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyCubes,	/* destroy */
		(XtWidgetProc) ResizeCubes,	/* resize */
		(XtExposeProc) ExposeCubes,	/* expose */
		(XtSetValuesFunc) SetValuesCubes,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsCubes,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass cubesWidgetClass = (WidgetClass) & cubesClassRec;

#ifndef HAVE_USLEEP
#if !defined( VMS ) || defined( XVMSUTILS ) || ( __VMS_VER >= 70000000 )
#ifdef USE_XVMSUTILS
#include <X11/unix_time.h>
#endif
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#if HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#endif
#endif
#if defined(SYSV) || defined(SVR4)
#ifdef LESS_THAN_AIX3_2
#include <sys/poll.h>
#else /* !LESS_THAN_AIX3_2 */
#include <poll.h>
#endif /* !LESS_THAN_AIX3_2 */
#endif /* defined(SYSV) || defined(SVR4) */

/* not static in case usleep found in system include */
int
usleep(unsigned int usec)
{
#if (defined (SYSV) || defined(SVR4)) && !defined(__hpux)
#if defined(HAVE_NANOSLEEP)
	{
		struct timespec rqt;

		rqt.tv_nsec = 1000 * (usec % (unsigned int) 1000000);
		rqt.tv_sec = usec / (unsigned int) 1000000;
		return nanosleep(&rqt, NULL);
	}
#else
	(void) poll(
#if defined(__cplusplus) || defined(c_plusplus)
		(pollfd *) /* guess */
#else
		(void *)
#endif
		0, (int) 0, usec / 1000);	/* ms resolution */
#endif
#else
#ifdef VMS
	long timadr[2];

	if (usec != 0) {
		timadr[0] = -usec * 10;
		timadr[1] = -1;

		sys$setimr(4, &timadr, 0, 0, 0);
		sys$waitfr(4);
	}
#else
	struct timeval time_out;

#if 0
	/* (!defined(AIXV3) && !defined(__hpux)) */
	extern int select(int, fd_set *, fd_set *, fd_set *, struct timeval *);

#endif

	time_out.tv_usec = usec % (unsigned int) 1000000;
	time_out.tv_sec = usec / (unsigned int) 1000000;
	(void) select(0, (void *) 0, (void *) 0, (void *) 0, &time_out);
#endif
#endif
	return 0;
}
#endif

void
SetCubes(CubesWidget w, int reason)
{
	cubesCallbackStruct cb;

	cb.reason = reason;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}
#endif

static void
loadFont(CubesWidget w)
{
#ifndef WINVER
	Display *display = XtDisplay(w);
	const char *altfontname = "-*-times-*-r-*-*-*-180-*";
	char buf[512];

	if (w->cubes.fontInfo) {
		XUnloadFont(XtDisplay(w), w->cubes.fontInfo->fid);
		XFreeFont(XtDisplay(w), w->cubes.fontInfo);
	}
	if ((w->cubes.fontInfo = XLoadQueryFont(display,
			w->cubes.font)) == NULL) {
		(void) sprintf(buf,
			"Cannot open %s font.\nAttempting %s font as alternate\n",
			w->cubes.font, altfontname);
		DISPLAY_WARNING(buf);
		if ((w->cubes.fontInfo = XLoadQueryFont(display,
				altfontname)) == NULL) {
			(void) sprintf(buf,
				"Cannot open %s alternate font.\nUse the -font option to specify a font to use.\n",
				altfontname);
			DISPLAY_WARNING(buf);
		}
	}
	if (w->cubes.fontInfo) {
		w->cubes.digitOffset.x = XTextWidth(w->cubes.fontInfo, "8", 1)
			/ 2;
		w->cubes.digitOffset.y = w->cubes.fontInfo->max_bounds.ascent
			/ 2;
	} else
#endif
	{
		w->cubes.digitOffset.x = 3;
		w->cubes.digitOffset.y = 4;
	}
}

#ifndef LOGFILE
#define LOGFILE "cubes.log"
#endif

#if (!defined(WINVER) || (WINVER <= 0x030a)) /* if X or WINDOWS 3.1 or less */
static void
Sleep(unsigned int cMilliseconds)
{
#if (defined(WINVER) && (WINVER <= 0x030a))
	unsigned long time_out = GetTickCount() + cMilliseconds;

	while (time_out > GetTickCount());
#else
	(void) usleep(cMilliseconds * 1000);
#endif
}
#endif

static void
initTimer(CubesWidget w)
{
#ifdef WINVER
	w->cubes.oldTime = GetTickCount();
#else
	XFlush(XtDisplay(w));
	(void) X_GETTIMEOFDAY(&(w->cubes.oldTime));
#endif
}


static void
useTimer(CubesWidget w, int delay)
{
	int sleepTime;
#ifdef WINVER
	long tv = GetTickCount();

	sleepTime = delay - (tv - w->cubes.oldTime);
	w->cubes.oldTime = tv;
	if (sleepTime > 0) {
		Sleep((unsigned int) sleepTime);
	}
#else
	struct timeval tv;

	XFlush(XtDisplay(w));
	(void) X_GETTIMEOFDAY(&tv);
	sleepTime = delay * 1000 - ((tv.tv_sec - w->cubes.oldTime.tv_sec) *
		1000000 + tv.tv_usec - w->cubes.oldTime.tv_usec);
	w->cubes.oldTime.tv_sec = tv.tv_sec;
	w->cubes.oldTime.tv_usec = tv.tv_usec;
	if (sleepTime > 0) {
		(void) usleep((unsigned int) sleepTime);
	}
#endif
}

void
intCat(char ** string, const char * var1, const int var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + 21))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	(void) sprintf(*string, "%s%d", var1, var2);
}

void
stringCat(char ** string, const char * var1, const char * var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + strlen(var2) + 1))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	(void) sprintf(*string, "%s%s", var1, var2);
}

static void
CheckBlocks(CubesWidget w)
{
	char *buf1 = NULL, *buf2 = NULL;

	if (w->cubes.sizeX < MINCUBES) {
		intCat(&buf1,
			"Number of cubes in X direction out of bounds, use at least ",
			MINCUBES);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTCUBESX);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->cubes.sizeX = DEFAULTCUBESX;
	}
	if (w->cubes.sizeY < MINCUBES) {
		intCat(&buf1,
			"Number of cubes in Y direction out of bounds, use at least ",
			MINCUBES);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTCUBESY);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->cubes.sizeY = DEFAULTCUBESY;
	}
	if (w->cubes.sizeZ < MINCUBES) {
		intCat(&buf1,
			"Number of cubes in Z direction out of bounds, use at least ",
			MINCUBES);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTCUBESZ);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->cubes.sizeZ = DEFAULTCUBESZ;
	}
	if (w->cubes.delay < 0) {
		intCat(&buf1, "Delay can not be negative (",
			w->cubes.delay);
		stringCat(&buf2, buf1, "), taking absolute value");
		free(buf1);
		DISPLAY_WARNING(buf2);
		free(buf2);
		w->cubes.delay = -w->cubes.delay;
	}
	if (w->cubes.base < MINBASE || w->cubes.base > MAXBASE) {
		intCat(&buf1, "Base out of bounds, use ", MINBASE);
		stringCat(&buf2, buf1, "..");
		free(buf1);
		intCat(&buf1, buf2, MAXBASE);
		free(buf2);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTBASE);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->cubes.base = DEFAULTBASE;
	}
}

static int
ToPosition(CubesWidget w, int column, int row, int stack)
{
	return (column + w->cubes.sizeX * row + w->cubes.sizeRect * stack);
}

static int
Column(CubesWidget w, int pos)
{
	return ((pos % w->cubes.sizeRect) % w->cubes.sizeX);
}

static int
Row(CubesWidget w, int pos)
{
	return ((pos % w->cubes.sizeRect) / w->cubes.sizeX);
}

static int
Stack(CubesWidget w, int pos)
{
	return (pos / w->cubes.sizeRect);
}

static void
Cartesian(CubesWidget w, int pos, int * x, int * y)
{
	*x = Column(w, pos) * w->cubes.offset.x + w->cubes.delta.x +
		w->cubes.puzzleOffset.x;
	*y = Row(w, pos) * w->cubes.offset.y + w->cubes.delta.y +
		w->cubes.puzzleOffset.y;
	if (w->cubes.vertical) {
		*y += Stack(w, pos) * (w->cubes.sizeY * w->cubes.offset.y + 1);
	} else {
		*x += Stack(w, pos) * (w->cubes.sizeX * w->cubes.offset.x + 1);
	}
}

static int
BlockNFromSpace(CubesWidget w, int n, int direction)
{
	int pos = w->cubes.spacePosition;

	if (direction == LEFT || direction == RIGHT)
		return (pos + ((direction == RIGHT) ? -n : n));
	else if (direction == TOP || direction == BOTTOM)
		return (pos + (w->cubes.sizeX * ((direction == BOTTOM) ?
			-n : n)));
	else /* direction == INWARDS || direction == OUTWARDS */
		return (pos + (w->cubes.sizeRect * ((direction == OUTWARDS) ?
			-n : n)));
}

static int
SolvedPosition(CubesWidget w, int block) {
	return (block + 1) % w->cubes.sizeBlock;
}

Boolean
CheckSolved(const CubesWidget w)
{
	int i;

	for (i = 1; i < w->cubes.sizeBlock; i++)
		if (w->cubes.blockOfPosition[i - 1] != i)
			return FALSE;
	return TRUE;
}

static int
int2String(CubesWidget w, char *buf, int number, int base, Boolean capital)
{
	int digit, mult = base, last, position;
	int a, i, j, s, r;

	if (capital) {
		a = 'A', i = 'I', j = 'J', s = 'S', r = 'R';
	} else {
		a = 'a', i = 'i', j = 'j', s = 's', r = 'r';
	}
	if (number < 0) {
		char *buff;

		intCat(&buff, "int2String: 0 > ", number);
		DISPLAY_WARNING(buff);
		free(buff);
		return 0;
	}
	last = 1;
	while (number >= mult) {
		last++;
		mult *= base;
	}
	for (position = 0; position < last; position++) {
		mult /= base;
		digit = number / mult;
		number -= digit * mult;
		buf[position] = digit + '0';
		if (buf[position] > '9') {	/* ASCII */
			buf[position] += (a - '9' - 1);
		} else if (buf[position] < '0') {	/* EBCDIC */
			buf[position] += (a - '9' - 1);
			if (buf[position] > i)
				buf[position] += (j - i - 1);
			if (buf[position] > r)
				buf[position] += (s - r - 1);
		}
	}
	buf[last] = '\0';
	return last;
}

static void
fill3DRect(CubesWidget w, Pixmap dr, GC gc, GC darkerGC, GC brighterGC,
		int x, int y, int width, int height, Boolean raised)
{
	GC currentGC = (raised) ? gc : darkerGC;

	if (width > 2 && height > 2) {
		FILLRECTANGLE(w, dr, currentGC,
			x + 1, y + 1, width - 2, height - 2);
	}
	currentGC = (raised) ? brighterGC : darkerGC;
	FILLRECTANGLE(w, dr, currentGC,
		x, y, 1, height);
	if (width > 1) {
		FILLRECTANGLE(w, dr, currentGC,
			x + 1, y, width - 1, 1);
	}
	FILLRECTANGLE(w, dr, currentGC,
		x + 1, y, 1, height);
	if (width > 2) {
		FILLRECTANGLE(w, dr, currentGC,
			x + 2, y + 1, width - 2, 1);
	}
	currentGC = (raised) ? darkerGC : gc;
	if (width > 1 && height > 1) {
		FILLRECTANGLE(w, dr, currentGC,
			x + 1, y + height - 1, width - 1, 1);
	}
	if (width > 1 && height > 1) {
		FILLRECTANGLE(w, dr, currentGC,
			x + width - 1, y, 1, height - 1);
	}
	if (width > 3 && height > 2) {
		FILLRECTANGLE(w, dr, currentGC,
			x + 2, y + height - 2, width - 3, 1);
	}
	if (width > 2 && height > 3) {
		FILLRECTANGLE(w, dr, currentGC,
			x + width - 2, y + 1, 1, height - 3);
	}
}

static void
drawShadow(CubesWidget w, Pixmap dr, GC gc, int startX, int startY,
		int sizeX, int sizeY)
{
	FILLRECTANGLE(w, dr, gc,
		startX, startY, sizeX + 1, 1);
	FILLRECTANGLE(w, dr, gc,
		startX, startY, 1, sizeY + 1);
}

static void
DrawBlock(CubesWidget w, int pos, Boolean blank, Boolean erase,
		int pressedOffset, int offsetX, int offsetY)
{
	int dx, dy, sx, sy;
	int block = (w->cubes.blockOfPosition[pos] + w->cubes.sizeBlock - 1) %
		w->cubes.sizeBlock;
	Pixmap dr = 0;

	Cartesian(w, pos, &dx, &dy);
	dx += offsetX + pressedOffset;
	dy += offsetY + pressedOffset;
	Cartesian(w, block, &sx, &sy);
	if (blank) {
		FILLRECTANGLE(w, dr,
			(erase) ? w->cubes.inverseGC : w->cubes.blockGC,
			dx, dy,
			w->cubes.blockSize.x, w->cubes.blockSize.y);
		return;
	}
#ifdef WINVER
	w->cubes.hOldBitmap = (HBITMAP) SelectObject(w->cubes.memDC,
		w->cubes.bufferBlocks[pressedOffset]);
	BitBlt(w->core.hDC,
		dx, dy,
		w->cubes.blockSize.x, w->cubes.blockSize.y,
		w->cubes.memDC,
		sx, sy,
		SRCCOPY);
	SelectObject(w->cubes.memDC, w->cubes.hOldBitmap);
#else
	XSetGraphicsExposures(XtDisplay(w), w->cubes.blockGC, False);
	XCopyArea(XtDisplay(w),
		w->cubes.bufferBlocks[pressedOffset],
		XtWindow(w),
		w->cubes.blockGC,
		sx, sy,
		w->cubes.blockSize.x, w->cubes.blockSize.y,
		dx, dy);
#endif
}

static void
DrawBlockPiece(CubesWidget w, int pos, Boolean blank, Boolean erase,
		int pressedOffset, int offsetX, int offsetY,
		int lox, int loy, int lsx, int lsy)
{
	int dx, dy, sx, sy;
	int block = (w->cubes.blockOfPosition[pos] + w->cubes.sizeBlock - 1) %
		w->cubes.sizeBlock;
	Pixmap dr = 0;

	Cartesian(w, pos, &dx, &dy);
	dx += offsetX + pressedOffset;
	dy += offsetY + pressedOffset;
	Cartesian(w, block, &sx, &sy);
	if (blank) {
		FILLRECTANGLE(w, dr,
			(erase) ? w->cubes.inverseGC : w->cubes.blockGC,
			dx + lox, dy + loy,
			lsx, lsy);
		return;
	}
#ifdef WINVER
	w->cubes.hOldBitmap = (HBITMAP) SelectObject(w->cubes.memDC,
		w->cubes.bufferBlocks[pressedOffset]);
	BitBlt(w->core.hDC,
		dx + lox, dy + loy,
		lsx, lsy,
		w->cubes.memDC,
		sx + lox, sy + loy,
		SRCCOPY);
	SelectObject(w->cubes.memDC, w->cubes.hOldBitmap);
#else
	XSetGraphicsExposures(XtDisplay(w), w->cubes.blockGC, False);
	XCopyArea(XtDisplay(w),
		w->cubes.bufferBlocks[pressedOffset],
		XtWindow(w),
		w->cubes.blockGC,
		sx + lox, sy + loy,
		lsx, lsy,
		dx + lox, dy + loy);
#endif
}

static void
DrawBufferedBlock(CubesWidget w, int pos, int pressedOffset)

{
	Pixmap *dr;
	GC blockGC, borderGC;
	int dx, dy, block = SolvedPosition(w, pos);
	int i = 0, digitOffsetX = 0;
	char buf[65];

	dr = &(w->cubes.bufferBlocks[pressedOffset]);
	blockGC = w->cubes.blockGC;
	borderGC = w->cubes.borderGC;
	Cartesian(w, pos, &dx, &dy);
	dx += pressedOffset;
	dy += pressedOffset;
	if (pressedOffset != 0) {
		drawShadow(w, *dr, w->cubes.blockDarkerGC,
			dx - pressedOffset, dy - pressedOffset,
			w->cubes.blockSize.x - 1, w->cubes.blockSize.y - 1);
	}
	fill3DRect(w, *dr, blockGC,
		w->cubes.blockDarkerGC, w->cubes.blockBrighterGC,
		dx, dy, w->cubes.blockSize.x, w->cubes.blockSize.y,
		pressedOffset == 0);
	(void) int2String(w, buf, block, w->cubes.base, True);
	while (block >= 1) {
		block /= w->cubes.base;
		digitOffsetX += w->cubes.digitOffset.x;
		i++;
	}
	DRAWTEXT(w, *dr, borderGC,
		dx + w->cubes.blockSize.x / 2 - digitOffsetX,
		dy + w->cubes.blockSize.y / 2 + w->cubes.digitOffset.y,
		buf, i);
}

static void
DrawAllBufferedBlocks(const CubesWidget w)
{
	int k, l;

	for (k = 0; k < w->cubes.sizeBlock; k++)
		for (l = 0; l < 2; l++)
			DrawBufferedBlock(w, k, l);
}

void
DrawAllBlocks(const CubesWidget w)
{
	int k;

	for (k = 0; k < w->cubes.sizeBlock; k++)
		DrawBlock(w, k, (w->cubes.blockOfPosition[k] <= 0),
			(w->cubes.blockOfPosition[k] <= 0), FALSE, 0, 0);
}

static void
DrawBlockFrame(CubesWidget w, int xR, int yR, int wR, int hR, int incAt)
{
	Pixmap dr = 0;

	if (wR > 2 * incAt && hR > 2 * incAt) {
		/* Perfectionism here...
		   have the erasing square be 3D also */
		FILLRECTANGLE(w, dr,
			w->cubes.blockBrighterGC,
			xR + incAt, yR + incAt, 2, hR - 2 * incAt);
		FILLRECTANGLE(w, dr,
			w->cubes.blockBrighterGC,
			xR + incAt, yR + incAt, wR - 2 * incAt, 2);
		FILLRECTANGLE(w, dr,
			w->cubes.blockDarkerGC,
			xR + wR - incAt - 2, yR + incAt, 2, hR - 2 * incAt);
		FILLRECTANGLE(w, dr,
			w->cubes.blockDarkerGC,
			xR + incAt, yR + hR - incAt - 2, wR - 2 * incAt, 2);
	}
}

void
AnimateSlide(CubesWidget w, int numBlocks, int dir, int fast, Boolean logMoves)
{
	int inc, aBlock;
	int fillBeginPos, fillBeginI = 0, gapI = 0, moveI = 0;
	int dx, dy, pos, posNext;
	int ix = 0, iy = 0;
	Pixmap dr = 0;

	fillBeginPos = BlockNFromSpace(w, numBlocks, dir);
	if ((dir == RIGHT) || (dir == LEFT)) {
	        Cartesian(w, fillBeginPos, &fillBeginI, &dy);
		gapI = w->cubes.blockSize.x * fast / w->cubes.numSlices;
		moveI = w->cubes.blockSize.x + w->cubes.delta.x;
	} else if ((dir == TOP) || (dir == BOTTOM)) {
	        Cartesian(w, fillBeginPos, &dx, &fillBeginI);
		gapI = w->cubes.blockSize.y * fast / w->cubes.numSlices;
		moveI = w->cubes.blockSize.y + w->cubes.delta.y;
	}
	if (gapI == 0)
		gapI++;
	if (numBlocks < 0)
	  numBlocks = -numBlocks;
	initTimer(w);
	for (inc = 0; inc < moveI + gapI; inc += gapI) {
	  if (inc > moveI)
	    inc = moveI;
	  for (aBlock = 0; aBlock < numBlocks; aBlock++) {
	    posNext = BlockNFromSpace(w, aBlock + 1, dir);
	    if (logMoves && inc == 0) {
	      SetCubes(w, CUBES_MOVED);
	      PutMove(dir);
	    }
	    /* Calculate deltas */
	    Cartesian(w, posNext, &dx, &dy);
	    if ((dir == RIGHT) || (dir == LEFT)) {
	      ix = ((dir == RIGHT) ? inc : -inc);
	      iy = 0;
	    } else if ((dir == TOP) || (dir == BOTTOM)) {
	      ix = 0;
	      iy = ((dir == BOTTOM) ? inc : -inc);
	    }
	    DrawBlock(w, posNext, False, False, FALSE, ix, iy);
	    /* Erase old slivers */
	    ix += dx;
	    iy += dy;
	    if (aBlock < numBlocks - 1) {
	      switch (dir) {
		case TOP:
		  FILLRECTANGLE(w, dr,
		    w->cubes.inverseGC,
		    ix, iy + w->cubes.blockSize.y,
		    w->cubes.blockSize.x, gapI);
		  break;
		case RIGHT:
		  FILLRECTANGLE(w, dr,
		    w->cubes.inverseGC,
		    ix - gapI, iy, gapI, w->cubes.blockSize.y);
		  break;
		case BOTTOM:
		  FILLRECTANGLE(w, dr,
		    w->cubes.inverseGC,
		    ix, iy - gapI, w->cubes.blockSize.x, gapI);
		  break;
		case LEFT:
		  FILLRECTANGLE(w, dr,
		    w->cubes.inverseGC,
		    ix + w->cubes.blockSize.x, iy,
		    gapI, w->cubes.blockSize.y);
		  break;
	      }
	    } else {
	      switch (dir) {
		case TOP:
		  FILLRECTANGLE(w, dr,
		    w->cubes.inverseGC,
		    ix, iy + w->cubes.blockSize.y,
		    w->cubes.blockSize.x, fillBeginI - iy);
		  break;
		case RIGHT:
		  FILLRECTANGLE(w, dr,
		    w->cubes.inverseGC,
		    fillBeginI, iy,
		    ix - fillBeginI, w->cubes.blockSize.y);
		  break;
		case BOTTOM:
		  FILLRECTANGLE(w, dr,
		    w->cubes.inverseGC,
		    ix, fillBeginI,
		    w->cubes.blockSize.x, iy - fillBeginI);
		  break;
		case LEFT:
		  FILLRECTANGLE(w, dr,
		    w->cubes.inverseGC,
		    ix + w->cubes.blockSize.x, iy,
		    fillBeginI - ix, w->cubes.blockSize.y);
		 break;
	      }
	    }
	  }
	  useTimer(w, w->cubes.delay);
	}
	pos = w->cubes.spacePosition;
	for (aBlock = 0; aBlock < numBlocks; aBlock++) {
	    posNext = BlockNFromSpace(w, aBlock + 1, dir);
	    w->cubes.blockOfPosition[pos] =
	      w->cubes.blockOfPosition[posNext];
	    pos = posNext;
	}
	w->cubes.spacePosition = fillBeginPos;
	w->cubes.blockOfPosition[fillBeginPos] = 0;
}

static void
AnimateMove(CubesWidget w, int fromPos, int fast)
{
	int pos = w->cubes.spacePosition;
	int dx, dy, fromdx, fromdy;
	int xToCenter, yToCenter;
	int incTo, incAt, inc;
	Pixmap dr = 0;

	Cartesian(w, pos, &dx, &dy);
	Cartesian(w, fromPos, &fromdx, &fromdy);
	xToCenter = w->cubes.blockSize.x / 2;
	yToCenter = w->cubes.blockSize.y / 2;
	incTo = MIN(w->cubes.blockSize.x, w->cubes.blockSize.y) / 2;
	incAt = MAX(incTo * fast / (2 * w->cubes.numSlices), 1);

	FILLRECTANGLE(w, dr,
		w->cubes.inverseGC,
		fromdx, fromdy, incAt, w->cubes.blockSize.y);
	FILLRECTANGLE(w, dr,
		w->cubes.inverseGC,
		fromdx, fromdy, w->cubes.blockSize.x, incAt);
	FILLRECTANGLE(w, dr,
		w->cubes.inverseGC,
		fromdx + w->cubes.blockSize.x - incAt, fromdy,
		incAt, w->cubes.blockSize.y);
	FILLRECTANGLE(w, dr,
		w->cubes.inverseGC,
		fromdx, fromdy + w->cubes.blockSize.y - incAt,
		w->cubes.blockSize.x, incAt);
	initTimer(w);
	for (inc = incAt; inc < incTo; inc += incAt) {
		int xR = fromdx + inc, yR = fromdy + inc;
		int wR = w->cubes.blockSize.x - 2 * inc;
		int hR = w->cubes.blockSize.y - 2 * inc;

		/* start disappear block */
		if (wR > 0 && hR > 0) {
			FILLRECTANGLE(w, dr,
				w->cubes.inverseGC,
				xR, yR, incAt, hR);
			FILLRECTANGLE(w, dr,
				w->cubes.inverseGC,
				xR, yR, wR, incAt);
			FILLRECTANGLE(w, dr,
				w->cubes.inverseGC,
				xR + wR - incAt, yR, incAt, hR);
			FILLRECTANGLE(w, dr,
				w->cubes.inverseGC,
				xR, yR + hR - incAt, wR, incAt);
		}
		if (!(w->cubes.picture && *(w->cubes.picture))) {
			DrawBlockFrame(w, xR, yR, wR, hR, incAt);
		}
		/* end disappear block */
		/* start reappear blocks */
		if (w->cubes.blockSize.x > w->cubes.blockSize.y) {
			/* horizontal blocks */
			DrawBlockPiece(w, pos, False, False, FALSE, 0, 0,
			  xToCenter - inc - (w->cubes.blockSize.x / 2 - w->cubes.blockSize.y / 2),
			  yToCenter - inc,
			  2 * (inc + w->cubes.blockSize.x / 2 - w->cubes.blockSize.y / 2),
			  2 * inc);
		if (!(w->cubes.picture && *(w->cubes.picture))) {
			DrawBlockFrame(w,
			    xToCenter - inc - (w->cubes.blockSize.x / 2 - w->cubes.blockSize.y / 2) + dx - incAt,
			    yToCenter - inc + dy - incAt,
			    2 * (inc + w->cubes.blockSize.x / 2 - w->cubes.blockSize.y / 2) + 2 * incAt,
			    2 * inc + 2 * incAt,
			    incAt);
		}
#if 0
			  fill3DRect(w, dr, w->cubes.blockGC,
			    w->cubes.blockDarkerGC,
			    w->cubes.blockBrighterGC,
			    xToCenter - inc - (w->cubes.blockSize.x / 2 - w->cubes.blockSize.y / 2) + dx,
			    yToCenter - inc + dy,
			    2 * (inc + w->cubes.blockSize.x / 2 - w->cubes.blockSize.y / 2),
			    2 * inc,
			    True);
#endif
		} else {
			/* vertical blocks */
			DrawBlockPiece(w, pos, False, False, FALSE, 0, 0,
			  xToCenter - inc,
			  yToCenter - inc - (w->cubes.blockSize.y / 2 - w->cubes.blockSize.x / 2),
			  2 * inc,
			  2 * (inc + w->cubes.blockSize.y / 2 - w->cubes.blockSize.x / 2));
		if (!(w->cubes.picture && *(w->cubes.picture))) {
			DrawBlockFrame(w,
			    xToCenter - inc + dx - incAt,
			    yToCenter - inc - (w->cubes.blockSize.y / 2 - w->cubes.blockSize.x / 2) + dy - incAt,
			    2 * inc + 2 * incAt,
			    2 * (inc + w->cubes.blockSize.y / 2 - w->cubes.blockSize.x / 2) + 2 * incAt,
			    incAt);
		}
#if 0
			  fill3DRect(w, dr, w->cubes.blockGC,
			    w->cubes.blockDarkerGC,
			    w->cubes.blockBrighterGC,
			    xToCenter - inc + dx,
			    yToCenter - inc - (w->cubes.blockSize.y / 2 - w->cubes.blockSize.x / 2) + dy,
			    2 * inc,
			    2 * (inc + w->cubes.blockSize.y / 2 - w->cubes.blockSize.x / 2),
			    True);
#endif
		}
		useTimer(w, w->cubes.delay);
	}
	DrawBlock(w, fromPos, True, True, FALSE, 0, 0);
	DrawBlock(w, pos, False, False, FALSE, 0, 0);
#ifndef WINVER
	XFlush(XtDisplay(w));
#endif
	w->cubes.spacePosition = fromPos;
}

static void
ResetBlocks(CubesWidget w)
{
	int i;

	w->cubes.sizeRect = w->cubes.sizeX * w->cubes.sizeY;
	w->cubes.sizeBlock = w->cubes.sizeRect * w->cubes.sizeZ;
	if (w->cubes.blockOfPosition)
		free((void *) w->cubes.blockOfPosition);
	if (!(w->cubes.blockOfPosition = (int *)
			malloc(sizeof (int) * w->cubes.sizeBlock))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}

	if (startPosition)
		free((void *) startPosition);
	if (!(startPosition = (int *)
			malloc(sizeof (int) * w->cubes.sizeBlock))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}

	w->cubes.spacePosition = w->cubes.sizeBlock - 1;
	w->cubes.blockOfPosition[w->cubes.sizeBlock - 1] = 0;
	for (i = 1; i < w->cubes.sizeBlock; i++)
		w->cubes.blockOfPosition[i - 1] = i;
	FlushMoves(w);
	w->cubes.currentPosition = -1;
	w->cubes.started = False;
}

static void
EraseFrame(const CubesWidget w, Pixmap dr)
{
	FILLRECTANGLE(w, dr, w->cubes.inverseGC,
		0, 0, w->core.width, w->core.height);
}

static void
DrawFrame(CubesWidget w, Pixmap dr, Boolean focus)
{
	int sumX, sumY, sumZ, offsetX, offsetY, dz, k;
	GC gc = (focus) ? w->cubes.frameGC : w->cubes.borderGC;

	offsetX = w->cubes.puzzleOffset.x;
	offsetY = w->cubes.puzzleOffset.y;
	FILLRECTANGLE(w, dr, w->cubes.borderGC,
		0, 0, w->core.width, offsetY);
	FILLRECTANGLE(w, dr, w->cubes.borderGC,
		0, w->core.height - offsetY,
		w->core.width, offsetY);
	FILLRECTANGLE(w, dr, w->cubes.borderGC,
		0, 0, offsetX, w->core.height);
	FILLRECTANGLE(w, dr, w->cubes.borderGC,
		w->core.width - offsetX, 0,
		offsetX, w->core.height);

	sumX = w->cubes.sizeX * w->cubes.offset.x + 1;
	sumY = w->cubes.sizeY * w->cubes.offset.y + 1;
	if (w->cubes.vertical) {
		sumZ = offsetY;
		dz = sumY;
	} else {
		sumZ = offsetX;
		dz = sumX;
	}
	for (k = 0; k < w->cubes.sizeZ; k++) {
		if (w->cubes.vertical) {
			FILLRECTANGLE(w, dr, gc,
				offsetX, sumZ, 1, sumY);
			FILLRECTANGLE(w, dr, gc,
				offsetX, sumZ, sumX, 1);
			FILLRECTANGLE(w, dr, gc,
				sumX + offsetX, sumZ, 1, sumY + 1);
			FILLRECTANGLE(w, dr, gc,
				offsetX, sumY + sumZ, sumX + 1, 1);
		} else {
			FILLRECTANGLE(w, dr, gc,
				sumZ, offsetY, 1, sumY);
			FILLRECTANGLE(w, dr, gc,
				sumZ, offsetY, sumX, 1);
			FILLRECTANGLE(w, dr, gc,
				sumX + sumZ, offsetY, 1, sumY + 1);
			FILLRECTANGLE(w, dr, gc,
				sumZ, sumY + offsetY, sumX + 1, 1);
		}
		sumZ += dz;
	}
}

static void
MoveNoBlocks(CubesWidget w)
{
	SetCubes(w, CUBES_IGNORE);
}

static void
MoveBlocks(CubesWidget w, int from, int fast)
{
	int tempBlock;
#ifdef USE_SOUND
	int oldSpace;

	oldSpace = w->cubes.spacePosition;
#endif

	tempBlock = w->cubes.blockOfPosition[from];
	if (fast != INSTANT && w->cubes.delay > 0) {
		int i = w->cubes.currentRow[0] = Column(w, from);
		int j = w->cubes.currentRow[1] = Row(w, from);
		int k = w->cubes.currentRow[2] = Stack(w, from);

		int l = i + w->cubes.sizeX * j - w->cubes.spacePosition % w->cubes.sizeRect;
		/* int m = i + w->cubes.sizeX * j + w->cubes.sizeRect * k - w->cubes.spacePosition; */
		/* Order important if w->cubes.sizeX = 1 */

		w->cubes.currentPosition = from;
		if (l % w->cubes.sizeX == 0 && k == Stack(w, w->cubes.spacePosition)) {
			if (l < 0) {
				AnimateSlide(w, -l / w->cubes.sizeX, BOTTOM,
					fast, False);
			} else if (l > 0) {
				AnimateSlide(w, l / w->cubes.sizeX, TOP, fast,
					False);
			}
		} else if (l / w->cubes.sizeX == 0 &&
				j == Row(w, w->cubes.spacePosition) &&
				k == Stack(w, w->cubes.spacePosition)) {
			if (l < 0) {
				AnimateSlide(w, -l, RIGHT, fast, False);
			} else {	/* (l > 0) */
				AnimateSlide(w, l, LEFT, fast, False);
			}
		} else {
			w->cubes.blockOfPosition[from] =
				w->cubes.blockOfPosition[w->cubes.spacePosition];
			w->cubes.blockOfPosition[w->cubes.spacePosition] = tempBlock;
			AnimateMove(w, from, fast);
		}
		w->cubes.currentPosition = -1;
	} else {
		w->cubes.blockOfPosition[from] =
			w->cubes.blockOfPosition[w->cubes.spacePosition];
		w->cubes.blockOfPosition[w->cubes.spacePosition] = tempBlock;
		DrawBlock(w, w->cubes.spacePosition, False, False, FALSE, 0, 0);
		w->cubes.spacePosition = from;
		DrawBlock(w, w->cubes.spacePosition, True, True, FALSE, 0, 0);
	}
#ifdef USE_SOUND
	if (w->cubes.sound) {
		playSound((char *) ((ABS(from - oldSpace) >=
			w->cubes.sizeRect) ? DRIPSOUND : BUMPSOUND));
	}
#endif
}

static int
MoveBlocksDir(CubesWidget w, int direction, int fast)
{
	switch (direction) {
		case TOP:
			if (Row(w, w->cubes.spacePosition) < w->cubes.sizeY - 1) {
				MoveBlocks(w, w->cubes.spacePosition +
					w->cubes.sizeX, fast);
				return TRUE;
			}
			break;
		case RIGHT:
			if (Column(w, w->cubes.spacePosition) > 0) {
				MoveBlocks(w, w->cubes.spacePosition - 1, fast);
				return TRUE;
			}
			break;
		case BOTTOM:
			if (Row(w, w->cubes.spacePosition) > 0) {
				MoveBlocks(w, w->cubes.spacePosition -
					w->cubes.sizeX, fast);
				return TRUE;
			}
			break;
		case LEFT:
			if (Column(w, w->cubes.spacePosition) < w->cubes.sizeX - 1) {
				MoveBlocks(w, w->cubes.spacePosition + 1, fast);
				return TRUE;
			}
			break;
		case INWARDS:
			if (Stack(w, w->cubes.spacePosition) > 0) {
				MoveBlocks(w, w->cubes.spacePosition -
					w->cubes.sizeRect, fast);
				return TRUE;
			}
			break;
		case OUTWARDS:
			if (Stack(w, w->cubes.spacePosition) < w->cubes.sizeZ - 1) {
				MoveBlocks(w, w->cubes.spacePosition +
					w->cubes.sizeRect, fast);
				return TRUE;
			}
			break;
		default:
			{
				char *buf;

				intCat(&buf, "MoveBlocksDir: direction ",
					direction);
				DISPLAY_WARNING(buf);
				free(buf);
			}
	}
	return FALSE;
}

int
MoveCubesDir(CubesWidget w, const int direction, const int fast)
{
	if (MoveBlocksDir(w, direction, fast)) {
		SetCubes(w, CUBES_MOVED);
		PutMove(direction);
		return TRUE;
	}
	return FALSE;
}


#ifndef WINVER
static
#endif
int
MoveCubes(CubesWidget w, const int direction, const int control)
{
	int reason;

	if (control) {
		reason = CUBES_IGNORE;
		switch (direction) {
			case TOP:
				if (w->cubes.sizeY <= MINCUBES)
					return FALSE;
				reason = CUBES_DEC_Y;
				break;
			case RIGHT:
				reason = CUBES_INC_X;
				break;
			case BOTTOM:
				reason = CUBES_INC_Y;
				break;
			case LEFT:
				if (w->cubes.sizeX <= MINCUBES)
					return FALSE;
				reason = CUBES_DEC_X;
				break;
			case INWARDS:
				if (w->cubes.sizeZ <= MINCUBES)
					return FALSE;
				reason = CUBES_DEC_Z;
				break;
			case OUTWARDS:
				reason = CUBES_INC_Z;
				break;
			default:
				{
					char *buf;

					intCat(&buf, "MoveCubes: direction ",
						direction);
					DISPLAY_WARNING(buf);
					free(buf);
					SetCubes(w, reason);
					return FALSE;
				}
		}
		SetCubes(w, reason);
		return TRUE;
	}
	if (CheckSolved(w)) {
		MoveNoBlocks(w);
		return FALSE;
	}
	if (!MoveCubesDir(w, direction, NORMAL)) {
		SetCubes(w, CUBES_BLOCKED);
		return FALSE;
	}
	if (CheckSolved(w)) {
		SetCubes(w, CUBES_SOLVED);
	}
	return TRUE;
}

static int
ExchangeBlocks(CubesWidget w, int pos1, int pos2)
{
	int tempBlock;

	if (w->cubes.blockOfPosition[pos1] <= 0)
		return FALSE;
	else if (w->cubes.blockOfPosition[pos2] <= 0)
		return FALSE;
	tempBlock = w->cubes.blockOfPosition[pos1];
	w->cubes.blockOfPosition[pos1] = w->cubes.blockOfPosition[pos2];
	w->cubes.blockOfPosition[pos2] = tempBlock;
	return TRUE;
}

static void
DiscreteMoves(CubesWidget w, int from, int dir, int fast)
{
	MoveBlocks(w, from, fast);
	SetCubes(w, CUBES_MOVED);
	PutMove(dir);
}

static int
PositionToBlock(CubesWidget w, int x, int y, int *i, int *j, int *k)
{
	if (w->cubes.vertical) {
		*i = (x - w->cubes.delta.x / 2 -
		      w->cubes.puzzleOffset.x) / w->cubes.offset.x;
		*j = ((y - w->cubes.delta.y / 2 -
		       w->cubes.puzzleOffset.y) % (w->cubes.sizeY * w->cubes.offset.y +
				w->cubes.delta.y - 1)) / w->cubes.offset.y;
		*k = (y - w->cubes.delta.y / 2 -
		      w->cubes.puzzleOffset.y) / (w->cubes.sizeY * w->cubes.offset.y +
				w->cubes.delta.y - 1);
	} else {
		*i = ((x - w->cubes.delta.x / 2 -
		       w->cubes.puzzleOffset.x) % (w->cubes.sizeX * w->cubes.offset.x +
				w->cubes.delta.x - 1)) / w->cubes.offset.x;
		*j = (y - w->cubes.delta.y / 2 -
		      w->cubes.puzzleOffset.y) / w->cubes.offset.y;
		*k = (x - w->cubes.delta.x / 2 -
		      w->cubes.puzzleOffset.x) / (w->cubes.sizeX * w->cubes.offset.x +
				w->cubes.delta.y - 1);
	}
	if (*i >= 0 && *j >= 0 && *k >= 0 && *i < w->cubes.sizeX &&
			*j < w->cubes.sizeY && *k < w->cubes.sizeZ)
		return ToPosition(w, *i, *j, *k);
	else
		return -1;
}

static int
MovableBlock(CubesWidget w)
{
	int i, j, k, l, m;

	i = w->cubes.currentRow[0];
	j = w->cubes.currentRow[1];
	k = w->cubes.currentRow[2];
	l = i + w->cubes.sizeX * j - w->cubes.spacePosition % w->cubes.sizeRect;
	m = i + w->cubes.sizeX * j + w->cubes.sizeRect * k - w->cubes.spacePosition;

	/* Order important if w->cubes.sizeX = 1 */
	if (l % w->cubes.sizeX == 0 && k == Stack(w, w->cubes.spacePosition)) {
		if (l == 0) {
			return CUBES_SPACE;
		}
	} else if (!(l / w->cubes.sizeX == 0 &&
			j == Row(w, w->cubes.spacePosition) &&
			k == Stack(w, w->cubes.spacePosition)) &&
			(m % w->cubes.sizeRect != 0)) {
		return CUBES_BLOCKED;
	}
	return 0;
}

static void
SelectBlocks(CubesWidget w)
{
	int i, j, k, l, m, n;
	char *buf;

	i = w->cubes.currentRow[0];
	j = w->cubes.currentRow[1];
	k = w->cubes.currentRow[2];
	l = i + w->cubes.sizeX * j - w->cubes.spacePosition % w->cubes.sizeRect;
	m = i + w->cubes.sizeX * j + w->cubes.sizeRect * k - w->cubes.spacePosition;

	/* Order important if w->cubes.sizeX = 1 */
	if (l % w->cubes.sizeX == 0 && k == Stack(w, w->cubes.spacePosition)) {
		if (l < 0) {
			if (w->cubes.delay > 0) {
				AnimateSlide(w, -l / w->cubes.sizeX, BOTTOM,
					NORMAL, True);
#ifdef USE_SOUND
				if (w->cubes.sound) {
					playSound((char *) BUMPSOUND);
				}
#endif
			} else {
				for (n = 1; n <= -l / w->cubes.sizeX; n++)
					DiscreteMoves(w,
						w->cubes.spacePosition - w->cubes.sizeX,
						BOTTOM, NORMAL);
			}
		} else if (l > 0) {
			if (w->cubes.delay > 0) {
				AnimateSlide(w, l / w->cubes.sizeX, TOP, NORMAL,
					True);
#ifdef USE_SOUND
				if (w->cubes.sound) {
					playSound((char *) BUMPSOUND);
				}
#endif
			} else {
				for (n = 1; n <= l / w->cubes.sizeX; n++)
					DiscreteMoves(w,
						w->cubes.spacePosition + w->cubes.sizeX,
						TOP, NORMAL);
			}
		} else {	/* (l == 0) */
			intCat(&buf, "SelectBlocks: ", CUBES_SPACE);
			DISPLAY_WARNING(buf);
			free(buf);
			return;
		}
	} else if (l / w->cubes.sizeX == 0 &&
			j == Row(w, w->cubes.spacePosition) &&
			k == Stack(w, w->cubes.spacePosition)) {
		if (l < 0) {
			if (w->cubes.delay > 0) {
				AnimateSlide(w, -l, RIGHT, NORMAL, True);
#ifdef USE_SOUND
				if (w->cubes.sound) {
					playSound((char *) BUMPSOUND);
				}
#endif
			} else {
				for (n = 1; n <= -l % w->cubes.sizeX; n++)
				  DiscreteMoves(w,
				    w->cubes.spacePosition - 1, RIGHT, NORMAL);
			}
		} else {	/* (l > 0) */
			if (w->cubes.delay > 0) {
				AnimateSlide(w, l, LEFT, NORMAL, True);
#ifdef USE_SOUND
				if (w->cubes.sound) {
					playSound((char *) BUMPSOUND);
				}
#endif
			} else {
				for (n = 1; n <= l % w->cubes.sizeX; n++)
				  DiscreteMoves(w,
				    w->cubes.spacePosition + 1, LEFT, NORMAL);
			}
		}
	} else if (m % w->cubes.sizeRect == 0) {
		if (m < 0) {
			for (n = 1; n <= -Stack(w, m); n++) {
				DiscreteMoves(w,
				  w->cubes.spacePosition - w->cubes.sizeRect,
				  INWARDS, (w->cubes.delay <= 0) ?
				  INSTANT : NORMAL);
			}
		} else {	/* (m > 0) */
			for (n = 1; n <= Stack(w, m); n++) {
				DiscreteMoves(w,
				  w->cubes.spacePosition + w->cubes.sizeRect,
				  OUTWARDS, (w->cubes.delay <= 0) ?
				  INSTANT : NORMAL);
			}
		}
	} else {
		intCat(&buf, "SelectBlocks: ", CUBES_BLOCKED);
		DISPLAY_WARNING(buf);
		free(buf);
		return;
	}
	if (CheckSolved(w)) {
		SetCubes(w, CUBES_SOLVED);
	}
}

static void
RandomizeBlocks(CubesWidget w)
{
	if (w->cubes.currentPosition >= 0)
		return;
	w->cubes.cheat = False;
	/* First interchange blocks an even number of times */
	/* Must be at least 3x2 or 2x2x2 */
	if ((w->cubes.sizeX > 2 && (w->cubes.sizeY > 1 || w->cubes.sizeZ > 1)) ||
			(w->cubes.sizeY > 2 && (w->cubes.sizeX > 1 || w->cubes.sizeZ > 1)) ||
			(w->cubes.sizeZ > 2 && (w->cubes.sizeX > 1 || w->cubes.sizeY > 1)) ||
			(w->cubes.sizeX == 2 && w->cubes.sizeY == 2 && w->cubes.sizeZ == 2)) {
		int pos, count = 0;

		for (pos = 0; pos < w->cubes.sizeBlock; pos++) {
			int randomPos = pos;

			while (randomPos == pos) {
				randomPos = NRAND(w->cubes.sizeBlock);
			}
			count += ExchangeBlocks(w, pos, randomPos);
		}
		if ((count & 1) &&
				!ExchangeBlocks(w, 0, 1) &&
				!ExchangeBlocks(w, w->cubes.sizeBlock - 2,
				w->cubes.sizeBlock - 1)) {
			DISPLAY_WARNING("RandomizeBlocks: should not get here");
		}
		DrawAllBlocks(w);
#ifndef WINVER
		XFlush(XtDisplay(w));
#endif
	}
	/* randomly position space */
	/* Idea for this came from "puzzle" by Don Bennett, HP Labs */
	{
		int n, s, e;

		s = Column(w, w->cubes.spacePosition);
		e = NRAND(w->cubes.sizeX);
		for (n = 0; n < e - s; n++)
			(void) MoveCubesDir(w, LEFT, INSTANT);
		for (n = 0; n < s - e; n++)
			(void) MoveCubesDir(w, RIGHT, INSTANT);
		s = Row(w, w->cubes.spacePosition);
		e = NRAND(w->cubes.sizeY);
		for (n = 0; n < e - s; n++)
			(void) MoveCubesDir(w, TOP, INSTANT);
		for (n = 0; n < s - e; n++)
			(void) MoveCubesDir(w, BOTTOM, INSTANT);
		s = Stack(w, w->cubes.spacePosition);
		e = NRAND(w->cubes.sizeZ);
		for (n = 0; n < e - s; n++)
			(void) MoveCubesDir(w, OUTWARDS, INSTANT);
		for (n = 0; n < s - e; n++)
			(void) MoveCubesDir(w, INWARDS, INSTANT);
		FlushMoves(w);
		SetCubes(w, CUBES_RANDOMIZE);
	}
#if 0
	/* Now move the space around randomly */
	if (w->cubes.sizeX > 1 || w->cubes.sizeY > 1 || w->cubes.sizeZ > 1) {
		int big = w->cubes.sizeBlock + NRAND(2);
		int lastDirection = -1;
		int randomDirection;

		SetCubes(w, CUBES_RESET);
#ifdef DEBUG
		big = 3;
#endif
		if (big > 1000)
			big = 1000;
		while (big--) {
			randomDirection = NRAND(COORD);
#ifdef DEBUG
			Sleep(1);
#endif
			if ((randomDirection < 4 && (randomDirection + 2) % 4 != lastDirection) ||
			    (randomDirection >= 4 && randomDirection + ((randomDirection % 2) ? -1 : 1) != lastDirection) ||
			    (w->cubes.sizeX == 1 && w->cubes.sizeY == 1) ||
			    (w->cubes.sizeY == 1 && w->cubes.sizeZ == 1) ||
			    (w->cubes.sizeZ == 1 && w->cubes.sizeX == 1)) {
				if (MoveCubesDir(w, randomDirection, INSTANT))
					lastDirection = randomDirection;
				else
					big++;
			}
		}
		FlushMoves(w);
		SetCubes(w, CUBES_RANDOMIZE);
	}
#endif
	if (CheckSolved(w)) {
		SetCubes(w, CUBES_SOLVED);
	}
}

static void
GetBlocks(CubesWidget w)
{
	FILE *fp;
	int c, i, sizeX, sizeY, sizeZ, moves;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "r")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read (get)", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read (get)", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	FlushMoves(w);
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &sizeX);
	if (sizeX >= MINCUBES) {
		for (i = w->cubes.sizeX; i < sizeX; i++) {
			SetCubes(w, CUBES_INC_X);
		}
		for (i = w->cubes.sizeX; i > sizeX; i--) {
			SetCubes(w, CUBES_DEC_X);
		}
	} else {
		stringCat(&buf1, name, " corrupted: sizeX ");
		intCat(&buf2, buf1, sizeX);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MINCUBES);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &sizeY);
	if (sizeY >= MINCUBES) {
		for (i = w->cubes.sizeY; i < sizeY; i++) {
			SetCubes(w, CUBES_INC_Y);
		}
		for (i = w->cubes.sizeY; i > sizeY; i--) {
			SetCubes(w, CUBES_DEC_Y);
		}
	} else {
		stringCat(&buf1, name, " corrupted: sizeY ");
		intCat(&buf2, buf1, sizeY);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MINCUBES);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &sizeZ);
	if (sizeZ >= MINCUBES) {
		for (i = w->cubes.sizeZ; i < sizeZ; i++) {
			SetCubes(w, CUBES_INC_Z);
		}
		for (i = w->cubes.sizeZ; i > sizeZ; i--) {
			SetCubes(w, CUBES_DEC_Z);
		}
	} else {
		stringCat(&buf1, name, " corrupted: sizeZ ");
		intCat(&buf2, buf1, sizeZ);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MINCUBES);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
	}
#ifdef WINVER
	ResetBlocks(w);
#endif
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &moves);
	ScanStartPosition(fp, w);
	SetCubes(w, CUBES_RESTORE);
	SetStartPosition(w);
	ScanMoves(fp, w, moves);
	(void) fclose(fp);
	(void) printf("%s: sizeX %d, sizeY %d, sizeZ %d, moves %d.\n",
		name, sizeX, sizeY, sizeZ, moves);
	free(lname);
	free(fname);
	w->cubes.cheat = True; /* Assume the worst. */
}

static void
WriteBlocks(CubesWidget w)
{
	FILE *fp;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "w")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	(void) fprintf(fp, "sizeX%c %d\n", SYMBOL, w->cubes.sizeX);
	(void) fprintf(fp, "sizeY%c %d\n", SYMBOL, w->cubes.sizeY);
	(void) fprintf(fp, "sizeZ%c %d\n", SYMBOL, w->cubes.sizeZ);
	(void) fprintf(fp, "moves%c %d\n", SYMBOL, NumMoves());
	PrintStartPosition(fp, w);
	PrintMoves(fp);
	(void) fclose(fp);
	(void) printf("Saved to %s.\n", name);
	free(lname);
	free(fname);
}

static void
ClearBlocks(CubesWidget w)
{
	if (w->cubes.currentPosition >= 0)
		return;
	ResetBlocks(w);
	DrawAllBlocks(w);
	SetCubes(w, CUBES_RESET);
}

static void
UndoBlocks(CubesWidget w)
{
	if (MadeMoves() && w->cubes.currentPosition < 0) {
		int direction;

		GetMove(&direction);
		direction = (direction < 4) ? (direction + 2) % 4 :
			((direction == INWARDS) ? OUTWARDS : INWARDS);

		if (MoveBlocksDir(w, direction, DOUBLE)) {
			SetCubes(w, CUBES_UNDO);
		} else {
			char *buf1, *buf2;

			intCat(&buf1, "Move ", direction);
			stringCat(&buf2, buf1, " can not be made");
			free(buf1);
			DISPLAY_WARNING(buf2);
			free(buf2);
		}
	}
}

static void
SolveBlocks(CubesWidget w)
{
	if (CheckSolved(w) || w->cubes.currentPosition >= 0)
		return;
	if (w->cubes.sizeX >= 4 && w->cubes.sizeY >= 4 &&
			w->cubes.sizeZ == 1) {
		SolveSomeBlocks(w);
	} else {
		SetCubes(w, CUBES_SOLVE_MESSAGE);
	}
}

static void
SpeedBlocks(CubesWidget w)
{
	w->cubes.delay -= 5;
	if (w->cubes.delay < 0)
		w->cubes.delay = 0;
#ifdef HAVE_MOTIF
	SetCubes(w, CUBES_SPEED);
#endif
}

static void
SlowBlocks(CubesWidget w)
{
	w->cubes.delay += 5;
#ifdef HAVE_MOTIF
	SetCubes(w, CUBES_SPEED);
#endif
}

static void
SoundBlocks(CubesWidget w)
{
	w->cubes.sound = !w->cubes.sound;
}

#define FACTOR 0.7

#ifdef WINVER
#define MAXINTENSITY 0xFF
static int
brighter(const int light)
{
	int i = (int) ((1 - FACTOR) * MAXINTENSITY);
	int temp = light;

	if (temp < i)
		temp = i;
	return MIN(temp / FACTOR, MAXINTENSITY);
}

static int
darker(const int light)
{
	return (int) (light * FACTOR);
}

static void
SetValuesCubes(CubesWidget w)
{
	struct tagColor {
		int red, green, blue;
	} color;
	char szBuf[80];

	w->cubes.sizeX = GetPrivateProfileInt(SECTION, "sizeX",
		DEFAULTCUBESX, INIFILE);
	w->cubes.sizeY = GetPrivateProfileInt(SECTION, "sizeY",
		DEFAULTCUBESY, INIFILE);
	w->cubes.sizeZ = GetPrivateProfileInt(SECTION, "sizeZ",
		DEFAULTCUBESZ, INIFILE);
	w->cubes.base = GetPrivateProfileInt(SECTION, "base",
		DEFAULTBASE, INIFILE);
	w->cubes.mono = (BOOL) GetPrivateProfileInt(SECTION, "mono",
		DEFAULTMONO, INIFILE);
	w->cubes.reverse = (BOOL) GetPrivateProfileInt(SECTION, "reverse",
		DEFAULTREVERSE, INIFILE);
	/* cyan */
	(void) GetPrivateProfileString(SECTION, "frameColor", "0 255 255",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->cubes.frameGC = RGB(color.red, color.green, color.blue);
	/* gray75 */
	(void) GetPrivateProfileString(SECTION, "blockColor", "191 191 191",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->cubes.blockGC = RGB(color.red, color.green, color.blue);
	w->cubes.blockBrighterGC = RGB(brighter(color.red),
		brighter(color.green), brighter(color.blue));
	w->cubes.blockDarkerGC = RGB(darker(color.red),
		darker(color.green), darker(color.blue));
	/* gray25 */
	(void) GetPrivateProfileString(SECTION, "blockBorder", "64 64 64",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->cubes.borderGC = RGB(color.red, color.green, color.blue);
	/* #AEB2C3 */
	(void) GetPrivateProfileString(SECTION, "background", "174 178 195",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->cubes.inverseGC = RGB(color.red, color.green, color.blue);
	(void) GetPrivateProfileString(SECTION, "picture", PICTURE,
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->cubes.picture, szBuf);
	w->cubes.picture[80] = 0;
	w->cubes.delay = GetPrivateProfileInt(SECTION, "delay",
		DEFAULTDELAY, INIFILE);
	w->cubes.sound = (BOOL)
		GetPrivateProfileInt(SECTION, "sound", 0, INIFILE);
	(void) GetPrivateProfileString(SECTION, "bumpSound", BUMPSOUND,
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->cubes.bumpSound, szBuf);
	w->cubes.bumpSound[80] = 0;
	(void) GetPrivateProfileString(SECTION, "dripSound", DRIPSOUND,
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->cubes.dripSound, szBuf);
	w->cubes.dripSound[80] = 0;
	(void) GetPrivateProfileString(SECTION, "name", "Guest",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->cubes.username, szBuf);
	w->cubes.username[80] = 0;
}

void
DestroyCubes(HBRUSH brush)
{
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

#else
#define MAXINTENSITY 0xFFFF

static Pixel
brighter(CubesWidget w, Pixel pixel)
{
	XColor color;
	Colormap colormap = (w->cubes.colormap == None) ?
		w->cubes.oldColormap : w->cubes.colormap;
	int i = (int) ((1 - FACTOR) * MAXINTENSITY);

	color.pixel = pixel;
	XQueryColor(XtDisplay(w), colormap, &color);
	if (color.red < i)
		color.red = i;
	if (color.green < i)
		color.green = i;
	if (color.blue < i)
		color.blue = i;
	color.red = (unsigned short) MIN(color.red / FACTOR, MAXINTENSITY);
	color.green = (unsigned short) MIN(color.green / FACTOR, MAXINTENSITY);
	color.blue = (unsigned short) MIN(color.blue / FACTOR, MAXINTENSITY);
	if (XAllocColor(XtDisplay(w), colormap, &color))
		return color.pixel;
	return pixel;
}

static Pixel
darker(CubesWidget w, Pixel pixel)
{
	XColor color;
	Colormap colormap = (w->cubes.colormap == None) ?
		w->cubes.oldColormap : w->cubes.colormap;

	color.pixel = pixel;
	XQueryColor(XtDisplay(w), colormap, &color);
	color.red = (unsigned short) (color.red * FACTOR);
	color.green = (unsigned short) (color.green * FACTOR);
	color.blue = (unsigned short) (color.blue * FACTOR);
	if (XAllocColor(XtDisplay(w), colormap, &color))
		return color.pixel;
	return pixel;
}

static void
SetAllColors(CubesWidget w)
{
	XGCValues values;
	XtGCMask valueMask;

	valueMask = GCForeground | GCBackground;

	if (w->cubes.reverse) {
		values.foreground = w->cubes.foreground;
		values.background = w->cubes.background;
	} else {
		values.foreground = w->cubes.background;
		values.background = w->cubes.foreground;
	}
	if (w->cubes.inverseGC)
		XtReleaseGC((Widget) w, w->cubes.inverseGC);
	w->cubes.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->cubes.mono) {
		if (w->cubes.reverse) {
			values.foreground = w->cubes.background;
			values.background = w->cubes.foreground;
		} else {
			values.foreground = w->cubes.foreground;
			values.background = w->cubes.background;
		}
	} else {
		values.foreground = w->cubes.frameColor;
		values.background = w->cubes.borderColor;
	}
	if (w->cubes.frameGC)
		XtReleaseGC((Widget) w, w->cubes.frameGC);
	w->cubes.frameGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->cubes.mono) {
		if (w->cubes.reverse) {
			values.foreground = w->cubes.background;
			values.background = w->cubes.foreground;
		} else {
			values.foreground = w->cubes.foreground;
			values.background = w->cubes.background;
		}
	} else {
		values.foreground = w->cubes.blockColor;
		values.background = w->cubes.borderColor;
	}

	if (w->cubes.blockGC)
		XtReleaseGC((Widget) w, w->cubes.blockGC);
	w->cubes.blockGC = XtGetGC((Widget) w, valueMask, &values);
	if (!w->cubes.mono) {
		values.foreground = brighter(w, w->cubes.blockColor);
	}
	if (w->cubes.blockBrighterGC)
		XtReleaseGC((Widget) w, w->cubes.blockBrighterGC);
	w->cubes.blockBrighterGC = XtGetGC((Widget) w, valueMask, &values);
	if (!w->cubes.mono) {
		values.foreground = darker(w, w->cubes.blockColor);
	}
	if (w->cubes.blockDarkerGC)
		XtReleaseGC((Widget) w, w->cubes.blockDarkerGC);
	w->cubes.blockDarkerGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->cubes.mono) {
		if (w->cubes.reverse) {
			values.foreground = w->cubes.foreground;
			values.background = w->cubes.background;
		} else {
			values.foreground = w->cubes.background;
			values.background = w->cubes.foreground;
		}
	} else {
		values.foreground = w->cubes.borderColor;
		values.background = w->cubes.blockColor;
	}
	if (w->cubes.borderGC)
		XtReleaseGC((Widget) w, w->cubes.borderGC);
	w->cubes.borderGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->cubes.fontInfo)
		XSetFont(XtDisplay(w), w->cubes.borderGC,
			w->cubes.fontInfo->fid);
}

static Boolean
SetValuesCubes(Widget current, Widget request, Widget renew)
{
	CubesWidget c = (CubesWidget) current, w = (CubesWidget) renew;
	Boolean redraw = False;
	Boolean redrawBlocks = False;

	CheckBlocks(w);
	if (w->cubes.font != c->cubes.font ||
			w->cubes.borderColor != c->cubes.borderColor ||
			w->cubes.reverse != c->cubes.reverse ||
			w->cubes.mono != c->cubes.mono) {
		loadFont(w);
		SetAllColors(w);
		redrawBlocks = True;
	} else if (w->cubes.background != c->cubes.background ||
			w->cubes.foreground != c->cubes.foreground ||
			w->cubes.blockColor != c->cubes.blockColor) {
		SetAllColors(w);
		redrawBlocks = True;
	}
	if (w->cubes.sizeX != c->cubes.sizeX ||
			w->cubes.sizeY != c->cubes.sizeY ||
			w->cubes.sizeZ != c->cubes.sizeZ ||
			w->cubes.base != c->cubes.base) {
		SizeCubes(w);
		redraw = True;
	} else if (w->cubes.offset.x != c->cubes.offset.x ||
			w->cubes.offset.y != c->cubes.offset.y) {
		ResizeCubes(w);
		redraw = True;
	}
	if (w->cubes.delay != c->cubes.delay) {
		w->cubes.numSlices = ((w->cubes.delay < MAXSLICES) ?
			w->cubes.delay + 1 : MAXSLICES);
	}
	if (w->cubes.menu != -1) {
		int menu = w->cubes.menu;

		w->cubes.menu = -1;
		switch (menu) {
		case 0:
			GetBlocks(w);
			break;
		case 1:
			WriteBlocks(w);
			break;
		case 3:
			ClearBlocks(w);
			break;
		case 4:
			UndoBlocks(w);
			break;
		case 5:
			RandomizeBlocks(w);
			break;
		case 6:
			SolveBlocks(w);
			break;
		case 7:
			SpeedBlocks(w);
			break;
		case 8:
			SlowBlocks(w);
			break;
		case 9:
			SoundBlocks(w);
			break;
		default:
			break;
		}
	}
	if (redrawBlocks && !redraw && XtIsRealized(renew) &&
			renew->core.visible) {
		EraseFrame(c, 0);
		if (w->cubes.focus)
			DrawFrame(w, 0, True);
		DrawAllBlocks(w);
	}
	return (redraw);
}

static void
QuitCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
#ifdef WINVER
	if (w->cubes.freeDC) {
		if (w->cubes.bufferBlocks != NULL) {
			DeleteObject(w->cubes.bufferBlocks);
		}
		DeleteDC(w->cubes.memDC);
		w->cubes.memDC = NULL;
	}
#else
	Display *display = XtDisplay(w);
	int sel;

	if (w->cubes.colormap != None) {
		XInstallColormap(display, w->cubes.oldColormap);
		XFreeColormap(display, w->cubes.colormap);
	}
	for (sel = 0; sel < 2; sel++)
		if (w->cubes.bufferBlocks[sel] != None)
			XFreePixmap(display, w->cubes.bufferBlocks[sel]);
	if (w->cubes.fontInfo) {
		XUnloadFont(XtDisplay(w), w->cubes.fontInfo->fid);
		XFreeFont(display, w->cubes.fontInfo);
	}
	XtCloseDisplay(display);
#endif
	exit(0);
}

static void
DestroyCubes(Widget old)
{
	CubesWidget w = (CubesWidget) old;

	XtReleaseGC(old, w->cubes.blockGC);
	XtReleaseGC(old, w->cubes.blockBrighterGC);
	XtReleaseGC(old, w->cubes.blockDarkerGC);
	XtReleaseGC(old, w->cubes.borderGC);
	XtReleaseGC(old, w->cubes.frameGC);
	XtReleaseGC(old, w->cubes.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->cubes.select);
}
#endif

static void
ResizeBlocks(CubesWidget w)
{
	int sel;

#ifdef WINVER
	if (w->cubes.memDC == NULL) {
		w->cubes.memDC = CreateCompatibleDC(w->core.hDC);
		if (w->cubes.memDC == NULL) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (sel = 0; sel < 2; sel++) {
		if (w->cubes.bufferBlocks[sel] != NULL) {
			DeleteObject(w->cubes.bufferBlocks[sel]);
				w->cubes.bufferBlocks[sel] = NULL;
		}
		if (!(w->cubes.picture && *(w->cubes.picture))) {
		    if ((w->cubes.bufferBlocks[sel] =
				CreateCompatibleBitmap(w->core.hDC,
				w->core.width, w->core.height)) == NULL) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		    }
		}
	}
#else
	Display *display = XtDisplay(w);
	Window window = XtWindow(w);
	XWindowAttributes xgwa;

	(void) XGetWindowAttributes(display, window, &xgwa);
	if (w->cubes.oldColormap == None) {
		w->cubes.mono = (xgwa.depth < 2 || w->cubes.mono);
		w->cubes.oldColormap = xgwa.colormap;
	}
	for (sel = 0; sel < 2; sel++) {
		if (w->cubes.bufferBlocks[sel] != None) {
			XFreePixmap(display, w->cubes.bufferBlocks[sel]);
			w->cubes.bufferBlocks[sel] = None;
		}
		if ((w->cubes.bufferBlocks[sel] = XCreatePixmap(display,
			window, w->core.width, w->core.height,
			xgwa.depth)) == None) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
#endif
	if (w->cubes.picture && *(w->cubes.picture)) {
#ifdef WINVER
		for (sel = 0; sel < 2; sel++) {
			w->cubes.bufferBlocks[sel] =
				LoadBitmap(w->core.hInstance,
				w->cubes.picture);
		}
#else
		if (w->cubes.image != NULL)  {
			destroyImage(&(w->cubes.image),
				&(w->cubes.graphicsFormat));
		}
		if (!getImage(display, window,
				xgwa.visual, w->cubes.oldColormap, xgwa.depth,
				&(w->cubes.image), w->cubes.picture,
				w->cubes.install, &(w->cubes.graphicsFormat),
				&(w->cubes.colormap))) {
			w->cubes.picture = NULL;
		} else if (w->cubes.image == NULL) {
			w->cubes.picture = NULL;
		}
#endif
	}
#ifndef WINVER
	if (!(w->cubes.picture && *(w->cubes.picture)) &&
	    !fixedColors(xgwa.visual, xgwa.depth, w->cubes.install) &&
	    w->cubes.colormap == None) {
		w->cubes.colormap = XCreateColormap(display, window,
			xgwa.visual, AllocNone);
	}
	SetAllColors(w);
	for (sel = 0; sel < 2; sel++) {
		FILLRECTANGLE(w, w->cubes.bufferBlocks[sel],
			w->cubes.inverseGC,
			0, 0, w->core.width, w->core.height);
		if ((w->cubes.picture && *(w->cubes.picture))) {

			(void) XPutImage(display, w->cubes.bufferBlocks[sel],
				w->cubes.inverseGC, w->cubes.image, 0, 0,
				0, 0,
				MIN(w->cubes.image->width, w->core.width),
				MIN(w->cubes.image->height, w->core.height));
		}
	}
#endif
	if (!(w->cubes.picture && *(w->cubes.picture))) {
		DrawAllBufferedBlocks(w);
	}
}

void
ResizeCubes(CubesWidget w)
{
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif

	w->cubes.delta.x = 1;
	w->cubes.delta.y = 1;
	w->cubes.offset.x = MAX(((int) w->core.width - 2) /
		w->cubes.sizeX, 0);
	w->cubes.offset.y = MAX(((int) w->core.height - 2) /
		w->cubes.sizeY, 0);
	if (w->cubes.offset.y >= w->cubes.offset.x) {
		w->cubes.vertical = TRUE;
		w->cubes.offset.y = MAX(((int) w->core.height /
			w->cubes.sizeZ - 2) / w->cubes.sizeY, 0);
	} else {
		w->cubes.vertical = FALSE;
		w->cubes.offset.x = MAX(((int) w->core.width /
			w->cubes.sizeZ - 2) / w->cubes.sizeX, 0);
	}
	w->cubes.faceSize.x = w->cubes.offset.x * w->cubes.sizeX +
		w->cubes.delta.x + 2;
	w->cubes.faceSize.y = w->cubes.offset.y * w->cubes.sizeY +
		w->cubes.delta.y + 2;
	if (w->cubes.vertical) {
		w->cubes.puzzleSize.x = w->cubes.faceSize.x;
		w->cubes.puzzleSize.y = (w->cubes.faceSize.y - 2) *
			w->cubes.sizeZ + 2;
	} else {
		w->cubes.puzzleSize.x = (w->cubes.faceSize.x - 2) *
			w->cubes.sizeZ + 2;
		w->cubes.puzzleSize.y = w->cubes.faceSize.y;
	}
	w->cubes.puzzleOffset.x = ((int) w->core.width -
		w->cubes.puzzleSize.x + 2) / 2;
	w->cubes.puzzleOffset.y = ((int) w->core.height -
		w->cubes.puzzleSize.y + 2) / 2;
	w->cubes.blockSize.x = MAX(w->cubes.offset.x - w->cubes.delta.x, 0);
	w->cubes.blockSize.y = MAX(w->cubes.offset.y - w->cubes.delta.y, 0);
}

#ifndef WINVER
static
#endif
void
SizeCubes(CubesWidget w)
{
	ResetBlocks(w);
	ResizeCubes(w);
}

#ifndef WINVER
static
#endif
void
InitializeCubes(
#ifdef WINVER
CubesWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
#ifdef WINVER
	SetValuesCubes(w);
	brush = CreateSolidBrush(w->cubes.inverseGC);
	SETBACK(w->core.hWnd, brush);
	(void) SRAND(time(NULL));
	w->cubes.bufferBlocks[0] = NULL;
	w->cubes.bufferBlocks[1] = NULL;
#else
	CubesWidget w = (CubesWidget) renew;

	(void) SRAND(getpid());
	w->cubes.bufferBlocks[0] = None;
	w->cubes.bufferBlocks[1] = None;
	w->cubes.colormap = None;
	w->cubes.oldColormap = None;
	w->cubes.fontInfo = NULL;
	w->cubes.blockGC = NULL;
	w->cubes.blockBrighterGC = NULL;
	w->cubes.blockDarkerGC = NULL;
	w->cubes.borderGC = NULL;
	w->cubes.frameGC = NULL;
	w->cubes.inverseGC = NULL;
#endif
	w->cubes.focus = False;
	loadFont(w);
	w->cubes.blockOfPosition = NULL;
	CheckBlocks(w);
	InitMoves();
	w->cubes.numSlices = ((w->cubes.delay < MAXSLICES) ?
		w->cubes.delay + 1 : MAXSLICES);
	w->cubes.cheat = False;
	SizeCubes(w);
}

#ifndef WINVER
static
#endif
void
ExposeCubes(
#ifdef WINVER
CubesWidget w
#else
Widget renew, XEvent * event, Region region
#endif
)
{
#ifndef WINVER
	CubesWidget w = (CubesWidget) renew;

	if (!w->core.visible)
		return;
#endif
	ResizeBlocks(w);
	EraseFrame(w, 0);
	DrawFrame(w, 0, w->cubes.focus);
	DrawAllBlocks(w);
}

#ifndef WINVER
static
#endif
void
HideCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SetCubes(w, CUBES_HIDE);
}

#ifndef WINVER
static
#endif
void
SelectCubes(CubesWidget w
#ifdef WINVER
, const int x, const int y
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
	int pos, i, j, k, rowType;
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
#endif

	pos = PositionToBlock(w, x, y, &i, &j, &k);
	if (pos >= 0) {
		if (CheckSolved(w)) {
			MoveNoBlocks(w);
			w->cubes.currentPosition = -1;
			return;
		}
		w->cubes.currentPosition = pos;
		w->cubes.currentRow[0] = i;
		w->cubes.currentRow[1] = j;
		w->cubes.currentRow[2] = k;
		rowType = MovableBlock(w);
		if (rowType < 0) {
			DrawBlock(w, w->cubes.currentPosition,
				rowType == CUBES_SPACE, False, TRUE, 0, 0);
#ifndef WINVER
			XFlush(XtDisplay(w));
#endif
			Sleep(100);
			DrawBlock(w, w->cubes.currentPosition,
				True, True, TRUE, 0, 0);
			if (rowType != CUBES_SPACE)
				DrawBlock(w, w->cubes.currentPosition,
					False, False, FALSE, 0, 0);
			SetCubes(w, rowType);
			w->cubes.currentPosition = -1;
			return;
		}
		DrawBlock(w, w->cubes.currentPosition,
			False, False, TRUE, 0, 0);
	} else
		w->cubes.currentPosition = -1;
}

#ifndef WINVER
static
#endif
void
ReleaseCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	if (w->cubes.currentPosition < 0)
		return;
	DrawBlock(w, w->cubes.currentPosition, True, True, TRUE, 0, 0);
	DrawBlock(w, w->cubes.currentPosition, False, False, FALSE, 0, 0);
	SelectBlocks(w);
	w->cubes.currentPosition = -1;
}

#ifndef WINVER
static
#endif
void
RandomizeCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	RandomizeBlocks(w);
}

#ifndef WINVER
static
void
RandomizeCubesMaybe(CubesWidget w
, XEvent * event, char **args, int nArgs
)
{
	if (!w->cubes.started) {
		RandomizeBlocks(w);
	}
#ifdef HAVE_MOTIF
	else {
		SetCubes(w, CUBES_RANDOMIZE_QUERY);
	}
#endif
}

static
void
RandomizeCubes2(CubesWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->cubes.started)
#endif
		RandomizeBlocks(w);
}
#endif

#ifndef WINVER
static
#endif
void
GetCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	GetBlocks(w);
}

#ifndef WINVER
static
#endif
void
WriteCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	WriteBlocks(w);
}

#ifndef WINVER
static
#endif
void
ClearCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	ClearBlocks(w);
}

#ifndef WINVER
static
#endif
void
UndoCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	UndoBlocks(w);
}

#ifndef WINVER
static
#endif
void
SolveCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SolveBlocks(w);
}

#ifndef WINVER
static
#endif
void
SpeedCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SpeedBlocks(w);
}

#ifndef WINVER
static
#endif
void
SlowCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SlowBlocks(w);
}

#ifndef WINVER
static
#endif
void
SoundCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SoundBlocks(w);
}

#ifndef WINVER
static
#endif
void
EnterCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->cubes.focus = True;
	DrawFrame(w, 0, w->cubes.focus);
#ifndef WINVER
	if (w->cubes.colormap != None)
		XInstallColormap(XtDisplay(w), w->cubes.colormap);
#endif
}

#ifndef WINVER
static
#endif
void
LeaveCubes(CubesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->cubes.focus = False;
	DrawFrame(w, 0, w->cubes.focus);
#ifndef WINVER
	if (w->cubes.colormap != None)
		XInstallColormap(XtDisplay(w), w->cubes.oldColormap);
#endif
}

#ifndef WINVER
static void
MoveCubesOut(CubesWidget w, XEvent * event, char **args, int nArgs
)
{
	(void) MoveCubes(w, OUTWARDS, (int) (event->xkey.state & ControlMask));
}

static void
MoveCubesTop(CubesWidget w, XEvent * event, char **args, int nArgs
)
{
	(void) MoveCubes(w, TOP, (int) (event->xkey.state & ControlMask));
}

static void
MoveCubesLeft(CubesWidget w, XEvent * event, char **args, int nArgs
)
{
	(void) MoveCubes(w, LEFT, (int) (event->xkey.state & ControlMask));
}

static void
MoveCubesIn(CubesWidget w, XEvent * event, char **args, int nArgs
)
{
	(void) MoveCubes(w, INWARDS, (int) (event->xkey.state & ControlMask));
}

static void
MoveCubesRight(CubesWidget w, XEvent * event, char **args, int nArgs
)
{
	(void) MoveCubes(w, RIGHT, (int) (event->xkey.state & ControlMask));
}

static void
MoveCubesBottom(CubesWidget w, XEvent * event, char **args, int nArgs
)
{
	(void) MoveCubes(w, BOTTOM, (int) (event->xkey.state & ControlMask));
}
#endif
