/* $Id: dialogs.c 19890 2006-02-15 17:14:43Z kelnos $ */
/*
 * Copyright (c) 2003,2005 Jasper Huijsmans <jasper@xfce.org>
 * Copyright (c) 2003 Benedikt Meurer <benny@xfce.org>
 *  Copyright (C) 2002 Olivier Fourdan (fourdan@xfce.org)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/**
 * xfce_create_header_with_image:
 * @image: a GtkImage or NULL if no image should be displayed in the header.
 * @text: the text to be displayed in the header.
 *
 * Creates a header with an optional @image (may be NULL) in larger bold
 * font. Background and foreground colors are taken from Gtk+ style.
 *
 * Return value: the container widget that contains the header widgets.
 **/


static void
private_cb_eventbox_style_set (GtkWidget * widget, GtkStyle * old_style)
{
    static gboolean recursive = 0;
    GtkStyle *style;

    if (recursive > 0)
	return;

    ++recursive;
    style = gtk_widget_get_style (widget);
    gtk_widget_modify_bg (widget, GTK_STATE_NORMAL,
			  &style->bg[GTK_STATE_SELECTED]);
    --recursive;
}

static void
private_cb_label_style_set (GtkWidget * widget, GtkStyle * old_style)
{
    static gboolean recursive = 0;
    GtkStyle *style;

    if (recursive > 0)
	return;

    ++recursive;
    style = gtk_widget_get_style (widget);
    gtk_widget_modify_fg (widget, GTK_STATE_NORMAL,
			  &style->fg[GTK_STATE_SELECTED]);
    --recursive;
}

static
GtkWidget*
xfce_create_header_with_image (GtkWidget   *image,
                               const gchar *text)
{
    GtkWidget *eventbox, *label, *hbox;
    GtkStyle *style;
    char *markup;

    eventbox = gtk_event_box_new ();
    gtk_widget_show (eventbox);

    hbox = gtk_hbox_new (FALSE, 12);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), 4);
    gtk_widget_show (hbox);
    gtk_container_add (GTK_CONTAINER (eventbox), hbox);

    if (image)
    {
	gtk_widget_show (image);
	gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);
    }

    style = gtk_widget_get_style (eventbox);
    gtk_widget_modify_bg (eventbox, GTK_STATE_NORMAL,
			  &style->bg[GTK_STATE_SELECTED]);

    markup =
	g_strconcat ("<span size=\"larger\" weight=\"bold\">", text,
		     "</span>", NULL);
    label = gtk_label_new (markup);
    g_free (markup);
    gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
    gtk_widget_show (label);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    
    style = gtk_widget_get_style (label);
    gtk_widget_modify_fg (label, GTK_STATE_NORMAL,
			  &style->fg[GTK_STATE_SELECTED]);

    g_signal_connect_after (G_OBJECT (eventbox), "style_set",
			    G_CALLBACK (private_cb_eventbox_style_set), NULL);
    g_signal_connect_after (G_OBJECT (label), "style_set",
			    G_CALLBACK (private_cb_label_style_set), NULL);

    return eventbox;
}

/**
 * xfce_create_header:
 * @icon: a GdkPixbuf or NULL if no icon should be displayed in the header.
 * @text: a text to be displayed in the header.
 *
 * Creates a header with an optional @icon (may be NULL) in larger bold
 * font. Background and foreground colors are taken from Gtk+ style.
 *
 * Return value: the container widget that contains the header widgets.
 **/
GtkWidget*
xfce_create_header (GdkPixbuf *icon, const gchar *text)
{
    GtkWidget *image = NULL;
    
    if (icon)
	image = gtk_image_new_from_pixbuf (icon);

    return xfce_create_header_with_image (image, text);
}



#include <gdk/gdk.h>
#include <gtk/gtkframe.h>

G_BEGIN_DECLS

#define XFCE_TYPE_FRAMEBOX          (xfce_framebox_get_type())
#define XFCE_FRAMEBOX(obj)          G_TYPE_CHECK_INSTANCE_CAST (obj, xfce_framebox_get_type (), XfceFramebox)
#define XFCE_FRAMEBOX_CLASS(klass)  G_TYPE_CHECK_CLASS_CAST (klass, xfce_framebox_get_type (), XfceFrameboxClass)
#define XFCE_IS_FRAMEBOX(obj)       G_TYPE_CHECK_INSTANCE_TYPE (obj, xfce_framebox_get_type ())

    typedef struct _XfceFramebox XfceFramebox;
    typedef struct _XfceFrameboxClass XfceFrameboxClass;

    struct _XfceFramebox
    {
        GtkFrame frame;
        GtkWidget *hbox;
        GtkWidget *indent_filler;
    };

    struct _XfceFrameboxClass
    {
        GtkFrameClass parent_class;
    };

static
    GType xfce_framebox_get_type        (void) G_GNUC_CONST;
static
    GtkWidget *xfce_framebox_new        (const gchar *text,
                                         gboolean indent);
static
    void xfce_framebox_add              (XfceFramebox *framebox, 
                                         GtkWidget *widget);

G_END_DECLS

/* Forward declarations */

static void xfce_framebox_class_init (XfceFrameboxClass * class);

/* Local data */

static GtkWidgetClass *parent_class = NULL;

static
GtkType
xfce_framebox_get_type (void)
{
    static GtkType framebox_type = 0;

    if (!framebox_type)
    {
        static const GTypeInfo framebox_info = {
            sizeof (XfceFrameboxClass),
            NULL,               /* base_init */
            NULL,               /* base_finalize */
            (GClassInitFunc) xfce_framebox_class_init,
            NULL,               /* class_finalize */
            NULL,               /* class_data */
            sizeof (XfceFramebox),
            0,                  /* n_preallocs */
            NULL,               /* instance_init */
            NULL                /* value_table */
        };

        framebox_type =
            g_type_register_static (GTK_TYPE_FRAME,
                                    "XfceFramebox", &framebox_info,
                                    0);
    }

    return framebox_type;
}

static void
xfce_framebox_class_init (XfceFrameboxClass * class)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (class);
    GtkObjectClass *object_class;
    GtkWidgetClass *widget_class;

    object_class = (GtkObjectClass *) class;
    widget_class = (GtkWidgetClass *) class;
    gobject_class = G_OBJECT_CLASS (class);

    parent_class = gtk_type_class (gtk_frame_get_type ());
}

/**
 * xfce_framebox_new:
 * @text    : header text.
 * @indent  : if TRUE, the children will be indented.
 *
 * Creates a new #XfceFramebox container widget.
 *
 * Return value: the newly allocated #XfceFramebox container widget.
 **/
static
GtkWidget*
xfce_framebox_new (const gchar *text, gboolean indent)
{
    XfceFramebox *framebox;
    
    framebox = g_object_new (xfce_framebox_get_type (), NULL);

    gtk_frame_set_shadow_type (GTK_FRAME (framebox), GTK_SHADOW_NONE);

    if (text)
    {
        GtkWidget *label;
        gchar *tmp;

        tmp = g_strdup_printf ("<b>%s</b>", text);
        label = gtk_label_new (tmp);
        gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
        gtk_frame_set_label_widget (GTK_FRAME (framebox), label);
        gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
        gtk_widget_show (label);
        g_free (tmp);
    }

    framebox->hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (framebox->hbox);
    gtk_container_set_border_width (GTK_CONTAINER (framebox->hbox), 5);
    gtk_container_add (GTK_CONTAINER (framebox), framebox->hbox);
   
    if (indent)
    {
        framebox->indent_filler = gtk_alignment_new (0.5, 0.5, 1, 1);
        gtk_widget_show (framebox->indent_filler);
        gtk_box_pack_start (GTK_BOX (framebox->hbox), framebox->indent_filler, 
                            FALSE, FALSE, 0);
        gtk_widget_set_size_request (framebox->indent_filler, 16, -1);
    }
    
    return GTK_WIDGET (framebox);
}


/**
 * xfce_framebox_add:
 * @framebox  : an #XfceFramebox object.
 * @widget    : child widget to add to the @framebox container.
 *
 * Adds a child @widget to the @framebox container widget.
 **/
static
void
xfce_framebox_add (XfceFramebox *framebox, GtkWidget *widget)
{
    g_return_if_fail(framebox != NULL);
    g_return_if_fail(widget != NULL);
    g_return_if_fail(XFCE_IS_FRAMEBOX(framebox));
    
    gtk_box_pack_start (GTK_BOX (framebox->hbox), widget, TRUE, TRUE, 0);
}
