/* trigger_launcher plugin */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
       
#include <signal.h>

#include <gtk/gtk.h>
#include <libxfcegui4/libxfcegui4.h>

#include <libxfce4panel/xfce-panel-plugin.h>
#include "xfce-trigger-launcher-options.h"
              
#define PLUGIN_NAME "triggerlauncher"
#define TRIGGER_LAUNCHER_RC_GROUP "Trigger Launcher"

/* Panel Plugin Interface */

static void trigger_launcher_construct (XfcePanelPlugin *plugin);

XFCE_PANEL_PLUGIN_REGISTER_EXTERNAL(trigger_launcher_construct);

typedef enum {
  TRIGGER_LAUNCHER_STATE_UNKNOWN,
  TRIGGER_LAUNCHER_STATE_DISABLED,
  TRIGGER_LAUNCHER_STATE_ENABLED,
  TRIGGER_LAUNCHER_STATE_CHECKING
} TriggerLauncherState;

/* data */

typedef struct {
  gchar* enable_command;
  gchar* disable_command;
  gchar* poke_command;
  gchar* check_status_command;
  
  gchar* enabled_icon;
  gchar* disabled_icon;
  gchar* undefined_icon;
  gchar* checking_icon;
  gchar* dodgy_icon;
  
  gboolean config_dirty;
  
  /* ------------ */
  
  TriggerLauncherState state;
  
  guint check_timer;
  guint child_watch;
  GPid child_pid;
  
  guint check_interval; /* ms */
  
  /* ------------ */
  
  XfcePanelPlugin* plugin;

  int pixbuf_size;
  
  XfceIconbutton* button;
  GdkPixbuf* enabled_pixbuf;
  GdkPixbuf* disabled_pixbuf;
  GdkPixbuf* undefined_pixbuf;
  GdkPixbuf* checking_pixbuf;
  GdkPixbuf* overtime_pixbuf;
  
  GdkPixbuf* fallback_pixbuf;
  
  XfceTriggerLauncherOptions* option_box;
  GtkDialog* dialog_1;
  
} TriggerLauncherData;

/* manage data */

static TriggerLauncherData* trigger_launcher_data_new(void)
{
    TriggerLauncherData* res;
  
    res = g_new0 (TriggerLauncherData, 1);
    res->check_interval = 15000;
    res->pixbuf_size = 1;
  
    return res;
}

static void free_pixbufs(TriggerLauncherData* data1)
{
    if (data1->overtime_pixbuf != NULL) {
        g_object_unref (G_OBJECT(data1->overtime_pixbuf));
        data1->overtime_pixbuf = NULL;
    }
  
    if (data1->checking_pixbuf != NULL) {
        g_object_unref (G_OBJECT(data1->checking_pixbuf));
        data1->checking_pixbuf = NULL;
    }
  
    if (data1->undefined_pixbuf != NULL) {
        g_object_unref (G_OBJECT(data1->undefined_pixbuf));
        data1->undefined_pixbuf = NULL;
    }
  
    if (data1->disabled_pixbuf != NULL) {
        g_object_unref (G_OBJECT(data1->disabled_pixbuf));
        data1->disabled_pixbuf = NULL;
    }
  
    if (data1->enabled_pixbuf != NULL) {
        g_object_unref (G_OBJECT(data1->enabled_pixbuf));
        data1->enabled_pixbuf = NULL;
    }
    
    if (data1->fallback_pixbuf != NULL) {
        g_object_unref (G_OBJECT(data1->fallback_pixbuf));
        data1->fallback_pixbuf = NULL;
    }
} 

static void load_pixbufs(TriggerLauncherData* data1)
{
    free_pixbufs (data1);
    
    
    data1->fallback_pixbuf = xfce_themed_icon_load ("xfce-unknown", data1->pixbuf_size);  
    data1->enabled_pixbuf = gdk_pixbuf_new_from_file_at_size (data1->enabled_icon, 
                                                              data1->pixbuf_size, 
                                                              data1->pixbuf_size,
                                                              NULL);
                                                              
    data1->disabled_pixbuf = gdk_pixbuf_new_from_file_at_size (data1->disabled_icon,  
                                                               data1->pixbuf_size, 
                                                               data1->pixbuf_size, 
                                                               NULL);
                                                              
    data1->undefined_pixbuf = gdk_pixbuf_new_from_file_at_size (data1->undefined_icon, 
                                                                data1->pixbuf_size, 
                                                                data1->pixbuf_size, 
                                                                NULL);
                                                                
    data1->checking_pixbuf = gdk_pixbuf_new_from_file_at_size (data1->checking_icon, 
                                                               data1->pixbuf_size, 
                                                               data1->pixbuf_size, 
                                                               NULL);
                                                               
    data1->overtime_pixbuf = gdk_pixbuf_new_from_file_at_size (data1->dodgy_icon, 
                                                               data1->pixbuf_size, 
                                                               data1->pixbuf_size, 
                                                               NULL);
          
}

static void kill_child(TriggerLauncherData* data1)
{
    int status;
    
    if (data1->child_watch != 0) {
        if (waitpid (data1->child_pid, &status, WUNTRACED) != -1) {
            if (WIFSTOPPED(status)) {
                kill (data1->child_pid, SIGKILL);
            }
        }
        g_spawn_close_pid (data1->child_pid);
        if (data1->child_watch != 0) {
            g_source_remove (data1->child_watch);
            data1->child_watch = 0;
        }
    }

}

static void trigger_launcher_data_free(TriggerLauncherData* data1)
{
  if (data1 != NULL) {
    if (data1->check_timer != 0) {
        g_source_remove (data1->check_timer);
        data1->check_timer = 0;
    }

    kill_child (data1);

    free_pixbufs (data1);

    g_free (data1->enable_command);
    g_free (data1->disable_command);
    g_free (data1->poke_command);
    g_free (data1->check_status_command);
    g_free (data1->enabled_icon);
    g_free (data1->disabled_icon);
    g_free (data1->undefined_icon);
    g_free (data1->checking_icon);
    g_free (data1->dodgy_icon);
    
    if (data1->option_box != NULL) {
        g_object_unref (G_OBJECT(data1->option_box));
        data1->option_box = NULL;
    }

    gtk_widget_destroy (GTK_WIDGET(data1->dialog_1));
    
    g_free (data1);
  }
}

/* internal functions */

static void trigger_launcher_orientation_changed_cb (
                     XfcePanelPlugin *plugin, 
                     GtkOrientation orientation, 
                     TriggerLauncherData* data1)
{
    /* data1->button */
    
}

static void trigger_launcher_free_data_cb (XfcePanelPlugin *plugin, TriggerLauncherData* data1)
{
    DBG ("Free data: %s", PLUGIN_NAME);
    /*gtk_main_quit ();*/
    
    trigger_launcher_data_free (data1);
}

static void trigger_launcher_do_action(TriggerLauncherData* data1, char const* command)
{
    gint child_argc;
    gchar** child_argv;
    GError* error;
    GPid child_pid;
                        
    error = NULL;
    child_argv = NULL;
    /* kill_child (data1); */
    

    if (command[0] != 0) {
        if (g_shell_parse_argv (command, &child_argc, &child_argv, &error)) {
            if (g_spawn_async (NULL, child_argv, NULL, 
                G_SPAWN_SEARCH_PATH,
                NULL, NULL, &child_pid, &error)
            ) {
            } else {
                g_error_free (error);
            }
            g_strfreev (child_argv);
        }
    }
    
    
}

static void update_gui(TriggerLauncherData* data1)
{
    GdkPixbuf* pixbuf_1;
    
    pixbuf_1 = NULL;

    switch (data1->state) {
    case TRIGGER_LAUNCHER_STATE_UNKNOWN:
        pixbuf_1 = data1->checking_pixbuf;
        break;
        
    case TRIGGER_LAUNCHER_STATE_DISABLED:
        pixbuf_1 = data1->disabled_pixbuf;
        break;
        
    case TRIGGER_LAUNCHER_STATE_ENABLED:
        pixbuf_1 = data1->enabled_pixbuf;
        break;
        
    case TRIGGER_LAUNCHER_STATE_CHECKING:
        pixbuf_1 = data1->overtime_pixbuf;
        break;
    }
    
    if (pixbuf_1 == NULL) {
        pixbuf_1 = data1->fallback_pixbuf;
    }
    
    xfce_iconbutton_set_pixbuf (data1->button, pixbuf_1);
    
}

static void trigger_launcher_check_child_died_cb(GPid pid, gint status, TriggerLauncherData* data1)
{
    if (WIFEXITED(status)) {
        if (WEXITSTATUS(status) != 0) {
            if (WEXITSTATUS(status) == 255) {
                /* child is allowed to return 255 to signal "unknown" state 
                   (i.e. on timeout etc) */
                data1->state = TRIGGER_LAUNCHER_STATE_UNKNOWN;
            } else {
                data1->state = TRIGGER_LAUNCHER_STATE_DISABLED;
            }
        } else {
            data1->state = TRIGGER_LAUNCHER_STATE_ENABLED;
        }
    } else { /* crash etc */
        data1->state = TRIGGER_LAUNCHER_STATE_UNKNOWN;
    }
    
    update_gui (data1);
    
    g_spawn_close_pid (pid);
}

static gboolean trigger_launcher_check_cb(TriggerLauncherData* data1)
{
    gint child_argc;
    gchar** child_argv;
    GError* error;
    error = NULL;
    
    child_argv = NULL;

    if (data1->state == TRIGGER_LAUNCHER_STATE_CHECKING) {
        /* still checking... this means, the check program got stuck.
           so signal that to the user. */
           
        data1->state = TRIGGER_LAUNCHER_STATE_UNKNOWN;
           
        update_gui (data1);
        return TRUE;
    }
    
    data1->state = TRIGGER_LAUNCHER_STATE_CHECKING;
    
    if (data1->check_status_command[0] != 0) {
        if (g_shell_parse_argv (data1->check_status_command, &child_argc, &child_argv, &error)) {
            if (g_spawn_async (NULL, child_argv, NULL, 
                G_SPAWN_DO_NOT_REAP_CHILD|G_SPAWN_SEARCH_PATH|G_SPAWN_STDOUT_TO_DEV_NULL,
                NULL, NULL, &data1->child_pid, &error)
            ) {
                if (data1->child_watch != 0) {
                    g_source_remove (data1->child_watch);
                    data1->child_watch = 0;
                }
                
                data1->child_watch = g_child_watch_add (data1->child_pid, 
                    (GChildWatchFunc) trigger_launcher_check_child_died_cb, 
                    data1);
            } else {
                g_error_free (error);
                error = NULL;
                data1->state = TRIGGER_LAUNCHER_STATE_UNKNOWN;
            }
            
            g_strfreev (child_argv);
        } else {
            g_error_free (error);
            error = NULL;
            data1->state = TRIGGER_LAUNCHER_STATE_UNKNOWN;
        }
    } else {
        data1->state = TRIGGER_LAUNCHER_STATE_UNKNOWN;
    }
    
    update_gui (data1);

    return TRUE;
}


static void trigger_launcher_read_config(XfcePanelPlugin* plugin, TriggerLauncherData* data1)
{
    XfceRc* rc;
    gchar* path;
    
    kill_child (data1);

    if (data1->check_timer != 0) {
        g_source_remove (data1->check_timer);
        data1->check_timer = 0;
    }
    
    g_free (data1->enable_command);
    g_free (data1->disable_command);
    g_free (data1->poke_command);
    g_free (data1->check_status_command);
    g_free (data1->enabled_icon);
    g_free (data1->disabled_icon);
    g_free (data1->undefined_icon);
    g_free (data1->checking_icon);
    g_free (data1->dodgy_icon);
    
    data1->config_dirty = FALSE;
    
    data1->enable_command = NULL;
    data1->disable_command = NULL;
    data1->poke_command = NULL;
    data1->check_status_command = NULL;
    data1->enabled_icon = NULL;
    data1->disabled_icon = NULL;
    data1->undefined_icon = NULL;
    data1->checking_icon = NULL;
    data1->dodgy_icon = NULL;

    data1->check_interval = 15000;
     
    rc = NULL;
    path = xfce_panel_plugin_lookup_rc_file (plugin);
                
    if (path != NULL) {
        rc = xfce_rc_simple_open (path, TRUE);
    }
    
    if (rc != NULL) {
        xfce_rc_set_group (rc, TRIGGER_LAUNCHER_RC_GROUP);

        data1->enable_command = g_strdup (xfce_rc_read_entry (rc, "enable_command", ""));
        data1->disable_command = g_strdup (xfce_rc_read_entry (rc, "disable_command", ""));
        data1->poke_command = g_strdup (xfce_rc_read_entry (rc, "poke_command", ""));
        data1->check_status_command = g_strdup (xfce_rc_read_entry (rc, "check_status_command", ""));
        data1->enabled_icon = g_strdup (xfce_rc_read_entry (rc, "enabled_icon", ""));
        data1->disabled_icon = g_strdup (xfce_rc_read_entry (rc, "disabled_icon", ""));
        data1->undefined_icon = g_strdup (xfce_rc_read_entry (rc, "undefined_icon", ""));
        data1->checking_icon = g_strdup (xfce_rc_read_entry (rc, "checking_icon", ""));
        data1->dodgy_icon = g_strdup (xfce_rc_read_entry (rc, "dodgy_icon", ""));
        data1->check_interval = xfce_rc_read_int_entry (rc, "check_interval", 15000);
        
        xfce_rc_close (rc);
    }

    g_free (path);
    
    if (data1->enable_command == NULL) {
        data1->enable_command = g_strdup (""); 
    }
    
    if (data1->disable_command == NULL) {
        data1->disable_command = g_strdup ("");
    }
    
    if (data1->poke_command == NULL) {
        data1->poke_command = g_strdup ("");
    }
    
    if (data1->check_status_command == NULL) {
        data1->check_status_command = g_strdup ("");
    }
    
    if (data1->enabled_icon == NULL) {
        data1->enabled_icon = g_strdup ("");
    }
    
    if (data1->disabled_icon == NULL) {
        data1->disabled_icon = g_strdup ("");
    }
    
    if (data1->undefined_icon == NULL) {
        data1->undefined_icon = g_strdup ("");
    }

    if (data1->checking_icon == NULL) {
        data1->checking_icon = g_strdup ("");
    }
    
    if (data1->dodgy_icon == NULL) {
        data1->dodgy_icon = g_strdup ("");
    }

    load_pixbufs (data1);
    
    data1->state = TRIGGER_LAUNCHER_STATE_UNKNOWN;
    update_gui (data1);
    
    if (trigger_launcher_check_cb (data1)) {
        data1->check_timer = g_timeout_add (data1->check_interval, 
           (GSourceFunc) trigger_launcher_check_cb, data1);
    }
}

static void 
trigger_launcher_save_cb (XfcePanelPlugin *plugin, TriggerLauncherData* data1)
{
    char *file;
    XfceRc *rc;
    
    DBG ("Save: %s", PLUGIN_NAME);
    
    if (!data1->config_dirty) {
        /* not modified */
        
        return;
    }
    

    if (!(file = xfce_panel_plugin_save_location (plugin, TRUE)))
        return;

    rc = xfce_rc_simple_open (file, FALSE);

    if (rc == NULL) {
        g_warning ("trigger_launcher: Could not save to \"%s\"", file);
        g_free (file);
        
        return;
    }
    
    g_free (file);
    
    
    xfce_rc_set_group (rc, TRIGGER_LAUNCHER_RC_GROUP);
       
    xfce_rc_write_entry (rc, "enable_command", data1->enable_command);
    xfce_rc_write_entry (rc, "disable_command", data1->disable_command);
    xfce_rc_write_entry (rc, "poke_command", data1->poke_command);
    xfce_rc_write_entry (rc, "check_status_command", data1->check_status_command);
    xfce_rc_write_entry (rc, "enabled_icon", data1->enabled_icon);
    xfce_rc_write_entry (rc, "disabled_icon", data1->disabled_icon);
    xfce_rc_write_entry (rc, "undefined_icon", data1->undefined_icon);
    xfce_rc_write_entry (rc, "checking_icon", data1->checking_icon);
    xfce_rc_write_entry (rc, "dodgy_icon", data1->dodgy_icon);
    xfce_rc_write_int_entry (rc, "check_interval", data1->check_interval);
        
    /*
    xfce_rc_write_bool_entry (rc, "bool", TRUE);
    */

    xfce_rc_close (rc);
    data1->config_dirty = FALSE;
}

static void
trigger_launcher_configure_cb (XfcePanelPlugin *plugin, TriggerLauncherData* data1)
{
    DBG ("Configure: %s", PLUGIN_NAME);

    g_object_set (G_OBJECT(data1->option_box),
                  "check-status-command",
                  data1->check_status_command,
                  "enable-command",
                  data1->enable_command,
                  "disable-command",
                  data1->disable_command,
                  "poke-command",
                  data1->poke_command,
                  "disabled-icon-path",
                  data1->disabled_icon,
                  "enabled-icon-path",
                  data1->enabled_icon,
                  "undefined-icon-path",
                  data1->undefined_icon,
                  "dodgy-icon-path",
                  data1->dodgy_icon,
                  "check-interval",
                  data1->check_interval,
/* TODO  gchar* checking_icon;*/
                  NULL);
    
    gtk_widget_show (GTK_WIDGET(data1->option_box));
    gtk_widget_show (GTK_WIDGET(data1->dialog_1));
    
    gtk_window_set_title (GTK_WINDOW(data1->dialog_1),
      xfce_panel_plugin_get_display_name (plugin)
    );
    
}

static gboolean 
trigger_launcher_set_size_cb (XfcePanelPlugin *plugin, int size, TriggerLauncherData* data1)
{
    DBG ("Set size to %d: %s", size, PLUGIN_NAME);

    if (xfce_panel_plugin_get_orientation (plugin) == 
        GTK_ORIENTATION_HORIZONTAL)
    {
        gtk_widget_set_size_request (GTK_WIDGET (plugin), -1, size);
    }
    else
    {
        gtk_widget_set_size_request (GTK_WIDGET (plugin), size, -1);
    }

    data1->pixbuf_size = size  - 2 - 2 * MAX (GTK_WIDGET(data1->button)->style->xthickness,
                                              GTK_WIDGET(data1->button)->style->ythickness);
                                             
    load_pixbufs (data1);
    update_gui (data1);

    return TRUE;
}

static gboolean
trigger_launcher_configure_delete_event_cb(GtkWidget* widget, GdkEvent* event, TriggerLauncherData* data1)
{
    g_free (data1->check_status_command);
    data1->check_status_command = NULL;
    
    g_free (data1->enable_command);
    data1->enable_command = NULL;
    
    g_free (data1->disable_command);
    data1->disable_command = NULL;
    
    g_free (data1->poke_command);
    data1->poke_command = NULL;
    
    g_free (data1->disabled_icon);
    data1->disabled_icon = NULL;
    
    g_free (data1->enabled_icon);
    data1->enabled_icon = NULL;
    
    g_free (data1->undefined_icon);
    data1->undefined_icon = NULL;

    g_object_get (G_OBJECT(data1->option_box),
                  "check-status-command",
                  &data1->check_status_command,
                  "enable-command",
                  &data1->enable_command,
                  "disable-command",
                  &data1->disable_command,
                  "poke-command",
                  &data1->poke_command,
                  "disabled-icon-path",
                  &data1->disabled_icon,
                  "enabled-icon-path",
                  &data1->enabled_icon,
                  "undefined-icon-path",
                  &data1->undefined_icon,
                  "dodgy-icon-path",
                  &data1->dodgy_icon,
                  "check-interval",
                  &data1->check_interval,
/* TODO  gchar* checking_icon;*/
                  NULL);

    gtk_widget_hide (widget);

    free_pixbufs (data1);
    load_pixbufs (data1);
    update_gui (data1);
    
    data1->config_dirty = TRUE;
    
    trigger_launcher_save_cb (data1->plugin, data1);

    /* restart timer (interval could have been changed) */
    
    if (data1->check_timer != 0) {
        g_source_remove (data1->check_timer);
        data1->check_timer = 0;
    }
    
   
    if (trigger_launcher_check_cb (data1)) {
        data1->check_timer = g_timeout_add (data1->check_interval, 
           (GSourceFunc) trigger_launcher_check_cb, data1);
    }
    
    return TRUE;
}

static void
trigger_launcher_configure_response_cb(GtkDialog* dialog_1, int response, TriggerLauncherData* data1)
{
    trigger_launcher_configure_delete_event_cb (GTK_WIDGET(dialog_1), NULL, data1);
}

static void
trigger_launcher_style_set_cb(XfcePanelPlugin *plugin, GtkStyle  *previous_style, TriggerLauncherData* data1)
{
    free_pixbufs (data1);
    load_pixbufs (data1);
    update_gui (data1);
}

static void
trigger_launcher_clicked_cb(GtkButton* button_1, TriggerLauncherData* data1)
{
    switch (data1->state) {
    case TRIGGER_LAUNCHER_STATE_UNKNOWN:
      trigger_launcher_do_action(data1, data1->poke_command);
      break;
      
    case TRIGGER_LAUNCHER_STATE_DISABLED:
      trigger_launcher_do_action(data1, data1->enable_command);
      break;
      
    case TRIGGER_LAUNCHER_STATE_ENABLED:
      trigger_launcher_do_action(data1, data1->disable_command);
      break;
      
    case TRIGGER_LAUNCHER_STATE_CHECKING:
      /* bad timing... */
      trigger_launcher_do_action(data1, data1->poke_command);
      break;
      
    default:
      trigger_launcher_do_action(data1, data1->poke_command);
      break;
    }
}

/* create widgets and connect to signals */ 

static void 
trigger_launcher_construct (XfcePanelPlugin *plugin)
{
    TriggerLauncherData* data1;
    
    xfce_textdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8"); 

    DBG ("Construct: %s", PLUGIN_NAME);
    
    DBG ("Properties: size = %d, panel_position = %d", 
         xfce_panel_plugin_get_size (plugin),
         xfce_panel_plugin_get_screen_position (plugin));

    data1 = trigger_launcher_data_new ();
    
    data1->plugin = plugin;

    data1->button = XFCE_ICONBUTTON(xfce_iconbutton_new ());
    data1->option_box = xfce_trigger_launcher_options_new ();
    gtk_widget_show (GTK_WIDGET(data1->option_box));

    g_object_ref (G_OBJECT(data1->option_box));
    /*gtk_object_sink (GTK_OBJECT(data1->option_box));*/
    
    data1->dialog_1 = GTK_DIALOG(xfce_titled_dialog_new_with_buttons (
      xfce_panel_plugin_get_display_name (plugin),
      GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (plugin))),
      GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
      GTK_STOCK_CLOSE, GTK_RESPONSE_OK,
      NULL));
      
    g_signal_connect (G_OBJECT(data1->dialog_1), 
      "delete-event", G_CALLBACK(trigger_launcher_configure_delete_event_cb), data1);
    
    g_signal_connect (G_OBJECT(data1->dialog_1), 
      "response", G_CALLBACK(trigger_launcher_configure_response_cb), data1);
    
    g_signal_connect (G_OBJECT(data1->button), 
      "clicked", G_CALLBACK(trigger_launcher_clicked_cb), data1);
    
    
    gtk_box_pack_start (GTK_BOX(data1->dialog_1->vbox), GTK_WIDGET(data1->option_box), TRUE, TRUE, 7);

    /*xfce_panel_plugin_get_display_name (plugin);*/
    
    gtk_widget_show (GTK_WIDGET(data1->button));
    /*gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_HALF);*/

    /*GtkOrientation orientation = xfce_panel_plugin_get_orientation (plugin);*/
    
    gtk_container_add (GTK_CONTAINER (plugin), GTK_WIDGET(data1->button));

    xfce_panel_plugin_add_action_widget (plugin, GTK_WIDGET(data1->button));

    g_signal_connect (plugin, "style-set", 
                      G_CALLBACK (trigger_launcher_style_set_cb),
                      data1);    

    g_signal_connect (plugin, "orientation-changed", 
                      G_CALLBACK (trigger_launcher_orientation_changed_cb), 
                      data1);

    g_signal_connect (plugin, "free-data", 
                      G_CALLBACK (trigger_launcher_free_data_cb), data1);

    trigger_launcher_read_config (plugin, data1);
    
    g_signal_connect (plugin, "save", 
                      G_CALLBACK (trigger_launcher_save_cb), data1);

    g_signal_connect (plugin, "size-changed", 
                      G_CALLBACK (trigger_launcher_set_size_cb), data1);

    xfce_panel_plugin_menu_show_configure (plugin);
    g_signal_connect (plugin, "configure-plugin", 
                      G_CALLBACK (trigger_launcher_configure_cb), data1);
}

