/*
	Copyright (C) 2004 - 2006 rt2x00 SourceForge Project
	<http://rt2x00.serialmonkey.com>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Module: rt2x00
	Abstract: rt2x00 global information.
	Supported chipsets: RT2460, RT2560, RT2570,
	rt2561, rt2561s, rt2661 & rt2573.
 */

#ifndef RT2X00_H
#define RT2X00_H

/*
 * Module information.
 */
#ifndef DRV_NAME
#define DRV_NAME			"rt2x00"
#endif /* DRV_NAME */
#define DRV_VERSION			"CVS"
#define DRV_RELDATE			"N/A"
#define DRV_PROJECT			"http://rt2x00.serialmonkey.com"

/*
 * Debug definitions.
 * Debug output has to be enabled during compile time,
 * and should be switched on using the module parameter.
 */
#ifdef CONFIG_RT2X00_DEBUG
/*
 * Module parameter.
 */
static int rt2x00_debug_level = 0;

#define DEBUG_PRINTK(__message...) \
	do { if (rt2x00_debug_level) printk(__message); } while (0)
#else /* CONFIG_RT2X00_DEBUG */
#define DEBUG_PRINTK(__message...) \
	do { } while (0)

#endif /* CONFIG_RT2X00_DEBUG */

/*
 * Hardware button support (through radiobtn)
 * The delay between each poll can be set by the user
 * using the module parameter. Default value is 0,
 * which means polling is disabled.
 */
#ifdef CONFIG_RT2X00_BUTTON
/*
 * Module parameter.
 */
static short rt2x00_poll_delay = 0;

#include <linux/radiobtn.h>
#endif /* CONFIG_RT2X00_BUTTON */

/*
 * Various debug levels.
 * The debug levels PANIC and ERROR both indicate serious problems,
 * for this reason they should never be ignored.
 */
#define PANIC(__message, __args...) \
	printk(KERN_PANIC DRV_NAME "->%s: Panic - " __message, \
		__FUNCTION__, ##__args);
#define ERROR(__message, __args...) \
	printk(KERN_ERR DRV_NAME "->%s: Error - " __message, \
		__FUNCTION__, ##__args);
#define WARNING(__message, __args...) \
	DEBUG_PRINTK(KERN_WARNING DRV_NAME "->%s: Warning - " __message, \
		__FUNCTION__, ##__args);
#define NOTICE(__message, __args...) \
	DEBUG_PRINTK(KERN_NOTICE DRV_NAME "->%s: Notice - " __message, \
		__FUNCTION__, ##__args);
#define INFO(__message, __args...) \
	DEBUG_PRINTK(KERN_INFO DRV_NAME "->%s: Info - " __message, \
		__FUNCTION__, ##__args);
#define DEBUG(__message, __args...) \
	DEBUG_PRINTK(KERN_DEBUG DRV_NAME "->%s: Debug - " __message, \
		__FUNCTION__, ##__args);

/*
 * Ring sizes.
 * Ralink PCI devices demand the Frame size to be a multiple of 128 bytes.
 * DATA_FRAME_SIZE is used for TX, RX, ATIM and PRIO rings.
 * MGMT_FRAME_SIZE is used for the BEACON ring.
 */
#define DATA_FRAME_SIZE			2432
#define MGMT_FRAME_SIZE			256

/*
 * Number of entries in a packet ring.
 */
#define RX_ENTRIES			12
#define TX_ENTRIES			12
#define ATIM_ENTRIES			1
#define BEACON_ENTRIES			1

/*
 * Flag handlers
 */
#define SET_FLAG(__dev, __flag)		( (__dev)->flags |= (__flag) )
#define GET_FLAG(__dev, __flag)		( !!((__dev)->flags & (__flag)) )
#define CLEAR_FLAG(__dev, __flag)	( (__dev)->flags &= ~(__flag) )
#define CLEAR_FLAGS(__dev)		( (__dev)->flags = 0 )

/*
 * Device flags
 */
#define DEVICE_ENABLED_RADIO		0x00000001
#define DEVICE_INITIALIZED		0x00000002
#define DEVICE_INITIALIZED_HW		0x00000004
#define DEVICE_INITIALIZED_MAC		0x00000008
#define FIRMWARE_LOADED			0x00000010
#define FIRMWARE_FAILED			0x00000020
#define INTERFACE_INITIALIZED		0x00000040
#define INTERFACE_ENABLED_MONITOR	0x00000080
#define INTERFACE_ENABLED_PROMISC	0x00000100
#define DEVICE_SUPPORT_ATIM		0x00000200
#define DEVICE_SUPPORT_HW_BUTTON	0x00000400
#define CONFIG_FRAME_TYPE		0x00000800
#define CONFIG_RF_SEQUENCE		0x00001000
#define CONFIG_EXTERNAL_LNA		0x00002000
#define CONFIG_EXTERNAL_LNA_A		0x00004000
#define CONFIG_EXTERNAL_LNA_BG		0x00008000
#define CONFIG_DOUBLE_ANTENNA		0x00010000
#define CONFIG_DISABLE_BBP_TUNING	0x00020000

/*
 * TX result flags.
 */
enum TX_STATUS {
	TX_SUCCESS			= 0,
	TX_SUCCESS_RETRY		= 1,
	TX_FAIL_RETRY			= 2,
	TX_FAIL_INVALID			= 3,
	TX_FAIL_OTHER			= 4,
};

/*
 * Led mode values.
 */
enum led_mode {
	LED_MODE_DEFAULT		= 0,
	LED_MODE_TXRX_ACTIVITY 		= 1,
	LED_MODE_SIGNAL_STRENGTH	= 2,
	LED_MODE_ASUS			= 3,
	LED_MODE_ALPHA			= 4,
};

/*
 * Device states
 */
enum dev_state {
	STATE_DEEP_SLEEP		= 0,
	STATE_SLEEP			= 1,
	STATE_STANDBY			= 2,
	STATE_AWAKE			= 3,
};

/*
 * Macros for determining which is the lowest or highest bit
 * set in a 16 or 32 bit variable.
 */
#define BIT_SET(__val, __bit)			(__val & (1 << __bit))

#define BIT_OK(__val, __bit, __low, __high) \
	(__bit < __low ? 1 : \
	(__bit > __high ? 1 : \
	BIT_SET(__val, __bit) ? 1 : 0))

#define LOWEST_BIT16(__val) \
	(BIT_SET(__val, 0) ? 0 : (BIT_SET(__val, 1) ? 1 : \
	(BIT_SET(__val, 2) ? 2 : (BIT_SET(__val, 3) ? 3 : \
	(BIT_SET(__val, 4) ? 4 : (BIT_SET(__val, 5) ? 5 : \
	(BIT_SET(__val, 6) ? 6 : (BIT_SET(__val, 7) ? 7 : \
	(BIT_SET(__val, 8) ? 8 : (BIT_SET(__val, 9) ? 9 : \
	(BIT_SET(__val, 10) ? 10 : (BIT_SET(__val, 11) ? 11 : \
	(BIT_SET(__val, 12) ? 12 : (BIT_SET(__val, 13) ? 13 : \
	(BIT_SET(__val, 14) ? 14 : (BIT_SET(__val, 15) ? 15 : \
	-EINVAL))))))))))))))))

#define LOWEST_BIT32(__val) \
	(BIT_SET(__val, 0) ? 0 : (BIT_SET(__val, 1) ? 1 : \
	(BIT_SET(__val, 2) ? 2 : (BIT_SET(__val, 3) ? 3 : \
	(BIT_SET(__val, 4) ? 4 : (BIT_SET(__val, 5) ? 5 : \
	(BIT_SET(__val, 6) ? 6 : (BIT_SET(__val, 7) ? 7 : \
	(BIT_SET(__val, 8) ? 8 : (BIT_SET(__val, 9) ? 9 : \
	(BIT_SET(__val, 10) ? 10 : (BIT_SET(__val, 11) ? 11 : \
	(BIT_SET(__val, 12) ? 12 : (BIT_SET(__val, 13) ? 13 : \
	(BIT_SET(__val, 14) ? 14 : (BIT_SET(__val, 15) ? 15 : \
	(BIT_SET(__val, 16) ? 16 : (BIT_SET(__val, 17) ? 17 : \
	(BIT_SET(__val, 18) ? 18 : (BIT_SET(__val, 19) ? 19 : \
	(BIT_SET(__val, 20) ? 20 : (BIT_SET(__val, 21) ? 21 : \
	(BIT_SET(__val, 22) ? 22 : (BIT_SET(__val, 23) ? 23 : \
	(BIT_SET(__val, 24) ? 24 : (BIT_SET(__val, 25) ? 25 : \
	(BIT_SET(__val, 26) ? 27 : (BIT_SET(__val, 27) ? 27 : \
	(BIT_SET(__val, 28) ? 28 : (BIT_SET(__val, 29) ? 29 : \
	(BIT_SET(__val, 30) ? 30 : (BIT_SET(__val, 31) ? 31 : \
	-EINVAL))))))))))))))))))))))))))))))))

#define HIGHEST_BIT16(__val) \
	(BIT_SET(__val, 15) ? 15 : (BIT_SET(__val, 14) ? 14 : \
	(BIT_SET(__val, 13) ? 13 : (BIT_SET(__val, 12) ? 12 : \
	(BIT_SET(__val, 11) ? 11 : (BIT_SET(__val, 10) ? 10 : \
	(BIT_SET(__val, 9) ? 9 : (BIT_SET(__val, 8) ? 8 : \
	(BIT_SET(__val, 7) ? 7 : (BIT_SET(__val, 6) ? 6 : \
	(BIT_SET(__val, 5) ? 5 : (BIT_SET(__val, 4) ? 4 : \
	(BIT_SET(__val, 3) ? 3 : (BIT_SET(__val, 2) ? 2 : \
	(BIT_SET(__val, 1) ? 1 : (BIT_SET(__val, 0) ? 0 : \
	-EINVAL))))))))))))))))

#define HIGHEST_BIT32(__val) \
	(BIT_SET(__val, 31) ? 31 : (BIT_SET(__val, 30) ? 30 : \
	(BIT_SET(__val, 29) ? 29 : (BIT_SET(__val, 28) ? 28 : \
	(BIT_SET(__val, 27) ? 27 : (BIT_SET(__val, 26) ? 26 : \
	(BIT_SET(__val, 25) ? 25 : (BIT_SET(__val, 24) ? 24 : \
	(BIT_SET(__val, 23) ? 23 : (BIT_SET(__val, 22) ? 22 : \
	(BIT_SET(__val, 21) ? 21 : (BIT_SET(__val, 20) ? 20 : \
	(BIT_SET(__val, 19) ? 19 : (BIT_SET(__val, 18) ? 18 : \
	(BIT_SET(__val, 17) ? 17 : (BIT_SET(__val, 16) ? 16 : \
	(BIT_SET(__val, 15) ? 15 : (BIT_SET(__val, 14) ? 14 : \
	(BIT_SET(__val, 13) ? 13 : (BIT_SET(__val, 12) ? 12 : \
	(BIT_SET(__val, 11) ? 11 : (BIT_SET(__val, 10) ? 10 : \
	(BIT_SET(__val, 9) ? 9 : (BIT_SET(__val, 8) ? 8 : \
	(BIT_SET(__val, 7) ? 7 : (BIT_SET(__val, 6) ? 6 : \
	(BIT_SET(__val, 5) ? 5 : (BIT_SET(__val, 4) ? 4 : \
	(BIT_SET(__val, 3) ? 3 : (BIT_SET(__val, 2) ? 2 : \
	(BIT_SET(__val, 1) ? 1 : (BIT_SET(__val, 0) ? 0 : \
	-EINVAL))))))))))))))))))))))))))))))))

#define BITRANGE_OK16(__val, __low, __high) \
	((BIT_OK(__val, 0, __low, __high) \
	&& BIT_OK(__val, 1, __low, __high) \
	&& BIT_OK(__val, 2, __low, __high) \
	&& BIT_OK(__val, 3, __low, __high) \
	&& BIT_OK(__val, 4, __low, __high) \
	&& BIT_OK(__val, 5, __low, __high) \
	&& BIT_OK(__val, 6, __low, __high) \
	&& BIT_OK(__val, 7, __low, __high) \
	&& BIT_OK(__val, 8, __low, __high) \
	&& BIT_OK(__val, 9, __low, __high) \
	&& BIT_OK(__val, 10, __low, __high) \
	&& BIT_OK(__val, 11, __low, __high) \
	&& BIT_OK(__val, 12, __low, __high) \
	&& BIT_OK(__val, 13, __low, __high) \
	&& BIT_OK(__val, 14, __low, __high) \
	&& BIT_OK(__val, 15, __low, __high)) ? \
	0 : -EINVAL)

#define BITRANGE_OK32(__val, __low, __high) \
	((BIT_OK(__val, 0, __low, __high) \
	&& BIT_OK(__val, 1, __low, __high) \
	&& BIT_OK(__val, 2, __low, __high) \
	&& BIT_OK(__val, 3, __low, __high) \
	&& BIT_OK(__val, 4, __low, __high) \
	&& BIT_OK(__val, 5, __low, __high) \
	&& BIT_OK(__val, 6, __low, __high) \
	&& BIT_OK(__val, 7, __low, __high) \
	&& BIT_OK(__val, 8, __low, __high) \
	&& BIT_OK(__val, 9, __low, __high) \
	&& BIT_OK(__val, 10, __low, __high) \
	&& BIT_OK(__val, 11, __low, __high) \
	&& BIT_OK(__val, 12, __low, __high) \
	&& BIT_OK(__val, 13, __low, __high) \
	&& BIT_OK(__val, 14, __low, __high) \
	&& BIT_OK(__val, 15, __low, __high) \
	&& BIT_OK(__val, 16, __low, __high) \
	&& BIT_OK(__val, 17, __low, __high) \
	&& BIT_OK(__val, 18, __low, __high) \
	&& BIT_OK(__val, 19, __low, __high) \
	&& BIT_OK(__val, 20, __low, __high) \
	&& BIT_OK(__val, 21, __low, __high) \
	&& BIT_OK(__val, 22, __low, __high) \
	&& BIT_OK(__val, 23, __low, __high) \
	&& BIT_OK(__val, 24, __low, __high) \
	&& BIT_OK(__val, 25, __low, __high) \
	&& BIT_OK(__val, 26, __low, __high) \
	&& BIT_OK(__val, 27, __low, __high) \
	&& BIT_OK(__val, 28, __low, __high) \
	&& BIT_OK(__val, 29, __low, __high) \
	&& BIT_OK(__val, 30, __low, __high) \
	&& BIT_OK(__val, 31, __low, __high)) ? \
	0 : -EINVAL)

extern int error_lowest_bit_not_constant;
extern int error_highest_bit_not_constant;
extern int error_bitrange_not_constant;
extern int error_bitrange_bad;

#define BUILD_LOWEST_BIT16(__val) \
 	(!__builtin_constant_p(__val) ? error_lowest_bit_not_constant : \
	LOWEST_BIT16(__val))

#define BUILD_LOWEST_BIT32(__val) \
 	(!__builtin_constant_p(__val) ? error_lowest_bit_not_constant : \
	LOWEST_BIT32(__val))

#define BUILD_HIGHEST_BIT16(__val) \
 	(!__builtin_constant_p(__val) ? error_highest_bit_not_constant : \
	HIGHEST_BIT16(__val))

#define BUILD_HIGHEST_BIT32(__val) \
 	(!__builtin_constant_p(__val) ? error_highest_bit_not_constant : \
	HIGHEST_BIT32(__val))

#define BUILD_BITRANGE_OK16(__val, __low, __high) \
	((!__builtin_constant_p(__val) || !__builtin_constant_p(__low) \
	|| !__builtin_constant_p(__high)) ? error_bitrange_not_constant : \
	BITRANGE_OK16(__val, __low, __high))

#define BUILD_BITRANGE_OK32(__val, __low, __high) \
	((!__builtin_constant_p(__val) || !__builtin_constant_p(__low) \
	|| !__builtin_constant_p(__high)) ? error_bitrange_not_constant : \
	BITRANGE_OK16(__val, __low, __high))

/*
 * Register handlers.
 * We store the position of a register field inside a field structure,
 * This will simplify the process of setting and reading a certain field
 * inside the register while making sure the process remains byte order safe.
 * Before setting the value into the structure we use macros to determine
 * whether all bits in the field are contineous and valid.
 * These additional checks will be optimized away at compile time,
 * but do have a major impact on compile speed, therefor we only make this
 * check when compiling with debug enabled.
 */
struct rt2x00_field16{
	u16	bit_offset;
	u16	bit_mask;
};

struct rt2x00_field32{
	u32	bit_offset;
	u32	bit_mask;
};

/*
 * Before intitializing the rt2x00_field# structures,
 * we will check if the bitmask is correct and does
 * not contain any gaps.
 * This check is only done in debug mode, since it severely
 * impacts compilation speed.
 */
#ifdef CONFIG_RT2X00_DEBUG
#define FIELD16(__mask) \
	( (struct rt2x00_field16) { \
		(BUILD_BITRANGE_OK16(__mask, BUILD_LOWEST_BIT16(__mask), \
			BUILD_HIGHEST_BIT16(__mask)) == 0) ? \
		BUILD_LOWEST_BIT16(__mask) : error_bitrange_bad, \
		(__mask) \
	} )

#define FIELD32(__mask) \
	( (struct rt2x00_field32) { \
		(BUILD_BITRANGE_OK32(__mask, BUILD_LOWEST_BIT32(__mask), \
			BUILD_HIGHEST_BIT32(__mask)) == 0) ? \
		BUILD_LOWEST_BIT32(__mask) : error_bitrange_bad, \
		(__mask) \
	} )
#else /* CONFIG_RT2X00_DEBUG */
#define FIELD16(__mask) \
	( (struct rt2x00_field16) { \
		BUILD_LOWEST_BIT16(__mask), (__mask) \
	} )

#define FIELD32(__mask) \
	( (struct rt2x00_field32) { \
		BUILD_LOWEST_BIT32(__mask), (__mask) \
	} )
#endif /* CONFIG_RT2X00_DEBUG */

static inline void rt2x00_set_field32(u32 *reg,
	const struct rt2x00_field32 field, const u32 value)
{
	*reg &= cpu_to_le32(~(field.bit_mask));
	*reg |= cpu_to_le32((value << field.bit_offset) & field.bit_mask);
}

static inline void rt2x00_set_field32_nb(u32 *reg,
	const struct rt2x00_field32 field, const u32 value)
{
	*reg &= ~(field.bit_mask);
	*reg |= (value << field.bit_offset) & field.bit_mask;
}

static inline u32 rt2x00_get_field32(const u32 reg,
	const struct rt2x00_field32 field)
{
	return (le32_to_cpu(reg) & field.bit_mask) >> field.bit_offset;
}

static inline u32 rt2x00_get_field32_nb(const u32 reg,
	const struct rt2x00_field32 field)
{
	return (reg & field.bit_mask) >> field.bit_offset;
}

static inline void rt2x00_set_field16(u16 *reg,
	const struct rt2x00_field16 field, const u16 value)
{
	*reg &= cpu_to_le16(~(field.bit_mask));
	*reg |= cpu_to_le16((value << field.bit_offset) & field.bit_mask);
}

static inline void rt2x00_set_field16_nb(u16 *reg,
	const struct rt2x00_field16 field, const u16 value)
{
	*reg &= ~(field.bit_mask);
	*reg |= (value << field.bit_offset) & field.bit_mask;
}

static inline u16 rt2x00_get_field16(const u16 reg,
	const struct rt2x00_field16 field)
{
	return (le16_to_cpu(reg) & field.bit_mask) >> field.bit_offset;
}

static inline u16 rt2x00_get_field16_nb(const u16 reg,
	const struct rt2x00_field16 field)
{
	return (reg & field.bit_mask) >> field.bit_offset;
}

/*
 * Chipset identification
 * The chipset on the device is composed of a RT and RF chip.
 * The chipset combination is important for determining device capabilities.
 */
struct rt2x00_chip{
	u16				rt;
	u16				rf;
};

/*
 * Set chipset data.
 */
static inline void rt2x00_set_chip(struct rt2x00_chip *chipset,
	const u16 rt, const u16 rf)
{
	INFO("Chipset detected - rt: %04x, rf: %04x.\n", rt, rf);

	chipset->rt = rt;
	chipset->rf = rf;
}

static inline u32 rt2x00_get_chip_id(struct rt2x00_chip *chipset)
{
	return chipset->rt << 16 | chipset->rf;
}

static inline char rt2x00_rt(const struct rt2x00_chip *chipset, const u16 chip)
{
	return (chipset->rt == chip);
}

static inline char rt2x00_rf(const struct rt2x00_chip *chipset, const u16 chip)
{
	return (chipset->rf == chip);
}

/*
 * data_entry
 * The data ring is a list of data entries.
 * Each entry holds a reference to the descriptor
 * and the data buffer. For TX rings the reference to the
 * sk_buff of the packet being transmitted is also stored here.
 */
struct data_entry{
	/*
	 * Status flags
	 */
	unsigned int			flags;
#define ENTRY_OWNER_NIC			0x00000001

	/*
	 * Ring we belong to.
	 */
	struct data_ring		*ring;

	/*
	 * sk_buff for the packet which is being transmitted
	 * in this entry (Only used with TX related rings).
	 */
	struct sk_buff			*skb;

	/*
	 * Store a ieee80211_tx_status structure in each
	 * ring entry, this will optimize the txdone
	 * handler.
	 */
	struct ieee80211_tx_status	tx_status;

	/*
	 * private pointer specific to driver.
	 */
	void				*priv;

	/*
	 * Data address for this entry.
	 */
	void				*data_addr;
	dma_addr_t			data_dma;
};

/*
 * data_ring
 * Data rings are used by the device to send and receive packets.
 * The data_addr is the base address of the data memory.
 * To determine at which point in the ring we are,
 * have to use the rt2x00_ring_index_*() functions.
 */
struct data_ring{
	/*
	 * net_device where this ring belongs to.
	 */
	struct net_device			*net_dev;

	/*
	 * Work structure for bottom half interrupt handling.
	 */
	struct work_struct			irq_work;

	/*
	 * Base address for the device specific data entries.
	 */
	struct data_entry			*entry;

	/*
	 * TX queue statistic info.
	 */
	struct ieee80211_tx_queue_stats_data	stats;

	/*
	 * TX Queue parameters.
	 */
	struct ieee80211_tx_queue_params	tx_params;

	/*
	 * Base address for data ring.
	 */
	dma_addr_t				data_dma;
	void					*data_addr;

	/*
	 * Index variables.
	 */
	u8					index;
	u8					index_done;

	/*
	 * Ring type.
	 */
	u16					type;

	/*
	 * Size of packet and descriptor in bytes.
	 */
	u16					data_size;
	u16					desc_size;
};

/*
 * Handlers to determine the address of the current device specific
 * data entry, where either index or index_done points to.
 */
static inline struct data_entry* rt2x00_get_data_entry(
	struct data_ring *ring)
{
	return &ring->entry[ring->index];
}

static inline struct data_entry* rt2x00_get_data_entry_done(
	struct data_ring *ring)
{
	return &ring->entry[ring->index_done];
}

/*
 * Total ring memory
 */
static inline int rt2x00_get_ring_size(struct data_ring *ring)
{
	return ring->stats.limit * (ring->desc_size + ring->data_size);
}

/*
 * Ring index manipulation functions.
 */
static inline void rt2x00_ring_index_inc(struct data_ring *ring)
{
	ring->index++;
	if (ring->index >= ring->stats.limit)
		ring->index = 0;
	ring->stats.len++;
}

static inline void rt2x00_ring_index_done_inc(struct data_ring *ring)
{
	ring->index_done++;
	if (ring->index_done >= ring->stats.limit)
		ring->index_done = 0;
	ring->stats.len--;
	ring->stats.count++;
}

static inline void rt2x00_ring_index_clear(struct data_ring *ring)
{
	ring->index = 0;
	ring->index_done = 0;
	ring->stats.len = 0;
	ring->stats.count = 0;
}

static inline int rt2x00_ring_empty(struct data_ring *ring)
{
	return ring->stats.len == 0;
}

static inline int rt2x00_ring_full(struct data_ring *ring)
{
	return ring->stats.len == ring->stats.limit;
}

/*
 * Scanning structure.
 * Swithing channel during scanning will be put
 * in a workqueue so we will be able to sleep
 * during the switch.
 * We also make use of the completion structure
 * in case a frame must be send before a
 * channel switch.
 */
struct scanning{
	/*
	 * Completion structure if an packet needs to be send.
	 */
	struct completion			completion;

	/*
	 * Scanning parameters.
	 */
	struct ieee80211_scan_conf		conf;

	/*
	 * Scanning state: IEEE80211_SCAN_START or IEEE80211_SCAN_END.
	 */
	short					state;

	/*
	 * Flag to see if this scan has been cancelled.
	 */
	short					status;
#define SCANNING_READY				0x0001
#define SCANNING_CANCELLED			0x0002
};


/*
 * Device specific rate value.
 * We will have to create the device specific rate value
 * passed to the ieee80211 kernel. We need to make it a consist of
 * multiple fields because we want to store more then 1 device specific
 * values inside the value.
 *	1 - rate, stored as 100 kbit/s.
 *	2 - preamble, short_preamble enabled flag.
 *	3 - MASK_RATE, which rates are enabled in this mode, this mask
 *	corresponds with the TX register format for the current device.
 *	4 - plcp, 802.11b rates are device specific,
 *	802.11g rates are set according to the ieee802.11a-1999 p.14.
 * The bit to enable preamble is set in a seperate define.
 */
#define DEV_RATE			FIELD32(0x000007ff)
#define DEV_PREAMBLE			FIELD32(0x00000800)
#define DEV_RATEMASK			FIELD32(0x00fff000)
#define DEV_PLCP			FIELD32(0xff000000)

/*
 * Macro's for creating the device specific rate value.
 */
#define DEVICE_RATE_VALUE(__rate, __mask, __plcp) \
	(int) ((((__rate) << DEV_RATE.bit_offset) & DEV_RATE.bit_mask) \
	| (((__mask) << DEV_RATEMASK.bit_offset) & DEV_RATEMASK.bit_mask) \
	| (((__plcp) << DEV_PLCP.bit_offset) & DEV_PLCP.bit_mask) )

#define DEVICE_RATE_PREAMBLE(__value) \
	(int)( (__value) | (1 << DEV_PREAMBLE.bit_offset) )

/*
 * Macro for reading the device specific rate value.
 */
#define DEVICE_RATE_FIELD(__value, __mask) \
	(int)( ((__value) & DEV_##__mask.bit_mask) >> DEV_##__mask.bit_offset)

#define DEVICE_RATE_1MB			DEVICE_RATE_VALUE(10, 0x001, 0x00)
#define DEVICE_RATE_2MB			DEVICE_RATE_VALUE(20, 0x003, 0x01)
#define DEVICE_RATE_2MB_PREAMBLE	DEVICE_RATE_PREAMBLE(DEVICE_RATE_2MB)
#define DEVICE_RATE_55MB		DEVICE_RATE_VALUE(55, 0x007, 0x02)
#define DEVICE_RATE_55MB_PREAMBLE	DEVICE_RATE_PREAMBLE(DEVICE_RATE_55MB)
#define DEVICE_RATE_11MB		DEVICE_RATE_VALUE(110, 0x00f, 0x03)
#define DEVICE_RATE_11MB_PREAMBLE	DEVICE_RATE_PREAMBLE(DEVICE_RATE_11MB)
#define DEVICE_RATE_6MB			DEVICE_RATE_VALUE(60, 0x01f, 0x0b)
#define DEVICE_RATE_9MB			DEVICE_RATE_VALUE(90, 0x03f, 0x0f)
#define DEVICE_RATE_12MB		DEVICE_RATE_VALUE(120, 0x07f, 0x0a)
#define DEVICE_RATE_18MB		DEVICE_RATE_VALUE(180, 0x0ff, 0x0e)
#define DEVICE_RATE_24MB		DEVICE_RATE_VALUE(240, 0x1ff, 0x09)
#define DEVICE_RATE_36MB		DEVICE_RATE_VALUE(360, 0x3ff, 0x0d)
#define DEVICE_RATE_48MB		DEVICE_RATE_VALUE(480, 0x7ff, 0x08)
#define DEVICE_RATE_54MB		DEVICE_RATE_VALUE(540, 0xfff, 0x0c)

/*
 * Sequence number handlers.
 */
static inline int rt2x00_require_sequence(struct ieee80211_hdr *ieee80211hdr)
{
	/*
	 * Check if the packet has a header
	 * that contains a seq_ctrl field.
	 */
	if (ieee80211_get_hdrlen(
		le16_to_cpu(ieee80211hdr->frame_control)) >= 24)
		return 1;
	return 0;
}

static inline void rt2x00_set_sequence(struct sk_buff *skb, u16 *seq)
{
	struct ieee80211_hdr *ieee80211hdr =
		(struct ieee80211_hdr*)skb->data;

	/*
	 * Only attach the sequence number, when
	 * the buffer requires it.
	 */
	if (!rt2x00_require_sequence(ieee80211hdr))
		return;

	/*
	 * Set the sequence number for this frame,
	 * but do not change the fragment number.
	 */
	ieee80211hdr->seq_ctrl |= cpu_to_le16((*seq << 4) & 0xfff0);

	/*
	 * Since the dscape stack sends all packets and
	 * fragments individually, we need to check if there
	 * will be new fragments following.
	 * If this is the last or only fragment, the sequence
	 * counter should be increased.
	 */
	if (!ieee80211_get_morefrag(ieee80211hdr))
		*seq = (*seq + 1) & 0x0fff;
}

/*
 * Duration calculations
 * The rate variable passed is: 100kbs.
 * To convert from bytes to bits we multiply size with 8,
 * then the size is multiplied with 10 to make the
 * real rate -> rate argument correction.
 */
static inline u16 get_duration(const unsigned int size, const u8 rate)
{
	return ((size * 8 * 10) / rate);
}

static inline u16 get_duration_res(const unsigned int size, const u8 rate)
{
	return ((size * 8 * 10) % rate);
}

#define ACK_SIZE			14
#define IEEE80211_HEADER		24
#define PLCP				48
#define BEACON				100
#define SIFS				10
#define PREAMBLE			144
#define SHORT_PREAMBLE			72
#define SLOT_TIME			20
#define SHORT_SLOT_TIME			7

#endif /* RT2X00_H */
