/*
	Copyright (C) 2006 Ivo van Doorn

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Radio hardware button support
	Poll frequently all registered hardware for hardware button status,
	if changed enabled or disable the radio of that hardware device.
	Send signal to input device to inform userspace about the new status.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/input.h>
#include <radiobtn.h>

MODULE_AUTHOR("Ivo van Doorn <IvDoorn@gmail.com>");
MODULE_VERSION("1.0");
MODULE_DESCRIPTION("Radio hardware button support");
MODULE_LICENSE("GPL");

static void radiobtn_poll(unsigned long data)
{
	struct radio_button *radiobtn = (struct radio_button*)data;
	int state;

	/*
	 * Poll for the new state.
	 * Check if the state has changed.
	 */
	state = !!radiobtn->button_poll(radiobtn->data);
	if (state != radiobtn->current_state) {
		radiobtn->current_state = state;

		/*
		 * Enable or disable the radio when this
		 * should be done in software.
		 */
		if (state && radiobtn->enable_radio)
			radiobtn->enable_radio(radiobtn->data);
		else if (!state && radiobtn->disable_radio)
			radiobtn->disable_radio(radiobtn->data);

		/*
		 * Report key event.
		 */
		input_report_key(radiobtn->input_dev, KEY_RADIO, 1);
		input_sync(radiobtn->input_dev);
		input_report_key(radiobtn->input_dev, KEY_RADIO, 0);
		input_sync(radiobtn->input_dev);
	}

	/*
	 * Check if polling has been disabled.
	 */
	if (radiobtn->poll_delay != 0) {
		radiobtn->poll_timer.expires =
			jiffies + msecs_to_jiffies(radiobtn->poll_delay);
		add_timer(&radiobtn->poll_timer);
	}
}

int radiobtn_register_device(struct radio_button *radiobtn)
{
	int status;

	/*
	 * Check if all mandatory fields have been set.
	 */
	if (radiobtn->poll_delay == 0 || radiobtn->button_poll == NULL)
		return -EINVAL;

	/*
	 * Allocate, initialize and register input device.
	 */
	radiobtn->input_dev = input_allocate_device();
	if (!radiobtn->input_dev) {
		printk(KERN_ERR "Failed to allocate input device %s.\n",
			radiobtn->dev_name);
		return -ENOMEM;
	}

	radiobtn->input_dev->name = "Radio button";
	radiobtn->input_dev->phys = radiobtn->dev_name;
	radiobtn->input_dev->id.bustype = BUS_HOST;
	set_bit(KEY_RADIO, radiobtn->input_dev->keybit);

	status = input_register_device(radiobtn->input_dev);
	if (status) {
		printk(KERN_ERR "Failed to register input device %s.\n",
			radiobtn->dev_name);
		input_free_device(radiobtn->input_dev);
		return status;
	}

	/*
	 * Set the initial state of the button.
	 */
	radiobtn->current_state = radiobtn->button_poll(radiobtn->data);

	/*
	 * Initialize timer.
	 */
	init_timer(&radiobtn->poll_timer);
	radiobtn->poll_timer.function = radiobtn_poll;
	radiobtn->poll_timer.data = (unsigned long)radiobtn;
	radiobtn->poll_timer.expires =
		jiffies + msecs_to_jiffies(radiobtn->poll_delay);
	add_timer(&radiobtn->poll_timer);

	printk(KERN_INFO "Created new %s: %s.\n",
		radiobtn->input_dev->name, radiobtn->input_dev->phys);

	return 0;
}

void radiobtn_unregister_device(struct radio_button *radiobtn)
{
	/*
	 * Stop timer.
	 */
	radiobtn->poll_delay = 0;
	del_timer_sync(&radiobtn->poll_timer);

	/*
	 * Remove input device.
	 */
	input_unregister_device(radiobtn->input_dev);
	input_free_device(radiobtn->input_dev);
}

static int __init radiobtn_init(void)
{
	printk(KERN_INFO "Loading radio button driver.\n");
	return 0;
}

static void __exit radiobtn_exit(void)
{
	printk(KERN_INFO "Unloading radio button driver.\n");
}

EXPORT_SYMBOL(radiobtn_register_device);
EXPORT_SYMBOL(radiobtn_unregister_device);

module_init(radiobtn_init);
module_exit(radiobtn_exit);
