/*
 * Linux driver for ADMtek ADM8211 (IEEE 802.11b wireless LAN card)
 *
 * Copyright (c) 2003, Jouni Malinen <jkmaline@cc.hut.fi>
 * Copyright (c) 2004-2005, Michael Wu <flamingice@sourmilk.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 */

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/wireless.h>
#include <net/iw_handler.h>
#include <linux/random.h>
#include <linux/if_arp.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/pci.h>
#undef CONFIG_IEEE80211_DEBUG
#include <net/ieee80211.h>

#include "adm8211.h"
#include "adm8211_ioctl.h"
#include "ieee80211_sta.h"


static const long freq_list[] = { 2412, 2417, 2422, 2427, 2432, 2437, 2442,
				  2447, 2452, 2457, 2462, 2467, 2472, 2484 };
#define FREQ_COUNT ARRAY_SIZE(freq_list)

#if 0
#define SET_IWFREQ(freq, chan) \
        freq->m = freq_list[chan - 1] * 100000; \
        freq->e = 1;
#else
#define SET_IWFREQ(freq, chan) \
	freq->m = chan; \
        freq->e = 0;
#endif

static int adm8211_ioctl_giwname(struct net_device *dev,
				 struct iw_request_info *info,
				 char *name, char *extra)
{
	strcpy(name, "IEEE 802.11b");
	return 0;
}

static int adm8211_ioctl_siwfreq(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_freq *freq, char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	struct ieee80211_device *ieee = netdev_priv(dev);

	/* freq => chan. */
	if (freq->e == 1 &&
	    freq->m / 100000 >= freq_list[0] &&
	    freq->m / 100000 <= freq_list[FREQ_COUNT - 1]) {
		int ch;
		int fr = freq->m / 100000;
		for (ch = 0; ch < FREQ_COUNT; ch++) {
			if (fr == freq_list[ch]) {
				freq->e = 0;
				freq->m = ch + 1;
				break;
			}
		}
	}

	if (freq->e != 0 || freq->m < priv->ieee80211.chan_range.min || freq->m > priv->ieee80211.chan_range.max)
		return 0; /* -EINVAL we allow this for now so wpa_supplicant doesn't complain */

	if (dev->flags & IFF_UP) {
		if (ieee->iw_mode == IW_MODE_MONITOR)
			priv->ieee80211.set_channel(dev, freq->m);
		else if (ieee->state != IEEE80211_SHUTDOWN)
			ieee80211_start_scan(&priv->ieee80211);
	}

	priv->ieee80211.channel = priv->ieee80211.pref_channel = freq->m;

	return 0;
}


static int adm8211_ioctl_giwfreq(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_freq *freq, char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	int chan = priv->ieee80211.channel;

	if (chan < 1 || chan > FREQ_COUNT)
		return -EINVAL;

	SET_IWFREQ(freq, chan);

	return 0;
}


static int adm8211_ioctl_siwmode(struct net_device *dev,
				 struct iw_request_info *info,
				 __u32 *mode, char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	struct ieee80211_device *ieee = netdev_priv(dev);
	
	if (*mode != IW_MODE_INFRA &&
	    *mode != IW_MODE_ADHOC &&
	    *mode != IW_MODE_MONITOR)
		return -EOPNOTSUPP;

	if (*mode == ieee->iw_mode)
		return 0;

	ieee80211_stop(data);
	ieee->iw_mode = *mode;
	
	switch (*mode) {
	case IW_MODE_AUTO:
		ieee->iw_mode = IW_MODE_INFRA;
	case IW_MODE_INFRA:
		priv->ieee80211.capab &= ~WLAN_CAPABILITY_IBSS;
		dev->type = ARPHRD_ETHER;
		dev->hard_header_parse = priv->eth_header_parse;
		break;
	case IW_MODE_ADHOC:
		priv->ieee80211.capab |= WLAN_CAPABILITY_IBSS;
		dev->type = ARPHRD_ETHER;
		dev->hard_header_parse = priv->eth_header_parse;
		break;
	case IW_MODE_MONITOR:
		dev->type = ARPHRD_IEEE80211_PRISM;
		dev->hard_header_parse = adm8211_80211_header_parse;
		break;
	}

	if (dev->flags & IFF_UP) {
		adm8211_update_mode(dev);
		ieee80211_start_scan(data);
	}

	return 0;
}


static int adm8211_ioctl_giwmode(struct net_device *dev,
				 struct iw_request_info *info,
				 __u32 *mode, char *extra)
{
	struct ieee80211_device *ieee = netdev_priv(dev);
	*mode = ieee->iw_mode;
	return 0;
}

static int adm8211_ioctl_giwrange(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_point *dwrq,
				  char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	struct iw_range *range = (struct iw_range *) extra;
	unsigned int i, j;

        dwrq->length = sizeof(struct iw_range);
        memset(range, 0, sizeof(*range));

	range->min_nwid = range->max_nwid = 0;

	IW_EVENT_CAPA_SET_KERNEL(range->event_capa);
	IW_EVENT_CAPA_SET(range->event_capa, SIOCGIWTHRSPY);
	IW_EVENT_CAPA_SET(range->event_capa, SIOCGIWAP);
	IW_EVENT_CAPA_SET(range->event_capa, SIOCGIWSCAN);

	range->max_qual.qual  = 255;
	range->max_qual.level = 0;
	range->max_qual.noise = 0;

	range->avg_qual.qual  = 0;
	range->avg_qual.level = 0;
	range->avg_qual.noise = 0;

	range->num_bitrates = data->num_supp_rates;
	i = 0;
	while (i++ < min(data->num_supp_rates, IW_MAX_BITRATES))
		range->bitrate[i - 1] = (data->supp_rates[i - 1]*1000000)/2;

	range->min_rts = 0;
	range->max_rts = 2346;
	range->min_frag = MIN_FRAG_THRESHOLD;
	range->max_frag = MAX_FRAG_THRESHOLD;

	/* are the values really in dBm? */
	range->txpower_capa = IW_TXPOW_RANGE | IW_TXPOW_DBM;

	range->we_version_compiled = WIRELESS_EXT;
	range->we_version_source = 19;
	range->num_channels = data->chan_range.max - data->chan_range.min + 1;
	range->num_frequency = range->num_channels;

	range->max_encoding_tokens = WEP_KEYS;
	range->encoding_size[0] = 5;
	range->encoding_size[1] = 13;
	range->num_encoding_sizes = 2;

	j = 0;
	for (i = data->chan_range.min; i <= data->chan_range.max; i++) {
		range->freq[j].m = freq_list[i - 1] * 100000;
		range->freq[j].e = 1;
		range->freq[j].i = i;
		j++;
	}

	range->enc_capa = IW_ENC_CAPA_CIPHER_TKIP | IW_ENC_CAPA_CIPHER_CCMP;
	return 0;
}

/*
static int adm8211_ioctl_giwsens(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_param *vwrq, char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	struct ieee80211_bss *bss = ieee80211_get_bss(data, data->bssid);
	return 0;
}
*/

static int adm8211_ioctl_siwscan(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_point *dwrq, char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	struct ieee80211_device *ieee = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;

	if (ieee->iw_mode == IW_MODE_MONITOR)
		return -EOPNOTSUPP;

	/* check if a scan is in progress */
	if (ieee->state == IEEE80211_INITIALIZED)
		return 0;

	if ((priv->last_userscan + 3*HZ) > jiffies)
		return -EAGAIN;
	
	if (dev->flags & IFF_UP) {
		data->flags |= SCAN_NOTIFY;
		priv->last_userscan = jiffies;
		ieee80211_start_scan(&priv->ieee80211);
	} else
		return -1;

	return 0;
}

static int adm8211_ioctl_giwscan(struct net_device *dev,
				 struct iw_request_info *info,
				 union iwreq_data *wrqu, char *extra)
{
	return ieee80211_wx_get_scan(netdev_priv(dev), info, wrqu, extra);
}

static int adm8211_ioctl_giwessid(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_point *dwrq, char *essid)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);

	dwrq->flags = 1; /* active */
	dwrq->length = min(priv->ieee80211.ssid_len, (size_t)IW_ESSID_MAX_SIZE);
	memset(essid, 0, IW_ESSID_MAX_SIZE);
	memcpy(essid, priv->ieee80211.ssid, dwrq->length);
	return 0;
}


static int adm8211_ioctl_siwessid(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_point *dwrq, char *essid)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	struct ieee80211_device *ieee = netdev_priv(dev);
	int len;

	if (dwrq->flags == 0 || dwrq->length < 1)
		len = 0;
	else
		len = min(dwrq->length - 1, IW_ESSID_MAX_SIZE);

	memcpy(priv->ieee80211.ssid, essid, len);
	priv->ieee80211.ssid_len = len;
	if (len)
		priv->ieee80211.flags &= ~ANY_SSID;
	else
		priv->ieee80211.flags |= ANY_SSID;

	if (dev->flags & IFF_UP && ieee->state != IEEE80211_SHUTDOWN)
		ieee80211_start_scan(&priv->ieee80211);

	return 0;
}

static int adm8211_ioctl_giwnickn(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_point *dwrq, char *nickn)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);

        memset(nickn, 0, IW_ESSID_MAX_SIZE);
        strncpy(nickn, priv->ieee80211.nick, IW_ESSID_MAX_SIZE);
        return 0;
}

static int adm8211_ioctl_siwnickn(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_point *dwrq, char *nickn)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	int len = dwrq->length;

	if (len > IW_ESSID_MAX_SIZE)
		len = IW_ESSID_MAX_SIZE;

	memset(priv->ieee80211.nick, 0, IW_ESSID_MAX_SIZE);
	strncpy(priv->ieee80211.nick, nickn, len);

	return 0;
}

static int adm8211_ioctl_giwap(struct net_device *dev,
			       struct iw_request_info *info,
			       struct sockaddr *ap_addr, char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	struct ieee80211_device *ieee = netdev_priv(dev);
	ap_addr->sa_family = ARPHRD_ETHER;
	if (dev->flags & IFF_UP || !(priv->ieee80211.flags & PREF_BSSID_SET))
		memcpy(ap_addr->sa_data, ieee->bssid, ETH_ALEN);
	else
		memcpy(ap_addr->sa_data, priv->ieee80211.pref_bssid, ETH_ALEN);

	return 0;
}

static int adm8211_ioctl_siwmlme(struct net_device *dev,
				 struct iw_request_info *info,
				 union iwreq_data *wrqu,
				 char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	struct iw_mlme *mlme = (struct iw_mlme *)extra;

	if (!(dev->flags & IFF_UP))
		return -EINVAL;

	switch (mlme->cmd) {
	case IW_MLME_DEAUTH:
		ieee80211_send_deauth(&priv->ieee80211, mlme->reason_code, mlme->addr.sa_data);
		break;
	case IW_MLME_DISASSOC:
		ieee80211_send_disassoc(&priv->ieee80211, mlme->reason_code, mlme->addr.sa_data);
		break;
	default:
		return -EOPNOTSUPP;
	}

	return 0;
}

static int adm8211_ioctl_siwap(struct net_device *dev,
			       struct iw_request_info *info,
			       struct sockaddr *ap_addr, char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	u8 *addr = ap_addr->sa_data;
	memcpy(priv->ieee80211.pref_bssid, addr, ETH_ALEN);

	if (is_zero_ether_addr(addr))
		priv->ieee80211.flags &= ~(AUTO_ASSOCIATE | PREF_BSSID_SET);
	else if (is_broadcast_ether_addr(addr))
		priv->ieee80211.flags |= AUTO_ASSOCIATE;
	else
		priv->ieee80211.flags |= PREF_BSSID_SET;

	if (dev->flags & IFF_UP && priv->ieee80211.flags & (AUTO_ASSOCIATE | PREF_BSSID_SET))
		ieee80211_start_scan(&priv->ieee80211);

	return 0;
}

static int adm8211_ioctl_siwrate(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_param *vwrq,
				 char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	unsigned int i, j;

	if (vwrq->value <= 0)
		return -EINVAL;

	data->flags &= ~AUTO_RATE;
	if (!vwrq->fixed) {
		data->flags |= AUTO_RATE;

		/* assumes last rate is the highest rate */
		data->rate = data->supp_rates[data->num_supp_rates-1];
	} else if (vwrq->value <= data->num_supp_rates) {
		data->rate = data->supp_rates[vwrq->value-1];
	} else {
		i = 0;
		j = (vwrq->value*2)/1000000;

		/* make sure the rate given matches a supported rate */
		while (i < data->num_supp_rates && data->supp_rates[i] != j)
			i+=1;

		/* give up if it doesn't */
		if (i >= data->num_supp_rates)
			return -EINVAL;

		data->rate = j;
	}

	if (dev->flags & IFF_UP)
		adm8211_set_rate(dev);

	return 0;
}

static int adm8211_ioctl_giwrate(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_param *vwrq,
				 char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;

	vwrq->fixed = !(data->flags & AUTO_RATE);
	vwrq->value = (data->rate * 1000000)/2;
	return 0;
}

static int adm8211_ioctl_siwrts(struct net_device *dev,
				struct iw_request_info *info,
				struct iw_param *vwrq,
				char *extra)
{
	struct ieee80211_device *ieee = netdev_priv(dev);

	if (vwrq->disabled)
		ieee->rts = 2347;
	else if ((vwrq->value >= 0) && (vwrq->value <= 2346))
		ieee->rts = vwrq->value;
	else
		return -EINVAL;

	return 0;
}

static int adm8211_ioctl_giwrts(struct net_device *dev,
				struct iw_request_info *info,
				struct iw_param *vwrq,
				char *extra)
{
	struct ieee80211_device *ieee = netdev_priv(dev);

	vwrq->value = ieee->rts;
	vwrq->disabled = (vwrq->value > 2346);
	vwrq->fixed = 1;

	return 0;
}

static int adm8211_ioctl_siwfrag(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_param *vwrq,
				 char *extra)
{
	struct ieee80211_device *ieee = netdev_priv(dev);

	if (vwrq->disabled)
		ieee->fts = MAX_FRAG_THRESHOLD;
	else if ((vwrq->value >= MIN_FRAG_THRESHOLD) && (vwrq->value <= MAX_FRAG_THRESHOLD))
		ieee->fts = vwrq->value;
	else
		return -EINVAL;

	return 0;
}

static int adm8211_ioctl_giwfrag(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_param *vwrq,
				 char *extra)
{
	struct ieee80211_device *ieee = netdev_priv(dev);

	vwrq->value = ieee->fts;
	vwrq->disabled = (vwrq->value >= 2346);
	vwrq->fixed = 1;

	return 0;
}

static int adm8211_ioctl_giwtxpow(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_param *vwrq,
				  char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);

	vwrq->flags = IW_TXPOW_DBM;
	if (priv->tx_power > 0x3F) {
		vwrq->fixed = 0;
		vwrq->value = priv->eeprom->tx_power[priv->ieee80211.channel-1];
	} else {
		vwrq->fixed = 1;
		vwrq->value = priv->tx_power;
	}

	if (!vwrq->value)
		vwrq->disabled = 1;
	else
		vwrq->disabled = 0;

	return 0;
}

static int adm8211_ioctl_siwtxpow(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_param *vwrq,
				  char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);

	if (vwrq->disabled || !vwrq->fixed) {
		priv->tx_power = 0x40;
	} else if (vwrq->value >= 0 && vwrq->value <= 0x3F) {
		priv->tx_power = vwrq->value;
	} else
		return -EINVAL;

	return 0;
}

static int adm8211_ioctl_siwretry(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_param *vwrq,
				  char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);

	if (!vwrq->disabled && (vwrq->flags & IW_RETRY_LIMIT))
		priv->retry_limit = vwrq->value;
	else
		return -EINVAL;

	return 0;
}

static int adm8211_ioctl_giwretry(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_param *vwrq,
				  char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);

	vwrq->disabled = 0;
	vwrq->value = priv->retry_limit;
	vwrq->flags = IW_RETRY_LIMIT;

	return 0;
}

static int adm8211_ioctl_siwencode(struct net_device *dev,
				   struct iw_request_info *info,
				   union iwreq_data *wrqu,
				   char *key)
{
	return ieee80211_wx_set_encode(netdev_priv(dev), info, wrqu, key);
}

static int adm8211_ioctl_giwencode(struct net_device *dev,
				   struct iw_request_info *info,
				   union iwreq_data *wrqu,
				   char *key)
{
	return ieee80211_wx_get_encode(netdev_priv(dev), info, wrqu, key);
}

static int adm8211_ioctl_siwpower(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_param *vwrq,
				  char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);

	if (vwrq->disabled)
		priv->powersave = PS_OFF;
	else
		priv->powersave = PS_FAST;

	/*if (dev->flags & IFF_UP)
		adm8211_update_powersave(dev);*/

	return 0;
}

static int adm8211_ioctl_giwpower(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_param *vwrq,
				  char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	
	if (priv->powersave)
		vwrq->disabled = 0;
	else
		vwrq->disabled = 1;

	return 0;
}

static int adm8211_ioctl_siwgenie(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_point *dwrq,
				  char *extra)
{
	struct ieee80211_device *ieee = netdev_priv(dev);

	kfree(ieee->wpa_ie);
	ieee->wpa_ie = NULL;
	ieee->wpa_ie_len = 0;

	if (dwrq->length > 0) {
		ieee->wpa_ie = kmalloc(dwrq->length, GFP_KERNEL);
		if (!ieee->wpa_ie)
			return -ENOMEM;
		ieee->wpa_ie_len = dwrq->length;
		memcpy(ieee->wpa_ie, extra, ieee->wpa_ie_len);
	}
	return 0;
}

static int adm8211_ioctl_giwgenie(struct net_device *dev,
				  struct iw_request_info *info,
				  struct iw_point *dwrq,
				  char *extra)
{
	struct ieee80211_device *ieee = netdev_priv(dev);

	dwrq->length = ieee->wpa_ie_len;
	if (ieee->wpa_ie_len)
		memcpy(extra, ieee->wpa_ie, ieee->wpa_ie_len);

	return 0;
}

static int adm8211_ioctl_siwauth(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_param *vwrq,
				 char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	struct ieee80211_device *ieee = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;

	switch (vwrq->flags & IW_AUTH_INDEX) {
	case IW_AUTH_WPA_VERSION:
		data->wpa_version = vwrq->value;
		break;
	case IW_AUTH_CIPHER_PAIRWISE:
		data->cipher_pairwise = vwrq->value;
		break;
	case IW_AUTH_CIPHER_GROUP:
		data->cipher_group = vwrq->value;
		break;
	case IW_AUTH_KEY_MGMT:
		if (vwrq->value & IW_AUTH_KEY_MGMT_802_1X)
			ieee->ieee802_1x = 1;
		else
			ieee->ieee802_1x = 0;
		break;
	case IW_AUTH_TKIP_COUNTERMEASURES:
		//ieee->tkip_countermeasures = vwrq->value;
		break;
	case IW_AUTH_DROP_UNENCRYPTED:
		ieee->drop_unencrypted = vwrq->value;

		// ieee80211_rx doesn't properly check drop_unencrypted yet
		//ieee->open_wep = !vwrq->value;
		break;
	case IW_AUTH_80211_AUTH_ALG:
		if (vwrq->value & IW_AUTH_ALG_LEAP)
			return -EOPNOTSUPP;

		data->auth_alg = vwrq->value;
		break;
	case IW_AUTH_WPA_ENABLED:
		ieee->wpa_enabled = vwrq->value;
		break;
	case IW_AUTH_RX_UNENCRYPTED_EAPOL:
		break;
	case IW_AUTH_ROAMING_CONTROL:
		break;
	case IW_AUTH_PRIVACY_INVOKED:
		ieee->privacy_invoked = vwrq->value;
		break;
	default:
		return -EOPNOTSUPP;
	}

	return 0;
}

static int adm8211_ioctl_giwauth(struct net_device *dev,
				 struct iw_request_info *info,
				 struct iw_param *vwrq,
				 char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	struct ieee80211_device *ieee = netdev_priv(dev);
	struct ieee80211_data *data = &priv->ieee80211;
	
	switch (vwrq->flags & IW_AUTH_INDEX) {
	case IW_AUTH_WPA_VERSION:
		vwrq->value = data->wpa_version;
		break;
	case IW_AUTH_CIPHER_PAIRWISE:
		vwrq->value = data->cipher_pairwise;
		break;
	case IW_AUTH_CIPHER_GROUP:
		vwrq->value = data->cipher_group;
		break;
	case IW_AUTH_KEY_MGMT:
		vwrq->value = ieee->ieee802_1x ? IW_AUTH_KEY_MGMT_802_1X : 0;
		break;
	case IW_AUTH_TKIP_COUNTERMEASURES:
		//vwrq->value = ieee->tkip_countermeasures;
		break;
	case IW_AUTH_DROP_UNENCRYPTED:
		vwrq->value = ieee->drop_unencrypted;
		break;
	case IW_AUTH_80211_AUTH_ALG:
		vwrq->value = data->auth_alg;
		break;
	case IW_AUTH_WPA_ENABLED:
		vwrq->value = ieee->wpa_enabled;
		break;
	case IW_AUTH_RX_UNENCRYPTED_EAPOL:
		vwrq->value = 1;
		break;
	case IW_AUTH_ROAMING_CONTROL:
		vwrq->value = IW_AUTH_ROAMING_ENABLE;
		break;
	case IW_AUTH_PRIVACY_INVOKED:
		vwrq->value = ieee->privacy_invoked;
		break;
	default:
		return -EOPNOTSUPP;
	}
	
	return 0;
}

static int adm8211_ioctl_siwencodeext(struct net_device *dev,
				      struct iw_request_info *info,
				      union iwreq_data *wrqu,
				      char *extra)
{
	return ieee80211_wx_set_encodeext(netdev_priv(dev), info, wrqu, extra);
}

static int adm8211_ioctl_giwencodeext(struct net_device *dev,
				      struct iw_request_info *info,
				      union iwreq_data *wrqu,
				      char *extra)
{
	return ieee80211_wx_get_encodeext(netdev_priv(dev), info, wrqu, extra);
}

static const iw_handler adm8211_handler[] =
{
	(iw_handler) NULL,				/* SIOCSIWCOMMIT */
	(iw_handler) adm8211_ioctl_giwname,		/* SIOCGIWNAME */
	(iw_handler) NULL,				/* SIOCSIWNWID */
	(iw_handler) NULL,				/* SIOCGIWNWID */
	(iw_handler) adm8211_ioctl_siwfreq,		/* SIOCSIWFREQ */
	(iw_handler) adm8211_ioctl_giwfreq,		/* SIOCGIWFREQ */
	(iw_handler) adm8211_ioctl_siwmode,		/* SIOCSIWMODE */
	(iw_handler) adm8211_ioctl_giwmode,		/* SIOCGIWMODE */
	(iw_handler) NULL,				/* SIOCSIWSENS */
	(iw_handler) NULL,				/* SIOCGIWSENS */
	(iw_handler) NULL /* not used */,		/* SIOCSIWRANGE */
	(iw_handler) adm8211_ioctl_giwrange,		/* SIOCGIWRANGE */
	(iw_handler) NULL /* not used */,		/* SIOCSIWPRIV */
	(iw_handler) NULL /* kernel code */,		/* SIOCGIWPRIV */
	(iw_handler) NULL /* not used */,		/* SIOCSIWSTATS */
	(iw_handler) NULL /* kernel code */,		/* SIOCGIWSTATS */
	iw_handler_set_spy,				/* SIOCSIWSPY */
	iw_handler_get_spy,				/* SIOCGIWSPY */
	iw_handler_set_thrspy,				/* SIOCSIWTHRSPY */
	iw_handler_get_thrspy,				/* SIOCGIWTHRSPY */
	(iw_handler) adm8211_ioctl_siwap,		/* SIOCSIWAP */
	(iw_handler) adm8211_ioctl_giwap,		/* SIOCGIWAP */
	(iw_handler) adm8211_ioctl_siwmlme,		/* SIOCSIWMLME */
	(iw_handler) NULL /* deprecated */,		/* SIOCGIWAPLIST */
	(iw_handler) adm8211_ioctl_siwscan,		/* SIOCSIWSCAN */
	(iw_handler) adm8211_ioctl_giwscan,		/* SIOCGIWSCAN */
	(iw_handler) adm8211_ioctl_siwessid,		/* SIOCSIWESSID */
	(iw_handler) adm8211_ioctl_giwessid,		/* SIOCGIWESSID */
	(iw_handler) adm8211_ioctl_siwnickn,		/* SIOCSIWNICKN */
	(iw_handler) adm8211_ioctl_giwnickn,		/* SIOCGIWNICKN */
	(iw_handler) NULL,				/* -- hole -- */
	(iw_handler) NULL,				/* -- hole -- */
	(iw_handler) adm8211_ioctl_siwrate,		/* SIOCSIWRATE */
	(iw_handler) adm8211_ioctl_giwrate,		/* SIOCGIWRATE */
	(iw_handler) adm8211_ioctl_siwrts,		/* SIOCSIWRTS */
	(iw_handler) adm8211_ioctl_giwrts,		/* SIOCGIWRTS */
	(iw_handler) adm8211_ioctl_siwfrag,		/* SIOCSIWFRAG */
	(iw_handler) adm8211_ioctl_giwfrag,		/* SIOCGIWFRAG */
	(iw_handler) adm8211_ioctl_siwtxpow,		/* SIOCSIWTXPOW */
	(iw_handler) adm8211_ioctl_giwtxpow,		/* SIOCGIWTXPOW */
	(iw_handler) adm8211_ioctl_siwretry,		/* SIOCSIWRETRY */
	(iw_handler) adm8211_ioctl_giwretry,		/* SIOCGIWRETRY */
	(iw_handler) adm8211_ioctl_siwencode,		/* SIOCSIWENCODE */
	(iw_handler) adm8211_ioctl_giwencode,		/* SIOCGIWENCODE */
	(iw_handler) adm8211_ioctl_siwpower,		/* SIOCSIWPOWER */
	(iw_handler) adm8211_ioctl_giwpower,		/* SIOCGIWPOWER */
	(iw_handler) NULL,				/* -- hole -- */
	(iw_handler) NULL,				/* -- hole -- */
	(iw_handler) adm8211_ioctl_siwgenie,		/* SIOCSIWGENIE */
	(iw_handler) adm8211_ioctl_giwgenie,		/* SIOCGIWGENIE */
	(iw_handler) adm8211_ioctl_siwauth,		/* SIOCSIWAUTH */
	(iw_handler) adm8211_ioctl_giwauth,		/* SIOCGIWAUTH */
	(iw_handler) adm8211_ioctl_siwencodeext,	/* SIOCSIWENCODEEXT */
	(iw_handler) adm8211_ioctl_giwencodeext,	/* SIOCGIWENCODEEXT */
	(iw_handler) NULL,				/* SIOCSIWPMKSA */
	(iw_handler) NULL,				/* -- hole -- */
};

static int adm8211_ioctl_sreg (struct net_device *dev,
			       struct iw_request_info *info,
			       union iwreq_data *wrqu,
			       char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	unsigned int *i = (unsigned int *) extra;
	u32 addr = i[0];
	u32 reg = i[1];

	if (addr >= 0 && addr <= 0x10C) {
		printk(KERN_NOTICE "%s: writing 0x%x to 0x%x\n",
		       dev->name, reg, addr);
		iowrite32(reg, (void __iomem *)priv->map + addr);
	} else
		printk(KERN_NOTICE "%s: setreg: register value out of range\n", dev->name);

	return 0;
}

static int adm8211_ioctl_greg (struct net_device *dev,
			       struct iw_request_info *info,
			       union iwreq_data *wrqu,
			       char *extra)
{
        struct adm8211_priv *priv = ieee80211_priv(dev);
        u32 addr = wrqu->param.value;

	if (addr >= 0 && addr <= 0x10C)
	        printk(KERN_NOTICE "%s: value of register 0x%x is 0x%x\n",
		       dev->name, addr, ioread32((void __iomem *)priv->map + addr));
	else
		printk(KERN_NOTICE "%s: getreg: register value out of range\n", dev->name);

        return 0;
}

static int adm8211_ioctl_print_eeprom (struct net_device *dev,
				       struct iw_request_info *info,
				       union iwreq_data *wrqu,
				       char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	int i, bl;

	if (priv->eeprom) {
	printk(KERN_NOTICE "%s: eeprom dump:\n", dev->name);

	for (bl = 0; bl < priv->eeprom_len; bl += 16) {
		printk(KERN_NOTICE "%s:", dev->name);
		for (i = 0; i<16; i++)
			printk(" %02x", ((u8 *)priv->eeprom)[bl+i]);
		printk("\n");
	}

	} else
		printk(KERN_NOTICE "%s: no eeprom data\n", dev->name);

	return 0;
}

static int adm8211_ioctl_print_sram (struct net_device *dev,
				     struct iw_request_info *info,
				     union iwreq_data *wrqu,
				     char *extra)
{
        struct adm8211_priv *priv = ieee80211_priv(dev);
	unsigned int i;
	u32 mask = ADM8211_WEPCTL_TABLE_RD | (priv->revid < ADM8211_REV_BA ? 0 : ADM8211_WEPCTL_SEL_WEPTABLE );

	printk(KERN_NOTICE "%s: Dump SRAM: ", dev->name);
	for (i = 0; i < ADM8211_SRAM_SIZE; i++) {
		ADM8211_CSR_WRITE(WEPCTL, cpu_to_le32(i | mask));
		ADM8211_CSR_READ(WEPCTL);
		mdelay(1);

		if ((i % 8) == 0)
			printk("\n" KERN_NOTICE);
		
		if (priv->revid < ADM8211_REV_BA)
			printk("%04x ", swab16(ADM8211_CSR_READ(WESK)));
		else
			printk("%08x ", swab32(ADM8211_CSR_READ(WESK)));
		mdelay(1);
	}
	printk("\n");

	return 0;
}

static int adm8211_ioctl_set_country (struct net_device *dev,
				      struct iw_request_info *info,
				      union iwreq_data *wrqu,
				      char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	int code = wrqu->param.value;

	if (code >= ARRAY_SIZE(cranges) || code < 0)
		return -EINVAL;

	/* potential, but unlikely race, I think. need to check */
	priv->ieee80211.chan_range = cranges[code];
	printk(KERN_DEBUG "%s: Channel range: %d-%d\n",
	       dev->name, (int)priv->ieee80211.chan_range.min, (int)priv->ieee80211.chan_range.max);

	return 0;
}

static int adm8211_ioctl_set_antpower(struct net_device *dev,
				      struct iw_request_info *info,
				      union iwreq_data *wrqu,
				      char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);

	if (wrqu->param.value > 0x3F) {
		priv->ant_power = 0x40;
	} else if (wrqu->param.value >= 0 && wrqu->param.value <= 0x3F) {
		priv->ant_power = wrqu->param.value;
	} else
		return -EINVAL;

	priv->ieee80211.set_channel(dev, priv->ieee80211.channel);
	return 0;
}

static int adm8211_ioctl_set_lpfcutoff(struct net_device *dev,
				       struct iw_request_info *info,
				       union iwreq_data *wrqu,
				       char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);

	if (wrqu->param.value >= 0 && wrqu->param.value <= 0xFF) {
		priv->lpf_cutoff = wrqu->param.value;
	} else
		return -EINVAL;

	priv->ieee80211.set_channel(dev, priv->ieee80211.channel);
	return 0;
}

static int adm8211_ioctl_set_lnagsthresh(struct net_device *dev,
					 struct iw_request_info *info,
					 union iwreq_data *wrqu,
					 char *extra)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);

	if (wrqu->param.value >= 0 && wrqu->param.value <= 0xFF) {
		priv->lnags_threshold = wrqu->param.value;
	} else
		return -EINVAL;

	priv->ieee80211.set_channel(dev, priv->ieee80211.channel);
	return 0;
}

static int adm8211_ioctl_print_radio (struct net_device *dev,
				      struct iw_request_info *info,
				      union iwreq_data *wrqu,
				      char *extra)
{
        struct adm8211_priv *priv = ieee80211_priv(dev);
	int channel = priv->ieee80211.channel - 1;

	printk(KERN_DEBUG "%s: Antenna Power: %d (%d) \n",
			dev->name, priv->ant_power,
			priv->eeprom->antenna_power[channel]);
	printk(KERN_DEBUG "%s: LPF Cutoff: %d (%d)\n",
			dev->name, priv->lpf_cutoff,
			priv->eeprom->lpf_cutoff[channel]);
	printk(KERN_DEBUG "%s: LNAGS Threshold: %d (%d)\n",
			dev->name, priv->lnags_threshold,
			priv->eeprom->lnags_threshold[channel]);
	
	return 0;
}

static const iw_handler adm8211_private_handler[] =
{							/* SIOCIWFIRSTPRIV + */
	(iw_handler) adm8211_ioctl_sreg,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_greg,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_print_eeprom,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_print_radio,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_set_country,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_set_antpower,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_set_lpfcutoff,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_set_lnagsthresh,
	(iw_handler) NULL,
	(iw_handler) adm8211_ioctl_print_sram,
};

static const struct iw_priv_args adm8211_priv[] = {
	{SIOCIWFIRSTPRIV, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 2, 0, "setreg" },
	{0},
	{SIOCIWFIRSTPRIV+2, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "getreg" },
	{0},
	{SIOCIWFIRSTPRIV+4, IW_PRIV_TYPE_NONE, 0, "print_eeprom" },
	{0},
	{SIOCIWFIRSTPRIV+6, IW_PRIV_TYPE_NONE, 0, "print_radio" },
	{0},
	{SIOCIWFIRSTPRIV+8, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "set_country" },
	{0},
	{SIOCIWFIRSTPRIV+10, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "set_antpower" },
	{0},
	{SIOCIWFIRSTPRIV+12, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "set_lpfcutoff" },
	{0},
	{SIOCIWFIRSTPRIV+14, IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "set_lnagsthresh" },
	{0},
	{SIOCIWFIRSTPRIV+16, IW_PRIV_TYPE_NONE, 0, "print_sram" },
};

static struct iw_statistics *adm8211_get_wireless_stats(struct net_device *dev)
{
	struct adm8211_priv *priv = ieee80211_priv(dev);
	return &priv->wstats;
}

const struct iw_handler_def adm8211_iw_handler_def =
{
	.num_standard   = ARRAY_SIZE(adm8211_handler),
	.num_private    = ARRAY_SIZE(adm8211_private_handler),
	.num_private_args = ARRAY_SIZE(adm8211_priv),
	.standard       = (iw_handler *) adm8211_handler,
	.private        = (iw_handler *) adm8211_private_handler,
	.private_args   = (struct iw_priv_args *) adm8211_priv,
	.get_wireless_stats = adm8211_get_wireless_stats,
};

