/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "font.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "font_data.h"

#ifdef WFTK_DEFAULT_RESOURCE_ENABLED
// Static const unsigned int array representing a TTF file
#include "text_font.h"
#endif

#include "application.h"
#include "debug.h" 

#include <assert.h>
#include <iostream>

#include <ft2build.h>
#include FT_FREETYPE_H

wftk::ResourceRegistry<wftk::Font,wftk::Font::ResLoad,wftk::Font::ResInval>
	wftk::Font::registry;

wftk::Font::Glyph wftk::Font::bad_glyph_;
wftk::Font::Metrics wftk::Font::bad_metrics_;

wftk::Font::Font(const std::string& fontfilename, unsigned ptsize,
	const Color& color, unsigned face_index) throw(BadFont)
{
  glyphs_ = (new FontData(fontfilename, ptsize << 6, face_index))->ref(color);
}

wftk::Font::Font(const unsigned char* buffer, unsigned buf_size, unsigned ptsize,
	const Color& color, unsigned face_index) throw(BadFont)
{
  glyphs_ = (new FontData(buffer, buf_size, ptsize << 6, face_index))->ref(color);
}

wftk::Font& wftk::Font::operator=(const Font& f)
{
  if(glyphs_ == f.glyphs_)
    return *this;

  if(glyphs_)
    glyphs_->unref();
  glyphs_ = f.glyphs_;
  if(glyphs_)
    glyphs_->ref();

  return *this;
}

void
wftk::Font::setColor(const Color& color)
{
  if(!glyphs_)
    return;

  SurfaceTable* new_glyphs = glyphs_->ref(color);
  glyphs_->unref();
  glyphs_ = new_glyphs;
}

const wftk::Font&
wftk::Font::textFont()
{
  // have to use get() instead of find(), since find() defaults
  // to textFont() if it can't find the font
  Resource* res = registry.get("text_font");

#ifdef WFTK_DEFAULT_RESOURCE_ENABLED 
	if(!res) {
    Font text_font((const unsigned char *)&STATIC_FONT_DATA, STATIC_FONT_DATA_SIZE, 12);
		res = new Font::Resource(text_font);
	}
#endif
	if(!res) {
		// Still no font?  We are in bad shape.
    throw Fatal("Font resource \"text_font\" not available !");
	}
	else {
		registry.insert("text_font", res);
	}

  return res->res();
}

const wftk::Color&
wftk::Font::textColor()
{
  Color::Resource *text_color = Color::registry.get("text_color");
  return text_color ? text_color->res() : Color::find("white");
}

void
wftk::Font::Glyph::set(FontData& font, const Color& color, unsigned char c)
{
  Debug out(Debug::FONTS);

  out << "Starting Glyph::set()" << Debug::endl;

  FT_GlyphSlot slot = font[c];
  FT_Bitmap& bitmap = slot->bitmap;
  setSurface(bitmap.width, bitmap.rows); // ABGR888 by default

  out << "Got an SDL surface" << Debug::endl;

  assert(pixelformat().bitspp() == 32 && pixelformat().aMask() == 0xff000000);
  assert(sdlSurface_);

  fill(color);

  out << "About to set pixel transparency" << Debug::endl;

  lock();

  for(int y=0; y < bitmap.rows; ++y)
  {
    Uint8* surface_pixel = (Uint8*) sdlSurface_->pixels
	+ sdlSurface_->pitch * y;
    unsigned char* slot_byte = bitmap.buffer + bitmap.pitch * y;

    for(int x=0; x < bitmap.width; ++x)
    {
      Uint8 trans_value =
#if SDL_ALPHA_TRANSPARENT
	255 -          // if SDL_ALPHA_TRANSPARENT is 255, we subtract
#endif
	*slot_byte;

      *((Uint32 *) surface_pixel) &= 0xffffff; // clear alpha bits
      *((Uint32 *) surface_pixel) |= ((Uint32) trans_value) << 24; // and set them

      surface_pixel += 4;
      ++slot_byte;
    }
  }

  unlock();

  out << "Copying metrics information" << Debug::endl;

  metrics_.width = slot->metrics.width;
  metrics_.height = slot->metrics.height;

  metrics_.horiBearingX = slot->metrics.horiBearingX;
  metrics_.horiBearingY = slot->metrics.horiBearingY;
  metrics_.horiAdvance = slot->metrics.horiAdvance;

  metrics_.vertBearingX = slot->metrics.vertBearingX;
  metrics_.vertBearingY = slot->metrics.vertBearingY;
  metrics_.vertAdvance = slot->metrics.vertAdvance;

  metrics_.linearHoriAdvance = slot->linearHoriAdvance;
  metrics_.linearVertAdvance = slot->linearVertAdvance;
  metrics_.advance.x = slot->advance.x;
  metrics_.advance.y = slot->advance.y;

  metrics_.bitmap_left = slot->bitmap_left;
  metrics_.bitmap_top = slot->bitmap_top;
}

void
wftk::Font::Glyph::copy(Surface& target, const Point& dest, const Region& destmask) const
{
  if(!sdlSurface_)
    return;

  Uint32 alpha_flags = sdlSurface_->flags & (SDL_SRCALPHA | SDL_RLEACCEL);
  Uint8 alpha = sdlSurface_->format->alpha;

  SDL_SetAlpha(sdlSurface_, 0, SDL_ALPHA_OPAQUE);

  blit(target, dest, destmask);

  SDL_SetAlpha(sdlSurface_, alpha_flags, alpha);
}

int
wftk::Font::blitString(const std::string& txt, Surface& target,
	const Point& p, const Region& mask, bool copy) const
{
  if(!glyphs_ || target.empty() || txt.empty())
    return 0;

  Debug out(Debug::FONTS);
	out << "In Font::blitString()" << Debug::endl;

  Point advance(0, 0);

  bool has_kerning = glyphs_->font().hasKerning();

  out << "Advancing... ";
  unsigned char prev = 0;
  for(unsigned i = 0; i < txt.size(); ++i) {
    unsigned char current = txt[i];
    Point kerning = (prev && has_kerning) ?
	glyphs_->font().getKerning(prev, current) : Point(0, 0);
    prev = current;

    const Glyph& glyph = (*glyphs_)[current];

    Point dest(p.x + glyph.metrics().bitmap_left + (advance.x + kerning.x) / 64,
	       p.y - glyph.metrics().bitmap_top + (advance.y + kerning.y) / 64);

    if(copy)
      glyph.copy(target, dest, mask);
    else
      glyph.blit(target, dest, mask);

    int char_advance = glyph.metrics().advance.x;
    if(char_advance + kerning.x > 0)
      advance.x += char_advance + kerning.x;
    out << txt[i] << ":" << advance.x << " ";
  }

	out << " !" << Debug::endl;

  return advance.x / 64;
}

wftk::Surface*
wftk::Font::getString(const std::string& txt, Point& offset) const
{
  Debug out(Debug::FONTS);

  if(!glyphs_ || txt.empty())
    return new Surface();

  out << "In Font::getString()" << Debug::endl;

  Rect extents = getExtents(txt);
  offset = Point(extents.x, extents.y);

  out << "About to create target surface for string blit" << Debug::endl;

  Surface* target = new Surface(extents.w, extents.h);

  out << "About to draw string on surface" << Debug::endl;

  target->fill(Color(0, 0, 0, Color::WFTK_TRANSPARENT));
//  target->fill(Color(0, 0, 0));
  blitString(txt, *target, -offset, target->rect(), true);

  out << "Returning surface with blitted string" << Debug::endl;

  return target;
}

wftk::Rect
wftk::Font::getExtents(const std::string& txt) const
{
  if(!glyphs_ || txt.empty())
    return Rect(0, 0, 0, 0);

  Debug::channel(Debug::FONTS) << "In Font::getExtents()" << Debug::endl;

  bool has_kerning = glyphs_->font().hasKerning();

  int advance = 0;
  unsigned char prev = 0;
  for(unsigned i = 0; i < txt.size(); ++i) {
    unsigned char current = txt[i];
    int char_advance = (*glyphs_)[current].metrics().advance.x;
    if(prev && has_kerning)
      char_advance += glyphs_->font().getKerning(prev, current).x;
    prev = current;
    if(char_advance > 0)
      advance += char_advance;
  }

  int rise = (metrics().height + metrics().ascender + metrics().descender) / 2;

  return Rect(0, -rise / 64, advance / 64, metrics().height / 64);
}

const wftk::Font::Metrics&
wftk::Font::SurfaceTable::metrics() const
{
  return font_.metrics();
}

std::pair<wftk::Font,bool>
wftk::Font::ResLoad::operator()(const std::string& fontdescriptor_)
{
  std::string tmp = fontdescriptor_;
  int pos;
  std::string filename;
  pos = tmp.find(",");
  filename = tmp.substr(0,pos);

  Pixelformat format(Pixelformat::ARGB8888);
  
  int size;
  unsigned long fontCol;
  sscanf(tmp.substr(pos+1,tmp.size()-pos-1).c_str(),"%d,%lx",
	 &size, &fontCol);
  Debug::channel(Debug::GENERIC) <<"requesting font from "<<filename<<" at "<<
	size<<" pt."<<Debug::endl;

  try {
    Font font(filename, size, format.mapToColor(fontCol));

    return std::make_pair(font, font.valid());
  }
  catch(wftk::Font::BadFont&) {
    return std::make_pair(Font(), false);
  }
}


const wftk::Font::Glyph&
wftk::Font::SurfaceTable::operator[](unsigned char c)
{
  Glyph& glyph = glyphs_[c];
  if(glyph.empty())
    glyph.set(font_, color_, c);
  return glyph;
}

void
wftk::Font::SurfaceTable::ref()
{
  const SurfaceTable* out = font_.ref(color_);
  assert(out == this);
}

wftk::Font::SurfaceTable*
wftk::Font::SurfaceTable::ref(const Color& c)
{
  return font_.ref(c);
}

void
wftk::Font::SurfaceTable::unref()
{
  font_.unref(color_);
}
