#!/usr/bin/perl -w
# ldap-users.pl - Administrate users in ldap
#
# $Id: ldap-users.pl 2921 2005-02-18 23:18:43Z andreas $

use Data::Dumper;
use Net::LDAP;
use Net::LDAP::Entry;
use Net::LDAP::Util qw( ldap_error_name ldap_error_text);
use Text::Unaccent;
use HTML::FromText;
use Time::HiRes qw(usleep);
use locale;
use POSIX;
use strict;

END {ldap_close()}

# setlocale(LC_COLLATE, "no_NO.UTF-8");    

my %g;    # this is "g" for global. but only global to this file,
          # not the whole process.

# Connect to LDAP server.
sub ldap_connect {
    my ( $server, $dn, $passwd, $basedn, $minid, $maxid ) = @_;

    my (@err, @dbg);

    $g{minid}         = $minid;
    $g{maxid}         = $maxid;
    $g{server}        = $server;
    $g{basedn}        = $basedn;
    $g{ldap}          = undef;
    $g{err}           = \@err;
    $g{dbg}           = \@dbg;
    $g{allreadyusers} = undef;
    $g{modified}      = undef;

    # Take a look at bug 404 if this doesn't work
    $g{ldap} = Net::LDAP->new($server,version => '3') 
	|| die "Unable to connect to LDAP server on host \"$server\": $! ";
    my $result = $g{ldap}->start_tls( verify => 'none');
#    $g{ldap} = Net::LDAP->new( $server, version => '3',debug=>15 )
#      || die "Configuration error or LDAP server not running on $server: $! ";
#    my $result = $g{ldap}->start_tls( verify => 'require',
#				      cafile => '/var/lib/pyca/Root/cacert.pem');
    die "LDAP Error: "
      . ldap_error_name( $result->code ) . " "
      . ldap_error_text( $result->code ) . " "
      if $result->code();

    if ( defined $passwd ) {
	debug ($passwd, $dn);
	$result = $g{ldap}->bind( $dn, 'password' => $passwd );
    
	if ( $result->code ) {
	    error_msg( "Could not bind to ldap server! "
		       . ldap_error_name( $result->code ) . ": "
		       . ldap_error_text( $result->code ) . "\n" );
	}
    }
}

sub ldap_check_capabilities {
    my ($needed_ref) = @_;

    my $base   = $g{basedn};
    my $filter = "cn=capabilities";
    my $result = $g{ldap}->search( base => $base, filter => $filter );
    my $entry  = $result->pop_entry;
    if ( $result->code or !$entry ) {
        error_msg( "No capabilities found:"
            . ldap_error_text( $result->code ) . "\n" );
        return undef;
    }

    foreach my $cap ( $entry->get_value("capability") ) {
        ( $cap, my $have_version ) = split ( / /, $cap );
        if ( $needed_ref->{$cap} and $needed_ref->{$cap} > $have_version ) {
            return undef;
        }
        else {
            delete $needed_ref->{$cap} if  $needed_ref->{$cap};
        }
    }

    if ( keys %$needed_ref ) {

        # we depend on capabilities the ldap directory does not meet.
	return undef;
    }
    return "can run";
}

# Make a unique username from common name.
sub make_uid {
    my ( $firstname, $lastname, $newnames_ref ) = @_;
    $g{allreadyusers} = $newnames_ref;
    my $username = make_unique_username( $firstname, $lastname );
    return $username;
}

sub change_slicing {
    my ( $filter_var, $slice_point, $uppercase_ref ) = @_;

    my @filter_var_list;
    my $n = $slice_point;    # this is done this way because
                             # the s/// replaces the k-1 occurance,
                             # so $n needs to lag one behind.
    $slice_point++;
    my $star_cnt = $filter_var =~ tr/*/*/;
    if ( $slice_point > $star_cnt ) {
        $slice_point = undef;
        return ( [], $slice_point );
    }

    for my $char ( 'a' .. 'z', '0' .. '9', @$uppercase_ref ) {
        my $filter_var_new = $filter_var;

        $filter_var_new =~ s/(^[^*]*(?:\*.*?){$n})\*/$1$char*/x;
        push @filter_var_list, $filter_var_new;
    }

    return ( \@filter_var_list, $slice_point );
}

sub check_slice_empty {
    my ($slice_ref) = @_;
    return 1 if $#$slice_ref == -1;
    return 0;
}

sub sort_result {
    my ( $sort_array_ref, $sort_key ) = @_;

    my @result =
      sort { $a->get_value($sort_key) cmp $b->get_value($sort_key); }
      @{$sort_array_ref};
    return @result;
}

sub get_slice {
    my ( $base, $filter_fix, $filter_var, $uppercase_ref, $depth ) = @_;
    my ( @slice, $filters );

    $depth++;

    # final filter:"(&(objectClass=lisGroup)(cn="*a*b*"))"
    # filter_fix: "&(objectClass=lisGroup)"
    # variable filter: cn="*a*b*"
    my $filter = "($filter_fix($filter_var))";
    my $mesg = $g{ldap}->search( base => $base, filter => $filter );
    if ( 4 == $mesg->code ) {

        # We got a LDAP_SIZELIMIT_EXEEDED error.
        #split up slice 
        my $slice_point = 0;
        do {
            ( my $filter_var_list_ref, $slice_point ) =
              change_slicing( $filter_var, $slice_point, $uppercase_ref );
            for my $filter_var_new ( @{$filter_var_list_ref} ) {
                if ( 9 < $depth ) { last }
                ;    # savety against infinit recursion
                 # we should not go deeper then 9 since logins have just 8 chars
                 #get sub-slices
                my ( $slice_ref, $filters_applied ) = get_slice(
                    $base,          $filter_fix, $filter_var_new,
                    $uppercase_ref, $depth
                );

                # merge new slices 
                push ( @slice, @{$slice_ref} );
                $filters .= $filters_applied if $filters_applied;
            }
        } while ( check_slice_empty( \@slice ) and defined $slice_point );
        if ( 1 == $depth ) { # this happens after we are done with our recursion
                #here we get all other entries,
                #which for some reason were not covered in the other searches.
            $filter = "($filter_fix$filters($filter_var))\n";
            $mesg = $g{ldap}->search( base => $base, filter => $filter );
            my @tmp = $mesg->all_entries();
            debug_msg( "key $filter  returned " . ( $#tmp + 1 ) . " users\n" );
            push ( @slice, $mesg->all_entries() );
        }
    }
    elsif ( $mesg->code ) {    # some other error!
        error_msg( "ldap problem: $base $filter : "
            . ldap_error_name( $mesg->code ) . " "
            . ldap_error_text( $mesg->code ) . "\n" )
          if $mesg->code;
        return ( [], "" );
    }
    else {    # We got the whole slice in one piece.
        @slice   = $mesg->all_entries();
        $filters = "(!$filter_var)";

        my @tmp = $mesg->all_entries();
        debug_msg( "key $filter  returned " . ( $#tmp + 1 ) . " users\n" )
          unless ( -1 == $#tmp );
    }
    if ( 1 == $depth ) {
        $filter_var =~ m/^(.*)=/;
        my $sort_key = $1;
        @slice = sort_result( \@slice, $sort_key );
    }
    return ( \@slice, $filters );
}

# Get all users in a given domain $basedn.
sub ldap_get_users {

    my $base = "ou=People," . $g{basedn};

    # since we dont have unique cn for users, it is unsave to search 
    # the cn with a limited SIZELIMIT therefore we must limit us to the 
    # uid, which is unique. it is much more efficent, too.
    my $filter_fix = "&(objectClass=posixAccount)";
    my $filter_var = "uid=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    return @{$slice_ref};
}

# Search for users in a given domain $basedn.
sub ldap_search_users {
    my ( $searchin, $searchfor ) = @_;

    my $base       = "ou=People," . $g{basedn};
    my $filter_fix = "&(objectClass=posixAccount)";
    my $filter_var = "$searchin=$searchfor*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    return @{$slice_ref};
}

# Search in a given domain $basedn.
sub ldap_search {
    my ( $searchin, $searchfor ) = @_;
    my ( $ou, $oc, $attribut );
    unless ($searchin) {
        goto error_exit;
    }
    elsif ( $searchin eq "class_cn" ) {
        $oc       = "lisGroup";
        $attribut = "cn";
        $ou       = "ou=Group,";
    }
    elsif ( $searchin eq "user_login" ) {
        $oc       = "posixAccount";
        $attribut = "uid";
        $ou       = "ou=People,";
    }
    elsif ( $searchin eq "user_cn" ) {
        $oc       = "posixAccount";
        $attribut = "cn";             #XXX cn ist nicht unique!!!
        $ou       = "ou=People,";
    }
    else {
        error_exit:
        return ( "illegal search type\n", undef );
    }
    unless ( $searchfor and $searchfor =~ /\*/ ) {
        $searchfor .= "*";    #XXX quick fix, make this an exact search instead.
    }
    my $base       = $ou . $g{basedn};
    my $filter_fix =
      "&(objectClass=$oc)(!(|(groupType=private)(cn=admin)(cn=nextID)))";
    my $filter_var = "$attribut=$searchfor";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    if ( $slice_ref->[0] and $slice_ref->[0]->get_value('cn') ) {
        return ( "OK", @{$slice_ref} );
    }
    else {
        return ( "EMPTY", undef );
    }
}

sub ldap_disable_user_login {
    my ( $user_login ) = @_;
    my $result;
    my $base   = "ou=People," . $g{basedn};
    my $filter = "&(objectClass=posixAccount)(uid=$user_login)";

    $result = $g{ldap}->search( base => $base, filter => $filter );
    my $entry = $result->pop_entry;
    if ( $result->code or !$entry ) {
        debug_msg( "No such user \"$user_login\" found:"
            . ldap_error_text( $result->code ) . "\n" );
        return undef;
    }

    if ( $config{sambasync} ){
	system( "smbpasswd -d $user_login  >/dev/null 2>&1" );
    }

    my $inactive_flag = $entry->get_value('shadowFlag');
    if ($inactive_flag) {
        debug_msg( "User $user_login is allready disabled by flag:"
            . ldap_error_text( $result->code ) . "\n" );
    }
    else {
        $result =
          $g{ldap}->modify( $entry,
            changes => [ add => [ 'shadowFlag' => 1 ] ] );
        if ( $result->code ) {
            debug_msg( "Error deactivating user's ldap-flag! "
                . ldap_error_name( $result->code ) . "\n" );
        }
    }

    my $passwd = $entry->get_value('userPassword');
    if ( $passwd =~ /^DISABLED!/ ) {
        debug_msg( "User $user_login is allready disabled by password:"
            . ldap_error_text( $result->code ) . "\n" );
    }
    else {
        $result =
          $g{ldap}->modify( $entry,
            changes => [ replace => [ 'userPassword' => "DISABLED!$passwd" ] ]
        );
        if ( $result->code ) {
            debug_msg( "Error deactivating user's password! "
                . ldap_error_name( $result->code ) . "\n" );
        }
    }

    my $shell = $entry->get_value('loginShell');
    if ( $shell =~ /^DISABLED!/ ) {
        debug_msg( "User $user_login is allready disabled by shell-distrortion:"
            . ldap_error_text( $result->code ) . "\n" );
    }
    else {
        $result =
          $g{ldap}->modify( $entry,
            changes => [ replace => [ 'loginShell' => "DISABLED!$shell" ] ] );
        if ( $result->code ) {
            debug_msg( "Error deactivating user's shell! "
                . ldap_error_name( $result->code ) . "\n" );
        }
    }

    my $homedir = $entry->get_value('homeDirectory');
    if ( ( ( stat($homedir) )[2] & 07777 ) == 00000 ) {
        debug_msg(
"User $user_login is allready disabled by homedirectory permissoions:"
            . ldap_error_text( $result->code ) . "\n" );
    }
    else {
        my $ret = chmod 0000, $homedir;
        if ( $ret != 1 ) {
            debug_msg("Error deactivating user's homedirectory! $homedir \n");
        }
    }
    $g{modified} = 1;
    return 1;
}

sub ldap_enable_user_login {
    my ( $user_login ) = @_;
    my $result;
    my $base   = "ou=People," . $g{basedn};
    my $filter = "(&(objectClass=posixAccount)(uid=$user_login))";

    $result = $g{ldap}->search( base => $base, filter => $filter );

    my $entry = $result->pop_entry;
    if ( $result->code or !$entry ) {
        debug_msg( "No such user \"$user_login\" found:"
            . ldap_error_text( $result->code ) . "\n" );
        return undef;
    }

    if ( $config{sambasync} ){
	system( "smbpasswd -e $user_login  >/dev/null 2>&1" );
    }

    my $inactive_flag = $entry->get_value('shadowFlag');
    unless ($inactive_flag) {
        debug_msg( "User $user_login is allready ldap-enabled:"
            . ldap_error_text( $result->code ) . "\n" );
    }
    else {
        $result =
          $g{ldap}->modify( $entry,
            changes => [ delete => [ 'shadowFlag' => 1 ] ] );
        if ( $result->code ) {
            debug_msg( "Error enabling user! "
                . ldap_error_name( $result->code ) . "\n" );
        }
    }

    my $homedir = $entry->get_value('homeDirectory');
    if ( ( ( stat($homedir) )[2] & 07777 ) == 00755 ) {
        debug_msg( "User $user_login is allready enabled for the homedir:"
            . ldap_error_text( $result->code ) . "\n" );
    }
    else {
        my $ret = chmod 0755, $homedir;
        if ( $ret != 1 ) {
            debug_msg("Error enabling user's home directory $homedir!\n");
        }
    }

    my $shell = $entry->get_value('loginShell');
    unless ( $shell =~ /^DISABLED!(.*)$/ ) {
        debug_msg("User $user_login is allready enabled on shell level.");
    }
    else {
        $shell  = $1;
        $result =
          $g{ldap}->modify( $entry,
            changes => [ replace => [ 'loginShell' => $shell ] ] );
        if ( $result->code ) {
            debug_msg( "Error enabling user's shell $shell! "
                . ldap_error_name( $result->code ) . "\n" );
        }
    }

    my $passwd = $entry->get_value('userPassword');
    unless ( $passwd =~ /^DISABLED!(.*)$/ ) {
        debug_msg("User $user_login is allready enabled on shell level.");
    }
    else {
        $passwd = $1;
        $result =
          $g{ldap}->modify( $entry,
            changes => [ replace => [ 'userPassword' => $passwd ] ] );
        if ( $result->code ) {
            debug_msg( "Error enabling user's shell $shell! "
                . ldap_error_name( $result->code ) . "\n" );
        }
    }
    $g{modified} = 1;
    return 1;
}

sub ldap_search_user_disabled_logins {
    my ( $searchin, $searchfor ) = @_;
    my ( $ou, $oc, $attribut );

    my $filter_fix =
      "&(shadowFlag=1)(!(|(groupType=private)(cn=admin)(cn=nextID)))";
    
    unless ($searchin) {
        goto error_exit;
    }
    elsif ( $searchin eq "class_cn" ) {
	my @uids; 
	my($msg, @list) = ldap_search($searchin, $searchfor);
	return  ( "EMPTY", undef ) if $msg ne "OK";
	for my $class (@list){
	    push @uids, $class->get_value("memberUID");
	}
	my $filter_append;
	for my $uid ( unique( @uids ) ){
	    $filter_append .= "(uid=$uid)";      
	}
	$filter_fix .= "(|$filter_append)" if $filter_append;
        $oc       = "posixAccount";
        $attribut = "uid";
        $ou       = "ou=People,";
    }
    elsif ( $searchin eq "user_login" ) {
        $oc       = "posixAccount";
        $attribut = "uid";
        $ou       = "ou=People,";
    }
    elsif ( $searchin eq "user_cn" ) {
        $oc       = "posixAccount";
        $attribut = "cn";       #XXX cn ist nicht unique!!!
        $ou       = "ou=People,";
    }
    else {
        error_exit:
        return ( "illegal search type\n", undef );
    }
    unless ( $searchfor and $searchfor =~ /\*/ ) {
        $searchfor .= "*";    #XXX quick fix, make this an exact search instead.
    }
    my $base       = $ou . $g{basedn};
    my $filter_var = "$attribut=$searchfor";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    if ( $slice_ref->[0] and $slice_ref->[0]->get_value('cn') ) {
        return ( "OK", @{$slice_ref} );
    }
    else {
        return ( "EMPTY", undef );
    }
}

sub ldap_get_group_login_status{
    my ( $cn ) = @_;

    my $base       = "ou=People," . $g{basedn};
    my $filter_var = "uid=*";

    my $group = ldap_get_group($cn);
    my @uids = $group->get_value("memberUID");
    
    my $filter_append;
    for my $uid ( unique( @uids ) ){
	$filter_append .= "(uid=$uid)";      
    }

    my ($login_enabled_flag, $login_disabled_flag) = (undef, undef);
    if ($filter_append) {
	my $filter_fix = "&(!(shadowFlag=1))(|$filter_append)"; 
	my ( $slice_ref, $filters_applied ) =
	    get_slice( $base, $filter_fix, $filter_var );
	if ( $slice_ref->[0] and $slice_ref->[0]->get_value('cn') ) {
	    $login_enabled_flag = "1";
	}

	$filter_fix = "&(shadowFlag=1)(|$filter_append)"; 
	( $slice_ref, $filters_applied ) =
	    get_slice( $base, $filter_fix, $filter_var );
	if ( $slice_ref->[0] and $slice_ref->[0]->get_value('cn') ) {
	    $login_disabled_flag = "1";
	}
    }
    return ($login_enabled_flag, $login_disabled_flag);
}

sub ldap_search_group_disabled_logins
{    # this wont work like this, need to loop over all users in group
    my ($searchfor) = @_;
    my $base       = "ou=People," . $g{basedn};
    my $filter_fix = "&(userPassword=\!*)(gidNumber=$searchfor)";
    my $filter_var = "uid=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    return @{$slice_ref};
}

sub ldap_search_user_enabled_logins {
    my ( $searchin, $searchfor ) = @_;
    my ( $oc, $ou, $attribut );

    my $filter_fix =
      "&(!(shadowFlag=1))(!(|(groupType=private)(cn=admin)(cn=nextID)))";

    unless ($searchin) {
        goto error_exit;
    }
    elsif ( $searchin eq "class_cn" ) {
	my @uids; 
	my($msg, @list) = ldap_search($searchin, $searchfor);
	return  ( "EMPTY", undef ) if $msg ne "OK";
	for my $class (@list){
	    push @uids, $class->get_value("memberUID");
	}
	my $filter_append;
	for my $uid ( unique( @uids ) ){
	    $filter_append .= "(uid=$uid)";      
	}
	$filter_fix .= "(|$filter_append)" if $filter_append;
        $oc       = "posixAccount";
        $attribut = "uid";
        $ou       = "ou=People,";
    }
    elsif ( $searchin eq "user_login" ) {
        $oc       = "posixAccount";
        $attribut = "uid";
        $ou       = "ou=People,";
    }
    elsif ( $searchin eq "user_cn" ) {
        $oc       = "posixAccount";
        $attribut = "cn";       #XXX cn ist nicht unique!!!
        $ou       = "ou=People,";
    }
    else {
        error_exit:
        return ( "illegal search type\n", undef );
    }
    unless ( $searchfor and $searchfor =~ /\*/ ) {
        $searchfor .= "*";    #XXX quick fix, make this an exact search instead.
    }
    my $base       = $ou . $g{basedn};
    my $filter_var = "$attribut=$searchfor";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    if ( $slice_ref->[0] and $slice_ref->[0]->get_value('cn') ) {
        return ( "OK", @{$slice_ref} );
    }
    else {
        return ( "EMPTY", undef );
    }
}

sub ldap_search_group_enabled_logins
{    # this wont work like this, need to loop over all users in group
    my ($searchfor) = @_;
    my $base       = "ou=People," . $g{basedn};
    my $filter_fix = "&(!userPassword=!*)(gidNumber=$searchfor)";
    my $filter_var = "uid=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    return @{$slice_ref};
}

# Get all groups in a domain $basedn.
sub ldap_get_groups {
    my ($group_type) = @_;

    my $base       = "ou=Group," . $g{basedn};
    my $filter_fix = "&(objectClass=lisGroup)";
    if ($group_type) {
        $filter_fix .= "(groupType=$group_type)";
    }
    my $filter_var = "cn=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );

    return @{$slice_ref};
}

# Get all groups a user $uid is member of, given a domain $basedn.
sub ldap_get_membergroups {
    my ($uid) = @_;

    my $base       = "ou=Group," . $g{basedn};
    my $filter_fix = "&(objectClass=lisGroup)(|(memberUid=$uid)(member=uid=$uid,ou=People,".$g{basedn}."))";
    my $filter_var = "cn=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );

    return @{$slice_ref};
}

# this is a more general function for retrieving a 
# list of groups a user is a member of with more 
# finegrained search criteria.
sub ldap_get_member_grouplist {
    my ( $uid, $grouptype ) = @_;

    my $base       = "ou=Group," . $g{basedn};
    my $filter_fix =
      "&(objectClass=lisGroup)(memberUid=$uid)(groupType=$grouptype)";
    my $filter_var = "cn=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );

    return @{$slice_ref};
}

sub ldap_get_not_member_grouplist {
    my ( $uid, $grouptype ) = @_;

    my $base       = "ou=Group," . $g{basedn};
    my $filter_fix =
      "&(objectClass=lisGroup)(groupType=$grouptype)(!(memberUid=$uid))";
    my $filter_var = "cn=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );

    return @{$slice_ref};
}

# takes gidNumber, returns list of complete user entries
sub ldap_get_member_userlist {
    my ($gidNumber) = @_;

    my @list;
    my $base   = "ou=Group," . $g{basedn};
    my $filter = "(&(objectClass=lisGroup)(gidNumber=$gidNumber))";
    my $mesg   = $g{ldap}->search( base => $base, filter => $filter );
    my $entry  = $mesg->pop_entry;
    unless ($entry) { return undef }
    my @uids = $entry->get_value("memberUid");

    for my $uid (@uids) {
        my $user = ldap_get_user($uid);
        push @list, $user;
    }
    return @list;
}

# get all users who are not yet members in a group
sub ldap_get_not_member_userlist {
    my ($gidNumber) = @_;

    # we need to get all users who are part of the 
    # ageGroups but not of the gid. 
    # first get all users in $gidNumber 
    my @list;
    my $base   = "ou=Group," . $g{basedn};
    my $filter = "(&(objectClass=lisGroup)(gidNumber=$gidNumber))";
    my $mesg   = $g{ldap}->search( base => $base, filter => $filter );
    my $entry  = $mesg->pop_entry;
    unless ($entry) { return undef }
    my @black_list = $entry->get_value("memberUid");
    my $cn         = $entry->get_value("cn");

    #make blacklist a hash, for faster access
    my (%blacklist, $blacklist_filter);
    for my $uid (@black_list) {
        $blacklist{$uid} = 1;    
	$blacklist_filter .= "(uid=$uid)";
    }
    
    # Get all users.
    $base = "ou=People," . $g{basedn};
    my $filter_fix = "&(objectClass=posixAccount)";
    unless ( -1 == $#black_list ) {
	$filter_fix .= "(!(|$blacklist_filter))";
    }
    my $filter_var = "uid=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
  
    return @$slice_ref;

}

# Get all groups a user $uid is not member of, given a domain $basedn.
sub ldap_get_not_membergroups {
    my ($uid) = @_;
    my @notGroups;

    my $grpBase    = "ou=Group,"  . $g{basedn};
    my $actBase    = "ou=People," . $g{basedn};
    my $filter_fix = "&(objectClass=lisGroup)(!(memberUid=$uid))";
    my $filter_var = "cn=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $grpBase, $filter_fix, $filter_var );

    foreach my $group ( @{$slice_ref} ) {
        my $gidNumber = $group->get_value('gidNumber');
        my $filter    = "(&(objectClass=posixAccount)(gidNumber=$gidNumber))";
        my $privGroup = $g{ldap}->search(
            base   => $actBase,
            filter => $filter
        );
        my $didfindgroup = $privGroup->count();
        unless ($didfindgroup) {
            push ( @notGroups, $group );
        }
    }
    return @notGroups;
}

# In a domain $basedn, get data about a user $uid. 
# If user is not found, undef is returned.
sub ldap_get_user {
    my ($uid) = @_;

    my $base   = "ou=People," . $g{basedn};
    my $filter = "uid=$uid";

    my $mesg = $g{ldap}->search( base => $base, filter => $filter );
    error_msg( "ldap_get_user: " . ldap_error_text( $mesg->code ) . "\n" )
      if $mesg->code;

    return $mesg->pop_entry;
}

# In a domain $basedn, get data about a user $uid. 
# If user is not found, undef is returned.

sub ldap_get_old_user {
    my ($uid) = @_;

    my $base   = "ou=Attic," . $g{basedn};
    my $filter = "uid=$uid";

    my $mesg = $g{ldap}->search( base => $base, filter => $filter );
    error_msg( "ldap_get_user: " . ldap_error_text( $mesg->code ) . "\n" )
      if $mesg->code;

    return $mesg->pop_entry;
}

# Returns if the group $gidNumber exists in domain $basedn.
sub ldap_group_exists {
    my ($gid) = @_;

    my $mesg = $g{ldap}->search(
        base   => "ou=Group," . $g{basedn},
        filter => "cn=$gid"
    );
    debug_msg( "ldap_group_exists: " . ldap_error_name( $mesg->code ) . "\n" )
      if $mesg->code;
    return $mesg->count();
}

# Returns if a given (group or user)-id exists in domain $basedn.
sub ldap_id_exists {
    my ($id) = @_;
    my $base = "ou=Group," . $g{basedn};

    my $mesg = $g{ldap}->search( base => $base, filter => "gidNumber=$id" );
    debug_msg( "ldap_id_exists: " . ldap_error_name( $mesg->code ) . "\n" )
      if $mesg->code;
    unless ( $mesg->count() ) {    # usually we should not need to do this,
            # because UID and GID are likely the same for users.
        $mesg = $g{ldap}->search( base => $base, filter => "uidNumber=$id" );
        debug_msg( "ldap_id_exists: " . ldap_error_name( $mesg->code ) . "\n" )
          if $mesg->code;
    }
    return $mesg->count();
}

# Returns the name of a group $gid in the domain $basedn,
# or the string "(not in ldap)" if the group is not found.
sub ldap_get_groupname {
    my ($gidNumber) = @_;

    my $mesg = $g{ldap}->search(
        base   => "ou=Group," . $g{basedn},
        filter => "gidNumber=$gidNumber",
        attrs  => "cn",
    );
    debug_msg( "ldap_get_groupname: " . ldap_error_name( $mesg->code ) . "\n" )
      if $mesg->code;

    return "(not in ldap)" unless ( $mesg->count() );

    my $entry = $mesg->pop_entry;
    return $entry->get_value('cn');
}

# Returns the group $gid in the domain $basedn,
# or undef if the group is not found.
sub ldap_get_group {
    my ($cn) = @_;

    my $mesg = $g{ldap}->search(
        base   => "ou=Group," . $g{basedn},
        filter => "cn=$cn"
    );
    debug_msg( "ldap_get_groupname: " . ldap_error_name( $mesg->code ) . "\n" )
      if $mesg->code;

    return $mesg->pop_entry;
}

# Returns the description of the $gid in the domain $basedn,
# or the string "(not in ldap)" if the group is not found.
sub ldap_get_groupdescription {
    my ($gidNumber) = @_;

    my $mesg = $g{ldap}->search(
        base   => "ou=Group," . $g{basedn},
        filter => "gidNumber=$gidNumber",
        attrs  => "cn",
    );
    debug_msg(
        "ldap_get_groupdescription: " . ldap_error_name( $mesg->code ) . "\n" )
      if $mesg->code;

    return "(not in ldap)" unless ( $mesg->count() );

    my $entry = $mesg->pop_entry;
    return $entry->get_value('description');
}

sub ldap_find_current_id {

    my @list  = ldap_get_groups();
    my $maxid = $g{minid};
    while (@list) {
        my $entry = pop @list;
        my $id    = $entry->get_value('gidNumber');
        $maxid = $id if ( $id > $maxid );
    }

    @list = ldap_get_users();
    while (@list) {
        my $entry = pop @list;
        my $id    = $entry->get_value('gidNumber');
        $maxid = $id if ( $id > $maxid );
        $id = $entry->get_value('uidNumber');
        $maxid = $id if ( $id > $maxid );
    }

    return $maxid;
}

sub ldap_create_entry_nextid {

    my $entry = Net::LDAP::Entry->new();
    $entry->dn( "ou=Variables," . $g{basedn} );
    $entry->add(
        objectclass => [ 'organizationalUnit', 'top' ],
        ou          => "Variables"
    );
    $entry->update( $g{ldap} );
    $entry = Net::LDAP::Entry->new();
    $entry->dn( "cn=nextID,ou=Variables," . $g{basedn} );
    my $current_id = ldap_find_current_id();
    $entry->add(
        objectclass => [ 'posixGroup', 'top' ],
        cn          => "nextID",
        gidNumber   => $current_id,
    );
    ldap_update( $entry );

}

# looks up cn=nextID,ouVariables,$basedn to get the new ID it should use
# function fetched from http://www.ccie.org.uk/resources/newuid.html 
# function somewhat rewritten to suit our needs
sub ldap_get_newid {
#    my ($rootpw) = @_;
    my $loop = 2;
    my ( $result, $object, $newid );
    my $minid = $g{minid};
    my $maxid = $g{maxid};
    my $base  = "ou=variables," . $g{basedn};

    while ($loop) {
        $result = $g{ldap}->search(
            base   => $base,
            filter => "cn=nextID"
        );
        $object = $result->pop_entry;
        if ( $result->code or !$object ) {

            # It the Search Fails it's most likely that this is an old
            # installation, with a missing dn=nextID, ou=Variables
            debug_msg( "Need to generate nextID!:"
                . ldap_error_text( $result->code ) . "\n" );
            ldap_create_entry_nextid();

            # Okay , try searching again
            $result = $g{ldap}->search(
                base   => $base,
                filter => "cn=nextID"
            );
            $object = $result->pop_entry;
        }

        if ($object) {
            $newid = $object->get_value('gidnumber');
            debug_msg("Got a valid nextID: $newid !\n");
        }
        else {
            debug_msg( "Got no nextID! Quit trying?!?"
                . ldap_error_text( $result->code ) . "\n" );
            undef $newid;
            last;
        }
        debug_msg("Increasing nextID!\n");
        $result = $g{ldap}->modify(
            $object,
            changes => [
                delete => [ 'gidNumber' => $newid ],
                add    => [ 'gidNumber' => $newid + 1 ]
            ]
        );
        if ( $result->code
            || getpwuid($newid)
            || getgrgid($newid)
            || ldap_id_exists($newid)
            || ( $newid > $maxid )
            || ( $newid < $minid ) )
        {
            debug_msg( "Something suspicous happend while increasing nextID! "
                . ldap_error_name( $result->code ) . "\n" );
            if ( ( $newid gt $maxid ) || ( $newid lt $minid ) ) {
                unless ($loop) {
                    error_msg("nextID out of bounds. But quit tying.\n");
                    undef $newid;
                    last;
                }
                debug_msg(
"now attempt to set nextID, since it was out of bounds: $minid, $maxid\n"
                );
                $result = $g{ldap}->modify(
                    $object,
                    changes => [
                        delete => [ 'gidNumber' => $newid + 1 ],
                        add    => [ 'gidNumber' => $minid ]
                    ]
                );
                if ( $result->code ) {
                    debug_msg(
                        "Something suspicous happend while setting nextID! "
                        . ldap_error_name( $result->code ) . "\n" );
                }
                $loop--;
            }
            debug_msg("Race Condition? Need to try again!\n");
            undef $newid;
            usleep( int( rand(500) ) );
        }
        else {
            debug_msg("I think i succeeded getting a nextID!\n");
            last;
        }
    }
    debug_msg("Returning now $newid!\n");
    $g{modified} = 1;
    return ($newid);
}

# Add a user to LDAP.
sub ldap_add_user {
    my (
        $cn,        $uid,       $userpw,
        $uidNumber, $gidNumber, 
        $homedir,   $maildir,   $userpw_crypt
      )
      = @_;

    # if there is no given crypt-hash generate it
    unless ( $userpw_crypt ) {
        $userpw_crypt = gen_crypt($userpw);
    }

    debug_msg("Trying to add this users: uidNumber:$uidNumber  uid:$uid!\n");
    my $entry     = Net::LDAP::Entry->new();
    my $new_entry = "uid=$uid,ou=People," . $g{basedn};
    $entry->dn($new_entry);
    $entry->add(
        objectclass   => [ 'posixAccount', 'top', 'shadowAccount', 'imapUser' ],
        cn            => $cn,
        uid           => $uid,
        uidNumber     => $uidNumber,
        gidNumber     => $gidNumber,
        homeDirectory => $homedir,
        mailMessageStore => $maildir,
        userPassword     => "{crypt}" . $userpw_crypt,
        loginShell       => "/bin/bash",
    );

    return ldap_update($entry);
}

# Add a machine to LDAP.
sub ldap_add_machine {
    my (
        $cn,        $uid,  , $uidNumber,
        $gidNumber 
	)
	= @_;
    
    debug_msg("Trying to add this machine: uidNumber:$uidNumber  uid:$uid!\n");
    my $entry     = Net::LDAP::Entry->new();
    my $new_entry = "uid=$uid,ou=Machines,ou=People," . $g{basedn};
    $entry->dn($new_entry);
    $entry->add(
		objectclass   => [ 'posixAccount', 'top', 'account'],
		cn            => $cn,
		uid           => $uid,
		uidNumber     => $uidNumber,
		gidNumber     => $gidNumber,
		homeDirectory => "/dev/null",
		loginShell       => "/bin/false",
		);
    
    return ldap_update($entry);;
}

# Change an attribute $attr to the value $value for the user $uid.
sub ldap_mod_user {
    my ( $uid, $attr, $value ) = @_;

    $g{ldap}->modify( "uid=$uid,ou=People," . $g{basedn},
        replace => { $attr => "$value" } );
    $g{modified} = 1;
}

# Change many attributes/value pairs in the 
#   hash-ref $changes_ref for the user $uid.
sub ldap_modify_user {
    my ( $uid, $changes_ref ) = @_;

    my %additions;    # this is for new fields in the ldap-entry
    my $user = ldap_get_user($uid);

    for my $key ( keys %$changes_ref ) {
        unless ( $user->get_value($key) ) {
	    next if ( $key eq "userPassword" );
            $additions{$key} = $changes_ref->{$key};
            delete $changes_ref->{$key};
        }
    }

    my $mesg = $g{ldap}->modify(
        "uid=$uid,ou=People," . $g{basedn}, replace => $changes_ref,
        add => \%additions
    );
    $g{modified} = 1;
    error_msg( "ldap_modify_user Error: "
        . ldap_error_name( $mesg->code ) . " "
        . ldap_error_text( $mesg->code ) . " " )
      if ( $mesg->code );
    return undef;
}

# Change attributes/value pairs in the 
#   hash-ref $changes_ref for the group $gidNumber.
sub ldap_modify_group {
    my ( $cn, $changes_ref ) = @_;

    my %additions;    # this is for new fields in the ldap-entry
    my $group = ldap_get_group($cn);
    for my $key ( keys %$changes_ref ) {
        unless ( $group->get_value($key) ) {
            $additions{$key} = $changes_ref->{$key};
            delete $changes_ref->{$key};
        }
    }

    if ( keys %additions ) {
        my $mesg =
          $g{ldap}->modify( "cn=$cn,ou=Group," . $g{basedn},
            add => \%additions );
        error_msg( "ldap_modify_user Error: "
            . ldap_error_name( $mesg->code ) . " "
            . ldap_error_text( $mesg->code ) . " " )
          if ( $mesg->code );
    }
    if ( keys %$changes_ref ) {
        my $mesg =
          $g{ldap}->modify( "cn=$cn,ou=Group," . $g{basedn},
            replace => $changes_ref );
        error_msg( "ldap_modify_user Error: "
            . ldap_error_name( $mesg->code ) . " "
            . ldap_error_text( $mesg->code ) . " " )
          if ( $mesg->code );
    }
    $g{modified} = 1;
}

# Make a new group $gid, with number $gidNumber, to LDAP.
sub ldap_add_group {
    my ( $cn, $gidNumber, $type, 
         $description ) = @_;

    if( getgrnam( $cn ) ) {
	error_msg("Group $cn exists allready. ");
	return undef;
    } 

    unless ($type)        { $type        = "dontcare" }
    unless ($description) { $description = "dontcare" }
    
    my $entry = Net::LDAP::Entry->new();
    $entry->dn( "cn=$cn,ou=Group," . $g{basedn} );
    
    if ( $type eq "authority_group" ) {
        $entry->add(
            objectclass => [ 'lisAclGroup', 'top', 'lisGroup' ],
            cn          => $cn,
            gidNumber   => $gidNumber,
            groupType   => $type,
            description => $description,
            member => "cn=admin,ou=People," . $g{basedn},
        );
    }
    else {
        $entry->add(
            objectclass => [ 'posixGroup', 'top', 'lisGroup' ],
            cn          => $cn,
            gidNumber   => $gidNumber,
            groupType   => $type,
            description => $description,
        );
    }
    return ldap_update($entry);
}


# Add a user $uid to group $gid.
#strings returned correspond to strings in the lang/en file!
sub ldap_add_user_to_group {
    my ( $uid, $gid ) = @_;

    my $mesg = $g{ldap}->search(
        base   => $g{basedn},
        filter => "(&(cn=$gid)(memberUid=$uid))"
    );
    return "useralreadymember" if ( $mesg->pop_entry );
    $mesg = $g{ldap}->search(
        base   => "ou=Group," . $g{basedn},
        filter => "(cn=$gid)"
    );
    my $group_entry = $mesg->pop_entry;
    return "groupdoesntexist" unless ($group_entry);
    my $group_type = $group_entry->get_value("grouptype");

    if ( $group_type ne "authority_group" ) {
        $g{ldap}->modify(
            "cn=$gid,ou=Group," . $g{basedn},
            add => { 'memberUid' => "$uid" }
        );
    }
    else {
        $g{ldap}->modify(
            "cn=$gid,ou=Group," . $g{basedn},
            add => {
                'memberUid' => "$uid",
                'member'    => "uid=$uid,ou=People," . $g{basedn},
            }
        );
    }
    $g{modified} = 1;
    return "addedgroupuser";
}

# Delete a user $uid from group $gid. 
sub ldap_del_user_from_group {
    my ( $uid, $gid ) = @_;

    my $mesg = $g{ldap}->search(
        base   => "ou=Group," . $g{basedn},
        filter => "(&(cn=$gid)(|(memberUid=$uid)(member=uid=$uid,ou=People,".$g{basedn}.")))"
    );
    my $group_entry = $mesg->pop_entry;
    return "(not a member)" unless ($group_entry);

    my $group_type = $group_entry->get_value("grouptype");
    if ( $group_type ne "authority_group" ) {
        $mesg = $g{ldap}->modify(
	    "cn=$gid,ou=Group," . $g{basedn},
	    delete => { 'memberUid' => "$uid" }
        );
    }
    else {
        $mesg = $g{ldap}->modify(
            "cn=$gid,ou=Group," . $g{basedn},
            delete => {
                'memberUid' => "$uid",
                'member'    => "uid=$uid,ou=People," . $g{basedn},
            }
        );
    }
    $g{modified} = 1;
    return "(User removed from group)";
}

# Delete a user $uid.
sub ldap_delete_user {
    my ( $uid ) = @_;
    
    $g{modified} = 1;
    my $result = $g{ldap}->delete( "uid=$uid,ou=People,".$g{basedn} );
    return $result->code;    
}

# Move a user $uid to the attic.
sub ldap_remove_user {
    my ( $uid ) = @_;
    
    $g{modified} = 1;
    my $result = $g{ldap}->moddn( "uid=$uid,ou=People,".$g{basedn},
                                  newrdn => "uid=$uid",
                                  deleteoldrdn => 1,
                                  newsuperior => "ou=Attic,".$g{basedn},
                                  );
    return $result->code;    
}

# Delete a group $gid.
sub ldap_delete_group {
    my ( $cn ) = @_;
    $g{modified} = 1;
    my $msg = $g{ldap}->delete( "cn=$cn,ou=Group," . $g{basedn} );
    return $msg->code;
}

# Print error message to browser if ldap_error.
sub print_ldap_error {
    my ( $errormsg, $result ) = @_;
    my $output = $errormsg;
    $output .= text('error') . ": \n";
    $output .= ldap_error_name( $result->code ) . ": "
      . ldap_error_text( $result->code );
    return $output;
}

# Close the connetction to the LDAP server.
sub ldap_close {
    $g{ldap}->unbind() if $g{ldap};
    if ($g{modified}) {
	system ("nscd -i passwd >/dev/null 2>&1; nscd -i group >/dev/null 2>&1") 
	    if( -e "/var/run/nscd.pid" );
    }
}

sub ldap_update {
    my( $entry ) = @_;
    my $ret = $entry->update( $g{ldap} ) if $entry;
    $g{modified} = 1;

    return $ret;
}

sub make_cn {
    my ( $first_name, $last_name ) = @_;

    my $cn = "$first_name";
    $cn .= " $last_name" if ($last_name);
    return $cn;
}


sub error_msg {
    my ($mesg) = @_;
    push @{ $g{err} }, $mesg;
}

sub ldap_err_output {
    my $ret;

    while ( @{ $g{err} } ) {
        $ret .= shift @{ $g{err} };
    }
    return $ret;
}

sub debug_msg {
    my ($mesg) = @_;
    push @{ $g{dbg} }, $mesg;
}

sub ldap_dbg_output {
    my $ret;

    while ( @{ $g{dbg} } ) {
        $ret .= shift @{ $g{dbg} };
    }
    return $ret;
}

1;

__END__
