// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
// 
// $Id: MPIPhysicalCommunicationLayer.cpp
// 
//---------------------------------------------------------------------------

#include <mpi.h>
#include <clutils/StringHashMap.h>
#include "MPIPhysicalCommunicationLayer.h"
#include "warped/SerializedInstance.h"

using std::ofstream;
using std::endl;

#define MPI_DATA_TAG  100

MPIPhysicalCommunicationLayer::MPIPhysicalCommunicationLayer(){
};

MPIPhysicalCommunicationLayer::~MPIPhysicalCommunicationLayer(){};

void
MPIPhysicalCommunicationLayer::getCStyleArguments( int &argc, 
						   char ** &argv,
						   const vector<string> &arguments ){
  argc = arguments.size();
  argv = new char*[argc];  
  for( int i = 0; i < argc; i++ ){
    argv[i] = const_cast<char *>(arguments[i].c_str());
    std::cout << "argv[" << i << "] = " << argv[i] << std::endl;
  }
}

void
MPIPhysicalCommunicationLayer::writeProcGroupFile( SimulationConfiguration &configuration ){
  ofstream procGroupFile( "procgroup" );
  if( !procGroupFile ){
    perror("Error writing procgroup file:");
    abort();
  }
  
  StringHashMap<int> count;
  procGroupFile << "local 0" << endl;
  const vector<string> nodeList = configuration.getNodes();
  for( vector<string>::const_iterator i = nodeList.begin();
       i < nodeList.end();
       i++ ){
//     string currentNode = *i;
//     int currentCount = count.find( currentNode );
//     count.insert( currentNode, ++currentCount );
//   }

//   vector<string> *keyVector = count.getKeyVector();
//   for( vector<string>::const_iterator i = keyVector->begin();
//        i < keyVector->end();
//        i++ ){
    procGroupFile << (*i) << " 1 " // << count.find( *i ) << " " 
		  << configuration.getBinaryName() << endl;
  }

  procGroupFile.close();
}

void
MPIPhysicalCommunicationLayer::startMPI( const vector<string> &arguments ){
  int argc = 0;
  char **argv = 0;
  getCStyleArguments( argc, argv, arguments );

  clutils::debug << "About to call MPI_Init" << endl;
  MPI_Init( &argc, &argv );
  clutils::debug << "Done with MPI_Init" << endl;

  // MPICH Rewrites these arguments out from under us - we can't delete the
  // values inside the array.  We'll delete the array at list.
  delete [] argv;
}

void
MPIPhysicalCommunicationLayer::physicalInit( SimulationConfiguration &configuration ){
  writeProcGroupFile( configuration );
  startMPI( configuration.getArguments() );
  mySimulationManagerID = physicalGetId();
}

int
MPIPhysicalCommunicationLayer::physicalGetId() const {
   // get the id of this simulation manager ...
   int id;
   MPI_Comm_rank(MPI_COMM_WORLD, &id);
   return id;
}

void
MPIPhysicalCommunicationLayer::checkPendingSends(){
  for_each( pendingSends.begin(),
	    pendingSends.end(),
	    MPIMessage::finalizeSend() );
  
  pendingSends.erase( 
		     remove_if( pendingSends.begin(),
				pendingSends.end(),
				MPIMessage::operationComplete() ),
		     pendingSends.end() );
}

void
MPIPhysicalCommunicationLayer::physicalSend( const SerializedInstance *messageToSend,
					     unsigned int dest){
   ASSERT( messageToSend != NULL);
   checkPendingSends();
   
   MPI_Request request;
   MPI_Isend( const_cast<char *>(&messageToSend->getData()[0]), 
	      messageToSend->getSize(), 
	      MPI_BYTE, 
	      dest, 
	      MPI_DATA_TAG, 
	      MPI_COMM_WORLD,
	      &request );

   pendingSends.push_back( MPIMessage( messageToSend,
				       request ) );
}

SerializedInstance *
MPIPhysicalCommunicationLayer::physicalProbeRecv(){
  SerializedInstance *retval = 0;

  MPI_Status status;
  char *message = NULL;
  int flag = 0;

  //  if( mySimulationManagerID != 0 )    cout << mySimulationManagerID << "- about to probe" << endl;
  // check to see if any messages have arrived ...
  MPI_Iprobe(MPI_ANY_SOURCE, MPI_ANY_TAG, MPI_COMM_WORLD, &flag, &status );
  //  if( mySimulationManagerID != 0 )  cout << mySimulationManagerID << "Done!" << endl;
  
  if (flag != 0) {
    // how many messages did we get ? ...
    message = new char[status.count];
    ASSERT(message != NULL);
    MPI_Recv( message, 
	      status.count, 
	      MPI_BYTE, 
	      MPI_ANY_SOURCE, 
	      MPI_ANY_TAG, 
	      MPI_COMM_WORLD, 
	      MPI_STATUS_IGNORE );
    retval = new SerializedInstance( message, status.count );
  }

  return retval;
}

bool 
MPIPhysicalCommunicationLayer::physicalProbeRecvBuffer(char *buffer,
                                                       int size,
                                                       bool& sizeStatus){
  MPI_Status status;
  int flag = 0, msgSize = 0;
  
  // check to see if any messages have arrived ...
  MPI_Iprobe(MPI_ANY_SOURCE, MPI_ANY_TAG, MPI_COMM_WORLD, &flag, &status);
  
  if (flag != 0) {
    MPI_Get_count(&status, MPI_BYTE, &msgSize);
    if ( msgSize > size) {
      sizeStatus = false;
    }
    else {
      sizeStatus = true;
      MPI_Recv( buffer, msgSize, MPI_BYTE, MPI_ANY_SOURCE,
		MPI_ANY_TAG, MPI_COMM_WORLD, &status );
    }
    return true;
  }
  else {
    return false;
  }
}

void
MPIPhysicalCommunicationLayer::physicalFinalize(){
  MPI_Finalize();
}

int
MPIPhysicalCommunicationLayer::physicalGetSize() const {
  int size;
  MPI_Comm_size(MPI_COMM_WORLD, &size);
  return size;
}


