// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
// 
// $Id: StateManagerImplementationBase.cpp
// 
//---------------------------------------------------------------------------

#include "StateManagerImplementationBase.h"
#include "SimulationObject.h"
#include "State.h"

using std::cerr;
using std::endl;

StateManagerImplementationBase::StateManagerImplementationBase( SimulationManager *simMgr, 
								unsigned int period) :
  mySimulationManager(simMgr), 
  statePeriod(period),
  myStateQueue( new multiset< SetObject<State> >[ simMgr->getNumberOfSimulationObjects() ] ){
  objectStatePeriod.resize( simMgr->getNumberOfSimulationObjects(), period);
  periodCounter.resize( simMgr->getNumberOfSimulationObjects(), -1 );
}

StateManagerImplementationBase::~StateManagerImplementationBase(){
  delete [] myStateQueue;
}

void
StateManagerImplementationBase::saveState(const VTime &currentTime,
					  SimulationObject *object){
  cerr << "StateManagerImplementationBase::saveState called" << endl;
  cerr << "Exiting simulation ..." << endl;
  exit(-1);
}

unsigned int
StateManagerImplementationBase::getStatePeriod(){
  return statePeriod;
}

const VTime&
StateManagerImplementationBase::restoreState(const VTime &rollbackTime,
					     SimulationObject *object){

  // store this object's id temporarily
  OBJECT_ID *currentObjectID = object->getObjectID();

  multiset< SetObject<State> >::iterator iter_begin =
    myStateQueue[currentObjectID->getSimulationObjectID()].begin();
   
  // start from the end of the queue
  multiset< SetObject<State> >::iterator iter_end =
    myStateQueue[currentObjectID->getSimulationObjectID()].end();

  if (iter_end != iter_begin){
    iter_end--;
      
    // restore current state to the last state in the state queue
    // that is less than the rollback time
    while((*iter_end).getTime() >= rollbackTime){
      object->deallocateState((*iter_end).getElement());
      myStateQueue[currentObjectID->getSimulationObjectID()].erase(iter_end--);
    }

    // at this point, the iterator points to the state we want to restore
    object->getState()->copyState((*iter_end).getElement());

    return (*iter_end).getTime();
  }
  else {
    // this implies that it is a rollback to the initial state
    // restore initial state
    cerr << "ERROR: Unable to restore state at " << rollbackTime << endl;
    cerr << "object" << *currentObjectID << ": Current Simulation Time is "
	 << object->getSimulationTime() << endl;
    return rollbackTime;
  }
}

const VTime&
StateManagerImplementationBase::garbageCollect( const VTime &garbageCollectTime,
						SimulationObject *object ){
  // store this object's id temporarily
  OBJECT_ID *objID = object->getObjectID();

  if(!myStateQueue[objID->getSimulationObjectID()].empty()){
    if( garbageCollectTime != mySimulationManager->getPositiveInfinity() ){
      // construct my search key
      SetObject<State> searchObject(garbageCollectTime);

      // get a handle to the beginning of this queue
      multiset< SetObject<State> >::iterator iter_begin =
	myStateQueue[objID->getSimulationObjectID()].begin();
   
      // find the last state upto which garbage collection will take place.
      // note: one state older than garbageCollectTime will be kept in the
      // queue since we can rollback to GVT. this is our marker.
      multiset< SetObject<State> >::iterator iter_end =
	myStateQueue[objID->getSimulationObjectID()].lower_bound(searchObject);

      if (iter_end != iter_begin) {
	--iter_end;
      }

      while(iter_begin != iter_end){
	object->deallocateState((*iter_begin).getElement());
	myStateQueue[objID->getSimulationObjectID()].erase(iter_begin++);
      }
    }
    else {
      // walk from the front of the queue and delete everything
      multiset< SetObject<State> >::iterator iter_begin =
	myStateQueue[objID->getSimulationObjectID()].begin();
      multiset< SetObject<State> >::iterator iter_end =
	myStateQueue[objID->getSimulationObjectID()].end();
      while(iter_begin != iter_end){
	object->deallocateState((*iter_begin).getElement());
	myStateQueue[objID->getSimulationObjectID()].erase(iter_begin++);
      }
    }
  } // else do nothing

  // Return the lowest timestamped state.
  multiset< SetObject<State> >::iterator iter_begin =
    myStateQueue[objID->getSimulationObjectID()].begin();
  if (!myStateQueue[objID->getSimulationObjectID()].empty()) {
    return (*iter_begin).getTime(); 
  }
  else {
    return mySimulationManager->getPositiveInfinity();
  }
}


void
StateManagerImplementationBase::printStateQueue(const VTime &currentTime,
                                                SimulationObject *object,
                                                ostream &out){
  // store this object's id temporarily
  OBJECT_ID *currentObjectID = object->getObjectID();
   
  if(!myStateQueue[currentObjectID->getSimulationObjectID()].empty()){
      
    // get a handle to the beginning of this queue
    multiset< SetObject<State> >::iterator iter_begin =
      myStateQueue[currentObjectID->getSimulationObjectID()].begin();
      
    // get a handle to the end of this queue
    multiset< SetObject<State> >::iterator iter_end =
      myStateQueue[currentObjectID->getSimulationObjectID()].end();
      
    // iterate through the queue and print each state's time
    out << "object " << currentObjectID->getSimulationObjectID()
	<< "@(" << currentTime << "): ";
    while (iter_begin != iter_end){
      out << *iter_begin << " ";
      ++iter_begin;
    }
    out << endl;
  }
}
