// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
// 
// $Id: main.cpp
// 
//---------------------------------------------------------------------------

#include "warped.h"
#include "Application.h"
#include "WarpedMain.h"
#include "SimulationConfiguration.h"
#include "Simulation.h"
#include "Spinner.h"
#include <string>
#include <fstream>
#include <clutils/StringUtilities.h>
#include <unistd.h>
#include <sstream>
#include <cstdio>

// We need these to register their deserializers
#include "CirculateInitializationMessage.h"
#include "DeserializerManager.h"
#include "EventMessage.h"
#include "GVTUpdateMessage.h"
#include "InitializationMessage.h"
#include "IntVTime.h"
#include "MatternGVTMessage.h"
#include "NegativeEventMessage.h"
#include "StartMessage.h"
#include "TerminateToken.h"
#include "WarpedMain.h"

using std::string;
using std::fstream;
using std::ofstream;
using std::ifstream;
using std::istringstream;
using std::cout;
using std::cerr;
using std::cin;
using std::endl;

// maximum length of command line arguments
#define MAX_COMMAND_LINE_LENGTH 256

ArgumentParser::ArgRecord *
WarpedMain::getArgumentList( WarpedMain &main ) {
  static ArgumentParser::ArgRecord args[] = {
    {"-configuration", "specify configuration file", &main.configurationFileName,
     ArgumentParser::STRING, false},
    {"-simulateUntil", "specify a simulation end time", &main.simulateUntil,
     ArgumentParser::STRING, false},
    {"-debug", "display debug messages", &main.debugFlag, ArgumentParser::BOOLEAN, false},
    {"", ""}
  };
    
  return args;
};

WarpedMain::WarpedMain( Application *initApplication ) :
  errors( 0 ),
  warnings( 0 ),
  configurationFileName( "" ),
  debugFlag( false ),
  simulateUntil( "" ),
  slaveArgumentFile( "slave-args" ),
  myApplication( initApplication ),
  mySimulation( 0 ){}

void
WarpedMain::displayParameters( string executableName ){
  ArgumentParser ap( getArgumentList( *this ) );
  cerr << "Usage " << executableName << " -options\n";
  cerr << "Where options is one of the following ['*'"
       << " indicates mandatory option(s)]:\n" << ap; 
}

vector<string>
WarpedMain::readSlaveArguments( const vector<string> &commandLineArgs ){
  string path(commandLineArgs[0]);
  string fileToRead = path.substr(0, path.find_last_of("/")+1) + slaveArgumentFile;
  
  cerr << "About to read \"" << fileToRead << "\"" << endl;

  ifstream infile( fileToRead.c_str() );
  if(!infile){
    perror( ("Slave error - unable to open file " + fileToRead + ":" ).c_str() );
    exit(-1);
  }
   
  string argument;
  vector<string> slaveArgs;
  slaveArgs.insert( slaveArgs.begin(),
		    commandLineArgs.begin(),
		    commandLineArgs.end() );

  while(infile >> argument){
    clutils::debug << "  arg:\"" << argument << "\"" << endl;
    slaveArgs.push_back( argument );
  }
  infile.close();

  return slaveArgs;
}

bool
WarpedMain::checkConfigFile( string configFileName ){
  string choice;
  if( configFileName == "" ){
    cerr << "A Simulation Configuration File has not been specified" << endl;
    cerr << "Shall I create a default configuration: [y/n]: ";
    cin >> choice;
    // we are going to uppercase whatever the choice is ...
    string upChoice = upperCase(choice);
    if( upChoice == "Y" ){
      cerr << "Creating default configuration file: simulation.conf"
	   << endl;
      cerr << "This has not been implemented yet ..." << endl;
      exit(-1);
    }
  }
  return true;
}

void
WarpedMain::registerKernelDeserializers(){
  CirculateInitializationMessage::registerDeserializer();
  EventMessage::registerDeserializer();
  GVTUpdateMessage::registerDeserializer();
  InitializationMessage::registerDeserializer();
  IntVTime::registerDeserializer();
  MatternGVTMessage::registerDeserializer();
  NegativeEventMessage::registerDeserializer();
  StartMessage::registerDeserializer();
  TerminateToken::registerDeserializer();
}

vector<string>
WarpedMain::doSlaveInit( const vector<string> &commandLineArgs ){
  cerr << "doSlaveInit called" << endl;

  // if i am a slave process, then i have to read the command line
  // parameters from a file instead of reading it from argc and
  // args as MPI hasn't sent us the simulation command line
  // parameters yet. So we are forced to read this from a file
  // called "mpich-args".

  // the following is for constructing the path to the mpich-args
  // file. MPI gives us the whole path to the executable in
  // args[0]. By ignoring the executable name in the path, we can
  // obtain the path to the file this slave needs to read.

  // array of parameters that slaves need
  return readSlaveArguments( commandLineArgs );
}

void
WarpedMain::writeSlaveArguments( const vector<string> &masterArgs ){
  ofstream outFile( slaveArgumentFile.c_str() );
  if(!outFile){
    perror( ("Error trying to write " + slaveArgumentFile + ":").c_str() );
    exit(-1);
  }
      
  for( unsigned int count = 1; count < masterArgs.size(); count++ ){
    outFile << masterArgs[count] << " ";
  }
  outFile << endl;
}

vector<string>
WarpedMain::doMasterInit( const vector<string> &masterArgs ){
  cerr << "doMasterInit called" << endl;
  // if i am the master, i should create the mpich-args file so
  // that user doesn't have to do this. The reason I need to do
  // this is because only I have complete access to the
  // simulation's command line parameters (at this point; before a
  // call to physicalInit).
  writeSlaveArguments( masterArgs );
  return masterArgs;
}

vector<string>
WarpedMain::buildArgumentVector( int argc, char **argv ){
  vector<string> retval;
  for( int i = 0; i < argc; i++ ){
    retval.push_back( string( argv[i] ) );
  }
  return retval;
}

bool
WarpedMain::amSlave( const vector<string> &args ) const {
  bool retval = false;
  for( vector<string>::const_iterator i = args.begin();
       i < args.end(); 
       i++ ){
    if( *i == "-p4amslave" ){
      retval = true;
    }
  }
  return retval;
}

SimulationConfiguration *
WarpedMain::readConfiguration( const string &configurationFileName,
			       const vector<string> &argumentVector ){
  SimulationConfiguration *configuration = 0;
  if( configurationFileName != "" ){
    if( checkConfigFile( configurationFileName ) == false ){
      cerr << "Can't read configuration file " << configurationFileName << endl;
      exit( -1 );
    }
    configuration = SimulationConfiguration::parseConfiguration( configurationFileName,
								 argumentVector );
    if( configuration == 0 ){
      cerr << "There was a problem parsing configuration " << configurationFileName 
	   << ", exiting." << endl;
      exit( -1 );
    }
  }
  return configuration;
}

vector<string>
WarpedMain::initMasterOrSlave( const vector<string> &args ){
  vector<string> argumentsToUse;

  // We have to do some special stuff here due to the fact that we haven't
  // read our configuration yet and don't even know if we're running
  // parallel.  Furthermore, we won't have our parameters for some time if
  // we happen to be an MPI slave process.

  if( amSlave( args ) ){
    argumentsToUse = doSlaveInit( args );
  }
  else {
    argumentsToUse = doMasterInit( args );
  }

  ArgumentParser ap( getArgumentList( *this ));
  ap.checkArgs( argumentsToUse, false );

  return argumentsToUse;
}

void
WarpedMain::initializeSimulation( const vector<string> &commandLineArgs ){
  registerKernelDeserializers();
  myApplication->registerDeserializers();

  if( debugFlag == true ){
    clutils::enableDebug();
  }

  vector<string> masterOrSlaveArgs = initMasterOrSlave( commandLineArgs );
  SimulationConfiguration *configuration = readConfiguration( configurationFileName,
							      masterOrSlaveArgs );
  if( configuration != 0 ){
    Spinner::spinIfRequested( "SpinBeforeConfiguration", *configuration );
  }


  // We have to let the application initialize before we can do much else.
  myApplication->initialize( masterOrSlaveArgs );

  // else, configuration is NULL, and we'll run with a default configuration
  mySimulation = Simulation::instance( configuration, myApplication );
  mySimulation->initialize();

  delete configuration;
}

void
WarpedMain::simulate( const VTime &simulateUntil ){
  mySimulation->simulate( simulateUntil );
}

bool
WarpedMain::simulationComplete( ){
  return mySimulation->simulationComplete();
}

void
WarpedMain::finalize(){
  mySimulation->finalize();
}

const VTime &
WarpedMain::getCommittedTime(){
  return mySimulation->getCommittedTime();
}

const VTime &
WarpedMain::getNextEventTime(){
  return mySimulation->getNextEventTime();
}


int 
WarpedMain::main( int argc, char **argv ){
  vector<string> args = buildArgumentVector( argc, argv );
  initializeSimulation( args );
    
  simulate( myApplication->getPositiveInfinity() );

  finalize();

  return errors;
}
