// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
// 
// $Id: FileQueue.cpp
// 
//---------------------------------------------------------------------------

#include "FileQueue.h"
#include <cstdio>
#include <fstream>
using std::ofstream;

int FileQueue::standardOut = -1;

FileQueue::FileQueue(const string &fileName) {
  open( fileName );
}

FileQueue::FileQueue(int standardOutFlag){
  // We instantiate a out file queue for writing output to stdout.  This
  // is done as follows: FileQueue is newed with the constructor having
  // the argument FileQueue::standardOut.  This tells the file queue to
  // output all the data that it recieves to stdout and not to a file.
  // The file id is also accessed as FileQueue::standardOut.
  if(standardOutFlag != 1){
    cerr << "FileQueue: Error in use of constructor" << endl;
    cerr << "Integer Parameter to constructor != 1" << endl;
    abort();
  }
  statusBit = standardOutFlag;
  
  // outFile is set to cout
  outFile = &cout;
  if (!outFile->good()) {
    cout << "FileQueue: error opening standard out " << "\n";
    abort();
  }
}


FileQueue::~FileQueue() {
  // garbageCollect(VTimeFactory::getPositiveInfinity());
  if(statusBit == 0){
     (static_cast<ofstream *>(outFile))->close();
  }
  delete outFile;
}

void
FileQueue::close() {

   if (statusBit == 0) {
      ((ofstream *)outFile)->close();
   }
}

void
FileQueue::open(const string &fileName) {
  open( fileName, std::ios::in );
}

void
FileQueue::open(const string &fileName, ios::openmode mode) {
  outFileName = fileName;
  statusBit = 0;
  
  outFile = new ofstream(fileName.c_str(), mode);
  if (!outFile->good()) {
    perror( ("FileQueue: Error opening file " + fileName ).c_str() );
    abort();
  }
}

void 
FileQueue::garbageCollect(const VTime& garbageCollectTime){

   multiset< FileData >::iterator iter_begin = begin();
   multiset< FileData >::iterator iter_end = end();
   
   while(iter_begin != iter_end){
      if ((*iter_begin).getTime() <= garbageCollectTime){
         outFile->seekp(0, ios::end);
         const string *line = (*iter_begin).getLine();
         
         *outFile << *line << endl;
         // this is a little trick using iterators.
         // the old value of iter_begin is saved and passed to erase
         // AFTER the iterator has been incremented.
         delete line;
         erase(iter_begin++);
      }
      else {
         break;
      }
   }
}


void
FileQueue::rollbackTo(const VTime& rollbackToTime){

   FileData findElement(rollbackToTime);
   multiset< FileData >::iterator iter_find = lower_bound(findElement);
   multiset< FileData >::iterator iter_end = end();
   // delete every thing after (and including) the found element
   while(iter_find != iter_end){
      const string *line = (*iter_find).getLine();
      delete line;
      erase(iter_find++);
   }
}

void
FileQueue::storeLine(const VTime &time, string *line){

   FileData input(time, line);
   insert(input);
}
