/*  $Id: console-app.cpp,v 1.2 2010/04/20 03:30:20 sarrazip Exp $
    console-app.cpp - Console application main function

    verbiste - French conjugation system
    Copyright (C) 2003-2009 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#include "gui/conjugation.h"
#include "verbiste/FrenchVerbDictionary.h"

#include <libintl.h>
#include <locale.h>
#include <stdlib.h>
#include <iostream>

using namespace std;




class ConsoleApp
{
public:

    ConsoleApp(verbiste::FrenchVerbDictionary &_fvd)
    :   fvd(_fvd),
	includePronouns(false)
    {
    }

    void processCommand(const string &utf8Command)
    {
	string lowerCaseLatin1UserText = fvd.tolowerLatin1(fvd.utf8ToLatin1(utf8Command));

	if (lowerCaseLatin1UserText == "/showpronouns" || lowerCaseLatin1UserText == "/sp")
	{
	    includePronouns = true;
	    return;
	}
	if (lowerCaseLatin1UserText == "/hidepronouns" || lowerCaseLatin1UserText == "/hp")
	{
	    includePronouns = false;
	    return;
	}

	/*
	    For each possible deconjugation, take the infinitive form and
	    obtain its complete conjugation.
	*/
	vector<InflectionDesc> v;
	fvd.deconjugate(lowerCaseLatin1UserText, v);

	string prevLatin1Infinitive;
	size_t numPages = 0;

	cout << "<result input='" << utf8Command << "'>\n";

	for (vector<InflectionDesc>::const_iterator it = v.begin();
						    it != v.end(); it++)
	{
	    const InflectionDesc &d = *it;

	    VVVS latin1Conjug;
	    getConjugation(fvd, d.infinitive, latin1Conjug, includePronouns);

	    if (latin1Conjug.size() == 0           // if no tenses
		|| latin1Conjug[0].size() == 0     // if no infinitive tense
		|| latin1Conjug[0][0].size() == 0  // if no person in inf. tense
		|| latin1Conjug[0][0][0].empty())  // if infinitive string empty
	    {
		continue;
	    }

	    string latin1Infinitive = latin1Conjug[0][0][0];

	    if (latin1Infinitive == prevLatin1Infinitive)
		continue;

	    cout << "<conjugation verb='" << fvd.latin1ToUTF8(latin1Infinitive) << "'>\n";
	    numPages++;

	    int i = 0;
	    for (VVVS::const_iterator t = latin1Conjug.begin();
				    t != latin1Conjug.end(); t++, i++)
	    {
		const VVS &latin1Tense = *t;

		if (i == 1)
		    i = 4;
		else if (i == 11)
		    i = 12;
		assert(i >= 0 && i < 16);

		int row = i / 4;
		int col = i % 4;

		string utf8TenseName = getTenseNameForTableCell(row, col);
		assert(!utf8TenseName.empty());

		string latin1Persons = createTableCellText(
						latin1Tense,
						lowerCaseLatin1UserText,
						"*",
						"");

		string utf8Persons = fvd.latin1ToUTF8(latin1Persons);

		cout << "<tense name='" << utf8TenseName << "'>\n";
		cout << utf8Persons << "\n";
		cout << "</tense>\n";
	    }

	    cout << "</conjugation>\n";

	    prevLatin1Infinitive = latin1Infinitive;

	}   // for

	cout << "</result>\n";
    }

private:

    verbiste::FrenchVerbDictionary &fvd;
    bool includePronouns;

};


int
main(int argc, char *argv[])
{
    bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
    bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    textdomain(GETTEXT_PACKAGE);

    verbiste::FrenchVerbDictionary *fvd;
    try
    {
	fvd = initVerbDict();
    }
    catch(logic_error &e)
    {
	cerr << PACKAGE_FULL_NAME << ": " << e.what() << endl;
	return EXIT_FAILURE;
    }

    ConsoleApp app(*fvd);

    string utf8Command;
    while (getline(cin, utf8Command))
	app.processCommand(utf8Command);

    return EXIT_SUCCESS;
}
