/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse m�l :
	BILLARD, non joignable par m�l ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant � visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est r�gi par la licence CeCILL soumise au droit fran�ais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffus�e par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez acc�der � cet en-t�te signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept� les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include <gtk/gtk.h>
#ifdef HAVE_LIBGTKGLEXT_X11_1_0
#include <gtk/gtkgl.h>
#endif
#include <unistd.h> /* For the access markers R_OK, W_OK ... */

#include "support.h"
#include "opengl.h"
#include "visu_gtk.h"
#include "visu_object.h"
#include "visu_basic.h"
#include "visu_commandLine.h"
#include "visu_configFile.h"
#include "visu_rendering.h"
#include "visu_extension.h"
#include "renderingMethods/renderingAtomic.h"
#include "renderingMethods/renderingSpin.h"
#include "coreTools/toolShade.h"
#include "coreTools/toolConfigFile.h"
#include "extraFunctions/plane.h"
#include "extraFunctions/dataFile.h"
#include "panelModules/panelSurfaces.h"
#include "gtk_renderingWindowWidget.h"
#include "extraGtkFunctions/gtk_colorComboBoxWidget.h"
#include "OSOpenGL/visu_openGL.h"
#include "extensions/fogAndBGColor.h"
#include "extensions/box.h"
#include "extensions/scale.h"

/**
 * SECTION: visu_gtk
 * @short_description: Basic GUI routines, for preview, standard
 * dialogs...
 */

/* Parameters. */
#define FLAG_PARAMETER_PREVIEW "main_usePreview"
#define DESC_PARAMETER_PREVIEW "Automatically compute preview in filechooser ; boolean"
static gboolean usePreview;

/* Static variables. */
static gint gtkFileChooserWidth = -1, gtkFileChooserHeight = -1;

/* This hashtable associate a #RenderingMethod with a Gtk dialog to
   choose files. */
static GHashTable *visuGtkLoadMethods = NULL;
/* Store the last open directory. It is initialised
   to current working directory. */
static gchar *visuGtkLastDir;

static GtkWindow *visuGtkPanel;
static GtkWindow *visuGtkRender;
static GtkWidget *visuGtkRenderArea = (GtkWidget*)0;

/* Local routines. */
static void exportParameters(GString *data, VisuData *dataObj);
static gboolean readUsePreview(gchar **lines, int nbLines,
			       int position, VisuData *dataObj, GError **error);
static void onPreviewToggled(GtkToggleButton *button, gpointer data);
static void initVisuGtk();


/* Draw the alert window with the specified message. */
void visuGtkRaise_warning(gchar *action, gchar *message, GtkWindow *window)
{
  GtkWidget *alert;
  gchar *str;
  
  if (!window)
    window = visuGtkRender;

  DBG_fprintf(stderr, "Visu Gtk: raise the error dialog (parent %p).\n",
	      (gpointer)window);

#if GTK_MAJOR_VERSION > 2 || GTK_MINOR_VERSION > 5
  str = action;
#else
  str = message;
#endif
  alert = gtk_message_dialog_new(GTK_WINDOW(window),
				 GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
				 GTK_MESSAGE_WARNING, GTK_BUTTONS_OK, "%s", str);
  gtk_window_set_title(GTK_WINDOW(alert), _("V_Sim error message"));
#if GTK_MAJOR_VERSION > 2 || GTK_MINOR_VERSION > 5
  gtk_widget_set_name(alert, "error");
#else
  gtk_widget_set_name(alert, action);
#endif
#if GTK_MAJOR_VERSION > 2 || GTK_MINOR_VERSION > 5
  gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(alert), "%s", message);
#endif
  gtk_widget_show_all(alert);

  /* block in a loop waiting for reply. */
  gtk_dialog_run(GTK_DIALOG(alert));
  gtk_widget_destroy(alert);
}
/* Draw the alert window with the specified message. */
void visuGtkRaise_warningLong(gchar *action, gchar *message, GtkWindow *window)
{
  GtkWidget *alert;
  GtkWidget *text, *scroll;
  GtkTextBuffer *buf;

  if (!window)
    window = visuGtkRender;

  alert = gtk_message_dialog_new(GTK_WINDOW(window),
				 GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
				 GTK_MESSAGE_WARNING,
				 GTK_BUTTONS_OK,
				 "%s", action);
  gtk_window_set_resizable(GTK_WINDOW(alert), TRUE);
  gtk_widget_set_name(alert, "error");
#if GTK_MAJOR_VERSION > 2 || GTK_MINOR_VERSION > 5
  gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(alert),
					   _("Output errors:"));
#endif
  scroll = gtk_scrolled_window_new((GtkAdjustment*)0, (GtkAdjustment*)0);
  gtk_widget_set_size_request(scroll, 300, 200);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scroll),
				      GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(alert))),
		     scroll, TRUE, TRUE, 2);
  text = gtk_text_view_new();
  gtk_text_view_set_editable(GTK_TEXT_VIEW(text), FALSE);
  gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(text), FALSE);
  gtk_container_add(GTK_CONTAINER(scroll), text);
  buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW(text));
  gtk_text_buffer_set_text(GTK_TEXT_BUFFER(buf), message, -1);
  gtk_widget_show_all(alert);

  /* block in a loop waiting for reply. */
  gtk_dialog_run (GTK_DIALOG (alert));
  gtk_widget_destroy(alert);
}
/* This function must be called in a blocking loop to update
   different things like the gtk for example. */
void visuGtkWait(void)
{
  while(gtk_events_pending())
    gtk_main_iteration();
}

struct _VisuGtkSetFilesFunc
{
  VisuGtkSetFilesFunc load;
};
void visuGtkSet_renderingSpecificMethods(VisuRendering *method,
					 VisuGtkSetFilesFunc methodLoad)
{
  struct _VisuGtkSetFilesFunc *ct;

  g_return_if_fail(method);

  if (!visuGtkLoadMethods)
    initVisuGtk();

  DBG_fprintf(stderr, "Visu Gtk: set rendering specific for method '%s'.\n",
	      visu_rendering_getName(method, FALSE));
  ct = g_malloc(sizeof(struct _VisuGtkSetFilesFunc));
  ct->load = methodLoad;
  g_hash_table_insert(visuGtkLoadMethods, method, ct);
}
VisuGtkSetFilesFunc visuGtkGet_renderingSpecificOpen(VisuRendering *method)
{
  struct _VisuGtkSetFilesFunc *ct;

  g_return_val_if_fail(method, (VisuGtkSetFilesFunc)0);
  
  if (!visuGtkLoadMethods)
    initVisuGtk();

  DBG_fprintf(stderr, "Visu Gtk: looking for a specific load interface for rendering"
	      " method '%s'...\n", visu_rendering_getName(method, FALSE));
  
  ct = (struct _VisuGtkSetFilesFunc *)g_hash_table_lookup(visuGtkLoadMethods, method);
  return (ct && ct->load)?ct->load:visuGtkGet_fileFromDefaultFileChooser;
}

gchar** visuGtkGet_selectedDirectory(GtkWindow *parent, gboolean multiple,
				     const gchar *dir)
{
  GtkWidget *file_selector, *hbox, *wd;
  gchar **dirnames;
  char *directory;
  GSList* names, *tmpLst;
  int i;

  /* Create the selector */
  if (!parent)
    parent = visuGtkRender;

  file_selector = gtk_file_chooser_dialog_new(_("Choose a directory"), parent,
					      GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OPEN, GTK_RESPONSE_OK,
					      NULL);
  if (gtkFileChooserWidth > 0 || gtkFileChooserHeight > 0)
    gtk_window_set_default_size(GTK_WINDOW(file_selector), gtkFileChooserWidth,
				gtkFileChooserHeight);
  if (multiple)
    {
      hbox = gtk_hbox_new(FALSE, 0);
      gtk_file_chooser_set_extra_widget(GTK_FILE_CHOOSER(file_selector), hbox);
      wd = gtk_image_new_from_stock(GTK_STOCK_HELP, GTK_ICON_SIZE_MENU);
      gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
      wd = gtk_label_new("");
      gtk_box_pack_start(GTK_BOX(hbox), wd, TRUE, TRUE, 5);
      gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
      gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
      gtk_label_set_markup(GTK_LABEL(wd), _("<span size=\"smaller\">Choose several"
					    " directories using the"
					    " <span font_desc=\"courier\">"
					    "Control</span> key.</span>"));
      gtk_widget_show_all(hbox);
    }
  if (!dir)
    directory = visuGtkGet_lastOpenDirectory();
  else
    directory = (char*)dir;
  if (directory)
    {
      DBG_fprintf(stderr, "Visu Gtk: open a directory chooser, set on '%s'.\n",
		  directory);
      gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(file_selector),
					  directory);
    }
  gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(file_selector), multiple);

  gtk_widget_set_name(file_selector, "filesel");
  gtk_window_set_position(GTK_WINDOW(file_selector), GTK_WIN_POS_CENTER_ON_PARENT);
  gtk_window_set_modal(GTK_WINDOW (file_selector), TRUE);

  if (gtk_dialog_run (GTK_DIALOG (file_selector)) == GTK_RESPONSE_OK)
    {
      names = gtk_file_chooser_get_filenames(GTK_FILE_CHOOSER(file_selector));
      dirnames = g_malloc(sizeof(gchar*) * (g_slist_length(names) + 1));
      tmpLst = names;
      i = 0;
      while(tmpLst)
	{
	  dirnames[i] = (gchar*)tmpLst->data;
	  i += 1;
	  tmpLst = g_slist_next(tmpLst);
	}
      dirnames[i] = (gchar*)0;
      g_slist_free(names);
    }
  else
    dirnames = (gchar**)0;

  gtk_window_get_size(GTK_WINDOW(file_selector), &gtkFileChooserWidth,
		      &gtkFileChooserHeight);

  gtk_widget_destroy (file_selector);

  return dirnames;
}

static void free_image(guchar *image, gpointer data _U_)
{
  DBG_fprintf(stderr, "Visu Gtk: free the preview image data.\n");
  g_free(image);
}
void visuGtkPreview_update(VisuGtkPreview *preview, const char *filenames[])
{
  gchar *text, *comment;
  VisuData *data;
  gboolean valid;
  GError *error;
  VisuPixmapContext *dumpData;
  GArray* image;
  GdkPixbuf *pixbuf;
  VisuRendering *method;
  gint width, height, nRows, nFiles, nb;
  VisuDataIter iter;
  GtkWidget *wd;
  VisuRenderingWindow *currentWindow;
  VisuOpenGLView *view;
  float centre[3], *extens;

  DBG_fprintf(stderr, "Visu Gtk: update preview with given filenames.\n");

  if (preview->table)
    {
      gtk_widget_destroy(preview->table);
      preview->table = (GtkWidget*)0;
    }

  method = visu_object_getRendering(VISU_INSTANCE);
  g_return_if_fail(method);

  nb = visu_rendering_getNFileTypes(method);
  for (nFiles = 0; filenames[nFiles]; nFiles++);

  if (nFiles < nb)
    gtk_image_set_from_pixbuf(GTK_IMAGE(preview->image), (GdkPixbuf*)0);
  else
    {
      if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(preview->check)))
	{
	  gtk_image_set_from_pixbuf(GTK_IMAGE(preview->image), (GdkPixbuf*)0);
	  return;
	}

      width = 150;
      height = 150;
      data = visu_data_newWithSize(width, height);
      for (nFiles = 0; filenames[nFiles]; nFiles++)
	visu_data_addFile(data, (char*)filenames[nFiles], nFiles, (ToolFileFormat*)0);

      /* We save the current rendering context. */
      currentWindow = RENDERING_WINDOW(visuGtkRenderArea);

      /* We change the context since loading a new data will
	 generate some changes on the OpenGL rendering. */
      dumpData = visu_pixmap_context_new((guint)width, (guint)height);
      if (!dumpData)
	{
	  /* We free the VisuData. */
	  g_object_unref(data);

	  gtk_image_set_from_stock(GTK_IMAGE(preview->image),
				   GTK_STOCK_DIALOG_ERROR,
				   GTK_ICON_SIZE_DIALOG);
	  wd = gtk_label_new(_("Internal error,\nno preview available"));
	  gtk_box_pack_start(GTK_BOX(preview->vbox), wd,
			     FALSE, FALSE, 0);
	  gtk_widget_show_all(wd);
	  preview->table = wd;

          /* Set the rendering window current for OpenGL. */
          renderingWindowSet_current(currentWindow, TRUE);

	  return;
	}

      error = (GError*)0;
      valid = visu_rendering_load(method, data, (ToolFileFormat*)0, 0, &error);
      if (!valid)
	{
	  gtk_image_set_from_stock(GTK_IMAGE(preview->image),
				   GTK_STOCK_DIALOG_QUESTION,
				   GTK_ICON_SIZE_DIALOG);
	  wd = gtk_label_new(_("Not a V_Sim file"));
	  gtk_box_pack_start(GTK_BOX(preview->vbox), wd,
			     FALSE, FALSE, 0);
	  gtk_widget_show_all(wd);
	  preview->table = wd;
	}
      else
	{
	  if (error)
	    {
	      gtk_image_set_from_stock(GTK_IMAGE(preview->image),
				       GTK_STOCK_DIALOG_ERROR,
				       GTK_ICON_SIZE_DIALOG);
	      wd = gtk_label_new(_("This file has errors"));
	      gtk_box_pack_start(GTK_BOX(preview->vbox), wd,
				 FALSE, FALSE, 0);
	      gtk_widget_show_all(wd);
	      preview->table = wd;
	      g_error_free(error);
	    }
	  else
	    {
	      extens = visu_data_getBoxExtens(data);
	      view = visu_data_getOpenGLView(data);
              openGLCameraSet_refLength(view->camera, extens[0], unit_undefined);
	      visu_data_createAllElements(data);
	      visu_data_getBoxCentre(data, centre);
	      /* We call the given draw method. */
	      openGLInit_context();
	      openGLModelize(view->camera);
	      openGLProject(view->window, view->camera, extens[1]);
	      visuExtensions_rebuildList(data, VISU_GLEXT_FOG_AND_BG_ID);
	      visuExtensions_rebuildList(data, EXT_VISU_DATA_ID);
	      visuExtensions_rebuildList(data, VISU_GLEXT_BOX_ID);
	      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	      glMatrixMode(GL_MODELVIEW);
	      glTranslated(-centre[0], -centre[1], -centre[2]);
	      visuExtensions_callList(VISU_GLEXT_FOG_AND_BG_ID, FALSE);
	      visuExtensions_callList(EXT_VISU_DATA_ID, FALSE);
	      visuExtensions_callList(VISU_GLEXT_BOX_ID, FALSE);
	      /* We copy the pixmap into generic data. */
	      image = visuOpenGLGet_pixmapData((guint)width,
					       (guint)height, FALSE);
	      pixbuf =
		gdk_pixbuf_new_from_data((guchar*)image->data, GDK_COLORSPACE_RGB,
					 FALSE, 8, width, height, 3 * width,
					 free_image, (gpointer)0);
	      g_array_free(image, FALSE);
	      gtk_image_set_from_pixbuf(GTK_IMAGE(preview->image), pixbuf);
	      gdk_pixbuf_unref(pixbuf);
	      /* We reset the flag of material. */
	      comment = visu_data_getFileCommentary(data, 0);
	      visu_data_iterNew(data, &iter);
	      nRows = (comment && comment[0])?
		iter.nElements + 3:iter.nElements + 1;
	      preview->table = gtk_table_new(iter.nElements + 1, 2, FALSE);
	      wd = gtk_label_new(_("<i>Box composition:</i>"));
	      gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
	      gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
	      gtk_table_attach(GTK_TABLE(preview->table), wd, 0, 2, 0, 1,
			       GTK_FILL | GTK_EXPAND, GTK_SHRINK, 2, 5);
	      for (visu_data_iterStart(data, &iter); iter.element;
		   visu_data_iterNextElement(data, &iter))
		{
		  iter.element->materialIsUpToDate = FALSE;
		  wd = gtk_label_new("");
		  text = g_markup_printf_escaped
		    (_("<span size=\"small\"><b>%s:</b></span>"),
		     iter.element->name);
		  gtk_label_set_markup(GTK_LABEL(wd), text);
		  gtk_misc_set_alignment(GTK_MISC(wd), 1., 0.5);
		  g_free(text);
		  gtk_table_attach(GTK_TABLE(preview->table), wd, 0, 1,
				   iter.iElement + 1, iter.iElement + 2,
				   GTK_FILL, GTK_SHRINK, 2, 0);
		  wd = gtk_label_new("");
		  if (iter.nStoredNodes[iter.iElement] > 1)
		    text = g_markup_printf_escaped
		      (_("<span size=\"small\">%d nodes</span>"),
		       iter.nStoredNodes[iter.iElement]);
		  else
		    text = g_markup_printf_escaped
		      (_("<span size=\"small\">%d node</span>"),
		       iter.nStoredNodes[iter.iElement]);
		  gtk_label_set_markup(GTK_LABEL(wd), text);
		  gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
		  g_free(text);
		  gtk_table_attach(GTK_TABLE(preview->table), wd, 1, 2,
				   iter.iElement + 1, iter.iElement + 2,
				   GTK_FILL | GTK_EXPAND, GTK_SHRINK, 2, 0);
		}
	      if (comment && comment[0])
		{
		  wd = gtk_label_new(_("<i>Description:</i>"));
		  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
		  gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
		  gtk_table_attach(GTK_TABLE(preview->table), wd, 0, 2,
				   iter.nElements + 2, iter.nElements + 3,
				   GTK_FILL | GTK_EXPAND, GTK_SHRINK, 2, 5);
		  wd = gtk_label_new("");
		  text = g_markup_printf_escaped("<span size=\"small\">%s</span>",
						 comment);
		  gtk_label_set_markup(GTK_LABEL(wd), text);
		  g_free(text);
		  gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
		  gtk_label_set_justify(GTK_LABEL(wd), GTK_JUSTIFY_FILL);
		  gtk_label_set_line_wrap(GTK_LABEL(wd), TRUE);
		  gtk_widget_set_size_request(wd, 150, -1);
		  gtk_table_attach(GTK_TABLE(preview->table), wd, 0, 2,
				   iter.nElements + 3, iter.nElements + 4,
				   GTK_FILL | GTK_EXPAND, GTK_SHRINK, 2, 0);
		}
	      gtk_box_pack_start(GTK_BOX(preview->vbox), preview->table,
				 FALSE, FALSE, 0);
	      gtk_widget_show_all(preview->table);
	    }
	}
      /* Set the rendering window current for OpenGL. */
      renderingWindowSet_current(currentWindow, TRUE);

      /* We free the pixmap context. */
      visu_pixmap_context_free(dumpData);

      /* We free the VisuData. */
      g_object_unref(data);
    }
  DBG_fprintf(stderr, "Visu Gtk: end of preview creation.\n");
}
static void onPreviewToggled(GtkToggleButton *button, gpointer data _U_)
{
  usePreview = gtk_toggle_button_get_active(button);
}

VisuGtkPreview* visuGtkPreview_add(VisuGtkPreview *preview, GtkFileChooser *chooser)
{
  GtkWidget *wd, *frame;

  g_return_val_if_fail(preview, (VisuGtkPreview*)0);

  preview->vbox = gtk_vbox_new(FALSE, 0);
  preview->check = gtk_check_button_new_with_mnemonic(_("_Preview:"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(preview->check), usePreview);
  g_signal_connect(G_OBJECT(preview->check), "toggled",
		   G_CALLBACK(onPreviewToggled), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(preview->vbox), preview->check, FALSE, FALSE, 5);
  wd = gtk_alignment_new(0.5, 0.5, 0., 0.);
  gtk_box_pack_start(GTK_BOX(preview->vbox), wd, TRUE, TRUE, 0);
  frame = gtk_frame_new(NULL);
  gtk_widget_set_size_request(frame, 150, 150);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_add(GTK_CONTAINER(wd), frame);
  preview->image = gtk_image_new();
  gtk_container_add(GTK_CONTAINER(frame), preview->image);
  preview->table = NULL;
  gtk_widget_show_all(preview->vbox);
  
  gtk_file_chooser_set_preview_widget(chooser, preview->vbox);
  gtk_file_chooser_set_use_preview_label(chooser, FALSE);
  gtk_file_chooser_set_preview_widget_active(chooser, TRUE);

  return preview;
}

static void update_preview(GtkFileChooser *chooser, VisuGtkPreview *preview)
{
  const char *filenames[2];

  filenames[0] = gtk_file_chooser_get_preview_filename(chooser);
  filenames[1] = (char*)0;

  /* We test if the selected filename is a directory or not. */
  if (filenames[0] && !g_file_test(filenames[0], G_FILE_TEST_IS_DIR))
    visuGtkPreview_update(preview, filenames);
  if (filenames[0])
    g_free((char*)filenames[0]);
}

gboolean visuGtkGet_fileFromDefaultFileChooser(VisuData *data, GtkWindow *parent)
{
  GtkWidget *fileSelection;
  GList *filters, *tmpLst;
  VisuRendering *method;
  gchar* directory, *filename;
  ToolFileFormat *selectedFormat;
  GtkFileFilter *filterDefault;
  gboolean res;
  VisuGtkPreview preview;

  g_return_val_if_fail(data, FALSE);

  DBG_fprintf(stderr, "Visu Gtk: default filechooser for file opening.\n");

  method = visu_object_getRendering(VISU_INSTANCE);
  g_return_val_if_fail(method, FALSE);

  fileSelection = gtk_file_chooser_dialog_new(_("Load session"),
					      GTK_WINDOW(parent),
					      GTK_FILE_CHOOSER_ACTION_OPEN,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OPEN, GTK_RESPONSE_OK,
					      NULL);
  if (gtkFileChooserWidth > 0 || gtkFileChooserHeight > 0)
    gtk_window_set_default_size(GTK_WINDOW(fileSelection), gtkFileChooserWidth,
				gtkFileChooserHeight);
  directory = visuGtkGet_lastOpenDirectory();
  if (directory)
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(fileSelection), directory);

  gtk_widget_set_name(fileSelection, "filesel");
  gtk_window_set_position(GTK_WINDOW(fileSelection), GTK_WIN_POS_CENTER_ON_PARENT);
  gtk_window_set_modal(GTK_WINDOW(fileSelection), TRUE);

  /* Create and add the filters. */
  tmpLst = visu_rendering_getFileFormat(method, 0);
  filters = visuGtkCreate_fileChooserFilter(tmpLst, fileSelection);
  g_list_free(tmpLst);

  /* Create and add the preview. */
  visuGtkPreview_add(&preview, GTK_FILE_CHOOSER(fileSelection));
  g_signal_connect(GTK_FILE_CHOOSER(fileSelection), "update-preview",
		   G_CALLBACK(update_preview), &preview);

  if (gtk_dialog_run(GTK_DIALOG(fileSelection)) == GTK_RESPONSE_OK)
    {
      /* Get the selected filter. */
      selectedFormat = (ToolFileFormat*)0;
      filterDefault = gtk_file_chooser_get_filter(GTK_FILE_CHOOSER(fileSelection));
      tmpLst = filters;
      while(tmpLst)
	{
	  if (filterDefault == ((FileFilterCustom*)tmpLst->data)->gtkFilter)
	    selectedFormat = ((FileFilterCustom*)tmpLst->data)->visuFilter;
	  tmpLst = g_list_next(tmpLst);
	}
      filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(fileSelection));
      visu_data_addFile(data, filename, 0, selectedFormat);
      g_free(filename);
      res = TRUE;
    }
  else
    res = FALSE;

  directory =
    gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(fileSelection));
  visuGtkSet_lastOpenDirectory((char*)directory, VISU_DIR_FILE);
  g_free(directory);

  gtk_window_get_size(GTK_WINDOW(fileSelection), &gtkFileChooserWidth,
		      &gtkFileChooserHeight);

  gtk_widget_destroy(fileSelection);

  /* Free the filters list. */
  tmpLst = filters;
  while(tmpLst)
    {
      g_free(tmpLst->data);
      tmpLst = g_list_next(tmpLst);
    }
  g_list_free(filters);
  DBG_fprintf(stderr, "Visu Gtk: free load dialog OK.\n");

  return res;
}

void visuGtkCreate_main(GtkWindow **panel, GtkWindow **renderWindow,
			GtkWidget **renderArea)
{
  int width, height;

  /* Force the creation of the Scale class. */
  commandLineGet_XWindowGeometry(&width, &height);
  DBG_fprintf(stderr,"Visu Gtk: create a rendering window (%dx%d).\n",
	      width, height);
  *renderArea =
    visu_rendering_window_newEmbedded(width, height);
  *renderWindow = GTK_WINDOW(g_object_get_data(G_OBJECT(*renderArea),
					       RENDERING_WINDOW_ID));
  g_signal_connect(G_OBJECT(*renderWindow), "delete-event",
		   G_CALLBACK(gtk_main_quit), (gpointer)0);
  g_signal_connect(G_OBJECT(*renderWindow), "destroy-event",
		   G_CALLBACK(gtk_main_quit), (gpointer)0);
  gtk_widget_show(GTK_WIDGET(*renderWindow));
  *panel = (GtkWindow*)0;

  return;
}

static void initVisuGtk(void)
{
  VisuConfigFileEntry *entry;

  /* Retrieve the current working directory. */
  visuGtkLastDir = g_get_current_dir();

  /* Set private variables. */
  usePreview = TRUE;
  entry = visu_configFile_addEntry(VISU_CONFIGFILE_PARAMETER,
				  FLAG_PARAMETER_PREVIEW,
				  DESC_PARAMETER_PREVIEW,
				  1, readUsePreview);
  visu_configFile_entry_setVersion(entry, 3.4f);
  visu_configFile_addExportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParameters);
  visuGtkLoadMethods = g_hash_table_new_full(g_direct_hash, g_direct_equal,
					     NULL, g_free);
}

void visuGtkMain(VisuGtkInitWidgetFunc panelFunc)
{
  gchar *message;

  g_return_if_fail(panelFunc);

  if (!visuGtkLoadMethods)
    initVisuGtk();

  panelFunc(&visuGtkPanel, &visuGtkRender, &visuGtkRenderArea);
  g_return_if_fail(visuGtkRender && visuGtkRenderArea);

  /* Add the gtk tag as a known tag to allow to read such parameters. */
  message = visu_basic_parseConfigFiles();
  if (message)
    {
      visuGtkRaise_warningLong(_("Reading the configuration files"), message,
			       visuGtkRender);
      g_free(message);
    }

  return;
}

gboolean visuGtkRun_commandLine(gpointer data _U_)
{
  VisuData *obj;
  VisuBasicCLISet *set;
  GError *error;

  obj = visu_rendering_window_getVisuData(RENDERING_WINDOW(visuGtkRenderArea));
  if (!obj)
    return FALSE;

  set = g_malloc(sizeof(VisuBasicCLISet));
  error = (GError*)0;
  if (!visu_basic_applyCommandLine(obj, set, &error))
    {
      visuGtkRaise_warning(_("Parsing command line"), error->message, (GtkWindow*)0);
      g_error_free(error);
    }
  else
    visu_basic_createExtensions(obj, set, TRUE);

  visu_data_createAllNodes(obj);

  VISU_ADD_REDRAW;
  
  return FALSE;
}

GList* visuGtkCreate_fileChooserFilter(GList *list, GtkWidget *fileChooser)
{
  GtkFileFilter *filter, *filterAll;
  GList *tmpLst;
  const GList *tmpLst2;
  const char *name;
  FileFilterCustom *data;
  GList *returnedFilters;

  DBG_fprintf(stderr, "Visu Gtk: creating list of filters.\n");
  returnedFilters = (GList*)0;
  filterAll = gtk_file_filter_new ();
  gtk_file_filter_set_name(filterAll, _("All supported formats"));
  for (tmpLst = list; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      filter = gtk_file_filter_new ();
      name = tool_file_format_getLabel((ToolFileFormat*)tmpLst->data);
      if (name)
	gtk_file_filter_set_name(filter, name);
      else
	gtk_file_filter_set_name(filter, _("No description"));
      for (tmpLst2 = tool_file_format_getFilePatterns((ToolFileFormat*)tmpLst->data);
           tmpLst2; tmpLst2 = g_list_next(tmpLst2))
	{
	  gtk_file_filter_add_pattern (filter, (gchar*)tmpLst2->data);
	  gtk_file_filter_add_pattern (filterAll, (gchar*)tmpLst2->data);
	}
      data = g_malloc(sizeof(FileFilterCustom));
      data->gtkFilter = filter;
      data->visuFilter = (ToolFileFormat*)tmpLst->data;
      returnedFilters = g_list_append(returnedFilters, (gpointer)data);
    }
  data = g_malloc(sizeof(FileFilterCustom));
  data->gtkFilter = filterAll;
  data->visuFilter = (ToolFileFormat*)0;
  returnedFilters = g_list_append(returnedFilters, (gpointer)data);
  filter = gtk_file_filter_new ();
  gtk_file_filter_set_name(filter, _("All files"));
  gtk_file_filter_add_pattern (filter, "*");
  data = g_malloc(sizeof(FileFilterCustom));
  data->gtkFilter = filter;
  data->visuFilter = (ToolFileFormat*)0;
  returnedFilters = g_list_append(returnedFilters, (gpointer)data);
  
  DBG_fprintf(stderr, "Gtk Main : attach list to the given filechooser.\n");
  tmpLst = returnedFilters;
  while(tmpLst)
    {
      gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(fileChooser),
				  ((FileFilterCustom*)tmpLst->data)->gtkFilter);
      tmpLst = g_list_next(tmpLst);
    }
  gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(fileChooser), filterAll);

  return returnedFilters;
}

static void exportParameters(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_PREVIEW);
  g_string_append_printf(data, "%s[gtk]: %d\n\n", FLAG_PARAMETER_PREVIEW,
			 usePreview);
}
static gboolean readUsePreview(gchar **lines, int nbLines,
			       int position, VisuData *dataObj _U_, GError **error)
{
  int res;
  int val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  res = sscanf(lines[0],"%d", &val);
  if (res != 1)
    {
      *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_MISSING,
			   _("Parse error at line %d,"
			     " 1 boolean value must appear"
			     " after the %s markup.\n"),
			   position, FLAG_PARAMETER_PREVIEW);
      return FALSE;
    }
  else
    usePreview = (gboolean)val;

  return TRUE;
}

char* visuGtkGet_lastOpenDirectory(void)
{
  DBG_fprintf(stderr, "Visu Gtk: get the last open directory : '%s'.\n",
	      visuGtkLastDir);
  return visuGtkLastDir;
}
void visuGtkSet_lastOpenDirectory(const char* directory, VisuGtkDirType type)
{
  if (visuGtkLastDir)
    g_free(visuGtkLastDir);

  if (!g_path_is_absolute(directory))
    visuGtkLastDir = g_build_filename(g_get_current_dir(),
					 directory, NULL);
  else
    visuGtkLastDir = g_build_filename(directory, NULL);
  DBG_fprintf(stderr, "Visu Gtk: set the last open directory to '%s', emit signal.\n",
	      visuGtkLastDir);

  g_signal_emit(VISU_INSTANCE, VISU_SIGNALS[DIR_SIGNAL],
		0 /* details */, type, NULL);
  DBG_fprintf(stderr, "Visu Gtk: emission done (DirectoryChanged).\n");
}

GdkPixbuf* visuGtkCreate_pixbuf(const gchar *filename)
{
  gchar *pathname = NULL;
  GdkPixbuf *pixbuf;
  GError *error = NULL;

  g_return_val_if_fail(filename && filename[0], (GdkPixbuf*)0);

  pathname = g_build_filename(V_SIM_PIXMAPS_DIR, filename, NULL);

  pixbuf = gdk_pixbuf_new_from_file(pathname, &error);
  if (!pixbuf)
    {
      g_warning(_("failed to load pixbuf file '%s': %s\n"),
		pathname, error->message);
      g_error_free(error);
    }
  g_free(pathname);

  return pixbuf;
}

GtkWindow* visuGtkGet_render(void)
{
  return visuGtkRender;
}
GtkWindow* visuGtkGet_panel(void)
{
  return visuGtkPanel;
}
GtkWidget* visuGtkGet_renderArea(void)
{
  return visuGtkRenderArea;
}
