/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2006)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2006)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "gtk_renderingWindowWidget.h"

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <math.h> /* for sqrt function... */

#include <pixmaps/icone-observe.xpm>

#include "support.h"
#include "visu_gtk.h"
#include "visu_tools.h"
#include "visu_object.h"
#include "visu_data.h"
#include "visu_pickMesure.h"
#include "visu_extension.h"
#include "OSOpenGL/visu_openGL.h"
#include "renderingBackend/visu_actionInterface.h"
#include "gtk_openGLWidget.h"
#include "extraFunctions/extraNode.h"
#include "coreTools/toolFileFormat.h"
#include "extraGtkFunctions/gtk_dumpDialogWidget.h"
#include "openGLFunctions/objectList.h"
#include "openGLFunctions/interactive.h"
#include "opengl.h"

/**
 * SECTION:renderingwindow
 * @short_description: Defines a complex widget used to render files
 * and print information.
 *
 * <para>This is a complex widget, inheriting from #GtkWindow, with a
 * rendering area and a status bar area. A #VisuData is always
 * attached to this widget, see renderingWindowSet_visuData(). If not
 * the V_Sim logo is displayed.</para>
 *
 * <para>The rendering area can receive keyboard or mouse events, see
 * renderingWindowSet_pickEventListener() or
 * renderingWindowSet_observeEventListener(). Then, callbacks are
 * defined using renderingWindowSet_interactiveHandlers().</para>
 *
 * <para>The status bar area has different buttons to load or export a
 * file. It also display some usefull information like the number of
 * rendered nodes. It has also a real status bar location displaying
 * tips about current available actions. One can add news using
 * renderingWindowPush_message().</para>
 */

typedef enum
  {
    event_button_press,
    event_button_release,
    event_motion_notify,
    event_key_press,
    event_key_release,
    event_scroll
  } InteractiveEventsId;

struct InteractiveEvents_struct
{
  gulong callbackId;
  InteractiveEventsId id;
};
typedef struct InteractiveEvents_struct InteractiveEvents;


struct GtkInfoArea_struct
{
  GtkWidget *area;

  GtkWidget *hboxFileInfo;
  GtkWidget *labelSize;
  GtkWidget *labelNb;
  GtkWidget *labelFileInfo;
  gboolean fileInfoFreeze;

  GtkWidget *dumpButton;
  GtkWidget *loadButton;
  GtkWidget *raiseButton;
  GtkWidget *reloadButton;

  GtkWidget *statusInfo;

  GtkWidget *infoButton;
  gulong infoClicked;

  guint statusInfoId;
};
typedef struct GtkInfoArea_struct GtkInfoArea;

#define GTK_STATUSINFO_NOFILEINFO _("<span style=\"italic\">No description is available</span>")
#define GTK_STATUSINFO_NONB       _("<span style=\"italic\">Nothing is loaded</span>")

/* Internal gtkStatusInfo functions. */
static GtkInfoArea *gtkStatusInfo_createBar(RenderingWindow *window, gint width,
					    gint height, gboolean withToolBar);
/**
 * gtkStatusInfo_setOpenGLSize:
 * @info: the #GtkInfoArea that needs modifications ;
 * @width: size of the OpenGL area ;
 * @height: idem.
 *
 * Set the label of the size for the OpenGL area.
 */
void gtkStatusInfo_setOpenGLSize(GtkInfoArea *info, gint width, gint height);
/**
 * gtkStatusInfo_setFileInfo:
 * @info: the #GtkInfoArea that needs modifications ;
 * @message: an UTF8 string to print on the status bar.
 *
 * Use this method to display the commentary associated to a file.
 */
void gtkStatusInfo_setFileDescription(GtkInfoArea *info, gchar *message);
/**
 * gtkStatusInfo_setNbNodes:
 * @info: the #GtkInfoArea that needs modifications ;
 * @nb: number of loaded nodes (-1 if no file loaded).
 *
 * Use this method to display the commentary associated to the number of allocated nodes.
 */
void gtkStatusInfo_setNbNodes(GtkInfoArea *info, gint nb);
static gulong addInteractiveEventListeners(RenderingWindow *window,
					   InteractiveEventsId id);
static GtkWidget* buildCameraMenu(RenderingWindow *window);

/* Local callbacks */
static void onMarkClearClicked(GtkButton *button, gpointer data);
static void onNodeInfoClicked(GtkToggleButton *button, gpointer data);
static void onLoadFileClicked(RenderingWindow *window, gpointer data);
static void onReloadClicked(GtkButton *button, gpointer data);
static void onDumpButtonClicked(GtkButton *button, gpointer user_data);
static void onRaiseButtonClicked(RenderingWindow *window, gpointer user_data);
static void displayFileInfoOnDataLoaded(RenderingWindow *window);
static void onRenderingMethodChanged(RenderingWindow *window, VisuRendering *method,
				     gpointer data);
static void setFileButtonsSensitive(RenderingWindow *window);
static void onNodePopulationChanged(VisuData *data, int *nodes, gpointer user_data);
static void onBoxSizeChanged(VisuData *data, gpointer user_data);
static gboolean onDragMotion(GtkWidget *widget, GdkDragContext *context,
			     gint x, gint y, guint t, gpointer user_data);
static void onDropData(RenderingWindow *window, GdkDragContext *context,
		       gint x, gint y, GtkSelectionData *selection_data,
		       guint target_type, guint time, GtkWidget *glArea);
static gboolean onCameraMenu(GtkEventBox *ev _U_, GdkEventButton *event,
			     gpointer window);
static void onCameraMenuSelected(GtkMenuShell *menushell, gpointer user_data _U_);
static void onCameraMenuClicked(GtkMenuItem *menuitem, gpointer user_data);

struct _RenderingWindow
{
  GtkVBox generalVBox;

  /*********************************/
  /* Dealing with the OpenGL area. */
  /*********************************/
  /* The OpenGL area and it's notification zone. */
  GtkWidget *openGLArea;
  /* Stored size of the OpenGL area. */
  gint socketWidth, socketHeight;
  /* This pointer give the handle to rule all interactive actions. */
  VisuInteractive *inter;
  gulong onSelection_id;
  /* This is a list of currently connected
     signal for the interactive mode. */
  GList *interactiveEventsId;
  /* A pointer on the current used cursor. */
  GdkCursor *currentCursor;
  GdkCursor *refCursor;
  /* A pixbuf that can be display in the OpenGL area. */
  GdkPixbuf *backLogo;

  /*************************************/
  /* Dealing with the information bar. */
  /*************************************/
  /* TO BE INTEGRATED. */
  GtkInfoArea *info;
  /* TO BE INTEGRATED. */
  int nbStatusMessage;
  /* The cooresponding command panel, if any. */
  GtkWidget *panel;

  /* a pointer to the currently loaded VisuData. */
  VisuData *currentData;
  /* Id of signals currently connected, related to the current #VisuData object. */
  gulong populationIncrease_id, populationDecrease_id, boxChanged_id;
};

struct _RenderingWindowClass
{
  GtkWindowClass parent_class;

  void (*renderingWindow) (RenderingWindow *window);

  GdkCursor *cursorRotate;
  GdkCursor *cursorWatch;
  GdkCursor *cursorPointer;
  GdkCursor *cursorPirate;
  GdkCursor *cursorGrab;
  GdkCursor *currentCursor;
};

G_DEFINE_TYPE(RenderingWindow, renderingWindow, GTK_TYPE_VBOX)

/* Local callbacks */
static gboolean timeOutPopMessage(gpointer data);
/* Alert user when size of the OpenGL area is changed. */
static void onSizeChangeEvent(GtkWidget *widget, GtkAllocation *allocation, gpointer user_data);
static void onRedraw(RenderingWindow *window, gpointer data);
static void onForceRedraw(RenderingWindow *window, gpointer data);
static void onRealiseEvent(GtkWidget *wd, gpointer data);

/* Interactive mode listeners. */
static gboolean onHomePressed(GtkWidget *widget, GdkEventKey *event, gpointer data);
static gboolean onKeyPressed(GtkWidget *widget, GdkEventKey *event, gpointer data);
static gboolean onKeyRelease(GtkWidget *widget, GdkEventKey *event, gpointer data);
static gboolean onMouseMotion(GtkWidget *widget, GdkEventMotion *event, gpointer user_data);
static gboolean onScrollEvent(GtkWidget *widget, GdkEventScroll *event, gpointer user_data);
static gboolean onButtonAction(RenderingWindow *window, GdkEventButton *event, gpointer user_data);
static void stopIneractiveMode(RenderingWindow *window);
static void startIneractiveMode(RenderingWindow *window);

static void renderingWindow_class_init(RenderingWindowClass *klass)
{
  DBG_fprintf(stderr, "Gtk RenderingWindow : creating the class of the widget.\n");

  /* Initialisation des crseurs utiles. */
  klass->cursorPirate  = gdk_cursor_new(GDK_PIRATE);
  klass->cursorRotate  = gdk_cursor_new(GDK_EXCHANGE);
  klass->cursorWatch   = gdk_cursor_new(GDK_WATCH);
  klass->cursorPointer = gdk_cursor_new(GDK_DOTBOX);
  klass->cursorGrab    = gdk_cursor_new(GDK_FLEUR);
}

static void renderingWindow_init(RenderingWindow *renderingWindow)
{
  DBG_fprintf(stderr, "Gtk RenderingWindow : initializing new object (%p).\n",
	      (gpointer)renderingWindow);

  renderingWindow->currentData           = (VisuData*)0;
  renderingWindow->populationDecrease_id = 0;
  renderingWindow->populationIncrease_id = 0;
  renderingWindow->boxChanged_id         = 0;
  renderingWindow->panel                 = (GtkWidget*)0;
}

GtkWidget* renderingWindow_newEmbedded(int width, int height)
{
  GtkWidget *renderingWindow;
  GtkWidget *window;
  gchar *windowRef = "V_Sim";
  gchar *window_name = "v_sim_render";
  gchar *class_name = "V_Sim";
  GdkPixbuf *iconPixBuf;

  renderingWindow = renderingWindow_new(width, height, FALSE, TRUE);
  
  /* We create the rendering window. */
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), windowRef);
  gtk_window_set_wmclass(GTK_WINDOW(window), window_name, class_name);
  iconPixBuf = gdk_pixbuf_new_from_xpm_data((const char**)icone_observe_xpm);
  gtk_window_set_icon(GTK_WINDOW(window), iconPixBuf);

  gtk_container_add(GTK_CONTAINER(window), renderingWindow);
  g_object_set_data(G_OBJECT(renderingWindow), RENDERING_WINDOW_ID, window);

  return renderingWindow;
}

GtkWidget* renderingWindow_new(int width, int height, gboolean withFrame,
			       gboolean withToolBar)
{
  RenderingWindow *renderingWindow;
  GtkWidget *wd;
  GtkTargetEntry target_list[] = {
    { "text/plain", 0, 0 },
  };


  DBG_fprintf(stderr, "Gtk RenderingWindow: create a new RenderingWindow object.\n");

  renderingWindow = RENDERING_WINDOW(g_object_new(renderingWindow_get_type(), NULL));

  /* We create the statusinfo area. */
  renderingWindow->info = gtkStatusInfo_createBar(renderingWindow, width, height,
						  withToolBar);
  wd = renderingWindow->info->area;
  gtk_box_pack_end(GTK_BOX(renderingWindow), wd, FALSE, FALSE, 0);

  renderingWindow->openGLArea = openGLWidgetNew(TRUE);
  g_signal_connect_swapped(G_OBJECT(renderingWindow->openGLArea), "realize",
			   G_CALLBACK(onRealiseEvent), (gpointer)renderingWindow);
  gtk_drag_dest_set(renderingWindow->openGLArea,
		    GTK_DEST_DEFAULT_DROP,
		    target_list, 1, GDK_ACTION_COPY);
  g_signal_connect(renderingWindow->openGLArea, "drag-motion",
		   G_CALLBACK(onDragMotion), NULL);
  g_signal_connect_swapped(renderingWindow->openGLArea, "drag-data-received",
			   G_CALLBACK(onDropData), (gpointer)renderingWindow);

  gtk_widget_set_size_request(renderingWindow->openGLArea, width, height);
  /* Attach no redraw redraw method. */
  openGLWidgetSet_redraw(OPENGL_WIDGET(renderingWindow->openGLArea),
			 openGL_drawToEmpty, (VisuData*)0);
  if (withFrame)
    {
      wd = gtk_frame_new(NULL);
      gtk_frame_set_shadow_type(GTK_FRAME(wd), GTK_SHADOW_ETCHED_IN);
      gtk_box_pack_start(GTK_BOX(renderingWindow), wd, TRUE, TRUE, 0);
      gtk_container_add(GTK_CONTAINER(wd), renderingWindow->openGLArea);
    }
  else
    gtk_box_pack_start(GTK_BOX(renderingWindow),
		       renderingWindow->openGLArea, TRUE, TRUE, 0);
  
  /* We physically show the vbox. */
  gtk_widget_show_all(GTK_WIDGET(renderingWindow));
  if (renderingWindow->info->raiseButton)
    gtk_widget_hide(renderingWindow->info->raiseButton);

  /* Unset the size request. */
  renderingWindow->socketWidth = width;
  renderingWindow->socketHeight = height;
  /* Set a listener on the size to show the warning icon accordingly. */
  g_signal_connect(G_OBJECT(renderingWindow->openGLArea), "size-allocate",
		   G_CALLBACK(onSizeChangeEvent), (gpointer)renderingWindow);
  g_signal_connect(G_OBJECT(renderingWindow), "key-press-event",
		   G_CALLBACK(onHomePressed), (gpointer)renderingWindow);

  /* Set local variables. */
  renderingWindow->nbStatusMessage          = 0;
  renderingWindow->interactiveEventsId      = (GList*)0;
  renderingWindow->backLogo                 = (GdkPixbuf*)0;
  renderingWindow->inter                    = visuInteractiveNew(renderingWindow);
  renderingWindow->currentCursor            = 
    RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(renderingWindow))->cursorPirate;
  renderingWindow->refCursor                = 
    RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(renderingWindow))->cursorPirate;

  g_signal_connect_swapped(VISU_INSTANCE, "OpenGLAskForReDraw",
			   G_CALLBACK(onRedraw), (gpointer)renderingWindow);
  g_signal_connect_swapped(VISU_INSTANCE, "OpenGLForceReDraw",
			   G_CALLBACK(onForceRedraw), (gpointer)renderingWindow);

  DBG_fprintf(stderr, "Gtk RenderingWindow: Building OK.\n");

  return GTK_WIDGET(renderingWindow);
}

static GtkInfoArea *gtkStatusInfo_createBar(RenderingWindow *window, gint width,
					    gint height, gboolean withToolBar)
{
  GtkWidget *hbox;
  GtkWidget *wd, *image, *ev;
  GtkInfoArea *info;
#if GTK_MINOR_VERSION < 12
  GtkTooltips *tooltips;
  tooltips = gtk_tooltips_new ();
#endif

  GtkWidget *handle, *hboxHandle;

  info = g_malloc(sizeof(GtkInfoArea));

  info->area = gtk_vbox_new(FALSE, 0);

  info->fileInfoFreeze = FALSE;
  info->hboxFileInfo = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(info->area), info->hboxFileInfo, FALSE, FALSE, 1);

  /* Size info */
  wd = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(info->hboxFileInfo), wd, FALSE, FALSE, 5);
  info->labelSize = gtk_label_new("");
  gtk_label_set_use_markup(GTK_LABEL(info->labelSize), TRUE);
  gtkStatusInfo_setOpenGLSize(info, width, height);
  gtk_box_pack_start(GTK_BOX(wd), info->labelSize, FALSE, FALSE, 0);

  wd = gtk_vseparator_new();
  gtk_box_pack_start(GTK_BOX(info->hboxFileInfo), wd, FALSE, FALSE, 0);

  /* Nb nodes */
  info->labelNb = gtk_label_new("");
  gtk_label_set_use_markup(GTK_LABEL(info->labelNb), TRUE);
  gtkStatusInfo_setNbNodes(info, -1);
  gtk_box_pack_start(GTK_BOX(info->hboxFileInfo), info->labelNb, FALSE, FALSE, 5);

  wd = gtk_vseparator_new();
  gtk_box_pack_start(GTK_BOX(info->hboxFileInfo), wd, FALSE, FALSE, 0);

  /* File info */
  wd = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(info->hboxFileInfo), wd, TRUE, TRUE, 5);
#if GTK_MINOR_VERSION > 7
  image = gtk_image_new_from_stock(GTK_STOCK_INFO,
				   GTK_ICON_SIZE_MENU);
#else
  image = gtk_image_new_from_stock(GTK_STOCK_SAVE,
				   GTK_ICON_SIZE_MENU);
#endif
  gtk_box_pack_start(GTK_BOX(wd), image, FALSE, FALSE, 1);
  info->labelFileInfo = gtk_label_new("");
  gtk_label_set_use_markup(GTK_LABEL(info->labelFileInfo), TRUE);
  gtk_misc_set_alignment(GTK_MISC(info->labelFileInfo), 0., 0.5);
#if GTK_MINOR_VERSION > 5
  gtk_label_set_ellipsize(GTK_LABEL(info->labelFileInfo), PANGO_ELLIPSIZE_END);
#endif
  gtkStatusInfo_setFileDescription(info, GTK_STATUSINFO_NOFILEINFO);
  gtk_box_pack_start(GTK_BOX(wd), info->labelFileInfo, TRUE, TRUE, 0);
  ev = gtk_event_box_new();
  gtk_widget_set_tooltip_text(ev, _("Click here to get the list of"
				    " saved camera positions.\n"
				    "Use 's' and 'r' keys to save and"
				    " restore camera settings."));
  g_signal_connect(G_OBJECT(ev), "button-release-event",
		   G_CALLBACK(onCameraMenu), (gpointer)window);
  gtk_box_pack_end(GTK_BOX(wd), ev, FALSE, FALSE, 0);
  image = gtk_image_new_from_stock(GTK_STOCK_ZOOM_FIT,
				   GTK_ICON_SIZE_MENU);
  gtk_container_add(GTK_CONTAINER(ev), image);

  /* Status */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(info->area), hbox, FALSE, FALSE, 0);

  /* Handle box for action buttons. */
  if (withToolBar)
    {
      handle = gtk_handle_box_new();
      gtk_box_pack_start(GTK_BOX(hbox), handle, FALSE, FALSE, 0);
      hboxHandle = gtk_hbox_new(TRUE, 1);
      gtk_container_add(GTK_CONTAINER(handle), hboxHandle);
      /* Load button */
      info->loadButton = gtk_button_new();
      gtk_widget_set_sensitive(info->loadButton, FALSE);
      gtk_button_set_focus_on_click(GTK_BUTTON(info->loadButton), FALSE);
      gtk_widget_set_tooltip_text(info->loadButton,
			    _("Open"));
      g_signal_connect_swapped(G_OBJECT(info->loadButton), "clicked",
			       G_CALLBACK(onLoadFileClicked), (gpointer)window);
      image = gtk_image_new_from_stock(GTK_STOCK_OPEN,
				       GTK_ICON_SIZE_MENU);
      gtk_container_add(GTK_CONTAINER(info->loadButton), image);
      gtk_box_pack_start(GTK_BOX(hboxHandle), info->loadButton, FALSE, FALSE, 0);
      /* Refresh button */
      wd = gtk_button_new();
      gtk_button_set_focus_on_click(GTK_BUTTON(wd), FALSE);
      gtk_widget_set_tooltip_text(wd,
			    _("Reload the current file"));
      g_signal_connect(G_OBJECT(wd), "clicked",
		       G_CALLBACK(onReloadClicked), (gpointer)window);
      image = gtk_image_new_from_stock(GTK_STOCK_REFRESH,
				       GTK_ICON_SIZE_MENU);
      gtk_container_add(GTK_CONTAINER(wd), image);
      gtk_box_pack_start(GTK_BOX(hboxHandle), wd, FALSE, FALSE, 0);
      info->reloadButton = wd;
      /* Save button */
      info->dumpButton = gtk_button_new();
      gtk_button_set_focus_on_click(GTK_BUTTON(info->dumpButton), FALSE);
      gtk_widget_set_tooltip_text(info->dumpButton,
			    _("Export"));
      g_signal_connect(G_OBJECT(info->dumpButton), "clicked",
		       G_CALLBACK(onDumpButtonClicked), (gpointer)window);
      gtk_widget_set_sensitive(info->dumpButton, FALSE);
      image = gtk_image_new_from_stock(GTK_STOCK_SAVE_AS,
				       GTK_ICON_SIZE_MENU);
      gtk_container_add(GTK_CONTAINER(info->dumpButton), image);
      gtk_box_pack_start(GTK_BOX(hboxHandle), info->dumpButton, FALSE, FALSE, 0);
      /* Auto-raise command panel button */
      info->raiseButton = gtk_button_new();
      gtk_button_set_focus_on_click(GTK_BUTTON(info->raiseButton), FALSE);
      gtk_widget_set_tooltip_text(info->raiseButton,
			    _("Raise the command panel window.\n"
			      "  Use <home> as key binding."));
      g_signal_connect_swapped(G_OBJECT(info->raiseButton), "clicked",
			       G_CALLBACK(onRaiseButtonClicked), (gpointer)window);
      image = gtk_image_new_from_stock(GTK_STOCK_GO_UP,
				       GTK_ICON_SIZE_MENU);
      gtk_container_add(GTK_CONTAINER(info->raiseButton), image);
      gtk_box_pack_start(GTK_BOX(hboxHandle), info->raiseButton, FALSE, FALSE, 0);
    }
  else
    {
      info->loadButton = (GtkWidget*)0;
      info->dumpButton = (GtkWidget*)0;
      info->raiseButton = (GtkWidget*)0;
    }

  info->statusInfo = gtk_statusbar_new();
  gtk_box_pack_start(GTK_BOX(hbox), info->statusInfo, TRUE, TRUE, 0);
  gtk_statusbar_set_has_resize_grip(GTK_STATUSBAR(info->statusInfo), FALSE);

  info->statusInfoId = gtk_statusbar_get_context_id(GTK_STATUSBAR(info->statusInfo),
						    "OpenGL statusbar.");
  
  /* Action button */
  wd = gtk_toggle_button_new();
  gtk_widget_set_sensitive(wd, FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(wd), FALSE);
  gtk_button_set_focus_on_click(GTK_BUTTON(wd), FALSE);
  g_object_set(G_OBJECT(wd), "can-default", FALSE, "can-focus", FALSE,
	       "has-default", FALSE, "has-focus", FALSE, NULL);
  info->infoClicked = g_signal_connect(G_OBJECT(wd), "clicked",
				       G_CALLBACK(onNodeInfoClicked),
				       (gpointer)window);
  gtk_widget_set_tooltip_text(wd, _("Measure / remove information"
				    " for the selected node."));
  image = gtk_image_new_from_stock(GTK_STOCK_PROPERTIES,
				   GTK_ICON_SIZE_MENU);
  gtk_container_add(GTK_CONTAINER(wd), image);
  gtk_box_pack_end(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  info->infoButton = wd;
  g_object_set_data_full(G_OBJECT(wd), "selectedNodeId",
			 g_malloc(sizeof(gint)), g_free);

  /* Clean marks button */
  wd = gtk_button_new();
  gtk_button_set_focus_on_click(GTK_BUTTON(wd), FALSE);
  g_object_set(G_OBJECT(wd), "can-default", FALSE, "can-focus", FALSE,
	       "has-default", FALSE, "has-focus", FALSE, NULL);
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onMarkClearClicked), (gpointer)window);
  gtk_widget_set_tooltip_text(wd, _("Remove all measurement marks."));
  image = gtk_image_new_from_stock(GTK_STOCK_CLEAR,
				   GTK_ICON_SIZE_MENU);
  gtk_container_add(GTK_CONTAINER(wd), image);
  gtk_box_pack_end(GTK_BOX(hbox), wd, FALSE, FALSE, 0);

  g_signal_connect_swapped(VISU_INSTANCE, "renderingChanged",
			   G_CALLBACK(onRenderingMethodChanged), (gpointer)window);

  return info;
}

static void onSizeChangeEvent(GtkWidget *widget _U_,
			      GtkAllocation *allocation, gpointer user_data)
{
  RenderingWindow *window;

  window = RENDERING_WINDOW(user_data);
  g_return_if_fail(window);

  /* Return if no changes in size (this event is called even the size
     is not really changed but has been negociated. */
  if (window->socketWidth == allocation->width &&
      window->socketHeight == allocation->height)
    return;

  window->socketWidth = allocation->width;
  window->socketHeight = allocation->height;
  gtkStatusInfo_setOpenGLSize(window->info,
			      window->socketWidth,
			      window->socketHeight);

  /* If data are currently rendered on this window,
     we ask these data to update to the new size. */
  if (window->currentData)
    visuDataSet_sizeOfView(window->currentData,
			   allocation->width, allocation->height);
}

static void onRealiseEvent(GtkWidget *wd, gpointer data _U_)
{
  guint w, h;

  DBG_fprintf(stderr, "Gtk RenderingWindow : initializing OpenGL variable for"
	      "the new OpenGL area.\n");
  /* Set V_Sim OpenGL options. */
  openGLInit_context();
  DBG_fprintf(stderr, " | openGL context OK\n");

  /* If we have a VisuData object attached, we set its size. */
  if (RENDERING_WINDOW(wd)->currentData)
    {
      renderingWindowGet_openGLAreaSize(RENDERING_WINDOW(wd), &w, &h);
      visuDataSet_sizeOfView(RENDERING_WINDOW(wd)->currentData, w, h);
      openGLViewCompute_matrixAndView(visuDataGet_openGLView(RENDERING_WINDOW(wd)->currentData));
      rebuildAllExtensionsLists(RENDERING_WINDOW(wd)->currentData);
    }
  DBG_fprintf(stderr, "Gtk RenderingWindow: changing the cursor.\n");
  /* We set the cursor. */
  gdk_window_set_cursor(GDK_WINDOW(RENDERING_WINDOW(wd)->openGLArea->window),
			RENDERING_WINDOW(wd)->currentCursor);  
  DBG_fprintf(stderr, " | cursor OK.\n");

  DBG_fprintf(stderr, "Gtk RenderingWindow: changing the minimum size.\n");
  gtk_widget_set_size_request(RENDERING_WINDOW(wd)->openGLArea, 100, 100);
}
static gboolean onHomePressed(GtkWidget *widget _U_, GdkEventKey *event,
			       gpointer data)
{
  DBG_fprintf(stderr, "Gtk RenderingWindow: get key pressed.\n");
  if(event->keyval == GDK_Home)
    {
      onRaiseButtonClicked(RENDERING_WINDOW(data), (gpointer)0);
      return TRUE;
    }
  return FALSE;
}

static gboolean onDragMotion(GtkWidget *widget, GdkDragContext *context,
			     gint x _U_, gint y _U_, guint t, gpointer data _U_)
{
/*   GList *tmpLst; */
  GdkAtom atom;

/*   DBG_fprintf(stderr, "RenderingWindow: Hey ! You dnd move something !\n"); */
/*   for (tmpLst = context->targets; tmpLst; tmpLst = g_list_next(tmpLst)) */
/*     { */
/*       DBG_fprintf(stderr, " | dnd: '%s'\n", */
/* 		  gdk_atom_name(GDK_POINTER_TO_ATOM(tmpLst->data))); */
/*     } */
  atom = gtk_drag_dest_find_target(widget, context,
				   gtk_drag_dest_get_target_list(widget));
  if (atom != GDK_NONE)
    gdk_drag_status(context, GDK_ACTION_COPY, t);
  else
    gdk_drag_status(context, 0, t);
  return (atom != GDK_NONE);
}

static void onDropData(RenderingWindow *window, GdkDragContext *context,
		       gint x _U_, gint y _U_, GtkSelectionData *selection_data,
		       guint target_type _U_, guint time, GtkWidget *glArea _U_)
{
  gchar **filenames;
  int i, n, delta;
  VisuData *newData;
  guint w, h;

  if (context->action != GDK_ACTION_COPY)
    return;

  DBG_fprintf(stderr, " | data: '%s' -> '%s'\n", gdk_atom_name(selection_data->type),
	      selection_data->data);

  filenames = g_strsplit((gchar*)selection_data->data, "\n", -1);
  gtk_drag_finish(context, TRUE, TRUE, time);

  if (window->currentData)
    newData = visuDataNew_withOpenGLView(visuDataGet_openGLView(window->currentData));
  else
    {
      if (GTK_WIDGET_REALIZED(window))
	{
	  renderingWindowGet_openGLAreaSize(window, &w, &h);
	  newData = visuDataNew_withSize(w, h);
	}
      else
	newData = visuDataNew();
    }
  g_return_if_fail(newData);

  n = 0;
  for (i = 0; filenames[i]; i++)
    {
      g_strstrip(filenames[i]);
      if (filenames[i][0] != '\0')
	{
	  delta = (strncmp("file://", filenames[i], 7))?0:7;
	  visuDataAdd_file(newData, filenames[i] + delta, n, (FileFormat*)0);
	  n += 1;
	}
    }
  g_strfreev(filenames);

  visuGtkLoad_file(newData, 0);
  g_idle_add(visuObjectRedraw, (gpointer)0);

}

static void minimalPickInfo(VisuInteractive *inter _U_, PickMesure *pickMesure,
			    gpointer data)
{
  float posSelect[3], posRef[3], dist;
  GString *str;
  RenderingWindow *window;
  VisuNode *node, *ref, *ref2;
  int i;
  gint *id;
  gchar *strBuf;
  PickMesureType type;
  const gchar *comment;

  window = RENDERING_WINDOW(data);
  g_return_if_fail(window);

  while (window->nbStatusMessage > 1)
    {
      renderingWindowPop_message(window);
      window->nbStatusMessage -= 1;
    }
  gtk_widget_set_sensitive(window->info->infoButton, FALSE);

  if (pickMesureGet_newsAvailable(pickMesure, &type))
    {
      DBG_fprintf(stderr, "Gtk RenderingWindow: update the status bar after pick.\n");
      switch (type)
	{
	case PICK_SELECTED:
	case PICK_REFERENCE_1:
	  node = pickMesureGet_selectedNode(pickMesure);
	  ref  = pickMesureGet_firstReference(pickMesure);
	  
	  gtk_widget_set_sensitive(window->info->infoButton,
				   (gboolean)(!ref && node));
	  id = (gint*)g_object_get_data(G_OBJECT(window->info->infoButton),
					 "selectedNodeId");
	  if (!ref && node)
	    {
	      *id = node->number;
	      g_signal_handler_block(G_OBJECT(window->info->infoButton),
				     window->info->infoClicked);
	      gtk_toggle_button_set_active
		(GTK_TOGGLE_BUTTON(window->info->infoButton),
		 pickMesureGet_active(pickMesure, *id));
	      g_signal_handler_unblock(G_OBJECT(window->info->infoButton),
				       window->info->infoClicked);
	    }
	  else
	    *id = -1;

	  /* No ref, then only position informations are displayed */
	  if (!ref && node)
	    {
	      visuDataGet_nodePosition(window->currentData, node, posSelect);
	      str = g_string_new(_("Selected node number "));
	      g_string_append_printf(str, "%d - %s (%7.3g;%7.3g;%7.3g)",
				     node->number + 1,
				     window->currentData->fromIntToVisuElement
				     [node->posElement]->name,
				     posSelect[0], posSelect[1], posSelect[2]);
	      comment = extraNodeGet_label(window->currentData, node);
	      if (comment)
		g_string_append_printf(str, " %s", comment);
	      renderingWindowPush_message(window, str->str);
	      window->nbStatusMessage += 1;
	      g_string_free(str, TRUE);
	    }
	  /* Have a ref and a selected node, then distance informations is compued. */
	  if (ref && node)
	    {
	      visuDataGet_nodePosition(window->currentData, node, posSelect);
	      visuDataGet_nodePosition(window->currentData, ref, posRef);
	      str = g_string_new(_("Distance between nodes "));
	      dist = 0.;
	      for (i = 0; i < 3; i++)
		dist += (posRef[i] - posSelect[i]) * (posRef[i] - posSelect[i]);
	      dist = sqrt(dist);
	      g_string_append_printf(str, _("%d and %d : %7.3f"),
				     ref->number + 1, node->number + 1, dist);
	      renderingWindowPush_message(window, str->str);
	      window->nbStatusMessage += 1;
	      g_string_free(str, TRUE);
	    }
	  /* Have a ref only, give some text info. */
	  if (ref && !node)
	    {
	      renderingWindowPush_message(window, _("<shift> right-click on"
						    " background to unset reference"));
	      window->nbStatusMessage += 1;
	    }
	  return /* FALSE */;
	case PICK_REFERENCE_2:
	  ref2 = pickMesureGet_secondReference(pickMesure);

	  if (ref2)
	    {
	      renderingWindowPush_message(window,
					  _("<ctrl> right-click on"
					    " background to unset reference"));
	      window->nbStatusMessage += 1;
	    }
	  return /* FALSE */;
	case (PICK_REGION):
	  /* If we selected several nodes. */
	  strBuf = g_strdup_printf(_("%d selected nodes."),
				   pickMesureGet_regionNNodes(pickMesure));
	  renderingWindowPush_message(window, strBuf);
	  window->nbStatusMessage += 1;
	  g_free(strBuf);
	  return /* FALSE */;
	case (PICK_HIGHLIGHT):
	case (PICK_UNHIGHLIGHT):
	case (PICK_INFORMATION):
	  return /* TRUE */;
	default:
	  g_warning("No other pick possibilities for the"
		    " rendering window (type=%d).", type);
	}
    }
  return /* FALSE */;
}

void emptyInteractiveEventListeners(RenderingWindow *window)
{
  GList* ptList;

  g_return_if_fail(IS_RENDERING_WINDOW(window));

  DBG_fprintf(stderr, "Gtk RenderingWindow: removing current interactive listeners.\n");
  ptList = window->interactiveEventsId;
  while (ptList)
    {
      DBG_fprintf(stderr, "  | disconnecting %ld signal.\n", *(gulong*)ptList->data);
      g_signal_handler_disconnect(G_OBJECT(window->openGLArea), *(gulong*)ptList->data);
      g_free(ptList->data);
      ptList = g_list_next(ptList);
    }
  if (window->interactiveEventsId)
    g_list_free(window->interactiveEventsId);
  window->interactiveEventsId = (GList*)0;
}
static gulong addInteractiveEventListeners(RenderingWindow *window,
					   InteractiveEventsId id)
{
  GList* ptList;
  InteractiveEvents *event;
  gboolean found;

  g_return_val_if_fail(IS_RENDERING_WINDOW(window), (gulong)0);

  found  = FALSE;
  for (ptList = window->interactiveEventsId; ptList && !found; ptList = g_list_next(ptList))
    {
      event = (InteractiveEvents*)ptList->data;
      if (event->id == id)
	found = TRUE;
    }
  if (found)
    return (gulong)0;

  event = g_malloc(sizeof(InteractiveEvents));
  event->id = id;
  switch (id)
    {
    case event_button_press:
      event->callbackId = g_signal_connect_swapped(G_OBJECT(window->openGLArea),
						   "button-press-event",
						   G_CALLBACK(onButtonAction), (gpointer)window);
      break;
    case event_button_release:
      event->callbackId = g_signal_connect_swapped(G_OBJECT(window->openGLArea),
						   "button-release-event",
						   G_CALLBACK(onButtonAction), (gpointer)window);
      break;
    case event_motion_notify:
      event->callbackId = g_signal_connect(G_OBJECT(window->openGLArea),
					   "motion-notify-event",
					   G_CALLBACK(onMouseMotion), (gpointer)window);
      break;
    case event_key_press:
      event->callbackId = g_signal_connect(G_OBJECT(window->openGLArea),
					   "key-press-event",
					   G_CALLBACK(onKeyPressed), (gpointer)window);
      break;
    case event_key_release:
      event->callbackId = g_signal_connect(G_OBJECT(window->openGLArea),
					   "key-release-event",
					   G_CALLBACK(onKeyRelease), (gpointer)window);
      break;
    case event_scroll:
      event->callbackId = g_signal_connect(G_OBJECT(window->openGLArea),
					   "scroll-event",
					   G_CALLBACK(onScrollEvent), (gpointer)window);
      break;
    default:
      g_warning("Unknown event to add.");
      g_free(event);
      return (gulong)0;
    };
  window->interactiveEventsId = g_list_prepend(window->interactiveEventsId,
					       (gpointer)event);
  return event->callbackId;
}
gulong removeInteractiveEventListeners(RenderingWindow *window, InteractiveEventsId id)
{
  GList* ptList;
  InteractiveEvents *event;
  gulong callid;

  g_return_val_if_fail(IS_RENDERING_WINDOW(window), (gulong)0);

  for (ptList = window->interactiveEventsId; ptList; ptList = g_list_next(ptList))
    {
      event = (InteractiveEvents*)ptList->data;
      if (event->id == id)
	{
	  g_signal_handler_disconnect(G_OBJECT(window->openGLArea),
				      event->callbackId);
	  window->interactiveEventsId = g_list_remove(window->interactiveEventsId,
						      (gpointer)event);
	  callid = event->callbackId;
	  g_free(event);
	  return callid;
	}
    }
  return (gulong)0;
}
void renderingWindowRemove_interactiveEventListener(RenderingWindow *window)
{
  DBG_fprintf(stderr, "Gtk RenderingWindow: remove all listeners.\n");
  emptyInteractiveEventListeners(window);
  if (GTK_WIDGET_REALIZED(GTK_WIDGET(window)))
    gdk_window_set_cursor(GDK_WINDOW(window->openGLArea->window),
			  RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorPirate);
  window->currentCursor = RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorPirate;
  window->refCursor = RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorPirate;
}
void renderingWindowSet_observeEventListener(RenderingWindow *window)
{
  gulong id;

  g_return_if_fail(IS_RENDERING_WINDOW(window));

  if (GTK_WIDGET_REALIZED(GTK_WIDGET(window)))
    gdk_window_set_cursor(GDK_WINDOW(window->openGLArea->window),
			  RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorRotate);
  window->currentCursor = RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorRotate;
  window->refCursor = RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorRotate;

  DBG_fprintf(stderr, "Gtk RenderingWindow: signals for observe.\n");
  id = addInteractiveEventListeners(window, event_button_release);
  if (id)
    DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
  id = addInteractiveEventListeners(window, event_button_press);
  if (id)
    DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
  id = addInteractiveEventListeners(window, event_motion_notify);
  if (id)
    DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
  id = addInteractiveEventListeners(window, event_key_press);
  if (id)
    DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
  id = addInteractiveEventListeners(window, event_key_release);
  if (id)
    DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
  id = addInteractiveEventListeners(window, event_scroll);
  if (id)
    DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
}
void renderingWindowSet_pickEventListener(RenderingWindow *window)
{
  gulong id;

  g_return_if_fail(IS_RENDERING_WINDOW(window));

  if (GTK_WIDGET_REALIZED(GTK_WIDGET(window)))
    gdk_window_set_cursor(GDK_WINDOW(window->openGLArea->window),
			  RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorPointer);
  window->currentCursor = RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorPointer;
  window->refCursor = RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorPointer;

  DBG_fprintf(stderr, "Gtk RenderingWindow: setting signals for pick.\n");
  id = addInteractiveEventListeners(window, event_button_press);
  if (id)
    DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
  id = addInteractiveEventListeners(window, event_button_release);
  if (id)
    DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
  id = addInteractiveEventListeners(window, event_motion_notify);
  if (id)
    DBG_fprintf(stderr, "  | connecting %ld signal.\n", id);
  id = removeInteractiveEventListeners(window, event_key_press);
  if (id)
    DBG_fprintf(stderr, "  | disconnecting %ld signal.\n", id);
  id = removeInteractiveEventListeners(window, event_scroll);
  if (id)
    DBG_fprintf(stderr, "  | disconnecting %ld signal.\n", id);
}

static gboolean onButtonAction(RenderingWindow *window, GdkEventButton *event,
			       gpointer user_data _U_)
{
  SimplifiedEvents ev;

  ev.button = 0;
  ev.motion = 0;
  ev.letter = '\0';
  ev.specialKey = Key_None;

  ev.x = event->x;
  ev.y = event->y;
  ev.button = event->button;
  ev.shiftMod = event->state & GDK_SHIFT_MASK;
  ev.controlMod = event->state & GDK_CONTROL_MASK;
  ev.buttonType = 0;
  if (event->type == GDK_BUTTON_PRESS)
    ev.buttonType = BUTTON_TYPE_PRESS;
  else if (event->type == GDK_BUTTON_RELEASE)
    ev.buttonType = BUTTON_TYPE_RELEASE;

  gdk_window_set_cursor(GDK_WINDOW(window->openGLArea->window),
			RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorWatch);
  visuInteractiveHandle_event(window->inter, &ev);
/*   if (window->callbacksInInteractiveMode.action(&ev, window->currentData) && */
/*       window->callbacksInInteractiveMode.stop) */
/*     window->callbacksInInteractiveMode.stop((gpointer)0); */

  gdk_window_set_cursor(GDK_WINDOW(window->openGLArea->window),
			window->currentCursor);

  return TRUE;
}
static gboolean onScrollEvent(GtkWidget *widget _U_, GdkEventScroll *event,
			      gpointer user_data)
{
  SimplifiedEvents ev;
  RenderingWindow *window;

  window = RENDERING_WINDOW(user_data);
  g_return_val_if_fail(window, TRUE);

  ev.button = 0;
  ev.motion = 0;
  ev.letter = '\0';
  ev.specialKey = Key_None;

  ev.x = event->x;
  ev.y = event->y;
  if (event->direction == GDK_SCROLL_UP)
    ev.button = 4;
  else if (event->direction == GDK_SCROLL_DOWN)
    ev.button = 5;
  ev.shiftMod = event->state & GDK_SHIFT_MASK;
  ev.controlMod = event->state & GDK_CONTROL_MASK;

  if (ev.button)
    {
      gdk_window_set_cursor(GDK_WINDOW(window->openGLArea->window),
			RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorWatch);
/*       window->callbacksInInteractiveMode.action(&ev, window->currentData); */
      visuInteractiveHandle_event(window->inter, &ev);
      gdk_window_set_cursor(GDK_WINDOW(window->openGLArea->window),
			window->currentCursor);
    }

  return TRUE;
}
static gboolean onMouseMotion(GtkWidget *widget _U_, GdkEventMotion *event,
			      gpointer user_data)
{
  SimplifiedEvents ev;
  RenderingWindow *window;

  window = RENDERING_WINDOW(user_data);
  g_return_val_if_fail(window, TRUE);

  ev.button = 0;
  ev.motion = 1;
  ev.letter = '\0';
  ev.specialKey = Key_None;

/*   DBG_fprintf(stderr, "Gtk RenderingWindow: motion at %gx%g.\n", event->x, event->y); */
  ev.x = event->x;
  ev.y = event->y;
  if (event->is_hint)
    {
#if GDK_MINOR_VERSION > 11
      gdk_event_request_motions(event);
#else
      gdk_window_get_pointer(event->window, NULL, NULL, NULL);
#endif
    }
  if (event->state & GDK_BUTTON1_MASK)
    ev.button = 1;
  else if (event->state & GDK_BUTTON2_MASK)
    ev.button = 2;
  else if (event->state & GDK_BUTTON3_MASK)
    ev.button = 3;
  ev.shiftMod = event->state & GDK_SHIFT_MASK;
  ev.controlMod = event->state & GDK_CONTROL_MASK;

  if (ev.button)
    {
      gdk_window_set_cursor(GDK_WINDOW(window->openGLArea->window),
			RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorWatch);
/*       window->callbacksInInteractiveMode.action(&ev, window->currentData); */
      visuInteractiveHandle_event(window->inter, &ev);
      gdk_window_set_cursor(GDK_WINDOW(window->openGLArea->window),
			window->currentCursor);
    }

  return TRUE;
}
static gboolean onKeyPressed(GtkWidget *widget _U_, GdkEventKey *event,
			     gpointer data)
{
  SimplifiedEvents ev;
  RenderingWindow *window;
  GList *cameras, *head;

  window = RENDERING_WINDOW(data);
  g_return_val_if_fail(window, TRUE);

  ev.button = 0;
  ev.motion = 0;
  ev.letter = '\0';
  ev.specialKey = Key_None;

  if(event->keyval == GDK_r || event->keyval == GDK_R)
    {
      ev.letter = 'r';
      /* If any camera, print a message. */
      visuInteractiveGet_savedCameras(window->inter, &cameras, &head);
      if (cameras)
	renderingWindowPush_message(window, _("Restore saved camera position."));
      else
	renderingWindowPush_message(window, _("No saved camera. Use 's' to save one."));
#if GLIB_MINOR_VERSION > 13
	  g_timeout_add_seconds(3, timeOutPopMessage, (gpointer)window);
#else
	  g_timeout_add(3000, timeOutPopMessage, (gpointer)window);
#endif
    }
  else if(event->keyval == GDK_s || event->keyval == GDK_S)
    {
      ev.letter = 's';
      renderingWindowPush_message(window, _("Save current camera position."));
#if GLIB_MINOR_VERSION > 13
      g_timeout_add_seconds(3, timeOutPopMessage, (gpointer)window);
#else
      g_timeout_add(3000, timeOutPopMessage, (gpointer)window);
#endif
    }
  else if(event->keyval == GDK_space)
    ev.letter = ' ';
  else if(event->keyval == GDK_Page_Up)
    ev.specialKey = Key_Page_Up;
  else if(event->keyval == GDK_Page_Down)
    ev.specialKey = Key_Page_Down;
  else if(event->keyval == GDK_Down)
    ev.specialKey = Key_Arrow_Down;
  else if(event->keyval == GDK_Up)
    ev.specialKey = Key_Arrow_Up;
  else if(event->keyval == GDK_Left)
    ev.specialKey = Key_Arrow_Left;
  else if(event->keyval == GDK_Right)
    ev.specialKey = Key_Arrow_Right;
  ev.shiftMod = event->state & GDK_SHIFT_MASK;
  ev.controlMod = event->state & GDK_CONTROL_MASK;

  if (ev.letter != '\0' || ev.specialKey != Key_None)
    {
      gdk_window_set_cursor(GDK_WINDOW(window->openGLArea->window),
			    RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorWatch);
/*       window->callbacksInInteractiveMode.action(&ev, window->currentData); */
      visuInteractiveHandle_event(window->inter, &ev);
      gdk_window_set_cursor(GDK_WINDOW(window->openGLArea->window),
			    window->currentCursor);
    }
  else if (event->keyval == GDK_Shift_L || event->keyval == GDK_Shift_R)
    {
      gdk_window_set_cursor(GDK_WINDOW(window->openGLArea->window),
			    RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorGrab);
      window->currentCursor = RENDERING_WINDOW_CLASS(G_OBJECT_GET_CLASS(window))->cursorGrab;
    }

  return FALSE;
}
static gboolean onKeyRelease(GtkWidget *widget _U_, GdkEventKey *event,
			     gpointer data)
{
  RenderingWindow *window;

  window = RENDERING_WINDOW(data);
  g_return_val_if_fail(window, TRUE);

  if (event->keyval == GDK_Shift_L || event->keyval == GDK_Shift_R)
    {
      gdk_window_set_cursor(GDK_WINDOW(window->openGLArea->window),
			    window->refCursor);
      window->currentCursor = window->refCursor;
    }

  return TRUE;
}

static void stopIneractiveMode(RenderingWindow *window)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));

  DBG_fprintf(stderr, "Gtk RenderingWindow: unset interactive listener %p.\n",
	      (gpointer)window->inter);
  /* Reset the statusbar informations. */
  while (window->nbStatusMessage > 0)
    {
      window->nbStatusMessage -= 1;
      renderingWindowPop_message(window);
    }
  visuInteractiveSet_type(window->inter, interactive_none);
  if (window->onSelection_id > 0)
    g_signal_handler_disconnect(G_OBJECT(window->inter), window->onSelection_id);
  window->onSelection_id = 0;
}
static void startIneractiveMode(RenderingWindow *window)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window) && window->currentData);

  /* Reset the statusbar informations. */
  while (window->nbStatusMessage > 0)
    {
      window->nbStatusMessage -= 1;
      renderingWindowPop_message(window);
    }
  /* Put a commentary in the statusbar. */
  renderingWindowPush_message(window, _("Observe mode &"
					" <shift / ctrl> right-click to pick."));
  window->nbStatusMessage += 1;

  pickMesureSet_formatedOutput(visuInteractiveGet_pickMesure(window->inter), FALSE);
  visuInteractiveSet_type(window->inter, interactive_pickAndObserve);

  window->onSelection_id = g_signal_connect(G_OBJECT(window->inter), "selection",
					    G_CALLBACK(minimalPickInfo),
					    (gpointer)window);
}
void renderingWindowBlock_defaultIneractiveMode(RenderingWindow *window)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));

  g_signal_handler_block(G_OBJECT(window->inter), window->onSelection_id);
  visuInteractiveSet_type(window->inter, interactive_none);
}
void renderingWindowUnblock_defaultIneractiveMode(RenderingWindow *window)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));

  g_signal_handler_unblock(G_OBJECT(window->inter), window->onSelection_id);
  visuInteractiveSet_type(window->inter, interactive_pickAndObserve);
}

static gboolean timeOutPopMessage(gpointer data)
{
  gtk_statusbar_pop(GTK_STATUSBAR(RENDERING_WINDOW(data)->info->statusInfo),
		    RENDERING_WINDOW(data)->info->statusInfoId);

  return FALSE;
}

void renderingWindowPush_message(RenderingWindow *window, gchar *message)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));

  gtk_statusbar_push(GTK_STATUSBAR(window->info->statusInfo),
		     window->info->statusInfoId, message);
}
void renderingWindowPop_message(RenderingWindow *window)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));

  gtk_statusbar_pop(GTK_STATUSBAR(window->info->statusInfo),
		    window->info->statusInfoId);
}
void renderingWindowGet_openGLAreaSize(RenderingWindow *window,
				       unsigned int *width, unsigned int *height)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window) && width && height);

  *width = 0;
  *height = 0;
  gdk_drawable_get_size(GDK_DRAWABLE(window->openGLArea->window),
			(gint*)width, (gint*)height);
}
GdkPixbuf* renderingWindowGet_backgroundImage(RenderingWindow *window,
					     guchar **rowData, gboolean *hasAlphaChannel,
					     int *width, int *height)
{
  g_return_val_if_fail(IS_RENDERING_WINDOW(window), (GdkPixbuf*)0);

  g_return_val_if_fail(rowData && hasAlphaChannel && width && height, (GdkPixbuf*)0);

  DBG_fprintf(stderr, "Gtk RenderingWindow: load the pixbuf 'logo_grey.png'.\n");
  if (!window->backLogo)
    window->backLogo = visuGtkCreate_pixbuf("logo_grey.png");
  g_return_val_if_fail(window->backLogo, (GdkPixbuf*)0);

  *rowData = gdk_pixbuf_get_pixels(window->backLogo);
  *hasAlphaChannel = gdk_pixbuf_get_has_alpha(window->backLogo);
  *width = gdk_pixbuf_get_width(window->backLogo);
  *height = gdk_pixbuf_get_height(window->backLogo);
  return window->backLogo;
}
void renderingWindowFree_backgroundImage(RenderingWindow *window)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));

  if (window->backLogo)
    g_object_unref(G_OBJECT(window->backLogo));
  window->backLogo = (GdkPixbuf*)0;
}
void renderingWindowSet_visuData(RenderingWindow *window, VisuData* data)
{
  gboolean noEmit;
  gchar *nom, *file;
  guint w, h;
  VisuData *oldData;
  gpointer *wd;

  g_return_if_fail(IS_RENDERING_WINDOW(window));
  DBG_fprintf(stderr, "Gtk renderingWindow: attach %p (%p) VisuData to %p window.\n",
	      (gpointer)data, (gpointer)window->currentData, (gpointer)window);
  noEmit = (!window->currentData && !data);
  oldData = window->currentData;
  if (oldData != data)
    {
      if (window->currentData)
	{
	  visuDataSet_renderingWindow(window->currentData, (GenericRenderingWindow)0);
	  g_signal_handler_disconnect(G_OBJECT(window->currentData),
				      window->populationIncrease_id);
	  g_signal_handler_disconnect(G_OBJECT(window->currentData),
				      window->populationDecrease_id);
	  g_signal_handler_disconnect(G_OBJECT(window->currentData),
				      window->boxChanged_id);
	}
    }

  /* Change the rendering window with the new data (may be the same
     but with a different content). */
  window->currentData = data;
  displayFileInfoOnDataLoaded(window);
  setFileButtonsSensitive(window);

  /* Find a widget to change the title of. */
  wd = g_object_get_data(G_OBJECT(window), RENDERING_WINDOW_ID);
  if (!wd)
    wd = (gpointer)visuGtkGet_render();
  if (data)
    {
      /* Ref the object. */
      g_object_ref(G_OBJECT(data));
      DBG_fprintf(stderr, "Gtk RenderingWindow: ref new object %p.\n",
		  (gpointer)data);

      /* Set the window as attachingRenderingWindow for the given VisuData. */
      visuDataSet_renderingWindow(data, (GenericRenderingWindow)window);
      /* Adapt the Camera to the window. */
      if (GTK_WIDGET_REALIZED(window))
	{
	  renderingWindowGet_openGLAreaSize(window, &w, &h);
	  visuDataSet_sizeOfView(data, w, h);
	  openGLViewCompute_matrixAndView(visuDataGet_openGLView(data));
	}
      /* Change the name of the window, according to the file loaded. */
      if (wd)
	{
	  nom = visuDataGet_file(data, 0, (FileFormat**)0);
	  if (nom)
	    file = g_path_get_basename(nom);
	  else
	    {
	      g_warning("Can't find the filename to label the rendering window.\n");
	      file = g_strdup(_("No filename"));
	    }
	  gtk_window_set_title(GTK_WINDOW(wd), file);
	  g_free(file);
	}

      /* Attach the default redraw method. */
      openGLWidgetSet_redraw(OPENGL_WIDGET(window->openGLArea),
			     openGL_reDraw, data);

      /* Attach signals to the new #VisuData object. */
      window->populationIncrease_id =
	g_signal_connect(G_OBJECT(data), "NodePopulationIncrease",
			 G_CALLBACK(onNodePopulationChanged), window);
      window->populationDecrease_id =
	g_signal_connect(G_OBJECT(data), "NodePopulationDecrease",
			 G_CALLBACK(onNodePopulationChanged), window);
      window->boxChanged_id =
	g_signal_connect(G_OBJECT(data), "BoxSizeChanged",
			 G_CALLBACK(onBoxSizeChanged), window);
    }
  else
    {
      /* Erase the name of the window. */
      if (wd)
	gtk_window_set_title(GTK_WINDOW(wd), _("No file loaded"));

      if (oldData)
	{
	  /* Attach the default redraw method. */
	  openGLWidgetSet_redraw(OPENGL_WIDGET(window->openGLArea),
				 openGL_drawToEmpty, (VisuData*)0);
	  /* Ask for redraw. */
	  renderingWindowRedraw(window, TRUE);
	}
    }

  if (!noEmit)
    {
      DBG_fprintf(stderr, "Gtk renderingWindow: emitting the 'dataReadyForRendering' signal.\n");
      g_signal_emit(VISU_INSTANCE, VISU_SIGNALS[DATAREADYFORRENDERING_SIGNAL],
		    0 /* details */, (gpointer)data, NULL);
      DBG_fprintf(stderr, "Gtk renderingWindow: emition done.\n");
    }

  visuInteractiveSet_visuData(window->inter, data);
  if (data)
    startIneractiveMode(window);
  else
    stopIneractiveMode(window);

  /* Update properties of new data from old one.
     TODO : This should be trgiggered by a signal. */
  pickMesureUpdate(window->currentData, oldData);

  if (oldData)
    {
      DBG_fprintf(stderr, "Gtk RenderingWindow: unref old object %p.\n",
		  (gpointer)oldData);
      g_object_unref(oldData);
    }
}
VisuData* renderingWindowGet_visuData(RenderingWindow *window)
{
  g_return_val_if_fail(IS_RENDERING_WINDOW(window), (VisuData*)0);
  return window->currentData;
}

/***************************/
/* GtkStatusInfo functions */
/***************************/
void gtkStatusInfo_setOpenGLSize(GtkInfoArea *info, gint width, gint height)
{
  GString *str;

  g_return_if_fail(info);

  if (info->fileInfoFreeze)
    return;

  str = g_string_new("<span size=\"smaller\">");
  g_string_append_printf(str, _("<b>Size:</b> %dx%d"), width, height);
  g_string_append_printf(str, "</span>");
  gtk_label_set_markup(GTK_LABEL(info->labelSize), str->str);
  g_string_free(str, TRUE);
}
void gtkStatusInfo_setFileDescription(GtkInfoArea *info, gchar* message)
{
  gchar *str;

  g_return_if_fail(info);

  str = g_strdup_printf("<span size=\"smaller\">%s</span>", message);
  gtk_label_set_markup(GTK_LABEL(info->labelFileInfo), str);
  g_free(str);
}
void gtkStatusInfo_setNbNodes(GtkInfoArea *info, gint nb)
{
  GString *str;

  g_return_if_fail(info);

  str = g_string_new("<span size=\"smaller\">");
  if (nb > 0)
    g_string_append_printf(str, _("<b>Nb nodes:</b> %d"), nb);
  else
    g_string_append(str, GTK_STATUSINFO_NONB);
  g_string_append_printf(str, "</span>");
  gtk_label_set_markup(GTK_LABEL(info->labelNb), str->str);
  g_string_free(str, TRUE);
}
static void onNodeInfoClicked(GtkToggleButton *button, gpointer data)
{
  RenderingWindow *window;
  gint *id;
  PickMesure *pickMesure;

  window = RENDERING_WINDOW(data);
  g_return_if_fail(window);

  id = g_object_get_data(G_OBJECT(button), "selectedNodeId");
  g_return_if_fail(id);

  g_return_if_fail(window->currentData);
  pickMesure = (PickMesure*)g_object_get_data(G_OBJECT(window->currentData),
					      "pickMesure_data");

  if (pickMesureSet_nodeMeasurements(pickMesure, (guint)(*id), 1.2f,
				     gtk_toggle_button_get_active(button)))
    g_idle_add(visuObjectRedraw, GINT_TO_POINTER(TRUE));
}
static void onMarkClearClicked(GtkButton *button _U_, gpointer data)
{
  RenderingWindow *window;
  PickMesure *pickMesure;

  window = RENDERING_WINDOW(data);
  g_return_if_fail(window);

  g_return_if_fail(window->currentData);
  pickMesure = (PickMesure*)g_object_get_data(G_OBJECT(window->currentData),
					      "pickMesure_data");

  if (pickMesureRemove_allMarks(pickMesure))
    g_idle_add(visuObjectRedraw, GINT_TO_POINTER(TRUE));
}
static gboolean onCameraMenu(GtkEventBox *ev _U_, GdkEventButton *event,
			     gpointer window)
{
  GtkWidget *wd;

  DBG_fprintf(stderr, "Gtk RenderingWindow: click on the camera menu.\n");
  wd = buildCameraMenu(window);
  if (!wd)
    return TRUE;;

  g_signal_connect(G_OBJECT(wd), "selection-done",
		   G_CALLBACK(onCameraMenuSelected), (gpointer)window);

  gtk_widget_show_all(wd);
  gtk_menu_popup(GTK_MENU(wd), NULL, NULL, NULL, NULL, 
		 1, event->time);

  return TRUE;
}
static GtkWidget* buildCameraMenu(RenderingWindow *window)
{
  GtkWidget *menu, *item;
  gchar *lbl;
  GList *cameras, *head, *tmpLst;
  OpenGLCamera *current;

  if (!window->currentData)
    return (GtkWidget*)0;

  /* All camera. */
  DBG_fprintf(stderr, "Gtk RenderingWindow: get the cameras.\n");
  visuInteractiveGet_savedCameras(window->inter, &cameras, &head);
/*   if (!cameras) */
/*     return (GtkWidget*)0; */

  DBG_fprintf(stderr, "Gtk RenderingWindow: build the menu.\n");
  menu = gtk_menu_new();
  
  /* Put the current camera. */
  current = visuDataGet_openGLView(window->currentData)->camera;
  lbl = g_strdup_printf(_("current:\n"
			  "(\316\270 %6.1f ; \317\206 %6.1f ; \317\211 %6.1f) "
			  "dx %4.1f dy %4.1f"),
			current->theta, current->phi, current->omega,
			current->xs, current->ys);
  item = gtk_menu_item_new_with_label(lbl);
  g_free(lbl);
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
  /* Separator. */
  item = gtk_separator_menu_item_new();
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
  if (!cameras)
    {
      item = gtk_menu_item_new_with_label(_("No saved camera. Use 's' to save one."));
      gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    }
  
  for (tmpLst = cameras; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      current = (OpenGLCamera*)tmpLst->data;
      lbl = g_strdup_printf(_("(\316\270 %6.1f ; \317\206 %6.1f ; \317\211 %6.1f) "
			      "dx %4.1f dy %4.1f"),
			    current->theta, current->phi, current->omega,
			    current->xs, current->ys);
/*       if (tmpLst-> == head) */
/* 	{ */
/* 	  item = gtk_check_menu_item_new_with_label(lbl); */
/* 	  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE); */
/* 	} */
/*       else */
      item = gtk_menu_item_new_with_label(lbl);
      g_free(lbl);
      g_signal_connect(G_OBJECT(item), "activate",
		       G_CALLBACK(onCameraMenuClicked), window);
      g_object_set_data(G_OBJECT(item), "Camera", (gpointer)current);
      gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    }
  DBG_fprintf(stderr, "Gtk RenderingWindow: create the camera menu %p.\n",
	      (gpointer)menu);
  return menu;
}
static void onCameraMenuSelected(GtkMenuShell *menushell, gpointer user_data _U_)
{
  DBG_fprintf(stderr, "Gtk RenderingWindow: destroy the camera menu %p.\n",
	      (gpointer)menushell);
  gtk_widget_destroy(GTK_WIDGET(menushell));
}
static void onCameraMenuClicked(GtkMenuItem *menuitem, gpointer user_data)
{
  OpenGLCamera *camera;
  gboolean reDrawNeeded;
  RenderingWindow *window;

  window = RENDERING_WINDOW(user_data);
  if (!window->currentData)
    return;

  camera = (OpenGLCamera*)g_object_get_data(G_OBJECT(menuitem), "Camera");
  if (!camera)
    return;
  
  visuInteractivePush_savedCamera(window->inter, camera);

  reDrawNeeded = visuDataSet_angleOfView(window->currentData,
					 camera->theta,
					 camera->phi,
					 camera->omega,
					 MASK_THETA | MASK_PHI | MASK_OMEGA);
  reDrawNeeded = visuDataSet_positionOfView(window->currentData,
					    camera->xs,
					    camera->ys,
					    MASK_XS | MASK_YS) ||
    reDrawNeeded;
  reDrawNeeded = visuDataSet_zoomOfView(window->currentData,
					camera->gross) ||
    reDrawNeeded;
  reDrawNeeded = visuDataSet_perspectiveOfView(window->currentData,
					       camera->d_red) ||
    reDrawNeeded;
  if (reDrawNeeded)
    g_idle_add(visuObjectRedraw, GINT_TO_POINTER(TRUE));
}
static void displayFileInfoOnDataLoaded(RenderingWindow *window)
{
  gchar* message;
  VisuDataIter iter;

  g_return_if_fail(window);

  if (window->currentData)
    {
      message = visuDataGet_fileCommentary(window->currentData,
					   visuDataGet_setId(window->currentData));
      visuDataIter_new(window->currentData, &iter);
      gtkStatusInfo_setNbNodes(window->info, iter.nAllStoredNodes);
    }
  else
    {
      message = (gchar*)0;
      gtkStatusInfo_setNbNodes(window->info, -1);
    }
  if (message && message[0])
    gtkStatusInfo_setFileDescription(window->info, message);
  else
    gtkStatusInfo_setFileDescription(window->info,
				     GTK_STATUSINFO_NOFILEINFO);
}

void renderingWindowLoad_file(RenderingWindow *window, GtkWindow *parent)
{
  gboolean res;
  VisuGtkSetFilesFunc loadAction;
  VisuData *newData;
  guint w, h;

  loadAction = visuGtkGet_renderingSpecificOpen(visuRenderingClassGet_current());
  g_return_if_fail(loadAction);

  if (window->currentData)
    newData = visuDataNew_withOpenGLView(visuDataGet_openGLView(window->currentData));
  else
    {
      if (GTK_WIDGET_REALIZED(window))
	{
	  renderingWindowGet_openGLAreaSize(window, &w, &h);
	  newData = visuDataNew_withSize(w, h);
	}
      else
	newData = visuDataNew();
    }
  g_return_if_fail(newData);

  res = loadAction(newData, parent);
  DBG_fprintf(stderr, "Gtk RenderingWindow: 'loadAction' OK.\n");
  
  if (res)
    {
      visuGtkLoad_file(newData, 0);
      g_idle_add(visuObjectRedraw, (gpointer)0);
    }
  else
    g_object_unref(newData);
}

static void onLoadFileClicked(RenderingWindow *window, gpointer data _U_)
{
  renderingWindowLoad_file(window, (GtkWindow*)0);
}

static void onReloadClicked(GtkButton *button _U_, gpointer data)
{
  RenderingWindow *window;
  VisuData *dataObj;
  int id;

  window = RENDERING_WINDOW(data);
  g_return_if_fail(window);

/*   if (RENDERING_WINDOW(data)->currentData) */
/*     rebuildAllExtensionsLists(RENDERING_WINDOW(data)->currentData); */
/*   renderingWindowRedraw(RENDERING_WINDOW(data), TRUE); */
  dataObj = window->currentData;
  g_return_if_fail(dataObj);

  id = visuDataGet_setId(dataObj);
  visuDataFree_population(dataObj);
  g_object_ref(dataObj);
  visuGtkLoad_file(dataObj, id);
  g_idle_add(visuObjectRedraw, GINT_TO_POINTER(TRUE));
}

static void onRenderingMethodChanged(RenderingWindow *window, VisuRendering *method,
				     gpointer data _U_)
{
  if (window->currentData)
    /* First free attached visuData. */
    renderingWindowSet_visuData(window, (VisuData*)0);

  /* Customize interface according to new method. */
  if (method)
    {
      renderingWindowPop_message(window);
      if (window->info->loadButton)
	gtk_widget_set_sensitive(window->info->loadButton, TRUE);
      renderingWindowPush_message(window,
				  _("Use the 'open' button to render a file."));
    }
  else
    {
      if (window->info->loadButton)
	gtk_widget_set_sensitive(window->info->loadButton, FALSE);
      renderingWindowPop_message(window);
    }
}
static void onNodePopulationChanged(VisuData *data, int *nodes _U_,
				    gpointer user_data)
{
  VisuDataIter iter;

  if (data != RENDERING_WINDOW(user_data)->currentData)
    return;

  /* Change the count of nodes. */
  visuDataIter_new(data, &iter);
  gtkStatusInfo_setNbNodes(RENDERING_WINDOW(user_data)->info, iter.nAllStoredNodes);
}
static void onBoxSizeChanged(VisuData *data, gpointer user_data _U_)
{
  DBG_fprintf(stderr, "Gtk RenderingWindow: apply OpenGL projection"
	      " on current VisuData.\n");
  openGLViewCompute_matrixAndView(visuDataGet_openGLView(data));
}
static void setFileButtonsSensitive(RenderingWindow *window)
{
  g_return_if_fail(window);

  if (!window->info->dumpButton || !window->info->reloadButton)
    return;

  if (window->currentData)
    {
      gtk_widget_set_sensitive(window->info->dumpButton, TRUE);
      gtk_widget_set_sensitive(window->info->reloadButton, TRUE);
    }
  else
    {
      gtk_widget_set_sensitive(window->info->dumpButton, FALSE);
      gtk_widget_set_sensitive(window->info->reloadButton, FALSE);
    }
}

void updateDumpProgressBar(gpointer data)
{
  gdouble val;

  g_return_if_fail(GTK_PROGRESS_BAR(data));

  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(data), _("Saving image..."));
  val = gtk_progress_bar_get_fraction(GTK_PROGRESS_BAR(data));
  if (val + 0.01 <= 1.0 && val >= 0.)
    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(data), val + 0.01);
  visuGtkWait();
}

static void redrawOffScreen(const char** lists, VisuData *data)
{
  gint viewport[4];

  glGetIntegerv(GL_VIEWPORT, viewport);
  openGLInit_context();
  visuDataSet_sizeOfView(data, viewport[2], viewport[3]);
  openGLViewCompute_matrixAndView(visuDataGet_openGLView(data)); 
  rebuildAllExtensionsLists(VISU_DATA(data));
  openGL_reDraw(lists, data);
}

gboolean renderingWindowDump(RenderingWindow *window, DumpType *format,
			     const char* fileName, gint width, gint height,
			     GError **error,
			     voidDataFunc functionWait, gpointer data)
{
  guchar *imageData;
  gboolean res;

  g_return_val_if_fail(IS_RENDERING_WINDOW(window), FALSE);
  g_return_val_if_fail(window->currentData, FALSE);
  g_return_val_if_fail(error && !*error, FALSE);
  g_return_val_if_fail(format && fileName, FALSE);

  if (format->bitmap)
    {
      DBG_fprintf(stderr, "Gtk RenderingWindow : dumping current OpenGL area.\n");
      DBG_fprintf(stderr, " | requested size %dx%d.\n", width, height);
      openGLWidgetSet_redraw(OPENGL_WIDGET(window->openGLArea),
			     redrawOffScreen, window->currentData);
      imageData = openGLWidgetGet_pixmapData(OPENGL_WIDGET(window->openGLArea),
					     &width, &height, TRUE,
					     format->hasAlpha);
      visuDataSet_sizeOfView(VISU_DATA(window->currentData),
			     window->socketWidth, window->socketHeight);
      openGLWidgetSet_redraw(OPENGL_WIDGET(window->openGLArea),
			     openGL_reDraw, window->currentData);
      DBG_fprintf(stderr, " | allocated size %dx%d.\n", width, height);
      if (!imageData)
	{
	  *error = g_error_new(VISU_ERROR_DUMP, DUMP_ERROR_OPENGL,
			       _("Can't dump OpenGL area to data.\n"));
	  return FALSE;
	}
    }
  else
    imageData = (guchar*)0;

  res = format->writeFunc(format->fileType, fileName,
			  width, height, window->currentData,
			  imageData, error, functionWait, data);

  if (imageData)
    g_free(imageData);
  return res;
}

static void onDumpButtonClicked(GtkButton *button _U_, gpointer user_data)
{
  GtkWidget *dump;
  char *filename;
  DumpType *format;
  gboolean res;
  GError *error;
  GdkCursor *cursorWatch;
  GtkProgressBar *dumpBar;

  dump = dumpDialog_new(RENDERING_WINDOW(user_data)->currentData,
			(GtkWindow*)0, (const gchar*)0);
  if (gtk_dialog_run(GTK_DIALOG(dump)) != GTK_RESPONSE_ACCEPT)
    {
      gtk_widget_destroy(dump);
      return;
    }

  filename = dumpDialogGet_fileName(DUMP_DIALOG(dump));
  format = dumpDialogGet_dumpType(DUMP_DIALOG(dump));
  g_return_if_fail(format && filename);

  DBG_fprintf(stderr, "Gtk StatusInfo : dump image to the file '%s' (format : %s)\n",
	      filename, format->fileType->description);
  if (format->writeFunc)
    {
      cursorWatch = gdk_cursor_new(GDK_WATCH);
      dumpBar = dumpDialogGet_progressBar(DUMP_DIALOG(dump));
      dumpDialogStart(DUMP_DIALOG(dump));
      gtk_progress_bar_set_fraction(dumpBar, 0.);
      if (format->bitmap)
	gtk_progress_bar_set_text(dumpBar,
				  _("Waiting for generating image in memory..."));
      visuGtkWait();
      gdk_window_set_cursor(GDK_WINDOW((GTK_WIDGET(dump))->window), cursorWatch);

      error = (GError*)0;
      res = renderingWindowDump(RENDERING_WINDOW(user_data), format, filename,
				dumpDialogGet_widthValue(DUMP_DIALOG(dump)),
				dumpDialogGet_heightValue(DUMP_DIALOG(dump)), &error,
				updateDumpProgressBar, (gpointer)dumpBar);

      if (!res && error)
	visuGtkRaise_warning(_("Saving a file"), error->message, (GtkWindow*)0);
      gdk_window_set_cursor(GDK_WINDOW((GTK_WIDGET(dump))->window), NULL);
      if (error)
	g_error_free(error);
    }
  gtk_widget_destroy(dump);
}

void renderingWindowSet_panel(RenderingWindow *window, GtkWidget *panel)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));
  window->panel = panel;
  
  if (GTK_IS_WIDGET(panel) && window->info->raiseButton)
    gtk_widget_show(window->info->raiseButton);
}

static void onRaiseButtonClicked(RenderingWindow *window, gpointer user_data _U_)
{
  DBG_fprintf(stderr, "Gtk RenderingWindow: get a raise main window event.\n");
  if (window->panel && GTK_IS_WINDOW(window->panel))
    gtk_window_present(GTK_WINDOW(window->panel));
}

void renderingWindowRedraw(RenderingWindow *window, gboolean forceRedraw)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));
  if (!GTK_WIDGET_REALIZED(GTK_WIDGET(window)))
    return;

  if (!openGLGet_immediate() && !forceRedraw)
    {
      DBG_fprintf(stderr, "Redraw rejected since drawing is deferred and not forced.\n");
      return;
    }
  DBG_fprintf(stderr, "Redraw accepted let's go...\n");

  openGLWidgetRedraw(OPENGL_WIDGET(window->openGLArea));
}

void renderingWindowSet_current(RenderingWindow *window, gboolean status)
{
  g_return_if_fail(IS_RENDERING_WINDOW(window));

  openGLWidgetSet_current(OPENGL_WIDGET(window->openGLArea), status);
}

static void onRedraw(RenderingWindow *window, gpointer data _U_)
{
  renderingWindowRedraw(window, FALSE);
}
static void onForceRedraw(RenderingWindow *window, gpointer data _U_)
{
  renderingWindowRedraw(window, TRUE);
}

VisuInteractive* renderingWindowGet_interactive(RenderingWindow *window)
{
  g_return_val_if_fail(IS_RENDERING_WINDOW(window), (VisuInteractive*)0);

  return window->inter;
}
