/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "support.h"
#include "gtk_interactive.h"
#include "visu_gtk.h"
#include "visu_pickMesure.h"
#include "gtk_main.h"
#include "gtk_pick.h"
#include "visu_object.h"
#include "visu_basic.h"
#include "gtk_renderingWindowWidget.h"
#include "renderingBackend/visu_windowInterface.h"
#include "extraFunctions/dataNode.h"
#include "extensions/extInfos.h"
#include "extraGtkFunctions/gtk_valueIOWidget.h"

/* The ids of the column used in the data treeview.
   This treeview is made on a list with NB_COLUMN_BASE + n * NB_COLUMN_DATA,
   where n is the number of data associated per node. */
enum
  {
    COLUMN_BASE_NUMBER,
    COLUMN_BASE_ELEMENT,
    COLUMN_BASE_HIGHLIGHT,
    NB_COLUMN_BASE
  };
enum
  {
    COLUMN_DATA_LABEL,
    COLUMN_DATA_EDITABLE,
    COLUMN_DATA_COLOR,
    NB_COLUMN_DATA
  };

/* The ids of the columns used in the combobox that identify
   each data node. */
enum
  {
    COLUMN_COMBO_LABEL,
    COLUMN_COMBO_STOCK,
    COLUMN_COMBO_POINTER,
    NB_COLUMN_COMBO
  };

static GtkWidget *comboDraw;

#define GTK_PICK_INFO				\
  _("left-button\t\t\t: standard pick\n"			\
    "control-left-button\t\t: toggle highlihgt node\n"		\
    "middle-button\t\t: measure node neighbouring\n"		\
    "shift-middle-button\t: pick 1st reference\n"		\
    "ctrl-middle-button\t\t: pick 2nd reference\n"		\
    "drag-left-button\t\t: make a rectangular selection\n"	\
    "right-button\t\t\t: switch to observe")

/* Treeview used to print data of nodes. */
static GtkListStore *listDataNode;
static GtkWidget *treeviewDataNode;
#define GTK_PICK_EDITABLE_NODE   "blue"
#define GTK_PICK_UNEDITABLE_NODE "black"
static GtkTreeViewColumn** dataCols;

/* Draw data widgets. */
static GtkWidget *radioDrawNever, *radioDrawSelected, *radioDrawAlways;
static GtkListStore *listComboInfos;
static GtkWidget *lblList;
static GtkWidget *valueIO;
static GtkWidget *hboxMarks, *tglMarks, *lblMarks;
static guint nbMarks;

/* The pick viewport. */
static GtkWidget *labelPickOut, *labelPickError;

/* Signals that need to be suspended */
static gulong radioInfosSignals[3];
static gulong comboInfosSignal;
static gulong askForHideSignal;
static gulong onSelection_id;

/* Local callbacks. */
static void onEditedPick(GtkCellRendererText *cellrenderertext,
			 gchar *path, gchar *text, gpointer user_data);
static void onNodePropertyUsed(DataNode* data, VisuData *dataObj, gpointer user_data);
static void onNodePropertyUnused(DataNode* data, VisuData *dataObj,
				 gpointer user_data);
static void onNodePropertyChanged(DataNode* data, VisuData *dataObj,
				  gpointer user_data);
static void onRadioDrawInfos(GtkToggleButton *togglebutton, gpointer user_data);
static void onDrawDistanceChecked(GtkToggleButton* button, gpointer data);
static void onEraseDistanceClicked(GtkButton *button, gpointer user_data);
static void onVisuDataChanged(GObject *obj, VisuData *dataObj, gpointer bool);
static void onNodeRemoved(VisuData *visuData, int *nodeNumbers, gpointer data);
static void onComboInfosChanged(GtkComboBox *combo, gpointer data);
static gboolean onTreeviewInfosKey(GtkWidget *widget, GdkEventKey *event,
				   gpointer user_data);
static void onEraseMarksClicked(GtkButton *button, gpointer user_data);
static void onAskForHideNodes(VisuData *visuData, gboolean *redraw, gpointer data);
static void onHighlightToggled(GtkCellRendererToggle *cell_renderer,
			       gchar *path, gpointer user_data);
static void onHighlightClicked(GtkButton *button, gpointer user_data);
static void onHighlightSetClicked(GtkButton *button, gpointer user_data);
static void onHighlightHideToggled(GtkButton *button, gpointer user_data);
static void onPageEnter(GtkNotebook *notebook, GtkNotebookPage *page,
			guint page_num, gpointer user_data);
static void updateInfos(VisuInteractive *inter,
			PickMesure *pickMesure, gpointer data);

/* Local routines. */
static int* getListedNodes();
static void updateLabelMarks();
static void updateLabelList();
static void getIterPick(guint nodeId, GtkTreeIter *iter);
static void addNodeToList(VisuData *dataObj, VisuNode *node, gboolean withRedraw);
static void drawDataOnNode(VisuData *data, DrawItem item);
static void addNodeAtIter(VisuData *dataObj, VisuNode *node, GtkTreeIter *iter,
			  gboolean highlight);
static void populateComboInfos(VisuData *dataObj);
static gboolean applyHidingScheme(VisuData *data);
static gboolean onLoadXML(const gchar *filename, GError **error);
static gboolean onSaveXML(const gchar *filename, GError **error);


/********************/
/* Public routines. */
/********************/
GtkWidget* gtkPickBuild_interface(GtkMain *main, gchar **label,
				  gchar **help, GtkWidget **radio)
{
  int i, j, nb, nbData;
  GList *tmpLst;
  GType *dataTypes;
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  const gchar *title;
  GtkWidget *lbl, *wd, *hbox, *vbox, *image;
  gchar *markup;
  RenderingWindow *window;
  VisuData *dataObj;
  gboolean visible;
#if GTK_MINOR_VERSION < 12
  GtkTooltips *tooltips;
  tooltips = gtk_tooltips_new ();
#endif

  *label = g_strdup("Pick");
  *help  = g_strdup(GTK_PICK_INFO);
  *radio = lookup_widget(main->interactiveDialog, "radioPick");

  /* Get the current VisuData object. */
  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  dataObj = renderingWindowGet_visuData(window);

  /* Create the liststore used for the DataNode. */
  tmpLst = nodeDataGet_list();
  nbData = g_list_length(tmpLst);
  dataCols = g_malloc(sizeof(GtkTreeViewColumn*) * nbData);
  nb = nbData * NB_COLUMN_DATA + NB_COLUMN_BASE;
  dataTypes = g_malloc(sizeof(GType) * nb);
  dataTypes[COLUMN_BASE_NUMBER] = G_TYPE_UINT;
  dataTypes[COLUMN_BASE_ELEMENT] = G_TYPE_STRING;
  dataTypes[COLUMN_BASE_HIGHLIGHT] = G_TYPE_BOOLEAN;
  for (i = NB_COLUMN_BASE; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      /* The string to put on screen. */
      dataTypes[i + COLUMN_DATA_LABEL] = G_TYPE_STRING;
      /* Wether this string is editable or not. */
      dataTypes[i + COLUMN_DATA_EDITABLE] = G_TYPE_BOOLEAN;
      /* Give the color used to render the string. */
      dataTypes[i + COLUMN_DATA_COLOR] = G_TYPE_STRING;
      i += NB_COLUMN_DATA;
    }
  listDataNode = gtk_list_store_newv(nb, dataTypes);
  g_free(dataTypes);

  /* Create the treeview and related buttons. */
  hbox = lookup_widget(main->interactiveDialog, "hbox74");

  wd = gtk_scrolled_window_new(NULL, NULL);
  gtk_box_pack_start(GTK_BOX(hbox), wd, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(wd),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(wd), GTK_SHADOW_IN);

  treeviewDataNode = gtk_tree_view_new ();
  gtk_container_add(GTK_CONTAINER(wd), treeviewDataNode);
  gtk_widget_set_size_request(treeviewDataNode, -1, 100);
  gtk_tree_view_set_model(GTK_TREE_VIEW(treeviewDataNode),
			  GTK_TREE_MODEL(listDataNode));
  g_signal_connect(G_OBJECT(treeviewDataNode), "key-press-event",
		   G_CALLBACK(onTreeviewInfosKey), (gpointer)0);
  gtk_tree_selection_set_mode
    (gtk_tree_view_get_selection(GTK_TREE_VIEW(treeviewDataNode)),
     GTK_SELECTION_MULTIPLE);

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 2);

  wd = gtk_button_new();
  gtk_widget_set_tooltip_text(wd, _("Remove all highlight marks for"
				       " the nodes of the list."));
  gtk_box_pack_start(GTK_BOX(vbox), wd, FALSE, FALSE, 1);
  image = create_pixmap((GtkWidget*)0, "stock-unselect-all_20.png");
  gtk_container_add(GTK_CONTAINER(wd), image);
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onHighlightClicked), GINT_TO_POINTER(FALSE));

  wd = gtk_button_new();
  gtk_widget_set_tooltip_text(wd, _("Put a highlight mark on all"
				       " the nodes of the list."));
  gtk_box_pack_end(GTK_BOX(vbox), wd, FALSE, FALSE, 1);
  image = create_pixmap((GtkWidget*)0, "stock-select-all_20.png");
  gtk_container_add(GTK_CONTAINER(wd), image);
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onHighlightClicked), GINT_TO_POINTER(TRUE));

  gtk_widget_show_all(hbox);

  /* Create the list for the combobox of data node. */
  listComboInfos = gtk_list_store_new(NB_COLUMN_COMBO, G_TYPE_STRING,
				      G_TYPE_STRING, G_TYPE_POINTER);
  /* Building headers. */
  /* Id colum. */
  renderer = gtk_cell_renderer_text_new();
  g_object_set(G_OBJECT(renderer), "scale", 0.75, NULL);
  lbl = gtk_label_new("");
  title = _("Id");
  markup = g_markup_printf_escaped("<span size=\"smaller\">%s</span>", title);
  gtk_label_set_markup(GTK_LABEL(lbl), markup);
  gtk_widget_show(lbl);
  g_free(markup);
  column = gtk_tree_view_column_new_with_attributes(_("Node"), renderer,
						    "text", COLUMN_BASE_NUMBER,
						    NULL);
  gtk_tree_view_column_set_sort_column_id(column, COLUMN_BASE_NUMBER);
  gtk_tree_view_column_set_sort_indicator(column, TRUE);
  gtk_tree_view_column_set_widget(column, lbl);
  gtk_tree_view_append_column (GTK_TREE_VIEW(treeviewDataNode), column);
  /* Highlight colum. */
  renderer = gtk_cell_renderer_toggle_new();
  g_signal_connect(G_OBJECT(renderer), "toggled",
		   G_CALLBACK(onHighlightToggled), (gpointer)0);
  g_object_set(G_OBJECT(renderer), "indicator-size", 10, NULL);
  column = gtk_tree_view_column_new_with_attributes("", renderer,
						    "active", COLUMN_BASE_HIGHLIGHT,
						    NULL);
  gtk_tree_view_column_set_sort_column_id(column, COLUMN_BASE_HIGHLIGHT);
  gtk_tree_view_append_column (GTK_TREE_VIEW(treeviewDataNode), column);
  /* Element column. */
  renderer = gtk_cell_renderer_text_new();
  g_object_set(G_OBJECT(renderer), "scale", 0.75, NULL);
  lbl = gtk_label_new("");
  title = _("Type");
  markup = g_markup_printf_escaped("<span size=\"smaller\">%s</span>", title);
  gtk_label_set_markup(GTK_LABEL(lbl), markup);
  gtk_widget_show(lbl);
  g_free(markup);
  column = gtk_tree_view_column_new_with_attributes(_("Node"), renderer,
						    "text", COLUMN_BASE_ELEMENT, NULL);
  gtk_tree_view_column_set_sort_column_id(column, COLUMN_BASE_ELEMENT);
  gtk_tree_view_column_set_widget(column, lbl);
  gtk_tree_view_append_column (GTK_TREE_VIEW(treeviewDataNode), column);

  i = 0;
  for (tmpLst = nodeDataGet_list(); tmpLst; tmpLst = g_list_next(tmpLst))
    {
      j = i * NB_COLUMN_DATA + NB_COLUMN_BASE;
      visible = nodeDataGet_used(DATA_NODE(tmpLst->data), dataObj);
      renderer = gtk_cell_renderer_text_new();
      g_signal_connect(G_OBJECT(renderer), "edited",
		       G_CALLBACK(onEditedPick), GINT_TO_POINTER(i));
      g_object_set(G_OBJECT(renderer), "scale", 0.75, NULL);
      title = nodeDataGet_label(DATA_NODE(tmpLst->data));
      lbl = gtk_label_new("");
      markup = g_markup_printf_escaped("<span size=\"smaller\">%s</span>", title);
      gtk_label_set_markup(GTK_LABEL(lbl), markup);
      gtk_widget_show(lbl);
      g_free(markup);
      dataCols[i] = gtk_tree_view_column_new_with_attributes
	("", renderer,
	 "markup", j + COLUMN_DATA_LABEL,
	 "editable", j + COLUMN_DATA_EDITABLE,
	 "foreground", j + COLUMN_DATA_COLOR, NULL);
/*       g_object_set(G_OBJECT(column), "editable-set", TRUE, NULL); */
      gtk_tree_view_column_set_widget(dataCols[i], lbl);
      gtk_tree_view_append_column(GTK_TREE_VIEW(treeviewDataNode), dataCols[i]);

      g_signal_connect(G_OBJECT(tmpLst->data), "propertyUsed",
		       G_CALLBACK(onNodePropertyUsed), (gpointer)0);
      g_signal_connect(G_OBJECT(tmpLst->data), "propertyUnused",
		       G_CALLBACK(onNodePropertyUnused), (gpointer)0);
      g_signal_connect(G_OBJECT(tmpLst->data), "valueChanged",
		       G_CALLBACK(onNodePropertyChanged), (gpointer)0);

      i += 1;
    }

  /* Set the load/save widget. */
  valueIO = valueIONew(GTK_WINDOW(main->interactiveDialog),
		       _("Import picked nodes from an existing XML file."),
		       _("Export listed picked nodes to the current XML file."),
		       _("Export listed picked nodes to a new XML file."));
  valueIOSet_openSensitive(VALUE_IO(valueIO), TRUE);
  valueIOConnect_open(VALUE_IO(valueIO), onLoadXML);
  valueIOConnect_save(VALUE_IO(valueIO), onSaveXML);
  gtk_widget_show_all(valueIO);
  wd = lookup_widget(main->interactiveDialog, "hboxPick");
  gtk_box_pack_end(GTK_BOX(wd), valueIO, TRUE, TRUE, 10);

  /* Set the names and load the widgets. */
  comboDraw = lookup_widget(main->interactiveDialog, "comboboxShowInfos");
  gtk_combo_box_set_model(GTK_COMBO_BOX(comboDraw), GTK_TREE_MODEL(listComboInfos));
  gtk_cell_layout_clear(GTK_CELL_LAYOUT(comboDraw));
  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(comboDraw), renderer, FALSE);
  gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(comboDraw), renderer, "stock-id",
				COLUMN_COMBO_STOCK);
  renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(comboDraw), renderer, FALSE);
  gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(comboDraw), renderer, "markup",
				COLUMN_COMBO_LABEL);
  gtk_combo_box_set_active(GTK_COMBO_BOX(comboDraw), 0);
  comboInfosSignal = g_signal_connect(G_OBJECT(comboDraw), "changed",
				      G_CALLBACK(onComboInfosChanged), (gpointer)0);
  labelPickOut = lookup_widget(main->interactiveDialog, "pickInfo");
  labelPickError = lookup_widget(main->interactiveDialog, "pickComment");
  gtk_widget_set_name(labelPickError, "label_error");
  wd = lookup_widget(main->interactiveDialog, "viewportPick");
  gtk_widget_set_name(wd, "message_viewport");
  wd = lookup_widget(main->interactiveDialog, "checkDrawDistance");
  gtk_widget_set_name(wd, "message_radio");
  radioDrawNever = lookup_widget(main->interactiveDialog, "radioDrawNever");
  gtk_widget_set_name(radioDrawNever, "message_radio");
  radioInfosSignals[DRAW_NEVER] =
    g_signal_connect(G_OBJECT(radioDrawNever), "toggled",
		     G_CALLBACK(onRadioDrawInfos), GINT_TO_POINTER(DRAW_NEVER));
  radioDrawSelected = lookup_widget(main->interactiveDialog, "radioDrawSelected");
  gtk_widget_set_name(radioDrawSelected, "message_radio");
  radioInfosSignals[DRAW_SELECTED] =
    g_signal_connect(G_OBJECT(radioDrawSelected), "toggled",
		     G_CALLBACK(onRadioDrawInfos), GINT_TO_POINTER(DRAW_SELECTED));
  radioDrawAlways = lookup_widget(main->interactiveDialog, "radioDrawAlways");
  gtk_widget_set_name(radioDrawAlways, "message_radio");
  radioInfosSignals[DRAW_ALWAYS] =
    g_signal_connect(G_OBJECT(radioDrawAlways), "toggled",
		     G_CALLBACK(onRadioDrawInfos), GINT_TO_POINTER(DRAW_ALWAYS));

  wd = lookup_widget(main->interactiveDialog, "checkDrawDistance");
  g_signal_connect(G_OBJECT(wd), "toggled",
		   G_CALLBACK(onDrawDistanceChecked), (gpointer)0);
  wd = lookup_widget(main->interactiveDialog, "buttonEraseDistances");
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onEraseDistanceClicked), (gpointer)0);
  hboxMarks = lookup_widget(main->interactiveDialog, "hboxMarks");
  gtk_widget_set_sensitive(hboxMarks, FALSE);
  wd = lookup_widget(main->interactiveDialog, "buttonEraseMarks");
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onEraseMarksClicked), (gpointer)0);
  wd = lookup_widget(main->interactiveDialog, "buttonSetMarks");
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onHighlightSetClicked), (gpointer)0);
  tglMarks = gtk_toggle_button_new();
  gtk_widget_set_tooltip_text(tglMarks, _("Hide all non-highlighted nodes."));
  image = create_pixmap((GtkWidget*)0, "stock-masking.png");
  gtk_container_add(GTK_CONTAINER(tglMarks), image);
  gtk_box_pack_start(GTK_BOX(hboxMarks), tglMarks, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(tglMarks), "toggled",
		   G_CALLBACK(onHighlightHideToggled), (gpointer)0);
  lblMarks = lookup_widget(main->interactiveDialog, "labelMarks");
  gtk_widget_show_all(hboxMarks);
  nbMarks = 0;
  lblList = lookup_widget(main->interactiveDialog, "labelList");
  wd = lookup_widget(main->interactiveDialog, "notebookAction");
  g_signal_connect(G_OBJECT(wd), "switch-page",
		   G_CALLBACK(onPageEnter), (gpointer)0);

  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onVisuDataChanged), (gpointer)0);

  if (dataObj)
    {
      onVisuDataChanged((GObject*)0, dataObj, (gpointer)0);
      onPageEnter((GtkNotebook*)0, (GtkNotebookPage*)0, 0, (gpointer)0);
    }

  return (GtkWidget*)0;
}
void gtkPickStart(RenderingWindow *window)
{
  VisuInteractive *inter;

  inter = renderingWindowGet_interactive(window);
  visuInteractiveSet_type(inter, interactive_pick);
  onSelection_id = g_signal_connect(G_OBJECT(inter),
				    "selection", G_CALLBACK(updateInfos),
				    (gpointer)renderingWindowGet_visuData(window));
  pickMesureSet_formatedOutput(visuInteractiveGet_pickMesure(inter), TRUE);
}
void gtkPickStop(RenderingWindow *window)
{
  VisuInteractive *inter;

  inter = renderingWindowGet_interactive(window);
  g_signal_handler_disconnect(G_OBJECT(inter), onSelection_id);
  visuInteractiveSet_type(inter, interactive_none);
  pickMesureSet_formatedOutput(visuInteractiveGet_pickMesure(inter), FALSE);
}
static void updateInfos(VisuInteractive *inter _U_,
			PickMesure *pickMesure, gpointer data)
{
  gchar *infos, *errors;
  PickMesureType type;
  GList *lst, *tmpLst;
  VisuNode *node;
  GtkTreeIter iter;
  VisuData *dataObj;

  g_return_if_fail(pickMesure);

  dataObj = VISU_DATA(data);

  if (pickMesureGet_newsAvailable(pickMesure, &type))
    {
      gtk_tree_selection_unselect_all(gtk_tree_view_get_selection
				      (GTK_TREE_VIEW(treeviewDataNode)));
      switch (type)
	{
	case PICK_SELECTED:
	  /* Add the clicked node to the listStore. */
	  node = pickMesureGet_selectedNode(pickMesure);
	  if (node)
	    addNodeToList(dataObj, node, TRUE);
	case PICK_REFERENCE_1:
	case PICK_REFERENCE_2:
	  infos = pickMesureGet_infos(pickMesure);
	  errors = pickMesureGet_errors(pickMesure);
	  if (infos)
	    gtk_label_set_markup(GTK_LABEL(labelPickOut), infos);
	  if (errors)
	    gtk_label_set_text(GTK_LABEL(labelPickError), errors);
	  return /* FALSE */;
	case PICK_REGION:
	  /* We first empty the list. */
/* 	  gtk_list_store_clear(listDataNode); */
	  /* Update the visibility of valueIO. */
	  valueIOSet_saveSensitive(VALUE_IO(valueIO), FALSE);
	  /* We add the new elements. */
	  lst = pickMesureGet_regionNodes(pickMesure);
	  tmpLst = lst;
	  while (tmpLst)
	    {
	      addNodeToList(dataObj, (VisuNode*)tmpLst->data, (!tmpLst->next));
	      tmpLst = g_list_next(tmpLst);
	    }
	  g_list_free(lst);
	  return /* FALSE */;
	case PICK_HIGHLIGHT:
	case PICK_UNHIGHLIGHT:
	  node = pickMesureGet_highlightedNode(pickMesure);
	  g_return_if_fail(node);

	  getIterPick(node->number, &iter);
	  addNodeAtIter(dataObj, node, &iter, (type == PICK_HIGHLIGHT));
	  
	  nbMarks += (type == PICK_HIGHLIGHT)?1:-1;
	  updateLabelMarks();
	  return /* TRUE */;
	case PICK_INFORMATION:
	  break;
	default:
	  g_warning("Not a pick event!");
	}
    }
  return /* FALSE */;
}
void gtkPick_onClose()
{
  gtk_label_set_text(GTK_LABEL(labelPickOut), "");
  gtk_label_set_text(GTK_LABEL(labelPickError), "");
}
GList* gtkPickGet_nodeSelection()
{
  GList *lst;
  gboolean validIter;
  GtkTreeIter iter;
  unsigned int currentNodeId;

  lst = (GList*)0;
  validIter = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(listDataNode), &iter);
  while (validIter)
    {
      gtk_tree_model_get(GTK_TREE_MODEL(listDataNode), &iter,
			 COLUMN_BASE_NUMBER, &currentNodeId, -1);
      lst = g_list_prepend(lst, GINT_TO_POINTER((int)(currentNodeId - 1)));
      validIter = gtk_tree_model_iter_next(GTK_TREE_MODEL(listDataNode), &iter);
    }
  return lst;
}


/*********************/
/* Private routines. */
/*********************/
static void getIterPick(guint nodeId, GtkTreeIter *iter)
{
  gboolean validIter, found;
  guint currentNodeId;

  g_return_if_fail(iter);

  /* Search if @node is already in the tree. */
  found = FALSE;
  validIter = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(listDataNode), iter);
  while (validIter)
    {
      gtk_tree_model_get(GTK_TREE_MODEL(listDataNode), iter,
			 COLUMN_BASE_NUMBER, &currentNodeId, -1);
      if (nodeId + 1 == currentNodeId)
	{
	  found = TRUE;
	  validIter = FALSE;
	}
      else
	validIter = gtk_tree_model_iter_next(GTK_TREE_MODEL(listDataNode), iter);
    }
  if (!found)
    {
      gtk_list_store_append(listDataNode, iter);
      updateLabelList();
    }
}
static void populateComboInfos(VisuData *dataObj)
{
  gchar *markup;
  const gchar *title;
  GtkTreeIter iter;
  GList *tmpLst;
  gint i, n;
  gboolean visibility, set;

  DBG_fprintf(stderr, "Gtk Pick: rebuild the data combo list.\n");

  n = gtk_combo_box_get_active(GTK_COMBO_BOX(comboDraw));
  DBG_fprintf(stderr, "Gtk Pick: combo list previously %d.\n", n);

  g_signal_handler_block(G_OBJECT(comboDraw), comboInfosSignal);

  gtk_list_store_clear(listComboInfos);

  title = _("Id");
  markup = g_markup_printf_escaped("<span size=\"smaller\">%s</span>", title);
  gtk_list_store_append(listComboInfos, &iter);
  gtk_list_store_set(listComboInfos, &iter,
		     COLUMN_COMBO_LABEL, markup,
		     COLUMN_COMBO_POINTER, GINT_TO_POINTER(1),
		     -1);
  DBG_fprintf(stderr, " | add '%s'\n", title);
  g_free(markup);

  title = _("Type");
  markup = g_markup_printf_escaped("<span size=\"smaller\">%s</span>", title);
  gtk_list_store_append(listComboInfos, &iter);
  gtk_list_store_set(listComboInfos, &iter,
		     COLUMN_COMBO_LABEL, markup,
		     COLUMN_COMBO_POINTER, GINT_TO_POINTER(2),
		     -1);
  DBG_fprintf(stderr, " | add '%s'\n", title);
  g_free(markup);

  i = 0;
  for (tmpLst = nodeDataGet_list(); tmpLst; tmpLst = g_list_next(tmpLst))
    {
      visibility = nodeDataGet_used(DATA_NODE(tmpLst->data), dataObj);
      if (visibility)
	{
	  title = nodeDataGet_label(DATA_NODE(tmpLst->data));
	  markup = g_markup_printf_escaped("<span size=\"smaller\">%s</span>", title);

	  gtk_list_store_append(listComboInfos, &iter);
	  gtk_list_store_set(listComboInfos, &iter,
			     COLUMN_COMBO_LABEL, markup,
			     COLUMN_COMBO_POINTER, tmpLst->data,
			     -1);
	  if (nodeDataGet_editable(DATA_NODE(tmpLst->data)))
	    gtk_list_store_set(listComboInfos, &iter,
			       COLUMN_COMBO_STOCK, GTK_STOCK_EDIT,
			       -1);
	  DBG_fprintf(stderr, " | add '%s'\n", title);
	  g_free(markup);
	}
      /* Update the column visibility. */
      DBG_fprintf(stderr, " | col %d -> %d\n", i, visibility);
      g_object_set(G_OBJECT(dataCols[i]), "visible", visibility,
		   NULL);
      i += 1;
    }
  g_signal_handler_unblock(G_OBJECT(comboDraw), comboInfosSignal);

  set = (n >= 0 && n < gtk_tree_model_iter_n_children(GTK_TREE_MODEL(listComboInfos),
						      (GtkTreeIter*)0));
  gtk_combo_box_set_active(GTK_COMBO_BOX(comboDraw), (set)?n:0);

  DBG_fprintf(stderr, "Gtk Pick: combo list is now %d.\n", (set)?n:0);
}
static void updateLabelMarks()
{
  gboolean st;
  gchar *str;
  
  st = (nbMarks > 0);
  /* Update the visibility of highlight buttons. */
  gtk_widget_set_sensitive(hboxMarks, st);
  if (!st)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tglMarks), FALSE);

  if (st)
    str = g_strdup_printf(_("Highlights <span size=\"small\">(%d)</span>:"), nbMarks);
  else
    str = g_strdup(_("Highlights <span size=\"small\">(none)</span>:"));
  gtk_label_set_markup(GTK_LABEL(lblMarks), str);
  g_free(str);
}
static void updateLabelList()
{
  gchar *str;
  gint n;

  n = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(listDataNode), NULL);
  if (n > 0)
    str = g_strdup_printf(_("<b>List of nodes <span size="
			    "\"small\">(%d)</span>:</b>"), n);
  else
    str = g_strdup(_("<b>List of nodes <span size="
		     "\"small\">(none)</span>:</b>"));
  gtk_label_set_markup(GTK_LABEL(lblList), str);
  g_free(str);
}
static void addNodeAtIter(VisuData *dataObj, VisuNode *node, GtkTreeIter *iter,
			  gboolean highlight)
{
  GList *tmpLst;
  gchar *values, *label, *color;
  int i;
  gboolean editable;
  GtkTreePath *path;

  /* Store the base data. */
  gtk_list_store_set(listDataNode, iter,
		     COLUMN_BASE_NUMBER, node->number + 1, 
		     COLUMN_BASE_ELEMENT,
		     dataObj->fromIntToVisuElement[node->posElement]->name,
		     COLUMN_BASE_HIGHLIGHT, highlight,
		     -1);
  /* Store the additional data informations. */
  i = NB_COLUMN_BASE;
  for (tmpLst = nodeDataGet_list(); tmpLst; tmpLst = g_list_next(tmpLst))
    {
      editable = nodeDataGet_editable((DataNode*)tmpLst->data);
      label = nodeDataGet_valueAsString((DataNode*)tmpLst->data,
					dataObj, node);
      if (!label)
	{
	  values = _("<i>None</i>");
	  editable = FALSE;
	}
      else if (label[0] == '\0')
	values = _("<i>None</i>");
      else
	values = label;
      if (editable)
	color = GTK_PICK_EDITABLE_NODE;
      else
	color = GTK_PICK_UNEDITABLE_NODE;
      gtk_list_store_set(listDataNode, iter,
			 i + COLUMN_DATA_LABEL, values,
			 i + COLUMN_DATA_EDITABLE, editable,
			 i + COLUMN_DATA_COLOR, color, -1);
      if (label)
	g_free(label);
      i += NB_COLUMN_DATA;
    }
  /* Update the selection. */
  gtk_tree_selection_select_iter(gtk_tree_view_get_selection
				 (GTK_TREE_VIEW(treeviewDataNode)), iter);
  path = gtk_tree_model_get_path(GTK_TREE_MODEL(listDataNode), iter);
  gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(treeviewDataNode), path,
			       NULL, FALSE, 0., 0.);
  gtk_tree_path_free(path);

  /* Update the visibility of valueIO. */
  valueIOSet_saveSensitive(VALUE_IO(valueIO), TRUE);
}
static void addNodeToList(VisuData *dataObj, VisuNode *node, gboolean withRedraw)
{
  GtkTreeIter iter;
  gboolean hl;
  PickMesure *pickMesure;
  PickMesureType type;

  pickMesure = (PickMesure*)g_object_get_data(G_OBJECT(dataObj),
					      "pickMesure_data");

  getIterPick(node->number, &iter);
  if (pickMesureGet_typeNode(pickMesure, node->number, &type))
    hl = (type == PICK_HIGHLIGHT);
  else
    hl = FALSE;
  addNodeAtIter(dataObj, node, &iter, hl);

  /* If daw informations is DRAW_SELECTED, update the list. */
  if (withRedraw &&
      gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawSelected)))
    {
      drawDataOnNode(dataObj, DRAW_SELECTED);
      g_idle_add(visuObjectRedraw, (gpointer)0);
    }
}
static int* getListedNodes()
{
  GtkTreeIter iter;
  gboolean valid;
  int *nodes, i;

  nodes = g_malloc(sizeof(int) * 
		   (gtk_tree_model_iter_n_children(GTK_TREE_MODEL(listDataNode),
						   (GtkTreeIter*)0) + 1));
  i = 0;
  valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(listDataNode), &iter);
  while (valid)
    {
      gtk_tree_model_get(GTK_TREE_MODEL(listDataNode), &iter,
			 COLUMN_BASE_NUMBER, nodes + i,
			 -1);
      nodes[i] -= 1;
      i += 1;
      valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(listDataNode), &iter);
    }
  nodes[i] = -1;

  return nodes;
}

static void drawDataOnNode(VisuData *data, DrawItem item)
{
  GtkTreeIter iter;
  gpointer infos;
  gboolean valid;
  int *nodes;
  DrawMethod method;
  DataNode *dt;

  /* We get what data to draw. */
  valid = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(comboDraw), &iter);
  g_return_if_fail(valid);

  gtk_tree_model_get(GTK_TREE_MODEL(listComboInfos), &iter,
		     COLUMN_COMBO_POINTER, &infos,
		     -1);

  /* We build the list of elements to be drawn in DRAW_SELECTED mode. */
  if (item == DRAW_SELECTED)
    nodes = getListedNodes();
  else
    nodes = (int*)0;
  if (GPOINTER_TO_INT(infos) > 2)
    dt = DATA_NODE(infos);
  else
    dt = (DataNode*)0;
  if (item == DRAW_NEVER)
    method = DRAW_METH_NONE;
  else
    {
      switch (GPOINTER_TO_INT(infos))
	{
	case 1:
	  method = DRAW_METH_ID; break;
	case 2:
	  method = DRAW_METH_TYPE; break;
	default:
	  method = DRAW_METH_OTHER; break;
	}
    }
  visuBasicSet_extInfos(data, method, nodes, dt);
}
static gboolean rebuildDataOnNodes(DataNode *data, VisuData *dataObj)
{
  gboolean valid;
  GtkTreeIter iter;
  gpointer infos;

  DBG_fprintf(stderr, "Gtk Pick: maybe rebuild the data on nodes.\n");

  /* We rebuild the list for the drawn data if the dataNode match the current one. */
  /* We get what data to draw. */
  valid = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(comboDraw), &iter);
  g_return_val_if_fail(valid, FALSE);

  gtk_tree_model_get(GTK_TREE_MODEL(listComboInfos), &iter,
		     COLUMN_COMBO_POINTER, &infos,
		     -1);
  if (infos == (gpointer)data &&
      !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawNever)))
    {
      DBG_fprintf(stderr, "Gtk Pick: will redraw the data on nodes.\n");
  
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawSelected)))
	drawDataOnNode(dataObj, DRAW_SELECTED);
      else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawAlways)))
	drawDataOnNode(dataObj, DRAW_ALWAYS);

      return TRUE;
    }
  return FALSE;
}

/*************/
/* Callbacks */
/*************/
static void onPageEnter(GtkNotebook *notebook _U_, GtkNotebookPage *page _U_,
			guint page_num, gpointer user_data _U_)
{
  GList *list;
  RenderingWindow *window;
  VisuData *current;
  PickMesure *mesure;

  if (page_num != 0)
    return;

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  current = renderingWindowGet_visuData(window);
  mesure = (PickMesure*)g_object_get_data(G_OBJECT(current), "pickMesure_data");

  /* Update the number of highlight. */
  list = pickMesureGet_highlightedList(mesure);
  nbMarks = g_list_length(list);
  updateLabelMarks();
  g_list_free(list);
}
static void onEditedPick(GtkCellRendererText *cellrenderertext _U_,
			 gchar *path, gchar *text, gpointer user_data)
{
  gboolean valid;
  GtkTreeIter iter;
  gchar *label, *values;
  GList *lst;
  DataNode *data;
  RenderingWindow *window;
  VisuData *dataObj;
  gint number;

  valid = gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(listDataNode),
						  &iter, path);
  if (!valid)
    {
      g_warning("Wrong 'path' argument for 'onEditedPick'.");
      return;
    }

  gtk_tree_model_get(GTK_TREE_MODEL(listDataNode), &iter,
		     COLUMN_BASE_NUMBER, &number, -1);

  DBG_fprintf(stderr, "Gtk Pick: edited value on the fly.\n");
  lst = nodeDataGet_list();
  data = (DataNode*)g_list_nth_data(lst, GPOINTER_TO_INT(user_data));
  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  dataObj = renderingWindowGet_visuData(window);
  valid = nodeDataSet_valueAsString
    (data, dataObj,visuDataGet_nodeFromNumber(dataObj, number - 1), text, &label);
  if (!valid)
    visuGtkRaise_warning(_("Reading values"),
			 _("Wrong format. Impossible to parse the data associated"
			   " to the selected node."), (GtkWindow*)0);
  /* Change the value of the text in the list store. */
  if (label[0] == '\0')
    values = _("<i>None</i>");
  else
    values = label;
  gtk_list_store_set(listDataNode, &iter,
		     GPOINTER_TO_INT(user_data) * NB_COLUMN_DATA + NB_COLUMN_BASE,
		     values, -1);
  g_free(label);

  if (rebuildDataOnNodes(data, dataObj))
    g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onHighlightClicked(GtkButton *button _U_, gpointer user_data)
{
  GtkTreeIter iter;
  gboolean valid;
  guint node;
  RenderingWindow *window;
  VisuData *current;
  PickMesure *mesure;
  
  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  current = renderingWindowGet_visuData(window);
  mesure = (PickMesure*)g_object_get_data(G_OBJECT(current), "pickMesure_data");
  for (valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(listDataNode), &iter);
       valid; valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(listDataNode), &iter))
    {
      gtk_tree_model_get(GTK_TREE_MODEL(listDataNode), &iter,
			 COLUMN_BASE_NUMBER, &node,
			 -1);
      gtk_list_store_set(listDataNode, &iter,
			 COLUMN_BASE_HIGHLIGHT, GPOINTER_TO_INT(user_data),
			 -1);

      if (pickMesureSet_highlight(mesure, node - 1, GPOINTER_TO_INT(user_data)))
	nbMarks += (GPOINTER_TO_INT(user_data))?1:-1;
    }
  pickMesureRebuild_classicalList(current);
  updateLabelMarks();
  
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tglMarks)) &&
      applyHidingScheme(current))
    visuData_createAllNodes(current);

  g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onHighlightToggled(GtkCellRendererToggle *cell_renderer  _U_,
			       gchar *path, gpointer user_data _U_)
{
  gboolean valid, hl;
  GtkTreeIter iter;
  guint node;
  RenderingWindow *window;
  VisuData *current;
  PickMesure *mesure;

  valid = gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(listDataNode),
					      &iter, path);
  g_return_if_fail(valid);

  gtk_tree_model_get(GTK_TREE_MODEL(listDataNode), &iter,
		     COLUMN_BASE_HIGHLIGHT, &hl,
		     COLUMN_BASE_NUMBER, &node,
		     -1);
  gtk_list_store_set(listDataNode, &iter,
		     COLUMN_BASE_HIGHLIGHT, !hl,
		     -1);

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  current = renderingWindowGet_visuData(window);
  mesure = (PickMesure*)g_object_get_data(G_OBJECT(current), "pickMesure_data");

  if (pickMesureSet_highlight(mesure, node - 1, !hl))
    nbMarks += (!hl)?1:-1;
  pickMesureRebuild_classicalList(current);
  updateLabelMarks();

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tglMarks)) &&
      applyHidingScheme(current))
    visuData_createAllNodes(current);

  g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onHighlightSetClicked(GtkButton *button _U_, gpointer user_data _U_)
{
  GList *lst, *tmpLst;
  RenderingWindow *window;
  VisuData *current;
  PickMesure *mesure;
  VisuNode *node;
  GtkTreeIter iter;

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  current = renderingWindowGet_visuData(window);
  mesure = (PickMesure*)g_object_get_data(G_OBJECT(current), "pickMesure_data");

  lst = pickMesureGet_highlightedList(mesure);
  for (tmpLst = lst; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      node = visuDataGet_nodeFromNumber(current, GPOINTER_TO_INT(tmpLst->data));
      if (node)
	{
	  getIterPick(node->number, &iter);
	  addNodeAtIter(current, node, &iter, TRUE);
	}
    }
  if (lst &&
      gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawSelected)))
    {
      drawDataOnNode(current, DRAW_SELECTED);
      g_idle_add(visuObjectRedraw, (gpointer)0);
    }
  g_list_free(lst);
}
static gboolean applyHidingScheme(VisuData *data)
{
  gboolean redraw;

  visuDataEmit_askForShowHideNodes(data, &redraw);
  if (redraw)
    visuDataEmit_nodeRenderedChange(data);

  return redraw;
}
static void onHighlightHideToggled(GtkButton *button _U_, gpointer user_data _U_)
{
  RenderingWindow *window;
  VisuData *data;

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  data = renderingWindowGet_visuData(window);

  if (applyHidingScheme(data))
    {
      visuData_createAllNodes(data);
      g_idle_add(visuObjectRedraw, (gpointer)0);
    }
}
static void onNodePropertyUsed(DataNode* data _U_, VisuData *dataObj,
			       gpointer user_data _U_)
{
  RenderingWindow *window;
  VisuData *current;

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  current = renderingWindowGet_visuData(window);
  DBG_fprintf(stderr, "Gtk Pick: caught 'PropertyUsed' for %p (current %p).\n",
	      (gpointer)dataObj, (gpointer)current);
  if (current == dataObj)
    /* We update the combo for properties. */
    populateComboInfos(dataObj);
}
static void onNodePropertyUnused(DataNode* data _U_, VisuData *dataObj,
				 gpointer user_data _U_)
{
  RenderingWindow *window;
  VisuData *current;

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  current = renderingWindowGet_visuData(window);
  DBG_fprintf(stderr, "Gtk Pick: caught 'PropertyUnused' for %p (current %p).\n",
	      (gpointer)dataObj, (gpointer)current);
  if (current == dataObj)
    /* We update the combo for properties. */
    populateComboInfos(dataObj);
}
static void onNodePropertyChanged(DataNode* data, VisuData *dataObj,
				  gpointer user_data _U_)
{
  GtkTreeIter iter;
  gboolean valid;
  int i, nodeId;
  GList *tmpLst;
  gchar *label, *values;
  VisuNode *node;

  DBG_fprintf(stderr, "Gtk Pick: caught the 'valueChanged' signal.\n");

  valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(listDataNode), &iter);
  while (valid)
    {
      gtk_tree_model_get(GTK_TREE_MODEL(listDataNode), &iter,
			 COLUMN_BASE_NUMBER, &nodeId, -1);
      node = visuDataGet_nodeFromNumber(dataObj, nodeId - 1);
      g_return_if_fail(node);

      /* Store the additional data informations. */
      tmpLst = nodeDataGet_list();
      i = NB_COLUMN_BASE + NB_COLUMN_DATA * g_list_index(tmpLst, data);
      label = nodeDataGet_valueAsString(data, dataObj, node);
      if (!label || label[0] == '\0')
	values = _("<i>None</i>");
      else
	values = label;
      gtk_list_store_set(listDataNode, &iter,
			 i + COLUMN_DATA_LABEL, values, -1);
      if (label)
	g_free(label);

      valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(listDataNode), &iter);
    }

  rebuildDataOnNodes(data, dataObj);
}
static void onComboInfosChanged(GtkComboBox *combo, gpointer data _U_)
{
  RenderingWindow *window;
  VisuData *dataObj;

  window = visuRenderingWindowGet_current();
  dataObj = renderingWindowGet_visuData(window);
  g_return_if_fail(dataObj);

  if (gtk_combo_box_get_active(combo) >= 0)
    {
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawNever)))
	return;
      else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawSelected)))
	drawDataOnNode(dataObj, DRAW_SELECTED);
      else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawAlways)))
	drawDataOnNode(dataObj, DRAW_ALWAYS);
    }
  else
    drawDataOnNode(dataObj, DRAW_NEVER);
  g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onRadioDrawInfos(GtkToggleButton *togglebutton, gpointer user_data)
{
  RenderingWindow *window;
  VisuData *data;

  if (!gtk_toggle_button_get_active(togglebutton))
    return;

  window = visuRenderingWindowGet_current();
  data = renderingWindowGet_visuData(window);
  g_return_if_fail(data);

  drawDataOnNode(data, GPOINTER_TO_INT(user_data));
  g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onDrawDistanceChecked(GtkToggleButton* button, gpointer data _U_)
{
  RenderingWindow *window;
  PickMesure *pickMesure;

  window = visuRenderingWindowGet_current();
  pickMesure = (PickMesure*)g_object_get_data
    (G_OBJECT(renderingWindowGet_visuData(window)), "pickMesure_data");
  g_return_if_fail(pickMesure);
  pickMesureSet_storeMeasures(pickMesure, gtk_toggle_button_get_active(button));
}
static void onEraseDistanceClicked(GtkButton *button _U_, gpointer user_data _U_)
{
  RenderingWindow *window;
  PickMesure *pickMesure;

  window = visuRenderingWindowGet_current();
  pickMesure = (PickMesure*)g_object_get_data
    (G_OBJECT(renderingWindowGet_visuData(window)), "pickMesure_data");
  g_return_if_fail(pickMesure);
  DBG_fprintf(stderr, "Gtk Pick: clicked on 'erase all measures' button.\n");
  if (pickMesureRemove_allMeasures(pickMesure))
    g_idle_add(visuObjectRedraw, GINT_TO_POINTER(TRUE));
}
static void onAskForHideNodes(VisuData *visuData, gboolean *redraw, gpointer data _U_)
{
  RenderingWindow *window;
  PickMesure *pickMesure;
  VisuData *dataObj;

  DBG_fprintf(stderr, "Gtk Pick: caught the 'NodeAskForShowHide' signal for"
	      " VisuData %p.\n", (gpointer)visuData);

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tglMarks)))
    {
      window = visuRenderingWindowGet_current();
      dataObj = renderingWindowGet_visuData(window);
      pickMesure = (PickMesure*)g_object_get_data(G_OBJECT(dataObj),
						  "pickMesure_data");
      g_return_if_fail(pickMesure);
      *redraw = pickMesureHide_highlight(pickMesure, dataObj, TRUE) || *redraw;
    }
}
static void onVisuDataChanged(GObject *obj _U_, VisuData *dataObj, gpointer data _U_)
{
  GtkTreeIter iter, removedIter;
  gboolean valid, removed;
  int number;
  VisuNode *node;

  DBG_fprintf(stderr, "Gtk Pick: caught 'dataReadyForRendering'"
	      " signal for %p.\n", (gpointer)dataObj);

  if (dataObj)
    {
      g_signal_connect(G_OBJECT(dataObj), "NodePopulationDecrease",
		       G_CALLBACK(onNodeRemoved), (gpointer)0);
      askForHideSignal = g_signal_connect(G_OBJECT(dataObj), "NodeAskForShowHide",
					  G_CALLBACK(onAskForHideNodes), (gpointer)0);

      /* Rebuild the combo of properties. */
      populateComboInfos(dataObj);

      /* Try to match the selected nodes to new ones in the new visuData. */
      valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(listDataNode), &iter);
      while (valid)
	{
	  removed = FALSE;
	  gtk_tree_model_get(GTK_TREE_MODEL(listDataNode), &iter,
			     COLUMN_BASE_NUMBER, &number,
			     -1);
	  node = visuDataGet_nodeFromNumber(dataObj, number - 1);
	  if (node)
	    addNodeAtIter(dataObj, node, &iter, FALSE);
	  else
	    {
	      removed = TRUE;
	      removedIter = iter;
	    }

	  valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(listDataNode), &iter);
	  if (removed)
	    gtk_list_store_remove(listDataNode, &removedIter);
	}

      /* If daw informations is DRAW_SELECTED, update the list. */
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawSelected)))
	drawDataOnNode(dataObj, DRAW_SELECTED);
      /* If daw informations is DRAW_ALWAYS, update the list. */
      else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawAlways)))
	drawDataOnNode(dataObj, DRAW_ALWAYS);
    }
  else
    {
      /* Clear the node data. */  
      gtk_list_store_clear(listDataNode);
      /* Update the visibility of valueIO. */
      valueIOSet_saveSensitive(VALUE_IO(valueIO), FALSE);
      /* Clear the combobox of properties. */
      g_signal_handler_block(G_OBJECT(comboDraw), comboInfosSignal);
      gtk_list_store_clear(listComboInfos);
      g_signal_handler_unblock(G_OBJECT(comboDraw), comboInfosSignal);
    }
  updateLabelList();

  DBG_fprintf(stderr, "Gtk Pick: Done\n");

}
static void onNodeRemoved(VisuData *visuData, int *nodeNumbers, gpointer data _U_)
{
  GtkTreeIter iter, removeIter;
  gboolean valid;
  int number, i;
  gboolean found;

  DBG_fprintf(stderr, "Gtk Pick: caught the 'NodePopulationDecrease' signal.\n");

  valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(listDataNode), &iter);
  while (valid)
    {
      gtk_tree_model_get(GTK_TREE_MODEL(listDataNode), &iter,
			 0, &number,
			 -1);
      found = FALSE;
      for (i = 0; !found && nodeNumbers[i] >= 0; i++)
	if (number == (nodeNumbers[i] + 1))
	  {
	    removeIter = iter;
	    found = TRUE;
	  }
			 
      valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(listDataNode), &iter);
      if (found)
	gtk_list_store_remove(listDataNode, &removeIter);
    };
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawNever)))
    drawDataOnNode(visuData, DRAW_NEVER);
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawSelected)))
    drawDataOnNode(visuData, DRAW_SELECTED);
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawAlways)))
    drawDataOnNode(visuData, DRAW_ALWAYS);
  updateLabelList();
}
static gboolean onTreeviewInfosKey(GtkWidget *widget, GdkEventKey *event,
				   gpointer user_data _U_)
{
  RenderingWindow *window;
  VisuData *data;
  GList *selectedPaths, *tmpLst;
  GtkTreeIter iter, *iterCpy;
  GtkTreeModel *model;
  gboolean valid;

  DBG_fprintf(stderr, "Gtk Pick: key pressed on treeview '%d'.\n", event->keyval);
  
  if (event->keyval != GDK_Delete && event->keyval != GDK_BackSpace)
    return FALSE;

  selectedPaths = gtk_tree_selection_get_selected_rows
    (gtk_tree_view_get_selection(GTK_TREE_VIEW(widget)), &model);

  /* Transform all paths to iters in the list. */
  DBG_fprintf(stderr, "Gtk Pick: possible removing.\n");
  tmpLst = selectedPaths;
  while (tmpLst)
    {
      DBG_fprintf(stderr, " | remove path '%s'.\n",
		  gtk_tree_path_to_string((GtkTreePath*)tmpLst->data));

      valid = gtk_tree_model_get_iter(model, &iter, (GtkTreePath*)tmpLst->data);
      g_return_val_if_fail(valid, FALSE);

      gtk_tree_path_free((GtkTreePath*)tmpLst->data);
      iterCpy = gtk_tree_iter_copy(&iter);
      /* Replace the path with the tree iter. */
      tmpLst->data = (gpointer)iterCpy;

      tmpLst = g_list_next(tmpLst);
    };
  /* Then remove all iters. */
  DBG_fprintf(stderr, "Gtk Pick: actualy remove the iters.\n");
  tmpLst = selectedPaths;
  while (tmpLst && tmpLst->next)
    {
      gtk_list_store_remove(listDataNode, (GtkTreeIter*)tmpLst->data);
      gtk_tree_iter_free((GtkTreeIter*)tmpLst->data);
      tmpLst = g_list_next(tmpLst);
    };
  if (tmpLst)
    {
      gtk_list_store_remove(listDataNode, (GtkTreeIter*)tmpLst->data);
      gtk_tree_iter_free((GtkTreeIter*)tmpLst->data);
    }
  g_list_free(selectedPaths);
  updateLabelList();
  
  /* If the draw informations is on selected nodes, we need to redraw. */
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawSelected)))
    {
      DBG_fprintf(stderr, "Gtk Pick: ask for redraw.\n");
      window = RENDERING_WINDOW(visuRenderingWindowGet_current());
      data = renderingWindowGet_visuData(window);
      g_return_val_if_fail(data, FALSE);

      drawDataOnNode(data, DRAW_SELECTED);

      g_idle_add(visuObjectRedraw, (gpointer)0);
    }
  return TRUE;
}
static void onEraseMarksClicked(GtkButton *button _U_, gpointer user_data _U_)
{
  RenderingWindow *window;
  PickMesure *pickMesure;
  gboolean valid, redraw;
  GtkTreeIter iter;

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  pickMesure = (PickMesure*)g_object_get_data
    (G_OBJECT(renderingWindowGet_visuData(window)), "pickMesure_data");
  g_return_if_fail(pickMesure);

  /* Update the sensitivity. */
  if (gtk_tree_model_iter_n_children(GTK_TREE_MODEL(listDataNode),
				     (GtkTreeIter*)0) == 0)
    valueIOSet_saveSensitive(VALUE_IO(valueIO), FALSE);
  nbMarks = 0;
  updateLabelMarks();

  /* Remove all the check box in the model. */
  for (valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(listDataNode), &iter);
       valid; valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(listDataNode), &iter))
    gtk_list_store_set(listDataNode, &iter, COLUMN_BASE_HIGHLIGHT, FALSE, -1);

  DBG_fprintf(stderr, "Gtk Pick: remove all highlights.\n");
  redraw = pickMesureRemove_allHighlights(pickMesure);

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tglMarks)))
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tglMarks), FALSE);
  else if (redraw)
    g_idle_add(visuObjectRedraw, (gpointer)0);
}

static gboolean onLoadXML(const gchar *filename, GError **error)
{
  RenderingWindow *window;
  VisuData *dataObj;

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  dataObj = renderingWindowGet_visuData(window);

  if (gtkPickParse_XMLFile(filename, dataObj, error))
    {
      /* Update the sensitivity. */
      valueIOSet_saveSensitive(VALUE_IO(valueIO), TRUE);
      /* We ask for redraw. */
      g_idle_add(visuObjectRedraw, (gpointer)0);
      return TRUE;
    }
  else
    return FALSE;
}

gboolean gtkPickParse_XMLFile(const gchar* filename, VisuData *data, GError **error)
{
  gboolean set;
  GList *list, *tmpLst;
  VisuNode *node;
  DrawItem mode;
  guint info;
  PickMesure *mesure;

  mesure = pickMesureParse_XMLFile(filename, data, &list, &mode, &info, error);
  if (!mesure)
    return FALSE;

  /* We stop the callbacks during upgrade. */
  g_signal_handler_block(G_OBJECT(radioDrawNever), radioInfosSignals[DRAW_NEVER]);
  g_signal_handler_block(G_OBJECT(radioDrawSelected),
			 radioInfosSignals[DRAW_SELECTED]);
  g_signal_handler_block(G_OBJECT(radioDrawAlways), radioInfosSignals[DRAW_ALWAYS]);
  g_signal_handler_block(G_OBJECT(comboDraw), comboInfosSignal);

  /* We set the general options. */
  DBG_fprintf(stderr, "Gtk Pick: set the general options (%d %d).\n", mode, info);
  switch (mode)
    {
    case DRAW_NEVER:
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radioDrawNever), TRUE);
      break;
    case DRAW_SELECTED:
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radioDrawSelected), TRUE);
      break;
    case DRAW_ALWAYS:
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radioDrawAlways), TRUE);
      break;
    }
  set = (info < (guint)gtk_tree_model_iter_n_children(GTK_TREE_MODEL(listComboInfos),
						      (GtkTreeIter*)0));
  gtk_combo_box_set_active(GTK_COMBO_BOX(comboDraw), (set)?info:0);

  /* Convert the list to new pick. */
  DBG_fprintf(stderr, "Pick parser: create %d new pick for %p.\n",
	      g_list_length(list), (gpointer)data);
  tmpLst = list;
  while(tmpLst)
    {
      if (GPOINTER_TO_INT(tmpLst->data) == PICK_SELECTED)
	{
	  tmpLst = g_list_next(tmpLst);
	  node = visuDataGet_nodeFromNumber(data, GPOINTER_TO_INT(tmpLst->data) - 1);
	  if (node)
	    addNodeToList(data, node, FALSE);
	}
      else if (GPOINTER_TO_INT(tmpLst->data) == PICK_REFERENCE_1)
	{
	  tmpLst = g_list_next(tmpLst);
	  /* We add the last selection. */
	  tmpLst = g_list_next(tmpLst);
	  node = visuDataGet_nodeFromNumber(data, GPOINTER_TO_INT(tmpLst->data) - 1);
	  if (node)
	    addNodeToList(data, node, FALSE);
	}

      tmpLst = g_list_next(tmpLst);
    }
  g_list_free(list);
  updateInfos((VisuInteractive*)0, mesure, data);

  /* Update the number of highlight. */
  list = pickMesureGet_highlightedList(mesure);
  nbMarks = g_list_length(list);
  updateLabelMarks();
  g_list_free(list);

  /* We reset the callbacks during upgrade. */
  g_signal_handler_unblock(G_OBJECT(radioDrawNever), radioInfosSignals[DRAW_NEVER]);
  g_signal_handler_unblock(G_OBJECT(radioDrawSelected),
			   radioInfosSignals[DRAW_SELECTED]);
  g_signal_handler_unblock(G_OBJECT(radioDrawAlways), radioInfosSignals[DRAW_ALWAYS]);
  g_signal_handler_unblock(G_OBJECT(comboDraw), comboInfosSignal);

  /* We update the list. */
  drawDataOnNode(data, mode);

  return TRUE;
}

static gboolean onSaveXML(const gchar *filename, GError **error)
{
  RenderingWindow *window;
  VisuData *dataObj;
  int *nodes;
  DrawItem mode;
  int info;
  gboolean valid;

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  dataObj = renderingWindowGet_visuData(window);

  nodes = getListedNodes();

  mode = DRAW_NEVER;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawNever)))
    mode = DRAW_NEVER;
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawSelected)))
    mode = DRAW_SELECTED;
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioDrawAlways)))
    mode = DRAW_ALWAYS;
  info = gtk_combo_box_get_active(GTK_COMBO_BOX(comboDraw));

  valid = pickMesureExport_XMLFile(filename, dataObj, nodes, mode, info, error);

  g_free(nodes);

  return valid;
}
