/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include <string.h>

#include "visu_plugins.h"
#include "visu_tools.h"
#include "visu_basic.h"

/**
 * SECTION:visu_plugins
 * @short_description: Introduces the basic handling of plug-ins.
 *
 * <para>
 * Plug-ins are made of shared library presenting some common routines. These routines are of kind:
 * <itemizedlist>
 * <listitem><para>
 * pluginsInitFunc(), such a routine is called at V_Sim startup. It should initialise all things required by the module. For instance, if the module is used to add a load method for atomic rendering, it should call renderingAtomicAdd_loadMethod() to declare itself to V_Sim.
 * </para></listitem>
 * <listitem><para>
 * pluginsDescriptionFunc(), is used to get a short description of the plug-in.
 * </para></listitem>
 * <listitem><para>
 * pluginsAuthorsFunc(), gives a list of authors for the plug-in.
 * </para></listitem>
 * <listitem><para>
 * pluginsIconFunc(), returns a path where a small icon representing the plug-in can be found.
 * </para></listitem>
 * </itemizedlist>
 * All these routines must be named using the name of the module. When the module is loaded into memory, V_Sim will try to find these methods and will store them into the #VisuPlugin structure for future calls.
 * </para>
 * <para>
 * At the present time, only the Unix version is functional. Moreover, currently, no stable API is available from the main program. This should be corrected quickly.
 * </para>
 */

/* Local variables. */
GList *presentPlugins;

/* Local methods. */
static gboolean visuPluginLoad_byPath(gchar* path);


GList* visuPluginsGet_listLoaded()
{
  return presentPlugins;
}

static gboolean visuPluginLoad_byPath(gchar* path)
{
  gchar *func, *name, *ptChar, *libname;
  VisuPlugin *plugin;
  gboolean res;

  plugin = g_malloc(sizeof(VisuPlugin));
  plugin->hook = g_module_open(path, 0); /* G_MODULE_BIND_LOCAL); */
  if (plugin->hook)
    {
      libname = g_path_get_basename(path);
      name = g_strdup(libname + 3);
      g_free(libname);
#if SYSTEM_WIN32 == 1
      ptChar = g_strrstr(name, "-");
#endif
#if SYSTEM_X11 == 1
      ptChar = strchr(name, '.');
#endif
      if (ptChar)
	*ptChar = '\0';
      func = g_strdup_printf("%sInit", name);
      res = g_module_symbol(plugin->hook, func, (void*)&(plugin->init));
      g_free(func);
      if (!res)
	{
	  g_warning("The plugin '%s' doesn't have any %sInit() method.",
		    name, name);
	  g_free(plugin);
	  g_free(name);
	  return FALSE;
	}
      func = g_strdup_printf("%sGet_description", name);
      res = g_module_symbol(plugin->hook, func, (void*)&(plugin->getDescription));
      g_free(func);
      if (!res)
	{
	  g_warning("The plugin '%s' doesn't have any %sGet_description() method.",
		    name, name);
	  g_free(plugin);
	  g_free(name);
	  return FALSE;
	}
      func = g_strdup_printf("%sGet_authors", name);
      res = g_module_symbol(plugin->hook, func, (void*)&(plugin->getAuthors));
      g_free(func);
      if (!res)
	{
	  g_warning("The plugin '%s' doesn't have any %sGet_authors() method.",
		    name, name);
	  g_free(plugin);
	  g_free(name);
	  return FALSE;
	}
      func = g_strdup_printf("%sGet_icon", name);
      res = g_module_symbol(plugin->hook, func, (void*)&(plugin->getIcon));
      if (!res)
	plugin->getIcon = (pluginsIconFunc)0;
      g_free(func);
      func = g_strdup_printf("%sInitGtk", name);
      res = g_module_symbol(plugin->hook, func, (void*)&(plugin->initGtk));
      g_free(func);
      if (!res)
	plugin->initGtk = (pluginsInitGtkFunc)0;
      plugin->name = g_strdup(name);
      g_free(name);
      presentPlugins = g_list_prepend(presentPlugins, (gpointer)plugin);
    }
  else
    {
      g_warning("The plugin '%s' is not a loadable module, error:\n%s.",
		path, g_module_error());
      g_free(plugin);
      return FALSE;
    }
  
  return TRUE;
}

GList* visuPluginsParseDir(const gchar* path)
{
  GDir *gdir;
  const gchar *fileFromDir;
  GPatternSpec *pattern;
  GList* lst;

  lst = (GList*)0;
  gdir = g_dir_open(path, 0, NULL);
  if (!gdir)
    return lst;

  pattern = g_pattern_spec_new("lib*."G_MODULE_SUFFIX);
  fileFromDir = g_dir_read_name(gdir);
  while (fileFromDir)
    {
      if (g_pattern_match_string(pattern, fileFromDir))
	lst = g_list_prepend(lst, g_build_filename(path, fileFromDir, NULL));
      fileFromDir = g_dir_read_name(gdir);
    }
  g_dir_close(gdir);
  
  return lst;
}

/* Parse install directory and $XDG_CONFIG_HOME/v_sim/plugins to find
   files and create a gchar** array of names (as full path). */
gchar** visuPluginsGet_installedPlugins()
{
  gchar **data, *path;
  GList* lst1, *lst2, *tmpLst;
  int size;

  /* Try the install dir. */
  lst1 = visuPluginsParseDir(V_SIM_PLUGINS_DIR);
  /* Try the home dir. */
  path = g_build_filename(V_SIM_LOCAL_CONF_DIR, "plugins", NULL);
  lst2 = visuPluginsParseDir(path);
  g_free(path);

  size = g_list_length(lst1) + g_list_length(lst2) + 1;
  data = g_malloc(sizeof(gchar*) * size);
  
  size = 0;
  tmpLst = lst1;
  while(tmpLst)
    {
      data[size] = (gchar*)tmpLst->data;
      size += 1;
      tmpLst = g_list_next(tmpLst);
    }
  g_list_free(lst1);
  tmpLst = lst2;
  while(tmpLst)
    {
      data[size] = (gchar*)tmpLst->data;
      size += 1;
      tmpLst = g_list_next(tmpLst);
    }
  g_list_free(lst2);
  data[size] = (gchar*)0;

  return data;
}

void visuPluginsInit()
{
  gchar **plugins;
  gboolean res;
  int i;

  presentPlugins = (GList*)0;
  if (g_module_supported())
    {
      plugins = visuPluginsGet_installedPlugins();
      for (i = 0; plugins[i]; i++)
	{
	  DBG_fprintf(stderr, "Visu Plugins : try to load the plugin '%s'.\n",
		      plugins[i]);
	  res = visuPluginLoad_byPath(plugins[i]);
	  if (res)
	    ((VisuPlugin*)presentPlugins->data)->init();
	  else
	    g_warning("failure loading plugin '%s'!\n", plugins[i]);
	}
      g_strfreev(plugins);
    }
}
