/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_pickMesure.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include "opengl.h"
#include "visu_tools.h"
#include "visu_data.h"
#include "visu_object.h"
#include "visu_extension.h"
#include "visu_rendering.h"
#include "openGLFunctions/text.h"
#include "openGLFunctions/objectList.h"
#include "renderingBackend/visu_windowInterface.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <glib.h>

int openGlListMarksId;

#define PICK_MESURE_MARK_BIG_SQUARE_SIZE 8
#define PICK_MESURE_MARK_SMALL_SQUARE_SIZE 4
guchar pickMesureMark_bigSquare[PICK_MESURE_MARK_BIG_SQUARE_SIZE *
				PICK_MESURE_MARK_BIG_SQUARE_SIZE * 4];
guchar pickMesureMark_smallSquare[PICK_MESURE_MARK_SMALL_SQUARE_SIZE *
				  PICK_MESURE_MARK_SMALL_SQUARE_SIZE * 4];

typedef enum
  {
    PICK_MESURE_MARK_BIG_SQUARE,
    PICK_MESURE_MARK_SMALL_SQUARE,
    PICK_MESURE_MARK_HIGHLIGHT,
    PICK_MESURE_END_MARK_DOT,
    PICK_MESURE_MARK_DISTANCE,
    PICK_MESURE_END_MARK_LINK
  } PickMesureMarkType;

struct MarkInfo_struct
{
  /* Mark type. */
  PickMesureMarkType type;
  /* Id used to address the VisuNode when the
     pointer node1 has changed, for example when a new VisuData
     is loaded with the same geometry and we want to apply this mark. */
  int idNode1;

  /* Idem for a second node. */
  int idNode2;

  /* Text not used. */
  gchar *text;
};
/**
 * _PickMesure:
 * @selected: a #VisuNode corresponding to the normal selected node (if set) ;
 * @ref1: a #VisuNode corresponding to reference 1 (if set) ;
 * @ref2: a #VisuNode corresponding to reference 2 (if set) ;
 * @info: a string containing the formated pick informations ;
 * @error: a string containing an error message (if any) ;
 * @newValuesAvailable: a flag to set if the pick action has canged anything ;
 * @formatFlag: if set, the errors and info GString are used and strings
 *              output for the pick action are created.
 *
 * This structure reflects the state of a current pick session used for
 * distance and angles measurements.
 */
struct _PickMesure
{
  VisuData *data;

  int idSelected;
  int idRef1;
  int idRef2;
  GList *idRegion;

  gboolean formatFlag;
  GString *info;
  GString *errors;

  gboolean newValuesAvailable;
  PickMesureType typeOfNews;

  float drag[3];

  /* A list of MarkInfo_struct elements. */
  GList *storedMarks;
  gboolean storeDistance;
};

/* Local methods. */
static void pickMesureFree(gpointer data);
static void pickMesureRebuild_classicalList(VisuData *dataObj);
static void pickMesureRebuild_colourInvList(VisuData *dataObj);

/**************************/
/* Local drawing methods. */
/**************************/
/* OpenGL routines. */
static void drawMarkList(VisuData *data, GList *list, int listType);
static void drawMarkDot(VisuData *data, VisuNode *node, PickMesureMarkType type);
static void putMarkDot(VisuData *data, int nodeId, PickMesureMarkType type);
static void drawMarkDistance(VisuData *data, VisuNode *nodeRef,
			     VisuNode *node, PickMesureMarkType type);
static void putMarkDistance(VisuData *data, int nodeRefId,
			    int nodeId, PickMesureMarkType type);
/* Manage mark lists. */
typedef gboolean (*MatchFunc)(struct MarkInfo_struct *markInfo,
			      int nodeId, gpointer data);
static void addMarkDotToList(PickMesure *mesureData, int idNode,
			     PickMesureMarkType type);
static void toggleMarkDistanceInList(PickMesure *mesureData, int nodeRefId,
				     int nodeId, PickMesureMarkType type);
static GList* lookupMarkInList(PickMesure *mesureData, MatchFunc match,
			       int nodeId, gpointer data);
static void removeMark(PickMesure *mesureData, GList *list);
static void removeMarkFromList(PickMesure *mesureData, MatchFunc match,
			       int nodeId, gpointer data);


/* Local callbacks. */
static void createPickMesureOnNewData(GObject *visu, VisuData *dataObj,
				      gpointer data);
static void updateListOnNodeChange(VisuData *dataObj, gpointer data);
static void updateListOnElementChange(VisuData *dataObj, VisuElement *ele,
				      gpointer data);
static void updateListOnPopulationChange(VisuData *dataObj, int *nodes,
					 gpointer data);


void initPick_module()
{
  OpenGLExtension *ext1, *ext2;
  int i;

  /* Get an id for the OpenGL list used to draw marks. */
  openGlListMarksId = openGLObjectList_new(2);
  /* Create the marks. */
  for (i = 0; i < PICK_MESURE_MARK_BIG_SQUARE_SIZE *
	 PICK_MESURE_MARK_BIG_SQUARE_SIZE * 4 ; i++)
    pickMesureMark_bigSquare[i] = 0xff;
  for (i = 0; i < PICK_MESURE_MARK_SMALL_SQUARE_SIZE *
	 PICK_MESURE_MARK_SMALL_SQUARE_SIZE * 4; i++)
    pickMesureMark_smallSquare[i] = 0xff;

  /* We listen to the dataReadyForRendering signal to create a pickMesure
     object and attach it to an object when created. */
  g_signal_connect(VISU_INSTANCE, "dataNew",
		   G_CALLBACK(createPickMesureOnNewData), (gpointer)0);

  /* Initialize an extension to draw the marks. */
  ext1 = OpenGLExtension_new("MarksInv", _("Marks - inverse color"),
			     _("Draw some marks on element in video inverse."),
			     openGlListMarksId, pickMesureRebuild_colourInvList);
  OpenGLExtensionRegister(ext1);
  OpenGLExtensionSet_priority(ext1, OPENGL_EXTENSION_PRIORITY_LAST);
  OpenGLExtensionSet_saveOpenGLState(ext1, TRUE);
  ext1->used = 1;

  ext2 = OpenGLExtension_new("Marks", _("Marks - classical"),
			     _("Draw some marks on element."),
			     openGlListMarksId + 1, pickMesureRebuild_classicalList);
  OpenGLExtensionRegister(ext2);
  OpenGLExtensionSet_priority(ext2, OPENGL_EXTENSION_PRIORITY_LOW);
/*   OpenGLExtensionSet_saveOpenGLState(ext2, TRUE); */
  ext2->used = 1;
}

static void createPickMesureOnNewData(GObject *visu _U_, VisuData *dataObj,
				      gpointer data _U_)
{
  PickMesure *mesureData;

  if (!dataObj)
    return;

  /* By default, we need no extended output but we
     draw measurements on screen. */
  mesureData                     = g_malloc(sizeof(PickMesure));
  mesureData->data               = dataObj;
  mesureData->idRef1             = -99;
  mesureData->idRef2             = -99;
  mesureData->idSelected         = -99;
  mesureData->idRegion           = (GList*)0;
  mesureData->info               = (GString*)0;
  mesureData->errors             = (GString*)0;
  mesureData->newValuesAvailable = FALSE;
  mesureData->formatFlag         = FALSE;
  mesureData->storedMarks        = (GList*)0;
  mesureData->storeDistance      = TRUE;
  DBG_fprintf(stderr, "Visu PickMesure: create a new object, %p.\n",
	      (gpointer)mesureData);

  if (mesureData->storeDistance)
    openGLText_initFontList();

  /* Connect a signal on file change to remove everything. */
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationDecrease",
		   G_CALLBACK(updateListOnPopulationChange), (gpointer)mesureData);
  g_signal_connect(G_OBJECT(dataObj), "NodePositionChanged",
		   G_CALLBACK(updateListOnNodeChange), (gpointer)mesureData);
  g_signal_connect(G_OBJECT(dataObj), "NodeRenderedChanged",
		   G_CALLBACK(updateListOnNodeChange), (gpointer)mesureData);
  g_signal_connect(G_OBJECT(dataObj), "ElementRenderedChanged",
		   G_CALLBACK(updateListOnElementChange), (gpointer)mesureData);
  
  visuDataSet_propertyWithDestroyFunc(dataObj, "pickMesure_data",
				      (gpointer)mesureData, pickMesureFree);

  /* Empty the OpenGL list. */
  glDeleteLists(openGlListMarksId, 2);
}

static void pickMesureFree(gpointer data)
{
  PickMesure *mesureData;
  GList *list;

  DBG_fprintf(stderr, "Visu PickMesure: freeing object %p.\n", data);
  mesureData = (PickMesure*)data;

  if (mesureData->idRegion)
    g_list_free(mesureData->idRegion);
  if (mesureData->info)
    g_string_free(mesureData->info, TRUE);
  if (mesureData->errors)
    g_string_free(mesureData->errors, TRUE);
  list = mesureData->storedMarks;
  while (list)
    {
      g_free(list->data);
      list = g_list_next(list);
    }
  g_list_free(mesureData->storedMarks);
  g_free(mesureData);
}




/****************************/
/* OpenGL drawing routines. */
/****************************/
/* This method recreate the opengl list associated to marks with
   all the marks given as argument. */
static void drawMarkList(VisuData *data, GList *list, int listType)
{
  struct MarkInfo_struct *mark;
  VisuNode *node1;
  GList *tmpLst;

  DBG_fprintf(stderr, "Visu pickMesure: update the list %p"
	      " of all marks.\n", (gpointer)list);
  if (listType == 0 || listType < 0)
    glDeleteLists(openGlListMarksId, 1);
  if (listType == 1 || listType < 0)
    glDeleteLists(openGlListMarksId + 1, 1);
  if (!list)
    return;
  tmpLst = list;
  if (listType == 0 || listType < 0)
    {
      /* Draw the colour inverse list. */
      glNewList(openGlListMarksId, GL_COMPILE);
      glDisable(GL_DEPTH_TEST);
      glDisable(GL_LIGHTING);
      glDisable(GL_FOG);
      /*   glDisable(GL_DITHER); */
      glEnable(GL_BLEND);
      glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR);
      while(list)
	{
	  mark = (struct MarkInfo_struct*)list->data;
	  if (mark->type != PICK_MESURE_MARK_HIGHLIGHT)
	    {
	      DBG_fprintf(stderr, " | draw mark of type %d.\n", mark->type);
	      node1 = visuDataGet_nodeFromNumber(data, mark->idNode1);
	      if (mark->type < PICK_MESURE_END_MARK_DOT)
		drawMarkDot(data, node1, mark->type);
	      else if (mark->type > PICK_MESURE_END_MARK_DOT &&
		       mark->type < PICK_MESURE_END_MARK_LINK)
		drawMarkDistance(data, node1,
				 visuDataGet_nodeFromNumber(data, mark->idNode2),
				 mark->type);
	    }
	  list = g_list_next(list);
	}
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glEndList();
    }
  if (listType == 1 || listType < 0)
    {
      /* Draw the classical list. */
      glNewList(openGlListMarksId + 1, GL_COMPILE);
      list = tmpLst;
      while(list)
	{
	  mark = (struct MarkInfo_struct*)list->data;
	  if (mark->type == PICK_MESURE_MARK_HIGHLIGHT)
	    {
	      DBG_fprintf(stderr, " | draw mark of type %d.\n", mark->type);
	      node1 = visuDataGet_nodeFromNumber(data, mark->idNode1);
	      drawMarkDot(data, node1, mark->type);
	    }
	  list = g_list_next(list);
	}
      glEndList();
    }
}
static void drawMarkDot(VisuData *data, VisuNode *node, PickMesureMarkType type)
{
  float xyz[3];
  OpenGLView *view;
  VisuElement *ele;
  int nlat;
  float eleSize, rgba[4], material[5];
  RenderingMethod *method;
  GLUquadricObj *obj;

  /* If the node is masked, then we don't draw the dot. */
  if (!node->rendered)
    return;
  /* If one of the element is masked, then we don't draw the distance. */
  ele = data->fromIntToVisuElement[node->posElement];
  if (!ele->rendered)
    return;

  DBG_fprintf(stderr, "Visu PickMesure: draw a mark (%d) on node %d.\n",
	      (int)type, node->number);

  visuDataGet_nodePosition(data, node, xyz);
  view = visuDataGet_openGLView(data);
  switch (type)
    {
    case PICK_MESURE_MARK_BIG_SQUARE:
      glRasterPos3f(xyz[0] - view->box->dxxs2,
		    xyz[1] - view->box->dyys2,
		    xyz[2] - view->box->dzzs2);
      glDrawPixels(PICK_MESURE_MARK_BIG_SQUARE_SIZE,
		   PICK_MESURE_MARK_BIG_SQUARE_SIZE,
		   GL_RGBA, GL_UNSIGNED_BYTE, pickMesureMark_bigSquare);
      break;
    case PICK_MESURE_MARK_SMALL_SQUARE:
      glRasterPos3f(xyz[0] - view->box->dxxs2,
		    xyz[1] - view->box->dyys2,
		    xyz[2] - view->box->dzzs2);
      glDrawPixels(PICK_MESURE_MARK_SMALL_SQUARE_SIZE,
		   PICK_MESURE_MARK_SMALL_SQUARE_SIZE,
		   GL_RGBA, GL_UNSIGNED_BYTE, pickMesureMark_smallSquare);
      break;
    case PICK_MESURE_MARK_HIGHLIGHT:
      obj = gluNewQuadric();
      method = getRenderingMethodInUse();
      eleSize = renderingMethodGet_sizeOfElement(method, ele);
      rgba[0] = 1. - ele->rgb[0];
      rgba[1] = 1. - ele->rgb[1];
      rgba[2] = 1. - ele->rgb[2];
      rgba[3] = 0.5;
      material[material_amb] = 1.;
      setOpenGlMaterial(material, rgba);
      nlat = OpenGLViewGet_numberOfFacettes(view, eleSize * 1.25);
      glPushMatrix();
      glTranslated(xyz[0] - view->box->dxxs2,
		   xyz[1] - view->box->dyys2,
		   xyz[2] - view->box->dzzs2);
      gluSphere(obj, (double)eleSize * 1.25, 2 * nlat, 2 * nlat);
      glPopMatrix();
      break;
    default:
      break;
    }
}
/* Directly draw the dot on the front buffer. */
static void putMarkDot(VisuData *data, int nodeId, PickMesureMarkType type)
{
  VisuElement *ele;
  VisuNode *node;

  node = visuDataGet_nodeFromNumber(data, nodeId);
  if (!node)
    return;
  /* If the node is masked, then we don't draw the dot. */
  if (!node->rendered)
    return;
  /* If one of the element is masked, then we don't draw the distance. */
  ele = data->fromIntToVisuElement[node->posElement];
  if (!ele->rendered)
    return;

  DBG_fprintf(stderr, "Visu pickMesure: draw directly a mark on front buffer.\n");
  glDrawBuffer(GL_FRONT);

  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_LIGHTING);
  glDisable(GL_FOG);
  glEnable(GL_BLEND);
  glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR);
  drawMarkDot(data, node, type);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glPopAttrib();

  glDrawBuffer(GL_BACK);

  glFlush();
}
static void drawMarkDistance(VisuData *data, VisuNode *nodeRef,
			     VisuNode *node, PickMesureMarkType type _U_)
{
  float xyzRef[3], xyz[3], dist;
  int i;
  char distStr[8];
  OpenGLView *view;
  VisuElement *ele;

  /* If one of the node is masked, then we don't draw the distance. */
  if (!nodeRef->rendered || !node->rendered)
    return;

  /* If one of the element is masked, then we don't draw the distance. */
  ele = data->fromIntToVisuElement[nodeRef->posElement];
  if (!ele->rendered)
    return;
  ele = data->fromIntToVisuElement[node->posElement];
  if (!ele->rendered)
    return;

  visuDataGet_nodePosition(data, nodeRef, xyzRef);
  visuDataGet_nodePosition(data, node, xyz);
  view = visuDataGet_openGLView(data);

  dist = 0.;
  for (i = 0; i < 3; i++)
    dist += (xyzRef[i] - xyz[i]) * (xyzRef[i] - xyz[i]);
  sprintf(distStr, "%7.3f", sqrt(dist));
  distStr[7] = '\0';
  DBG_fprintf(stderr, "Visu pickMesure: draw a link with distance %s.\n", distStr);

  glPushMatrix();
  glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);
  glLineWidth(1);
  glColor4f(1., 1., 1., 0.);
  glBegin(GL_LINES);
  glVertex3fv(xyzRef);
  glVertex3fv(xyz);
  glEnd();
  glPointSize(8.);
  glBegin(GL_POINTS);
  glVertex3fv(xyzRef);
  glVertex3fv(xyz);
  glEnd();
  glRasterPos3f((xyzRef[0] + xyz[0])/2., (xyzRef[1] + xyz[1])/2, (xyzRef[2] + xyz[2])/2.); 
  openGLText_drawChars(distStr); 
/*   fprintf(stderr, "----> '%s'\n", distStr); */
  glPopMatrix();
  
/*   drawMarkDot(data, node, PICK_MESURE_MARK_SMALL_SQUARE); */
}
/* Directly draw the distance on the front buffer. */
static void putMarkDistance(VisuData *data, int nodeRefId,
			    int nodeId, PickMesureMarkType type)
{
  VisuElement *ele;
  VisuNode *nodeRef, *node;

  node    = visuDataGet_nodeFromNumber(data, nodeId);
  nodeRef = visuDataGet_nodeFromNumber(data, nodeRefId);
  if (!node || !nodeRef)
    return;
  /* If one of the node is masked, then we don't draw the distance. */
  if (!nodeRef->rendered || !node->rendered)
    return;
  /* If one of the element is masked, then we don't draw the distance. */
  ele = data->fromIntToVisuElement[nodeRef->posElement];
  if (!ele->rendered)
    return;
  ele = data->fromIntToVisuElement[node->posElement];
  if (!ele->rendered)
    return;

  DBG_fprintf(stderr, "Visu pickMesure: draw directly a distance on front buffer.\n");
  glDrawBuffer(GL_FRONT);

  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_LIGHTING);
  glDisable(GL_FOG);
  glEnable(GL_BLEND);
  glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR);
  drawMarkDistance(data, nodeRef, node, type);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glPopAttrib();

  glDrawBuffer(GL_BACK);

  glFlush();
}


/**********************/
/* Manage mark lists. */
/**********************/
/* static gboolean matchDot(struct MarkInfo_struct *markInfo, int nodeId, */
/* 			 gpointer data _U_) */
/* { */
/*   return (markInfo->idNode1 == nodeId && */
/* 	  markInfo->type < PICK_MESURE_END_MARK_DOT); */
/* } */
/* static gboolean matchType(struct MarkInfo_struct *markInfo, */
/* 			  int nodeId _U_, gpointer data) */
/* { */
/*   return (markInfo->type ==(unsigned int)GPOINTER_TO_INT(data)); */
/* } */
static gboolean matchDotType(struct MarkInfo_struct *markInfo,
			     int nodeId, gpointer data)
{
  return (markInfo->idNode1 == nodeId &&
	  markInfo->type == (unsigned int)GPOINTER_TO_INT(data));
}
/* static gboolean matchDistance(struct MarkInfo_struct *markInfo, */
/* 			      int nodeId, gpointer data _U_) */
/* { */
/*   return ((markInfo->idNode1 == nodeId || markInfo->idNode2 == nodeId) && */
/* 	  (markInfo->type > PICK_MESURE_END_MARK_DOT && */
/* 	   markInfo->type < PICK_MESURE_END_MARK_LINK)); */
/* } */
static gboolean matchAny(struct MarkInfo_struct *markInfo, int nodeId,
			 gpointer data _U_)
{
  return (markInfo->idNode1 == nodeId || markInfo->idNode2 == nodeId);
}
static void addMarkDotToList(PickMesure *mesureData, int idNode,
			     PickMesureMarkType type)
{
  struct MarkInfo_struct *mark;

  g_return_if_fail(type < PICK_MESURE_END_MARK_DOT && idNode >= 0 && mesureData);

  mark          = g_malloc(sizeof(struct MarkInfo_struct));
  mark->type    = type;
  mark->idNode1 = idNode;
  mark->idNode2 = -1;
  mesureData->storedMarks = g_list_append(mesureData->storedMarks, (gpointer)mark);
}
static GList* lookupMarkInList(PickMesure *mesureData, MatchFunc match,
			       int nodeId, gpointer data)
{
  GList *list, *rtList;

  g_return_val_if_fail(mesureData, (GList*)0);

  list = mesureData->storedMarks;
  rtList = (GList*)0;
  while (list)
    {
      if (match((struct MarkInfo_struct*)list->data, nodeId, data))
	rtList = g_list_append(rtList, list);
      list = g_list_next(list);
    }
  return rtList;
}
static void removeMark(PickMesure *mesureData, GList *list)
{
  g_return_if_fail(mesureData && list);

  DBG_fprintf(stderr, "Visu PickMesure: remove mark of type %d, node %d/%d.\n",
	      ((struct MarkInfo_struct*)list->data)->type,
	      ((struct MarkInfo_struct*)list->data)->idNode1,
	      ((struct MarkInfo_struct*)list->data)->idNode2);
  g_free(list->data);
  if (list->prev)
    list->prev->next = list->next;
  if (list->next)
    list->next->prev = list->prev;
  if (list == mesureData->storedMarks)
    mesureData->storedMarks = list->next;
  g_list_free_1(list);
}
static void removeMarkFromList(PickMesure *mesureData, MatchFunc match,
			       int nodeId, gpointer data)
{
  GList *list, *rmList;

  g_return_if_fail(mesureData);

  rmList = lookupMarkInList(mesureData, match, nodeId, data);
  list = rmList;
  while(list)
    {
      removeMark(mesureData, (GList*)list->data);
      list = g_list_next(list);
    }
  g_list_free(rmList);
}
/* Method that add/remove a distance mark to the list of drawn marks. */
static void toggleMarkDistanceInList(PickMesure *mesureData, int nodeRefId,
				     int nodeId, PickMesureMarkType type)
{
  struct MarkInfo_struct *mark;
  GList *tmpLst;

  g_return_if_fail(type > PICK_MESURE_END_MARK_DOT &&
		   type < PICK_MESURE_END_MARK_LINK &&
		   nodeRefId >= 0 && nodeId >= 0 && mesureData);

  /* Look for the mark. */
  tmpLst = mesureData->storedMarks;
  while (tmpLst)
    {
      mark = (struct MarkInfo_struct*)tmpLst->data;
      if (mark->type == type && mark->idNode1 == nodeRefId && mark->idNode2 == nodeId)
	{
	  DBG_fprintf(stderr, "Visu PickMesure: remove a distance mark.\n");
	  removeMark(mesureData, tmpLst);
	  return;
	}
      tmpLst = g_list_next(tmpLst);
    }
  /* Found none, create a new one. */
  mark          = g_malloc(sizeof(struct MarkInfo_struct));
  mark->type    = type;
  mark->idNode1 = nodeRefId;
  mark->idNode2 = nodeId;
  DBG_fprintf(stderr, "Visu PickMesure: add a distance mark.\n");
  mesureData->storedMarks = g_list_append(mesureData->storedMarks, (gpointer)mark);
}
/* Method that removes all distance marks from the list of drawn marks. */
gboolean pickMesureRemove_allDistanceMarks(PickMesure *mesureData)
{
  GList *list, *rmList;

  g_return_val_if_fail(mesureData, FALSE);

  list = mesureData->storedMarks;
  rmList = (GList*)0;
  while (list)
    {
      if (((struct MarkInfo_struct*)list->data)->type > PICK_MESURE_END_MARK_DOT &&
	  ((struct MarkInfo_struct*)list->data)->type < PICK_MESURE_END_MARK_LINK)
	rmList = g_list_append(rmList, list);
      list = g_list_next(list);
    }
  if (!rmList)
    return FALSE;

  list = rmList;
  while (list)
    {
      removeMark(mesureData, (GList*)list->data);
      list = g_list_next(list);
    }  
  g_list_free(rmList);
  drawMarkList(mesureData->data, mesureData->storedMarks, 0);
  return TRUE;
}
gboolean pickMesureRemove_allMarks(PickMesure *mesureData)
{
  gboolean redraw;

  g_return_val_if_fail(mesureData, FALSE);

  if (mesureData->idRef1 >= 0)
    removeMarkFromList(mesureData, matchDotType, mesureData->idRef1,
		       GINT_TO_POINTER(PICK_MESURE_MARK_BIG_SQUARE));
  if (mesureData->idRef2 >= 0)
    removeMarkFromList(mesureData, matchDotType, mesureData->idRef2,
		       GINT_TO_POINTER(PICK_MESURE_MARK_SMALL_SQUARE));
  mesureData->idRef1 = -1;
  mesureData->idRef2 = -1;
  mesureData->idSelected = -1;
  redraw = pickMesureRemove_allDistanceMarks(mesureData);
  
  return redraw;
}
gboolean pickMesureRemove_allHighlights(PickMesure *mesureData)
{
  GList *list, *rmList;

  g_return_val_if_fail(mesureData, FALSE);

  list = mesureData->storedMarks;
  rmList = (GList*)0;
  while (list)
    {
      if (((struct MarkInfo_struct*)list->data)->type == PICK_MESURE_MARK_HIGHLIGHT)
	rmList = g_list_append(rmList, list);
      list = g_list_next(list);
    }
  if (!rmList)
    return FALSE;

  list = rmList;
  while (list)
    {
      removeMark(mesureData, (GList*)list->data);
      list = g_list_next(list);
    }  
  g_list_free(rmList);
  drawMarkList(mesureData->data, mesureData->storedMarks, 1);
  return TRUE;
}


void pickMesureSet_highlight(PickMesure *mesureData, int idNode, gboolean status)
{
  GList *mark;

  DBG_fprintf(stderr, "Visu PickMesure: set an highlight (%d) for the node %d.\n",
	      status, idNode);
  
  mark = lookupMarkInList(mesureData, matchDotType, idNode,
			  GINT_TO_POINTER(PICK_MESURE_MARK_HIGHLIGHT));

  if (mark && !status)
    {
      g_return_if_fail(!mark->next);
      removeMark(mesureData, (GList*)mark->data);
    }
  if (!mark && status)
    addMarkDotToList(mesureData, idNode, PICK_MESURE_MARK_HIGHLIGHT);

  drawMarkList(mesureData->data, mesureData->storedMarks, 1);
}

void pickMesureToggle_highlight(PickMesure *mesureData, GList *idNodes)
{
  GList *mark;

  g_return_if_fail(mesureData);

  while (idNodes)
    {
      DBG_fprintf(stderr, "Visu PickMesure: toggle an highlight for the node %d.\n",
		  GPOINTER_TO_INT(idNodes->data));
      mark = lookupMarkInList(mesureData, matchDotType,
			      GPOINTER_TO_INT(idNodes->data),
			      GINT_TO_POINTER(PICK_MESURE_MARK_HIGHLIGHT));

      if (mark)
	{
	  g_return_if_fail(!mark->next);
	  removeMark(mesureData, (GList*)mark->data);
	}
      else
	addMarkDotToList(mesureData, GPOINTER_TO_INT(idNodes->data),
			 PICK_MESURE_MARK_HIGHLIGHT);
      idNodes = g_list_next(idNodes);
    }

  drawMarkList(mesureData->data, mesureData->storedMarks, 1);
}

void pickMesureSet_pickNode(PickMesure *mesureData, int nodeId, PickMesureType type)
{
  double dx, dy, dz, dr, dx1, dy1, dz1, dr1,  dx2, dy2, dz2, dr2;
  double ang;
  float posRef1[3], posRef2[3], posSelect[3];
  VisuNode *ref1, *ref2, *selected;

  g_return_if_fail(mesureData && (type == PICK_SELECTED ||
				  type == PICK_REFERENCE_1 ||
				  type == PICK_REFERENCE_2));

  /* No changes. */
  if (/* (type == PICK_SELECTED && nodeId == mesureData->idSelected) || */
      (type == PICK_REFERENCE_1 && nodeId == mesureData->idRef1) ||
      (type == PICK_REFERENCE_2 && nodeId == mesureData->idRef2))
    return;

  /* Prepare the values for the strings. */
  mesureData->newValuesAvailable = TRUE;
  mesureData->typeOfNews = type;
  if (mesureData->formatFlag)
    {
      if (!mesureData->info)
	mesureData->info = g_string_new("");
      else
	mesureData->info = g_string_erase(mesureData->info, 0, -1);
      if (!mesureData->errors)
	mesureData->errors = g_string_new("");
      else
	mesureData->errors = g_string_erase(mesureData->errors, 0, -1);
    }
  /* Cancel a selected region, if any. */
  if (mesureData->idRegion)
    g_list_free(mesureData->idRegion);
  mesureData->idRegion = (GList*)0;

  /* Update idSelected, idRef1 and idRef2, depending on @type. */
  switch (type)
    {
    case (PICK_SELECTED):
      if (nodeId >= 0)
	{
	  mesureData->idSelected = nodeId;
	  if (mesureData->idRef1 >= 0 && mesureData->storeDistance)
	    {
	      toggleMarkDistanceInList(mesureData,
				       mesureData->idRef1, mesureData->idSelected,
				       PICK_MESURE_MARK_DISTANCE);
	      putMarkDistance(mesureData->data,
			      mesureData->idRef1, mesureData->idSelected,
			      PICK_MESURE_MARK_DISTANCE);
	      drawMarkList(mesureData->data, mesureData->storedMarks, 0);
	    }
	}
      else
	{
	  if (mesureData->formatFlag)
	    mesureData->errors = g_string_append(mesureData->errors,
						 _("You picked nothing.\n"));
	  mesureData->idSelected = -1;
	}
      break;
    case (PICK_REFERENCE_1):
      if (nodeId >= 0)
	{
	  if (mesureData->idRef2 == nodeId)
	    {
	      if (mesureData->formatFlag)
		mesureData->errors =
		  g_string_append(mesureData->errors,
				  _("1st reference node should be distinct\n"
				    "from 2nd one.\n"));
	    }
	  else
	    {
	      if (mesureData->idRef1 >= 0)
		{
		  /* Remove old one. */
		  removeMarkFromList(mesureData, matchDotType,
				     mesureData->idRef1,
				     GINT_TO_POINTER(PICK_MESURE_MARK_BIG_SQUARE));
		  putMarkDot(mesureData->data, mesureData->idRef1,
			     PICK_MESURE_MARK_BIG_SQUARE);
		}
	      mesureData->idRef1 = nodeId;
	      addMarkDotToList(mesureData, mesureData->idRef1,
			       PICK_MESURE_MARK_BIG_SQUARE);
	      putMarkDot(mesureData->data, mesureData->idRef1,
			 PICK_MESURE_MARK_BIG_SQUARE);
	    }
	}
      else
	{
	  if (mesureData->formatFlag)
	    mesureData->errors = g_string_append(mesureData->errors,
						 _("Unset reference 1.\n"));
	  if (mesureData->idRef1 >= 0)
	    {
	      removeMarkFromList(mesureData, matchDotType,
				 mesureData->idRef1,
				 GINT_TO_POINTER(PICK_MESURE_MARK_BIG_SQUARE));
	      putMarkDot(mesureData->data, mesureData->idRef1,
			 PICK_MESURE_MARK_BIG_SQUARE);
	    }
	  mesureData->idRef1 = -1;
	  if (mesureData->idRef2)
	    {
	      removeMarkFromList(mesureData, matchDotType,
				 mesureData->idRef2,
				 GINT_TO_POINTER(PICK_MESURE_MARK_SMALL_SQUARE));
	      putMarkDot(mesureData->data, mesureData->idRef1,
			 PICK_MESURE_MARK_SMALL_SQUARE);
	    }
	  mesureData->idRef2 = -1;
	}
      drawMarkList(mesureData->data, mesureData->storedMarks, 0);
      break;
    case (PICK_REFERENCE_2):
      if (mesureData->idRef2)
	{
	  removeMarkFromList(mesureData, matchDotType,
			     mesureData->idRef2,
			     GINT_TO_POINTER(PICK_MESURE_MARK_SMALL_SQUARE));
	  putMarkDot(mesureData->data, mesureData->idRef2,
		     PICK_MESURE_MARK_SMALL_SQUARE);
	}
      mesureData->idRef2 = -1;
      if (nodeId >= 0)
	{
	  if (mesureData->idRef1 < 0)
	    {
	      if (mesureData->formatFlag)
		mesureData->errors =
		  g_string_append(mesureData->errors,
				  _("Unable to set reference 2\n"
				    "without existing reference 1, use\n"
				    "middle button alone to set reference 1.\n"));
	    }
	  else if (mesureData->idRef1 == nodeId)
	    {
	      if (mesureData->formatFlag)
		mesureData->errors =
		  g_string_append(mesureData->errors,
				  _("2nd reference node should be distinct\n"
				    "from 1st one.\n"));
	    }
	  else
	    {
	      mesureData->idRef2 = nodeId;
	      addMarkDotToList(mesureData, mesureData->idRef2,
			       PICK_MESURE_MARK_SMALL_SQUARE);
	      putMarkDot(mesureData->data, mesureData->idRef2,
			 PICK_MESURE_MARK_SMALL_SQUARE);
	    }
	}
      else
	{
	  if (mesureData->formatFlag)
	    mesureData->errors =
	      g_string_append(mesureData->errors,
			      _("Unset reference 2.\n"));
	}
      drawMarkList(mesureData->data, mesureData->storedMarks, 0);
      break;
    default:
      g_error("Can't be here!");
    }

  if (!mesureData->formatFlag)
    return;

  /* Create a string with informations. */
  selected = (mesureData->idSelected < 0)?(VisuNode*)0:
    visuDataGet_nodeFromNumber(mesureData->data, mesureData->idSelected);
  ref1 = (mesureData->idRef1 < 0)?(VisuNode*)0:
    visuDataGet_nodeFromNumber(mesureData->data, mesureData->idRef1);
  ref2 = (mesureData->idRef2 < 0)?(VisuNode*)0:
    visuDataGet_nodeFromNumber(mesureData->data, mesureData->idRef2);
  if (ref1)
    {
      g_string_append_printf(mesureData->info,
			     _("Reference node #%d\n"
			       "  x = %7.3f\n"
			       "  y = %7.3f\n"
			       "  z = %7.3f\n"),
			     ref1->number + 1, ref1->xyz[0],
			     ref1->xyz[1],     ref1->xyz[2]);
    }
  if (ref2)
    {
      visuDataGet_nodePosition(mesureData->data, ref1, posRef1);
      visuDataGet_nodePosition(mesureData->data, ref2, posRef2);
      dx = posRef2[0] - posRef1[0];
      dy = posRef2[1] - posRef1[1];
      dz = posRef2[2] - posRef1[2];
      g_string_append_printf(mesureData->info,
			     _("2nd Reference node #%d\n"
			       "  x = %7.3f   dx = %7.3f\n"
			       "  y = %7.3f   dy = %7.3f\n"
			       "  z = %7.3f   dz = %7.3f\n"),
			     ref2->number + 1,
			     ref2->xyz[0], dx,
			     ref2->xyz[1], dy,
			     ref2->xyz[2], dz);
      dr = sqrt(dx*dx + dy*dy + dz*dz);
      if (selected)
	g_string_append_printf(mesureData->info,
			       _("           i.e. dr = %7.3f\n"), dr);
      else
	g_string_append_printf(mesureData->info,
			       _("           i.e. dr = %7.3f"), dr);
    }
  if (selected)
    {
      if (!ref2)
        {
	  if (ref1)
	    {
	      visuDataGet_nodePosition(mesureData->data, selected, posSelect);
	      visuDataGet_nodePosition(mesureData->data, ref1, posRef1);
	      dx = posSelect[0] - posRef1[0];
	      dy = posSelect[1] - posRef1[1];
	      dz = posSelect[2] - posRef1[2];
	      g_string_append_printf(mesureData->info,
				     _("Newly picked node #%d\n"
				       "  x = %7.3f   dx = %7.3f\n"
				       "  y = %7.3f   dy = %7.3f\n"
				       "  z = %7.3f   dz = %7.3f\n"),
				     selected->number + 1,
				     selected->xyz[0], dx,
				     selected->xyz[1], dy,
				     selected->xyz[2], dz);
	      dr = sqrt(dx*dx + dy*dy + dz*dz);
	      g_string_append_printf(mesureData->info,
				     _("           i.e. dr = %7.3f"), dr);
	    }
	  else
	    g_string_append_printf(mesureData->info,
				   _("Newly picked node #%d\n"
				     "  x = %7.3f\n"
				     "  y = %7.3f\n"
				     "  z = %7.3f\n"),
				   selected->number +1,
				   selected->xyz[0],
				   selected->xyz[1],
				   selected->xyz[2]
				   );
	}
      else
        {
	  visuDataGet_nodePosition(mesureData->data, selected, posSelect);
	  visuDataGet_nodePosition(mesureData->data, ref1, posRef1);
	  visuDataGet_nodePosition(mesureData->data, ref2, posRef2);
	  dx1 = posSelect[0] - posRef1[0];
	  dy1 = posSelect[1] - posRef1[1];
	  dz1 = posSelect[2] - posRef1[2];
	  dx2 = posRef2[0] - posRef1[0];
	  dy2 = posRef2[1] - posRef1[1];
	  dz2 = posRef2[2] - posRef1[2];
	  g_string_append_printf(mesureData->info,
				 _("Newly picked node #%d\n"
				   "  x = %7.3f   dx1 = %7.3f  dx2 = %7.3f\n"
				   "  y = %7.3f   dy1 = %7.3f  dx2 = %7.3f\n"
				   "  z = %7.3f   dz1 = %7.3f  dx2 = %7.3f\n"),
				 selected->number + 1,
				 selected->xyz[0], dx1, dx2,
				 selected->xyz[1], dy1, dy2,
				 selected->xyz[2], dz1, dz2);
	  dr1 = sqrt(dx1*dx1 + dy1*dy1 + dz1*dz1);
	  dr2 = sqrt(dx2*dx2 + dy2*dy2 + dz2*dz2);
	  ang = acos((dx2*dx1+dy2*dy1+dz2*dz1)/(dr2*dr1))/PI180;
	    g_string_append_printf(mesureData->info,
				   _("           i.e. dr = %7.3f  dr2 = %7.3f  \n"
				     "  (Ref-Ref2, Ref-New) = %5.2f degrees"),
				   dr1, dr2, ang);
        }
    }

  DBG_fprintf(stderr, "Pick mesure info: %s", mesureData->info->str);
  DBG_fprintf(stderr, "Pick mesure error: %s", mesureData->errors->str);
}
void pickMesureSet_pickRegion(PickMesure *mesureData, GList *nodeIds)
{
  g_return_if_fail(mesureData);

  /* Prepare the values for the strings. */
  mesureData->newValuesAvailable = TRUE;
  mesureData->typeOfNews = PICK_REGION;
  if (mesureData->formatFlag)
    {
      if (mesureData->info)
	g_string_free(mesureData->info, TRUE);
      if (mesureData->errors)
	g_string_free(mesureData->errors, TRUE);
      mesureData->info   = (GString*)0;
      mesureData->errors = (GString*)0;
    }

  /* Copy the region list. */
  if (mesureData->idRegion)
    g_list_free(mesureData->idRegion);
  mesureData->idRegion = g_list_copy(nodeIds);
}
void pickMesureSet_dragStart(PickMesure *mesureData, int nodeId)
{
  g_return_if_fail(mesureData);
  
  mesureData->newValuesAvailable = TRUE;
  mesureData->typeOfNews = PICK_DRAG_START;
  mesureData->idSelected = nodeId;
}
void pickMesureSet_dragStop(PickMesure *mesureData)
{
  g_return_if_fail(mesureData);
  
  mesureData->newValuesAvailable = TRUE;
  mesureData->typeOfNews = PICK_DRAG_STOP;
}
void pickMesureSet_dragMove(PickMesure *mesureData, float dx, float dy, float dz)
{
  g_return_if_fail(mesureData);
  
  mesureData->newValuesAvailable = TRUE;
  mesureData->typeOfNews = PICK_DRAG_MOVE;
  mesureData->drag[0] = dx;
  mesureData->drag[1] = dy;
  mesureData->drag[2] = dz;
}
float* pickMesureGet_drag(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, (float*)0);
  
  return (float*)mesureData->drag;
}
gboolean pickMesureGet_newsAvailable(PickMesure *mesureData, PickMesureType *type)
{
  g_return_val_if_fail(mesureData, FALSE);

  if (type)
    *type = mesureData->typeOfNews;
  return mesureData->newValuesAvailable;
}

gchar* pickMesureGet_infos(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, (gchar*)0);

  if (!mesureData->info || !mesureData->formatFlag)
    return (gchar*)0;
  return mesureData->info->str;
}
gchar* pickMesureGet_errors(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, (gchar*)0);

  if (!mesureData->errors || !mesureData->formatFlag)
    return (gchar*)0;
  return mesureData->errors->str;
}
VisuNode* pickMesureGet_selectedNode(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, (VisuNode*)0);

  if (mesureData->idSelected >= 0)
    return visuDataGet_nodeFromNumber(mesureData->data, mesureData->idSelected);
  else
    return (VisuNode*)0;
}
VisuNode* pickMesureGet_firstReference(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, (VisuNode*)0);

  if (mesureData->idRef1 >= 0)
    return visuDataGet_nodeFromNumber(mesureData->data, mesureData->idRef1);
  else
    return (VisuNode*)0;
}
GList* pickMesureGet_regionNodes(PickMesure *mesureData)
{
  GList *lst, *tmpLst;
  VisuNode *node;

  g_return_val_if_fail(mesureData, (GList*)0);
  
  lst = (GList*)0;
  tmpLst = mesureData->idRegion;
  while (tmpLst)
    {
      node = visuDataGet_nodeFromNumber(mesureData->data,
					GPOINTER_TO_INT(tmpLst->data));
      if (node)
	lst = g_list_prepend(lst, (gpointer)node);
      tmpLst = g_list_next(tmpLst);
    }
  return lst;
}
int pickMesureGet_regionNNodes(PickMesure *mesureData)
{
  g_return_val_if_fail(mesureData, -1);

  return g_list_length(mesureData->idRegion);
}


struct MarkInfo_struct* pickMesureCopyMark(struct MarkInfo_struct* mark)
{
  struct MarkInfo_struct *newMark;

  g_return_val_if_fail(mark, (struct MarkInfo_struct*)0);

  newMark = g_malloc(sizeof(struct MarkInfo_struct));
  newMark->type = mark->type;
  newMark->idNode1 = mark->idNode1;
  newMark->idNode2 = mark->idNode2;
  newMark->text = (gchar*)0;

  return newMark;
}

void pickMesureUpdate(VisuData *newData, VisuData *oldData)
{
  GList *list;
  PickMesure *newPick, *oldPick;
  struct MarkInfo_struct* mark, *newMark;
  gboolean remove;

  if (!newData || !oldData)
    return;

  DBG_fprintf(stderr, "Visu PickMesure : updating list of marks.\n");

  newPick = (PickMesure*)visuDataGet_property(newData, "pickMesure_data");
  oldPick = (PickMesure*)visuDataGet_property(oldData, "pickMesure_data");
  g_return_if_fail(newPick && oldPick);

  /* Try to match previous marks on new VisuData. */
  list = oldPick->storedMarks;
  while (list)
    {
      mark = (struct MarkInfo_struct*)list->data;
      DBG_fprintf(stderr, " | old mark %p of type %d.\n", (gpointer)mark, mark->type);
      remove = FALSE;
      /* We remove mark if one of the node can't be matched. */
      if (mark->type < PICK_MESURE_END_MARK_DOT)
	remove = remove || !visuDataGet_nodeFromNumber(newData, mark->idNode1);
      else if (mark->type > PICK_MESURE_END_MARK_DOT &&
	       mark->type < PICK_MESURE_END_MARK_LINK)
	remove = remove || !visuDataGet_nodeFromNumber(newData, mark->idNode1) ||
	  !visuDataGet_nodeFromNumber(newData, mark->idNode2);
      if (!remove && newPick != oldPick)
	{
	  /* We keep it, and add it in the list of newPick. */
	  newMark = pickMesureCopyMark(mark);
	  newPick->storedMarks = g_list_prepend(newPick->storedMarks, newMark);
	  DBG_fprintf(stderr, " | adding new mark %p of type %d.\n",
		      (gpointer)newMark, newMark->type);
	}
      if (remove && newPick == oldPick)
	{
	  /* We remove it, from newPick. */
	  DBG_fprintf(stderr, " | delete old mark %p of type %d.\n",
		      (gpointer)mark, mark->type);
	  newPick->storedMarks = g_list_delete_link(newPick->storedMarks, list);
	}
      list = g_list_next(list);
    }
  DBG_fprintf(stderr, " | New mark list is pointing to %p.\n",
	      (gpointer)newPick->storedMarks);
  if (oldPick->idRef1 >= 0)
    {
      DBG_fprintf(stderr, "Visu PickMesure : trying to match ref1 node (id: %d).\n",
		  oldPick->idRef1);
      if (visuDataGet_nodeFromNumber(newData, oldPick->idRef1))
	newPick->idRef1 = oldPick->idRef1;
    }
  if (oldPick->idRef2 >= 0)
    {
      DBG_fprintf(stderr, "Visu PickMesure : trying to match ref2 node (id: %d).\n",
	      oldPick->idRef2);
      if (visuDataGet_nodeFromNumber(newData, oldPick->idRef2))
	newPick->idRef2 = oldPick->idRef2;
    }
  if (oldPick->idSelected >= 0)
    {
      DBG_fprintf(stderr, "Visu PickMesure : trying to match selected node (id: %d).\n",
	      oldPick->idSelected);
      if (visuDataGet_nodeFromNumber(newData, oldPick->idSelected))
	newPick->idSelected = oldPick->idSelected;
    }
  newPick->storeDistance = oldPick->storeDistance;
  newPick->formatFlag = oldPick->formatFlag;
  drawMarkList(newPick->data, newPick->storedMarks, -1);
}
static void updateListOnPopulationChange(VisuData *dataObj, int *nodes,
					 gpointer data)
{
  int i;
  GList *list, *rmList;

  g_return_if_fail(data);

  DBG_fprintf(stderr, "Visu PickMesure: caught 'NodePopulationDecrease'"
	      " signal (%p).\n", data);
  
  /* Run through the mark list to get all nodeId and look into nodes
     to find if mark must be removed or not. */
  list = ((PickMesure*)data)->storedMarks;
  rmList = (GList*)0;
  while (list)
    {
      for (i = 0; nodes[i] >= 0; i++)
	if (matchAny((struct MarkInfo_struct*)list->data, nodes[i], (gpointer)0))
	  rmList = g_list_append(rmList, list);
      list = g_list_next(list);
    }

  if (!rmList)
    return;

  list = rmList;
  while (list)
    {
      removeMark((PickMesure*)data, (GList*)list->data);
      list = g_list_next(list);
    }  
  g_list_free(rmList);

  /* Remove also ref1, ref2 and selected from the PickMesure object. */
  for (i = 0; nodes[i] >= 0; i++)
    {
      if (((PickMesure*)data)->idRef1 == nodes[i])
	((PickMesure*)data)->idRef1 = -1;
      if (((PickMesure*)data)->idRef2 == nodes[i])
	((PickMesure*)data)->idRef2 = -1;
      if (((PickMesure*)data)->idSelected == nodes[i])
	((PickMesure*)data)->idSelected = -1;
    }

  drawMarkList(dataObj, ((PickMesure*)data)->storedMarks, -1);
}
static void updateListOnNodeChange(VisuData *dataObj, gpointer data)
{
  g_return_if_fail(data);

  DBG_fprintf(stderr, "Visu PickMesure: caught a node rendering"
	      " changing signal (%p).\n", data);
  drawMarkList(dataObj, ((PickMesure*)data)->storedMarks, -1);
}
static void updateListOnElementChange(VisuData *dataObj, VisuElement *ele,
				      gpointer data)
{
  g_return_if_fail(data);

  DBG_fprintf(stderr, "Visu PickMesure: caught an element rendering"
	      " changing signal (%s-%p).\n", ele->name, data);
  drawMarkList(dataObj, ((PickMesure*)data)->storedMarks, -1);
}
void pickMesureSet_storeDistance(PickMesure *mesureData, gboolean storeDistance)
{
  g_return_if_fail(mesureData);

  DBG_fprintf(stderr, "Visu PickMesure : set store distance to %d.\n", (int)storeDistance);
  mesureData->storeDistance = storeDistance;
  if (storeDistance)
    openGLText_initFontList();
}
void pickMesureSet_formatedOutput(PickMesure *mesureData, gboolean formatedOutput)
{
  g_return_if_fail(mesureData);

  DBG_fprintf(stderr, "Visu PickMesure : set formated output to %d.\n", (int)formatedOutput);
  mesureData->formatFlag = formatedOutput;
}

static void pickMesureRebuild_classicalList(VisuData *dataObj)
{
  PickMesure *pickMesure;

  pickMesure = visuDataGet_property(dataObj, "pickMesure_data");
  if (!pickMesure)
    return;

  DBG_fprintf(stderr, "Visu PickMesure: rebuilding classical object"
	      " list for PickMesure %p.\n", (gpointer)pickMesure);

  drawMarkList(dataObj, pickMesure->storedMarks, 1);
}
static void pickMesureRebuild_colourInvList(VisuData *dataObj)
{
  PickMesure *pickMesure;

  pickMesure = visuDataGet_property(dataObj, "pickMesure_data");
  if (!pickMesure)
    return;

  DBG_fprintf(stderr, "Visu PickMesure: rebuilding colour inverse object"
	      " list for PickMesure %p.\n", (gpointer)pickMesure);

  drawMarkList(dataObj, pickMesure->storedMarks, 0);
}
