/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_elements.h"
#include "visu_configFile.h"

#include <stdlib.h>
#include <string.h>

#include "openGLFunctions/objectList.h"
#include "opengl.h"
#include "coreTools/toolColor.h"
#include "coreTools/toolConfigFile.h"

#include <GL/gl.h>
#include <GL/glu.h>

/**
 * SECTION:visu_elements
 * @short_description: defines methods to create and acccess to
 * #VisuElement.
 *
 * <para>V_Sim is used to rendered at given position several object of
 * the same kind. The #VisuElement object is used to control that
 * kind. Typically, it corresponds to chemical element. It can
 * represent the silicon, the iron...</para>
 *
 * <para>#VisuElement are defined by their name and have some
 * characteristic like their color or if they are rendered or not. The
 * color characteristic is defined by an RGBA array and different
 * value for the behavior of the light, as defined in OpenGL :
 * diffusivity, shiningness, emissivity, specular and ambient. These
 * values can be controlled with the following methods :
 * visuElementSet_allColorValues(), visuElementSet_allRGBValues(),
 * visuElementSet_RGBValue(), visuElementSet_allMaterialValues() and
 * visuElementSet_materialValue().</para>
 *
 * <para>Moreover, the user can add new properties to one element
 * using visuElementSet_property() and visuElementGet_property(). The
 * values stored with these functions are not copied and should be
 * reachable until being remove or until the end of the
 * program.</para>
 *
 * <para>If the OpenGL representation of one element is not dependent
 * of its position, it is recommended to use the OpenGL list
 * associated to each #VisuElement that can be accessed by a call to
 * visuElementGet_identifierMaterial().</para>
 */

/**********************/
/* Private prototypes */
/**********************/

/* This is a pointer to the last added VisuElement. */
/* VisuElement *lastAddedVisuElement; */
static int identifierMaterials;

/* This hashtable contains a list of all different
   elements loaded in visu. ntype is _not_ the number of items
   in this hashtable. */
static GHashTable *differentVisuElements;
static GList *visuElement_list;

/* This int stores how many user functions have needs
   to recreate the nodes of an element when its material
   has been changed. */
static int flagCreateNodesAfterMaterialChange;


#define FLAG_ELEMENT_COLOR      "element_color"
#define DESC_ELEMENT_COLOR      "Codes the main color in RedGreenBlueAlpha format" \
  "and the light effects on material, nine floats between 0. and 1."
#define FLAG_ELEMENT_PROPERTIES "element_properties"
#define DESC_ELEMENT_PROPERTIES "Define some properties ; rendered (0 or 1) masked" \
  "(0 or 1)."

/* These functions detail how to read the RGB resource and
   the material resource introduced by this part.
   These routines are obsolete but kept for backward compatibility. */
static gboolean readMaterial(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error);
static gboolean readRendered(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error);

/* Read routines for the config file. */
static gboolean readElementColor(gchar **lines, int nbLines, int position,
				 VisuData *dataObj, GError **error);
static gboolean readElementProperties(gchar **lines, int nbLines, int position,
				      VisuData *dataObj, GError **error);


/* These functions write all the element list to export there associated resources. */
static gboolean exportResourcesRenderingBase(GString *data, int *nbLinesWritten,
					     VisuData *dataObj);

static void freeHashValue(gpointer data);



/* Add a VisuElement to the hashtable. */
int visuElementAdd(VisuElement* element)
{
  int nbElements;

  if (element &&
      !g_hash_table_lookup(differentVisuElements, (gpointer)element->name))
    {
      DBG_fprintf(stderr, "Visu Elements : adding %s (%d).\n", element->name,
		  GPOINTER_TO_INT(element));
      nbElements = g_hash_table_size(differentVisuElements);
      if (nbElements == NMAX_TP)
	{
	  g_warning("The system requires more than %d elements. This value is"
		    " set at compilation time with the variable NMAX_TP. If you"
		    " need more, ask the maintainer to raise this value.\n", NMAX_TP);
	  return 1;
	}
      element->typeNumber = g_hash_table_size(differentVisuElements);
      g_hash_table_insert(differentVisuElements, (gpointer)element->name, (gpointer)element);
      visuElement_list = g_list_append(visuElement_list, (gpointer)element);
/*       lastAddedVisuElement = element; */
    }
  return 0;
}

GList* visuElementGet_allElements()
{
  return visuElement_list;
}
VisuElement* visuElementGet_fromName(char* name)
{
  return g_hash_table_lookup(differentVisuElements, (gpointer)name);
}



/* It saves the values of rgb in the specified VisuElement. */
int visuElementSet_allColorValues(VisuElement* ele, float rgb[4], float material[5])
{
  int chgt;

  chgt = (ele->rgb[0] != rgb[0]) || (ele->rgb[1] != rgb[1]) ||
    (ele->rgb[2] != rgb[2]) || (ele->rgb[3] != rgb[3]);
  ele->rgb[0] = rgb[0];
  ele->rgb[1] = rgb[1];
  ele->rgb[2] = rgb[2];
  ele->rgb[3] = rgb[3];
  chgt = chgt || (ele->material[0] != material[0]) || (ele->material[1] != material[1]) ||
    (ele->material[2] != material[2]) || (ele->material[3] != material[3]) ||
    (ele->material[4] != material[4]);
  ele->material[0] = material[0];
  ele->material[1] = material[1];
  ele->material[2] = material[2];
  ele->material[3] = material[3];
  ele->material[4] = material[4];

  if (chgt)
    visuElement_createMaterial(ele);

  if (chgt && ele->rendered)
    return 2 * flagCreateNodesAfterMaterialChange - 1;
  else
    return 0;
}

int visuElementSet_allRGBValues(VisuElement* ele, float rgb[4])
{
  if (!ele)
    return 0;

  DBG_fprintf(stderr, "Visu Element : set color (%f, %f, %f, %f)\n",
	      ele->rgb[0], ele->rgb[1], ele->rgb[2], ele->rgb[3]);
  DBG_fprintf(stderr, "                      -> (%f, %f, %f, %f).\n",
	      rgb[0], rgb[1], rgb[2], rgb[3]);

  if (ele->rgb[0] == rgb[0] &&
      ele->rgb[1] == rgb[1] &&
      ele->rgb[2] == rgb[2] &&
      ele->rgb[3] == rgb[3])
    return 0;

  ele->rgb[0] = rgb[0];
  ele->rgb[1] = rgb[1];
  ele->rgb[2] = rgb[2];
  ele->rgb[3] = rgb[3];

  visuElement_createMaterial(ele);

  if (ele->rendered)
    return 2 * flagCreateNodesAfterMaterialChange - 1;
  else
    return 0;
}

/* It saves the specific value of rgb (0 for red, 1 for green and 2
   for bluein the specified VisuElement. */
int visuElementSet_RGBValue(VisuElement* ele, int rgb, float value)
{
  if (rgb < 0 || rgb > 3)
    return 0;

  if (value == ele->rgb[rgb])
    return 0;

  ele->rgb[rgb] = value;
  visuElement_createMaterial(ele);
  if (ele->rendered)
    return 2 * flagCreateNodesAfterMaterialChange - 1;
  else
    return 0;
}

/* It saves the values of material in the specified VisuElement. */
int visuElementSet_allMaterialValues(VisuElement* ele, float material[5])
{
  if (!ele)
    return 0;

  ele->material[0] = material[0];
  ele->material[1] = material[1];
  ele->material[2] = material[2];
  ele->material[3] = material[3];
  ele->material[4] = material[4];

  visuElement_createMaterial(ele);
  if (ele->rendered)
    return 2 * flagCreateNodesAfterMaterialChange - 1;
  else
    return 0;
}

/* It saves the specific value of material (use the enum as the parameter
   material) in the specified VisuElement. */
int visuElementSet_materialValue(VisuElement* ele, int material, float value)
{
  if (material < 0 || material > 4)
    return 0;

  if (value == ele->material[material])
    return 0;

  ele->material[material] = value;
  visuElement_createMaterial(ele);
  if (ele->rendered)
    return 2 * flagCreateNodesAfterMaterialChange - 1;
  else
    return 0;
}
int visuElementSet_sensitiveToPlanes(VisuElement *element, gboolean status)
{
  g_return_val_if_fail(element, 0);

  if (element->sensitiveToMaskingPlanes == status)
    return 0;

  element->sensitiveToMaskingPlanes = status;
  return 1;
}
gboolean visuElementGet_sensitiveToPlanes(VisuElement *element)
{
  g_return_val_if_fail(element, TRUE);
  
  return element->sensitiveToMaskingPlanes;
}
int visuElementSet_showNodeInfos(VisuElement *element, gboolean status)
{
  g_return_val_if_fail(element, 0);

  if (element->showNodeInfos == status)
    return 0;

  element->showNodeInfos = status;
  return 1;
}
gboolean visuElementGet_showNodeInfos(VisuElement *element)
{
  g_return_val_if_fail(element, FALSE);
  
  return element->showNodeInfos;
}
VisuElement *visuElementNew_withName(const char *key)
{
  VisuElement *ele;
  int i;

  ele = g_malloc(sizeof(VisuElement));
  ele->name = g_strdup(key);
  ele->typeNumber = -1;
  ele->materialIsUpToDate = FALSE;
  ele->rendered = TRUE;
  ele->sensitiveToMaskingPlanes = TRUE;
  ele->showNodeInfos = TRUE;

  ele->properties = g_hash_table_new_full(g_str_hash, g_str_equal,
					  NULL, g_free);
  g_return_val_if_fail(ele->properties, (VisuElement*)0);

  for (i = 0; i < 4; i++)
    ele->rgb[i] = 1.;

  for (i = 0; i < 5; i++)
    ele->material[i] = 0.25;
  DBG_fprintf(stderr, "Visu Elements : creating a new VisuElement '%s' -> %d.\n",
	      key, GPOINTER_TO_INT(ele));
  return ele;
}

void visuElement_createMaterial(VisuElement *ele)
{
  g_return_if_fail(ele);

  DBG_fprintf(stderr, "Visu Elements: creating material for %s (ele id %d - OpenGL id %d)\n", ele->name, ele->typeNumber, identifierMaterials + ele->typeNumber);
/*   fprintf(stderr, "%f %f %f - %f %f %f %f %f\n", ele->rgb[0], ele->rgb[1], ele->rgb[2], ele->material[0], ele->material[1], ele->material[2], ele->material[3], ele->material[4]); */

  glDeleteLists(identifierMaterials + ele->typeNumber, 1);
  glNewList(identifierMaterials + ele->typeNumber, GL_COMPILE);
  glColor4fv(ele->rgb);
  setOpenGlMaterial(ele->material, ele->rgb);
  glEndList();
  ele->materialIsUpToDate = TRUE;
}

/* return the OpenGL identifier of the specified VisuElement. */
int visuElementGet_identifierMaterial(VisuElement *ele)
{
  if (!ele)
    return 0;
/*   if (!ele->materialIsUpToDate) */
/*     visuElement_createMaterial(ele); */

  return identifierMaterials + ele->typeNumber;
}

void visuElementSet_updateNodesOnMaterialChange()
{
  flagCreateNodesAfterMaterialChange += 1;
  DBG_fprintf(stderr, "Visu Element : set flagCreateNodesAfterMaterialChange to %d.\n",
	      flagCreateNodesAfterMaterialChange);
}
void visuElementUnset_updateNodesOnMaterialChange()
{
  flagCreateNodesAfterMaterialChange -= 1;
  if (flagCreateNodesAfterMaterialChange < 0)
    flagCreateNodesAfterMaterialChange = 0;
  DBG_fprintf(stderr, "Visu Element : set flagCreateNodesAfterMaterialChange to %d.\n",
	      flagCreateNodesAfterMaterialChange);
}
gboolean visuElementGet_updateNodesOnMaterialChange()
{
  return (flagCreateNodesAfterMaterialChange > 0);
}

int initVisuElements()
{
  VisuConfigFileEntry *resourceEntry, *oldEntry;

  differentVisuElements =
    g_hash_table_new_full(g_str_hash, g_str_equal,
			  g_free, (GDestroyNotify)freeHashValue);
  g_return_val_if_fail(differentVisuElements, 0);

  visuElement_list = (GList*)0;
/*   lastAddedVisuElement = (VisuElement*)0; */

  /* Create a VisuModule to registered the new resources as
     rgb and material. */
  oldEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
				     "material",
				     "Obsolete entry for element_color",
				     1, readMaterial);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_ELEMENT_COLOR,
					  DESC_ELEMENT_COLOR,
					  1, readElementColor);
  visuConfigFileSet_version(resourceEntry, 3.4f);
  visuConfigFileSet_replace(resourceEntry, oldEntry);
  oldEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
				     "element_is_rendered",
				     "Obsolete entry included in element_properties",
				     1, readRendered);
  visuConfigFileSet_version(resourceEntry, 3.1f);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_ELEMENT_PROPERTIES,
					  DESC_ELEMENT_PROPERTIES,
					  1, readElementProperties);
  visuConfigFileSet_version(resourceEntry, 3.4f);
  visuConfigFileSet_replace(resourceEntry, oldEntry);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesRenderingBase);


  /* Get an OpenGL identifier to store all the materials. */
  identifierMaterials = openGLObjectList_new(NMAX_TP);

  flagCreateNodesAfterMaterialChange = 0;

  return 1;
}

/******************/
/* Resources part */
/******************/
static gboolean readElementColor(gchar **lines, int nbLines, int position,
				 VisuData *dataObj _U_, GError **error)
{
  VisuElement* ele;
  float all[9];

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_floatWithElement(lines[0], position, all, 9, &ele, error))
    return FALSE;
  ele->rgb[0] = CLAMP(all[0], 0., 1.);
  ele->rgb[1] = CLAMP(all[1], 0., 1.);
  ele->rgb[2] = CLAMP(all[2], 0., 1.);
  ele->rgb[3] = CLAMP(all[3], 0., 1.);
  ele->material[0] = CLAMP(all[4], 0., 1.);
  ele->material[1] = CLAMP(all[5], 0., 1.);
  ele->material[2] = CLAMP(all[6], 0., 1.);
  ele->material[3] = CLAMP(all[7], 0., 1.);
  ele->material[4] = CLAMP(all[8], 0., 1.);
  colorAdd_floatRGBA(ele->rgb, (int*)0);
  
  return TRUE;
}
static gboolean readElementProperties(gchar **lines, int nbLines, int position,
				 VisuData *dataObj _U_, GError **error)
{
  VisuElement* ele;
  float values[2];

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_floatWithElement(lines[0], position, values, 2, &ele, error))
    return FALSE;
  visuElementSet_rendered(ele, (gboolean)values[0]);
  visuElementSet_sensitiveToPlanes(ele, (gboolean)values[1]);
  
  return TRUE;
}

static gboolean readMaterial(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error)
{
  return readElementColor(lines, nbLines, position, dataObj, error);
}
static gboolean readRendered(gchar **lines, int nbLines, int position,
			     VisuData *dataObj _U_, GError **error)
{
  VisuElement* ele;
  float value;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_floatWithElement(lines[0], position, &value, 1, &ele, error))
    return FALSE;
  visuElementSet_rendered(ele, (gboolean)value);
  
  return TRUE;
}

/* These functions read all the element list to export there associated resources. */
gboolean exportResourcesRenderingBase(GString *data, int *nbLinesWritten,
				      VisuData *dataObj)
{
  GList *pos, *eleList;
  unsigned int i;

  g_string_append_printf(data, "# %s\n", DESC_ELEMENT_COLOR);
  *nbLinesWritten = 1;
  /* We create a list of elements, or get the whole list. */
  eleList = (GList*)0;
  if (dataObj)
    {
      for (i = 0; i < dataObj->ntype; i++)
	eleList = g_list_prepend(eleList, (gpointer)dataObj->fromIntToVisuElement[i]);
      pos = eleList;
    }
  else
    pos = visuElementGet_allElements();
  while(pos)
    {
      g_string_append_printf(data, "%s:\n", FLAG_ELEMENT_COLOR);
      g_string_append_printf(data, "    %s %4.3f %4.3f %4.3f %4.3f"
			     "   %4.2f %4.2f %4.2f %4.2f %4.2f\n",
			     ((VisuElement*)pos->data)->name,
			     ((VisuElement*)pos->data)->rgb[0],
			     ((VisuElement*)pos->data)->rgb[1],
			     ((VisuElement*)pos->data)->rgb[2],
			     ((VisuElement*)pos->data)->rgb[3],
			     ((VisuElement*)pos->data)->material[0],
			     ((VisuElement*)pos->data)->material[1],
			     ((VisuElement*)pos->data)->material[2],
			     ((VisuElement*)pos->data)->material[3],
			     ((VisuElement*)pos->data)->material[4]);
      *nbLinesWritten += 2;
      pos = g_list_next(pos);
    }
  g_string_append_printf(data, "# %s\n", DESC_ELEMENT_PROPERTIES);
  *nbLinesWritten = 1;
  if (dataObj)
    pos = eleList;
  else
    pos = visuElementGet_allElements();
  while(pos)
    {
      g_string_append_printf(data, "%s:\n", FLAG_ELEMENT_PROPERTIES);
      g_string_append_printf(data, "    %s %d %d\n",
			     ((VisuElement*)pos->data)->name,
			     ((VisuElement*)pos->data)->rendered,
			     ((VisuElement*)pos->data)->sensitiveToMaskingPlanes);
      *nbLinesWritten += 2;
      pos = g_list_next(pos);
    }
  g_string_append_printf(data, "\n");
  *nbLinesWritten += 1;
  if (eleList)
    g_list_free(eleList);

  return TRUE;
}

static void freeHashValue(gpointer data)
{
  g_free(((VisuElement*)data)->name);
  g_free((VisuElement*)data);
}


void visuElementSet_property(VisuElement* element, char* key, gpointer value)
{
  g_return_if_fail(element && key);

  DBG_fprintf(stderr, "Visu Element : set a new (key, value) to the element"
	      " '%s' : ('%s', %p).\n", element->name, key, value);
  g_hash_table_insert(element->properties, (gpointer)key, (gpointer)value);
}
gpointer visuElementGet_property(const VisuElement* element, char* key)
{
  g_return_val_if_fail(element && key, (gpointer)0);

  return g_hash_table_lookup(element->properties, (gpointer)key);
}
void visuElementRemove_property(VisuElement* element, char* key)
{
  g_return_if_fail(element && key);

  g_hash_table_remove(element->properties, (gpointer)key);
}

int visuElementSet_rendered(VisuElement *element, gboolean rendered)
{
  g_return_val_if_fail(element, 0);

  if (element->rendered == rendered)
    return 0;

  element->rendered = rendered;
  return 1;
}
gboolean visuElementGet_rendered(VisuElement *element)
{
  g_return_val_if_fail(element, FALSE);

  return element->rendered;
}


/* Debug : print all informations about the specified VisuElement. */
void debug_VisuElement(gpointer key, gpointer value, gpointer data _U_)
{
  VisuElement *ele;

  ele = (VisuElement*)value;
  fprintf(stderr, "Element '%s'\n", (gchar*)key);
  fprintf(stderr, "  | rgba [%f %f %f %f]\n", ele->rgb[0], ele->rgb[1], ele->rgb[2], ele->rgb[3]);
}

/* Debug : print all informations about all the stored VisuElements. */
void debug_AllVisuElements()
{
  fprintf(stderr,"DEBUG info : %d elements available.\n", g_hash_table_size(differentVisuElements));
  g_hash_table_foreach(differentVisuElements, (GHFunc)debug_VisuElement, (gpointer)0);
}

