/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "abinit.h"

#include <glib.h>

#include <config.h>
#include <ab_dtset_c.h>
#include <visu_tools.h>
#include <visu_basic.h>
#include <coreTools/toolMatrix.h>
#include <coreTools/toolElements.h>
#include <renderingMethods/renderingAtomic.h>

#define ABINIT_DESCRIPTION _("<span size=\"smaller\">" \
			     "This plug-in introduces support for\n"	\
			     "crystallographic structures in\n"		\
			     "<b>ABINIT</b> input files.</span>")
#define ABINIT_AUTHORS     "Caliste Damien"

/* Local methods */
static gboolean loadAbinitIn(VisuData *data, const gchar* filename,
			     FileFormat *format _U_, int nSet, GError **error);
static gpointer loadAbinitThread(gpointer data);
/* static RenderingFormatLoad* abStructuralInit(); */
/* Local variables */
static gchar *iconPath;

/* Required methods for a loadable module. */
gboolean abinitInit()
{
  RenderingFormatLoad* meth;

  DBG_fprintf(stderr, "Abinit: loading plug-in 'abinit'...\n");

  DBG_fprintf(stderr, "Abinit: declare a new rendering load method.\n");
  meth = abStructuralInit();
  renderingAtomicAdd_loadMethod(meth);

  iconPath = g_build_filename(V_SIM_PIXMAPS_DIR, "abinit.png", NULL);

  return TRUE;
}

const char* abinitGet_description()
{
  return ABINIT_DESCRIPTION;
}

const char* abinitGet_authors()
{
  return ABINIT_AUTHORS;
}

const char* abinitGet_icon()
{
  return iconPath;
}

RenderingFormatLoad* abStructuralInit()
{
  char *type[] = {"*.in", (char*)0};
  char *descr = _("ABINIT input file format");
  RenderingFormatLoad *meth;
  
  meth = g_malloc(sizeof(RenderingFormatLoad));
  meth->name = "ABINIT input file format";
  meth->fmt = fileFormatNew(descr, type);
  if (!meth->fmt)
    g_error("Can't initialize the ABINIT loading method, aborting...\n");
  meth->priority = 90;
  meth->load = loadAbinitIn;

  return meth;
}

struct Load_struct
{
  VisuData *data;
  gchar *filename;
  int nSet;
};

static gboolean loadAbinitIn(VisuData *data, const gchar* filename,
			     FileFormat *format _U_, int nSet, GError **error)
{
  struct Load_struct ld_data;
#ifdef G_THREADS_ENABLED
  GThread *ld_thread;
#endif
  gboolean res;

  g_return_val_if_fail(error && !*error, FALSE);

  ld_data.data = data;
  ld_data.filename = g_strdup(filename);
  ld_data.nSet = nSet;
#ifdef G_THREADS_ENABLED
  DBG_fprintf(stderr, "AB structure: run ABINIT parsing into a thread.\n");
  ld_thread = g_thread_create(loadAbinitThread, (gpointer)&ld_data,
			      TRUE, error);
  if (ld_thread)
    *error = (GError*)g_thread_join(ld_thread);
  else
    g_warning("Can't run thread for ABINIT parsing.");
#else
  DBG_fprintf(stderr, "AB structure: run ABINIT parsing directly.\n");
  *error = (GError*)loadAbinitThread((gpointer)&ld_data);
#endif
  g_free(ld_data.filename);

  if (*error && (*error)->code == RENDERING_ERROR_FILE)
    {
      g_error_free(*error);
      *error = (GError*)0;
      res = FALSE;
    }
  else
    res = TRUE;
  
  return res;
}


static AbDtsets *dt;

static void onAbError(AbDtsets *dt, AbError error, AbDtsetIds id)
{
  GError *err;

  err = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
		      "Abinit parser report error %d.", (int)error);

  DBG_fprintf(stderr, "AB(1) structure: an error (%d) occured"
	      " while parsing attribute %d.\n", (int)error, (int)id);
  ab_dtset_free(dt);
#ifdef G_THREADS_ENABLED
  g_thread_exit((gpointer)err);
#else
  g_error("ABINIT plug-in requires threads.");
#endif
}

static gpointer loadAbinitThread(gpointer data)
{
  struct Load_struct *ld_data;
  int ndtset, i, res;
  int ntypat, natom, nzero, *index;
  int *typat;
  double *znucl, rprimd[3][3], *coord;
  VisuElement **types, **ntypes;
  float rcov, box[6], cart[3], red[3];
  char *ptChar, *name;
  unsigned int *nattyp;
  AbError error;

  ld_data = (struct Load_struct*)data;

  /* Read the file and store its contain as a string. */
  DBG_fprintf(stderr, "AB(1) structure: read and store input file.\n");
  dt = (AbDtsets*)0;
  dt = ab_dtset_new(ld_data->filename);

  /* Ok, try to find the required keywords. */
  error = ab_dtset_get_ndtset(dt, &ndtset);
  if (error != AB_NO_ERROR) onAbError(dt, error, AB_DTSET_N_IDS);
  DBG_fprintf(stderr, "AB(1) structure: found %d dtsets.\n", ndtset);
  /* Store the number of datasets. */
  visuDataSet_nSet(ld_data->data, ndtset);

  g_return_val_if_fail(ld_data->nSet >= 0 &&
		       ld_data->nSet < ndtset, (gpointer)0);

  error = ab_dtset_get_integer(dt, AB_DTSET_NTYPAT, ld_data->nSet + 1, &ntypat);
  if (error != AB_NO_ERROR) onAbError(dt, error, AB_DTSET_NTYPAT);
  error = ab_dtset_get_integer(dt, AB_DTSET_NATOM,  ld_data->nSet + 1, &natom);
  if (error != AB_NO_ERROR) onAbError(dt, error, AB_DTSET_NATOM);
  DBG_fprintf(stderr, "AB(1) structure: with %d atoms and %d types.\n",
	      natom, ntypat);
  
  typat = g_malloc(sizeof(int) * natom);
  error = ab_dtset_get_integer_array(dt, typat, natom,
				     AB_DTSET_TYPAT, ld_data->nSet + 1);
  if (error != AB_NO_ERROR) onAbError(dt, error, AB_DTSET_TYPAT);

  znucl = g_malloc(sizeof(double) * ntypat);
  types = g_malloc(sizeof(VisuElement*) * ntypat);
  nattyp = g_malloc(sizeof(unsigned int) * ntypat);
  error = ab_dtset_get_real_array(dt, znucl, ntypat,
				  AB_DTSET_ZNUCL, ld_data->nSet + 1);
  if (error != AB_NO_ERROR) onAbError(dt, error, AB_DTSET_ZNUCL);
  DBG_fprintf(stderr, "AB(1) structure: read znucl OK.\n");
  for (i = 0; i < ntypat; i++)
    {
      /* Try to find a name instead of a z number. */
      toolElementsGet_element(&ptChar, &rcov, znucl[i]);
      name = g_strdup(ptChar);
      /* adding name to the hashtable */
      types[i] = visuElementGet_fromName(name);
      if (!types[i])
	{
	  types[i] = visuElementNew_withName(name);
	  g_return_val_if_fail(types[i], (gpointer)0);
	  res = visuElementAdd(types[i]);
	  g_return_val_if_fail(!res, (gpointer)0);
	  renderingAtomicSet_radius(types[i], rcov);
	}
      g_free(name);
    }
  g_free(znucl);
  DBG_fprintf(stderr, "AB(1) structure: all new elements created.\n");
  memset(nattyp, 0, sizeof(unsigned int) * ntypat);
  ntypes = g_malloc(sizeof(VisuElement*) * natom);
  for (i = 0; i < natom; i++)
    {
      nattyp[typat[i] - 1] += 1;
      ntypes[i] = types[typat[i] - 1];
    }
  /* Reduce the arrays when nattyp is 0. */
  nzero = 0;
  index = g_malloc(sizeof(int) * ntypat);
  for (i = 0; i < ntypat; i++)
    {
      if (i > nzero)
	{
	  nattyp[nzero] = nattyp[i];
	  types[nzero]  = types[i];
	}
      index[i] = nzero;
      if (nattyp[i] > 0)
	nzero += 1;
    }
  DBG_fprintf(stderr, "AB(1) structure: removing null types.\n");
  for (i = 0; i < natom; i++)
    {
      DBG_fprintf(stderr, "AB(1) structure: atom %d (%d)", i, typat[i] - 1);
      DBG_fprintf(stderr, " -> %d.\n", index[typat[i] - 1]);
      ntypes[i] = types[index[typat[i] - 1]];
    }
  g_free(typat);
  g_free(index);
  ntypat = nzero;

  DBG_fprintf(stderr, "AB(1) structure: there are %d types in this file.\n", ntypat);
  if (DEBUG)
    for (i = 0; i < ntypat; i++)
      fprintf(stderr, " | %d atom(s) for type %d.\n", nattyp[i], i);
  if (DEBUG)
    for (i = 0; i < natom; i++)
      fprintf(stderr, " | atom %d of type %p.\n", i, (gpointer)ntypes[i]);
  res = visuDataSet_population(ld_data->data, ntypat, nattyp, types);
  if (!res)
    g_error("Can't store the nodes in the VisuData object.");
  g_free(nattyp);
  g_free(types);

  error = ab_dtset_get_real_array(dt, (double*)rprimd, 9,
				  AB_DTSET_RPRIMD_ORIG, ld_data->nSet + 1);
  if (error != AB_NO_ERROR) onAbError(dt, error, AB_DTSET_RPRIMD_ORIG);
  DBG_fprintf(stderr, "AB(1) structure : File '%s' parsed.\n", ld_data->filename);
  DBG_fprintf(stderr, " | box definition : ( %f %f %f )\n",
	      rprimd[0][0], rprimd[0][1], rprimd[0][2]);
  DBG_fprintf(stderr, " |                  ( %f %f %f )\n",
	      rprimd[1][0], rprimd[1][1], rprimd[1][2]);
  DBG_fprintf(stderr, " |                  ( %f %f %f )\n",
	      rprimd[2][0], rprimd[2][1], rprimd[2][2]);
  g_return_val_if_fail(matrix_reducePrimitiveVectors(box, rprimd), (gpointer)0);
  visuDataSet_boxGeometry(ld_data->data, box, TRUE);
  
  coord = g_malloc(sizeof(double) * 3 * natom);
  error = ab_dtset_get_real_array(dt, coord, 3 * natom,
				  AB_DTSET_XRED_ORIG, ld_data->nSet + 1);
  if (error != AB_NO_ERROR) onAbError(dt, error, AB_DTSET_XRED_ORIG);
  for (i = 0; i < natom; i++)
    {
      red[0] = (float)*(coord + 3 * i + 0);
      red[1] = (float)*(coord + 3 * i + 1);
      red[2] = (float)*(coord + 3 * i + 2);
      DBG_fprintf(stderr, " |                  ( %f %f %f )\n",
		  red[0], red[1], red[2]);
      /* Transform the reduced coordinates into cartesians. */
      visuDataConvert_boxCoordinatestoXYZ(ld_data->data, cart, red);
      DBG_fprintf(stderr, " |               -> ( %f %f %f )\n",
		  cart[0], cart[1], cart[2]);
      visuDataAdd_nodeFromElement(ld_data->data, ntypes[i], cart);
    }
  g_free(ntypes);

  ab_dtset_free(dt);

  return (gpointer)0;
}

void FC_FUNC(wrtout, WRTOUT)(int *unit,char message[500], char mode_paral[4])
{
  gchar *buf, *ptError, *ptInvars0, *ptInstrng, *ptSize;
  GError *error;

  buf = g_strndup(message, 500);
  g_strstrip(buf);
  DBG_fprintf(stderr, "AB(1) structure: (%d %c%c%c%c) %s\n",
	      *unit, mode_paral[0], mode_paral[1], mode_paral[2], mode_paral[3], buf);
  /* We analyse buf. If, it contains an error, we test if it is
     about natom in inarvs0. If so, the file is not a valid ABINIT
     file. On the contrary, we get the message and raise an error. */
  ptError = strstr(buf, "ERROR");
  ptInvars0 = strstr(buf, "Input natom must be defined");
  ptInstrng = strstr(buf, "The occurence of a tab");
  ptSize    = strstr(buf, "The size of your input file");
  if (ptError && !ptInvars0 && !ptInstrng && !ptSize)
    error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT, buf);
  else if (ptError && (ptInvars0 || ptInstrng || ptSize))
    error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			"Not an ABINIT file");
  else
    error = (GError*)0;

  g_free(buf);

  if (error)
    {
      DBG_fprintf(stderr, "AB(1) structure: an error occured while parsing.\n");
      if (dt)
	ab_dtset_free(dt);
#ifdef G_THREADS_ENABLED
      g_thread_exit((gpointer)error);
#else
      g_error("ABINIT plug-in requires threads.");
#endif
    }
}
void FC_FUNC(leave_new, LEAVE_NEW)(char mode_paral[4])
{
  GError *error;

  DBG_fprintf(stderr, "AB(1) structure: leave_new(%c%c%c%c)\n",
	      mode_paral[0], mode_paral[1], mode_paral[2], mode_paral[3]);
  ab_dtset_free(dt);
  error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
		      "Not an ABINIT file");
#ifdef G_THREADS_ENABLED
  g_thread_exit((gpointer)error);
#else
  g_error("ABINIT plug-in requires threads.");
#endif
}
void FC_FUNC(timab, TIMAB)()
{
}
