/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD, Damien
	CALISTE, Olivier D'Astier, laboratoire L_Sim, (2001-2005)
  
	Adresses ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD and Damien
	CALISTE and Olivier D'Astier, laboratoire L_Sim, (2001-2005)

	E-mail addresses :
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "scalarFields.h"

#include <string.h>
#include <math.h>
#include <GL/gl.h>

#include <visu_tools.h>
#include <coreTools/toolFileFormat.h>
#include <coreTools/toolMatrix.h>
#include "surfaces_points.h"
#include "isoline.h"

/**
 * SECTION:scalarFields
 * @short_description:Gives capabilities to load a scalar field.
 *
 * <para>A scalar field is represented by the given of datas on a
 * regular grid meshing the bounding box. Scalar field can be read
 * from several kind of files by adding load methods using
 * scalarFieldAdd_loadMethod(). The basic implementation gives access
 * to ASCII encoded files following a simple format.</para>
 *
 * <para>In coordination with #Plane and #Shade, scalar field can be
 * represented as coloured map calling scalarFieldDraw_map(). The
 * current implementation of interpolation is very limited since basic
 * linear approximation is used.</para>
 *
 * <para>If a structure file also contains a scalar field, when
 * loaded, it should add a #VisuData property called
 * #SCALAR_FIELD_DEFINED_IN_STRUCT_FILE using
 * visuDataAdd_property(). Then V_Sim will be able to handle the
 * structure file as a density file also.</para>
 */

#define MESH_FLAG             "meshType"
#define MESH_FLAG_UNIFORM     "uniform"
#define MESH_FLAG_NON_UNIFORM "nonuniform"

/**
 * ScalarField_struct:
 * @filename: the path to the file from which the scalar field
 *            has been read ;
 * @commentary: a commentary read from the file (must be in UTF-8) ;
 * @box: description of the associated bounding box ;
 * @nElements: number of points in each direction ;
 * @data: the values ;
 * @min: the minimum value ;
 * @max: the maximum value ;
 * @options: a GList of #Option values.
 *
 * The structure used to store a scalar field.
 */
struct ScalarField_struct
{
  /* The name of the file, where the data were read from. */
  gchar *filename;

  /* An associated commentary. */
  gchar *commentary;

  /* Description of the box. */
  float box[6];
  float fromXYZtoReducedCoord[3][3];

  /* Number of elements in each directions [x, y, z]. */
  int nElements[3];

  /* Mesh. */
  double *meshx;
  double *meshy;
  double *meshz;

  /* Datas. */
  double ***data;

  /* Minimum and maximum values. */
  double min, max, secondMin;

  /* Set to TRUE if the data can replicate by periodicity. */
  gboolean periodic;

  /* mesh type: uniform or non uniform */
  ScalarField_meshflag mesh_type;

  /* A GList to store some options (key, values) associated
     to the data. */
  GList *options;
};

enum
  {
    MAP_X,
    MAP_Y,
    MAP_Z,
    MAP_VISIBILITY,
    MAP_VALUE,
    MAP_R,
    MAP_G,
    MAP_B,
    MAP_A
  };

/* Local variables. */
static GList *loadMethods;

/* Local methods. */
static gboolean scalarFieldLoad_fromAscii(const gchar *filename, GList **fieldList, GError **error,
					  OptionTable *table);
static gint compareLoadPriority(gconstpointer a, gconstpointer b);


ScalarField* scalarFieldNew(const gchar *filename)
{
  ScalarField *field;

  g_return_val_if_fail(filename && filename[0], (ScalarField*)0);

  field               = g_malloc(sizeof(ScalarField));
  field->nElements[0] = 0;
  field->nElements[1] = 0;
  field->nElements[2] = 0;
  field->filename     = g_strdup(filename);
  field->commentary   = (gchar*)0;
  field->meshx        = (double*)0;
  field->meshy        = (double*)0;
  field->meshz        = (double*)0;
  field->data         = (double***)0;
  field->min          = G_MAXFLOAT;
  field->max          = -G_MAXFLOAT;
  field->secondMin    = G_MAXFLOAT;
  field->periodic     = FALSE;
  field->mesh_type    = uniform;
  field->options      = (GList*)0;

  return field;
}
void scalarFieldFree(ScalarField *field)
{
  int i, j;
  GList *tmplst;

  g_return_if_fail(field);

  if (field->filename)
    g_free(field->filename);

  if (field->commentary)
    g_free(field->commentary);

  if (field->meshx)
    g_free(field->meshx);

  if (field->meshy)
    g_free(field->meshy);

  if (field->meshz)
    g_free(field->meshz);

  if (field->data)
    {
      for (i = 0; i < field->nElements[0]; i++)
	{
	  for (j = 0; j < field->nElements[1]; j++)
	    g_free(field->data[i][j]);
	  g_free(field->data[i]);
	}
      g_free(field->data);
    }

  if (field->options)
    {
      tmplst = field->options;
      while(tmplst)
        {
          toolOptionsFree_option(tmplst->data);
          tmplst = g_list_next(tmplst);
        }
      g_list_free(field->options);
    }
}

gboolean scalarFieldLoad_fromFile(const gchar *filename, GList **fieldList,
				  GError **error, OptionTable *table)
{
  gboolean validFormat;
  GList *tmpLst;

  g_return_val_if_fail(filename, FALSE);
  g_return_val_if_fail(*fieldList == (GList*)0, FALSE);
  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);

  /* Try all supported format. */
  validFormat = FALSE;
  tmpLst = loadMethods;
  while(tmpLst && !validFormat)
    {
      DBG_fprintf(stderr, "Scalar Fields : try to open the file as a '%s'.\n",
		  ((ScalarFieldLoadStruct*)tmpLst->data)->name);
      validFormat = ((ScalarFieldLoadStruct*)tmpLst->data)->load(filename, fieldList, error, table);
      if (!validFormat && *error)
        {
          g_error_free(*error);
          *error = (GError*)0;
        }
      tmpLst = g_list_next(tmpLst);
    }
  
  if (!validFormat)
    g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_UNKNOWN_FORMAT, 
		_("unknown density/potential format.\n")); 
  return validFormat;
}


static gboolean scalarFieldLoad_fromAscii(const gchar *filename, GList **fieldList,
					  GError **error, OptionTable *table _U_)
{
  FILE *in;
  char rep[MAX_LINE_LENGTH], flag[MAX_LINE_LENGTH];
  char format[MAX_LINE_LENGTH], period[MAX_LINE_LENGTH];
  char *feed;
  gchar *comment;
  int res, i, j, k;
  int size[3];
  float box[6];
  ScalarField *field;
  gboolean periodic;
  ScalarField_meshflag meshtype;

  g_return_val_if_fail(filename, FALSE);
  g_return_val_if_fail(*fieldList == (GList*)0, FALSE);
  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);

  DBG_fprintf(stderr, "ScalarField : try to read '%s' as a ASCII scalar"
	      " field data file.\n", filename);

  in = fopen(filename, "r");
  if (!in)
    {
      g_set_error(error, G_FILE_ERROR, G_FILE_ERROR_ACCES,
		  _("impossible to open the file.\n"));
      return FALSE;
    }

  /* 1st line (comment) */
  (void)fgets(rep, MAX_LINE_LENGTH, in);
  rep[strlen(rep)-1] = 0; /* skipping \n */
  comment = g_locale_to_utf8(rep, -1, NULL, NULL, NULL);
  if (!comment)
    comment = g_strdup("");

  (void)fgets(rep, MAX_LINE_LENGTH, in);
  res = sscanf(rep, "%d %d %d", size, size + 1, size + 2);
  if (res != 3)
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }

  (void)fgets(rep, MAX_LINE_LENGTH, in);
  res = sscanf(rep, "%f %f %f", box, box + 1, box + 2);
  if (res != 3)
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }
  (void)fgets(rep, MAX_LINE_LENGTH, in); 
  res = sscanf(rep, "%f %f %f", box + 3, box + 4, box + 5);
  if (res != 3)
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }

  (void)fgets(rep, MAX_LINE_LENGTH, in);
  res = sscanf(rep, "%s %s", format, period);
  if (res < 1 || (strcmp(format, "xyz") && strcmp(format, "zyx")))
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }
  periodic = !strcmp(period, "periodic");

  /* OK, from now on, the format is supposed to be ASCII. */
  field = scalarFieldNew(filename);
  if (!field)
    {
      g_warning("impossible to create a ScalarField object.");
      g_free(comment);
      fclose(in);
      return FALSE;
    }

  /* by default the meshtype is set to uniform to keep working previous version.*/
  meshtype = uniform;
  feed = fgets(rep, MAX_LINE_LENGTH, in);
  while (rep[0] == '#' && feed)
    {
      if (strncmp(rep + 2, MESH_FLAG, strlen(MESH_FLAG)) == 0)
	{
	  DBG_fprintf(stderr, "ScalarField: found flag '%s'.\n", MESH_FLAG);
	  res = sscanf(rep + 2 + strlen(MESH_FLAG) + 1, "%s", flag);
	  if (res == 1 && strcmp(flag, MESH_FLAG_UNIFORM) == 0)
	    meshtype = uniform;
	  else if (res == 1 && strcmp(flag, MESH_FLAG_NON_UNIFORM) == 0)
	    meshtype = nonuniform;
	  else if (res == 1)
	    {
              g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("wrong '%s' value for flag '%s'.\n"), flag, MESH_FLAG);
              fclose(in);
              return TRUE;
	    }
	}
      feed = fgets(rep, MAX_LINE_LENGTH, in);
    }

  scalarFieldSet_periodic(field, periodic);
  scalarFieldSet_commentary(field, comment);
  scalarFieldSet_meshtype(field, meshtype);
  scalarFieldSet_gridSize(field, size);
  scalarFieldSet_box(field, box);
  *fieldList = g_list_append(*fieldList, (gpointer)field);

  if (meshtype == nonuniform)
    {
      DBG_fprintf(stderr, "ScalarField : Start to read meshx.\n");

      for ( i = 0; i < size[0]; i++ )
        {
	  if (!feed)
	    {
	      g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("not enough meshx values.\n"));
	      fclose(in);
	      return TRUE;
	    }
          res = sscanf(rep, "%lf", &field->meshx[i]);
	  do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
          if (res != 1)
            {
              g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("impossible to read meshx values.\n"));
              fclose(in);
              return TRUE;
            }
        }

      DBG_fprintf(stderr, "ScalarField : Start to read meshy.\n");

      for ( j = 0; j < size[1]; j++ )
        {
	  if (!feed)
	    {
	      g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("not enough meshy values.\n"));
	      fclose(in);
	      return TRUE;
	    }
          res = sscanf(rep, "%lf", &field->meshy[j]);
	  do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
          if (res != 1)
            {
              g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("impossible to read meshy values.\n"));
              fclose(in);
              return TRUE;
            }
        }

      DBG_fprintf(stderr, "ScalarField : Start to read meshz.\n");

      for ( k = 0; k < size[2]; k++ )
        {
	  if (!feed)
	    {
	      g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("not enough meshz values.\n"));
	      fclose(in);
	      return TRUE;
	    }
          res = sscanf(rep, "%lf", &field->meshz[k]);
	  do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
          if (res != 1)
            {
              g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			  _("impossible to read meshz values.\n"));
              fclose(in);
              return TRUE;
            }
        }
    }

  DBG_fprintf(stderr, "ScalarField : Start to read data.\n");

  field->min = G_MAXFLOAT;
  field->max = G_MINFLOAT;
  if(!strcmp(format, "xyz"))
    {
      for ( k = 0; k < size[2]; k++ ) 
	for ( j = 0; j < size[1]; j++ ) 
	  for ( i = 0; i < size[0]; i++ ) 
	    {
	      if (!feed)
		{
		  g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			      _("not enough density values.\n"));
		  fclose(in);
		  return TRUE;
		}

	      res = sscanf(rep, "%lf", &field->data[i][j][k]);

	      do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
	      if (res != 1)
		{
		  g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			      _("impossible to read density values.\n"));
		  fclose(in);
		  return TRUE;
		}
	      field->min = MIN(field->data[i][j][k], field->min);
	      field->max = MAX(field->data[i][j][k], field->max);
	    }
    }
  else
    {
      for ( i = 0; i < size[0]; i++ ) 
	for ( j = 0; j < size[1]; j++ ) 
	  for ( k = 0; k < size[2]; k++ ) 
	    {
	      if (!feed)
		{
		  g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			      _("not enough density values.\n"));
		  fclose(in);
		  return TRUE;
		}
	      res = sscanf(rep, "%lf", &field->data[i][j][k]);
	      do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
	      if (res != 1)
		{
		  g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			      _("impossible to read density values.\n"));
		  fclose(in);
		  return TRUE;
		}
	      field->min = MIN(field->data[i][j][k], field->min);
	      field->max = MAX(field->data[i][j][k], field->max);
	    }
    }
  
  fclose(in);
  return TRUE;
}

gchar* scalarFieldGet_commentary(ScalarField *field)
{
  g_return_val_if_fail(field, (gchar*)0);
  return field->commentary;
}
void scalarFieldSet_commentary(ScalarField *field, gchar* comment)
{
  g_return_if_fail(field);
  
  field->commentary = g_strdup(comment);
}

gboolean scalarFieldGet_periodic(ScalarField *field)
{
  g_return_val_if_fail(field, FALSE);
  return field->periodic;
}
void scalarFieldSet_periodic(ScalarField *field, gboolean periodic)
{
  g_return_if_fail(field);
  
  field->periodic = periodic;
}

ScalarField_meshflag scalarFieldGet_meshtype(ScalarField *field)
{
  g_return_val_if_fail(field, uniform);
  return field->mesh_type;
}

void scalarFieldSet_meshtype(ScalarField *field, ScalarField_meshflag meshtype)
{
  g_return_if_fail(field);

  field->mesh_type = meshtype;
}

gchar* scalarFieldGet_filename(ScalarField *field)
{
  g_return_val_if_fail(field, (gchar*)0);
  return field->filename;
}

void scalarFieldSet_gridSize(ScalarField *field, int grid[3])
{
  int i, j;
  
  g_return_if_fail(field);
  
  if (field->nElements[0] == grid[0] &&
      field->nElements[1] == grid[1] &&
      field->nElements[2] == grid[2])
    return;
  
  DBG_fprintf(stderr, "ScalarField : changing size from (%d ; %d ; %d)"
                      " to (%d ; %d ; %d).\n", field->nElements[0], field->nElements[1],
                      field->nElements[2], grid[0], grid[1], grid[2]);

  if (field->mesh_type == nonuniform)
    {
      /* If mesh was already allocated, we free it. */
      if (field->meshx)
        g_free(field->meshx);

      if (field->meshy)
        g_free(field->meshy);

      if (field->meshz)
        g_free(field->meshz);
    }

  /* If data was already allocated, we free it. */
  if (field->data)
    {
      for (i = 0; i < field->nElements[0]; i++)
	      {
          for (j = 0; j < field->nElements[1]; j++)
	          g_free(field->data[i][j]);
      	  g_free(field->data[i]);
      	}
      g_free(field->data);
    }

  /* We change the size and reallocate mesh and data. */
  field->nElements[0] = grid[0];
  field->nElements[1] = grid[1];
  field->nElements[2] = grid[2];

  if (field->mesh_type == nonuniform)
    {
      DBG_fprintf(stderr, "ScalarField : allocating meshx array.\n");
      field->meshx = g_malloc(sizeof(double) * grid[0]);

      DBG_fprintf(stderr, "ScalarField : allocating meshy array.\n");
      field->meshy = g_malloc(sizeof(double) * grid[1]);

      DBG_fprintf(stderr, "ScalarField : allocating meshz array.\n");
      field->meshz = g_malloc(sizeof(double) * grid[2]);
    }

  DBG_fprintf(stderr, "ScalarField : allocating data array.\n");
  field->data = g_malloc(sizeof(double **) * grid[0]);
  for(i = 0; i < grid[0]; i++)
    {
      field->data[i] = g_malloc(sizeof(double *) * grid[1]);
      for(j = 0; j < grid[1]; j++)
        field->data[i][j] = g_malloc(sizeof(double) * grid[2]);
    }
}

void scalarFieldGet_minMax(ScalarField *field, double minmax[2])
{
  g_return_if_fail(field);
  
  minmax[0] = field->min;
  minmax[1] = field->max;
}
double scalarFieldGet_secondaryMin(ScalarField *field)
{
  int i, j, k;

  g_return_val_if_fail(field, 0.);
  
  /* In log scale, we need the second minimum value. */
  if (field->secondMin == G_MAXFLOAT)
    {
      for (i = 0; i < field->nElements[0]; i++)
	for (j = 0; j < field->nElements[1]; j++)
	  for (k = 0; k < field->nElements[2]; k++)
	    field->secondMin = (field->data[i][j][k] == field->min)?
	      field->secondMin:MIN(field->data[i][j][k], field->secondMin);
      DBG_fprintf(stderr, "Scalar Fields: second minimum is %g (%g < %g).\n",
		  field->secondMin, field->min, field->max);
    }
  return field->secondMin;
}
void scalarFieldSet_data(ScalarField *field, double *data)
{
  int i, j, k, ii;
  
  g_return_if_fail(field && data);
  
  field->min = G_MAXFLOAT;
  field->max = -G_MAXFLOAT;
  ii = 0;
  for (k = 0 ; k < field->nElements[2] ; k++)
    for (j = 0 ; j < field->nElements[1] ; j++)
      for (i = 0 ; i < field->nElements[0] ; i++)
        {
          field->data[i][j][k] = data[ii];
          ii += 1;
	  field->min = MIN(field->data[i][j][k], field->min);
	  field->max = MAX(field->data[i][j][k], field->max);
        }
}
void scalarFieldDraw_map(ScalarField *field, Plane *plane, Shade *shade,
                         OpenGLView *view, float precision, float drawnMinMax[2],
			 gboolean logScale, guint nIsoLines)
{
  float xyz[2][3], normal[3], spherical[3], center[3], uMinMax[2], vMinMax[2];
  float dist, Xcoeff, Ycoeff, coeff, uCoord, vCoord;
  SurfacesPoints points;
  double value, minmax[2], minlog;
  float valMinMax[2];
  GList *inter;
  int alpha, i, j, k, uAlpha[2], vAlpha[2], nU, nV, nPolys, nPoints;
  int iPolys[4];
  Line *line;
  #if DEBUG == 1
  GTimer *timer;
  gulong fractionTimer;
  #endif

  g_return_if_fail(field && shade && plane);
  g_return_if_fail(view && precision > 0);

  #if DEBUG == 1
    timer = g_timer_new();
    g_timer_start(timer);
  #endif

  planeGet_nVectUser(plane, normal);
  matrix_cartesianToSpherical(spherical, normal);

  planeGet_barycentre(plane, center);
  /* We compute the min and max in the 3D basis. */
  inter = planeGet_intersection(plane);
  /* In case of no intersections, we build an empty list. */
  if (!inter)
    return;
  
  /* We compute the plane basis. */
  xyz[0][0] = cos(spherical[1] * PI180) * cos(spherical[2] * PI180);
  xyz[0][1] = cos(spherical[1] * PI180) * sin(spherical[2] * PI180);
  xyz[0][2] = -sin(spherical[1] * PI180);
  xyz[1][0] = -sin(spherical[2] * PI180);
  xyz[1][1] =  cos(spherical[2] * PI180);
  xyz[1][2] = 0.;

  /* We look for the min and max position in the plane basis for the
     different intersections. */
  dist = 0.;
  uMinMax[0] = 0.;
  uMinMax[1] = 0.;
  vMinMax[0] = 0.;
  vMinMax[1] = 0.;
  while(inter)
    {
      uCoord = xyz[0][0] * (((float*)inter->data)[0] - center[0]) +
	xyz[0][1] * (((float*)inter->data)[1] - center[1]) +
	xyz[0][2] * (((float*)inter->data)[2] - center[2]);
      uMinMax[0] = MIN(uCoord, uMinMax[0]);
      uMinMax[1] = MAX(uCoord, uMinMax[1]);
      vCoord = xyz[1][0] * (((float*)inter->data)[0] - center[0]) +
	xyz[1][1] * (((float*)inter->data)[1] - center[1]) +
	xyz[1][2] * (((float*)inter->data)[2] - center[2]);
      vMinMax[0] = MIN(vCoord, vMinMax[0]);
      vMinMax[1] = MAX(vCoord, vMinMax[1]);
      inter = g_list_next(inter);
    }
  /* So, in plane coordinates, X range in uMinMax and Y in vMinMax. */

  /* We put by default (precision and gross = 1) 30 triangles along
     the longest line. */
  alpha = (int)(60. * (OpenGLViewGet_precision() / 2. + .5) *
                precision * (view->camera->gross / 2. + .5) / 100.);
  dist = sqrt(view->box->p7[0] * view->box->p7[0] +
	      view->box->p7[1] * view->box->p7[1] +
	      view->box->p7[2] * view->box->p7[2]);
/*   dist = MAX(uMinMax[1] - uMinMax[0], vMinMax[1] - vMinMax[0]); */
  uAlpha[0] = (int)(uMinMax[0] / dist * (float)alpha) - 1;
  uAlpha[1] = (int)(uMinMax[1] / dist * (float)alpha) + 2;
  vAlpha[0] = (int)(vMinMax[0] / dist * (float)alpha * 1.154700538379251448) - 1;
  vAlpha[1] = (int)(vMinMax[1] / dist * (float)alpha * 1.154700538379251448) + 2;
  scalarFieldGet_minMax(field, minmax);
  drawnMinMax[0] = minmax[1];
  drawnMinMax[1] = minmax[0];

  DBG_fprintf(stderr, "Panel Map: allocating storing values (%dx%d).\n",
	      vAlpha[0] + vAlpha[1], uAlpha[0] + uAlpha[1]);
  /* We store 6 additional data per point.
     - 0: used or not ;
     - 1: field value ;
     - 2~5: RGBA. */
  isosurfacesPointsInit(&points, 6);
  nU = uAlpha[1] - uAlpha[0];
  nV = vAlpha[1] - vAlpha[0];
  isosurfacesPointsAllocate(&points, 1, (nV - 1) * (2 * (nU - 1) - 2), nV * nU);

  /* In log scale, we need the second minimum value. */
  minlog = minmax[0];
  if (logScale)
    minlog = scalarFieldGet_secondaryMin(field);

  /* We use an hexagonal padding for equilateral triangles. */
  minlog = log10((minlog - minmax[0]) / (minmax[1] - minmax[0]));
  coeff =  dist / (float)alpha;
  nPoints = 0;
  for (i = 0; i < vAlpha[1] - vAlpha[0]; i++)
    {
      Ycoeff = (float)(i + vAlpha[0]) * coeff * 0.8660254; /* the num coef is cos 30 */
      for (j = 0; j < uAlpha[1] - uAlpha[0]; j++)
	{
	  if (i%2)
	    Xcoeff = (float)(j + uAlpha[0]) * coeff;
	  else
	    Xcoeff = ((float)(j + uAlpha[0]) + 0.5) * coeff;
	  points.poly_points_data[nPoints][MAP_X] = center[0] +
	    Xcoeff * xyz[0][0] + Ycoeff * xyz[1][0];
	  points.poly_points_data[nPoints][MAP_Y] = center[1] +
	    Xcoeff * xyz[0][1] + Ycoeff * xyz[1][1];
	  points.poly_points_data[nPoints][MAP_Z] = center[2] +
	    Xcoeff * xyz[0][2] + Ycoeff * xyz[1][2];
	  points.poly_points_data[nPoints][MAP_VISIBILITY] =
	    (float)scalarFieldGet_value(field, points.poly_points_data[nPoints],
					&value, view->box->extension);
	  if ((gboolean)points.poly_points_data[nPoints][MAP_VISIBILITY])
	    {
	      points.poly_points_data[nPoints][MAP_VALUE] =
		(float)((CLAMP(value, minmax[0], minmax[1]) - minmax[0]) /
			(minmax[1] - minmax[0]));
	      if (logScale)
		points.poly_points_data[nPoints][MAP_VALUE] =
		  (points.poly_points_data[nPoints][MAP_VALUE] == 0.f)?0.f:
		  - (log10(points.poly_points_data[nPoints][MAP_VALUE])
		     - minlog) / minlog;
	      shadeGet_valueTransformedInRGB
		(shade,
		 points.poly_points_data[nPoints] + MAP_R,
		 points.poly_points_data[nPoints][MAP_VALUE]);
	      drawnMinMax[0] = MIN(drawnMinMax[0], (float)value);
	      drawnMinMax[1] = MAX(drawnMinMax[1], (float)value);
	    }
/* 	  DBG_fprintf(stderr, " | %d (%g, %g, %g) -> %g\n", (gboolean)values[i][j][0], */
/* 		      values[i][j][1], values[i][j][2], values[i][j][3], values[i][j][4]); */
	  nPoints += 1;
	}
    }
  nPolys = 0;
  points.num_polys_surf[0] = 0;
  for (i = 0; i < vAlpha[1] - vAlpha[0] - 1; i++)
    {
      iPolys[3] = 0;
      for (j = 0; j < uAlpha[1] - uAlpha[0] - 1; j++)
	{
	  for (k = 1; k >= 0; k--)
	    {
	      iPolys[3] += 1;
	      iPolys[iPolys[3]%3] = (i + (i + k)%2) * nU + j;
	      if (iPolys[3] > 2)
		{
		  /* The triangle is visible if all of its vertices are
		     visible. */
		  if ((gboolean)points.poly_points_data[iPolys[0]][MAP_VISIBILITY] &&
		      (gboolean)points.poly_points_data[iPolys[1]][MAP_VISIBILITY] &&
		      (gboolean)points.poly_points_data[iPolys[2]][MAP_VISIBILITY])
		    {
		      points.poly_surf_index[nPolys] = 1;
		      points.num_polys_surf[0]      += 1;
		    }
		  else
		    points.poly_surf_index[nPolys] = -1;
		  points.poly_num_vertices[nPolys] = 3;
		  points.poly_vertices[nPolys] = g_malloc(sizeof(int) * 3);
		  points.poly_vertices[nPolys][0] = iPolys[0];
		  points.poly_vertices[nPolys][1] = iPolys[1];
		  points.poly_vertices[nPolys][2] = iPolys[2];
		  nPolys += 1;
		}
	    }
	}
    }
#if DEBUG == 1
  isosurfacesPointsCheck(&points);
#endif

  glDisable(GL_CULL_FACE);
  glDisable(GL_LIGHTING);
  glPushMatrix();
  glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);
/*   glBegin(GL_LINES); */
/*   glVertex3d(center[0] - dist * xyz[0][0] - dist * xyz[1][0], */
/* 	     center[1] - dist * xyz[0][1] - dist * xyz[1][1], */
/* 	     center[2] - dist * xyz[0][2] - dist * xyz[1][2]); */
/*   glVertex3d(center[0] + dist * xyz[0][0] - dist * xyz[1][0], */
/* 	     center[1] + dist * xyz[0][1] - dist * xyz[1][1], */
/* 	     center[2] + dist * xyz[0][2] - dist * xyz[1][2]); */
/*   glVertex3d(center[0] + dist * xyz[0][0] - dist * xyz[1][0], */
/* 	     center[1] + dist * xyz[0][1] - dist * xyz[1][1], */
/* 	     center[2] + dist * xyz[0][2] - dist * xyz[1][2]); */
/*   glVertex3d(center[0] + dist * xyz[0][0] + dist * xyz[1][0], */
/* 	     center[1] + dist * xyz[0][1] + dist * xyz[1][1], */
/* 	     center[2] + dist * xyz[0][2] + dist * xyz[1][2]); */
/*   glVertex3d(center[0] + dist * xyz[0][0] + dist * xyz[1][0], */
/* 	     center[1] + dist * xyz[0][1] + dist * xyz[1][1], */
/* 	     center[2] + dist * xyz[0][2] + dist * xyz[1][2]); */
/*   glVertex3d(center[0] - dist * xyz[0][0] + dist * xyz[1][0], */
/* 	     center[1] - dist * xyz[0][1] + dist * xyz[1][1], */
/* 	     center[2] - dist * xyz[0][2] + dist * xyz[1][2]); */
/*   glVertex3d(center[0] - dist * xyz[0][0] + dist * xyz[1][0], */
/* 	     center[1] - dist * xyz[0][1] + dist * xyz[1][1], */
/* 	     center[2] - dist * xyz[0][2] + dist * xyz[1][2]); */
/*   glVertex3d(center[0] - dist * xyz[0][0] - dist * xyz[1][0], */
/* 	     center[1] - dist * xyz[0][1] - dist * xyz[1][1], */
/* 	     center[2] - dist * xyz[0][2] - dist * xyz[1][2]); */
/*   glEnd(); */
  for (i = 0; i < points.num_polys; i++)
    {
      if (points.poly_surf_index[i] > 0)
	{
	  glBegin(GL_TRIANGLE_STRIP);
	  glColor3fv(points.poly_points_data[points.poly_vertices[i][0]] + MAP_R);
	  glVertex3fv(points.poly_points_data[points.poly_vertices[i][0]]);
	  glColor3fv(points.poly_points_data[points.poly_vertices[i][1]] + MAP_R);
	  glVertex3fv(points.poly_points_data[points.poly_vertices[i][1]]);
	  glColor3fv(points.poly_points_data[points.poly_vertices[i][2]] + MAP_R);
	  glVertex3fv(points.poly_points_data[points.poly_vertices[i][2]]);
	  glEnd();
	}
    }

  valMinMax[0] =
    (float)((CLAMP(drawnMinMax[0], minmax[0], minmax[1]) - minmax[0]) /
	    (minmax[1] - minmax[0]));
  if (logScale)
    valMinMax[0] = (valMinMax[0] == 0.f)?0.f:
      - (log10(valMinMax[0]) - minlog) / minlog;
  valMinMax[1] =
    (float)((CLAMP(drawnMinMax[1], minmax[0], minmax[1]) - minmax[0]) /
	    (minmax[1] - minmax[0]));
  if (logScale)
    valMinMax[1] = (valMinMax[1] == 0.f)?0.f:
      - (log10(valMinMax[1]) - minlog) / minlog;
  for (i = 1; i <= (int)nIsoLines; i++)
    if (isolineBuild(&line, &points, MAP_VALUE, MAP_VISIBILITY,
		     valMinMax[0] + (valMinMax[1] - valMinMax[0]) *
		     (float)i / (float)(nIsoLines + 1)))
      {
	isolineDraw(line);
	isolineFree(line);
      }

  glPopMatrix();  
  glEnable(GL_CULL_FACE);
  glEnable(GL_LIGHTING);

  isosurfacesPointsFree(&points);

  #if DEBUG == 1
    g_timer_stop(timer);
    fprintf(stderr, "ScalarFields : Draw Map in %g micro-s.\n",
	    g_timer_elapsed(timer, &fractionTimer)/1e-6);
    g_timer_destroy(timer);
  #endif
}

gboolean scalarFieldGet_value(ScalarField *field, float xyz[3],
			      double *value, float extension[3])
{
  float redXyz[3], factor[3], pos;
  int l, m, n, ijk[3], dijk[3], nMax;
  int nval1, nval2;
  double *mesh;
  ScalarField_meshflag meshtype;

  /* Taken from ABINIT */
  float x1,x2,x3;

  g_return_val_if_fail(field, FALSE);

  meshtype = scalarFieldGet_meshtype(field);

  /* First, we transform the coordinates into reduced coordinates. */
  matrix_productVector(redXyz, field->fromXYZtoReducedCoord, xyz);

  /* We compute i, j, k. */
  for (l = 0; l < 3; l++)
    {
      /* If we are periodic and inside the extension, we put back in
	 the box. */
      if (field->periodic && redXyz[l] > -extension[l] &&
	  redXyz[l] < 1. + extension[l])
	redXyz[l] = fModulo(redXyz[l], 1);
      if (field->periodic)
	nMax = field->nElements[l];
      else
	nMax = field->nElements[l] - 1;

      switch (meshtype)
        {
        case uniform:
          pos = (float)nMax * redXyz[l];
          ijk[l] = (int)pos;
          factor[l] = pos - (float)ijk[l];
          break;
        case nonuniform:
	  mesh = (double*)0;
	  switch (l)
	    {
	    case 0:
	      mesh = scalarFieldGet_meshx(field);
	      break;
	    case 1:
	      mesh = scalarFieldGet_meshy(field);
	      break;
	    case 2:
	      mesh = scalarFieldGet_meshz(field);
	      break;
	    }
          nval1 = 0;
          nval2 = nMax-1;
	  n = 0;
          for (m = 0; m < nMax/2; m++)
            {
              n = (int)((nval2-nval1)/2);
              if (n == 0) {
                n = nval1;
                break;    }
              else
                n = nval1+n;
              if (redXyz[l] > mesh[n])
                nval1 = n;
              else
                nval2 = n;
            }
          ijk[l] = n;
          factor[l] = (redXyz[l]-mesh[n])/(mesh[n+1]-mesh[n]);
          break;
        default:
          g_warning("Wrong value for 'meshtype'.");
          return FALSE;
        }
	
      if (ijk[l] < 0 || redXyz[l] < 0.)
	return FALSE;
      if (ijk[l] >= nMax)
	return FALSE;
    }

  /* We linearly interpolate the value. */
  *value = field->data[ijk[0]][ijk[1]][ijk[2]];
  for (l = 0; l < 3; l++)
    {
      dijk[0] = ijk[0];
      dijk[1] = ijk[1];
      dijk[2] = ijk[2];
      if (ijk[l] == field->nElements[l] - 1)
	dijk[l] = 0;
      else
	dijk[l] += 1;
      *value += (field->data[dijk[0]][dijk[1]][dijk[2]] -
		 field->data[ijk[0]][ijk[1]][ijk[2]]) * factor[l];
    }

  /* lower left is ijk. */
  /* upper right is dijk. */
  dijk[0] = (ijk[0] + 1) % field->nElements[0];
  dijk[1] = (ijk[1] + 1) % field->nElements[1];
  dijk[2] = (ijk[2] + 1) % field->nElements[2];
  
  /* weight is factor. */
  x1 = factor[0];
  x2 = factor[1];
  x3 = factor[2];

  /* calculation of the density value */
  *value  = 0.f;
  *value += field->data[ ijk[0]][ ijk[1]][ ijk[2]] *
    (1.f - x1) * (1.f - x2) * (1.f - x3);
  *value += field->data[dijk[0]][ ijk[1]][ ijk[2]] * x1 * (1.f - x2) * (1.f - x3);
  *value += field->data[ ijk[0]][dijk[1]][ ijk[2]] * (1.f - x1) * x2 * (1.f - x3);
  *value += field->data[ ijk[0]][ ijk[1]][dijk[2]] * (1.f - x1) * (1.f - x2) * x3;
  *value += field->data[dijk[0]][dijk[1]][ ijk[2]] * x1 * x2 * (1.f - x3);
  *value += field->data[ ijk[0]][dijk[1]][dijk[2]] * (1.f  -x1) * x2 * x3;
  *value += field->data[dijk[0]][ ijk[1]][dijk[2]] * x1 * (1.f - x2) * x3;
  *value += field->data[dijk[0]][dijk[1]][dijk[2]] * x1 * x2 * x3;

  return TRUE;
}
void scalarFieldSet_box(ScalarField *field, float box[6])
{
  int i;

  g_return_if_fail(field);

  DBG_fprintf(stderr, "ScalarField : set the bounding box.\n");

  /* Change the box. */
  for (i = 0; i < 6; i++)
    field->box[i] = box[i];

  /* Update the transformation matrix. */
  field->fromXYZtoReducedCoord[0][0] =
    1 / field->box[VISU_DATA_BOX_DXX];
  field->fromXYZtoReducedCoord[0][1] =
    - field->box[VISU_DATA_BOX_DYX] /
    field->box[VISU_DATA_BOX_DXX] /
    field->box[VISU_DATA_BOX_DYY];
  field->fromXYZtoReducedCoord[0][2] =
    - (field->box[VISU_DATA_BOX_DZX] /
       field->box[VISU_DATA_BOX_DXX] -
       field->box[VISU_DATA_BOX_DYX] *
       field->box[VISU_DATA_BOX_DZY] / 
       field->box[VISU_DATA_BOX_DXX] / 
       field->box[VISU_DATA_BOX_DYY] ) /
    field->box[VISU_DATA_BOX_DZZ];
  field->fromXYZtoReducedCoord[1][0] = 0.;
  field->fromXYZtoReducedCoord[1][1] =
    1 / field->box[VISU_DATA_BOX_DYY];
  field->fromXYZtoReducedCoord[1][2] =
    - field->box[VISU_DATA_BOX_DZY] /
    field->box[VISU_DATA_BOX_DYY] /
    field->box[VISU_DATA_BOX_DZZ];
  field->fromXYZtoReducedCoord[2][0] = 0.;
  field->fromXYZtoReducedCoord[2][1] = 0.;
  field->fromXYZtoReducedCoord[2][2] = 1 /
    field->box[VISU_DATA_BOX_DZZ];
}
void scalarFieldGet_box(ScalarField *field, float box[6])
{
  int i;

  g_return_if_fail(field);

  for (i = 0; i < 6; i++)
    box[i] = field->box[i];
}
void scalarFieldGet_gridSize(ScalarField *field, int grid[3])
{
  int i;

  g_return_if_fail(field);

  for (i = 0; i < 3; i++)
    grid[i] = field->nElements[i];
}
double* scalarFieldGet_meshx(ScalarField *field)
{
  g_return_val_if_fail(field, (double*)0);
  return field->meshx;
}
double* scalarFieldGet_meshy(ScalarField *field)
{
  g_return_val_if_fail(field, (double*)0);
  return field->meshy;
}
double* scalarFieldGet_meshz(ScalarField *field)
{
  g_return_val_if_fail(field, (double*)0);
  return field->meshz;
}
double*** scalarFieldGet_data(ScalarField *field)
{
  g_return_val_if_fail(field, (double***)0);

  return field->data;
}

void scalarFieldSet_fitToBox(VisuData *data, ScalarField *field)
{
  float box[6];
  int i;

  g_return_if_fail(data);

  DBG_fprintf(stderr, "ScalarField : change the current box to fit to %p.\n", (gpointer) data);
  for (i = 0; i < 6; i++)
    box[i] = visuDataGet_boxGeometry(data, i);
  scalarFieldSet_box(field, box);
}  
GList* scalarFieldGet_allOptions(ScalarField *field)
{
  g_return_val_if_fail(field, (GList*)0);
  
  return g_list_copy(field->options);
}
void scalarFieldAdd_option(ScalarField *field, Option *option)
{
  g_return_if_fail(field && option);
  
  field->options = g_list_append(field->options, (gpointer)option);
}


/* Load method handling. */
void scalarFieldInit()
{
  char *type[] = {"*.pot", "*.dat", (char*)0};
  char *descr = _("Potential/density files");
  FileFormat *fmt;
  
  loadMethods = (GList*)0;
  
  fmt = fileFormatNew(descr, type);
  scalarFieldAdd_loadMethod("Plain ascii", scalarFieldLoad_fromAscii,
			    fmt, G_PRIORITY_LOW);
}

void scalarFieldAdd_loadMethod(const gchar *name, ScalarFieldLoadMethod method,
			       FileFormat *format, int priority)
{
  ScalarFieldLoadStruct *meth;

  g_return_if_fail(name && method && format);

  meth = g_malloc(sizeof(ScalarFieldLoadStruct));
  meth->name = g_strdup(name);
  meth->fmt = format;
  meth->load = method;
  meth->priority = priority;

  loadMethods = g_list_prepend(loadMethods, meth);
  loadMethods = g_list_sort(loadMethods, compareLoadPriority);
}

static gint compareLoadPriority(gconstpointer a, gconstpointer b)
{
  if (((ScalarFieldLoadStruct*)a)->priority <
      ((ScalarFieldLoadStruct*)b)->priority)
    return (gint)-1;
  else if (((ScalarFieldLoadStruct*)a)->priority >
	   ((ScalarFieldLoadStruct*)b)->priority)
    return (gint)+1;
  else
    return (gint)0;
}
GList* scalarFieldGet_allLoadMethods()
{
  return loadMethods;
}
