/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD, Damien
	CALISTE, Olivier D'Astier, laboratoire L_Sim, (2001-2005)
  
	Adresses ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD and Damien
	CALISTE and Olivier D'Astier, laboratoire L_Sim, (2001-2005)

	E-mail addresses :
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "scalarFields.h"

#include <string.h>
#include <math.h>
#include <GL/gl.h>

#include <visu_tools.h>
#include <coreTools/toolFileFormat.h>
#include <coreTools/toolMatrix.h>

/**
 * ScalarField_struct:
 * @filename: the path to the file from which the scalar field
 *            has been read ;
 * @commentary: a commentary read from the file (must be in UTF-8) ;
 * @box: description of the associated bounding box ;
 * @nElements: number of points in each direction ;
 * @data: the values ;
 * @min: the minimum value ;
 * @max: the maximum value ;
 * @options: a GList of #Option values.
 *
 * The structure used to store a scalar field.
 */
struct ScalarField_struct
{
  /* The name of the file, where the data were read from. */
  gchar *filename;

  /* An associated commentary. */
  gchar *commentary;

  /* Description of the box. */
  float box[6];
  float fromXYZtoReducedCoord[3][3];

  /* Number of elements in each directions [x, y, z]. */
  int nElements[3];

  /* Datas. */
  double ***data;

  /* Minimum and maximum values. */
  double min, max;
  
  /* A GList to store some options (key, values) associated
     to the data. */
  GList *options;
};

/* Local variables. */
static GList *loadMethods;

/* Local methods. */
static gboolean scalarFieldLoad_fromAscii(const gchar *filename, GList **fieldList, GError **error,
					  OptionTable *table);
static gint compareLoadPriority(gconstpointer a, gconstpointer b);


ScalarField* scalarFieldNew(const gchar *filename)
{
  ScalarField *field;

  g_return_val_if_fail(filename && filename[0], (ScalarField*)0);

  field = g_malloc(sizeof(ScalarField));
  field->nElements[0] = 0;
  field->nElements[1] = 0;
  field->nElements[2] = 0;
  field->filename = g_strdup(filename);
  field->commentary = (gchar*)0;
  field->data = (double***)0;
  field->min = G_MAXFLOAT;
  field->max = G_MINFLOAT;
  field->options = (GList*)0;

  return field;
}
void scalarFieldFree(ScalarField *field)
{
  int i, j;
  GList *tmplst;

  g_return_if_fail(field);

  if (field->filename)
    g_free(field->filename);

  if (field->commentary)
    g_free(field->commentary);

  if (field->data)
    {
      for (i = 0; i < field->nElements[0]; i++)
	{
	  for (j = 0; j < field->nElements[1]; j++)
	    g_free(field->data[i][j]);
	  g_free(field->data[i]);
	}
      g_free(field->data);
    }
    
  if (field->options)
    {
      tmplst = field->options;
      while(tmplst)
        {
          toolOptionsFree_option(tmplst->data);
          tmplst = g_list_next(tmplst);
        }
      g_list_free(field->options);
    }
}

gboolean scalarFieldLoad_fromFile(const gchar *filename, GList **fieldList,
				  GError **error, OptionTable *table)
{
  gboolean validFormat;
  GList *tmpLst;

  g_return_val_if_fail(filename, FALSE);
  g_return_val_if_fail(*fieldList == (GList*)0, FALSE);
  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);

  /* Try all supported format. */
  validFormat = FALSE;
  tmpLst = loadMethods;
  while(tmpLst && !validFormat)
    {
      DBG_fprintf(stderr, "Scalar Fields : try to open the file as a '%s'.\n",
		  ((ScalarFieldLoadStruct*)tmpLst->data)->name);
      validFormat = ((ScalarFieldLoadStruct*)tmpLst->data)->load(filename, fieldList, error, table);
      if (!validFormat && *error)
        {
          g_error_free(*error);
          *error = (GError*)0;
        }
      tmpLst = g_list_next(tmpLst);
    }
  
  if (!validFormat)
    g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_UNKNOWN_FORMAT, 
		_("WARNING : unknown density/potential format.\n")); 
  return validFormat;
}


static gboolean scalarFieldLoad_fromAscii(const gchar *filename, GList **fieldList, GError **error,
					  OptionTable *table)
{
  FILE *in;
  char rep[MAX_LINE_LENGTH], format[MAX_LINE_LENGTH];
  gchar *comment;
  int res, i, j, k;
  int size[3];
  float box[6];
  ScalarField *field;

  g_return_val_if_fail(filename, FALSE);
  g_return_val_if_fail(*fieldList == (GList*)0, FALSE);
  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);

  DBG_fprintf(stderr, "ScalarField : try to read '%s' as a ASCII scalar"
	      " field data file.\n", filename);

  in = fopen(filename, "r");
  if (!in)
    {
      g_set_error(error, G_FILE_ERROR, G_FILE_ERROR_ACCES,
		  _("WARNING : impossible to open the file.\n"));
      return FALSE;
    }

  /* 1st line (comment) */
  (void)fgets(rep, MAX_LINE_LENGTH, in);
  rep[strlen(rep)-1] = 0; /* skipping \n */
  comment = getStringInUTF8(rep);
   
  (void)fgets(rep, MAX_LINE_LENGTH, in);
  res = sscanf(rep, "%d %d %d", size, size + 1, size + 2);
  if (res != 3)
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }

  (void)fgets(rep, MAX_LINE_LENGTH, in);
  res = sscanf(rep, "%f %f %f", box, box + 1, box + 2);
  if (res != 3)
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }
  (void)fgets(rep, MAX_LINE_LENGTH, in); 
  res = sscanf(rep, "%f %f %f", box + 3, box + 4, box + 5);
  if (res != 3)
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }

  (void)fgets(rep, MAX_LINE_LENGTH, in);
  res = sscanf(rep, "%s", format);
  if (res != 1 || (strcmp(format, "xyz") && strcmp(format, "zyx")))
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }

  /* OK, from now on, the format is supposed to be ASCII. */
  field = scalarFieldNew(filename);
  if (!field)
    {
      g_warning("impossible to create a ScalarField object.");
      g_free(comment);
      fclose(in);
      return FALSE;
    }
  scalarFieldSet_commentary(field, comment);
  scalarFieldSet_gridSize(field, size);
  scalarFieldSet_box(field, box);
  *fieldList = g_list_append(*fieldList, (gpointer)field);
 
  /* TODO change the error flag. */
  field->min = G_MAXFLOAT;
  field->max = G_MINFLOAT;
  if(!strcmp(format, "xyz"))
    {   
      for ( k = 0; k < size[2]; k++ ) 
	for ( j = 0; j < size[1]; j++ ) 
	  for ( i = 0; i < size[0]; i++ ) 
	    {
	      res = fscanf(in, "%lf", &field->data[i][j][k]);
	      if (res != 1)
		{
		  g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			      _("WARNING : impossible to read the density value.\n"));
		  fclose(in);
		  return TRUE;
		}
	      field->min = MIN(field->data[i][j][k], field->min);
	      field->max = MAX(field->data[i][j][k], field->max);
	    }
    }
  else
    {
      for ( i = 0; i < size[0]; i++ ) 
	for ( j = 0; j < size[1]; j++ ) 
	  for ( k = 0; k < size[2]; k++ ) 
	    {
	      res = fscanf(in, "%lf", &field->data[i][j][k]);
	      if (res != 1)
		{
		  g_set_error(error, VISU_FILEFORMAT_ERROR, FILEFORMAT_ERROR_FORMAT,
			      _("WARNING : impossible to read the density value.\n"));
		  fclose(in);
		  return TRUE;
		}
	      field->min = MIN(field->data[i][j][k], field->min);
	      field->max = MAX(field->data[i][j][k], field->max);
	    }
    }
  
  fclose(in);
  return TRUE;
}

gchar* scalarFieldGet_commentary(ScalarField *field)
{
  g_return_val_if_fail(field, (gchar*)0);
  return field->commentary;
}
void scalarFieldSet_commentary(ScalarField *field, gchar* comment)
{
  g_return_if_fail(field);
  
  field->commentary = g_strdup(comment);
}


gchar* scalarFieldGet_filename(ScalarField *field)
{
  g_return_val_if_fail(field, (gchar*)0);
  return field->filename;  
}

void scalarFieldSet_gridSize(ScalarField *field, int grid[3])
{
  int i, j;
  
  g_return_if_fail(field);
  
  if (field->nElements[0] == grid[0] &&
      field->nElements[1] == grid[1] &&
      field->nElements[2] == grid[2])
    return;
  
  DBG_fprintf(stderr, "ScalarField : changing size from (%d ; %d ; %d)"
                      " to (%d ; %d ; %d).\n", field->nElements[0], field->nElements[1],
                      field->nElements[2], grid[0], grid[1], grid[2]);
  /* If data was already allocated, we free it. */
  if (field->data)
    {
      for (i = 0; i < field->nElements[0]; i++)
	      {
          for (j = 0; j < field->nElements[1]; j++)
	          g_free(field->data[i][j]);
      	  g_free(field->data[i]);
      	}
      g_free(field->data);
    }
    
  /* We change the size and reallocate data. */
  field->nElements[0] = grid[0];
  field->nElements[1] = grid[1];
  field->nElements[2] = grid[2];

  DBG_fprintf(stderr, "ScalarField : allocating array.\n");
  field->data = g_malloc(sizeof(double **) * grid[0]);
  for(i = 0; i < grid[0]; i++)
    {
      field->data[i] = g_malloc(sizeof(double *) * grid[1]);
      for(j = 0; j < grid[1]; j++)
        field->data[i][j] = g_malloc(sizeof(double) * grid[2]);
    }
}

void scalarFieldGet_minMax(ScalarField *field, double minmax[2])
{
  g_return_if_fail(field);
  
  minmax[0] = field->min;
  minmax[1] = field->max;
}
void scalarFieldSet_data(ScalarField *field, double *data)
{
  int i, j, k, ii;
  
  g_return_if_fail(field && data);
  
  field->min = G_MAXFLOAT;
  field->max = -G_MAXFLOAT;
  ii = 0;
  for (k = 0 ; k < field->nElements[2] ; k++)
    for (j = 0 ; j < field->nElements[1] ; j++)
      for (i = 0 ; i < field->nElements[0] ; i++)
        {
          field->data[i][j][k] = data[ii];
          ii += 1;
	  field->min = MIN(field->data[i][j][k], field->min);
	  field->max = MAX(field->data[i][j][k], field->max);
        }
}
void scalarFieldDraw_map(ScalarField *field, Plane *plane, Shade *shade,
                         OpenGLView *view, float precision, float drawnMinMax[2])
{
  float xyz[2][3], normal[3], spherical[3], center[3];
  float xyzMax[3], dist, Xcoeff, Ycoeff, coeff;
  float ***values;
  double value, minmax[2];
  GList *inter;
  int alpha, i, j, k;
  float rgba[4][4];
  int posRgba;
  
  g_return_if_fail(field && shade && plane);
  g_return_if_fail(view && precision > 0);

  planeGet_nVectUser(plane, normal);
  matrix_cartesianToSpherical(spherical, normal);

  planeGet_barycentre(plane, center);
  /* We compute the min and max in the 3D basis. */
  inter = planeGet_intersection(plane);
  /* In case of no intersections, we build an empty list. */
  if (!inter)
    return;
  
  dist = 0.;
  while(inter)
    {
      xyzMax[0] = ((float*)inter->data)[0] - center[0];
      xyzMax[1] = ((float*)inter->data)[1] - center[1];
      xyzMax[2] = ((float*)inter->data)[2] - center[2];
      dist = MAX(dist, xyzMax[0] * xyzMax[0] + xyzMax[1] * xyzMax[1] + xyzMax[2] * xyzMax[2]);
      inter = g_list_next(inter);
    }

  /* We compute the plane basis. */
  xyz[0][0] = cos(spherical[1] * PI180) * cos(spherical[2] * PI180);
  xyz[0][1] = cos(spherical[1] * PI180) * sin(spherical[2] * PI180);
  xyz[0][2] = -sin(spherical[1] * PI180);
  xyz[1][0] = -sin(spherical[2] * PI180);
  xyz[1][1] =  cos(spherical[2] * PI180);
  xyz[1][2] = 0.;

  /* We transform the 3D min and max values in the plane basis. */
  dist = sqrt(dist);
  alpha = (int)(20. * (OpenGLViewGet_precision() / 2. + .5) *
                precision * (view->camera->gross / 2. + .5) / 100.);
  scalarFieldGet_minMax(field, minmax);
  drawnMinMax[0] = minmax[1];
  drawnMinMax[1] = minmax[0];

/*   DBG_fprintf(stderr, "Panel Map : storing values.\n"); */
  values = g_malloc(sizeof(float**) * 2 * alpha);
  for (i = -alpha; i < alpha; i++)
    {
      values[i + alpha] = g_malloc(sizeof(float*) * 2 * alpha);
      for (j = -alpha; j < alpha; j++)
	/* The first value if 0 if the point is not used,
	   the next three are the coordinates,
	   the fourth is the scalar field value,
	   and the next 4 values are a storage for RGBA values. */
	values[i + alpha][j + alpha] = g_malloc(sizeof(float) * 9);
    }
  /* We use an hexagonal padding for equilateral triangles. */
  coeff =  dist / (float)alpha * 1.154700538379251448;
  for (i = 0; i < 2 * alpha; i++)
    {
      Ycoeff = (float)(i - alpha) * coeff * 0.8660254; /* the num coef is cos 30 */
      for (j = 0; j < 2 * alpha; j++)
	{
	  if (i%2)
	    Xcoeff = (float)(j - alpha) * coeff;
	  else
	    Xcoeff = ((float)(j - alpha) + 0.5) * coeff;
	  values[i][j][1] = center[0] + Xcoeff * xyz[0][0] + Ycoeff * xyz[1][0];
	  values[i][j][2] = center[1] + Xcoeff * xyz[0][1] + Ycoeff * xyz[1][1];
	  values[i][j][3] = center[2] + Xcoeff * xyz[0][2] + Ycoeff * xyz[1][2];
	  values[i][j][0] = (float)scalarFieldGet_value(field, &values[i][j][1], &value);
	  if ((gboolean)values[i][j][0])
	    {
	      values[i][j][4] = (float)((value - minmax[0]) / (minmax[1] - minmax[0]));
	      shadeGet_valueTransformedInRGB(shade, &values[i][j][5], values[i][j][4]);
	      drawnMinMax[0] =MIN(drawnMinMax[0], (float)value);
	      drawnMinMax[1] =MAX(drawnMinMax[1], (float)value);
	    }
/* 	  DBG_fprintf(stderr, " | %d (%g, %g, %g) -> %g\n", (gboolean)values[i][j][0], */
/* 		      values[i][j][1], values[i][j][2], values[i][j][3], values[i][j][4]); */
	}
    }

  glDisable(GL_CULL_FACE);
  glDisable(GL_LIGHTING);
  glPushMatrix();
  glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);
/*   glBegin(GL_LINES); */
/*   glVertex3d(center[0] - dist * xyz[0][0] - dist * xyz[1][0], */
/* 	     center[1] - dist * xyz[0][1] - dist * xyz[1][1], */
/* 	     center[2] - dist * xyz[0][2] - dist * xyz[1][2]); */
/*   glVertex3d(center[0] + dist * xyz[0][0] - dist * xyz[1][0], */
/* 	     center[1] + dist * xyz[0][1] - dist * xyz[1][1], */
/* 	     center[2] + dist * xyz[0][2] - dist * xyz[1][2]); */
/*   glVertex3d(center[0] + dist * xyz[0][0] - dist * xyz[1][0], */
/* 	     center[1] + dist * xyz[0][1] - dist * xyz[1][1], */
/* 	     center[2] + dist * xyz[0][2] - dist * xyz[1][2]); */
/*   glVertex3d(center[0] + dist * xyz[0][0] + dist * xyz[1][0], */
/* 	     center[1] + dist * xyz[0][1] + dist * xyz[1][1], */
/* 	     center[2] + dist * xyz[0][2] + dist * xyz[1][2]); */
/*   glVertex3d(center[0] + dist * xyz[0][0] + dist * xyz[1][0], */
/* 	     center[1] + dist * xyz[0][1] + dist * xyz[1][1], */
/* 	     center[2] + dist * xyz[0][2] + dist * xyz[1][2]); */
/*   glVertex3d(center[0] - dist * xyz[0][0] + dist * xyz[1][0], */
/* 	     center[1] - dist * xyz[0][1] + dist * xyz[1][1], */
/* 	     center[2] - dist * xyz[0][2] + dist * xyz[1][2]); */
/*   glVertex3d(center[0] - dist * xyz[0][0] + dist * xyz[1][0], */
/* 	     center[1] - dist * xyz[0][1] + dist * xyz[1][1], */
/* 	     center[2] - dist * xyz[0][2] + dist * xyz[1][2]); */
/*   glVertex3d(center[0] - dist * xyz[0][0] - dist * xyz[1][0], */
/* 	     center[1] - dist * xyz[0][1] - dist * xyz[1][1], */
/* 	     center[2] - dist * xyz[0][2] - dist * xyz[1][2]); */
/*   glEnd(); */

  /* The rgba[3][:] are the average color for the last three vertex. */
  for (i = 0; i < 4; i++)
    for (j = 0; j < 4; j++)
      rgba[i][j] = 0.;
  posRgba = 0;
  for (i = 0; i < 2 * alpha - 1; i++)
    {
      glBegin(GL_TRIANGLE_STRIP);
      for (j = 0; j < 2 * alpha - 1; j++)
	{
	  if ((gboolean)values[i + (i + 1)%2][j][0])
	    {
	      for (k = 0; k < 3; k++)
		{
		  rgba[3][k] += (values[i + (i + 1)%2][j][5 + k] - rgba[posRgba][k]) / 3.;
		  rgba[posRgba][k] = values[i + (i + 1)%2][j][5 + k];
		}
	      posRgba = (posRgba + 1)%3;
	      glColor3fv(rgba[3]);
	      glNormal3fv(normal);
	      glVertex3fv(&values[i + (i + 1)%2][j][1]);
	    }
	  if ((gboolean)values[i + i%2][j][0])
	    {
	      for (k = 0; k < 3; k++)
		{
		  rgba[3][k] += (values[i + i%2][j][5 + k] - rgba[posRgba][k]) / 3.;
		  rgba[posRgba][k] = values[i + i%2][j][5 + k];
		}
	      posRgba = (posRgba + 1)%3;
	      glColor3fv(rgba[3]); 
	      glNormal3fv(normal);
	      glVertex3fv(&values[i + i%2][j][1]);
	    }
	}
      glEnd();
    }
  glPopMatrix();  
  glEnable(GL_LIGHTING);
  glEnable(GL_CULL_FACE);

  for (i = 0; i < 2 * alpha; i++)
    {
      for (j = 0; j < 2 * alpha; j++)
	g_free(values[i][j]);
      g_free(values[i]);
    }
  g_free(values);
}

gboolean scalarFieldGet_value(ScalarField *field, float xyz[3], double *value)
{
  float redXyz[3], localXyz[3], pos;
  int l, ijk[3], dijk[3];

  g_return_val_if_fail(field, FALSE);
  
  /* First, we transform the coordinates into reduced coordinates. */
  matrix_productVector(redXyz, field->fromXYZtoReducedCoord, xyz);

  /* We compute i, j, k. */
  for (l = 0; l < 3; l++)
    {
      pos = (float)field->nElements[l] * redXyz[l];
      ijk[l] = (int)pos;
      if (ijk[l] < 0 || redXyz[l] < 0.)
	return FALSE;
      if (ijk[l] >= field->nElements[l])
	return FALSE;
      localXyz[l] = pos - (float)ijk[l];
    }

  /* We linearly interpolate the value. */
  *value = field->data[ijk[0]][ijk[1]][ijk[2]];
  for (l = 0; l < 3; l++)
    {
      dijk[0] = ijk[0];
      dijk[1] = ijk[1];
      dijk[2] = ijk[2];
      if (ijk[l] == field->nElements[l] - 1)
	dijk[l] = 0;
      else
	dijk[l] += 1;
      *value += (field->data[dijk[0]][dijk[1]][dijk[2]] -
		 field->data[ijk[0]][ijk[1]][ijk[2]]) * localXyz[l];
    }
  return TRUE;
}
void scalarFieldSet_box(ScalarField *field, float box[6])
{
  int i;

  g_return_if_fail(field);

  DBG_fprintf(stderr, "ScalarField : set the bouding box.\n");

  /* Change the box. */
  for (i = 0; i < 6; i++)
    field->box[i] = box[i];

  /* Update the transformation matrix. */
  field->fromXYZtoReducedCoord[0][0] =
    1 / field->box[VISU_DATA_BOX_DXX];
  field->fromXYZtoReducedCoord[0][1] =
    - field->box[VISU_DATA_BOX_DYX] /
    field->box[VISU_DATA_BOX_DXX] /
    field->box[VISU_DATA_BOX_DYY];
  field->fromXYZtoReducedCoord[0][2] =
    - (field->box[VISU_DATA_BOX_DZX] /
       field->box[VISU_DATA_BOX_DXX] -
       field->box[VISU_DATA_BOX_DYX] *
       field->box[VISU_DATA_BOX_DZY] / 
       field->box[VISU_DATA_BOX_DXX] / 
       field->box[VISU_DATA_BOX_DYY] ) /
    field->box[VISU_DATA_BOX_DZZ];
  field->fromXYZtoReducedCoord[1][0] = 0.;
  field->fromXYZtoReducedCoord[1][1] =
    1 / field->box[VISU_DATA_BOX_DYY];
  field->fromXYZtoReducedCoord[1][2] =
    - field->box[VISU_DATA_BOX_DZY] /
    field->box[VISU_DATA_BOX_DYY] /
    field->box[VISU_DATA_BOX_DZZ];
  field->fromXYZtoReducedCoord[2][0] = 0.;
  field->fromXYZtoReducedCoord[2][1] = 0.;
  field->fromXYZtoReducedCoord[2][2] = 1 /
    field->box[VISU_DATA_BOX_DZZ];
}
void scalarFieldGet_box(ScalarField *field, float box[6])
{
  int i;

  g_return_if_fail(field);

  for (i = 0; i < 6; i++)
    box[i] = field->box[i];
}
void scalarFieldGet_gridSize(ScalarField *field, int grid[3])
{
  int i;

  g_return_if_fail(field);

  for (i = 0; i < 3; i++)
    grid[i] = field->nElements[i];
}
double*** scalarFieldGet_data(ScalarField *field)
{
  g_return_val_if_fail(field, (double***)0);

  return field->data;
}

void scalarFieldSet_fitToBox(VisuData *data, ScalarField *field)
{
  float box[6];
  int i;

  g_return_if_fail(data);

  DBG_fprintf(stderr, "ScalarField : change the current box to fit to %p.\n", (gpointer) data);
  for (i = 0; i < 6; i++)
    box[i] = visuDataGet_boxGeometry(data, i);
  scalarFieldSet_box(field, box);
}  
GList* scalarFieldGet_allOptions(ScalarField *field)
{
  g_return_val_if_fail(field, (GList*)0);
  
  return g_list_copy(field->options);
}
void scalarFieldAdd_option(ScalarField *field, Option *option)
{
  g_return_if_fail(field && option);
  
  field->options = g_list_append(field->options, (gpointer)option);
}


/* Load method handling. */
void scalarFieldInit()
{
  char *type[] = {"*.pot", "*.dat", (char*)0};
  char *descr = _("Potential/density files");
  FileFormat *fmt;
  
  loadMethods = (GList*)0;
  
  fmt = fileFormatNew(descr, type);
  scalarFieldAdd_loadMethod("Plain ascii", scalarFieldLoad_fromAscii,
			    fmt, G_PRIORITY_LOW);
}

void scalarFieldAdd_loadMethod(const gchar *name, ScalarFieldLoadMethod method,
			       FileFormat *format, int priority)
{
  ScalarFieldLoadStruct *meth;

  g_return_if_fail(name && method && format);

  meth = g_malloc(sizeof(ScalarFieldLoadStruct));
  meth->name = g_strdup(name);
  meth->fmt = format;
  meth->load = method;
  meth->priority = priority;

  loadMethods = g_list_prepend(loadMethods, meth);
  loadMethods = g_list_sort(loadMethods, compareLoadPriority);
}

static gint compareLoadPriority(gconstpointer a, gconstpointer b)
{
  if (((ScalarFieldLoadStruct*)a)->priority <
      ((ScalarFieldLoadStruct*)b)->priority)
    return (gint)-1;
  else if (((ScalarFieldLoadStruct*)a)->priority >
	   ((ScalarFieldLoadStruct*)b)->priority)
    return (gint)+1;
  else
    return (gint)0;
}
GList* scalarFieldGet_allLoadMethods()
{
  return loadMethods;
}
