/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "scale.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include <opengl.h>
#include <openGLFunctions/text.h>
#include <visu_object.h>
#include <visu_tools.h>
#include <visu_configFile.h>
#include <openGLFunctions/objectList.h>
#include <coreTools/toolColor.h>
#include <coreTools/toolMatrix.h>
#include <math.h>


/* Parameters & resources*/
/* This is a boolean to control is the axes is render or not. */
#define FLAG_RESOURCE_SCALE_USED   "scales_are_on"
#define DESC_RESOURCE_SCALE_USED   "Control if scales are drawn ; boolean (0 or 1)"
#define RESOURCE_SCALE_USED_DEFAULT 0
static gboolean readScaleIsOn(gchar **lines, int nbLines,
			      int position, GString *errorMessage);

/* A resource to control the color used to render the lines of the Scale. */
#define FLAG_RESOURCE_SCALE_COLOR   "scales_color"
#define DESC_RESOURCE_SCALE_COLOR   "Define the color of all scales ; three floating point values (0. <= v <= 1.)"
static float scaleRGB[3];
static float scaleRGBDefault[3] = {0.0, 0.0, 0.0};
static gboolean readScaleColor(gchar **lines, int nbLines,
			       int position, GString *errorMessage);

/* A resource to control the width to render the lines of the Scale. */
#define FLAG_RESOURCE_SCALE_LINE   "scales_line_width"
#define DESC_RESOURCE_SCALE_LINE   "Define the width of the lines of all scales ; one floating point values (1. <= v <= 10.)"
#define RESOURCE_SCALE_LINE_DEFAULT 1.
static float scaleLineWidth;
static gboolean readscaleLineWidth(gchar **lines, int nbLines,
				   int position, GString *errorMessage);

/* A resource to control the elements of a scale (origin, direction, length... */
#define FLAG_RESOURCE_SCALE_DEFINITION "scale_definition"
#define DESC_RESOURCE_SCALE_DEFINITION "Define the position, the direction, the length and the legend of a scale ; position[3] direction[3] length legend"
/* Position of the scale origin.*/
static float origin[3] = {0.0, 0.0, 0.0};
/* Direction of the scale in space. */
static float direction[3] = {1.0, 0.0, 0.0};
/* Length of drawn scale. */
static float length = 5.;
/* Drawn legend. */
static GString *legend;
static gboolean readScaleDefinition(gchar **lines, int nbLines,
				    int position, GString *errorMessage);

/* Export function that is called by visu_module to write the
   values of resources to a file. */
static gboolean exportResourcesScale(GString *data, int *nbLinesWritten, VisuData *dataObj);


/* This routine is the callback of the extension when it must be
   rebuilt. */
static void rebuildScale(VisuData *dataObj);

/* This routine is responsible for the scale drawing. */
static void LigneDraw(VisuData *dataObj,OpenGLView *view,char*legend);

void rectDraw(VisuData *dataObj,OpenGLView *view,char*legend);  /*rectangle*/

/* Routine to set the default values (colour, origin, length... */
static void scaleSet_defaults();

/* This flag can be set internally to specify that Scale should be recreated. */
static gboolean scaleHasBeenBuilt;

/* Local callbacks */
static void onScaleParametersChange(VisuData *dataObj, OpenGLView *view, gpointer data);
static void rebuildScaleOnResourcesLoaded(GObject *visu, gpointer data);
static void onDataReadySignal(GObject *visu, VisuData *dataObj, gpointer data);
static void onDataNewSignal(GObject *visu, VisuData *dataObj, gpointer data);

OpenGLExtension* initExtensionScale()
{
  char *name = "Scale";
  char *description = _("Draw a scale in the rendering area.");
  int listScale;
  VisuConfigFileEntry *resourceEntry;

  DBG_fprintf(stderr,"Initialising the Scale OpenGL extension...\n");
  listScale = openGLObjectList_new(1);
  extensionScale = OpenGLExtension_new(name, description, listScale, rebuildScale);

  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_SCALE_USED,
					  DESC_RESOURCE_SCALE_USED,
					  1, readScaleIsOn);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_SCALE_COLOR,
					  DESC_RESOURCE_SCALE_COLOR,
					  1, readScaleColor);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_SCALE_LINE,
					  DESC_RESOURCE_SCALE_LINE,
					  1, readscaleLineWidth);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_SCALE_DEFINITION,
					  DESC_RESOURCE_SCALE_DEFINITION,
					  1, readScaleDefinition);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesScale);

  /* Set default values. */
  scaleSet_RGBValues(scaleRGBDefault, MASK_RGB_ALL);
  extensionScale->used = RESOURCE_SCALE_USED_DEFAULT;
  OpenGLExtensionSet_saveOpenGLState(extensionScale, TRUE);
  scaleLineWidth = RESOURCE_SCALE_LINE_DEFAULT;
  scaleSet_defaults();


  g_signal_connect(G_OBJECT(visu), "dataReadyForRendering",
		   G_CALLBACK(onDataReadySignal), (gpointer)0);
  g_signal_connect(G_OBJECT(visu), "dataNew",
		   G_CALLBACK(onDataNewSignal), (gpointer)0);

  g_signal_connect(G_OBJECT(visu), "resourcesLoaded",
 		   G_CALLBACK(rebuildScaleOnResourcesLoaded), (gpointer)0);


  scaleHasBeenBuilt = FALSE;
  
  return extensionScale;
}

/* Method used to change the value of the parameter scale_color. */
int scaleSet_RGBValues(float rgb[3], int mask)
{
  int diff = 0;
  
  if (mask & MASK_RGB_R && scaleRGB[0] != rgb[0])
    {
      scaleRGB[0] = rgb[0];
      diff = 1;
    }
  if (mask & MASK_RGB_G && scaleRGB[1] != rgb[1])
    {
      scaleRGB[1] = rgb[1];
      diff = 1;
    }
  if (mask & MASK_RGB_B && scaleRGB[2] != rgb[2])
    {
      scaleRGB[2] = rgb[2];
      diff = 1;
    }
  if (!diff)
    return 0;

  /* Set the building tag to false. */
  scaleHasBeenBuilt = FALSE;
  return extensionScale->used;
}

/* Method used to change the value of the parameter scale_line_width. */
int scaleSet_lineWidth(float width)
{
  if (width < 1. || width > 10. || width == scaleLineWidth)
    return 0;

  scaleLineWidth = width;
  /* Set the building tag to false. */
  scaleHasBeenBuilt = FALSE;
  return extensionScale->used;
}

/* Method used to change the value of the parameter scale_is_on. */
int scaleSet_isOn(int value)
{
  if (value == extensionScale->used)
    return 0;

  DBG_fprintf(stderr, "Scale: set scale status, %d.\n", value);
  extensionScale->used = value;
  return (value && !scaleHasBeenBuilt);
}

/* Get methods. */
float scaleGet_redValue()
{
  return scaleRGB[0];
}

float scaleGet_greenValue()
{
  return scaleRGB[1];
}

float scaleGet_blueValue()
{
  return scaleRGB[2];
}

int scaleGet_isOn()
{
  return extensionScale->used;
}

float scaleGet_lineWidth()
{
  return scaleLineWidth;
}

float scaleGet_length()
{
  return length;
}

void scaleGet_origin(float xyz[3])
{
  xyz[0] = origin[0];
  xyz[1] = origin[1];
  xyz[2] = origin[2];
}

void scaleGet_orientation(float xyz[3])
{
  xyz[0] = direction[0];
  xyz[1] = direction[1];
  xyz[2] = direction[2];
}

void scaleSet_defaults() /* donne des valeurs par defaut a tous les points */
{
  origin[0] = 0.;
  origin[1] = 0.;
  origin[2] = 0.;
  direction[0] = 1.;
  direction[1] = 0.;
  direction[2] = 0.;
  length = 5.;
  legend = g_string_new("");
  g_string_printf(legend, _("Length: %6.2f"), length);
}

/* Set the coordinates of scale origin. */
int scaleSet_origin(float xyz[3], int mask)
{
  int difference; /*permet de savoir s'il y a u des modification*/

  difference= 0;

  if (mask & MASK_XYZ_X && origin[0] != xyz[0])
    {
      origin[0] = xyz[0];
      difference = 1;
    }
  if (mask & MASK_XYZ_Y && origin[1] != xyz[1])
    {
      origin[1] = xyz[1];
      difference = 1;
    }
  if (mask & MASK_XYZ_Z && origin[2] != xyz[2])
    {
      origin[2] = xyz[2];
      difference = 1;
    }
  if (!difference)
    return 0;

  scaleHasBeenBuilt = FALSE;
  return (difference && extensionScale->used);
}

/* Set the direction of the scale. */
int scaleSet_orientation(float xyz[3], int mask)
{
  int difference; /*permet de savoir s'il y a u des modifications*/

  difference= 0;

  if (mask & MASK_XYZ_X && direction[0] != xyz[0])
    {
      direction[0] = xyz[0];
      difference = 1;
    }
  if (mask & MASK_XYZ_Y && direction[1] != xyz[1])
    {
      direction[1] = xyz[1];
      difference = 1;
    }
  if (mask & MASK_XYZ_Z && direction[2] != xyz[2])
    {
      direction[2] = xyz[2];
      difference = 1;
    }
  if (!difference)
    return 0;

  scaleHasBeenBuilt = FALSE;
  return (difference && extensionScale->used);
}

int scaleSet_length(float lg) /* saisie de la distance de l'scale*/
{
  if (lg == length)
    return 0;
  else
    {
      length = lg;
      g_string_printf(legend, _("Length: %6.2f"), length);
      scaleHasBeenBuilt = FALSE;
      return extensionScale->used;
    }
}

void scaleSet_legend(const gchar *value)/* pour modifier le texte de la legende*/
{
  g_string_printf(legend, "%s", value);
}


/****************/
/* Private part */
/****************/

void rebuildScale(VisuData *dataObj)
{
  openGLText_rebuildFontList();
  /* Set flag to false to force recreation of Scale. */
  scaleHasBeenBuilt = FALSE;
  scaleDraw(dataObj);
}
static void rebuildScaleOnResourcesLoaded(GObject *visu, gpointer data)
{
  GList *tmpLst;

  tmpLst = visuDataGet_allObjects();
  while (tmpLst)
    {
      scaleDraw(VISU_DATA(tmpLst->data));
      tmpLst = g_list_next(tmpLst);
    }
}
static void onDataReadySignal(GObject *visu, VisuData *dataObj, gpointer data)
{
  if (!dataObj)
    return;

  scaleHasBeenBuilt = FALSE;
  scaleDraw(dataObj);
}
static void onDataNewSignal(GObject *visu, VisuData *dataObj, gpointer data)
{
  g_signal_connect(G_OBJECT(dataObj), "OpenGLNearFar",
		   G_CALLBACK(onScaleParametersChange), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "OpenGLWidthHeight",
		   G_CALLBACK(onScaleParametersChange), (gpointer)0);
}

static void onScaleParametersChange(VisuData *dataObj, OpenGLView *view, gpointer data)
{
  scaleHasBeenBuilt = FALSE;
  scaleDraw(dataObj);
}

void scaleDraw(VisuData *dataObj)
{
  /* Nothing to draw; */
  if(!scaleGet_isOn() || scaleHasBeenBuilt) return;

  /* Nothing to draw if no data is associated to
     the current rendering window. */
  if (!dataObj)
    return;

  DBG_fprintf(stderr, "Extension Scale : creating scale.\n");

  scaleHasBeenBuilt = TRUE;

  openGLText_initFontList();
  
  glDeleteLists(extensionScale->objectListId, 1);

  glNewList(extensionScale->objectListId, GL_COMPILE);

  /* Deactivate light and fog. */
  glDisable(GL_LIGHTING);
  glDisable(GL_FOG);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  LigneDraw( dataObj, visuDataGet_openGLView(dataObj), legend->str);
/*   rectDraw(dataObj,(OpenGLView*)0,"legend"); */

  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  glEndList();
}

/* Parameters & resources*/

/* This is a boolean to control is the Scale is render or not. */
static gboolean readScaleIsOn(gchar **lines, int nbLines,
		       int position, GString *errorMessage)
{
  int res, val;

  res = sscanf(lines[0],"%d", &val);
  if (res != 1)
    {
      if (errorMessage)
	g_string_append_printf(errorMessage, _("WARNING! Parse error at line %d,"
					       " 1 boolean value must appear"
					       " after the %s markup.\n"),
			       position, FLAG_RESOURCE_SCALE_USED);
      scaleSet_isOn(RESOURCE_SCALE_USED_DEFAULT);
      return FALSE;
    }

  scaleSet_isOn(val);
  return TRUE;
}
/* A resource to control the color used to render the lines of the Scale. */
static gboolean readScaleColor(gchar **lines, int nbLines,
		       int position, GString *errorMessage)
{
  int res;
  float rgb[3];
  
  res = sscanf(lines[0],"%f %f %f", &rgb[0], &rgb[1], &rgb[2]);
  if (res != 3 ||
      visuConfigFileSet_floatValue(&rgb[0], rgb[0], 0., 1.) ||
      visuConfigFileSet_floatValue(&rgb[1], rgb[1], 0., 1.) ||
      visuConfigFileSet_floatValue(&rgb[2], rgb[2], 0., 1.))
    {
      if (errorMessage)
	g_string_append_printf(errorMessage, _("WARNING! Parse error at line %d, 3 floating "
					       "point values (0 <= v <= 1) must appear after"
					       " the %s markup.\n"),
			       position, FLAG_RESOURCE_SCALE_COLOR);
      scaleSet_RGBValues(scaleRGBDefault, MASK_RGB_ALL);
      return FALSE;
    }
  scaleSet_RGBValues(rgb, MASK_RGB_ALL);

  return TRUE;
}
/* A resource to control the width to render the lines of the Scale. */
static gboolean readscaleLineWidth(gchar **lines, int nbLines,
			   int position, GString *errorMessage)
{
  int res;
  float width;
  
  res = sscanf(lines[0],"%f", &width);
  if (res != 1 ||
      visuConfigFileSet_floatValue(&width, width, 1., 10.))
    {
      if (errorMessage)
	g_string_append_printf(errorMessage, _("WARNING! Parse error at line %d, 1 floating "
					       "point value (1 <= v <= 10) must appear after"
					       " the %s markup.\n"),
			       position, FLAG_RESOURCE_SCALE_COLOR);
      scaleSet_lineWidth(RESOURCE_SCALE_LINE_DEFAULT);
      return FALSE;
    }
  scaleSet_lineWidth(width);

  return TRUE;
}
static gboolean readScaleDefinition(gchar **lines, int nbLines,
				    int position, GString *errorMessage)
{
  int res, i, iToken;
  float xyz[3], orientation[3], len;
  gchar **tokens;
  gchar *remains;
  
  tokens = g_strsplit(g_strchug(lines[0]), " ", 0);
/*   fprintf(stderr, "'%s' '%s' '%s' '%s' '%s' '%s' '%s' '%s'\n", tokens[0], tokens[1], */
/* 	  tokens[2], tokens[3], tokens[4], tokens[5], tokens[6], tokens[7]); */

  iToken = 0;
  for (i = 0; i < 3; i++)
    {
      while (tokens[iToken] && !tokens[iToken][0]) iToken++;
      if (tokens[iToken])
	res = sscanf(tokens[iToken], "%f", &xyz[i]);
      else
	res = 0;
      iToken += 1;
      if (res != 1)
	{
	  if (errorMessage)
	    g_string_append_printf(errorMessage, _("WARNING! Parse error at line %d, "
						   "can't read scale definition.\n"),
				   position);
	  g_strfreev(tokens);
	  return FALSE;
	}
    }
  for (i = 0; i < 3; i++)
    {
      while (tokens[iToken] && !tokens[iToken][0]) iToken++;
      if (tokens[iToken])
	res = sscanf(tokens[iToken], "%f", &orientation[i]);
      else
	res = 0;
      iToken += 1;
      if (res != 1)
	{
	  if (errorMessage)
	    g_string_append_printf(errorMessage, _("WARNING! Parse error at line %d, "
						   "can't read scale definition.\n"),
				   position);
	  g_strfreev(tokens);
	  return FALSE;
	}
    }
  while (tokens[iToken] && !tokens[iToken][0]) iToken++;
  if (tokens[iToken])
    res = sscanf(tokens[iToken], "%f", &len);
  else
    res = 0;
  iToken += 1;
  if (res != 1)
    {
      if (errorMessage)
	g_string_append_printf(errorMessage, _("WARNING! Parse error at line %d, "
					       "can't read scale definition.\n"),
			       position);
      g_strfreev(tokens);
      return FALSE;
    }
  scaleSet_origin(xyz, MASK_XYZ_ALL);
  scaleSet_orientation(orientation, MASK_XYZ_ALL);
  scaleSet_length(len);
  if (tokens[iToken])
    {
      remains = g_strjoinv(" ", &tokens[iToken]);
      scaleSet_legend(g_strchomp(remains));
      g_free(remains);
    }
  g_strfreev(tokens);

  return TRUE;
}

/* Export function that is called by visu_module to write the
   values of resources to a file. */
static gboolean exportResourcesScale(GString *data, 
				     int *nbLinesWritten,
				     VisuData *dataObj)
{
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_SCALE_USED);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_SCALE_USED);
  g_string_append_printf(data, "    %d\n", extensionScale->used);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_SCALE_COLOR);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_SCALE_COLOR);
  g_string_append_printf(data, "    %4.3f %4.3f %4.3f\n",
	  scaleRGB[0], scaleRGB[1], scaleRGB[2]);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_SCALE_LINE);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_SCALE_LINE);
  g_string_append_printf(data, "    %4.0f\n", scaleLineWidth);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_SCALE_DEFINITION);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_SCALE_DEFINITION);
  g_string_append_printf(data, "    %g %g %g  %g %g %g  %g  %s\n",
			 origin[0], origin[1], origin[2],
			 direction[0], direction[1], direction[2],
			 length, legend->str);
  g_string_append_printf(data, "\n");
  *nbLinesWritten += 13;

  return TRUE;

}



/* fonction qui dessine la legende*/
static void LigneDraw(VisuData *dataObj, OpenGLView *view, char* legend)
{
  float x2,y2,z2, norm;
  float angles[3]; /*declaration des tableaux angles et coordonnées a 3 cellules*/
  float coord[3];
  int nlat;
  float radius = 0.3;

  glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);

  norm = length / sqrt(direction[0] * direction[0] + direction[1] * direction[1] +
		       direction[2] * direction[2]);
  x2 = origin[0] + direction[0] * norm;
  y2 = origin[1] + direction[1] * norm;
  z2 = origin[2] + direction[2] * norm;

  glLineWidth(scaleLineWidth);

  glBegin (GL_LINES); /*premier ligne */
  glColor4f(scaleRGB[0], scaleRGB[1], scaleRGB[2], 1.);
  glVertex3f(origin[0], origin[1], origin[2]);
  glVertex3f(x2, y2, z2);
  glEnd();

  coord[0] = x2 - origin[0];
  coord[1] = y2 - origin[1];
  coord[2] = z2 - origin[2];

  matrix_cartesianToSpherical(angles, coord); /*appel de la fonction pour transformé les angles en coordonnée*/

  nlat = OpenGLViewGet_numberOfFacettes(view, radius);

  glPushMatrix(); /*aide damien*/
  glTranslated(origin[0], origin[1], origin[2]); /*aide damien*/
  glRotated(angles[2], 0., 0., 1.);
  glRotated(angles[1], 0., 1., 0.);
  glTranslated(0., 0., length); /*aide damien*/
  glColor4f(scaleRGB[0], scaleRGB[1], scaleRGB[2], 1.);
  gluCylinder(obj, radius, 0., 1.0, nlat, 1); /* 20 NOMBRE DE FACETTES */
  glRotated(180., 1., 0., 0.);
  gluDisk(obj, 0, radius, nlat, 1);
  glPopMatrix();/*aide damien*/

  glColor4f(scaleRGB[0], scaleRGB[1], scaleRGB[2], 1.);
  glRasterPos3f(origin[0], origin[1], origin[2]);
  openGLText_drawChars(legend); /*affichage de la  legende*/
}



/* code pour second graphique*/

void rectDraw(VisuData *dataObj,OpenGLView *view,char*legend)  /*je declare mes variables*/
{
float len; /* variable pour definire la taille de la fenetre de dessin */
float e,f; /*variable des points pour la zone de dessin*/
float z;

view = visuDataGet_openGLView(dataObj);  /*je positionne la zone de dessin*/
len = 0.5f*view->box->extens/2;
  e = 0.20*view->window->width;
  f = 0.20*view->window->height; /*saisie des valeurs de la longueur et la largeur*/
z=len+4;
glViewport (0,0,e,f); /*saisie des cordonées de la fenetre*/


 /*debut code dessin premier rectangle*/

glBegin (GL_LINE_LOOP);
glColor4d(0,0,0,1);
glVertex3f(0,0,0);
glVertex3f(0,0,5.5);
glVertex3f(z,0.0,5.5);
glVertex3f(z,0.0,0.0);
glEnd();

glColor3i(0,0,0);  /*affichage de la legende*/
glRasterPos3f(0.0,0.0,0.0); openGLText_drawChars("c");
glRasterPos3f(0.0,0.0,5.5); openGLText_drawChars("a");
glRasterPos3f(z,0.0,5.5); openGLText_drawChars("b");
glRasterPos3f(z,0.0,0.0); openGLText_drawChars("d");

/*debut code dessin seconds rectangle*/

glBegin (GL_LINE_LOOP);
glColor4d(0,0,0,1);
glVertex3f(0,2,0);
glVertex3f(0,2,5.5);
glVertex3f(z,2,5.5);
glVertex3f(z,2,0);
glEnd();

/*code des lignelignes*/

glBegin (GL_LINES);
glColor4d(0,0,0,1);
glVertex3f(0,0,0);
glVertex3f(0,2,0);
glVertex3f(0,0,5.5);
glVertex3f(0,2,5.5);
glVertex3f(z,0,5.5);
glVertex3f(z,2,5.5);
glVertex3f(z,0,0);
glVertex3f(z,2,0);
glEnd();
}

