/* Chromium Unity integration extension
 * 
 *   Copyright 2012 Canonical Ltd.
 *
 *   This program is free software: you can redistribute it and/or modify it 
 *   under the terms of the GNU General Public License version 3, as published 
 *   by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but 
 *   WITHOUT ANY WARRANTY; without even the implied warranties of 
 *   MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
 *   PURPOSE.  See the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License along 
 *   with this program.  If not, see <http://www.gnu.org/licenses/>.
 **/

var background_page = (function () {
   
   // TODO extract based on logging settings
   var log = {
     info: function (msg) {
       if (msg) {
	 console.log ('Info: ' + msg);
       }
     }
     ,
     error: function (msg) {
       if (msg) {
	 console.log ('Error: ' + msg);
       }
     }
   };


   // connection & init w/ plugin NPAPI layer
   var plugin = document.getElementById ('unityChromiumExtensionId');
   if ( ! plugin) {
     log.error ("Unable to retrieve the Unity Chromium plugin");
     return null;
   }
   var service = plugin.service_new ();
   if (! service) {
     log.error ("Unable to retrieve a connection to Unity Webapps service");
     return null;
   }


   /////////////////////////////////////////////////////////
   // 
   // Handle login & connection to online-accounts project's
   // extension
   // 
   ////////////////////////////////////////////////////////
   var loginListeners = [];
   var onLoginListenerRequest = function (request, sender, sendResponse) {
     if (request.action == "registerLoginListener") {
       log.info ("Login listener request from " + sender.id);
       loginListeners.push (sender.id);
     }
   };
   chrome.extension.onRequestExternal.addListener (onLoginListenerRequest);

   var onLoginEvent = function (request, sender, sendResponse) {
     if (request.action == "loginEvent") {
       log.info ("Login on " + request.domain + ": " + request.login);
       var len = loginListeners.length;
       for (var i = 0; i < len; i++) {
         var extensionId = loginListeners[i];
         log.info("Notifying login listener: " + extensionId);
         chrome.extension.sendMessage (extensionId, {
                                         type: "login",
                                         login: request.login,
                                         domain: request.domain
                                       });
       }
     }
   };
   chrome.extension.onRequest.addListener (onLoginEvent);


   /////////////////////////////////////////////////////////
   // 
   // Scafolding to keep track of data associated w/ infobar requests 
   // (Chromium's structure imposes some kind of state being maintained
   // in order to communicate data) 
   // 
   ////////////////////////////////////////////////////////
   // 
   // list of callback that are to be called asynchronously
   //  per tab. Used in the user integration/installation resquests context.
   // 
   // One thing to keep in mind is that one constraint, that bring some amount of
   //  'soundness' is that there is a hard limit (provided by the browser) of one infobar per tab.

   var user_infobar_request_callbacks = {};
   var addInfobarRequestCallbackFor = function (infobarRequestId, callback, message, details) {
     user_infobar_request_callbacks[infobarRequestId] = {callback: callback
						         , message: message
                                                         , details: details
						        };
   };
   var getDataIfAnyFor = function (infobarRequestId) {
     if (user_infobar_request_callbacks[infobarRequestId] === undefined) {
       return "";
     }
     return { message: user_infobar_request_callbacks[infobarRequestId].message,
              details: user_infobar_request_callbacks[infobarRequestId].details };
   };
   var invokeAndRemoveCallbackIfAnyFor = function (infobarRequestId, arguments) {
     if (user_infobar_request_callbacks[infobarRequestId] === undefined) {
       return;
     }
     var callback = user_infobar_request_callbacks[infobarRequestId].callback;
     user_infobar_request_callbacks[infobarRequestId] = undefined;
     if (typeof(callback) === 'function') {
       callback(arguments);
     }
   };


   /////////////////////////////////////////////////////////
   // 
   // Extract installed userscripts & webapps info
   // 
   ////////////////////////////////////////////////////////
   var initializeIntegrationScriptRepositoryAccess = function (plugin) {

     var repo = plugin.application_repository_new_default ();
     plugin.application_repository_prepare (repo);
     return repo;
   };
   var repo_ = initializeIntegrationScriptRepositoryAccess(plugin);
   var repo_install_request_stamps = {};

   /**
    * Performs a match on the list of loaded integration scripts
    * given a url.
    *
    */
   var matchesIntegrationScripts = function (plugin, url, repo, windowInfos, callback) {
     function askForApplicationInstallation(plugin, appPackageName, url, installationCallback) {
       var appDomain = plugin.application_repository_get_resolved_application_domain(repo, appPackageName);

       if (plugin.permissions_get_domain_dontask(appDomain)) {
         log.info ("WebApp domain was found in the 'dont ask' list, won't install: " + appPackageName);
         installationCallback ();
         return;
       }
       if (repo_install_request_stamps[appPackageName] !== undefined) {
         log.info ('WebApp not being considered for install (request is only issues once per session): ' + appPackageName);
         installationCallback ();
         return;
       }
       repo_install_request_stamps[appPackageName] = true;

       function installApp() {
         log.info ('Installing application: ' + appPackageName);
         plugin.permissions_allow_domain(appDomain);
         plugin.application_repository_install_application(repo, appPackageName, installationCallback, null);
       }
       function addToIgnoreList() {
         plugin.permissions_dontask_domain(appDomain);
       }
       var isInstallationConfirmed = function (response) {
	  return response && response.integrate;
       };

       if (!plugin.permissions_get_domain_preauthorized(appDomain)) {
         var appName = plugin.application_repository_get_resolved_application_name(repo, appPackageName);
         addInfobarRequestCallbackFor (windowInfos.tabId,
                                       function (result) {
                                         log.info ('Asking for installation : ' + windowInfos.tabId);
                                         if (result === undefined) {
                                           installationCallback();
                                         }
                                         if (isInstallationConfirmed(result))
                                           installApp();
                                         else
                                           addToIgnoreList();
                                       },
	                               chrome.i18n.getMessage("integration_copy", [appName, appDomain]),
                                       null);
         chrome.infobars.show ({tabId: windowInfos.tabId, path: "infobar.html"});
       }
       else {
         plugin.permissions_allow_domain(appDomain);
         installApp();
       }

     } // function askForApplicationInstallation

     var APPLICATION_STATUS_AVAILABLE = 0;
     var APPLICATION_STATUS_INSTALLED = 1;
     var APPLICATION_STATUS_UNRESOLVED = 2;

     var formatInstalledAppInfo = function (name, src) {
       return {
         name: name,
         content: src,
         requires: null,
         includes: null
       };
     };

     /**
      * Gathers the lists of installed and (if any) available apps for a given list of app names
      * (available for a given url).
      * 
      * @returns object with "installed" and "available" list properties
      */
     var gatherApplicationsStatuses = function (names) {
       var installed = [];
       var available = [];
       for (var i = 0; i < names.length; ++i) {
         var status = plugin.application_repository_get_resolved_application_status (repo, names[i]);
         log.info ('A WebApp application has been found: ' + names[i] + ', status: ' + status);
         switch (status) {
           case APPLICATION_STATUS_INSTALLED:
             // we have found an installed application, use it directly
             var src = plugin.application_repository_get_userscript_contents (repo, names[i]);
             installed.push (formatInstalledAppInfo(names[i], src));
             break;
           case APPLICATION_STATUS_AVAILABLE:
             // we have found an application that can apply and be installed
             available.push (names[i]);
             break;
         }
       }
       return {
         installed: installed,
         available: available
       };
     };

     var names = JSON.parse(plugin.application_repository_resolve_url_as_json(repo_, url));
     if (null != names) {
       var scripts = gatherApplicationsStatuses(names);
       if (scripts && scripts.installed && scripts.installed.length > 0) {
         callback (scripts.installed);
         return;
       }

       // we should have at most one script for a given app
       if (scripts && scripts.available && scripts.available.length > 0) {
         var name = scripts.available[0];
         askForApplicationInstallation (plugin,
                                        name,
                                        url,
                                        function (_repository, name, status, data) {
                                          log.info ('Application installed: ' + name + ', (status ' + status + ')');
                                          if (name !== undefined && status == APPLICATION_STATUS_INSTALLED) {
                                            var src = plugin.application_repository_get_userscript_contents (repo_, name);
                                            scripts.installed.push (formatInstalledAppInfo(name, src));
                                          }
                                          callback (scripts.installed);
                                        }
                                       );
       }
       else  {
         callback (null);
       }
     } // if (null != names) {

   }; // var matchesIntegrationScripts = function (plugin, url, repo, windowInfos, callback) {


   /**
    * Handles tab change & activation code.
    * TODO(FIXME): better window tracking for active windows & tabs
    *   -> there are some glitches when a given webapp tab is moved around:
    *      1. open chromeless webapp w/ webapp (A)
    *      2. open normal window webapp (B) & unrelated other tab in same window
    *      3. undock (B) and dock it in (A): issues w/ activation (in laucnher)
    *      4. close the "first webapp" (A) -> actiating the webapp icon now
    *         refers back to first window, might be bc of BAMF window tracking
    *
    */
   chrome.tabs.onActivated.addListener (
     function (info) {
       // TODO(FIXME): pretty bad ... assumed that all tabs are interested
       chrome.windows.getAll (
         {populate: true}
	 , function (windows) {
	   for (var winIdx in windows) {
	     var tabs = windows[winIdx].tabs;
	     for (var tabIdx in tabs) {
	       var tabId = tabs[tabIdx].id;
	       chrome.tabs.sendRequest (tabId
					, {method: "on_tab_active_changed"
					   , tabId: info.tabId
					  }
					, function (response) {});
	     } // for (var tabIdx
	   } // for (var winIdx
	 });
     }
   );

   // TODO move elsewhere
   // TODO check that's it's fine (it should), since contrary to
   //      FF this is being called on the same environment every time (background page)
   function toDataURL (uri, callback, sx, sy, sw, sh) {
     if (uri.match(/^\s{0,}file:/i))
       return;
     
     if (uri.match(/^\s{0,}https:/i))
       return;
     
     var self = this;
     var canvas = document.createElementNS("http://www.w3.org/1999/xhtml", "canvas");
     var img = new window.Image();
     
     img.onload = function() {
       var width = img.width, height = img.height;

       var hasExtraArgs = sx || sx === 0;
       if (hasExtraArgs) {
	 width = sw;
	 height = sh;
       }

       var dx  = 0, dy = 0;

       if (width > height)
	 dy = width - height;
       else
	 dx = height - width;

       canvas.width = width + dx;
       canvas.height = height + dy;
       
       var ctx = canvas.getContext('2d');
       if (hasExtraArgs)
	 ctx.drawImage(img, sx, sy, sw, sh, dx / 2, dy / 2, width, height);
       else
	 ctx.drawImage(img, dx / 2, dy / 2);
       
       callback (true, canvas.toDataURL());
     };
     img.src = uri;
   };
   
   
   ////////////////////////////////////////
   // 
   // main request handler
   // 
   ///////////////////////////////////////
   
   /**
    * Handles & responds to content script requests.
    * 
    */
   var contentScriptsRequestHandler = function (request, sender, callback) {
     
     var handlers = {
       get_unity_script: function (request, sender, callback) {
	 
         var withMatchedScripts = function (scripts) {
	   var results = [];
	   if (scripts) {
             for (var i = 0; i < scripts.length; ++i) {
               var result = {
	         userscript: null
	         , requires: null
	       };
               result.userscript = scripts[i].content;
	       if (scripts[i].requires && scripts[i].requires.length !== 0) {
	         result.requires = scripts[i].requires;
	       }
	       results.push(result);
             }
	   }
	   callback (results);
         };
	 matchesIntegrationScripts (plugin,
                                    request.url,
                                    repo_,
                                    {
                                      windowId: sender.windowId,
                                      tabId: sender.tab.id
                                    },
                                    withMatchedScripts
                                   );
       }
       ,
       get_base_domain: function (request, sender, callback) {
	 
	 chrome.tld.getBaseDomain (request.url, callback);
       }
       ,
       add_integrationscript_for_tabid: function (request, sender, callback) {
         
         addIntegrationScriptForTab(sender.tab.id
                                    , request.name
                                    , request.domain
                                    , request.interest_id);
       }
       ,
       // TODO(alex-abreu): those two should be refactored and be cleaner
       on_user_infobar_request_result: function (request, sender, callback) {
	 
	 invokeAndRemoveCallbackIfAnyFor (request.tabId, request);
       }
       ,
       request_user_integration: function (request, sender, callback) {
	 
         // keep information about the context of the infobar request
	 addInfobarRequestCallbackFor (sender.tab.id, callback, request.message, request.details);
	 
	 chrome.infobars.show ({tabId: sender.tab.id, path: "infobar.html"});
       }
       , 
       take_snapshot: function (request, sender, callback) {
	 
	 chrome.tabs.captureVisibleTab (sender.tab.windowId
					, null
					, function (dataURL) {
					  callback (dataURL);
					});
       }
       , 
       get_data_url_from: function (request, sender, callback) {
	 
	 toDataURL (request.url
		    , function (flag, dataUrl) {
		      
		      callback ({flag: flag, dataUrl: dataUrl});
		    });
       }
       , 
       get_extension_settings: function (request, sender, callback) {
	 
	 var settings = {
	   logging: false,
           incognito: sender.tab.incognito
	 };
	 
	 try {
	   if (window.localStorage) {
	     settings.logging = localStorage['logging'];
	   }
	 }
	 catch (e) {
	   log.error ("Error while trying to retrieve logging information: " + str(e));
	 }
	 
	 callback (settings);
       }
       , 
       get_tab_info: function (request, sender, callback) {
	 
	 callback ({tabId: sender.tab.id, windowId: sender.tab.windowId});
       }
       , 
       kill_this_tab: function (request, sender, callback) {
	 
	 chrome.tabs.remove ([request.tabId], function () {});
       }
       , 
       raise_this_tab: function (request, sender, callback) {
	 
	 // bring the window & the tab to the front
	 chrome.windows.update (request.windowId
				, {focused: true}
				, function (window) {
				  chrome.tabs.update (request.tabId
						      , {active: true, highlighted: true});
				});
       }
       , 
       is_selected: function (request, sender, callback) {
	 
	 chrome.tabs.getCurrent (
	   function (tab) {
	     if (tab) {
	       callback ({isSelected: tab.id === request.tabId});
	     }
	   }
	 );
       }
     }; // handlers
     
     
     // validate request
     if ( ! request  || ! request.method) {
       callback ({error: "Invalid request structure"});
       return true;
     }
     if ( ! sender) {
       callback ({error: "Invalid sender"});
       return true;
     }
     if ( typeof (request.method) != 'string' || request.method.length == 0) {
       callback ({error: "Invalid request method"});
       return true;
     }
     
     log.info ('Got request: ' + request.method + ', ' + sender.tab.id);
     
     var handler = handlers [request.method];
     
     if (handler !== undefined && typeof(handler) == 'function') {
       
       log.info ('Calling handler for: ' + request.method);
       
       handler (request, sender, callback);
       
       return true;
     }
     return false;
   }; // var contentScriptsRequestHandler =

   // Main event handler and communication link
   //   w/ content scripts
   chrome.extension.onMessage.addListener (contentScriptsRequestHandler);


   ///////////////////////////////////////////////
   // 
   // window/tab management handling
   //
   ///////////////////////////////////////////////

   // Every tab that's being integrated as a webapp (chromeless or not)
   //  actually registers here. This allows tabs life cycle to be managed
   //  a bit when removed, or in chromeless mode when moved around.
   var integration_script_tabs = {};
   var addIntegrationScriptForTab = function (tabId
                                              , name
                                              , domain
                                              , interest_id) {
     if (integration_script_tabs[tabId] !== undefined) {
       log.info('Overriding tab information for existing tab: ' + tabId);
     }
     log.info ('Integrating tab with name: ' + name + ', for domain ' + domain);
     integration_script_tabs[tabId] = {name: name
                                       , domain: domain
                                       , interest_id: interest_id};
   };
   var removeIntegrationScriptForTab = function (tabId) {
     if (integration_script_tabs[tabId] !== undefined) {
       integration_script_tabs[tabId] = undefined;
     }
   };
   chrome.tabs.onRemoved.addListener (
     function (tabId, removeInfo) {
       var info = integration_script_tabs[tabId];
       removeIntegrationScriptForTab (tabId);
       if (info !== undefined) {
         log.info('Destroying interest: ' + info.name + ', ' + info.domain + ', ' + info.interest_id);
         try {
           plugin.service_destroy_interest_for_context (service
                                                        , info.name
                                                        , info.domain
                                                        , info.interest_id
                                                        , 1);
         }
         catch (e) {
           log.error('Error while trying to destroy interest for tab ' + tabId + ', ' + e);
         }
       }
     }
   );


   ///////////////////////////////////////////////////////////////////////
   // 
   // Window management related functions. In chromeless mode, we have specific
   //  rules for tab management to make webapps feel more "native" than plain
   //  web applications.
   // 
   ///////////////////////////////////////////////////////////////////////

   // BIG TODO: due to async specific of chrome api, all this is sensitive to timing
   //  issues / race conditions, so besides adding support for this directly in
   //  Chromium, one way to mitigate this would be to add support for the 'ischromeless'
   //  field in the tabs info structure retrieved from chrome API.
   // 

   /**
    * Validates the presence of a given tab in a chromeless window. The validation
    *  occurs based on specific rules:
    *    - for a given chromeless window, only tabs related to a gievn webapp should
    *       be allowed,
    *    - tabs can be 'undocked' but not 'external' tab can be docked in,
    * 
    * TODO: right now the scheme is pretty naive and brittle, fix that
    */
   var validateChromelessTabForWindowId =
     function (windowId, tabId, url, onInvalidTab) {
       // TODO bail out right away?
       var continuation =
         (onInvalidTab && typeof(onInvalidTab) == "function")
         ? onInvalidTab
         : function (tabId) { log.error ("onInvalidTab callback is invalid"); };

       // TODO use the new chrome.extension.getViews ({type: "tab"}) extension?
       // TODO(bis): mmmh one level of indirection again?
       chrome.windows.get(
         windowId
         , {populate: true}
         , function (win) {
           if (win.tabs.length == 1) {
             return;
           }
           var diff = win.tabs.some (
             function (curtab) {
               if (curtab.id == tabId) {
                 return false;
               }
               // TODO: usage of integration_script_tabs structure is bad
               var infos = integration_script_tabs[curtab.id];
               return infos
                 && infos.domain
                 && url.indexOf(infos.domain) == -1;
             }
           );
           if (diff) {
             // the tab corresponding to tabId
             // shoudldn't be there
             continuation(tabId);
           }
         }
       );
     }; // ) {

   var onTabChanged = function (tabId, windowId, url) {
     chrome.extension.isChromelessWindow(
       tabId
       , function (isChromeless) {
         if (!isChromeless) {
           return;
         }
         validateChromelessTabForWindowId(
           windowId
           , tabId
           , url
           , function (_tabId) {
             chrome.windows.create({tabId: tabId});
           }
         );
       } // function (isChromeless) {
     ); // chrome.extension.isChromelessWindow
   };
   chrome.tabs.onCreated.addListener(
     function(tab) {
       if (tab && tab.url) {
         onTabChanged(tab.id, tab.windowId, tab.url);
       } // if (tab && tab.url) {
     }
   );
   chrome.tabs.onUpdated.addListener (
     function(tabId, changeInfo, tab) {
       if (changeInfo && changeInfo.url) {
         onTabChanged(tabId, tab.windowId, changeInfo.url);
       }
     }
   );
   chrome.tabs.onAttached.addListener(
     function(tabId, attachInfo) {
       chrome.extension.isChromelessWindow(
           tabId
         , function (isChromeless) {
             if (!isChromeless) {
               return;
             }
             // retrieve info about the given tab
             // TODO: this get tiring bc of the sensitivity to
             //  race conditions (or equivalent) created by request lags
             chrome.tabs.get (
               tabId
               , function (tab) {
                   // Finally perform the actual validation or the
                   // new tab that got docked in
                   validateChromelessTabForWindowId(
                     attachInfo.newWindowId
                     , tabId
                     , tab.url
                     , function (tabId) {
                       chrome.windows.create({tabId: tab.id});
                     }
                   );
                 } // function (tab) {
             ); // chrome.tabs.get
         } // function (isChromeless) {
       ); // chrome.extension.isChromelessWindow
     }
   ); // tabs.onAttached

   return {
     /*
      *  Returns a potential message associated with a tab id (infobar)
      */
     getMessageForTabId: function (tabId)  {
       return getDataIfAnyFor (tabId).message;
     }
   };
 }) ();

