#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009-2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Utilities for finding and running a dbus session bus for testing."""

import os
import signal
import subprocess

from distutils.spawn import find_executable
from xdg.BaseDirectory import load_data_paths


class DBusLaunchError(Exception):
    """Error while launching dbus-daemon"""
    pass

class NotFoundError(Exception):
    """Not found error"""
    pass


class DBusRunner(object):
    """Class for running dbus-daemon with a private session."""

    def __init__(self):
        self.dbus_address = None
        self.dbus_pid = None
        self.running = False

    def _find_config_file(self):
        """Find the first appropriate dbus-session.conf to use."""
        # In case we're running from within the source tree
        path = os.path.join(os.path.dirname(__file__), "..", "..",
                            "data", "dbus-session.conf")
        if os.path.exists(path):
            return path

        # Use the installed file in $pkgdatadir
        for i in load_data_paths("ubuntuone-dev-tools", "dbus-session.conf"):
            if os.path.exists(i):
                return i


    def startDBus(self):
        """Start our own session bus daemon for testing."""
        dbus = find_executable("dbus-daemon")
        if not dbus:
            raise NotFoundError("dbus-daemon was not found.")

        config_file = self._find_config_file()

        dbus_args = ["--fork",
                     "--config-file=" + config_file,
                     "--print-address=1",
                     "--print-pid=2"]
        p = subprocess.Popen([dbus] + dbus_args,
                             bufsize=4096, stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE)

        self.dbus_address = "".join(p.stdout.readlines()).strip()
        self.dbus_pid = int("".join(p.stderr.readlines()).strip())

        if self.dbus_address != "":
            os.environ["DBUS_SESSION_BUS_ADDRESS"] = self.dbus_address
        else:
            os.kill(self.dbus_pid, signal.SIGKILL)
            raise DBusLaunchError("There was a problem launching dbus-daemon.")
        self.running = True

    def stopDBus(self):
        """Stop our DBus session bus daemon."""
        del os.environ["DBUS_SESSION_BUS_ADDRESS"]
        os.kill(self.dbus_pid, signal.SIGKILL)
        self.running = False

