#!/bin/sh
src=`pwd`
tmp=$src/tests-tmp
rm -rf $tmp
mkdir -p $tmp
PATH="$src:/bin:/usr/bin:/usr/local/bin"
tests_failed=0
tests_count=0
xw=$src/xfer-watch
uid=`id -u`
gid=`id -g`
home=$tmp/home
mkdir -p $home
cvm=$tmp/cvm
TCPLOCALIP=127.0.0.1
TCPREMOTEIP=127.0.0.1
LOGREQUESTS=1
SOFTCHROOT=1
export LOGREQUESTS TCPLOCALIP TCPREMOTEIP SOFTCHROOT

env="env UID=$uid GID=$gid USER=ftp GROUP=ftp HOME=$home TZ=Canada/Saskatchewan"

# Support functions

strippid() { sed -e 's/\[[0123456789]*\]/[PID]/'; }

echolines() {
  for line in "$@"
  do
    printf "$line\r\n"
  done
}

authcmd() { $src/twoftpd-auth "$@" 2>&1 </dev/null | strippid; }

auth() {
  local cvm=$1
  local backend=$2
  shift 2
  echolines "$@" | $src/twoftpd-auth $cvm $backend 2>&1 | strippid
}

makecvm() {
  (
    echo '#!/bin/sh'
    for line in "$@"; do echo "$line"; done
  ) >$cvm
  chmod 755 $cvm
}

anon() { echolines "$@" | $env $src/twoftpd-anon 2>/dev/null | tail +2; }

xfer() { echolines "$@" | $env $src/twoftpd-xfer 2>/dev/null | tail +2; }

testparams() {
  local backend=$1
  local command=$2
  shift 2
  local param
  for param in "$@"; do
    $backend "$command $param"
  done
}

xfertest() {
  local backend=$1
  local mode=$2
  local io=$3
  local conv=$4
  shift 4
  $env $src/xfer-test $src/twoftpd-$backend $mode $io $conv "$@" | tail +2
}


run_compare_test() {
  local name=$1
  shift
  cat >$tmp/expected
  ( runtest "$@" ) 2>&1 >$tmp/actual-raw
  cat -v $tmp/actual-raw >$tmp/actual
  if ! cmp $tmp/expected $tmp/actual >/dev/null 2>&1
  then
    echo "Test $name $@ failed:"
	( cd $tmp; diff -u9999 expected actual )
	let tests_failed=tests_failed+1
  fi
  rm -f $tmp/expected $tmp/actual
  let tests_count=tests_count+1
}

##### Test tests/auth-invocation #####

runtest() {
authcmd
authcmd 1
authcmd 1 2
authcmd 1 2 3
}
run_compare_test tests/auth-invocation  <<END_OF_TEST_RESULTS
421 Configuration error, insufficient paramenters.^M
421 Configuration error, insufficient paramenters.^M
220-TwoFTPd server ready.^M
220 Authenticate first.^M
220-TwoFTPd server ready.^M
220 Authenticate first.^M
END_OF_TEST_RESULTS


##### Test tests/auth-timeout #####

runtest() {
TIMEOUT=60 auth 1 2
TIMEOUT=00 auth 1 2
TIMEOUT=-6 auth 1 2
}
run_compare_test tests/auth-timeout  <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
421 Configuration error, invalid timeout value^M
421 Configuration error, invalid timeout value^M
END_OF_TEST_RESULTS


##### Test tests/auth-output #####

runtest() {
( echolines QUIT | $src/twoftpd-auth 1 2 >/dev/null; ) 2>&1 | strippid
echolines QUIT | $src/twoftpd-auth 1 2 2>/dev/null
}
run_compare_test tests/auth-output  <<END_OF_TEST_RESULTS
twoftpd-auth[PID]: QUIT
220-TwoFTPd server ready.^M
220 Authenticate first.^M
221 Bye.^M
END_OF_TEST_RESULTS


##### Test tests/auth-commands #####

runtest() {
local command="$1"
local parameter="$2"
if [ $parameter = 'no-parameter' ]; then
  cmd=$command
else
  cmd="$command xyzzy"
fi
auth foo foo "$cmd"
}
run_compare_test tests/auth-commands 'QUIT' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: QUIT xyzzy
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'QUIT' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: QUIT
221 Bye.^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'QuIt' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: QUIT xyzzy
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'QuIt' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: QUIT
221 Bye.^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'quit' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: QUIT xyzzy
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'quit' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: QUIT
221 Bye.^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'HELP' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: HELP xyzzy
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'HELP' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: HELP
502 No help is available.^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'SYST' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: SYST xyzzy
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'SYST' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: SYST
215 UNIX Type: L8^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'NOOP' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: NOOP xyzzy
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'NOOP' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: NOOP
200 Awaiting your commands, master...^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'USER' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER xyzzy
331 Send PASS.^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'USER' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'PASS' 'with-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: PASS XXXXXXXX
503 Send USER first.^M
END_OF_TEST_RESULTS

run_compare_test tests/auth-commands 'PASS' 'no-parameter' <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: PASS
504 Command requires a parameter^M
END_OF_TEST_RESULTS


##### Test tests/backend-greeting #####

runtest() {
local backend="$1"
$env $src/twoftpd-$backend 2>/dev/null
$env BANNER='foo
bar' $src/twoftpd-$backend 2>/dev/null
}
run_compare_test tests/backend-greeting 'anon' <<END_OF_TEST_RESULTS
220 Ready to transfer files.^M
220-foo^M
220-bar^M
220 Ready to transfer files.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-greeting 'xfer' <<END_OF_TEST_RESULTS
230 Ready to transfer files.^M
230-foo^M
230-bar^M
230 Ready to transfer files.^M
END_OF_TEST_RESULTS


##### Test tests/make-good-cvm #####

runtest() {
makecvm 'cat >/dev/null' \
	'printf "\\0"' \
	'printf "\\0001username\\0"' \
	'printf "\\0002102\\0"' \
	'printf "\\0003103\\0"' \
	'printf "\\0004Real Name\\0"' \
	'printf "\\0005/home/dir\\0"' \
	'printf "\\0006/bin/false\\0"' \
	'printf "\\0007grpname\\0"' \
	'printf "\\0000"' \
	'exit 0'
}
run_compare_test tests/make-good-cvm  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/auth-cvm-good-data #####

runtest() {
(
  echo '#!/bin/sh'
  echo 'echo UID=$UID GID=$GID HOME=$HOME USER=$USER'
) >$tmp/env
chmod 755 $tmp/env
auth $cvm $tmp/env 'USER username' 'PASS password'
rm -f $tmp/env
}
run_compare_test tests/auth-cvm-good-data  <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER username
331 Send PASS.^M
twoftpd-auth[PID]: PASS XXXXXXXX
UID=102 GID=103 HOME=/home/dir USER=username
END_OF_TEST_RESULTS


##### Test tests/delete-good-cvm #####

runtest() {
rm -f $cvm
}
run_compare_test tests/delete-good-cvm  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/make-error-cvm #####

runtest() {
makecvm 'echo SERVICE=$SERVICE >&2' 'exit 111'
}
run_compare_test tests/make-error-cvm  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/auth-cvm-invocation #####

runtest() {
auth $cvm printenv 'USER username' 'PASS password'
}
run_compare_test tests/auth-cvm-invocation  <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER username
331 Send PASS.^M
twoftpd-auth[PID]: PASS XXXXXXXX
SERVICE=ftp
530 Authentication failed.^M
END_OF_TEST_RESULTS


##### Test tests/auth-cvm-service #####

runtest() {
SERVICE=noftp
export SERVICE
auth $cvm printenv 'USER username' 'PASS password'
unset SERVICE
}
run_compare_test tests/auth-cvm-service  <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER username
331 Send PASS.^M
twoftpd-auth[PID]: PASS XXXXXXXX
SERVICE=noftp
530 Authentication failed.^M
END_OF_TEST_RESULTS


##### Test tests/delete-error-cvm #####

runtest() {
rm -f $cvm
}
run_compare_test tests/delete-error-cvm  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/make-bad-cvm #####

runtest() {
makecvm 'exit 0'
}
run_compare_test tests/make-bad-cvm  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/auth-cvm-bad-data #####

runtest() {
auth $cvm 'printf Help!' 'USER username' 'PASS password'
}
run_compare_test tests/auth-cvm-bad-data  <<END_OF_TEST_RESULTS
220-TwoFTPd server ready.^M
220 Authenticate first.^M
twoftpd-auth[PID]: USER username
331 Send PASS.^M
twoftpd-auth[PID]: PASS XXXXXXXX
530 Authentication failed.^M
END_OF_TEST_RESULTS


##### Test tests/delete-bad-cvm #####

runtest() {
rm -f $cvm
}
run_compare_test tests/delete-bad-cvm  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/make-home #####

runtest() {
mkdir $home/subdir
date >$home/subdir/date
for i in {0,1,2,3,4,5}{0,1,2,3,4,5,6,7,8,9}; do
  echo 01234567890123456789012345678901234567890123456789012345678901234567890
done >$home/large
TZ=GMT touch -t 200001020304.05 $home/empty
TZ=GMT touch -t 200102030405.06 $home/subdir
TZ=GMT touch -t 200301180607.08 $home/large
chmod 664 $home/empty
chmod 644 $home/large
chmod 775 $home/subdir
}
run_compare_test tests/make-home  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


##### Test tests/backend-commands #####

runtest() {
local backend="$1"
local command="$2"
local parameter="$3"
if [ $parameter = no ]; then
  cmd=$command
else
  cmd="$command xyzzy"
fi
$backend "$cmd"
}
run_compare_test tests/backend-commands 'anon' 'QUIT' 'no' <<END_OF_TEST_RESULTS
221 Bye.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'QUIT' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'HELP' 'no' <<END_OF_TEST_RESULTS
502 No help is available.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'HELP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'SYST' 'no' <<END_OF_TEST_RESULTS
215 UNIX Type: L8^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'SYST' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'NOOP' 'no' <<END_OF_TEST_RESULTS
200 Awaiting your commands, master...^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'NOOP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'USER' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'USER' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'PASS' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'PASS' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'ACCT' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'ACCT' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'TYPE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'TYPE' 'yes' <<END_OF_TEST_RESULTS
501 Unknown transfer type.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'STRU' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'STRU' 'yes' <<END_OF_TEST_RESULTS
504 Invalid parameter.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'MODE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'MODE' 'yes' <<END_OF_TEST_RESULTS
504 Invalid parameter.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'CWD' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'CWD' 'yes' <<END_OF_TEST_RESULTS
550 Directory does not exist.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'XCWD' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'XCWD' 'yes' <<END_OF_TEST_RESULTS
550 Directory does not exist.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'PWD' 'no' <<END_OF_TEST_RESULTS
257 "/"^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'PWD' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'XPWD' 'no' <<END_OF_TEST_RESULTS
257 "/"^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'XPWD' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'CDUP' 'no' <<END_OF_TEST_RESULTS
250 Changed directory.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'CDUP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'XCUP' 'no' <<END_OF_TEST_RESULTS
250 Changed directory.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'XCUP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'SIZE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'SIZE' 'yes' <<END_OF_TEST_RESULTS
550 Could not determine file size: No such file or directory^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'MDTM' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'anon' 'MDTM' 'yes' <<END_OF_TEST_RESULTS
550 Could not determine file time: No such file or directory^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'QUIT' 'no' <<END_OF_TEST_RESULTS
221 Bye.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'QUIT' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'HELP' 'no' <<END_OF_TEST_RESULTS
502 No help is available.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'HELP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'SYST' 'no' <<END_OF_TEST_RESULTS
215 UNIX Type: L8^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'SYST' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'NOOP' 'no' <<END_OF_TEST_RESULTS
200 Awaiting your commands, master...^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'NOOP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'USER' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'USER' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'PASS' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'PASS' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'ACCT' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'ACCT' 'yes' <<END_OF_TEST_RESULTS
230 Access has already been granted.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'TYPE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'TYPE' 'yes' <<END_OF_TEST_RESULTS
501 Unknown transfer type.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'STRU' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'STRU' 'yes' <<END_OF_TEST_RESULTS
504 Invalid parameter.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'MODE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'MODE' 'yes' <<END_OF_TEST_RESULTS
504 Invalid parameter.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'CWD' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'CWD' 'yes' <<END_OF_TEST_RESULTS
550 Directory does not exist.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'XCWD' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'XCWD' 'yes' <<END_OF_TEST_RESULTS
550 Directory does not exist.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'PWD' 'no' <<END_OF_TEST_RESULTS
257 "/"^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'PWD' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'XPWD' 'no' <<END_OF_TEST_RESULTS
257 "/"^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'XPWD' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'CDUP' 'no' <<END_OF_TEST_RESULTS
250 Changed directory.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'CDUP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'XCUP' 'no' <<END_OF_TEST_RESULTS
250 Changed directory.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'XCUP' 'yes' <<END_OF_TEST_RESULTS
501 Command requires no parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'SIZE' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'SIZE' 'yes' <<END_OF_TEST_RESULTS
550 Could not determine file size: No such file or directory^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'MDTM' 'no' <<END_OF_TEST_RESULTS
504 Command requires a parameter^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-commands 'xfer' 'MDTM' 'yes' <<END_OF_TEST_RESULTS
550 Could not determine file time: No such file or directory^M
END_OF_TEST_RESULTS


##### Test tests/backend-cdup-pwd #####

runtest() {
local backend="$1"
$backend PWD 'CWD subdir' PWD CDUP PWD CDUP PWD
}
run_compare_test tests/backend-cdup-pwd 'anon' <<END_OF_TEST_RESULTS
257 "/"^M
250 Changed directory.^M
257 "/subdir"^M
250 Changed directory.^M
257 "/"^M
250 Changed directory.^M
257 "/"^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-cdup-pwd 'xfer' <<END_OF_TEST_RESULTS
257 "/"^M
250 Changed directory.^M
257 "/subdir"^M
250 Changed directory.^M
257 "/"^M
250 Changed directory.^M
257 "/"^M
END_OF_TEST_RESULTS


##### Test tests/backend-type #####

runtest() {
local backend="$1"
testparams $backend TYPE A a "A N" "a n" I i "L 8" "l 8" "AN" "L8"
}
run_compare_test tests/backend-type 'anon' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
501 Unknown transfer type.^M
501 Unknown transfer type.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-type 'xfer' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to ASCII.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
200 Transfer mode changed to BINARY.^M
501 Unknown transfer type.^M
501 Unknown transfer type.^M
END_OF_TEST_RESULTS


##### Test tests/backend-stru #####

runtest() {
local backend="$1"
testparams $backend STRU F f T t
}
run_compare_test tests/backend-stru 'anon' <<END_OF_TEST_RESULTS
200 OK.^M
200 OK.^M
504 Invalid parameter.^M
504 Invalid parameter.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-stru 'xfer' <<END_OF_TEST_RESULTS
200 OK.^M
200 OK.^M
504 Invalid parameter.^M
504 Invalid parameter.^M
END_OF_TEST_RESULTS


##### Test tests/backend-mode #####

runtest() {
local backend="$1"
testparams $backend MODE S s X x
}
run_compare_test tests/backend-mode 'anon' <<END_OF_TEST_RESULTS
200 OK.^M
200 OK.^M
504 Invalid parameter.^M
504 Invalid parameter.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-mode 'xfer' <<END_OF_TEST_RESULTS
200 OK.^M
200 OK.^M
504 Invalid parameter.^M
504 Invalid parameter.^M
END_OF_TEST_RESULTS


##### Test tests/backend-cwd-pwd #####

runtest() {
local backend="$1"
$backend PWD 'CWD subdir' PWD 'CWD ..' PWD 'CWD ..' PWD
}
run_compare_test tests/backend-cwd-pwd 'anon' <<END_OF_TEST_RESULTS
257 "/"^M
250 Changed directory.^M
257 "/subdir"^M
250 Changed directory.^M
257 "/"^M
250 Changed directory.^M
257 "/"^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-cwd-pwd 'xfer' <<END_OF_TEST_RESULTS
257 "/"^M
250 Changed directory.^M
257 "/subdir"^M
250 Changed directory.^M
257 "/"^M
250 Changed directory.^M
257 "/"^M
END_OF_TEST_RESULTS


##### Test tests/backend-list-modes #####

runtest() {
local backend="$1"
local mode="$2"
local conv="$3"
local list="$4"
xfertest $backend $mode retr $conv $list
}
run_compare_test tests/backend-list-modes 'anon' 'pasv' 'bin' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'anon' 'pasv' 'bin' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 Jan  1  2000 empty^M
-rw-r--r--    1 ftp      ftp          4320 Jan 18 00:07 large^M
drwxrwxr-x    1 ftp      ftp          4096 Feb  2  2001 subdir^M
226 Listing complete (190 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'anon' 'pasv' 'asc' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'anon' 'pasv' 'asc' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 Jan  1  2000 empty^M
-rw-r--r--    1 ftp      ftp          4320 Jan 18 00:07 large^M
drwxrwxr-x    1 ftp      ftp          4096 Feb  2  2001 subdir^M
226 Listing complete (190 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'anon' 'port' 'bin' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'anon' 'port' 'bin' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 Jan  1  2000 empty^M
-rw-r--r--    1 ftp      ftp          4320 Jan 18 00:07 large^M
drwxrwxr-x    1 ftp      ftp          4096 Feb  2  2001 subdir^M
226 Listing complete (190 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'anon' 'port' 'asc' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'anon' 'port' 'asc' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 Jan  1  2000 empty^M
-rw-r--r--    1 ftp      ftp          4320 Jan 18 00:07 large^M
drwxrwxr-x    1 ftp      ftp          4096 Feb  2  2001 subdir^M
226 Listing complete (190 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'xfer' 'pasv' 'bin' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'xfer' 'pasv' 'bin' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 Jan  1  2000 empty^M
-rw-r--r--    1 ftp      ftp          4320 Jan 18 00:07 large^M
drwxrwxr-x    1 ftp      ftp          4096 Feb  2  2001 subdir^M
226 Listing complete (190 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'xfer' 'pasv' 'asc' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'xfer' 'pasv' 'asc' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 Jan  1  2000 empty^M
-rw-r--r--    1 ftp      ftp          4320 Jan 18 00:07 large^M
drwxrwxr-x    1 ftp      ftp          4096 Feb  2  2001 subdir^M
226 Listing complete (190 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'xfer' 'port' 'bin' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'xfer' 'port' 'bin' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 Jan  1  2000 empty^M
-rw-r--r--    1 ftp      ftp          4320 Jan 18 00:07 large^M
drwxrwxr-x    1 ftp      ftp          4096 Feb  2  2001 subdir^M
226 Listing complete (190 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'xfer' 'port' 'asc' 'NLST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-modes 'xfer' 'port' 'asc' 'LIST' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 Jan  1  2000 empty^M
-rw-r--r--    1 ftp      ftp          4320 Jan 18 00:07 large^M
drwxrwxr-x    1 ftp      ftp          4096 Feb  2  2001 subdir^M
226 Listing complete (190 bytes sent).^M
END_OF_TEST_RESULTS


##### Test tests/backend-size #####

runtest() {
local backend="$1"
$backend 'SIZE empty' 'SIZE large'
}
run_compare_test tests/backend-size 'anon' <<END_OF_TEST_RESULTS
213 0^M
213 4320^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-size 'xfer' <<END_OF_TEST_RESULTS
213 0^M
213 4320^M
END_OF_TEST_RESULTS


##### Test tests/backend-mdtm #####

runtest() {
local backend="$1"
$backend 'MDTM empty' 'MDTM large'
}
run_compare_test tests/backend-mdtm 'anon' <<END_OF_TEST_RESULTS
213 20000102030405^M
213 20030118060708^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-mdtm 'xfer' <<END_OF_TEST_RESULTS
213 20000102030405^M
213 20030118060708^M
END_OF_TEST_RESULTS


##### Test tests/backend-mkd-rmd #####

runtest() {
local backend="$1"
$backend 'MKD dir' 'MKD dir'
test -d $home/dir && echo dir was created
$backend 'RMD dir' 'RMD dir'
test -d $home/dir && echo dir still exists
$backend 'XMKD dir' 'XRMD dir'
test -d $home/dir && echo dir still exists
}
run_compare_test tests/backend-mkd-rmd 'anon' <<END_OF_TEST_RESULTS
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-mkd-rmd 'xfer' <<END_OF_TEST_RESULTS
257 Directory created successfully.^M
550 Could not create directory: File exists^M
dir was created
250 Directory removed successfully.^M
550 Could not remove directory: No such file or directory^M
257 Directory created successfully.^M
250 Directory removed successfully.^M
END_OF_TEST_RESULTS


##### Test tests/backend-list-params #####

runtest() {
local backend="$1"
local param="$2"
xfertest $backend port retr asc "NLST $param"
}
run_compare_test tests/backend-list-params 'anon' '.' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'anon' '..' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
553 Paths containing '..' not allowed.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'anon' '*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'anon' 'l*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
large^M
226 Listing complete (7 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'anon' 's*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
date^M
226 Listing complete (6 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'anon' 'subdir' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
date^M
226 Listing complete (6 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'anon' 'subdir/*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
subdir/date^M
226 Listing complete (13 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'anon' 'subdir/d*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
subdir/date^M
226 Listing complete (13 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'anon' '-A' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'anon' '-a' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'anon' '-L' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'anon' '-l' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 Jan  1  2000 empty^M
-rw-r--r--    1 ftp      ftp          4320 Jan 18 00:07 large^M
drwxrwxr-x    1 ftp      ftp          4096 Feb  2  2001 subdir^M
226 Listing complete (190 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'anon' '-alL' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 Jan  1  2000 empty^M
-rw-r--r--    1 ftp      ftp          4320 Jan 18 00:07 large^M
drwxrwxr-x    1 ftp      ftp          4096 Feb  2  2001 subdir^M
226 Listing complete (190 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'anon' '-a subdir/d*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
subdir/date^M
226 Listing complete (13 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' '.' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' '..' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
553 Paths containing '..' not allowed.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' '*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' 'l*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
large^M
226 Listing complete (7 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' 's*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
date^M
226 Listing complete (6 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' 'subdir' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
date^M
226 Listing complete (6 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' 'subdir/*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
subdir/date^M
226 Listing complete (13 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' 'subdir/d*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
subdir/date^M
226 Listing complete (13 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' '-A' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' '-a' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' '-L' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
empty^M
large^M
subdir^M
226 Listing complete (22 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' '-l' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 Jan  1  2000 empty^M
-rw-r--r--    1 ftp      ftp          4320 Jan 18 00:07 large^M
drwxrwxr-x    1 ftp      ftp          4096 Feb  2  2001 subdir^M
226 Listing complete (190 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' '-alL' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
-rw-rw-r--    1 ftp      ftp             0 Jan  1  2000 empty^M
-rw-r--r--    1 ftp      ftp          4320 Jan 18 00:07 large^M
drwxrwxr-x    1 ftp      ftp          4096 Feb  2  2001 subdir^M
226 Listing complete (190 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-list-params 'xfer' '-a subdir/d*' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
subdir/date^M
226 Listing complete (13 bytes sent).^M
END_OF_TEST_RESULTS


##### Test tests/backend-rest #####

runtest() {
local backend="$1"
xfertest $backend port retr bin 'REST 4000' 'RETR large'
}
run_compare_test tests/backend-rest 'anon' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
350 Start position for transfer has been set.^M
200 OK.^M
150 Opened data connection.^M
0123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
226 File sent successfully (320 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-rest 'xfer' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
350 Start position for transfer has been set.^M
200 OK.^M
150 Opened data connection.^M
0123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
226 File sent successfully (320 bytes sent).^M
END_OF_TEST_RESULTS


##### Test tests/backend-rnfrto #####

runtest() {
local backend="$1"
$backend 'RNFR subdir/date' 'RNTO date'
ls -1 $home
$backend 'RNFR date' 'RNTO subdir/date' 'RNFR date' 'RNTO subdir/date'
ls -1 $home
}
run_compare_test tests/backend-rnfrto 'anon' <<END_OF_TEST_RESULTS
502 Command not supported.^M
502 Command not supported.^M
empty
large
subdir
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
502 Command not supported.^M
empty
large
subdir
END_OF_TEST_RESULTS

run_compare_test tests/backend-rnfrto 'xfer' <<END_OF_TEST_RESULTS
350 OK, file exists.^M
250 File renamed.^M
date
empty
large
subdir
350 OK, file exists.^M
250 File renamed.^M
450 Could not locate file.^M
425 Send RNFR first.^M
empty
large
subdir
END_OF_TEST_RESULTS


##### Test tests/backend-retr #####

runtest() {
local backend="$1"
local mode="$2"
local conv="$3"
xfertest $backend $mode retr $conv 'RETR large'
}
run_compare_test tests/backend-retr 'anon' 'pasv' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
226 File sent successfully (4320 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-retr 'anon' 'pasv' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
226 File sent successfully (4380 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-retr 'anon' 'port' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
226 File sent successfully (4320 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-retr 'anon' 'port' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
226 File sent successfully (4380 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-retr 'xfer' 'pasv' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
226 File sent successfully (4320 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-retr 'xfer' 'pasv' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
226 File sent successfully (4380 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-retr 'xfer' 'port' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
01234567890123456789012345678901234567890123456789012345678901234567890
226 File sent successfully (4320 bytes sent).^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-retr 'xfer' 'port' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
01234567890123456789012345678901234567890123456789012345678901234567890^M
226 File sent successfully (4380 bytes sent).^M
END_OF_TEST_RESULTS


##### Test tests/backend-appe #####

runtest() {
local backend="$1"
cat $home/large >$home/tmp
xfertest $backend port store bin 'APPE tmp' <$home/large
wc <$home/tmp
xfertest $backend port store bin 'APPE tmpx' <$home/large
test -e $home/tmpx || echo tmpx was not created
rm -f $home/tmp $home/tmpx
}
run_compare_test tests/backend-appe 'anon' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
502 Command not supported.^M
     60      60    4320
200 Transfer mode changed to BINARY.^M
200 OK.^M
502 Command not supported.^M
tmpx was not created
END_OF_TEST_RESULTS

run_compare_test tests/backend-appe 'xfer' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
226 File received successfully (4320 bytes received).^M
    120     120    8640
200 Transfer mode changed to BINARY.^M
200 OK.^M
550 Could not open output file: No such file or directory^M
tmpx was not created
END_OF_TEST_RESULTS


##### Test tests/backend-store #####

runtest() {
local backend="$1"
local mode="$2"
local conv="$3"
sed -e 's/$/
/' $home/large >$tmp/large
xfertest $backend $mode store $conv 'STOR tmp' <$tmp/large
if [ -e $home/tmp ]; then
  wc <$home/tmp
  rm -f $home/tmp
else
  echo tmp was not created
fi
rm -f $tmp/large
}
run_compare_test tests/backend-store 'anon' 'pasv' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
502 Command not supported.^M
tmp was not created
END_OF_TEST_RESULTS

run_compare_test tests/backend-store 'anon' 'pasv' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
502 Command not supported.^M
tmp was not created
END_OF_TEST_RESULTS

run_compare_test tests/backend-store 'anon' 'port' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
502 Command not supported.^M
tmp was not created
END_OF_TEST_RESULTS

run_compare_test tests/backend-store 'anon' 'port' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
502 Command not supported.^M
tmp was not created
END_OF_TEST_RESULTS

run_compare_test tests/backend-store 'xfer' 'pasv' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
226 File received successfully (4380 bytes received).^M
     60      60    4380
END_OF_TEST_RESULTS

run_compare_test tests/backend-store 'xfer' 'pasv' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
227 Entering Passive Mode (#,#,#,#,#,#).^M
150 Opened data connection.^M
226 File received successfully (4380 bytes received).^M
     60      60    4320
END_OF_TEST_RESULTS

run_compare_test tests/backend-store 'xfer' 'port' 'bin' <<END_OF_TEST_RESULTS
200 Transfer mode changed to BINARY.^M
200 OK.^M
150 Opened data connection.^M
226 File received successfully (4380 bytes received).^M
     60      60    4380
END_OF_TEST_RESULTS

run_compare_test tests/backend-store 'xfer' 'port' 'asc' <<END_OF_TEST_RESULTS
200 Transfer mode changed to ASCII.^M
200 OK.^M
150 Opened data connection.^M
226 File received successfully (4380 bytes received).^M
     60      60    4320
END_OF_TEST_RESULTS


##### Test tests/backend-dele #####

runtest() {
local backend="$1"
touch $home/tmp
$backend "DELE tmp"
test -e $home/tmp && echo tmp was not deleted
$backend "DELE tmp"
rm -f $home/tmp
}
run_compare_test tests/backend-dele 'anon' <<END_OF_TEST_RESULTS
502 Command not supported.^M
tmp was not deleted
502 Command not supported.^M
END_OF_TEST_RESULTS

run_compare_test tests/backend-dele 'xfer' <<END_OF_TEST_RESULTS
250 File removed successfully.^M
550 Could not remove file: No such file or directory^M
END_OF_TEST_RESULTS


##### Test tests/backend-chmod #####

runtest() {
local backend="$1"
touch $home/tmp
chmod 444 $home/tmp
$backend "SITE CHMOD 755 tmp" "SITE CHMOD 755 tmpx"
ls -l $home/tmp | cut -d' ' -f1
rm -f $home/tmp
}
run_compare_test tests/backend-chmod 'anon' <<END_OF_TEST_RESULTS
502 Command not supported.^M
502 Command not supported.^M
-r--r--r--
END_OF_TEST_RESULTS

run_compare_test tests/backend-chmod 'xfer' <<END_OF_TEST_RESULTS
250 File modes changed.^M
550 Could not change modes on file: No such file or directory^M
-rwxr-xr-x
END_OF_TEST_RESULTS


##### Test tests/delete-home #####

runtest() {
rm -r $home/*
}
run_compare_test tests/delete-home  <<END_OF_TEST_RESULTS
END_OF_TEST_RESULTS


rm -rf $tmp
echo $tests_count tests executed, $tests_failed failures
if [ $tests_failed != 0 ]; then exit 1; fi
