/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    Copyright (C) 2009-2012 Martin Brehm

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "timestep.h"
#include "travis.h"

void CTimeStep::CalcCenters()
{
	BTIN;
	int z, z2, z3, z4, c;
	CxVector3 tv;
	CMolecule *m;
	CVirtualAtom *v;
	CSingleMolecule *sm;

	if (m_vaCoords.GetSize() < g_iGesVirtAtomCount)
		m_vaCoords.SetSize(g_iGesVirtAtomCount);

//	mprintf("CalcCenters(): Size is now %d.\n",m_vaCoords.GetSize());

	if (g_bUseVelocities)
		if (m_vaVelocities.GetSize() < g_iGesVirtAtomCount)
			m_vaVelocities.SetSize(g_iGesVirtAtomCount);

	if (g_bUseForces)
		if (m_vaForces.GetSize() < g_iGesVirtAtomCount)
			m_vaForces.SetSize(g_iGesVirtAtomCount);

	for (z=0;z<g_oaVirtualAtoms.GetSize();z++)
	{
		v = (CVirtualAtom*)g_oaVirtualAtoms[z];
		m = (CMolecule*)g_oaMolecules[v->m_iMolecule];
		for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
		{
			sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];
			if (v->m_iMode == 0) // Mittel aus Atompositionen
			{
				tv = 0;
				c = 0;
				for (z3=0;z3<v->m_oCenterAtoms.m_baAtomType.GetSize();z3++)
				{
					for (z4=0;z4<((CxIntArray*)v->m_oCenterAtoms.m_oaAtoms[z3])->GetSize();z4++)
					{
						tv += m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[v->m_oCenterAtoms.m_baAtomType[z3]])->GetAt(((CxIntArray*)v->m_oCenterAtoms.m_oaAtoms[z3])->GetAt(z4))] * v->m_faWeight[c];
						c++;
					}
				}
				tv /= v->m_fGesWeight;
			} else if (v->m_iMode == 1) // Abstand, Winkel, Dihedralwinkel
			{
				tv = PointFromRAD(m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[v->m_iAtomType[0]])->GetAt(v->m_iAtom[0])],m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[v->m_iAtomType[1]])->GetAt(v->m_iAtom[1])],m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[v->m_iAtomType[2]])->GetAt(v->m_iAtom[2])],v->m_fValues[0],v->m_fValues[1],v->m_fValues[2]);
			} else if (v->m_iMode == 2) // Dipolvektor
			{
				if (!g_bDipole)
				{
					eprintf("Cannot use dipole vectors.\n");
					BTOUT;
					return;
				}
				tv = sm->m_vDipole + m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[sm->m_baAtomIndex.GetSize()-1])->GetAt(0)];
			} else if (v->m_iMode == 3) // Geschwindigkeitsvektor
			{
			} else if (v->m_iMode == 4) // Kraftvektor
			{
			}
			m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[sm->m_baAtomIndex.GetSize()-1])->GetAt(v->m_iMolVirtAtom)] = tv;
		}
	}
	BTOUT;
}

bool CTimeStep::BondRange(int i1, int i2, double *f)
{
	BTIN;
	int a, b;
	double d;
	double x, y, z;

	if (g_iScanMolStep == -1)
		return false;

	x = m_vaCoords[i1][0] - m_vaCoords[i2][0];
	y = m_vaCoords[i1][1] - m_vaCoords[i2][1];
	z = m_vaCoords[i1][2] - m_vaCoords[i2][2];

	if (g_bPeriodic)
	{
		if (g_bPeriodicX)
		{
			while (x < -g_fBoxX/2)
				x += g_fBoxX;
			while (x > g_fBoxX/2)
				x -= g_fBoxX;
		}

		if (g_bPeriodicY)
		{
			while (y < -g_fBoxY/2)
				y += g_fBoxY;
			while (y > g_fBoxY/2)
				y -= g_fBoxY;
		}

		if (g_bPeriodicZ)
		{
			while (z < -g_fBoxZ/2)
				z += g_fBoxZ;
			while (z > g_fBoxZ/2)
				z -= g_fBoxZ;
		}
	}

	d = (float)sqrt(x*x+y*y+z*z);

	if (f != NULL)
		*f = d;

	a = g_baAtomIndex[i1];
	b = g_baAtomIndex[i2];

	if ((a >= g_oaAtoms.GetSize()) || (b >= g_oaAtoms.GetSize()))
	{
		BTOUT; 
		return false;
	}

	if (((CAtom*)g_oaAtoms[a])->m_pElement->m_fRadius == 0)
	{
		BTOUT;
		return false;
	}
	
	if (((CAtom*)g_oaAtoms[b])->m_pElement->m_fRadius == 0)
	{
		BTOUT;
		return false;
	}

	if (d < (((CAtom*)g_oaAtoms[a])->m_pElement->m_fRadius+((CAtom*)g_oaAtoms[b])->m_pElement->m_fRadius)*g_fBondFactor)
	{
//		printf("    \"%s\" br=%f, \"%s\" br=%f Hat Nachbarn %s%d im Abstand von %.3f.\n",m_pLabels[i1],c1,m_pLabels[i2],c2,m_pLabels[i2],i2+1,d);
		BTOUT; 
		return true;
	} else 
	{
		BTOUT;
		return false;
	}
}

bool CTimeStep::MirrorBond(int i1, int i2)
{
	BTIN;
	bool changed;

	changed = false;

	if (g_bPeriodicX)
	{
		while (m_vaCoords[i1][0]-m_vaCoords[i2][0] > g_fBoxX/2)
		{
			m_vaCoords[i2][0] += g_fBoxX;
			changed = true;
		}
		while (m_vaCoords[i2][0]-m_vaCoords[i1][0] > g_fBoxX/2)
		{
			m_vaCoords[i2][0] -= g_fBoxX;
			changed = true;
		}
	}

	if (g_bPeriodicY)
	{
		while (m_vaCoords[i1][1]-m_vaCoords[i2][1] > g_fBoxY/2)
		{
			m_vaCoords[i2][1] += g_fBoxY;
			changed = true;
		}
		while (m_vaCoords[i2][1]-m_vaCoords[i1][1] > g_fBoxY/2)
		{
			m_vaCoords[i2][1] -= g_fBoxY;
			changed = true;
		}
	}

	if (g_bPeriodicZ)
	{
		while (m_vaCoords[i1][2]-m_vaCoords[i2][2] > g_fBoxZ/2)
		{
			m_vaCoords[i2][2] += g_fBoxZ;
			changed = true;
		}
		while (m_vaCoords[i2][2]-m_vaCoords[i1][2] > g_fBoxZ/2)
		{
			m_vaCoords[i2][2] -= g_fBoxZ;
			changed = true; 
		}
	}

	if (changed)
	{
		BTOUT; 
		return true;
	} 
	BTOUT; 
	return false;
}


void CTimeStep::RECURSION_ScanMolecules(int i, CxByteArray *ta, CSingleMolecule *sm, int depth, int *stack, unsigned long bmask, bool w)
{
	BTIN;
	int z, z2;
	int nblist[64], nbs;
	double f;
	CxIntArray *wa;	
	CMolAtom *ma;
//	mprintf("  Rekursion fuer Atom %d.\n",i+1);

//	mprintf("Depth=%d, i=%d.\n",depth,i);
	if (g_bVerbose)
	{
		mprintf("  ");
		for (z=1;z<depth;z++)
		{
			if ((bmask & (int)pow(2.0,z)) != 0)
				mprintf(WHITE,"|  ");
					else mprintf("   ");
		}
		if (depth != 0)
		{
			if (w)
				mprintf(WHITE,"|--");
					else mprintf(WHITE,"`--");
		}
		mprintf(CYAN,"%s",((CAtom*)g_oaAtoms[g_baAtomIndex[i]])->m_sName);
		mprintf("(%d)",i+1);
	}

	stack[depth] = i;
	g_laAtomSMIndex[i] = (unsigned long)g_oaSingleMolecules.GetSize()-1;
	(*ta)[i] = 1;
//	mprintf("  sm->m_baAtomIndex.GetSize()=%d\n",sm->m_baAtomIndex.GetSize());
	for (z=0;z<sm->m_baAtomIndex.GetSize();z++)
	{
		if (sm->m_baAtomIndex[z] == g_baAtomIndex[i])
		{
			try { ma = new CMolAtom(); } catch(...) { ma = NULL; }
			if (ma == NULL) NewException((double)sizeof(CMolAtom),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			ma->m_iOffset = i;
			ma->m_iType = z;
			sm->m_oaMolAtoms.Add(ma);
			goto _ok;
		}
	}

	try { ma = new CMolAtom(); } catch(...) { ma = NULL; }
	if (ma == NULL) NewException((double)sizeof(CMolAtom),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	ma->m_iOffset = i;
	ma->m_iType = sm->m_baAtomIndex.GetSize();
	ma->m_iNumber = 0;
	sm->m_oaMolAtoms.Add(ma);

	sm->m_baAtomIndex.Add(g_baAtomIndex[i]);
_ok:
	m_iConnectedAtoms++;

	if (!g_bVerbose)
		if ((m_iConnectedAtoms % m_iGesAtomModulo)==0)
			mprintf(WHITE,"#");

	nbs = 0;
	for (z=0;z<(long)m_iGesAtomCount;z++) // Schon mal alle Nachbarn raussuchen
	{
		if (z == i)
			continue;
		if (((CAtom*)g_oaAtoms[g_baAtomIndex[z]])->m_bExclude)
			continue;
		if (BondRange(i,z,&f))
		{
			if ((*ta)[z] != 0)
			{
				if ((depth > 0) && (z != stack[depth-1]))
				{
					if (g_bVerbose)
					{
						mprintf(GREEN,"  <-- Ring closure: ");
						mprintf("%s(%d)",((CAtom*)g_oaAtoms[g_baAtomIndex[stack[depth]]])->m_sName,stack[depth]+1);
					}

					try { wa = new CxIntArray(); } catch(...) { wa = NULL; }
					if (wa == NULL) NewException((double)sizeof(CxIntArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					sm->m_oaRings.Add(wa);
					wa->Add(stack[depth]);
					z2 = depth-1;
					while ((stack[z2] != z) && (z2 >= 0))
					{
						wa->Add(stack[z2]);
						if (g_bVerbose)
							mprintf(" - %s(%d)",((CAtom*)g_oaAtoms[g_baAtomIndex[stack[z2]]])->m_sName,stack[z2]+1);
						z2--;
					}
					wa->Add(z);
					if (g_bVerbose)
						mprintf(" - %s(%d)",((CAtom*)g_oaAtoms[g_baAtomIndex[z]])->m_sName,z+1);
				}
				continue;
			}
/*			for (z2=0;z2<g_waBondBlackList.GetSize();z2+=2)
				if (((g_waBondBlackList[z2] == i) && (g_waBondBlackList[z2+1] == z)) || 
					((g_waBondBlackList[z2] == z) && (g_waBondBlackList[z2+1] == i)))
				{
					if (g_bVerbose)
					{
						mprintf("\n  ");
						for (z2=1;z2<depth+1;z2++)
						{
							if ((bmask & (int)pow(2,z2)) != 0)
								mprintf(WHITE,"|  ");
							else mprintf("   ");
						}
						if (z+1 < nbs)
							mprintf(WHITE,"|--");
								else mprintf("`--");
						mprintf("%s(%d)",((CAtom*)g_oaAtoms[g_baAtomIndex[z]])->m_sName,z+1);
						mprintf(GREEN,"  <-- This bond has been broken, skipping.\n");
					}
					g_iBondBlackListUsed++;
					goto _nextnb;
				}*/
			if (f < 50.0f)
			{
				g_bCloseAtoms = true;
				mprintf(RED,"\n### The atoms %s(%d) and %s(%d) are VERY close to each other. (Distance=%.4f pm)",((CAtom*)g_oaAtoms[g_baAtomIndex[i]])->m_sName,i+1,((CAtom*)g_oaAtoms[g_baAtomIndex[z]])->m_sName,z+1,f);
			}
			nblist[nbs] = z;
			nbs++;
			if (nbs >= 64)
			{
				mprintf(RED,"\n### Atom %s(%d) has more than 64 bonds. Ignoring further bonds.",((CAtom*)g_oaAtoms[g_baAtomIndex[i]])->m_sName,i+1);
				goto _nbdone;
			}
		}
	}
_nbdone:
	if (g_bVerbose)
		mprintf("\n");
	for (z=0;z<nbs;z++) // Fuer das aktuelle Atom z2 alle Nachbarn durchgehen
	{
		for (z2=0;z2<g_laBondBlackList.GetSize();z2+=2)
		{
			if (((g_laBondBlackList[z2] == i) && (g_laBondBlackList[z2+1] == nblist[z])) || 
				((g_laBondBlackList[z2] == nblist[z]) && (g_laBondBlackList[z2+1] == i)))
			{
				if (g_bVerbose)
				{
					mprintf("  ");
					for (z2=1;z2<depth+1;z2++)
					{
						if ((bmask & (int)pow(2.0,z2)) != 0)
							mprintf(WHITE,"|  ");
						else mprintf("   ");
					}
					if (z+1 < nbs)
						mprintf(WHITE,"|--");
							else mprintf(WHITE,"`--");
					mprintf(CYAN,"%s",((CAtom*)g_oaAtoms[g_baAtomIndex[nblist[z]]])->m_sName);
					mprintf("(%d)",nblist[z]+1);
					mprintf(GREEN,"  <-- This bond shall be broken, skipping.\n");
				}
				g_iBondBlackListUsed++;
				goto _nextnb;
			}
		}

		sm->m_laBonds.Add(i);
		sm->m_laBonds.Add(nblist[z]);

		if ((*ta)[nblist[z]] == 0) // Der Nachbar ist noch immer frei
		{
			if (z+1 == nbs)
				bmask -= (int)pow(2.0,depth+1);
			RECURSION_ScanMolecules(nblist[z],ta,sm,depth+1,stack,bmask,(z+1==nbs)?false:true);
		} else // Ringschluss
		{
			if (g_bVerbose)
			{
				mprintf("  ");
				for (z2=1;z2<depth+1;z2++)
				{
					if ((bmask & (int)pow(2.0,z2)) != 0)
						mprintf(WHITE,"|  ");
					else mprintf("   ");
				}
				if (z+1 < nbs)
					mprintf(WHITE,"|--");
						else mprintf(WHITE,"`--");
				mprintf(CYAN,"%s",((CAtom*)g_oaAtoms[g_baAtomIndex[nblist[z]]])->m_sName);
				mprintf("(%d)",nblist[z]+1);
				mprintf(GREEN,"  <-- Ring closure\n");
			}
		}
_nextnb:;
	}
	BTOUT; 
}

void CTimeStep::BuildAtomIndex()
{
	BTIN;
	int z, z2;
	CAtom *at;
	char buf[64];

	g_baAtomIndex.SetSize(m_iGesAtomCount);
	for (z=0;z<(long)m_iGesAtomCount;z++)
	{
		strcpy(buf,(char*)m_paLabels[z]);
		ReplaceDigits(buf);
		for (z2=0;z2<g_oaAtoms.GetSize();z2++)
		{
			if (mystricmp(buf,((CAtom*)g_oaAtoms[z2])->m_sName)==0)
			{
				at = (CAtom*)g_oaAtoms[z2];
	//			mprintf("Atom %d: Type %d (%s).\n",z,z2,at->m_sName);
				while (at->m_pMergedTo != NULL)
				{
					at = at->m_pMergedTo;
	//				mprintf("  ...was merged to %s.\n",at->m_sName);
				}
	//			mprintf("  Index is %d.\n",at->m_iIndex);
				g_baAtomIndex[z] = (unsigned char)at->m_iIndex;
				goto _e;
			}
		}
		g_baAtomIndex[z] = 255;
		eprintf("Error: CTimeStep::BuildAtomIndex(): Atom type \"%s\" not known.\n",buf);
		_e:;
	}
	BTOUT; 
}

void CTimeStep::ScanMolecules()
{
	BTIN;
	int z, z2, z3, z4, z5, i, i2, ti, ti2, i1;
	CxByteArray ta;
	CSingleMolecule *sm, *sm2;
	CxIntArray *wa, *waz2, *waz3, *waneu;
	CMolecule *m;
	CMolAtom *ma, *ma2, *ma3;
	CMolBond *bond, *bond2;
	CMolBondGroup *bg;
	CMolAngle *angle, *angle2;
	CMolAngleGroup *ag;
	CAtom *at;
	float f;
	bool b;
	double tfs;
	int *stack;

	BuildAtomIndex();
//	CalcRadii();

	try { stack = new int[g_iGesAtomCount]; } catch(...) { stack = NULL; }
	if (stack == NULL) NewException((double)g_iGesAtomCount*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	b = false;
	for (z=0;z<g_oaAtoms.GetSize();z++)
	{
		if (z == g_iVirtAtomType)
			continue;
		at = (CAtom*)g_oaAtoms[z];
		if (at->m_pMergedTo != NULL)
			continue;
		if (at->m_pElement->m_fRadius == 0)
		{
			if (!b)
			{
				b = true;
				mprintf("\n");
			}
			at->m_bExclude = AskYesNo("    Atom type \"%s\" has bond radius 0. Exclude it from the system (y/n)? [yes] ",true,at->m_sName);
	//		if (!at->m_bExclude)
	//			at->m_pElement->m_fRadius = AskFloat_ND("    Please enter covalent bond radius for %s (in pm): ",at->m_sName);
		}
	}

	ta.SetSize(g_iGesAtomCount);
	g_laAtomSMIndex.SetSize(g_iGesAtomCount);
	for (z=0;z<g_iGesAtomCount;z++)
	{
		ta[z] = 0;
		g_laAtomSMIndex[z] = 0;
	}
	g_iBondBlackListUsed = 0;
	g_oaSingleMolecules.RemoveAll();
	g_oaMolecules.RemoveAll();
	m_iConnectedAtoms = 0;
	m_iGesAtomModulo = (m_iGesAtomCount / 60)+1;
	if (g_bVerbose)
	{
		mprintf(WHITE,"\n    Molecule recognition...\n\n");
		mprintf(WHITE,">>> Output of the molecule tree >>>\n");
	} else mprintf("\n    Molecule recognition [");
//	fflush(stdout);
	for (z=0;z<(long)m_iGesAtomCount;z++)
	{
		if (ta[z] != 0) // Dieses Atom wurde bereits in irgendein Molekuel eingebaut
		{
//			mprintf("> Atom %d: Schon vergeben.\n",z+1);
			continue;
		}
//		mprintf("# Atom %d: Starte Rekursion.\n",z+1);

		if (((CAtom*)g_oaAtoms[g_baAtomIndex[z]])->m_bExclude)
			continue;
		
		try { sm = new CSingleMolecule(); } catch(...) { sm = NULL; }
		if (sm == NULL) NewException((double)sizeof(CSingleMolecule),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		g_oaSingleMolecules.Add(sm);

//		sm->m_iAtomGes = 0;
/*		for (z2=0;z2<16;z2++)
			sm->m_iAtomCount[z2] = 0;
		sm->m_iElements = 0;*/
		if (g_bVerbose)
			mprintf(YELLOW,"\nThe next molecule starts with %s(%d):\n",m_paLabels[z],z+1);
		
		RECURSION_ScanMolecules(z,&ta,sm,0,stack,0xFFFFFFFF,true);

//		printf("%d Atome in diesem Molekuel.\n",g_pSingleMolecules[g_oaSingleMolecules.GetSize()].AtomGes);

//		g_oaSingleMolecules.GetSize()++;
	}
	if (g_bVerbose)
	{
		mprintf(WHITE,"\n<<< Output of the molecule tree <<<\n\n");
		mprintf("%d molecules found.\n\n",g_oaSingleMolecules.GetSize());
	} else mprintf("]\n\n    %d molecules found.\n\n",g_oaSingleMolecules.GetSize());
	mprintf("    Sorting atom types...\n");
	SortSingleMolAtomTypes();

	delete[] stack;

/*	mprintf("\n");
	for (z=0;z<g_oaSingleMolecules.GetSize();z++)
	{
		mprintf("%d ",z+1);
		((CSingleMolecule*)g_oaSingleMolecules[z])->Dump();
	}

	mprintf("$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$\n");*/

/*	mprintf("\n");
	for (z=0;z<g_oaSingleMolecules.GetSize();z++)
	{
		mprintf("%d ",z+1);
		((CSingleMolecule*)g_oaSingleMolecules[z])->Dump();
	}
	mprintf("$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$\n");*/
	if (g_iBondBlackListUsed != 0)
		mprintf("    %d bonds have been broken.\n",g_iBondBlackListUsed);
	
/*	if (g_oaSingleMolecules.GetSize() >= 4096)
	{
		mprintf("\n>>> Mehr als 4096 Molekuele! Dies wird nicht unterstuetzt. <<<\n\n");
//		g_oaSingleMolecules.GetSize() = 4096;
	}*/

	mprintf("    Setting up bond lists...\n");
	// Die Bindungen in m_oaMolAtoms aufbauen
	for (z=0;z<g_oaSingleMolecules.GetSize();z++) 
	{
		sm = (CSingleMolecule*)g_oaSingleMolecules[z];
		for (z2=0;z2<sm->m_laBonds.GetSize()/2;z2++)
		{
			i = sm->m_laBonds[z2*2];
			i2 = sm->m_laBonds[z2*2+1];
			ti = -1;
			for (z3=0;z3<sm->m_oaMolAtoms.GetSize();z3++)
				if (((CMolAtom*)sm->m_oaMolAtoms[z3])->m_iOffset == i)
				{
					ti = z3;
					break;
				}
			if (ti == -1)
			{
				eprintf("CTimeStep::ScanMolecules(): Atom 1 of bond (%d) not found.\n",i);
				return;
			}
			ti2 = -1;
			for (z3=0;z3<sm->m_oaMolAtoms.GetSize();z3++)
				if (((CMolAtom*)sm->m_oaMolAtoms[z3])->m_iOffset == i2)
				{
					ti2 = z3;
					break;
				}
			if (ti2 == -1)
			{
				eprintf("CTimeStep::ScanMolecules(): Atom 2 of bond (%d) not found.\n",i2);
				return;
			}
			((CMolAtom*)sm->m_oaMolAtoms[ti])->m_oaBonds.Add((CMolAtom*)sm->m_oaMolAtoms[ti2]);
			((CMolAtom*)sm->m_oaMolAtoms[ti2])->m_oaBonds.Add((CMolAtom*)sm->m_oaMolAtoms[ti]);
		}
	}

	// Die AtomCodes berechnen
	mprintf("    Building atom codes...\n");
	for (z=0;z<g_oaSingleMolecules.GetSize();z++) 
	{
//		mprintf("Molecule %d: ",z+1);
		sm = (CSingleMolecule*)g_oaSingleMolecules[z];
		if (g_bVerbose)
			mprintf(YELLOW,"\n*** Singlemolecule %d ***\n\n",z+1);
		sm->BuildAtomCodes();
	}

	mprintf("    Creating topological atom order...\n");
	for (z=0;z<g_oaSingleMolecules.GetSize();z++) 
	{
		sm = (CSingleMolecule*)g_oaSingleMolecules[z];
		for (z2=0;z2<sm->m_baAtomIndex.GetSize();z2++)
		{
			try { wa = new CxIntArray(); } catch(...) { wa = NULL; }
			if (wa == NULL) NewException((double)sizeof(CxIntArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			sm->m_oaAtomOffset.Add(wa);
			for (z3=0;z3<sm->m_oaMolAtoms.GetSize();z3++)
			{
				ma = (CMolAtom*)sm->m_oaMolAtoms[z3];
				if (ma->m_iType != z2)
					continue;
				ma->m_iNumber = wa->GetSize();
				wa->Add(ma->m_iOffset);
			}
		}
	}

	mprintf("    Creating bond list...\n");
	for (z=0;z<g_oaSingleMolecules.GetSize();z++) 
	{
		sm = (CSingleMolecule*)g_oaSingleMolecules[z];
//		mprintf("%d:\n",z+1);
		for (z2=0;z2<sm->m_oaMolAtoms.GetSize();z2++)
		{
			ma = (CMolAtom*)sm->m_oaMolAtoms[z2];
			for (z3=0;z3<ma->m_oaBonds.GetSize();z3++)
			{
				ma2 = (CMolAtom*)ma->m_oaBonds[z3];
				if (ma2->m_iMolAtomNumber < z2 /* == ma->m_iMolAtomNumber*/)
					continue;

		//		mprintf("  %s%2d <--> %s%2d\n",((CAtom*)g_oaAtoms[sm->m_baAtomIndex[ma->m_iType]])->m_sName,ma->m_iNumber+1,((CAtom*)g_oaAtoms[sm->m_baAtomIndex[ma2->m_iType]])->m_sName,ma2->m_iNumber+1);

				try { bond = new CMolBond(); } catch(...) { bond = NULL; }
				if (bond == NULL) NewException((double)sizeof(CMolBond),__FILE__,__LINE__,__PRETTY_FUNCTION__);
				
				bond->m_iAtom[0] = ma->m_iNumber;
				bond->m_iAtom[1] = ma2->m_iNumber;
				bond->m_iAtomType[0] = ma->m_iType;
				bond->m_iAtomType[1] = ma2->m_iType;
				bond->m_iMolAtom[0] = z2;
				bond->m_iMolAtom[1] = ma2->m_iMolAtomNumber;
				bond->m_iAtomOffset[0] = ma->m_iOffset;
				bond->m_iAtomOffset[1] = ma2->m_iOffset;
				sm->m_oaBonds.Add(bond);
			}
		}
/*		if (z == 0)
			mprintf("  %d Bonds added.\n",oatemp.GetSize());*/
		for (z2=0;z2<sm->m_oaBonds.GetSize();z2++)
		{
			bond = (CMolBond*)sm->m_oaBonds[z2];
			for (z3=0;z3<sm->m_oaBondGroups.GetSize();z3++)
			{
				bg = (CMolBondGroup*)sm->m_oaBondGroups[z3];
				for (z4=0;z4<bg->m_oaBonds.GetSize();z4++)
				{
					bond2 = (CMolBond*)bg->m_oaBonds[z4];
					if (((((CMolAtom*)sm->m_oaMolAtoms[bond->m_iMolAtom[0]])->m_fAtomCode == ((CMolAtom*)sm->m_oaMolAtoms[bond2->m_iMolAtom[0]])->m_fAtomCode) &&
						 (((CMolAtom*)sm->m_oaMolAtoms[bond->m_iMolAtom[1]])->m_fAtomCode == ((CMolAtom*)sm->m_oaMolAtoms[bond2->m_iMolAtom[1]])->m_fAtomCode)) ||
						((((CMolAtom*)sm->m_oaMolAtoms[bond->m_iMolAtom[0]])->m_fAtomCode == ((CMolAtom*)sm->m_oaMolAtoms[bond2->m_iMolAtom[1]])->m_fAtomCode) &&
						 (((CMolAtom*)sm->m_oaMolAtoms[bond->m_iMolAtom[1]])->m_fAtomCode == ((CMolAtom*)sm->m_oaMolAtoms[bond2->m_iMolAtom[0]])->m_fAtomCode)))	
					{
						bg->m_oaBonds.Add(bond);
						goto _bonddone;
					}
				}
			}

			try { bg = new CMolBondGroup(); } catch(...) { bg = NULL; }
			if (bg == NULL) NewException((double)sizeof(CMolBondGroup),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			bg->m_oaBonds.Add(bond);
			sm->m_oaBondGroups.Add(bg);
_bonddone:;
		}
/*		if (z == 0)
		{
			mprintf("  %d Bond groups found:\n",sm->m_oaBondGroups.GetSize());
			for (z2=0;z2<sm->m_oaBondGroups.GetSize();z2++)
			{
				mprintf("  - ");
				bg = (CMolBondGroup*)sm->m_oaBondGroups[z2];
				for (z3=0;z3<bg->m_oaBonds.GetSize();z3++)
				{
					bond = (CMolBond*)bg->m_oaBonds[z3];
					mprintf("%s%2d <--> %s%2d",((CAtom*)g_oaAtoms[sm->m_baAtomIndex[bond->m_iAtomType[0]]])->m_sName,bond->m_iAtom[0]+1,((CAtom*)g_oaAtoms[sm->m_baAtomIndex[bond->m_iAtomType[1]]])->m_sName,bond->m_iAtom[1]+1);
					if (z3 < bg->m_oaBonds.GetSize()-1)
						mprintf(", ");
				}
				mprintf("\n");
			}
		}*/
	}

	mprintf("    Creating angle list...\n");
	for (z=0;z<g_oaSingleMolecules.GetSize();z++) 
	{
		sm = (CSingleMolecule*)g_oaSingleMolecules[z];
//		mprintf("%d:\n",z+1);
		for (z2=0;z2<sm->m_oaMolAtoms.GetSize();z2++)
		{
			ma = (CMolAtom*)sm->m_oaMolAtoms[z2];
			for (z3=0;z3<ma->m_oaBonds.GetSize();z3++)
			{
				ma2 = (CMolAtom*)ma->m_oaBonds[z3];
				for (z4=z3+1;z4<ma->m_oaBonds.GetSize();z4++)
				{
					ma3 = (CMolAtom*)ma->m_oaBonds[z4];
			//		mprintf("  (%s%2d, %s%2d, %s%2d)\n",((CAtom*)g_oaAtoms[sm->m_baAtomIndex[ma2->m_iType]])->m_sName,ma2->m_iNumber+1,((CAtom*)g_oaAtoms[sm->m_baAtomIndex[ma->m_iType]])->m_sName,ma->m_iNumber+1,((CAtom*)g_oaAtoms[sm->m_baAtomIndex[ma3->m_iType]])->m_sName,ma3->m_iNumber+1);

					try { angle = new CMolAngle(); } catch(...) { angle = NULL; }
					if (angle == NULL) NewException((double)sizeof(CMolAngle),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					angle->m_iAtom[0] = ma2->m_iNumber;
					angle->m_iAtom[1] = ma->m_iNumber;
					angle->m_iAtom[2] = ma3->m_iNumber;
					angle->m_iAtomType[0] = ma2->m_iType;
					angle->m_iAtomType[1] = ma->m_iType;
					angle->m_iAtomType[2] = ma3->m_iType;
					angle->m_iMolAtom[0] = ma2->m_iMolAtomNumber;
					angle->m_iMolAtom[1] = z2;
					angle->m_iMolAtom[2] = ma3->m_iMolAtomNumber;
					angle->m_iAtomOffset[0] = ma2->m_iOffset;
					angle->m_iAtomOffset[1] = ma->m_iOffset;
					angle->m_iAtomOffset[2] = ma3->m_iOffset;
					sm->m_oaAngles.Add(angle);
				}
			}
		}
/*		if (z ==  0)
			mprintf("  %d Angles added.\n",oatemp.GetSize());*/
		if (sm->m_oaAngles.GetSize() > 1000)
			mprintf(WHITE,"      [");
		tfs = sm->m_oaAngles.GetSize() / 50.0;
		for (z2=0;z2<sm->m_oaAngles.GetSize();z2++)
		{
			if (sm->m_oaAngles.GetSize() > 1000)
				if (fmod(z2,tfs) < 1)
					mprintf(WHITE,"#");
			angle = (CMolAngle*)sm->m_oaAngles[z2];
			for (z3=0;z3<sm->m_oaAngleGroups.GetSize();z3++)
			{
				ag = (CMolAngleGroup*)sm->m_oaAngleGroups[z3];
				for (z4=0;z4<ag->m_oaAngles.GetSize();z4++)
				{
					angle2 = (CMolAngle*)ag->m_oaAngles[z4];
					if (( ((CMolAtom*)sm->m_oaMolAtoms[angle->m_iMolAtom[1]])->m_fAtomCode == ((CMolAtom*)sm->m_oaMolAtoms[angle2->m_iMolAtom[1]])->m_fAtomCode) &&
						(((( ((CMolAtom*)sm->m_oaMolAtoms[angle->m_iMolAtom[0]])->m_fAtomCode == ((CMolAtom*)sm->m_oaMolAtoms[angle2->m_iMolAtom[0]])->m_fAtomCode) &&
						( ((CMolAtom*)sm->m_oaMolAtoms[angle->m_iMolAtom[2]])->m_fAtomCode == ((CMolAtom*)sm->m_oaMolAtoms[angle2->m_iMolAtom[2]])->m_fAtomCode))) ||
						(( ((CMolAtom*)sm->m_oaMolAtoms[angle->m_iMolAtom[0]])->m_fAtomCode == ((CMolAtom*)sm->m_oaMolAtoms[angle2->m_iMolAtom[2]])->m_fAtomCode) &&
						( ((CMolAtom*)sm->m_oaMolAtoms[angle->m_iMolAtom[2]])->m_fAtomCode == ((CMolAtom*)sm->m_oaMolAtoms[angle2->m_iMolAtom[0]])->m_fAtomCode))))	
					{
						ag->m_oaAngles.Add(angle);
						goto _angledone;
					}
				}
			}

			try { ag = new CMolAngleGroup(); } catch(...) { ag = NULL; }
			if (ag == NULL) NewException((double)sizeof(CMolAngleGroup),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			ag->m_oaAngles.Add(angle);
			sm->m_oaAngleGroups.Add(ag);
_angledone:;
		}
		if (sm->m_oaAngles.GetSize() > 1000)
			mprintf(WHITE,"]\n");
/*		if (z == 0)
		{
			mprintf("  %d Angle groups found:\n",sm->m_oaAngleGroups.GetSize());
			for (z2=0;z2<sm->m_oaAngleGroups.GetSize();z2++)
			{
				mprintf("  - ");
				ag = (CMolAngleGroup*)sm->m_oaAngleGroups[z2];
				for (z3=0;z3<ag->m_oaAngles.GetSize();z3++)
				{
					angle = (CMolAngle*)ag->m_oaAngles[z3];
					mprintf("  (%s%2d, %s%2d, %s%2d)",((CAtom*)g_oaAtoms[sm->m_baAtomIndex[angle->m_iAtomType[0]]])->m_sName,angle->m_iAtom[0]+1,((CAtom*)g_oaAtoms[sm->m_baAtomIndex[angle->m_iAtomType[1]]])->m_sName,angle->m_iAtom[1]+1,((CAtom*)g_oaAtoms[sm->m_baAtomIndex[angle->m_iAtomType[2]]])->m_sName,angle->m_iAtom[2]+1);
					if (z3 < ag->m_oaAngles.GetSize()-1)
						mprintf(", ");
				}
				mprintf("\n");
			}
		}*/
	}

	mprintf("    Grouping together equivalent molecules...\n");
	for (z=0;z<g_oaSingleMolecules.GetSize();z++) // Alle Molekuelvertreter durchgehen
	{
//		mprintf("Pruefe SM %d...\n",z+1);
		sm = (CSingleMolecule*)g_oaSingleMolecules[z];
		for (z2=0;z2<g_oaMolecules.GetSize();z2++) // Alle schon vorhandenen Molekueltemplates durchgehen
		{
			m = (CMolecule*)g_oaMolecules[z2];
			sm2 = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[0]];
/*			if (m->m_baAtomIndex.GetSize() != sm->m_baAtomIndex.GetSize())
				goto _cont; // Nich die Gleiche Anzahl verschiedener Elemente -> Keine Uebereinstimmung
			for (z3=0;z3<m->m_baAtomIndex.GetSize();z3++)
			{
				if (m->m_baAtomIndex[z3] != sm->m_baAtomIndex[z3])
					goto _cont; // Elemente an Position z3 unterscheiden sich -> keine Uebereinstimmung
				if (m->m_waAtomCount[z3] != ((CxIntArray*)sm->m_oaAtomOffset[z3])->GetSize())
					goto _cont; // Anzahl an Atomen dieser Elementsorte unterscheidet sich -> Keine Uebereinstimmung*/
//				mprintf("  Stimmt mit M%d ueberein.\n",z2);
//				((CMolecule*)g_oaMolecules[z2])->Count++;
//				g_pSingleMolecules[z].m_iMoleculeOffset = z2;
//			}
			if (sm->m_oaMolAtoms.GetSize() != sm2->m_oaMolAtoms.GetSize())
				goto _cont;
			for (z3=0;z3<sm->m_oaMolAtoms.GetSize();z3++)
				if (((CMolAtom*)sm->m_oaMolAtoms[z3])->m_fAtomCode != ((CMolAtom*)sm2->m_oaMolAtoms[z3])->m_fAtomCode)
					goto _cont;
			sm->m_iMolSMIndex = m->m_laSingleMolIndex.GetSize();
			m->m_laSingleMolIndex.Add(z);
//			sm->m_iMolType = z2;
			goto _fert;
			_cont:;
		}

		try { m = new CMolecule(); } catch(...) { m = NULL; }
		if (m == NULL) NewException((double)sizeof(CMolecule),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		g_oaMolecules.Add(m);
//		m->Count = 1; // Neues Molekueltemplate hinzufuegen
//		m->Elements = ((CSingleMolecule*)g_oaSingleMolecules[z])->m_iElements;
		m->m_iAtomGes = 0;
		f = 0;
		for (z2=0;z2<sm->m_oaAtomOffset.GetSize();z2++)
		{
			m->m_iAtomGes += ((CxIntArray*)sm->m_oaAtomOffset[z2])->GetSize();
			f += ((CAtom*)g_oaAtoms[sm->m_baAtomIndex[z2]])->m_pElement->m_fMass * ((CxIntArray*)sm->m_oaAtomOffset[z2])->GetSize();
		}
		m->m_iAtomGesNoVirt = m->m_iAtomGes;
		m->m_fMass = f;
		((CSingleMolecule*)g_oaSingleMolecules[z])->m_iMolType = g_oaMolecules.GetSize()-1;
		for (z2=0;z2<((CSingleMolecule*)g_oaSingleMolecules[z])->m_baAtomIndex.GetSize();z2++)
		{
			m->m_baAtomIndex.Add(((CSingleMolecule*)g_oaSingleMolecules[z])->m_baAtomIndex[z2]);
			m->m_waAtomCount.Add(((CxIntArray*)((CSingleMolecule*)g_oaSingleMolecules[z])->m_oaAtomOffset[z2])->GetSize());
//			m->AtomCount[z2] = ((CSingleMolecule*)g_oaSingleMolecules[z])->m_iAtomCount[z2];
		}
		sm->m_iMolSMIndex = m->m_laSingleMolIndex.GetSize();
		m->m_laSingleMolIndex.Add(z);
//		g_pSingleMolecules[z].m_iMoleculeOffset = g_oaMolecules.GetSize();
		m->BuildName();

		// Standardmaessig geom. Zentrum als Dipolzentrum
		m->m_iDipoleCenterType = m->m_baAtomIndex.GetSize();
		m->m_iDipoleCenterIndex = 0;

//		mprintf("### Erzeuge neues Molekuel %d \"%s\".\n",g_oaMolecules.GetSize(),m->Name);
//		g_oaMolecules.GetSize()++;
		_fert:;
	}
	mprintf("    Found %d unique molecule types.\n",g_oaMolecules.GetSize());

	// Sort by molecular mass
	mprintf("    Sorting molecule types by molecular mass...\n");
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
//		mprintf("@ z=%d\n",z);
		f = 0;
		for (z2=z;z2<g_oaMolecules.GetSize();z2++)
		{
//			mprintf("@   z2=%d\n",z2);
			if (((CMolecule*)g_oaMolecules[z2])->m_fMass > f)
			{
//				mprintf("@     %f > %f --> z3 = %d\n",((CMolecule*)g_oaMolecules[z2])->m_fMass,f,z2);
				f = ((CMolecule*)g_oaMolecules[z2])->m_fMass;
				z3 = z2;
			}
		}
		if (f > 0)
		{
//			mprintf("@ Swapping %d with %d.\n",z,z3);
			m = (CMolecule*)g_oaMolecules[z3];
			g_oaMolecules[z3] = g_oaMolecules[z];
			g_oaMolecules[z] = m;
		}
	}

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		((CMolecule*)g_oaMolecules[z])->m_iIndex = z;
		for (z2=0;z2<((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex.GetSize();z2++)
			((CSingleMolecule*)g_oaSingleMolecules[((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex[z2]])->m_iMolType = z;
	}


/*	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[z];
		sm = (CSingleMolecule*)g_oaSingleMolecules[((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex[0]];
		mprintf("Molecule %d Ring systems:\n",z+1);
		for (z2=0;z2<sm->m_oaRings.GetSize();z2++)
		{
			wa = (CxIntArray*)sm->m_oaRings[z2];
			mprintf("  - Ring %d: ",z2+1);
			for (z3=0;z3<wa->GetSize();z3++)
			{
				mprintf("%d",(*wa)[z3]);
				if (z3+1 < wa->GetSize())
					mprintf(", ");
			}
			mprintf("\n");
		}
		mprintf("\n");
	}*/

	ti = 0;
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		sm = (CSingleMolecule*)g_oaSingleMolecules[((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex[0]];
		ti += sm->m_oaRings.GetSize();
	}

	mprintf("    Found %d rings.\n",ti);
	if (ti != 0)
	{
		mprintf("    Refining ring systems...\n");

		try { wa = new CxIntArray(); } catch(...) { wa = NULL; }
		if (wa == NULL) NewException((double)sizeof(CxIntArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		// Process ring systems
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			m = (CMolecule*)g_oaMolecules[z];
			sm = (CSingleMolecule*)g_oaSingleMolecules[((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex[0]];
			if (sm->m_oaRings.GetSize() > 100)
			{
				eprintf("      More than 100 rings in molecule %d, skipping refinement. Ring system may be improper.\n",z+1);
				continue;
			}
			ti2 = 0;
_again:
			ti2++;
			if (ti2 > 1000)
			{
				eprintf("      Too many iterations in molecule %d, aborting refinement. Ring system may be improper.\n",z+1);
				continue;
			}
//			mprintf("### again\n");
			for (z2=0;z2<sm->m_oaRings.GetSize();z2++)
			{
				waz2 = (CxIntArray*)sm->m_oaRings[z2];
				for (z3=z2+1;z3<sm->m_oaRings.GetSize();z3++)
				{
					waz3 = (CxIntArray*)sm->m_oaRings[z3];

//					mprintf("*A* Checking if %d(%d) contains %d(%d)...\n",z2+1,waz2->GetSize(),z3+1,waz3->GetSize());
					// Check if waz2 contains waz3
					wa->SetSize(waz2->GetSize());
					for (z4=0;z4<waz2->GetSize();z4++)
						(*wa)[z4] = 0;
					for (z4=0;z4<waz3->GetSize();z4++)
					{
						for (z5=0;z5<waz2->GetSize();z5++)
						{
							if ((*waz3)[z4] == (*waz2)[z5])
							{
								(*wa)[z5] = 1;
								goto _found2;
							}
						}
						goto _notfound2;
_found2:;
					}
//					mprintf("  Yes!\n");

					try { waneu = new CxIntArray(); } catch(...) { waneu = NULL; }
					if (waneu == NULL) NewException((double)sizeof(CxIntArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					for (z4=0;z4<waz2->GetSize();z4++)
					{
						if (z4 == 0)
						{
							if (((*wa)[z4] == 0) || ((*wa)[z4+1] == 0) || ((*wa)[waz2->GetSize()-1] == 0))
								waneu->Add((*waz2)[z4]);
						} else if (z4 == waz2->GetSize()-1)
						{
							if (((*wa)[z4] == 0) || ((*wa)[z4-1] == 0) || ((*wa)[0] == 0))
								waneu->Add((*waz2)[z4]);
						} else
						{
							if (((*wa)[z4] == 0) || ((*wa)[z4+1] == 0) || ((*wa)[z4-1] == 0))
								waneu->Add((*waz2)[z4]);
						}
					}
//					mprintf("   Took %d from %d atoms from waz2.\n",waneu->GetSize(),waz2->GetSize());
					i1 = waz2->GetSize();
					delete waz2;
					sm->m_oaRings[z2] = waneu;
					if (waneu->GetSize() != i1)
						goto _again;
_notfound2:

//					mprintf("*B* Checking if %d contains %d...\n",z3+1,z2+1);
					// Check if waz3 contains waz2
					wa->SetSize(waz3->GetSize());
					for (z4=0;z4<waz3->GetSize();z4++)
						(*wa)[z4] = 0;
					for (z4=0;z4<waz2->GetSize();z4++)
					{
						for (z5=0;z5<waz3->GetSize();z5++)
						{
							if ((*waz2)[z4] == (*waz3)[z5])
							{
								(*wa)[z5] = 1;
								goto _found3;
							}
						}
						goto _notfound3;
_found3:;
					}
//					mprintf("  Yes!\n");

					try { waneu = new CxIntArray(); } catch(...) { waneu = NULL; }
					if (waneu == NULL) NewException((double)sizeof(CxIntArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					for (z4=0;z4<waz3->GetSize();z4++)
					{
						if (z4 == 0)
						{
							if (((*wa)[z4] == 0) || ((*wa)[z4+1] == 0) || ((*wa)[waz3->GetSize()-1] == 0))
								waneu->Add((*waz3)[z4]);
						} else if (z4 == waz3->GetSize()-1)
						{
							if (((*wa)[z4] == 0) || ((*wa)[z4-1] == 0) || ((*wa)[0] == 0))
								waneu->Add((*waz3)[z4]);
						} else
						{
							if (((*wa)[z4] == 0) || ((*wa)[z4+1] == 0) || ((*wa)[z4-1] == 0))
								waneu->Add((*waz3)[z4]);
						}
					}
//					mprintf("   Took %d from %d atoms from waz3.\n",waneu->GetSize(),waz3->GetSize());
					i1 = waz3->GetSize();
					delete waz3;
					sm->m_oaRings[z3] = waneu;
					if (waneu->GetSize() != i1)
						goto _again;
_notfound3:;
				}
			}
		}
		delete wa;

		mprintf("    Assigning ring systems to molecule types...\n");
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			m = (CMolecule*)g_oaMolecules[z];
			sm = (CSingleMolecule*)g_oaSingleMolecules[((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex[0]];
			for (z2=0;z2<sm->m_oaRings.GetSize();z2++)
			{
				wa = (CxIntArray*)sm->m_oaRings[z2];

				try { waz2 = new CxIntArray(); } catch(...) { waz2 = NULL; }
				if (waz2 == NULL) NewException((double)sizeof(CxIntArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
				
				try { waz3 = new CxIntArray(); } catch(...) { waz3 = NULL; }
				if (waz3 == NULL) NewException((double)sizeof(CxIntArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
				
				m->m_oaRingAtomTypes.Add(waz2);
				m->m_oaRingAtoms.Add(waz3);
				for (z3=0;z3<wa->GetSize();z3++)
				{
					for (z4=0;z4<sm->m_baAtomIndex.GetSize();z4++)
					{
						for (z5=0;z5<((CxIntArray*)sm->m_oaAtomOffset[z4])->GetSize();z5++)
						{
							if (((CxIntArray*)sm->m_oaAtomOffset[z4])->GetAt(z5) == wa->GetAt(z3))
							{
								waz2->Add(z4);
								waz3->Add(z5);
								goto _done;
							}
						}
					}
_done:;
				}
			}
		}

		mprintf("    Sorting rings by size...\n");
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			m = (CMolecule*)g_oaMolecules[z];

			for (z2=0;z2<m->m_oaRingAtoms.GetSize();z2++)
			{
				i1 = 0;
				ti = -1;
				for (z3=z2;z3<m->m_oaRingAtoms.GetSize();z3++)
				{
					if (((CxIntArray*)m->m_oaRingAtoms[z3])->GetSize() > i1)
					{
						i1 = ((CxIntArray*)m->m_oaRingAtoms[z3])->GetSize();
						ti = z3;
					}
				}
				if ((ti != -1) && (ti != z2))
				{
					wa = (CxIntArray*)m->m_oaRingAtoms[z2];
					m->m_oaRingAtoms[z2] = m->m_oaRingAtoms[ti];
					m->m_oaRingAtoms[ti] = wa;
					wa = (CxIntArray*)m->m_oaRingAtomTypes[z2];
					m->m_oaRingAtomTypes[z2] = m->m_oaRingAtomTypes[ti];
					m->m_oaRingAtomTypes[ti] = wa;
				}
			}
		}
	}

/*	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[z];
		sm = (CSingleMolecule*)g_oaSingleMolecules[((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex[0]];
		mprintf("Molecule %d Ring systems:\n",z+1);
		for (z2=0;z2<sm->m_oaRings.GetSize();z2++)
		{
			wa = (CxIntArray*)sm->m_oaRings[z2];
			mprintf("  - Ring %d: ",z2+1);
			for (z3=0;z3<wa->GetSize();z3++)
			{
				mprintf("%d",(*wa)[z3]);
				if (z3+1 < wa->GetSize())
					mprintf(", ");
			}
			mprintf("\n");
		}
		mprintf("\n");
	}*/

/*	mprintf("\n%d Molekuelsorten gefunden.\n",g_oaMolecules.GetSize());
	for (z=0;z<g_oaMolecules.GetSize();z++)
	mprintf("Molekuel \"%s\": %d Atome gesamt, %d mal in der Simulation.\n",g_pMolecules[z].Name,g_pMolecules[z].AtomGes,g_pMolecules[z].Count);
	*/
/*	mprintf("\n");
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		mprintf("%d ",z+1);
		((CMolecule*)g_oaMolecules[z])->Dump();
	}*/
	mprintf("    Molecule recognition finished.\n");
	BTOUT; 
}

void CTimeStep::RECURSION_MegaTree(int i, char *ta, int depth, unsigned long bmask, bool w, int *stack)
{
	BTIN;
	int z, z2;
	int nblist[20], nbs;

	stack[depth] = i;
	ta[i] = 1;
	for (z=1;z<depth;z++)
	{
		if ((bmask & (int)pow(2.0,z)) != 0)
			mprintf(WHITE,"|  ");
				else mprintf("   ");
	}
	if (depth != 0)
	{
		if (w)
			mprintf(WHITE,"|--");
				else mprintf(WHITE,"`--");
	}
	mprintf("%s(%d)",((CAtom*)g_oaAtoms[g_baAtomIndex[i]])->m_sName,i+1);
	nbs = 0;
	for (z=0;z<(long)m_iGesAtomCount;z++) // Schon mal alle Nachbarn raussuchen
	{
		if (z == i)
			continue;
		if (BondRange(i,z,NULL))
		{
			if (ta[z] != 0)
			{
				if ((depth > 0) && (z != stack[depth-1]))
				{
					mprintf(GREEN,"  <-- Ring closure: ");
					mprintf("%s(%d)",((CAtom*)g_oaAtoms[g_baAtomIndex[stack[depth]]])->m_sName,stack[depth]+1);
					z2 = depth-1;
					while ((stack[z2] != z) && (z2 >= 0))
					{
						mprintf(" - %s(%d)",((CAtom*)g_oaAtoms[g_baAtomIndex[stack[z2]]])->m_sName,stack[z2]+1);
						z2--;
					}
					mprintf(" - %s(%d)",((CAtom*)g_oaAtoms[g_baAtomIndex[z]])->m_sName,z+1);
//					mprintf("Ringschluss! [d-1=%d,d=%d,i=%d] %s(%d)\n",stack[depth-1]+1,stack[depth]+1,i+1,((CAtom*)g_oaAtoms[g_baAtomIndex[z]])->m_sName,z+1);
				}
				continue;
			}
			nblist[nbs] = z;
			nbs++;
		}
	}
	mprintf("\n");
	for (z=0;z<nbs;z++) // Fuer das aktuelle Atom z2 alle Nachbarn finden
	{
		if (ta[nblist[z]] != 0) // Der Nachbar wurde uns weggeschnappt -> Ringschluss
		{
			for (z2=1;z2<depth+1;z2++)
			{
				if ((bmask & (int)pow(2.0,z2)) != 0)
					mprintf(WHITE,"|  ");
						else mprintf("   ");
			}
			if (z+1 < nbs)
				mprintf(WHITE,"|--");
					else mprintf("`--");
			mprintf("%s(%d)",((CAtom*)g_oaAtoms[g_baAtomIndex[nblist[z]]])->m_sName,nblist[z]+1);
			mprintf(GREEN,"  <-- Ring closure\n");
		} else
		{
			if (z+1 == nbs)
				bmask -= (int)pow(2.0,depth+1);
			RECURSION_MegaTree(nblist[z],ta,/*sm,*/depth+1,bmask,(z+1==nbs)?false:true,stack);
		}
	}
	BTOUT; 
}

void CTimeStep::PrintMegaTree()
{
	BTIN;
	int z;
	char ta[16384];
	int stack[16384];

	for (z=0;z<16384;z++)
		ta[z] = 0;
	for (z=0;z<(long)m_iGesAtomCount;z++)
	{
		if (ta[z] != 0) // Dieses Atom wurde bereits in irgendein Molekuel eingebaut
			continue;
		mprintf(YELLOW,"\nThe next molecule starts with %s(%d):\n",m_paLabels[z],z+1);
		RECURSION_MegaTree(z,ta,0,0xFFFFFFFF,true,stack);
	}
	mprintf("\n");
	BTOUT; 
}

void CTimeStep::PrintMatrix(bool onlyfirst, bool onlybind)
{
	BTIN;
	int z, z2, z3, z4, z5, z6, z7, z8, ti, ti2;
	float tf;
	bool b, c, noh;
	CxVector3 vec1, vec2;
	CSingleMolecule *sm;
	CMolecule *m;
	CMolAtom *ma, *ma0;
	CxIntArray *wa, *wat;

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[z];
		for (z6=0;z6<(onlyfirst?1:m->m_laSingleMolIndex.GetSize());z6++)
		{
			mprintf(YELLOW,"\n*** Molecule %d: %s; Representant %d; Distances in pm ***\n",z+1,m->m_sName,z6+1);
			if ((m->m_baAtomIndex.GetSize() == 1) && (m->m_waAtomCount[0] == 1))
			{
				mprintf("[only 1 Atom]\n");
				continue;
			}
			if (m->m_iAtomGes > 200)
			{
				mprintf(WHITE,"\n    This molecule has > 200 atoms. Output would flood the screen. Skipping.\n");
				continue;
			}
			noh = false;
_matagain:
			if (m->m_oaRingAtoms.GetSize() != 0)
				mprintf(GREEN,"    (ring bonds are shown in green)\n");
			mprintf("\n");
			mprintf("   ");
			for (z2=0;z2<m->m_baAtomIndex.GetSize();z2++)
			{
				if (noh) // Skip H atoms
					if (mystricmp(((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_sName,"H") == 0)
						continue;
				for (z3=0;z3<m->m_waAtomCount[z2];z3++)
				{
					if (((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_sName[1] == 0)
						mprintf(" ");
					mprintf(WHITE,"%s%d",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_sName,z3+1);	
					if (z3+1 < 10)
						mprintf(" ");
				}
			}
			mprintf("\n");
			for (z2=0;z2<m->m_baAtomIndex.GetSize();z2++)
			{
				if (noh) // Skip H atoms
					if (mystricmp(((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_sName,"H") == 0)
						continue;
				for (z3=0;z3<m->m_waAtomCount[z2];z3++)
				{
					mprintf(WHITE,"%s%d",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_sName,z3+1);
					if (((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_sName[1] == 0)
						mprintf(" ");
					if (z3+1 < 10)
						mprintf(" ");
					for (z4=0;z4<m->m_baAtomIndex.GetSize();z4++)
					{
						if (noh) // Skip H atoms
							if (mystricmp(((CAtom*)g_oaAtoms[m->m_baAtomIndex[z4]])->m_sName,"H") == 0)
								continue;
						for (z5=0;z5<m->m_waAtomCount[z4];z5++)
						{
							if ((z2 == z4) && (z3 == z5))
							{
								mprintf(BLUE,"*** ");
								continue;
							}
							ti = ((CxIntArray*)((CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z6]])->m_oaAtomOffset[z2])->GetAt(z3);
							vec1 = g_TimeStep.m_vaCoords[ti];
							ti2 = ((CxIntArray*)((CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z6]])->m_oaAtomOffset[z4])->GetAt(z5);
							vec2 = g_TimeStep.m_vaCoords[ti2];
							vec1 -= vec2;
							tf = vec1.GetLength();
							if ((g_TimeStep.BondRange(ti,ti2,NULL)) || (!onlybind))
							{
								for (z8=0;z8<m->m_oaRingAtomTypes.GetSize();z8++)
								{
									wat = (CxIntArray*)m->m_oaRingAtomTypes[z8];
									wa = (CxIntArray*)m->m_oaRingAtoms[z8];
									for (z7=0;z7<wa->GetSize();z7++)
									{
										if (((*wat)[z7] == z2) && ((*wa)[z7] == z3))
										{
											if (z7 == 0)
											{
												if (((*wat)[z7+1] == z4) && ((*wa)[z7+1] == z5))
													goto _green;
												if (((*wat)[wa->GetSize()-1] == z4) && ((*wa)[wa->GetSize()-1] == z5))
													goto _green;
											} else if (z7 == wa->GetSize()-1)
											{
												if (((*wat)[0] == z4) && ((*wa)[0] == z5))
													goto _green;
												if (((*wat)[z7-1] == z4) && ((*wa)[z7-1] == z5))
													goto _green;
											} else
											{
												if (((*wat)[z7+1] == z4) && ((*wa)[z7+1] == z5))
													goto _green;
												if (((*wat)[z7-1] == z4) && ((*wa)[z7-1] == z5))
													goto _green;
											}
										}
									}
								}
								mprintf("%3.0f ",tf);
								continue;
_green:
								mprintf(GREEN,"%3.0f ",tf);
							} else mprintf(" -  ");
						}
					}	
					mprintf("\n");
				}
			}
			if ((!noh) && (m->m_iAtomGes > 30))
			{
				mprintf(WHITE,"\n    This was a very large molecule.\n    Printing the matrix again without H atoms.\n");
				noh = true;
				goto _matagain;
			}
//			if (((!onlyfirst) && ((z2 < ((CMolecule*)g_oaMolecules[z])->Elements-1) || (g_oaMolecules.GetSize()-1))) || (z < g_oaMolecules.GetSize()-1))
//				mprintf(">>>");
		}

		if (m->m_oaRingAtoms.GetSize() != 0)
		{
			mprintf("\n");
			for (z2=0;z2<m->m_oaRingAtoms.GetSize();z2++)
			{
				mprintf("    %d-ring: ",((CxIntArray*)m->m_oaRingAtoms[z2])->GetSize());
				if (((CxIntArray*)m->m_oaRingAtoms[z2])->GetSize() > 20)
				{
					for (z3=0;z3<5;z3++)
					{
						mprintf("%s%d",((CAtom*)g_oaAtoms[m->m_baAtomIndex[((CxIntArray*)m->m_oaRingAtomTypes[z2])->GetAt(z3)]])->m_sName,((CxIntArray*)m->m_oaRingAtoms[z2])->GetAt(z3)+1);
						if (z3+1 < ((CxIntArray*)m->m_oaRingAtoms[z2])->GetSize())
							mprintf(" - ");
					}
					mprintf(RED,"...");
					mprintf(" - ");
					for (z3=((CxIntArray*)m->m_oaRingAtoms[z2])->GetSize()-5;z3<((CxIntArray*)m->m_oaRingAtoms[z2])->GetSize();z3++)
					{
						mprintf("%s%d",((CAtom*)g_oaAtoms[m->m_baAtomIndex[((CxIntArray*)m->m_oaRingAtomTypes[z2])->GetAt(z3)]])->m_sName,((CxIntArray*)m->m_oaRingAtoms[z2])->GetAt(z3)+1);
						if (z3+1 < ((CxIntArray*)m->m_oaRingAtoms[z2])->GetSize())
							mprintf(" - ");
					}
				} else
				{
					for (z3=0;z3<((CxIntArray*)m->m_oaRingAtoms[z2])->GetSize();z3++)
					{
						mprintf("%s%d",((CAtom*)g_oaAtoms[m->m_baAtomIndex[((CxIntArray*)m->m_oaRingAtomTypes[z2])->GetAt(z3)]])->m_sName,((CxIntArray*)m->m_oaRingAtoms[z2])->GetAt(z3)+1);
						if (z3+1 < ((CxIntArray*)m->m_oaRingAtoms[z2])->GetSize())
							mprintf(" - ");
					}
				}
				mprintf("\n");
				if (z2 >= 99)
				{
					mprintf(RED,"\n    Only showing the first 100 rings.\n");
					break;
				}
			}
		}

		c = false;
		sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[0]];
		for (z2=0;z2<m->m_baAtomIndex.GetSize();z2++)
		{
			b = false;
			ma = NULL;
			for (z3=0;z3<sm->m_oaMolAtoms.GetSize();z3++)
			{
				if (((CMolAtom*)sm->m_oaMolAtoms[z3])->m_iType != z2)
					continue;
				ma0 = ma;
				ma = (CMolAtom*)sm->m_oaMolAtoms[z3];
				if (ma0 == NULL)
					continue;
				z4 = ma0->m_iNumber;
				if (ma->m_fAtomCode == ma0->m_fAtomCode)
				{
					if (b)
					{
						mprintf(", %s%d",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_sName,ma->m_iNumber+1);
					} else
					{
						if (!c)
						{
							c = true;
							mprintf("\n");
						}
						mprintf("    Atoms %s%d, %s%d",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_sName,z4+1,((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_sName,ma->m_iNumber+1);
						b = true;
					}
				} else if (b)
				{
					mprintf(" are equivalent.\n");
					b = false;
				}
			}
			if (b)
				mprintf(" are equivalent.\n");
		}
	} // Tabelle Ende
	BTOUT; 
}

void CTimeStep::Transform(const CxMatrix3 &mat)
{
	BTIN;
	int z;
//	CxVector3 v;

	for (z=0;z<g_iGesVirtAtomCount;z++)
		m_vaCoords[z] = mat * m_vaCoords[z];

	if (g_bUseVelocities)
		for (z=0;z<g_iGesVirtAtomCount;z++)
			m_vaVelocities[z] = mat * m_vaVelocities[z];

	if (g_bUseForces)
		for (z=0;z<g_iGesVirtAtomCount;z++)
			m_vaForces[z] = mat * m_vaForces[z];
	BTOUT; 
}

void CTimeStep::SubVelocities(const CxVector3 &vec)
{
	BTIN;
	int z;

	for (z=0;z<g_iGesVirtAtomCount;z++)
		m_vaVelocities[z] -= vec;
	BTOUT; 
}

void CTimeStep::FoldMolecules()
{
	BTIN;
	int z, z2, z3, z4;
	CxVector3 v, vc;
	CMolecule *m;
	CSingleMolecule *sm;

//	mprintf("*** Fold ***\n");
	
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[z];
		if (m->m_bPseudo)
			continue;
		for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
		{
			sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];

			// Massenzentrum
			v = m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[m->m_baAtomIndex.GetSize()-1])->GetAt(1)];

/*			mprintf("  .Mol ");
			v.Dump();
			mprintf("\n");			

*/			vc = CxVector3(0,0,0);
			
			while (v[0] < -g_fBoxX/2) { v[0] += g_fBoxX; vc[0] += g_fBoxX; }
			while (v[0] >= g_fBoxX/2) { v[0] -= g_fBoxX; vc[0] -= g_fBoxX; }
			while (v[1] < -g_fBoxY/2) { v[1] += g_fBoxY; vc[1] += g_fBoxY; }
			while (v[1] >= g_fBoxY/2) { v[1] -= g_fBoxY; vc[1] -= g_fBoxY; }
			while (v[2] < -g_fBoxZ/2) { v[2] += g_fBoxZ; vc[2] += g_fBoxZ; }
			while (v[2] >= g_fBoxZ/2) { v[2] -= g_fBoxZ; vc[2] -= g_fBoxZ; }
			
			if ((vc[0] == 0) && (vc[1] == 0) && (vc[2] == 0))
				 continue;
			
			for (z3=0;z3<m->m_baAtomIndex.GetSize();z3++)
				for (z4=0;z4<m->m_waAtomCount[z3];z4++)
					 m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)] += vc;

			if (g_bWannier)
				for (z3=0;z3<sm->m_laWannier.GetSize();z3++)
					 m_vaCoords[sm->m_laWannier[z3]] += vc;
		}
	}
//	mprintf("   Fold Done\n");
	BTOUT;
}

void CTimeStep::FoldAtoms()
{
	BTIN;
	int z;

	for (z=0;z<g_iGesVirtAtomCount;z++)
	{
		while (m_vaCoords[z][0] < -g_fBoxX/2) m_vaCoords[z][0] += g_fBoxX;
		while (m_vaCoords[z][0] >= g_fBoxX/2) m_vaCoords[z][0] -= g_fBoxX;
		while (m_vaCoords[z][1] < -g_fBoxY/2) m_vaCoords[z][1] += g_fBoxY;
		while (m_vaCoords[z][1] >= g_fBoxY/2) m_vaCoords[z][1] -= g_fBoxY;
		while (m_vaCoords[z][2] < -g_fBoxZ/2) m_vaCoords[z][2] += g_fBoxZ;
		while (m_vaCoords[z][2] >= g_fBoxZ/2) m_vaCoords[z][2] -= g_fBoxZ;
	}
	BTOUT; 
}

void CTimeStep::CenterPos(const CxVector3 &vec)
{
	BTIN;
	int z/*, z2, z3, z4*/;

/*	 for (z=0;z<g_oaMolecules.GetSize();z++)
		 for (z2=0;z2<((CMolecule*)g_oaMolecules[z])->Count;z2++)
			 for (z3=0;z3<((CMolecule*)g_oaMolecules[z])->Elements;z3++)
				 for (z4=0;z4<((CMolecule*)g_oaMolecules[z])->AtomCount[z3];z4++)
					 m_vaCoords[((CSingleMolecule*)g_oaSingleMolecules[((CMolecule*)g_oaMolecules[z])->SingleMolIndex[z2]])->m_iAtomOffset[z3][z4]] -= vec;*/

	for (z=0;z<g_iGesVirtAtomCount;z++)
		m_vaCoords[z] -= vec;
	BTOUT; 
}

void CTimeStep::AddAtoms()
{
	BTIN;
	char *q;
	int z;
	bool labeledatoms;

	labeledatoms = false;
	for (z=0;z<(long)m_iGesAtomCount;z++)
	{
		q = (char*)m_paLabels[z];
		q+=strlen(q)-1;
/*		while ((*q >= '0') && (*q <= '9'))
		{
			if (!labeledatoms)
			{
				mprintf(">>>\n>>> The atoms in the input file are numbered.\n>>> Ignoring this, using own numbers ;-)\n>>>\n");
				labeledatoms = true;
			}
			*q = 0;
			q--;
		}*/
//		printf("AddAtoms: \"%s\"\n",q);
		xAddAtom((char*)m_paLabels[z]);
	}
	BTOUT; 
}

void CTimeStep::WriteTimestep(FILE *a)
{
	BTIN;
	int z, z2, z3, z4, z0;
	CMolecule *m;
	CSingleMolecule *sm;

	if (g_bSaveVirtAtoms)
		fprintf(a,"  %d\n",g_iGesVirtAtomCount);
			else fprintf(a,"  %d\n",g_iGesAtomCount);
	if (m_pComment != NULL)
		fprintf(a,"%s\n",m_pComment);
			else fprintf(a,"\n");

	if (g_bWriteAtomwise)
	{
		for (z0=0;z0<g_oaAtoms.GetSize();z0++)
		{
			for (z=0;z<g_oaMolecules.GetSize();z++)
			{
				m = (CMolecule*)g_oaMolecules[z];
				if (m->m_bPseudo)
					continue;
				for (z3=0;z3<m->m_baAtomIndex.GetSize();z3++)
				{
					if (m->m_baAtomIndex[z3] != z0)
						continue;
					if ((!g_bSaveVirtAtoms) && (m->m_baAtomIndex[z3] == g_iVirtAtomType))
						continue;
					for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
					{
						sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];
						for (z4=0;z4<((CxIntArray*)sm->m_oaAtomOffset[z3])->GetSize();z4++)
							fprintf(a,"  %s  %8.5f  %8.5f  %8.5f\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z3]])->m_sName,m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][0]/100.0f,m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][1]/100.0f,m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][2]/100.0f);
					}
				}
			}
		}
	} else
	{
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			m = (CMolecule*)g_oaMolecules[z];
			if (m->m_bPseudo)
				continue;
			for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
			{
				sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];
				for (z3=0;z3<m->m_baAtomIndex.GetSize();z3++)
				{
					if ((!g_bSaveVirtAtoms) && (m->m_baAtomIndex[z3] == g_iVirtAtomType))
						continue;
					for (z4=0;z4<((CxIntArray*)sm->m_oaAtomOffset[z3])->GetSize();z4++)
						fprintf(a,"  %s  %8.5f  %8.5f  %8.5f\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z3]])->m_sName,m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][0]/100.0f,m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][1]/100.0f,m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][2]/100.0f);
				}
			}
		}
	}
			
	BTOUT; 
}
	
void CTimeStep::WriteTimestepNb(FILE *a, CNbSet *nbs)
{
	BTIN;
	int z, z2, z3, z4, z0, n;
	CMolecule *m;
	CSingleMolecule *sm;

	n = 0;
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		if (nbs->m_oaConditionGroups[z] == NULL)
			continue;
		m = (CMolecule*)g_oaMolecules[z];
		for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
		{
			if (((CConditionGroup*)nbs->m_oaConditionGroups[z])->Contains(z2))
			{
				n += m->m_iAtomGes;
				if (!g_bSaveVirtAtoms)
					n -= m->m_laVirtualAtoms.GetSize();
			}
		}
	}

	fprintf(a,"  %d\n",n);
	if (m_pComment != NULL)
		fprintf(a,"%s\n",m_pComment);
			else fprintf(a,"\n");

	if (g_bWriteAtomwise)
	{
		for (z0=0;z0<g_oaAtoms.GetSize();z0++)
		{
			for (z=0;z<g_oaMolecules.GetSize();z++)
			{
				if (nbs->m_oaConditionGroups[z] == NULL)
					continue;
				m = (CMolecule*)g_oaMolecules[z];
				for (z3=0;z3<m->m_baAtomIndex.GetSize();z3++)
				{
					if (m->m_baAtomIndex[z3] != z0)
						continue;
					if ((!g_bSaveVirtAtoms) && (m->m_baAtomIndex[z3] == g_iVirtAtomType))
						continue;
					for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
					{
						if (((CConditionGroup*)nbs->m_oaConditionGroups[z])->Contains(z2))
						{
							sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];
							for (z4=0;z4<((CxIntArray*)sm->m_oaAtomOffset[z3])->GetSize();z4++)
								fprintf(a,"  %s  %8.5f  %8.5f  %8.5f\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z3]])->m_sName,m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][0]/100.0f,m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][1]/100.0f,m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][2]/100.0f);
						}
					}
				}
			}
		}
	} else
	{
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			if (nbs->m_oaConditionGroups[z] == NULL)
				continue;
			m = (CMolecule*)g_oaMolecules[z];
			for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
			{
				if (((CConditionGroup*)nbs->m_oaConditionGroups[z])->Contains(z2))
				{
					sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];
					for (z3=0;z3<m->m_baAtomIndex.GetSize();z3++)
					{
						if ((!g_bSaveVirtAtoms) && (m->m_baAtomIndex[z3] == g_iVirtAtomType))
							continue;
						for (z4=0;z4<((CxIntArray*)sm->m_oaAtomOffset[z3])->GetSize();z4++)
							fprintf(a,"  %s  %8.5f  %8.5f  %8.5f\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z3]])->m_sName,m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][0]/100.0f,m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][1]/100.0f,m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][2]/100.0f);
					}
				}
			}
		}
	}
			
	BTOUT; 
}

/*void CTimeStep::WriteTimestepNb(int refmol, FILE *a)
{
	BTIN;
	int z, c, z2, z3, z4, z0;
	int ti;
	CMolecule *m, *m2;

	m = (CMolecule*)g_oaMolecules[g_iFixMol];

	c = 0;
	if (g_bRefEnvVirt)
	{
		for (z=0;z<g_oaMolecules.GetSize();z++)
			c += g_pNbAll->m_waScanNeighborCount[z]*((CMolecule*)g_oaMolecules[z])->m_iAtomGes;
		if (g_bSaveRefWithEnv)
			c += m->m_iAtomGes;
	} else
	{
		for (z=0;z<g_oaMolecules.GetSize();z++)
			c += g_pNbAll->m_waScanNeighborCount[z]*(((CMolecule*)g_oaMolecules[z])->m_iAtomGes-((CMolecule*)g_oaMolecules[z])->m_waVirtualAtoms.GetSize());
		if (g_bSaveRefWithEnv)
			c += m->m_iAtomGes - m->m_waVirtualAtoms.GetSize();
	}

	fprintf(a,"  %d\n\n",c);

	if (g_bRefEnvAtomwise)
	{
		for (z0=0;z0<g_oaAtoms.GetSize();z0++)
		{
			if (g_bSaveRefWithEnv)
			{
				for (z3=0;z3<m->m_baAtomIndex.GetSize();z3++)
				{
					if (m->m_baAtomIndex[z3] != z0)
						continue;
					if (!g_bRefEnvVirt)
						if (m->m_baAtomIndex[z3] == g_iVirtAtomType)
							continue;
					for (z4=0;z4<m->m_waAtomCount[z3];z4++)
					{
						ti = ((CxIntArray*)((CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[refmol]])->m_oaAtomOffset[z3])->GetAt(z4);
						fprintf(a,"  %s  %f  %f  %f\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z3]])->m_sName,m_vaCoords[ti][0]/100.0f,m_vaCoords[ti][1]/100.0f,m_vaCoords[ti][2]/100.0f);
					}
				}
			}
			for (z=0;z<g_oaMolecules.GetSize();z++)
			{
				m2 = (CMolecule*)g_oaMolecules[z];
				for (z2=0;z2<g_pNbAll->m_waScanNeighborCount[z];z2++)
				{
					for (z3=0;z3<m2->m_baAtomIndex.GetSize();z3++)
					{
						if (m2->m_baAtomIndex[z3] != z0)
							continue;
						if (!g_bRefEnvVirt)
							if (m2->m_baAtomIndex[z3] == g_iVirtAtomType)
								continue;
						for (z4=0;z4<m2->m_waAtomCount[z3];z4++)
						{
							ti = ((CxIntArray*)((CSingleMolecule*)g_oaSingleMolecules[m2->m_laSingleMolIndex[((CxIntArray*)g_pNbAll->m_oaScanNeighbors[z])->GetAt(z2)]])->m_oaAtomOffset[z3])->GetAt(z4);
							fprintf(a,"  %s  %f  %f  %f\n",((CAtom*)g_oaAtoms[m2->m_baAtomIndex[z3]])->m_sName,m_vaCoords[ti][0]/100.0f,m_vaCoords[ti][1]/100.0f,m_vaCoords[ti][2]/100.0f);
						}
					}
				}
			}
		}
	} else
	{
		if (g_bSaveRefWithEnv)
		{
			for (z3=0;z3<m->m_baAtomIndex.GetSize();z3++)
			{
				if (!g_bRefEnvVirt)
					if (m->m_baAtomIndex[z3] == g_iVirtAtomType)
						continue;
				for (z4=0;z4<m->m_waAtomCount[z3];z4++)
				{
					ti = ((CxIntArray*)((CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[refmol]])->m_oaAtomOffset[z3])->GetAt(z4);
					fprintf(a,"  %s  %f  %f  %f\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z3]])->m_sName,m_vaCoords[ti][0]/100.0,m_vaCoords[ti][1]/100.0,m_vaCoords[ti][2]/100.0);
				}
			}
		}
		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			m2 = (CMolecule*)g_oaMolecules[z];
			for (z2=0;z2<g_pNbAll->m_waScanNeighborCount[z];z2++)
			{
				for (z3=0;z3<m2->m_baAtomIndex.GetSize();z3++)
				{
					if (!g_bRefEnvVirt)
						if (m2->m_baAtomIndex[z3] == g_iVirtAtomType)
							continue;
					for (z4=0;z4<m2->m_waAtomCount[z3];z4++)
					{
						ti = ((CxIntArray*)((CSingleMolecule*)g_oaSingleMolecules[m2->m_laSingleMolIndex[((CxIntArray*)g_pNbAll->m_oaScanNeighbors[z])->GetAt(z2)]])->m_oaAtomOffset[z3])->GetAt(z4);
						fprintf(a,"  %s  %f  %f  %f\n",((CAtom*)g_oaAtoms[m2->m_baAtomIndex[z3]])->m_sName,m_vaCoords[ti][0]/100.0,m_vaCoords[ti][1]/100.0,m_vaCoords[ti][2]/100.0);
					}
				}
			}
		}
	}
	BTOUT; 
}*/

/*float CTimeStep::MolDist(CSingleMolecule *ref, CSingleMolecule *sm2, CNbSearch *nb)
{
	BXIN;
	int z3, z4;
	float d;
	CxVector3 vec;

	d = 99999.0f;
	for (z3=0;z3<nb->m_waRefElements.GetSize();z3++)
		for (z4=0;z4<((CxIntArray*)nb->m_oaNbElements[sm2->m_iMolType])->GetSize();z4++)
		{
			vec = m_vaCoords[((CxIntArray*)ref->m_oaAtomOffset[nb->m_waRefElements[z3]])->GetAt(nb->m_waRefAtoms[z3])] - m_vaCoords[((CxIntArray*)sm2->m_oaAtomOffset[((CxIntArray*)nb->m_oaNbElements[sm2->m_iMolType])->GetAt(z4)])->GetAt(((CxIntArray*)nb->m_oaNbAtoms[sm2->m_iMolType])->GetAt(z4))];
			if (g_bFold)
			{
				while (vec[0] >= g_fBoxX/2) vec[0] -= g_fBoxX;
				while (vec[0] < -g_fBoxX/2) vec[0] += g_fBoxX;
				while (vec[1] >= g_fBoxY/2) vec[1] -= g_fBoxY;
				while (vec[1] < -g_fBoxY/2) vec[1] += g_fBoxY;
				while (vec[2] >= g_fBoxZ/2) vec[2] -= g_fBoxZ;
				while (vec[2] < -g_fBoxZ/2) vec[2] += g_fBoxZ;
			}
			if (vec.GetLength() < d)
				d = vec.GetLength();
		}
	BXOUT;
	if (d < 90000.0f)
		return d;
			else return -1.0f;
}*/

/*void CTimeStep::ScanNeighborhood(int fixmol, int refmol, CNbSearch *nb, CNbSearch *prev)
{
	BTIN;
	float *del, d;
	int *best, b;
	int z, z2, z3, c;
	CMolecule *m, *m2;
	CSingleMolecule *sm, *sm2;
	CxVector3 vec;

	m = (CMolecule*)g_oaMolecules[fixmol];
	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[refmol]];
	del = new float[g_oaSingleMolecules.GetSize()];
	best = new int[g_oaSingleMolecules.GetSize()];

//	printf("Suche Nachbarschaft von Molekuel %d...\n",refmol+1);

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		if (((!nb->m_bDistMode) && (nb->m_waMolCount[z] == 0)) || (nb->m_bDistMode && (nb->m_faMolDist[z] == 0)))
		{
			nb->m_waScanNeighborCount[z] = 0;
			continue;
		}
		m2 = (CMolecule*)g_oaMolecules[z];
//		printf("*** Molekuel %d: %s\n",z+1,m2->Name);
		for (z2=0;z2<((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex.GetSize();z2++)
		{
			del[z2] = 99999.0f;
			if ((z == fixmol) && (z2 == refmol))
				continue;
			if (prev != NULL)
				if (!prev->Contains(z,z2))
					continue;
//			printf("  - Vertreter %d\n",z2+1);
			sm2 = (CSingleMolecule*)g_oaSingleMolecules[m2->m_laSingleMolIndex[z2]];
			del[z2] = MolDist(sm,sm2,nb);
//			printf("  - Finaler Abstand: %f\n",del[z2]);
		}
		if (nb->m_bDistMode)
		{
//			printf("**DistMode**\n");
//			mprintf("Die Nachbarn: ");
			c = 0;
			for (z3=0;z3<((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex.GetSize();z3++)
				if ((del[z3] <= nb->m_faMolDist[z]) && (del[z3] >= nb->m_faMolMinDist[z]))
				{
//					mprintf("%d (%f), ",z3+1,del[z3]);
					best[c] = z3;
					c++;
				}
//			mprintf("\n");
//			printf("  # Die naechsten Nachbarn: ");
			for (z2=0;z2<c;z2++)
			{
//				printf("%d, ",best[z2]+1);
				nb->m_iNeighbourCount++;
				for (z3=0;z3<nb->m_waScanNeighborCount[z];z3++)
					if (((CxIntArray*)nb->m_oaScanNeighbors[z])->GetAt(z3) == best[z2])
					{
						((CxIntArray*)nb->m_oaScanNeighborHits[z])->GetAt(z3)++;
						goto _enddist;
					}
				((CxIntArray*)nb->m_oaScanNeighbors[z])->Add(best[z2]);
				((CxIntArray*)nb->m_oaScanNeighborHits[z])->Add(1);
				nb->m_waScanNeighborCount[z]++;
_enddist:;
			}
//			printf("\n");
		} else
		{
//			printf("**CountMode**\n");
			for (z2=0;z2<nb->m_waMolCount[z];z2++)
			{
				d = 999999.0f;
				b = -1;
				for (z3=0;z3<((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex.GetSize();z3++)
				{
					if (del[z3] < d)
					{
						d = del[z3];
						b = z3;
					}
				}
				best[z2] = b;
				del[b] = 1000000.0f;
			}
//			printf("  # Die naechsten Nachbarn: ");
			for (z2=nb->m_waMolCountMin[z]-1;z2<nb->m_waMolCount[z];z2++)
			{
//				printf("%d, ",best[z2]);
				nb->m_iNeighbourCount++;
				for (z3=0;z3<nb->m_waScanNeighborCount[z];z3++)
					if (((CxIntArray*)nb->m_oaScanNeighbors[z])->GetAt(z3) == best[z2])
					{
						((CxIntArray*)nb->m_oaScanNeighborHits[z])->GetAt(z3)++;
						goto _end;
					}
				((CxIntArray*)nb->m_oaScanNeighbors[z])->Add(best[z2]);
				((CxIntArray*)nb->m_oaScanNeighborHits[z])->Add(1);
				nb->m_waScanNeighborCount[z]++;
_end:;
			}
//			printf("\n");
		}
	}
	delete del;
	delete best;
	BTOUT; 
}*/


/*void CTimeStep::ScanAngle(int fixmol, int refmol, CCondition *co, CNbSearch *prev)
{
	BTIN;
	int z2, z3, z4;
	float tf;
	CMolecule *m, *m2;
	CSingleMolecule *sm, *sm2;
	CxVector3 vec0, vec1, vec2, vec3, vec4, vec5;
	CxIntArray tempwa;
	CNbSearch *nb;

//	mprintf("*** ScanAngle ***\n");
//	mprintf("  FixMol = %d, RefMol = %d\n",fixmol,refmol);
	m = (CMolecule*)g_oaMolecules[fixmol];
//	mprintf("  m->m_laSingleMolIndex[refmol] = %d\n",m->m_laSingleMolIndex[refmol]);

	nb = co->m_pTempNbS;
	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[refmol]];

//	printf("Suche Nachbarschaft von Molekuel %d...\n",refmol+1);

//	mprintf("  SecondMol = %d\n",co->m_iSecondMol);
	m2 = (CMolecule*)g_oaMolecules[co->m_iSecondMol];

//	printf("*** Molekuel %d: %s\n",z+1,m2->Name);
	for (z2=0;z2<m2->m_laSingleMolIndex.GetSize();z2++)
	{
		if ((co->m_iSecondMol == fixmol) && (z2 == refmol))
			continue;
		if (prev != NULL)
			if (!prev->Contains(co->m_iSecondMol,z2))
				continue;
//		printf("  - Vertreter %d\n",z2+1);
		sm2 = (CSingleMolecule*)g_oaSingleMolecules[m2->m_laSingleMolIndex[z2]];

		co->m_pADF->BuildAtomList(sm,sm2,NULL,&tempwa);

		for (z4=0;z4<tempwa.GetSize();z4+=6)
		{
			if (co->m_pADF->m_bOrtho[0])
			{
				vec0 = m_vaCoords[tempwa[z4]];
				vec2 = m_vaCoords[tempwa[z4+1]];
				vec3 = m_vaCoords[tempwa[z4+2]];
				vec1 = CrossP(vec2-vec0,vec3-vec0);
			} else
			{
				vec0 = m_vaCoords[tempwa[z4]];
				vec2 = m_vaCoords[tempwa[z4+1]];
				vec1 = vec2-vec0;
			}
			if (co->m_pADF->m_bOrtho[1])
			{
				vec4 = m_vaCoords[tempwa[z4+3]];
				vec3 = m_vaCoords[tempwa[z4+4]];
				vec5 = m_vaCoords[tempwa[z4+5]];
				vec2 = CrossP(vec3-vec4,vec5-vec4);
			} else
			{
				vec4 = m_vaCoords[tempwa[z4+3]];
				vec3 = m_vaCoords[tempwa[z4+4]];
				vec2 = vec3-vec4;
			}

			tf = Angle_Deg(vec1,vec2);

  if ((tf >= co->m_pADF->m_fMinAngle) && (tf <= co->m_pADF->m_fMaxAngle))
			{
				nb->m_iNeighbourCount++;
				for (z3=0;z3<nb->m_waScanNeighborCount[co->m_iSecondMol];z3++)
					if (((CxIntArray*)nb->m_oaScanNeighbors[co->m_iSecondMol])->GetAt(z3) == z2)
					{
						((CxIntArray*)nb->m_oaScanNeighborHits[co->m_iSecondMol])->GetAt(z3)++;
						goto _endang;
					}
				((CxIntArray*)nb->m_oaScanNeighbors[co->m_iSecondMol])->Add(z2);
				((CxIntArray*)nb->m_oaScanNeighborHits[co->m_iSecondMol])->Add(1);
				nb->m_waScanNeighborCount[co->m_iSecondMol]++;
_endang:;		break;
			}
		}
	}
	BTOUT; 
}*/

/*void CTimeStep::GatherNbDiagram(int refmol, CNbSearch *nb)
{
	BTIN;
	float *del, d;
	int *best, b;
	bool *done;
	int z, z2, z3, z4, c, z0;
	CMolecule *m, *m2;
	CSingleMolecule *sm, *sm2;
	CxVector3 vec;
//	FILE *a;

	m = (CMolecule*)g_oaMolecules[g_iFixMol];
	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[refmol]];
	del = new float[g_oaSingleMolecules.GetSize()];
	best = new int[g_oaSingleMolecules.GetSize()];
	done = new bool[g_oaSingleMolecules.GetSize()];

//	printf("Suche Nachbarschaft von Molekuel %d...\n",refmol+1);

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		if (nb->m_bDistMode && (nb->m_faMolDist[z] == 0))
		{
			nb->m_waScanNeighborCount[z] = 0;
			continue;
		}
		m2 = (CMolecule*)g_oaMolecules[z];
//		printf("*** Molekuel %d: %s\n",z+1,m2->Name);
//		for (z2=0;z2<=((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex.GetSize();z2++)
//			del[z2] = 99999.0f;
		for (z2=0;z2<((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex.GetSize();z2++)
		{
			del[z2] = 99999.0f;
			if ((z == g_iFixMol) && (z2 == refmol))
				continue;
//			printf("  - Vertreter %d\n",z2+1);
			sm2 = (CSingleMolecule*)g_oaSingleMolecules[m2->m_laSingleMolIndex[z2]];
			for (z3=0;z3<nb->m_waRefElements.GetSize();z3++)
				for (z4=0;z4<((CxIntArray*)nb->m_oaNbElements[z])->GetSize();z4++)
				{
					vec = m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[nb->m_waRefElements[z3]])->GetAt(nb->m_waRefAtoms[z3])] - m_vaCoords[((CxIntArray*)sm2->m_oaAtomOffset[((CxIntArray*)nb->m_oaNbElements[z])->GetAt(z4)])->GetAt(((CxIntArray*)nb->m_oaNbAtoms[z])->GetAt(z4))];
					if (g_bFold)
					{
						while (vec[0] >= g_fBoxX/2) vec[0] -= g_fBoxX;
						while (vec[0] < -g_fBoxX/2) vec[0] += g_fBoxX;
						while (vec[1] >= g_fBoxY/2) vec[1] -= g_fBoxY;
						while (vec[1] < -g_fBoxY/2) vec[1] += g_fBoxY;
						while (vec[2] >= g_fBoxZ/2) vec[2] -= g_fBoxZ;
						while (vec[2] < -g_fBoxZ/2) vec[2] += g_fBoxZ;
					}
					d = vec.GetLength();
//					printf("    = m1 %d (%d|%d); m2 %d (%d|%d); Dist = %f\n",z3+1,m->m_waNbElements[z3]+1,m->m_waNbAtoms[z3]+1,z4+1,m2->m_waNbElements[z4]+1,m2->m_waNbAtoms[z4]+1,d);
					if (d < del[z2])
						del[z2] = d;
				}
//			printf("  - Finaler Abstand: %f\n",del[z2]);
		}

		for (z0=0;z0<nb->m_pAF->m_iResolution;z0++)
		{
			c = 0;
			for (z2=0;z2<((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex.GetSize();z2++)
				done[z2] = false;
			while (true)
			{
				d = 999999.0f;
				b = -1;
				for (z3=0;z3<((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex.GetSize();z3++)
				{
					if (done[z3])
						continue;
					if (del[z3] < d)
					{
						d = del[z3];
						b = z3;
					}
				}
				if (d > nb->m_pAF->m_fMaxVal*z0/nb->m_pAF->m_iResolution)
					break;
				best[c] = b;
				done[b] = true;
				c++;
//				if (c >= ((CMolecule*)g_oaMolecules[z])->m_laSingleMolIndex.GetSize())
//					break;
			}
			nb->m_pAF->AddToBin(nb->m_pAF->m_fMaxVal*z0/nb->m_pAF->m_iResolution,(float)c);
//			if (g_iSteps == 1)
//				fprintf(a,"%f;%f\n",nb->m_pAF->m_fMaxVal*z0/nb->m_pAF->m_iResolution,(float)c);
		}
//		if (g_iSteps == 1)
//			fclose(a);
	}
	delete del;
	delete best;
	BTOUT; 
}*/

int CTimeStep::REC_UniteMolecules(CSingleMolecule *sm, int i0, int depth)
{
	int z, z2, n;
	CMolAtom *m;

	n = 0;
	m = (CMolAtom*)sm->m_oaMolAtoms[i0];

/*	if (g_bVerbose)
	{
		mprintf("# ");
		for (z2=0;z2<depth;z2++)
			mprintf(". ");
		mprintf(">>> REC_UniteMolecules MolAtom=%d, Offset=%d.\n",i0,m->m_iOffset);
	}*/

	g_pUniteTemp[m->m_iOffset] = true;

	for (z=0;z<m->m_oaBonds.GetSize();z++)
	{
		if (!g_pUniteTemp[((CMolAtom*)m->m_oaBonds[z])->m_iOffset])
		{
			if (MirrorBond(m->m_iOffset,((CMolAtom*)m->m_oaBonds[z])->m_iOffset))
			{
				if (g_bVerbose)
				{
					mprintf("# ");
					for (z2=0;z2<depth;z2++)
						mprintf(". ");
					mprintf("    Bond %s(%d) <--> %s(%d) unwrapped.\n",((CAtom*)g_oaAtoms[g_baAtomIndex[m->m_iOffset]])->m_sName,m->m_iOffset+1,((CAtom*)g_oaAtoms[g_baAtomIndex[((CMolAtom*)m->m_oaBonds[z])->m_iOffset]])->m_sName,((CMolAtom*)m->m_oaBonds[z])->m_iOffset+1);
				}
				n++;
			}
			n += REC_UniteMolecules(sm,((CMolAtom*)m->m_oaBonds[z])->m_iMolAtomNumber,depth+1);
		}
	}
/*	if (g_bVerbose)
	{
		mprintf("# ");
		for (z2=0;z2<depth;z2++)
			mprintf(". ");
		mprintf("<<< REC_UniteMolecules MolAtom=%d, Offset=%d.\n",i0,m->m_iOffset);
	}*/
	return n;
}

void CTimeStep::UniteMolecules(bool verbose)
{
	BTIN;
	int z, z2, n;
	CMolecule *m;
	CSingleMolecule *sm;

	for (z=0;z<g_iGesAtomCount;z++)
		g_pUniteTemp[z] = false;
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[z];
		if (m->m_bPseudo)
			continue;
		for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
		{
			sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];

			if (g_bVerbose)
				mprintf("  # UniteMolecules molecule %s (%d)\n",m->m_sName,z2+1);

			n = REC_UniteMolecules(sm,0,0);

			if ((n != 0) && verbose)
				mprintf("      - Molecule %s (%d) united, %d bonds unwrapped.\n",m->m_sName,z2+1,n);
	
		}
	}
	BTOUT; 
}


bool CTimeStep::ReadTimestep(FILE *a, bool needinfo)
{
	BTIN;
	switch(g_iTrajFormat)
	{
		case 0:
			if (!ReadXYZ(a,needinfo,&m_vaCoords))
				return false;
			break;

		case 1:
			if (!ReadPDB(a,needinfo,&m_vaCoords))
				return false;
			break;

		case 2:
			if (!ReadMol2(a,needinfo,&m_vaCoords))
				return false;
			break;

		case 3:
			if (!ReadLAMMPS(a,needinfo,&m_vaCoords))
				return false;
			break;

		case 4:
			if (!ReadDLPOLY(a,needinfo,&m_vaCoords))
				return false;
			break;
	}
	if (g_bDoubleBox)
		DoubleBox();
	BTOUT; 
	return true;
}


bool CTimeStep::SkipTimestep(FILE *a)
{
	BTIN;
	switch(g_iTrajFormat)
	{
		case 0:
			if (!SkipXYZ(a))
				return false;
			break;

		case 1:
			if (!SkipPDB(a))
				return false;
			break;

		case 2:
			if (!SkipMol2(a))
				return false;
			break;

		case 3:
			if (!SkipLAMMPS(a))
				return false;
			break;

		case 4:
			if (!SkipDLPOLY(a))
				return false;
			break;
	}
	BTOUT; 
	return true;
}


bool CTimeStep::ReadTimestepVel(FILE *a)
{
	BTIN;
	char buf[256], *p, *q;
	int z, tc;

	buf[0] = 0;
	fgets(buf,256,a);
	if (feof(a))
	{
		BTOUT; 
		return false;
	}
	if (strlen(buf) > 0)
		buf[strlen(buf)-1] = 0;
	tc = atoi(buf);
	if (tc == 0)
	{
		BTOUT; 
		return false;
	}
	m_vaVelocities.SetSize(tc);
	buf[0] = 0;
	fgets(buf,256,a); // Zeitschritt - egal hier
	if (strlen(buf) > 0)
		buf[strlen(buf)-1] = 0;
	for (z=0;z<tc;z++) // den ersten Zeitschritt einlesen
	{
		buf[0] = 0;
		fgets(buf,256,a);
		if (feof(a))
		{
			BTOUT; 
			return false;
		}
		if (strlen(buf) > 0)
			buf[strlen(buf)-1] = 0;
		q = buf;
		while (*q == ' ')
			q++;
		p = strchr(q,' ');
		if (p == NULL)
		{
			eprintf("CTimeStep::ReadTimestepVel(): Error 1. %d, \"%s\"\n",z+1,buf);
			continue;
		}
		while (isdigit(*(p-1)) && (p > buf))
			p--;
		if (p == buf)
		{
			eprintf("CTimeStep::ReadTimestepVel(): No Atom laben found. %d, \"%s\"\n",z+1,buf);
			continue;
		}
		*p = 0;
		p++;
		q = strchr(p,' ');
		if (q == NULL)
		{
			eprintf("CTimeStep::ReadTimestepVel(): Error 2. %d, \"%s\"\n",z+1,p);
			continue;
		}
		while (*q == ' ')
			q++;
		p = strchr(q,' ');
		if (p == NULL)
		{
			eprintf("CTimeStep::ReadTimestepVel(): Error 3. %d, \"%s\"\n",z+1,q);
			continue;
		}
		*p = 0;
		m_vaVelocities[z][0] = (float)atof(q);
		q = p+1;
		while (*q == ' ')
			q++;
		p = strchr(q,' ');
		if (p == NULL)
		{
			eprintf("CTimeStep::ReadTimestepVel(): Error 4. %d, \"%s\"\n",z+1,q);
			continue;
		}
		*p = 0;
		m_vaVelocities[z][1] = (float)atof(q);
		q = p+1;
		while (*q == ' ')
			q++;
		p = strchr(q,' ');
		if (p != NULL)
			*p = 0;
		m_vaVelocities[z][2] = (float)atof(q);
	}
	if (g_bDoubleBox)
		DoubleBoxVelocity();
	BTOUT; 
	return true;
}

bool CTimeStep::ReadTimestepForce(FILE *a)
{
	BTIN;
	char buf[256], *p, *q;
	int z, tc;

	buf[0] = 0;
	fgets(buf,256,a);
	if (feof(a))
	{
		BTOUT; 
		return false;
	}
	if (strlen(buf) > 0)
		buf[strlen(buf)-1] = 0;
	tc = atoi(buf);
	if (tc == 0)
	{
		BTOUT; 
		return false;
	}
	m_vaForces.SetSize(tc);
	buf[0] = 0;
	fgets(buf,256,a); // Zeitschritt - egal hier
	if (strlen(buf) > 0)
		buf[strlen(buf)-1] = 0;
	for (z=0;z<tc;z++) // den ersten Zeitschritt einlesen
	{
		buf[0] = 0;
		fgets(buf,256,a);
		if (feof(a))
		{
			BTOUT; 
			return false;
		}
		if (strlen(buf) > 0)
			buf[strlen(buf)-1] = 0;
		q = buf;
		while (*q == ' ')
			q++;
		p = strchr(q,' ');
		if (p == NULL)
		{
			eprintf("CTimeStep::ReadTimestepForce(): Error 1. %d, \"%s\"\n",z+1,buf);
			continue;
		}
		while (isdigit(*(p-1)) && (p > buf))
			p--;
		if (p == buf)
		{
			eprintf("CTimeStep::ReadTimestepForce(): No Atom laben found. %d, \"%s\"\n",z+1,buf);
			continue;
		}
		*p = 0;
		p++;
		q = strchr(p,' ');
		if (q == NULL)
		{
			eprintf("CTimeStep::ReadTimestepForce(): Error 2. %d, \"%s\"\n",z+1,p);
			continue;
		}
		while (*q == ' ')
			q++;
		p = strchr(q,' ');
		if (p == NULL)
		{
			eprintf("CTimeStep::ReadTimestepForce(): Error 3. %d, \"%s\"\n",z+1,q);
			continue;
		}
		*p = 0;
		m_vaForces[z][0] = (float)atof(q);
		q = p+1;
		while (*q == ' ')
			q++;
		p = strchr(q,' ');
		if (p == NULL)
		{
			eprintf("CTimeStep::ReadTimestepForce(): Error 4. %d, \"%s\"\n",z+1,q);
			continue;
		}
		*p = 0;
		m_vaForces[z][1] = (float)atof(q);
		q = p+1;
		while (*q == ' ')
			q++;
		p = strchr(q,' ');
		if (p != NULL)
			*p = 0;
		m_vaForces[z][2] = (float)atof(q);
	}
	if (g_bDoubleBox)
		DoubleBoxForce();
	BTOUT; 
	return true;
}

bool CTimeStep::SkipXYZ(FILE *a)
{
	BTIN;
	char buf[256];
	int z, tc;

//	mprintf("*** Skip Anfang ***\n");
	buf[0] = 0;
	fgets_bin(buf,256,a);
	if (feof(a))
	{
		BTOUT; 
		return false;
	}
	if (strlen(buf)==0)
	{
		BTOUT; 
		return false;
	}
	buf[strlen(buf)-1] = 0;
	tc = atoi(buf);
//	mprintf("SkipA: \"%s\".\n",buf);
	if (tc == 0)
	{
		BTOUT; 
		return false;
	}
//	buf[0] = 0;
	fgets_bin(buf,256,a); // Zeitschritt - egal hier
//	mprintf("SkipB: \"%s\".\n",buf);
	for (z=0;z<tc;z++) // den ersten Zeitschritt einlesen
	{
//		buf[0] = 0;
		fgets_bin(buf,256,a);
//		mprintf("SkipC%d: \"%s\".\n",z,buf);
		if (feof(a))
		{
			BTOUT; 
			return false;
		}
	}
//	mprintf("*** Skip Ende ***\n");
	BTOUT; 
	return true;
}

void CTimeStep::CopyFrom(CTimeStep *t)
{
	BTIN;
	int z;
	char *p;
	CxVector3 v;

	m_iGesAtomCount = t->m_iGesAtomCount;
	m_vaCoords.CopyFrom(&t->m_vaCoords);
	if (g_bKeepUnfoldedCoords)
		m_vaCoords_Unfolded.CopyFrom(&t->m_vaCoords_Unfolded);
	m_vaForces.CopyFrom(&t->m_vaForces);
	m_vaVelocities.CopyFrom(&t->m_vaVelocities);
	if (t->m_paLabels.GetSize() != 0)
	{
		for (z=0;z<m_paLabels.GetSize();z++)
			delete[] (char*)m_paLabels[z];
		m_paLabels.RemoveAll();
		for (z=0;z<t->m_paLabels.GetSize();z++)
		{
			try { p = new char[strlen((char*)t->m_paLabels[z])+1]; } catch(...) { p = NULL; }
			if (p == NULL) NewException((double)(strlen((char*)t->m_paLabels[z])+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			strcpy(p,(char*)t->m_paLabels[z]);
			m_paLabels.Add(p);
		}
	}
	if (t->m_pComment != NULL)
	{
		if (m_pComment == NULL)
		{
			try { m_pComment = new char[256]; } catch(...) { m_pComment = NULL; }
			if (m_pComment == NULL) NewException((double)256*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		}
		strcpy(m_pComment,t->m_pComment);
	}
	BTOUT; 
}
	  
long CTimeStep::ExtractNumber(int i)
{
	BXIN;
	char *p, *q, buf[20];
	int z;
	long l;

	if (m_pComment == NULL)
		return -1;
	p = m_pComment;
	for (z=0;z<i;z++)
	{
		while ((!isdigit(*p)) && (*p != 0))
			p++;
		if (*p == 0)
		{
			BXOUT;
			return -1;
		}
		q = p+1;
		while ((isdigit(*q)) && (*q != 0))
			q++;
		if (*q == 0)
		{
			BXOUT;
			return -1;
		}
		p = q+1;
	}
	while ((!isdigit(*p)) && (*p != 0))
		p++;
	if (*p == 0)
	{
		BXOUT;
		return -1;
	}
	q = p;
	while ((isdigit(*q)) && (*q != 0))
		q++;
	if (q == p)
	{
		BXOUT;
		return -1;
	}
	memcpy(buf,p,q-p);
	buf[q-p] = 0;
	l = atoi(buf);
	BXOUT;
	return l;
}


int CTimeStep::GetCommentNumberCount()
{
	BXIN;
	char *p, *q;
	int z;

	if (m_pComment == NULL)
		return 0;
	p = m_pComment;
	z = 0;
	while (true)
	{
		while ((!isdigit(*p)) && (*p != 0))
			p++;
		if (*p == 0)
		{
//			mprintf("GetCommentNumberCount 1: %d (p=\"%s\", q=\"%s\")\n",z,p,q);
			BXOUT;
			return z;
		}
		q = p;
		while ((isdigit(*q)) && (*q != 0))
			q++;
		if (q == p)
		{
//			mprintf("GetCommentNumberCount 2: %d (p=\"%s\", q=\"%s\")\n",z,p,q);
			BXOUT;
			return z;
		}
		z++;
		if (*q == 0)
		{
//			mprintf("GetCommentNumberCount 3: %d (p=\"%s\", q=\"%s\")\n",z,p,q);
			BXOUT;
			return z;
		}
		p = q+1;
	}
	return 0; // Never happens
}


void CTimeStep::ScanWannier(bool verbose)
{
	BTIN;
	int z, z2, z3;
	float td, d, dx, dy, dz;
	CMolecule *m;

	for (z=0;z<g_oaSingleMolecules.GetSize();z++)
		((CSingleMolecule*)g_oaSingleMolecules[z])->m_laWannier.RemoveAll_KeepSize();

	for (z=0;z<g_iGesAtomCount;z++)
	{
		if (g_baAtomIndex[z] != g_iWannierAtomType)
			continue;
		d = 9999.0f;
		for (z2=0;z2<g_iGesAtomCount;z2++)
		{
			if (g_baAtomIndex[z2] == g_iWannierAtomType)
				continue;
			dx = m_vaCoords[z][0]-m_vaCoords[z2][0];
			dy = m_vaCoords[z][1]-m_vaCoords[z2][1];
			dz = m_vaCoords[z][2]-m_vaCoords[z2][2];
			if (g_bPeriodicX)
			{
				while (dx > g_fBoxX/2.0f) dx -= g_fBoxX;
				while (dx < -g_fBoxX/2.0f) dx += g_fBoxX;
			}
			if (g_bPeriodicY)
			{
				while (dy > g_fBoxY/2.0f) dy -= g_fBoxY;
				while (dy < -g_fBoxY/2.0f) dy += g_fBoxY;
			}
			if (g_bPeriodicZ)
			{
				while (dz > g_fBoxZ/2.0f) dz -= g_fBoxZ;
				while (dz < -g_fBoxZ/2.0f) dz += g_fBoxZ;
			}
			td = (float)sqrt(dx*dx+dy*dy+dz*dz);
			if (td < d)
			{
				d = td;
				z3 = z2;
			}
		}
		if (g_bPeriodicX)
		{
			while (m_vaCoords[z][0]-m_vaCoords[z3][0] > g_fBoxX/2.0f) m_vaCoords[z][0] -= g_fBoxX;
			while (m_vaCoords[z][0]-m_vaCoords[z3][0] < -g_fBoxX/2.0f) m_vaCoords[z][0] += g_fBoxX;
		}
		if (g_bPeriodicY)
		{
			while (m_vaCoords[z][1]-m_vaCoords[z3][1] > g_fBoxY/2.0f) m_vaCoords[z][1] -= g_fBoxY;
			while (m_vaCoords[z][1]-m_vaCoords[z3][1] < -g_fBoxY/2.0f) m_vaCoords[z][1] += g_fBoxY;
		}
		if (g_bPeriodicZ)
		{
			while (m_vaCoords[z][2]-m_vaCoords[z3][2] > g_fBoxZ/2.0f) m_vaCoords[z][2] -= g_fBoxZ;
			while (m_vaCoords[z][2]-m_vaCoords[z3][2] < -g_fBoxZ/2.0f) m_vaCoords[z][2] += g_fBoxZ;
		}
		z2 = g_laAtomSMIndex[z3];
		if (d > 200.0)
		{
			eprintf("Wannier center %d does not belong to an atom (closest atom is %s %d %s%d, %.0f pm).\n",z+1,((CMolecule*)g_oaMolecules[((CSingleMolecule*)g_oaSingleMolecules[z2])->m_iMolType])->m_sName,((CSingleMolecule*)g_oaSingleMolecules[z2])->m_iMolSMIndex+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[z3]])->m_sName,g_waAtomMolNumber[z3]+1,d);
		} else if (verbose)
			mprintf("  - Wannier center %d belongs to %s %d %s%d (%.0f pm).\n",z+1,((CMolecule*)g_oaMolecules[((CSingleMolecule*)g_oaSingleMolecules[z2])->m_iMolType])->m_sName,((CSingleMolecule*)g_oaSingleMolecules[z2])->m_iMolSMIndex+1,((CAtom*)g_oaAtoms[g_waAtomRealElement[z3]])->m_sName,g_waAtomMolNumber[z3]+1,d);
/*		if (z2 == -1)
		{
			eprintf("Wannier center %d: Atom %d does not belong to any molecule.\n",z+1,z3+1);
			continue;
		}*/
//		mprintf("Wannier Center %d hat Abstand %.2f zu Atom %d in SM %d.\n",z+1,d,z3+1,z2+1);
		((CSingleMolecule*)g_oaSingleMolecules[z2])->m_laWannier.Add(z);
	}
	if (verbose)
		mprintf("\n");
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[z];
		if (m->m_iWannierCount == 0)
		{
			m->m_iWannierCount = ((CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[0]])->m_laWannier.GetSize();
			td = 0;
			for (z2=0;z2<m->m_baAtomIndex.GetSize();z2++)
				td += ((CAtom*)g_oaAtoms[m->m_baAtomIndex[z2]])->m_fCharge * m->m_waAtomCount[z2];
			m->m_fCharge = td-m->m_iWannierCount*g_fWannierCharge;
			mprintf("  - Molecule %s contains %d wannier centers. Total charge is %.2f - %.2f = %.2f\n",m->m_sName,m->m_iWannierCount,td,m->m_iWannierCount*g_fWannierCharge,m->m_fCharge);
		}
		for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
		{
			if (m->m_iWannierCount != ((CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]])->m_laWannier.GetSize())
				eprintf("Error: Molecule %s, representant %d, contains %d instead of %d wannier centers.\n",m->m_sName,z2+1,((CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]])->m_laWannier.GetSize(),m->m_iWannierCount);
		}
	}
	BTOUT;
}


void CTimeStep::CalcDipoles()
{
	BTIN;
	int z, z2, z3, z4;
	CMolecule *m;
	CSingleMolecule *sm;
	CAtom *a;
	CxVector3 dc;

	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[z];
		for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
		{
			if (g_bVerbose)
				mprintf("\nCalcDipoles %s (%d):\n",m->m_sName,z2+1);

			sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];

			sm->m_vDipole = CxVector3(0,0,0);

			if ((!g_bWannier) && (!m->m_bChargesAssigned))
				continue;

			dc = m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[m->m_iDipoleCenterType])->GetAt(m->m_iDipoleCenterIndex)];

			if (g_bVerbose)
				mprintf("  Ref. point is ( %f | %f | %f )\n",dc[0],dc[1],dc[2]);

			for (z3=0;z3<m->m_baAtomIndex.GetSize();z3++)
			{
				if (m->m_baAtomIndex[z3] == g_iVirtAtomType)
					continue;

				if (g_bWannier)
					if (m->m_baAtomIndex[z3] == g_iWannierAtomType)
						continue;

				a = (CAtom*)g_oaAtoms[m->m_baAtomIndex[z3]];
				for (z4=0;z4<m->m_waAtomCount[z3];z4++)
				{
//					mprintf("  %s%d %.2f x ",a->m_sName,z4+1,a->m_fCharge);
//					(m_vaCoords[sm->m_iAtomOffset[z3][z4]]-m_vaCoords[sm->m_iAtomOffset[m->Elements-1][0]]).Dump();
//					mprintf("\n");
//					mprintf("  %s%d %.2f (%d)\n",a->m_sName,z4+1,a->m_fCharge,((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4));

					if (g_bWannier)
					{
						sm->m_vDipole += a->m_fCharge * (m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)] - dc);

						if (g_bVerbose)
							mprintf("  %.2f: ( %f | %f | %f )\n",a->m_fCharge,m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][0],m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][1],m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][2]);
					} else
					{
						sm->m_vDipole += ((CxFloatArray*)m->m_oaCharges[z3])->GetAt(z4) * (m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)] - dc);

						if (g_bVerbose)
							mprintf("  %.2f: ( %f | %f | %f )\n",((CxFloatArray*)m->m_oaCharges[z3])->GetAt(z4),m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][0],m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][1],m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][2]);
					}
				}
			}
			for (z3=0;z3<sm->m_laWannier.GetSize();z3++)
			{
//				mprintf("  - %.2f x ",g_fWannierCharge);
//				(m_vaCoords[sm->m_waWannier[z3]] - m_vaCoords[sm->m_iAtomOffset[m->Elements-1][0]]).Dump();
//				mprintf("\n");
				sm->m_vDipole -= g_fWannierCharge * (m_vaCoords[sm->m_laWannier[z3]] - dc);

				if (g_bVerbose)
					mprintf("  %.2f: ( %f | %f | %f )\n",-g_fWannierCharge,m_vaCoords[sm->m_laWannier[z3]][0],m_vaCoords[sm->m_laWannier[z3]][1],m_vaCoords[sm->m_laWannier[z3]][2]);
			}
//			mprintf("  = ");
//			sm->m_vDipole.Dump();
//			mprintf("\n");
			sm->m_vDipole *= 0.048008f; // Conversion e*Angstrom --> Debye

			if (g_bVerbose)
				mprintf("  Result: %f.\n",sm->m_vDipole.GetLength());

//			mprintf("Molecule %s - Dipole %.3f\n",m->Name,sm->m_vDipole.GetLength());
		}
	}
	BTOUT;
}


void CTimeStep::DoubleBox()
{
	int px, py, pz, z;
	char *p;

	if (m_vaCoords.GetSize() < g_iGesAtomCount)
		m_vaCoords.SetSize(g_iGesAtomCount);

	if (m_paLabels.GetSize() != 0)
		m_paLabels.SetSize(g_iGesAtomCount);

	for (pz=0;pz<g_iDoubleBoxZ;pz++)
	{
		for (py=0;py<g_iDoubleBoxY;py++)
		{
			for (px=0;px<g_iDoubleBoxX;px++)
			{
				if ((px == 0) && (py == 0) && (pz == 0))
					continue;
				for (z=0;z<g_iGesAtomCount/g_iDoubleBoxFactor;z++)
				{
					m_vaCoords[(pz*g_iDoubleBoxX*g_iDoubleBoxY+py*g_iDoubleBoxX+px)*g_iGesAtomCount/g_iDoubleBoxFactor+z] = m_vaCoords[z] + CxVector3(px*g_fBoxX/g_iDoubleBoxX,py*g_fBoxY/g_iDoubleBoxY,pz*g_fBoxZ/g_iDoubleBoxZ);
					if (m_paLabels.GetSize() != 0)
					{
						try { p = new char[strlen((char*)m_paLabels[z])+1]; } catch(...) { p = NULL; }
						if (p == NULL) NewException((double)(strlen((char*)m_paLabels[z])+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
						
						strcpy(p,(char*)m_paLabels[z]);
						m_paLabels[(pz*g_iDoubleBoxX*g_iDoubleBoxY+py*g_iDoubleBoxX+px)*g_iGesAtomCount/g_iDoubleBoxFactor+z] = p;
					}
				}
			}
		}
	}
	m_iGesAtomCount = g_iGesAtomCount;
}


void CTimeStep::DoubleBoxVelocity()
{
	int px, py, pz, z;

	if (m_vaVelocities.GetSize() < g_iGesAtomCount)
		m_vaVelocities.SetSize(g_iGesAtomCount);

	for (pz=0;pz<g_iDoubleBoxZ;pz++)
	{
		for (py=0;py<g_iDoubleBoxY;py++)
		{
			for (px=0;px<g_iDoubleBoxX;px++)
			{
				if ((px == 0) && (py == 0) && (pz == 0))
					continue;
				for (z=0;z<g_iGesAtomCount/g_iDoubleBoxFactor;z++)
					m_vaVelocities[(pz*g_iDoubleBoxX*g_iDoubleBoxY+py*g_iDoubleBoxX+px)*g_iGesAtomCount/g_iDoubleBoxFactor+z] = m_vaVelocities[z];
			}
		}
	}
}


void CTimeStep::DoubleBoxForce()
{
	int px, py, pz, z;

	if (m_vaForces.GetSize() < g_iGesAtomCount)
		m_vaForces.SetSize(g_iGesAtomCount);

	for (pz=0;pz<g_iDoubleBoxZ;pz++)
	{
		for (py=0;py<g_iDoubleBoxY;py++)
		{
			for (px=0;px<g_iDoubleBoxX;px++)
			{
				if ((px == 0) && (py == 0) && (pz == 0))
					continue;
				for (z=0;z<g_iGesAtomCount/g_iDoubleBoxFactor;z++)
					m_vaForces[(pz*g_iDoubleBoxX*g_iDoubleBoxY+py*g_iDoubleBoxX+px)*g_iGesAtomCount/g_iDoubleBoxFactor+z] = m_vaForces[z];
			}
		}
	}
}


bool CTimeStep::ReadXYZ(FILE *a, bool needinfo, CxVec3Array *v)
{
	BTIN;
	char buf[256], obuf[256],  *p, *q, *r;
	int z, i, j;
	const char *separators = " ,;\"'\t";

//	mprintf("*** Read Anfang.\n");
	j = 0;
//	m_iSizeBytes = 0;
_readagain:
	buf[0] = 0;
	fgets_bin(buf,256,a);
//	m_iSizeBytes += strlen(buf);
	if (strlen(buf) > 0)
		buf[strlen(buf)-1] = 0;
//	mprintf("Read1: \"%s\".\n",buf);
	goto _firsttry;
_again:
	eprintf("\nTrajectory file seems to be damaged. Searching for next time step...");
_firsttry:
	if (feof(a))
	{
//		mprintf("CTimeStep::ReadTimestep(): Unexpected End of File (1).\n"); 
		BTOUT; 
		return false;
	}
	if (strchr(buf,'.') != NULL)
	{
		if (j < 10)
			eprintf("x",j,buf);
//			mprintf("x: %d - \"%s\"\n",j,buf);
		j++;
		goto _readagain;
	}
	m_iGesAtomCount = atoi(buf);
	if (m_iGesAtomCount == 0)
	{
		eprintf("\nCTimeStep::ReadXYZ(): Error - atom count = 0. \"%s\"",buf);
		goto _readagain;
	}
	if (m_pComment == NULL)
	{
		try { m_pComment = new char[128]; } catch(...) { m_pComment = NULL; }
		if (m_pComment == NULL) NewException((double)128*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	}
	if (needinfo)
	{
		for (z=0;z<m_paLabels.GetSize();z++)
			delete[] (char*)m_paLabels[z];
		m_paLabels.RemoveAll();
		if (g_bDoubleBox)
			m_paLabels.SetSize(m_iGesAtomCount*g_iDoubleBoxFactor);
				else m_paLabels.SetSize(m_iGesAtomCount);
	}
	if (v->GetSize() < (long)m_iGesAtomCount)
		v->SetSize(m_iGesAtomCount);
	m_pComment[0] = 0;
	fgets_bin(m_pComment,128,a); // Zeitschritt - egal hier
	if (strlen(m_pComment) > 0)
		m_pComment[strlen(m_pComment)-1] = 0;
//	mprintf("Comment: \"%s\".\n",m_pComment);
//	m_iSizeBytes += strlen(m_pComment);
	for (z=0;z<(long)m_iGesAtomCount;z++) // den ersten Zeitschritt einlesen
	{
		buf[0] = 0;
		fgets_bin(buf,256,a);
		if (feof(a))
		{
			eprintf("\nCTimeStep::ReadXYZ(): Unexpected end of file (2). \"%s\"\n",buf);
			BTOUT; 
			return false;
		}
//		m_iSizeBytes += strlen(buf);
		buf[strlen(buf)-1] = 0;
//		mprintf("  %d: \"%s\".\n",z,buf);
		strcpy(obuf,buf);
		i = 0;

		p = &buf[0];
		while (strchr(separators,*p) != NULL)
			p++;
		q = p+1;
		while ((strchr(separators,*q) == NULL) && (*q != 0))
			q++;
		if (*q == 0)
		{
			eprintf("\nCTimeStep::ReadXYZ(): %d: Incomplete line (1): \"%s\"\n",z+1,obuf);
			BTOUT; 
			return false;
		}
		*q = 0;

		if (needinfo)
		{
			if (strlen(p) > 7)
			{
				eprintf("\nCTimeStep::ReadXYZ(): \"%s\" - maximum length for atom labels is 7 chars; truncating.\n",p);
				p[7] = 0;
			}

			try { r = new char[strlen(p)+1]; } catch(...) { r = NULL; }
			if (r == NULL) NewException((double)(strlen(p)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			strcpy(r,p);
			m_paLabels[z] = r;
		}

		q++;
		while (strchr(separators,*q) != NULL)
			q++;
		p = q;
		while ((strchr(separators,*p) == NULL) && (*p != 0))
			p++;
		if (*p == 0)
		{
			eprintf("\nCTimeStep::ReadXYZ(): %d: Incomplete line (2): \"%s\"",z+1,obuf);
			goto _again;
		}
		*p = 0;
		(*v)[z][0] = (float)atof(q) * 100.0f;
		q = p+1;
		while (strchr(separators,*q) != NULL)
			q++;
		p = q;
		while ((strchr(separators,*p) == NULL) && (*p != 0))
			p++;
		if (*p == 0)
		{
			eprintf("\nCTimeStep::ReadXYZ(): %d: Incomplete line (3) \"%s\"",z+1,obuf);
			goto _again;
		}
		*p = 0;
		(*v)[z][1] = (float)atof(q) * 100.0f;
		q = p+1;
		while (strchr(separators,*q) != NULL)
			q++;
		p = q;
		while ((strchr(separators,*p) == NULL) && (*p != 0))
			p++;
		if (*p != 0)
			*p = 0;
		(*v)[z][2] = (float)atof(q) * 100.0f;


/*		q = strchr(buf,'.');
		if (q != NULL)
		{
			i++;
			q = strchr(q+1,'.');
		}
		if (q != NULL)
		{
			i++;
			q = strchr(q+1,'.');
		}
		if (q==NULL)
		{
			eprintf("\nCTimeStep::ReadXYZ(): Error - only %d/3 dots. %d, \"%s\"",i,z+1,buf);
			goto _again;
		}
		q = buf;
		while (*q == ' ')
			q++;
		p = strchr(q,' ');
		if (p == NULL)
		{
			eprintf("\nCTimeStep::ReadXYZ(): Error 1. %d, \"%s\"",z+1,buf);
			goto _again;
		}
		while (isdigit(*(p-1)) && (p > buf))
			p--;
		if (p == buf)
		{
			eprintf("\nCTimeStep::ReadXYZ(): No Atom label found. %d, \"%s\"",z+1,buf);
			goto _again;
		}
		*p = 0;
		if (needinfo)
		{
			r = new char[strlen(q)+1];
			strcpy(r,q);
			m_paLabels[z] = r;
		}
		q = p+1;
		while (*q == ' ')
			q++;
		p = strchr(q,' ');
		if (p == NULL)
		{
			eprintf("\nCTimeStep::ReadXYZ(): Error 3. %d, \"%s\"",z+1,q);
			goto _again;
		}
		*p = 0;
		(*v)[z][0] = (float)atof(q) * 100.0f;
		q = p+1;
		while (*q == ' ')
			q++;
		p = strchr(q,' ');
		if (p == NULL)
		{
			eprintf("\nCTimeStep::ReadXYZ(): Error 4. %d, \"%s\"",z+1,q);
			goto _again;
		}
		*p = 0;
		(*v)[z][1] = (float)atof(q) * 100.0f;
		q = p+1;
		while (*q == ' ')
			q++;
		p = strchr(q,' ');
		if (p != NULL)
			*p = 0;
		(*v)[z][2] = (float)atof(q) * 100.0f;*/


	}
//	mprintf("*** Read Ende.\n");
	BTOUT; 
	return true;
}


bool CTimeStep::ReadPDB(FILE *a, bool needinfo, CxVec3Array *v)
{
	int i;
	char buf[256], obuf[256], *p, *q, *r;
	bool b;
	float x, y, z;

	b = false;
	m_vaCoords.RemoveAll_KeepSize();
	for (i=0;i<m_paLabels.GetSize();i++)
		delete[] (char*)m_paLabels[i];
	m_paLabels.RemoveAll();
	while (true)
	{
		fgets_bin(buf,256,a);
		if (feof(a))
			return false;
		if (strlen(buf) == 0)
			continue;
		buf[strlen(buf)-1] = 0;
		strcpy(obuf,buf);
		if (g_bNPT)
		{
			if (strstr(buf,"CRYST1") != 0) // Boxlaenge
			{
	//			mprintf(GREY,"\"%s\".\n",buf);
				p = &buf[6];
				while (*p == ' ')
					p++;
				q = p;
				while ((*q != ' ') && (*q != 0))
					q++;
				if (*q == 0)
				{
					eprintf("Error 5 reading PDB line: \"%s\".\n",obuf);
					return false;
				}
				*q = 0;
				g_fBoxX = (float)(atof(p)*100.0);
				p = q+1;

				while (*p == ' ')
					p++;
				q = p;
				while ((*q != ' ') && (*q != 0))
					q++;
				if (*q == 0)
				{
					eprintf("Error 6 reading PDB line: \"%s\".\n",obuf);
					return false;
				}
				*q = 0;
				g_fBoxY = (float)(atof(p)*100.0);
				p = q+1;

				while (*p == ' ')
					p++;
				q = p;
				while ((*q != ' ') && (*q != 0))
					q++;
				if (*q != 0)
					*q = 0;
				g_fBoxZ = (float)(atof(p)*100.0);
//				mprintf(GREY,"--> %f %f %f\n",g_fBoxX,g_fBoxY,g_fBoxZ);
			}
		}
		if ((strstr(buf,"ATOM")==0) && (strstr(buf,"HETATM")==0))
		{
			if (b)
				break;
					else continue;
		} else
			b = true;
		p = &buf[5];
		while (!(isalpha(*p) || (*p == '_')) && (*p != 0))
			p++;
		if (*p == 0)
		{
			eprintf("Error 4 reading PDB line: \"%s\".\n",obuf);
			return false;
		}
		q = p;
		while (isalpha(*q) || (*q == '_'))
			q++;
		if (needinfo)
		{
			*q = 0;
			if (strlen(p) > 7)
			{
				eprintf("\nCTimeStep::ReadPDB(): \"%s\" - maximum length for atom labels is 7 chars; truncating.\n",p);
				p[7] = 0;
			}

			try { r = new char[strlen(p)+1]; } catch(...) { r = NULL; }
			if (r == NULL) NewException((double)(strlen(p)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			strcpy(r,p);
			m_paLabels.Add(r);
		}
		p = q+1;

		while ((*p != '.') && (*p != 0))
			p++;

		while (*p != ' ')
			p--;

		p++;

		q = strchr(p,' ');
		if (q == NULL)
		{
			eprintf("\nCTimeStep::ReadPDB(): Error 1. %d. \"%s\".",z+1,obuf);
			return false;
		}
		*q = 0;
		x = (float)atof(p) * 100.0f;
		p = q+1;
		while (*p == ' ')
			p++;

		q = strchr(p,' ');
		if (q == NULL)
		{
			eprintf("\nCTimeStep::ReadPDB(): Error 2. %d. \"%s\".",z+1,obuf);
			return false;
		}
		*q = 0;
		y = (float)atof(p) * 100.0f;
		p = q+1;
		while (*p == ' ')
			p++;

		q = strchr(p,' ');
		if (q == NULL)
		{
			eprintf("\nCTimeStep::ReadPDB(): Error 3. %d. \"%s\".",z+1,obuf);
			return false;
		}
		*q = 0;
		z = (float)atof(p) * 100.0f;
		m_vaCoords.Add(CxVector3(x,y,z));
	}
	m_iGesAtomCount = m_vaCoords.GetSize();

	return true;
}


bool CTimeStep::SkipPDB(FILE *a)
{
	char buf[256];
	bool b;

	while (true)
	{
		fgets_bin(buf,256,a);
		if (feof(a))
			return false;
		if (strlen(buf) == 0)
			continue;
		buf[strlen(buf)-1] = 0;
		if ((strstr(buf,"ATOM")==0) || (strstr(buf,"HETATM")==0))
		{
			if (b)
				break;
					else continue;
		} else
			b = true;
	}
	return true;
}


bool CTimeStep::ReadLAMMPS(FILE *a, bool needinfo, CxVec3Array *v)
{
	int i;
	char buf[256], obuf[256], *p, *q, *r;
	float x, y, z;

	m_vaCoords.RemoveAll_KeepSize();
	for (i=0;i<m_paLabels.GetSize();i++)
		delete[] (char*)m_paLabels[i];
	m_paLabels.RemoveAll();
	m_iGesAtomCount = 0;
	while (true)
	{
		fgets_bin(buf,256,a);
		if (feof(a))
			return false;
		if (strlen(buf) == 0)
			continue;
		buf[strlen(buf)-1] = 0;
		strcpy(obuf,buf);
		if (g_bNPT)
		{
			if (strstr(buf,"ITEM: BOX BOUNDS") != 0) // Boxlaenge
			{
				for (i=0;i<3;i++)
				{
					fgets_bin(buf,256,a);
					if (feof(a))
						return false;

					p = &buf[0];
					while (strchr(" ",*p) != NULL)
						p++;
					q = p+1;
					while ((strchr(" ",*q) == NULL) && (*q != 0))
						q++;
					if (*q == 0)
					{
						eprintf("\nCTimeStep::ReadLAMMPS(): Incomplete line: \"%s\"\n",obuf);
						return false;
					}
					*q = 0;
					x = atof(p);

					p = q+1;
					while (strchr(" ",*p) != NULL)
						p++;

					y = atof(p);

					switch(i)
					{
						case 0: g_fBoxX = (y-x)*100.0f; break;
						case 1: g_fBoxY = (y-x)*100.0f; break;
						case 2: g_fBoxZ = (y-x)*100.0f; break;
					}
				}
			}
		}
		if (strstr(buf,"ITEM: NUMBER OF ATOMS") != 0)
		{
			fgets_bin(buf,256,a);
			if (feof(a))
				return false;
			buf[strlen(buf)-1] = 0;
			m_iGesAtomCount = atoi(buf);
			continue;
		}
		if (strstr(buf,"ITEM: ATOMS") != 0)
		{
			if (strstr(buf,"ITEM: ATOMS element xu yu zu") == 0)
			{
				eprintf("CTimeStep::ReadLAMMPS(): Wrong LAMMPS dump style: \"%s\".\n",buf);
				eprintf("You need to use \"dump custom element xu yu zu\".\n");
				return false;
			}

			if (m_iGesAtomCount == 0)
			{
				eprintf("CTimeStep::ReadLAMMPS(): \"ITEM: ATOMS\" before \"ITEM: NUMBER OF ATOMS\".\n");
				return false;
			}

			if (needinfo)
			{
				if (g_bDoubleBox)
					m_paLabels.SetSize(m_iGesAtomCount*g_iDoubleBoxFactor);
						else m_paLabels.SetSize(m_iGesAtomCount);
			}

			for (i=0;i<m_iGesAtomCount;i++)
			{
				fgets_bin(buf,256,a);
				if (feof(a))
					return false;

				p = &buf[0];
				while (strchr(" ",*p) != NULL)
					p++;
				q = p+1;
				while ((strchr(" ",*q) == NULL) && (*q != 0))
					q++;
				if (*q == 0)
				{
					eprintf("\nCTimeStep::ReadLAMMPS(): %d: Incomplete line (1): \"%s\"\n",i+1,obuf);
					BTOUT; 
					return false;
				}
				*q = 0;

				if (needinfo)
				{
					if (strlen(p) > 7)
					{
						eprintf("\nCTimeStep::ReadLAMMPS(): \"%s\" - maximum length for atom labels is 7 chars; truncating.\n",p);
						p[7] = 0;
					}

					try { r = new char[strlen(p)+1]; } catch(...) { r = NULL; }
					if (r == NULL) NewException((double)(strlen(p)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					strcpy(r,p);
					m_paLabels[i] = r;
				}

				q++;
				while (strchr(" ",*q) != NULL)
					q++;
				p = q;
				while ((strchr(" ",*p) == NULL) && (*p != 0))
					p++;
				if (*p == 0)
				{
					eprintf("\nCTimeStep::ReadLAMMPS(): %d: Incomplete line (2): \"%s\"",i+1,obuf);
					return false;
				}
				*p = 0;
				x = (float)atof(q) * 100.0f;
				q = p+1;
				while (strchr(" ",*q) != NULL)
					q++;
				p = q;
				while ((strchr(" ",*p) == NULL) && (*p != 0))
					p++;
				if (*p == 0)
				{
					eprintf("\nCTimeStep::ReadLAMMPS(): %d: Incomplete line (3) \"%s\"",i+1,obuf);
					return false;
				}
				*p = 0;
				y = (float)atof(q) * 100.0f;
				q = p+1;
				while (strchr(" ",*q) != NULL)
					q++;
				p = q;
				while ((strchr(" ",*p) == NULL) && (*p != 0))
					p++;
				if (*p != 0)
					*p = 0;
				z = (float)atof(q) * 100.0f;

				m_vaCoords.Add(CxVector3(x,y,z));
			}
			break;
		}
	}
	return true;
}


bool CTimeStep::SkipLAMMPS(FILE *a)
{
	int i, j;
	char buf[256];

	j = 0;
	while (true)
	{
		fgets_bin(buf,256,a);
		if (feof(a))
			return false;
		if (strlen(buf) == 0)
			continue;
		buf[strlen(buf)-1] = 0;
		if (strstr(buf,"ITEM: NUMBER OF ATOMS") != 0)
		{
			fgets_bin(buf,256,a);
			if (feof(a))
				return false;
			buf[strlen(buf)-1] = 0;
			j = atoi(buf);
			continue;
		}
		if (strstr(buf,"ITEM: ATOMS") != 0)
		{
			if (j == 0)
			{
				eprintf("CTimeStep::SkipLAMMPS():  \"ITEM: ATOMS\" before \"ITEM: NUMBER OF ATOMS\".\n");
				return false;
			}
			for (i=0;i<j;i++)
			{
				fgets_bin(buf,256,a);
				if (feof(a))
					return false;
			}
			break;
		}
	}
	return true;
}


bool CTimeStep::ReadDLPOLY(FILE *a, bool needinfo, CxVec3Array *v)
{
	int i;
	char buf[256], obuf[256], *p, *q, *r;
	float x, y, z;

	m_vaCoords.RemoveAll_KeepSize();
	for (i=0;i<m_paLabels.GetSize();i++)
		delete[] (char*)m_paLabels[i];
	m_paLabels.RemoveAll();
	m_iGesAtomCount = 0;
	while (true)
	{
		fgets_bin(buf,256,a);
		if (feof(a))
			return false;
		if (strlen(buf) == 0)
			continue;
		buf[strlen(buf)-1] = 0;
		strcpy(obuf,buf);

		if (strstr(buf,"timestep") != 0)
		{
			p = &buf[0];
			while (*p == ' ')
				p++;
			q = p;
			while ((*q != ' ') && (*q != 0))
				q++;
			if (*q == 0)
			{
				eprintf("\nCTimeStep::ReadDLPOLY(): Incomplete line A: \"%s\"\n",obuf);
				return false;
			}
			*q = 0;
			p=q+1;

			while (*p == ' ')
				p++;
			q = p;
			while ((*q != ' ') && (*q != 0))
				q++;
			if (*q == 0)
			{
				eprintf("\nCTimeStep::ReadDLPOLY(): Incomplete line B: \"%s\"\n",obuf);
				return false;
			}
			*q = 0;
			p=q+1;

			while (*p == ' ')
				p++;
			q = p;
			while ((*q != ' ') && (*q != 0))
				q++;
			if (*q == 0)
			{
				eprintf("\nCTimeStep::ReadDLPOLY(): Incomplete line C: \"%s\"\n",obuf);
				return false;
			}
			*q = 0;
			m_iGesAtomCount = atoi(p);

			if (g_bNPT)
			{
				for (i=0;i<3;i++)
				{
					fgets_bin(buf,256,a);
					if (feof(a))
						return false;

					p = &buf[0];

					while (*p == ' ')
						p++;
					q = p;
					while ((*q != ' ') && (*q != 0))
						q++;
					if (*q == 0)
					{
						eprintf("\nCTimeStep::ReadDLPOLY(): Incomplete line D: \"%s\"\n",obuf);
						return false;
					}
					*q = 0;
					x = atof(p);
					p = q+1;

					while (*p == ' ')
						p++;
					q = p;
					while ((*q != ' ') && (*q != 0))
						q++;
					if (*q == 0)
					{
						eprintf("\nCTimeStep::ReadDLPOLY(): Incomplete line E: \"%s\"\n",obuf);
						return false;
					}
					*q = 0;
					y = atof(p);
					p = q+1;

					while (*p == ' ')
						p++;
					q = p;
					while ((*q != ' ') && (*q != 0))
						q++;
					*q = 0;
					z = atof(p);

					switch(i)
					{
						case 0:
							if ((y != 0) || (z != 0))
							{
								eprintf("\nCTimeStep::ReadDLPOLY(): X: Only orthorhombic cells are supported.\n");
								return false;
							}
							g_fBoxX = x*100.0f;
							break;

						case 1:
							if ((x != 0) || (z != 0))
							{
								eprintf("\nCTimeStep::ReadDLPOLY(): Y: Only orthorhombic cells are supported.\n");
								return false;
							}
							g_fBoxY = y*100.0f;
							break;

						case 2:
							if ((x != 0) || (y != 0))
							{
								eprintf("\nCTimeStep::ReadDLPOLY(): Z: Only orthorhombic cells are supported.\n");
								return false;
							}
							g_fBoxZ = z*100.0f;
							break;

					}
				}
			} else
			{
				fgets_bin(buf,256,a);
				fgets_bin(buf,256,a);
				fgets_bin(buf,256,a);
			}

			if (m_iGesAtomCount == 0)
			{
				eprintf("CTimeStep::ReadDLPOLY(): Error: Atom count is 0.\n");
				return false;
			}

			if (needinfo)
			{
				if (g_bDoubleBox)
					m_paLabels.SetSize(m_iGesAtomCount*g_iDoubleBoxFactor);
						else m_paLabels.SetSize(m_iGesAtomCount);
			}

			for (i=0;i<m_iGesAtomCount;i++)
			{
				fgets_bin(buf,256,a);
				if (feof(a))
					return false;

				p = &buf[0];
				while (*p == ' ')
					p++;
				q = p;
				while ((*q != ' ') && (*q != 0))
					q++;
				if (*q == 0)
				{
					eprintf("\nCTimeStep::ReadDLPOLY(): %d: Incomplete line F: \"%s\"\n",i+1,obuf);
					return false;
				}
				*q = 0;

				if (needinfo)
				{
					if (strlen(p) > 7)
					{
						eprintf("\nCTimeStep::ReadDLPOLY(): \"%s\" - maximum length for atom labels is 7 chars; truncating.\n",p);
						p[7] = 0;
					}

					try { r = new char[strlen(p)+1]; } catch(...) { r = NULL; }
					if (r == NULL) NewException((double)(strlen(p)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					strcpy(r,p);
					m_paLabels[i] = r;
				}

				fgets_bin(buf,256,a);
				if (feof(a))
					return false;

				p = &buf[0];

				while (*p == ' ')
					p++;
				q = p;
				while ((*q != ' ') && (*q != 0))
					q++;
				if (*q == 0)
				{
					eprintf("\nCTimeStep::ReadDLPOLY(): %d: Incomplete line G: \"%s\"\n",i+1,obuf);
					return false;
				}
				*q = 0;
				x = atof(p)*100.0f;
				p = q+1;

				while (*p == ' ')
					p++;
				q = p;
				while ((*q != ' ') && (*q != 0))
					q++;
				if (*q == 0)
				{
					eprintf("\nCTimeStep::ReadDLPOLY(): %d: Incomplete line H: \"%s\"\n",i+1,obuf);
					return false;
				}
				*q = 0;
				y = atof(p)*100.0f;
				p = q+1;

				while (*p == ' ')
					p++;
				q = p;
				while ((*q != ' ') && (*q != 0))
					q++;
				*q = 0;
				z = atof(p)*100.0f;

				m_vaCoords.Add(CxVector3(x,y,z));
			}
			break;
		}
	}
	return true;
}


bool CTimeStep::SkipDLPOLY(FILE *a)
{
	int i, j;
	char buf[256], obuf[256], *p, *q;

	while (true)
	{
		fgets_bin(buf,256,a);
		if (feof(a))
			return false;
		if (strlen(buf) == 0)
			continue;
		buf[strlen(buf)-1] = 0;
		strcpy(obuf,buf);

		if (strstr(buf,"timestep") != 0)
		{
			p = &buf[0];
			while (*p == ' ')
				p++;
			q = p;
			while ((*q != ' ') && (*q != 0))
				q++;
			if (*q == 0)
			{
				eprintf("\nCTimeStep::ReadDLPOLY(): Incomplete line A: \"%s\"\n",obuf);
				return false;
			}
			*q = 0;
			p=q+1;

			while (*p == ' ')
				p++;
			q = p;
			while ((*q != ' ') && (*q != 0))
				q++;
			if (*q == 0)
			{
				eprintf("\nCTimeStep::ReadDLPOLY(): Incomplete line B: \"%s\"\n",obuf);
				return false;
			}
			*q = 0;
			p=q+1;

			while (*p == ' ')
				p++;
			q = p;
			while ((*q != ' ') && (*q != 0))
				q++;
			if (*q == 0)
			{
				eprintf("\nCTimeStep::ReadDLPOLY(): Incomplete line C: \"%s\"\n",obuf);
				return false;
			}
			*q = 0;
			j = atoi(p);

			fgets_bin(buf,256,a);
			fgets_bin(buf,256,a);
			fgets_bin(buf,256,a);

			if (j == 0)
			{
				eprintf("CTimeStep::ReadDLPOLY(): Error: Atom count is 0.\n");
				return false;
			}

			for (i=0;i<j;i++)
			{
				fgets_bin(buf,256,a);
				fgets_bin(buf,256,a);
				if (feof(a))
					return false;
			}
			break;
		}
	}
	return true;
}


bool CTimeStep::ReadMol2(FILE *a, bool needinfo, CxVec3Array *v)
{
	int i;
	char buf[256], obuf[256], *p, *q, *r;
	float x, y, z;

	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);
	p = buf;
	while (*p == ' ')
		p++;
	q = p;
	while ((*q != ' ') && (*q != 0))
		q++;
	*q = 0;
	m_iGesAtomCount = atoi(p);
	
	if (needinfo)
	{
		m_vaCoords.RemoveAll_KeepSize();
		for (i=0;i<m_paLabels.GetSize();i++)
			delete[] (char*)m_paLabels[i];
		m_paLabels.RemoveAll();
		m_paLabels.SetSize(m_iGesAtomCount);
		for (i=0;i<m_paMol2Types.GetSize();i++)
			delete[] (char*)m_paMol2Types[i];
		m_paMol2Types.RemoveAll();
		m_paMol2Types.SetSize(m_iGesAtomCount);
	}
	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);

	for (i=0;i<(long)m_iGesAtomCount;i++)
	{
		fgets_bin(buf,256,a);
		if (feof(a))
			return false;
		buf[strlen(buf)-1] = 0;
		strcpy(obuf,buf);
//		mprintf("  \"%s\"\n",buf);
		p = buf;
		while (*p == ' ')
			p++;
		while ((*p != ' ') && (*p != 0))
			p++;
		while (*p == ' ')
			p++;
		q = p;
		while (isalpha(*q) || (*q == '_'))
			q++;
		if (needinfo)
		{
			*q = 0;
			if (strlen(p) > 7)
			{
				eprintf("\nCTimeStep::ReadMol2(): \"%s\" - maximum length for atom labels is 7 chars; truncating.\n",p);
				p[7] = 0;
			}

			try { r = new char[strlen(p)+1]; } catch(...) { r = NULL; }
			if (r == NULL) NewException((double)(strlen(p)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			strcpy(r,p);
			m_paLabels[i] = r;
		}
		p = q+1;
		while ((*p != ' ') && (*p != 0))
			p++;
		while (*p == ' ')
			p++;

		q = strchr(p,' ');
		if (q == NULL)
		{
			eprintf("\nCTimeStep::ReadMol2(): Error 1. %d. \"%s\".",z+1,obuf);
			return false;
		}
		*q = 0;
		x = (float)atof(p) * 100.0f;
		p = q+1;
		while (*p == ' ')
			p++;

		q = strchr(p,' ');
		if (q == NULL)
		{
			eprintf("\nCTimeStep::ReadMol2(): Error 2. %d. \"%s\".",z+1,obuf);
			return false;
		}
		*q = 0;
		y = (float)atof(p) * 100.0f;
		p = q+1;
		while (*p == ' ')
			p++;

		q = strchr(p,' ');
		if (q == NULL)
		{
			eprintf("\nCTimeStep::ReadMol2(): Error 3. %d. \"%s\".",z+1,obuf);
			return false;
		}
		*q = 0;
		z = (float)atof(p) * 100.0f;
		m_vaCoords.Add(CxVector3(x,y,z));

		p = q+1;
		while (*p == ' ')
			p++;
		q = p;
		while ((*q != ' ') && (*q != 0))
			q++;
		if (needinfo)
		{
			*q = 0;

			try { r = new char[strlen(p)+1]; } catch(...) { r = NULL; }
			if (r == NULL) NewException((double)(strlen(p)+1)*sizeof(char),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			strcpy(r,p);
			m_paMol2Types[i] = r;
		}
/*		mprintf("  X=%f, Y=%f, Z=%f",x,y,z);
		if (needinfo)
			mprintf(", A=%s, B=%s",(char*)m_paLabels[i],(char*)m_paMol2Types[i]);
		mprintf("\n");*/
	}
	m_iGesAtomCount = m_vaCoords.GetSize();

	return true;
}


bool CTimeStep::SkipMol2(FILE *a)
{
	int i;
	char buf[256];

	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);
	fgets_bin(buf,256,a);

	for (i=0;i<(long)m_iGesAtomCount;i++)
	{
		fgets_bin(buf,256,a);
		if (feof(a))
			return false;
	}

	return true;
}


void CTimeStep::CalcMinMax()
{
	int z;

	m_vMin[0] = 1E20f;
	m_vMin[1] = 1E20f;
	m_vMin[2] = 1E20f;
	m_vMax[0] = -1E20f;
	m_vMax[1] = -1E20f;
	m_vMax[2] = -1E20f;

	for (z=0;z<(long)m_iGesAtomCount;z++)
	{
		if (m_vMin[0] > m_vaCoords[z][0])
			m_vMin[0] = m_vaCoords[z][0];
		if (m_vMin[1] > m_vaCoords[z][1])
			m_vMin[1] = m_vaCoords[z][1];
		if (m_vMin[2] > m_vaCoords[z][2])
			m_vMin[2] = m_vaCoords[z][2];
		if (m_vMax[0] < m_vaCoords[z][0])
			m_vMax[0] = m_vaCoords[z][0];
		if (m_vMax[1] < m_vaCoords[z][1])
			m_vMax[1] = m_vaCoords[z][1];
		if (m_vMax[2] < m_vaCoords[z][2])
			m_vMax[2] = m_vaCoords[z][2];
	}
}

void CTimeStep::WriteMol2(FILE *a)
{
	int z, z2, z3, z4, ti, c, mc;
	CMolecule *m;
	CSingleMolecule *sm;
	int bonds;
	int *tpi;
	char *cc;

	bonds = 0;
	for (z=0;z<g_oaSingleMolecules.GetSize();z++)
		bonds += ((CSingleMolecule*)g_oaSingleMolecules[z])->m_oaBonds.GetSize();

	try { tpi = new int[g_iGesAtomCount]; } catch(...) { tpi = NULL; }
	if (tpi == NULL) NewException((double)g_iGesAtomCount*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	fprintf(a," @<TRIPOS>MOLECULE\n");
	fprintf(a,"MOL\n");
	fprintf(a,"    %d    %d     %d  0  0\n",g_iGesAtomCount,bonds,g_oaSingleMolecules.GetSize());
	fprintf(a," SMALL\n");
	fprintf(a,"resp\n\n\n");
	fprintf(a," @<TRIPOS>ATOM\n");
	c = 0;
	mc = 0;
	if (m_faCharge.GetSize() != g_iGesAtomCount)
	{
		m_faCharge.SetSize(g_iGesAtomCount);
		for (z=0;z<g_iGesAtomCount;z++)
			m_faCharge[z] = 0;
	}
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[z];
		for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
		{
			sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];
			mc++;
			for (z3=0;z3<m->m_baAtomIndex.GetSize();z3++)
			{
				if (m->m_baAtomIndex[z3] == g_iVirtAtomType)
					continue;
				for (z4=0;z4<((CxIntArray*)sm->m_oaAtomOffset[z3])->GetSize();z4++)
				{
					ti = ((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4);
					if (m_paMol2Types.GetSize() != 0)
						cc = (char*)m_paMol2Types[ti];
							else cc = ((CAtom*)g_oaAtoms[m->m_baAtomIndex[z3]])->m_sName;
					tpi[ti] = c++;
					fprintf(a,"  %6d  %2s  % 11.4f  % 11.4f  % 11.4f  %2s  %4d  MOL  % 8.4f\n",c,((CAtom*)g_oaAtoms[m->m_baAtomIndex[z3]])->m_sName,m_vaCoords[ti][0]/100.0,m_vaCoords[ti][1]/100.0,m_vaCoords[ti][2]/100.0,cc,mc,m_faCharge[ti]);
				}
			}
		}
	}
	fprintf(a," @<TRIPOS>BOND\n");
	c = 0;
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[z];
		for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
		{
			sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];
			for (z3=0;z3<sm->m_oaBonds.GetSize();z3++)
				fprintf(a,"  %6d  %6d  %6d  1\n",++c,tpi[((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[0]]+1,tpi[((CMolBond*)sm->m_oaBonds[z3])->m_iAtomOffset[1]]+1);
		}
	}
	fprintf(a," @<TRIPOS>SUBSTRUCTURE\n");
	c = 0;
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		m = (CMolecule*)g_oaMolecules[z];
		for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
		{
			fprintf(a,"  %4d  MOL  %4d  TEMP              0 ****  ****    0 ROOT\n",c+1,c+1);
			c++;
		}
	}
	delete[] tpi;
}

void CTimeStep::ReadCellVector(FILE *a)
{
	char buf[256], obuf[256];
	char *p, *q;
	float tf;

	fgets(buf,256,a);
	if (feof(a))
	{
		eprintf("\nReadCellVector: End of file.\n");
		eprintf("Your cell vector text file is too short.\n");
		return;
	}
	if (strlen(buf) == 0)
	{
		eprintf("\nReadCellVector: Empty line.\n");
		return;
	}
	buf[strlen(buf)-1] = 0;
	strcpy(obuf,buf);
//	mprintf(GREY,"\nReadCellVector: \"%s\".\n",buf);
	p = buf;
	while ((!isnumeric(*p)) && (*p != 0))
		p++;
	if (*p == 0)
	{
		eprintf("\nReadCellVector: Incomplete line (1) \"%s\".\n",obuf);
		return;
	}
	q = p;
	while (isnumeric(*q))
		q++;
	if (*q == 0)
	{
		eprintf("\nReadCellVector: Incomplete line (2) \"%s\".\n",obuf);
		return;
	}
	*q = 0;
	tf = (float)(atof(p)*100.0);
	if (tf <= 0)
	{
		eprintf("\nReadCellVector: Cell vectors need to be > 0 (X) \"%s\".\n",obuf);
		return;
	}
	g_fBoxX = tf;

	p = q+1;
	while ((!isnumeric(*p)) && (*p != 0))
		p++;
	if (*p == 0)
	{
		eprintf("\nReadCellVector: Incomplete line (3) \"%s\".\n",obuf);
		return;
	}
	q = p;
	while (isnumeric(*q))
		q++;
	if (*q == 0)
	{
		eprintf("\nReadCellVector: Incomplete line (4) \"%s\".\n",obuf);
		return;
	}
	*q = 0;
	tf = (float)(atof(p)*100.0);
	if (tf <= 0)
	{
		eprintf("\nReadCellVector: Cell vectors need to be > 0 (Y) \"%s\".\n",obuf);
		return;
	}
	g_fBoxY = tf;
	p = q+1;

	while ((!isnumeric(*p)) && (*p != 0))
		p++;
	if (*p == 0)
	{
		eprintf("\nReadCellVector: Incomplete line (5) \"%s\".\n",obuf);
		return;
	}
	q = p;
	while (isnumeric(*q))
		q++;
	if (*q != 0)
		*q = 0;
	tf = (float)(atof(p)*100.0);
	if (tf <= 0)
	{
		eprintf("\nReadCellVector: Cell vectors need to be > 0 (Z) \"%s\".\n",obuf);
		return;
	}
	g_fBoxZ = tf;

//	mprintf(GREY," --> %f %f %f\n",g_fBoxX,g_fBoxY,g_fBoxZ);
}


void CTimeStep::CenterCOM()
{
	CxVector3 vc;
	double m;
	int z;

	vc = CxVector3(0,0,0);
	m = 0;
	for (z=0;z<g_iGesAtomCount;z++)
	{
		vc += m_vaCoords[z] * ((CAtom*)g_oaAtoms[g_baAtomIndex[z]])->m_pElement->m_fMass;
		m += ((CAtom*)g_oaAtoms[g_baAtomIndex[z]])->m_pElement->m_fMass;
	}
	vc /= m;
	for (z=0;z<g_iGesAtomCount;z++)
		m_vaCoords[z] -= vc;
}

float CTimeStep::FoldedDistance(int i1, int i2)
{
	CxVector3 t;

	t = m_vaCoords[i2]-m_vaCoords[i1];

	while (t[0] < -g_fBoxX/2) t[0] += g_fBoxX;
	while (t[0] >= g_fBoxX/2) t[0] -= g_fBoxX;
	while (t[1] < -g_fBoxY/2) t[1] += g_fBoxY;
	while (t[1] >= g_fBoxY/2) t[1] -= g_fBoxY;
	while (t[2] < -g_fBoxZ/2) t[2] += g_fBoxZ;
	while (t[2] >= g_fBoxZ/2) t[2] -= g_fBoxZ;

	return t.GetLength();
}

void CTimeStep::FoldAtomsPositive()
{
	BTIN;
	int z;

	for (z=0;z<g_iGesVirtAtomCount;z++)
	{
		while (m_vaCoords[z][0] < 0) m_vaCoords[z][0] += g_fBoxX;
		while (m_vaCoords[z][0] >= g_fBoxX) m_vaCoords[z][0] -= g_fBoxX;
		while (m_vaCoords[z][1] < 0) m_vaCoords[z][1] += g_fBoxY;
		while (m_vaCoords[z][1] >= g_fBoxY) m_vaCoords[z][1] -= g_fBoxY;
		while (m_vaCoords[z][2] < 0) m_vaCoords[z][2] += g_fBoxZ;
		while (m_vaCoords[z][2] >= g_fBoxZ) m_vaCoords[z][2] -= g_fBoxZ;
	}
	BTOUT; 
}
