/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    Copyright (C) 2009-2012 Martin Brehm

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "cluster.h"
#include "grace.h"
#include "globalvar.h"
#include "maintools.h"


CClusterNode::CClusterNode()
{
}


CClusterNode::~CClusterNode()
{
}


CClusterAnalysis::CClusterAnalysis()
{
	m_poaClusterPoly = NULL;
	m_pDistCache = NULL;
}


CClusterAnalysis::~CClusterAnalysis()
{
	if (m_poaClusterPoly != NULL)
		delete[] m_poaClusterPoly;
	if (m_pDistCache != NULL)
		delete[] m_pDistCache;
}


void CClusterAnalysis::AddCluster(int i)
{
	CClusterNode *n;

	try { n = new CClusterNode(); } catch(...) { n = NULL; }
	if (n == NULL) NewException((double)sizeof(CClusterNode),__FILE__,__LINE__,__PRETTY_FUNCTION__);

//	n->m_iParent = -1;
	n->m_iIndex = m_oaClusters.GetSize();
	n->m_fDistance = 1.0E30f;
	n->m_iMonomers = 1;
	n->m_pParent = NULL;
	if (m_bDistCache)
		n->m_iaAtoms.SetMaxSize(i);
			else n->m_vaAtoms.SetMaxSize(i);
	n->m_iChildren[0] = -1;

	m_oaClusters.Add(n);
	m_oaBaseClusters.Add(n);
	m_poaClusterPoly[0].Add(n);
}


void CClusterAnalysis::AddParticle(float x, float y, float z)
{
	CClusterNode *n;

	n = (CClusterNode*)m_oaClusters[m_oaClusters.GetSize()-1];
	n->m_vaAtoms.Add(CxVector3(x,y,z));
}


void CClusterAnalysis::AddParticle(int i)
{
	CClusterNode *n;

	n = (CClusterNode*)m_oaClusters[m_oaClusters.GetSize()-1];
	n->m_iaAtoms.Add(i);
}


void CClusterAnalysis::BuildTree()
{
	int z, i, j, st/*, m*/;
	float d;
	CClusterNode *n, *c1, *c2;

//	mprintf("BuildTree online.\n");
	for (z=0;z<m_oaClusters.GetSize();z++)
		m_oaTopClusters.Add(m_oaClusters[z]);

	for (z=0;z<m_oaClusters.GetSize();z++)
	{
		n = (CClusterNode*)m_oaClusters[z];
		FindNearestNeighbor(n);
	}

//	mprintf("Initialization done.\n");

	st = 0;
	while (m_oaTopClusters.GetSize() > 1)
	{
		st++;
//		mprintf("Step %d, %d Top-Clusters remaining.\n",st,m_oaTopClusters.GetSize());
		d = 1.0E30f;
		for (z=0;z<m_oaTopClusters.GetSize();z++)
		{
			if (((CClusterNode*)m_oaTopClusters[z])->m_fNNDistance < d)
			{
				d = ((CClusterNode*)m_oaTopClusters[z])->m_fNNDistance;
				i = z;
			}
		}
		j = ((CClusterNode*)m_oaTopClusters[i])->m_iNextNeighbor;

		c1 = (CClusterNode*)m_oaTopClusters[i];
		c2 = (CClusterNode*)m_oaClusters[j];

		try { n = new CClusterNode(); } catch(...) { n = NULL; }
		if (n == NULL) NewException((double)sizeof(CClusterNode),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		if (m_bDistCache)
			n->m_vaAtoms.SetMaxSize(c1->m_iaAtoms.GetSize()+c2->m_iaAtoms.GetSize());
				else n->m_vaAtoms.SetMaxSize(c1->m_vaAtoms.GetSize()+c2->m_vaAtoms.GetSize());

		n->m_pParent = NULL;
//		n->m_iParent = -1;
		n->m_fDistance = d;
//		n->m_laChildren.Add(((CClusterNode*)m_oaTopClusters[i])->m_iIndex);
//		n->m_laChildren.Add(j);
		n->m_iChildren[0] = c1->m_iIndex;
		n->m_iChildren[1] = j;
		n->m_iMonomers = c1->m_iMonomers + c2->m_iMonomers;
//		m = n->m_iMonomers - 1;
		m_poaClusterPoly[n->m_iMonomers-1].Add(n);
//		mprintf("  Combining %d and %d, d=%f...\n",((CClusterNode*)m_oaTopClusters[i])->m_iIndex+1,j+1,d);
		n->m_iIndex = m_oaClusters.GetSize();

/*		m_pClusterSizeDF->AddToBinInt_Fast(m);

		m_pClusterDistribution2DF->AddToBinInt_Fast(m,d-((CClusterNode*)m_oaTopClusters[i])->m_fDistance);

		if (((CClusterNode*)m_oaTopClusters[i])->m_iMonomers == 1)
			m_pClusterDistribution2DF->AddToBinInt_Fast(0,d);
				else m_pClusterDistribution2DF->AddToBinInt_Fast(((CClusterNode*)m_oaTopClusters[i])->m_iMonomers-1,d-((CClusterNode*)m_oaTopClusters[i])->m_fDistance);

		if (((CClusterNode*)m_oaClusters[j])->m_iMonomers == 1)
			m_pClusterDistribution2DF->AddToBinInt_Fast(0,d);
				else m_pClusterDistribution2DF->AddToBinInt_Fast(((CClusterNode*)m_oaClusters[j])->m_iMonomers-1,d-((CClusterNode*)m_oaClusters[j])->m_fDistance);
*/
//		((CClusterNode*)m_oaTopClusters[i])->m_iParent = n->m_iIndex;
		((CClusterNode*)m_oaTopClusters[i])->m_pParent = n;

//		((CClusterNode*)m_oaClusters[j])->m_iParent = n->m_iIndex;
		((CClusterNode*)m_oaClusters[j])->m_pParent = n;

		if (m_bDistCache)
		{
			for (z=0;z<c1->m_iaAtoms.GetSize();z++)
				n->m_iaAtoms.Add(c1->m_iaAtoms[z]);

			for (z=0;z<c2->m_iaAtoms.GetSize();z++)
				n->m_iaAtoms.Add(c2->m_iaAtoms[z]);
		} else
		{
			for (z=0;z<c1->m_vaAtoms.GetSize();z++)
				n->m_vaAtoms.Add(c1->m_vaAtoms[z]);

			for (z=0;z<c2->m_vaAtoms.GetSize();z++)
				n->m_vaAtoms.Add(c2->m_vaAtoms[z]);
		}

		m_oaClusters.Add(n);
		m_oaTopClusters.Add(n);
		m_oaTopClusters.RemoveAt_NoShrink(i,1);
		for (z=0;z<m_oaTopClusters.GetSize();z++)
		{
			if (((CClusterNode*)m_oaTopClusters[z])->m_iIndex == j)
			{
				m_oaTopClusters.RemoveAt_NoShrink(z,1);
				break;
			}
		}
//		for (z=0;z<m_oaTopClusters.GetSize();z++)
//			FindNearestNeighbor((CClusterNode*)m_oaTopClusters[z]);

		FindNearestNeighbor(n);

		TraceNeighbors();
	}
	m_pTop = n;
//	m_pClusterDistribution2DF->AddToBinInt_Fast(m_iMonomers-1,m_fMaxDist-n->m_fDistance);
//	mprintf("BuildTree done.\n");
}


void CClusterAnalysis::FindNearestNeighbor(CClusterNode *n)
{
	int z2, z3, z4;
	float t;
	CClusterNode *n2;

	n->m_fNNDistance = 1.0E30f;
	n->m_iNextNeighbor = -1;

	if (m_bDistCache)
	{
		for (z2=0;z2<m_oaTopClusters.GetSize();z2++)
		{
			n2 = (CClusterNode*)m_oaTopClusters[z2];
			if (n->m_iIndex == n2->m_iIndex)
				continue;
			for (z3=0;z3<n->m_iaAtoms.GetSize();z3++)
			{
				for (z4=0;z4<n2->m_iaAtoms.GetSize();z4++)
				{
	//				mprintf("  z2=%d, z3=%d, z4=%d, dist=%f.\n",z2,z3,z4,n->m_fNNDistance);
					t = Distance_Cache(n->m_iaAtoms[z3],n2->m_iaAtoms[z4]);
					if (t < n->m_fNNDistance)
					{
						n->m_fNNDistance = t;
						n->m_iNextNeighbor = n2->m_iIndex;
					}
				}
			}
		}
	} else
	{
		for (z2=0;z2<m_oaTopClusters.GetSize();z2++)
		{
			n2 = (CClusterNode*)m_oaTopClusters[z2];
			if (n->m_iIndex == n2->m_iIndex)
				continue;
			for (z3=0;z3<n->m_vaAtoms.GetSize();z3++)
			{
				for (z4=0;z4<n2->m_vaAtoms.GetSize();z4++)
				{
	//				mprintf("  z2=%d, z3=%d, z4=%d, dist=%f.\n",z2,z3,z4,n->m_fNNDistance);
					t = Distance(&n->m_vaAtoms[z3],&n2->m_vaAtoms[z4]);
					if (t < n->m_fNNDistance)
					{
						n->m_fNNDistance = t;
						n->m_iNextNeighbor = n2->m_iIndex;
					}
				}
			}
		}
	}
}


void CClusterAnalysis::TraceNeighbors()
{
	int z;
	CClusterNode *n, *n2;

	for (z=0;z<m_oaTopClusters.GetSize();z++)
	{
		n = (CClusterNode*)m_oaTopClusters[z];

		if (n->m_iNextNeighbor == -1)
			continue;

		n2 = (CClusterNode*)m_oaClusters[n->m_iNextNeighbor];

//		while (n2->m_iParent != -1)
//			n2 = (CClusterNode*)m_oaClusters[n2->m_iParent];

		while (n2->m_pParent != NULL)
			n2 = n2->m_pParent;

		n->m_iNextNeighbor = n2->m_iIndex;
	}
}


void CClusterAnalysis::DumpDot(const char *s)
{
	FILE *a;
	int z;
	CClusterNode *n;

	a = OpenFileWrite(s,true);

	fprintf(a,"digraph test {\n");
	for (z=0;z<m_oaClusters.GetSize();z++)
	{
		n = (CClusterNode*)m_oaClusters[z];
//		if (n->m_laChildren.GetSize() == 0)
		if (n->m_iChildren[0] == -1)
			fprintf(a,"  n%d [label=\"%d X%f\"];\n",z+1,n->m_iIndex+1,n->m_fPosX);
				else fprintf(a,"  n%d [label=\"X%f Y%.2f\"];\n",z+1,n->m_fPosX,n->m_fDistance);
	}

	fprintf(a,"\n");

	REC_DumpDot(a,m_pTop);

	fprintf(a,"}\n");

	fclose(a);
}


void CClusterAnalysis::REC_DumpDot(FILE *a, CClusterNode *n)
{
/*	int z;

	for (z=0;z<n->m_laChildren.GetSize();z++)
	{
		fprintf(a,"  n%d -> n%d;\n",n->m_iIndex+1,(int)(n->m_laChildren[z]+1));
		REC_DumpDot(a,(CClusterNode*)m_oaClusters[n->m_laChildren[z]]);
	}*/

	if (n->m_iChildren[0] != -1)
	{
		fprintf(a,"  n%d -> n%d;\n",n->m_iIndex+1,(int)(n->m_iChildren[0]+1));
		REC_DumpDot(a,(CClusterNode*)m_oaClusters[n->m_iChildren[0]]);
		fprintf(a,"  n%d -> n%d;\n",n->m_iIndex+1,(int)(n->m_iChildren[1]+1));
		REC_DumpDot(a,(CClusterNode*)m_oaClusters[n->m_iChildren[1]]);
	}
}


int CA_compareX(const void *arg1, const void *arg2 )
{
	if ((*((CClusterNode**)arg1))->m_fPosX > (*((CClusterNode**)arg2))->m_fPosX)
		return 1;
			else return -1;
}


void CClusterAnalysis::DumpAgr(const char *s)
{
	int z;
//	FILE *a;
	CxFloatArray **fa;
	int *p;
	float d, d2, d2l;
	bool b, c;
	int i;
	CGrace *g;
	char buf[256];

	REC_SortX(1,0,m_pTop);

	qsort(&m_oaBaseClusters[0],m_oaBaseClusters.GetSize(),sizeof(CxObject*),CA_compareX);

	for (z=0;z<m_oaBaseClusters.GetSize();z++)
	{
//		mprintf("%d: %f\n",z+1,((CClusterNode*)m_oaBaseClusters[z])->m_fPosX);
		((CClusterNode*)m_oaBaseClusters[z])->m_fPosX = z+1;
	}

	REC_AvgX(m_pTop);

	try { fa = new CxFloatArray*[m_oaBaseClusters.GetSize()]; } catch(...) { fa = NULL; }
	if (fa == NULL) NewException((double)m_oaBaseClusters.GetSize()*sizeof(CxFloatArray*),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	try { p = new int[m_oaBaseClusters.GetSize()]; } catch(...) { p = NULL; }
	if (p == NULL) NewException((double)m_oaBaseClusters.GetSize()*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<m_oaBaseClusters.GetSize();z++)
	{
		try { fa[z] = new CxFloatArray(); } catch(...) { fa[z] = NULL; }
		if (fa[z] == NULL) NewException((double)sizeof(CxFloatArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		p[z] = 0;
	}
	d = 0;

//	DumpDot("C:\\Software\\test.dot");

//	a = OpenFileWrite(s,true);

	try { g = new CGrace(); } catch(...) { g = NULL; }
	if (g == NULL) NewException((double)sizeof(CGrace),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	for (z=0;z<m_oaBaseClusters.GetSize();z++)
	{
		g->AddDataset();
		g->SetSetLineColor(z,0,0,0);
		sprintf(buf,"Molecule %d",((CClusterNode*)m_oaBaseClusters[z])->m_iIndex+1);
		g->SetDatasetName(z,buf);
	}

//	fprintf(a,"0");
	for (z=0;z<m_oaBaseClusters.GetSize();z++)
	{
		fa[z]->Add(((CClusterNode*)m_oaBaseClusters[z])->m_fPosX);
		fa[z]->Add(0);
		REC_DumpPoints(fa[z],(CClusterNode*)m_oaBaseClusters[z]);
//		fprintf(a,";  %f",((CClusterNode*)m_oaBaseClusters[z])->m_fPosX);
		g->AddXYTupel(z,((CClusterNode*)m_oaBaseClusters[z])->m_fPosX,0);
	}
//	fprintf(a,"\n");

/*	for (z=0;z<fa[0]->GetSize()/2;z++)
		fprintf(a,"%f; %f\n",fa[0]->GetAt(z*2),fa[0]->GetAt(z*2+1));
	fclose(a);
	return;*/

	i = 0;
	while (true)
	{
		i++;
//		mprintf("Stage %d starting.\n",i);
		d2l = d2;
		d2 = 1.0e35f;
		for (z=0;z<m_oaBaseClusters.GetSize();z++)
		{
//			mprintf("    Cluster %d: %d Pairs, Pos=%d.\n",z+1,fa[z]->GetSize()/2,p[z]);
			if ((p[z]+1)*2+1 < fa[z]->GetSize())
			{
//				mprintf("    %f < %f?\n",fa[z]->GetAt((p[z]+1)*2+1),d2);
				if (fa[z]->GetAt((p[z]+1)*2+1) < d2)
					d2 = fa[z]->GetAt((p[z]+1)*2+1);
			}
		}
//		mprintf("d2=%f.\n",d2);
		if (d2 > 1.0e30)
			break;
_next:
		b = false;
		c = false;
		for (z=0;z<m_oaBaseClusters.GetSize();z++)
		{
			if ((p[z]+1)*2+1 < fa[z]->GetSize())
			{
				if (fa[z]->GetAt((p[z]+1)*2+1) <= d2)
				{
					b = true;
					p[z]++;
				}
				if (!c)
				{
					c = true;
//					fprintf(a,"%f",d2);
				}
//				fprintf(a,";  %f",fa[z]->GetAt(p[z]*2));
				g->AddXYTupel(z,fa[z]->GetAt(p[z]*2),d2);
			}
		}
		if (!c)
		{
//			fprintf(a,"%f",d2+1.0);
			for (z=0;z<m_oaBaseClusters.GetSize();z++)
			{
				g->AddXYTupel(z,fa[0]->GetAt(p[0]*2),d2+100.0);
//				fprintf(a,";  %f",fa[0]->GetAt(p[0]*2));
			}
//			fprintf(a,"\n");
			break;
		}
//		fprintf(a,"\n");
		if (b)
			goto _next;
	}

	g->SetRangeX(0,m_oaBaseClusters.GetSize()+1);
	g->SetRangeY(0,m_fMaxDist);
	g->MakeTicks();
	g->SetLabelX("Molecule");
	g->SetLabelY("Distance [pm]");

	g->WriteAgr(s,true);

//	fclose(a);
}


void CClusterAnalysis::REC_SortX(double pos, int depth, CClusterNode *n)
{
	n->m_fPosX = pos;

	if (n->m_iChildren[0] == -1)
		return;

	if (((CClusterNode*)m_oaClusters[n->m_iChildren[0]])->m_fDistance > ((CClusterNode*)m_oaClusters[n->m_iChildren[1]])->m_fDistance)
	{
		REC_SortX(pos+pow(0.5,depth+1),depth+1,(CClusterNode*)m_oaClusters[n->m_iChildren[0]]);
		REC_SortX(pos-pow(0.5,depth+1),depth+1,(CClusterNode*)m_oaClusters[n->m_iChildren[1]]);
	} else
	{
		REC_SortX(pos-pow(0.5,depth+1),depth+1,(CClusterNode*)m_oaClusters[n->m_iChildren[0]]);
		REC_SortX(pos+pow(0.5,depth+1),depth+1,(CClusterNode*)m_oaClusters[n->m_iChildren[1]]);
	}
}


void CClusterAnalysis::REC_AvgX(CClusterNode *n)
{
	double d;
//	int z;

	if (n->m_iChildren[0] == -1)
		return;

	d = 0;

/*	for (z=0;z<n->m_laChildren.GetSize();z++)
	{
		REC_AvgX((CClusterNode*)m_oaClusters[n->m_laChildren[z]]);
		d += ((CClusterNode*)m_oaClusters[n->m_laChildren[z]])->m_fPosX;
	}*/

	REC_AvgX((CClusterNode*)m_oaClusters[n->m_iChildren[0]]);
	d += ((CClusterNode*)m_oaClusters[n->m_iChildren[0]])->m_fPosX;
	REC_AvgX((CClusterNode*)m_oaClusters[n->m_iChildren[1]]);
	d += ((CClusterNode*)m_oaClusters[n->m_iChildren[1]])->m_fPosX;

	d /= 2.0; //n->m_laChildren.GetSize();
//	mprintf("PosX=%f.\n",d);
	n->m_fPosX = d;
}


void CClusterAnalysis::REC_DumpPoints(CxFloatArray *fa, CClusterNode *n)
{
	CClusterNode *n2;

//	if (n->m_iParent != -1)
	if (n->m_pParent != NULL)
	{
//		n2 = (CClusterNode*)m_oaClusters[n->m_iParent];
		n2 = n->m_pParent;
		fa->Add(n->m_fPosX);
		fa->Add(n2->m_fDistance);
		fa->Add(n2->m_fPosX);
		fa->Add(n2->m_fDistance);
		REC_DumpPoints(fa,n2);
	}
}


void CClusterAnalysis::BinDistances()
{
	int z, m;
	CClusterNode *n;

	for (z=0;z<m_oaClusters.GetSize();z++)
	{
		n = (CClusterNode*)m_oaClusters[z];

		if (n->m_fDistance > 1.0E10)
		{
			m_pClusterSizeDF->AddToBinInt_Fast(0);
			m_pClusterDistributionDF->AddToBinInt_Fast(0,n->m_pParent->m_fDistance);
			continue;
		}

		m = n->m_iMonomers-1;
		m_pClusterSizeDF->AddToBinInt_Fast(m);

		if (n->m_pParent != NULL)
			m_pClusterDistributionDF->AddToBinInt_Fast(m,n->m_pParent->m_fDistance-n->m_fDistance);
				else m_pClusterDistributionDF->AddToBinInt_Fast(m,m_fMaxDist-n->m_fDistance);

		m_pClusterDistanceDF->AddToBin_Fast(n->m_fDistance);
	}
}


void CClusterAnalysis::Parse()
{
	int ti, z;
	char buf[256], buf2[256];
	CAtomGroup *ag;

	mprintf(WHITE,">>> Cluster Analysis >>>\n");

_canextmol:
	mprintf("\n");
	sprintf(buf,"    Which molecule type to use for cluster analysis (");
	for (z=0;z<g_oaMolecules.GetSize();z++)
	{
		sprintf(buf2,"%s=%d",((CMolecule*)g_oaMolecules[z])->m_sName,z+1);
		strcat(buf,buf2);
		if (z < g_oaMolecules.GetSize()-1)
			strcat(buf,", ");
	}
	strcat(buf,")? ");

	ti = AskRangeInteger_ND(buf,1,g_oaMolecules.GetSize()) - 1;

	for (z=0;z<m_oaAtomGroups.GetSize();z++)
	{
		if (((CAtomGroup*)m_oaAtomGroups[z])->m_pMolecule->m_iIndex == ti)
		{
			eprintf("This molecule type is already used for cluster analysis.\n");
			goto _canextmol;
		}
	}

	try { ag = new CAtomGroup(); } catch(...) { ag = NULL; }
	if (ag == NULL) NewException((double)sizeof(CAtomGroup),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
_caagain:
	AskString("    Which atoms to take into account from %s (*=all)? [#2] ",buf,"#2",((CMolecule*)g_oaMolecules[ti])->m_sName);
	if (!ag->ParseAtoms((CMolecule*)g_oaMolecules[ti],buf))
		goto _caagain;

	m_oaAtomGroups.Add(ag);

	mprintf("\n");
	if (AskYesNo("    Add another molecule type to the cluster analysis (y/n)? [no] ",false))
		goto _canextmol;

	m_fMaxDist = AskFloat("    Enter max. distance for cluster analysis distribution (in pm): [%d] ",(float)HalfBoxSq3(),HalfBoxSq3());
	m_iRes = AskUnsignedInteger("    Enter resolution of the cluster distance distribution: [300] ",300);
	m_bAnim = AskYesNo("    Create Cluster Graph animation (y/n)? [yes] ",true);

	if (m_bAnim)
	{
		m_iResX = AskUnsignedInteger("    Enter width (in pixel) of the animation images: [640] ",640);
		m_iResY = AskUnsignedInteger("    Enter height (in pixel) of the animation images: [480] ",480);
	}

	m_bDistCache = AskYesNo("    Use distance caching (y/n)? [yes] ",true);

	mprintf(WHITE,"\n<<< End of Cluster Analysis <<<\n\n");
}


void CClusterAnalysis::Create()
{
	int z, z2, z3, z4;
	char buf[64];
	CAtomGroup *ag;
	CSingleMolecule *sm;
	CMolecule *m;

	if (m_bAnim)
		m_fAnim = OpenFileWrite("render_cluster_anim",true);

	m_iCounter = 0;
	m_iMonomers = 0;

	for (z=0;z<m_oaAtomGroups.GetSize();z++)
		m_iMonomers += ((CAtomGroup*)m_oaAtomGroups[z])->m_pMolecule->m_laSingleMolIndex.GetSize();

	try { m_poaClusterPoly = new CxObArray[m_iMonomers]; } catch(...) { m_poaClusterPoly = NULL; }
	if (m_poaClusterPoly == NULL) NewException((double)m_iMonomers*sizeof(CxObArray),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	if (m_bDistCache)
	{
		for (z=0;z<m_oaAtomGroups.GetSize();z++)
		{
			ag = (CAtomGroup*)m_oaAtomGroups[z];
			m = ag->m_pMolecule;
			for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
			{
				sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];
				for (z3=0;z3<ag->m_oaAtoms.GetSize();z3++)
				{
					for (z4=0;z4<((CxIntArray*)ag->m_oaAtoms[z3])->GetSize();z4++)
						m_iaAtomList.Add(((CxIntArray*)sm->m_oaAtomOffset[ag->m_baAtomType[z3]])->GetAt(((CxIntArray*)ag->m_oaAtoms[z3])->GetAt(z4)));
				}
			}
		}

		m_iAtomListSize = m_iaAtomList.GetSize();

		try { m_pAtomIndex = new int[g_iGesVirtAtomCount]; } catch(...) { m_pAtomIndex = NULL; }
		if (m_pAtomIndex == NULL) NewException((double)g_iGesVirtAtomCount*sizeof(int),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		for (z=0;z<g_iGesVirtAtomCount;z++)
			m_pAtomIndex[z] = -1;

		for (z=0;z<m_iaAtomList.GetSize();z++)
			m_pAtomIndex[m_iaAtomList[z]] = z;

		try { m_pDistCache = new float[m_iaAtomList.GetSize()*m_iaAtomList.GetSize()]; } catch(...) { m_pDistCache = NULL; }
		if (m_pDistCache == NULL) NewException((double)m_iaAtomList.GetSize()*m_iaAtomList.GetSize()*sizeof(float),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	}	

	try { m_pClusterDistributionDF = new CDF(); } catch(...) { m_pClusterDistributionDF = NULL; }
	if (m_pClusterDistributionDF == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	m_pClusterDistributionDF->m_iResolution = m_iMonomers;
	m_pClusterDistributionDF->m_fMinVal = 1;
	m_pClusterDistributionDF->m_fMaxVal = m_iMonomers*((m_iMonomers+1.0)/m_iMonomers);
	m_pClusterDistributionDF->SetLabelX("Cluster size");
	m_pClusterDistributionDF->SetLabelY("Percentage");
	m_pClusterDistributionDF->m_bLeft = true;
	m_pClusterDistributionDF->Create();

/*	try { m_pClusterDistribution2DF = new CDF(); } catch(...) { m_pClusterDistribution2DF = NULL; }
	if (m_pClusterDistribution2DF == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	m_pClusterDistribution2DF->m_iResolution = m_iMonomers;
	m_pClusterDistribution2DF->m_fMinVal = 1;
	m_pClusterDistribution2DF->m_fMaxVal = m_iMonomers*((m_iMonomers+1.0)/m_iMonomers);
	m_pClusterDistribution2DF->SetLabelX("Cluster size");
	m_pClusterDistribution2DF->SetLabelY("Percentage");
	m_pClusterDistribution2DF->m_bLeft = true;
	m_pClusterDistribution2DF->Create();*/

	try { m_pClusterSizeDF = new CDF(); } catch(...) { m_pClusterSizeDF = NULL; }
	if (m_pClusterSizeDF == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	m_pClusterSizeDF->m_iResolution = m_iMonomers;
	m_pClusterSizeDF->m_fMinVal = 1;
	m_pClusterSizeDF->m_fMaxVal = m_iMonomers*((m_iMonomers+1.0)/m_iMonomers);
	m_pClusterSizeDF->SetLabelX("Cluster size");
	m_pClusterSizeDF->SetLabelY("Percentage");
	m_pClusterSizeDF->m_bLeft = true;
	m_pClusterSizeDF->Create();

	try { m_pClusterDistanceDF = new CDF(); } catch(...) { m_pClusterDistanceDF = NULL; }
	if (m_pClusterDistanceDF == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	m_pClusterDistanceDF->m_iResolution = m_iRes;
	m_pClusterDistanceDF->m_fMinVal = 0;
	m_pClusterDistanceDF->m_fMaxVal = m_fMaxDist;
	m_pClusterDistanceDF->SetLabelX("Distance [pm]");
	m_pClusterDistanceDF->SetLabelY("Occurrence");
	m_pClusterDistanceDF->Create();

	try { m_pPolymerDF = new CDF(); } catch(...) { m_pPolymerDF = NULL; }
	if (m_pPolymerDF == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	m_pPolymerDF->m_iResolution = m_iRes;
	m_pPolymerDF->m_fMinVal = 0;
	m_pPolymerDF->m_fMaxVal = m_fMaxDist;
	m_pPolymerDF->SetLabelX("Distance [pm]");
	m_pPolymerDF->SetLabelY("Occurrence");
	m_pPolymerDF->CreateMulti(m_iMonomers);
	m_pPolymerDF->SetLabelMulti(0,"Monomer");

	for (z=1;z<m_iMonomers;z++)
	{
		sprintf(buf,"%d-mer",z+1);
		m_pPolymerDF->SetLabelMulti(z,buf);
	}

	try { m_pClusterCountDF = new CDF(); } catch(...) { m_pClusterCountDF = NULL; }
	if (m_pClusterCountDF == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	m_pClusterCountDF->m_iResolution = m_iRes;
	m_pClusterCountDF->m_fMinVal = 0;
	m_pClusterCountDF->m_fMaxVal = m_fMaxDist;
	m_pClusterCountDF->SetLabelX("Distance [pm]");
	m_pClusterCountDF->SetLabelY("Number of Clusters");
	m_pClusterCountDF->Create();
}


void CClusterAnalysis::Process(CTimeStep *ts)
{
	int z0, z, z2, z3, ti;
	CMolecule *m;
	CSingleMolecule *sm;
	CAtomGroup *ag;
	char buf[256];

	CleanUp();

	if (m_bDistCache)
		BuildDistCache(ts);

	for (z0=0;z0<m_oaAtomGroups.GetSize();z0++)
	{
		ag = (CAtomGroup*)m_oaAtomGroups[z0];
		m = (CMolecule*)g_oaMolecules[ag->m_pMolecule->m_iIndex];
		for (z=0;z<m->m_laSingleMolIndex.GetSize();z++)
		{
			sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z]];
			AddCluster(ag->m_iAtomGes);
			for (z2=0;z2<ag->m_baAtomType.GetSize();z2++)
			{
				for (z3=0;z3<((CxIntArray*)ag->m_oaAtoms[z2])->GetSize();z3++)
				{
					ti = ((CxIntArray*)sm->m_oaAtomOffset[ag->m_baAtomType[z2]])->GetAt(((CxIntArray*)ag->m_oaAtoms[z2])->GetAt(z3));
					if (m_bDistCache)
						AddParticle(m_pAtomIndex[ti]);
							else AddParticle(ts->m_vaCoords[ti][0],ts->m_vaCoords[ti][1],ts->m_vaCoords[ti][2]);
				}
			}
		}
	}

	BuildTree();
	BinDistances();
	BinPoly();

	if (m_iCounter == 0)
	{
		DumpAgr("cluster_diagram.agr");
	}

	if (m_bAnim)
	{
		sprintf(buf,"cluster_anim_%05lu.agr",g_iSteps/g_iStride);
		DumpAgr(buf);
		fprintf(m_fAnim,"echo 'Printing frame %lu'\n",g_iSteps/g_iStride);
		fprintf(m_fAnim,"xmgrace %s -batch gracebatch -nosafe -hardcopy\n",buf);
		fprintf(m_fAnim,"mv output.png frame%05lu.png\n",g_iSteps/g_iStride);
	}

	m_iCounter++;
}


void CClusterAnalysis::CleanUp()
{
	int z;

	for (z=0;z<m_oaClusters.GetSize();z++)
		delete (CClusterNode*)m_oaClusters[z];
	m_oaClusters.RemoveAll_KeepSize();

	m_oaTopClusters.RemoveAll_KeepSize();

	m_oaBaseClusters.RemoveAll_KeepSize();

	for (z=0;z<m_iMonomers;z++)
		m_poaClusterPoly[z].RemoveAll_KeepSize();
}


void CClusterAnalysis::BinPoly()
{
	int z, z2, z3, l, u;
	CClusterNode *n;

	for (z=0;z<m_iMonomers;z++)
	{
		for (z2=0;z2<m_poaClusterPoly[z].GetSize();z2++)
		{
			n = (CClusterNode*)(m_poaClusterPoly[z])[z2];

			if (n->m_iChildren[0] == -1)
				l = 0;
					else l = n->m_fDistance / m_fMaxDist * m_iRes;

			if (l >= m_iRes)
				l = m_iRes-1;

//			if (n->m_iParent == -1)
//				u = m_iRes-1;
//					else u = ((CClusterNode*)m_oaClusters[n->m_iParent])->m_fDistance / m_fMaxDist * m_iRes - 1;

			if (n->m_pParent == NULL)
				u = m_iRes-1;
					else u = n->m_pParent->m_fDistance / m_fMaxDist * m_iRes - 1;

			if (u >= m_iRes)
				u = m_iRes-1;

			for (z3=l;z3<=u;z3++)
			{
				m_pPolymerDF->AddToBin_Multi_Int_Fast(z,z3,z+1.0);
				m_pClusterCountDF->AddToBinInt_Fast(z3);
			}
		}
	}
}


void CClusterAnalysis::BuildDistCache(CTimeStep *ts)
{
	int z, z2;
	CxVector3 *a, *b;
	float t;

	for (z=0;z<m_iAtomListSize;z++)
	{
		a = &ts->m_vaCoords[m_iaAtomList[z]];
		for (z2=z+1;z2<m_iAtomListSize;z2++)
		{
			b = &ts->m_vaCoords[m_iaAtomList[z2]];
			t = Distance(a,b);
			m_pDistCache[z*m_iAtomListSize+z2] = t;
			m_pDistCache[z2*m_iAtomListSize+z] = t;
		}
	}
}


void CClusterAnalysis::BuildClusterDistribution()
{
	int z/*, z2*/;

	for (z=0;z<m_iMonomers;z++)
	{
//		for (z2=0;z2<m_iRes;z2++)
//			m_pClusterDistributionDF->m_pBin[z] += m_pPolymerDF->m_pMultiBin[z][z2];

//		m_pClusterDistributionDF->m_pBin[z] *= 100.0 / m_iRes / m_iMonomers / m_iCounter;

		m_pClusterDistributionDF->m_pBin[z] *= 100.0 * (z+1.0) / m_iMonomers / m_iCounter / m_fMaxDist;

		m_pClusterSizeDF->m_pBin[z] *= 100.0 * (z+1.0) / m_iMonomers / m_iCounter;
	}
}
