/*
	**
	** htmlfe.c 
	**
	** HTML report frontend to network traffic visualiser
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: htmlfe.c,v 1.3 1999/02/15 12:21:56 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <glib.h>

#include "htmlfe.h"
#include "report.h"
#include "util.h"

static char			rcsid[] = "$Id: htmlfe.c,v 1.3 1999/02/15 12:21:56 dmiller Exp $";

/* Prototypes */
static void print_host(host_t *host, FILE *out);
static void print_peers(peer_t *peers, FILE *out);
static void format_hostname(host_t *host, char *s, size_t s_len);
static const char *format_time(time_t t);

/* Displays summary of network traffic */
void html_display_summary(report_t *r, const char *output_file)
{
	host_t	*h;
	FILE		*out;
	
	if (output_file == NULL)
	{
		setlinebuf(stdout);
		out = stdout;
	} else
	{
		/* Open output file */
		out = fopen(output_file, "w");
		if (out == NULL)
		{
			fprintf(stderr, "Couldn't open file '%s' for writing.\n", output_file);
			return;
		}
	}
	
	fprintf(out, "\
<HTML>\n\
	<HEAD>\n\
		<TITLE>traffic-vis report</TITLE>\n\
		<STYLE TYPE=\"text/css\"><!--\n\
			BODY { font-family: helvetica, arial, sans-serif; }\n\
			TD, TH { font-family: helvetica, arial, sans-serif; }\n\
			H { font-family: helvetica, arial, sans-serif; }\n\
		--></STYLE>\n\
	</HEAD>\n\
	<BODY BGCOLOR=\"#FFFFFF\">\n\
		<TABLE WIDTH=\"90%%\" ALIGN=\"CENTER\">\n\
			<TR>\n\
				<TD ALIGN=\"LEFT\">\n\
					<H1>traffic-vis report</H1>\n\
");					

	fprintf(out, "					<BIG><B>Collection started at %s</B></BIG><BR>\n", format_time(r->summary_start));
	fprintf(out, "					<BIG><B>Collection concluded at %s</B></BIG><BR><BR>", format_time(r->summary_finish));

	fprintf(out, "\
					<TABLE>\n\
						<TR>\n\
							<TD>\n\
								<B>Number of active hosts:</B>\n\
							</TD>\n\
							<TD>\n\
								%u\n\
							</TD>\n\
						</TR>\n\
						<TR>\n\
							<TD>\n\
								<B>Total bytes transferred:</B>\n\
							</TD>\n\
							<TD>\n\
								%llu\n\
							</TD>\n\
						</TR>\n\
						<TR>\n\
							<TD>\n\
								<B>Total packets recorded:</B>\n\
							</TD>\n\
							<TD>\n\
								%llu\n\
							</TD>\n\
						</TR>\n\
						<TR>\n\
							<TD>\n\
								<B>Total TCP connection requests:</B>\n\
							</TD>\n\
							<TD>\n\
								%llu\n\
							</TD>\n\
						</TR>\n\
					</TABLE>\n\
				<BR>\n\
", r->active_hosts, r->total_bytes, r->total_packets, r->total_connections);

	h = r->hosts;
	while(h != NULL)
	{
		print_host(h, out);
		h = h->next;
	}

	fprintf(out, "\
				</TD>\n\
			</TR>\n\
		</TABLE>\n\
	</BODY>\n\
</HTML>\n\
");

	if (output_file != NULL)
		fclose(out);
}

static void print_host(host_t *host, FILE *out)
{
	char			hostname[1024];

	format_hostname(host, hostname, sizeof(hostname));
	
	fprintf(out, "\
					<TABLE BORDER=1 WIDTH=\"100%%\" CELLSPACING=0>\n\
						<TR>\n\
							<TD BGCOLOR=\"#000000\" ALIGN=\"LEFT\">\n\
								<FONT COLOR=\"#FFFFFF\">\n\
									<B>%s</B>\n\
								</FONT>\n\
							</TD>\n\
						</TR>\n\
", hostname);

	fprintf(out, "\
						<TR>\n\
							<TD ALIGN=\"LEFT\">\n\
");
	print_peers(host->peers, out);
	
	fprintf(out, "\
								<BR>\n\
								<TABLE WIDTH=\"100%%\" CELLSPACING=0 BORDER=0>\n\
									<TR BGCOLOR=\"#E0E0E0\">\n\
										<TD>&nbsp</TD>\n\
										<TD ALIGN=\"RIGHT\"><B>Sent</B></TD>\n\
										<TD ALIGN=\"RIGHT\"><B>Received</B></TD>\n\
									</TR>\n\
");

	fprintf(out, "\
									<TR>\n\
										<TD>\n\
											<B>Total Bytes</B>\n\
										</TD>\n\
										<TD ALIGN=\"RIGHT\">\n\
											%llu\n\
										</TD>\n\
										<TD ALIGN=\"RIGHT\">\n\
											%llu\n\
										</TD>\n\
									</TR>
", host->bytes_sent, host->bytes_received);

	fprintf(out, "\
									<TR>\n\
										<TD>\n\
											<B>Total Packets</B>\n\
										</TD>\n\
										<TD ALIGN=\"RIGHT\">\n\
											%llu\n\
										</TD>\n\
										<TD ALIGN=\"RIGHT\">\n\
											%llu\n\
										</TD>\n\
									</TR>\n\
", host->packets_sent, host->packets_received);

	fprintf(out, "\
									<TR>\n\
										<TD>\n\
											<B>Total Connect requests</B>\n\
										</TD>\n\
										<TD ALIGN=\"RIGHT\">\n\
											%llu\n\
										</TD>\n\
										<TD ALIGN=\"RIGHT\">\n\
											%llu\n\
										</TD>\n\
									</TR>\n\
", host->connections_sent, host->connections_received);

	fprintf(out, "\
									<TR>\n\
										<TD COLSPAN=\"3\">&nbsp;</TD>\n\
									</TR>\n\
									<TR>\n\
										<TD ALIGN=\"LEFT\"><B>First traffic recorded at:</B></TD>\n\
										<TD ALIGN=\"LEFT\" COLSPAN=\"2\">%s</TD>\n\
									</TR>\n\
", format_time(host->first_seen));
	fprintf(out, "\
									<TR>\n\
										<TD ALIGN=\"LEFT\"><B>Last traffic recorded at:</B></TD>\n\
										<TD ALIGN=\"LEFT\" COLSPAN=\"2\">%s</TD>\n\
									</TR>\n\
								</TABLE>\n\
", format_time(host->last_seen));

	fprintf(out, "\
							</TD>\n\
						</TR>\n\
					</TABLE>\n\
");
	fprintf(out, "<BR>\n");
}

static void print_peers(peer_t *peers, FILE *out)
{
	char			src_hostname[1024];
	char			dst_hostname[1024];
	
	fprintf(out, "\
								<TABLE WIDTH=\"100%%\" BORDER=0 CELLSPACING=0>\n\
									<TR BGCOLOR=\"#E0E0E0\">\n\
										<TD><B>Source host</B></TD>\n\
										<TD><B>Destination host</B></TD>\n\
										<TD ALIGN=\"RIGHT\"><B>Bytes sent</B></TD>\n\
										<TD ALIGN=\"RIGHT\"><B>Packets sent</B></TD>\n\
									</TR>\n\
");

	while(peers != NULL)
	{
		format_hostname(peers->src, src_hostname, sizeof(src_hostname));
		format_hostname(peers->dst, dst_hostname, sizeof(dst_hostname));
		
		fprintf(out, "\
									<TR>\n\
										<TD>%s</TD>\n\
										<TD>%s</TD>\n\
										<TD ALIGN=\"RIGHT\">%llu</TD>\n\
										<TD ALIGN=\"RIGHT\">%llu</TD>\n\
									</TR>\n\
", src_hostname, dst_hostname, peers->bytes_sent, peers->packets_sent);
		
		peers = peers->next;
	}

	fprintf(out, "\
								</TABLE>\n\
");
}

static void format_hostname(host_t *host, char *s, size_t s_len)
{
	static struct	in_addr i;

	i.s_addr = host->ip_addr;
	
	if (host->hostname != NULL)
		g_snprintf(s, s_len, "%s", host->hostname);
	else	
		g_snprintf(s, s_len, "Unknown (%s)", inet_ntoa(i));
}

static const char *format_time(time_t t)
{
	static char	ts[128];
	struct tm	*tm;
	
	tm = localtime(&t);
	
	strftime(ts, sizeof(ts), "%a, %d %b %Y %H:%M:%S %Z", tm);
	
	return(ts);
}
