// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "Kernel.h"

namespace Torch {

Kernel::Kernel(DataSet *data_)
{
  data = data_;
}

void Kernel::reset()
{
}


Kernel::~Kernel()
{
}

//=====

DotKernel::DotKernel(EuclideanDataSet *edata_) : Kernel(edata_)
{
  edata = edata_;
}

real DotKernel::eval(int i, int j)
{
  return(edata->dotProduct(i, j));
}

real DotKernel::realEval(int i, List *y)
{
  return(edata->realDotProduct(i, y));
}

DotKernel::~DotKernel()
{
}

//=====

PolynomialKernel::PolynomialKernel(EuclideanDataSet *edata_) : Kernel(edata_)
{
  edata = edata_;
  addIOption("degree", &d, 2, "degree", true);
  addROption("mul cst", &s, 1, "multiplicative constant", true);
  addROption("add cst", &r, 1, "additive constant", true);
}

real PolynomialKernel::eval(int i, int j)
{
  real z = s*edata->dotProduct(i, j) + r;

  // la fonction pow rame a donf
  real julie = z;
  for(int t = 1; t < d; t++)
    julie *= z;
    
  return(julie);
}

real PolynomialKernel::realEval(int i, List *y)
{
  real z = s*edata->realDotProduct(i, y) + r;

  // la fonction pow rame a donf
  real julie = z;
  for(int t = 1; t < d; t++)
    julie *= z;
    
  return(julie);
}

PolynomialKernel::~PolynomialKernel()
{
}

//=====

GaussianKernel::GaussianKernel(EuclideanDataSet *edata_) : Kernel(edata_)
{
  edata = edata_;
  precalc = NULL;
  precalc_real = NULL;
  addROption("gamma", &g, 1./(10.*10.), "gamma", true);
}

void GaussianKernel::reset()
{
  precalc      = (real *)xrealloc(precalc,      sizeof(real)*data->n_examples);
  precalc_real = (real *)xrealloc(precalc_real, sizeof(real)*data->n_real_examples);
  message("GaussianKernel: precalculating...");
  for(int i = 0; i < data->n_examples; i++)
    precalc[i] = edata->dotProduct(i, i);
  for(int i = 0; i < data->n_real_examples; i++)
    precalc_real[i] = edata->realRealDotProduct(i, i);
  message("...ok");
}

real GaussianKernel::eval(int i, int j)
{    
  return(exp(g*( 2.*edata->dotProduct(i,j)-precalc[i]-precalc[j] )));
}

real GaussianKernel::realEval(int i, List *y)
{
  return(exp(g*( 2.*edata->realDotProduct(i,y)-precalc_real[i]-edata->dotProduct(y,y) )));
}

GaussianKernel::~GaussianKernel()
{
  free(precalc);
  free(precalc_real);
}

//=====

SigmoidKernel::SigmoidKernel(EuclideanDataSet *edata_) : Kernel(edata_)
{
  edata = edata_;
  addROption("mul cst", &s, 1, "multiplicative constant", true);
  addROption("add cst", &r, 1, "additive constant", true);
}

real SigmoidKernel::eval(int i, int j)
{
  return(tanh(s*edata->dotProduct(i, j) + r));
}

real SigmoidKernel::realEval(int i, List *y)
{
  return(tanh(s*edata->realDotProduct(i, y) + r));
}

SigmoidKernel::~SigmoidKernel()
{
}

//=====

}

