/*
** iksemel (XML Parser for Jabber)
** Copyright (C) 2000-2001 Gurer Ozen <palpa@jabber.org>
**
** This code is free software; you can redistribute it and/or
** modify it under the terms of GNU Lesser General Public License.
**
** pool memory allocation functions
*/

#include "common.h"
#include "iksemel.h"

#include <stdlib.h>
#include <stdio.h>

typedef struct _page
{
  struct _page *next;
  guint16 size;
} page;

typedef struct _freeitem
{
  guint16 size;
  struct _freeitem *next;
} freeitem;
 
struct ikspool_s
{
  void *owner;
  page *firstpage;
  freeitem *firstfree;
  guint16 pagesize;
};

/* XXX: make this architecture-dependent. */

#define ALIGN_ON 8

/* XXX: there's gotta be a better way! */

#define ALIGNED(x) (((x)%ALIGN_ON)?((x)+(ALIGN_ON-((x)%ALIGN_ON))):(x))

/* we use these instead of sizeof() to make sure (?) that
 * all our pointers end up aligned on word boundaries or
 * what have you. */

#define POOL_OFFSET ALIGNED(sizeof(ikspool))
#define PAGE_OFFSET ALIGNED(sizeof(page))
#define FI_OFFSET ALIGNED(sizeof(freeitem))

/* NOTE: this has to be aligned, too. */

#define MIN_FREESIZE ALIGNED(8)

//#define MAX(x, y) (((x) > (y)) ? (x) : (y)) //already in glib

void insert_free(ikspool *p, freeitem *newfi);
int abut(freeitem *fi1, freeitem *fi2);
static void combine(freeitem *fi1, freeitem *fi2);
void free_extra(ikspool *p, freeitem *fi, guint16 size);
void unlink_free(ikspool *p, freeitem *fi, freeitem *prev);

/* Create a new ikspool. */

ikspool *iks_pool_new(guint16 pagesize)
{
  ikspool *p;
  page *pg;
  freeitem *fi;

  if (pagesize < MIN_FREESIZE)
    pagesize = MIN_FREESIZE;

  pagesize = ALIGNED(pagesize);

  p = malloc(POOL_OFFSET + PAGE_OFFSET + FI_OFFSET + pagesize);
  if (!p) return NULL;

  pg = (page *)(((void*)p) + POOL_OFFSET);
  pg->next = NULL;
  pg->size = pagesize;

  fi = (freeitem *) (((void*)pg) + PAGE_OFFSET);
  fi->next = NULL;
  fi->size = pagesize;
   
  p->pagesize = pagesize;

  p->firstpage = pg;
  p->firstfree = fi;
  p->owner = NULL;

  return p;
}

/* Get the owner of this ikspool. */

void *iks_pool_owner(ikspool *p)
{
  return p->owner;
}

/* Return the owner of this ikspool. */

void *iks_pool_set_owner(ikspool *p, void *owner)
{
  void * old = p->owner;
  p->owner = owner;
  return old;
}

/* Allocate some data. */

void *iks_pool_alloc(ikspool *p, guint16 size) 
{
  freeitem *fi, *prev, *found;
  page *newpg;

  /* Make sure size is at least MIN_FREESIZE */

  if (size < MIN_FREESIZE)
    size = MIN_FREESIZE;

  /* Align on ALIGN_ON blocks. */

  size = ALIGNED(size);
     
  found = prev = NULL;
   
  for (fi = p->firstfree; fi; fi = fi->next) 
    {
      if (fi->size >= size)	       /* first fit */
        {
          found = fi;
          break;
        }
	
      prev = fi;
    }
   
  /* If we get here, no free items fit. */

  if (!found) 
    {
      guint16 ps = MAX(p->pagesize, size);

      newpg = malloc(PAGE_OFFSET + FI_OFFSET + ps);
      if (!newpg) return NULL;

      newpg->size = ps;
      newpg->next = p->firstpage->next;

      p->firstpage->next = newpg;

      found = (freeitem *)(((void*)newpg) + PAGE_OFFSET);
      found->size = ps;
      found->next = NULL;
    }
   
  unlink_free(p, found, prev);
  free_extra(p, found, size);

  /* Mark this so if we get it again, we know it's ours. */
  /* XXX: figure out better mark than casting. */

  found->next = (freeitem *) p;

  return (((void*)found) + FI_OFFSET);
}

/* Utility to alloc an array. Note that it zeroes memory, like calloc(). */

void * iks_pool_calloc(ikspool *p, guint16 number, guint16 size)
{
  guint16 total = number * size; /* XXX: overflow */
  void * ptr = iks_pool_alloc(p, total);
  memset(ptr, 0, total);
  return ptr;
}

/* Free allocated data. */

void iks_pool_free(ikspool *p, void *data)
{
  freeitem *newfi = (freeitem *)(data - FI_OFFSET);
  freeitem *fi, *prev;

  /* Check that this is ours. */

  if (newfi->next != (freeitem *) p)
    return;

  /* Try to combine it with abutting higher freeitem. */

  prev = NULL;
   
  for (fi = p->firstfree; fi; fi = fi->next)
    {
      if (abut(newfi, fi))
        {
          unlink_free(p, fi, prev); 
          combine(newfi, fi);
          break;
        }
      prev = fi;
    }

  /* Try to combine it with abutting lower freeitem. */

  prev = NULL;
   
  for (fi = p->firstfree; fi; fi = fi->next)
    {
      if (abut(fi, newfi))
        {
          unlink_free(p, fi, prev);
          combine(fi, newfi);
          newfi = fi;
          break;
        }
      prev = fi;
    }

  /* Add it to the free list. */

  insert_free(p, newfi);
}

/* Re-size some memory. */

void *iks_pool_realloc(ikspool *p, void *data, guint16 newsize)
{
  freeitem *refi = NULL;
  freeitem *prev, *fi;
  void * newdata = NULL;

  /* Like realloc(), just do a malloc when data is NULL. */

  if (!data)
    return iks_pool_alloc(p, newsize);

  /* make sure it's big enough. */

  if (newsize < MIN_FREESIZE)
    newsize = MIN_FREESIZE;

  /* make sure it's aligned. */

  newsize = ALIGNED(newsize);

  /* our free item stuff should still be there. */

  refi = (freeitem *)(data - FI_OFFSET);

  /* Check that this is ours. */

  if (refi->next != (freeitem *) p)
    return NULL;

  /* If it's already big enough, just return. */

  if (refi->size >= newsize)
    return data;

  /* Try to find abutting free space, so no copying. */

  prev = NULL;
   
  for (fi = p->firstfree; fi; fi = fi->next)
    {
      if (abut(refi, fi))
        {
          if ((refi->size + fi->size + FI_OFFSET) >= newsize)
            {
              unlink_free(p, fi, prev); 
              combine(refi, fi);
              free_extra(p, refi, newsize);
              return data;
            }
          else
            break;
        }
    }

  /* otherwise, just make new data, copy to it. */

  newdata = iks_pool_alloc(p, newsize);
  memcpy(newdata, data, refi->size);
  iks_pool_free(p, data);
   
  return newdata;
}

/* Zark the entire pool. */

void iks_pool_delete(ikspool *p)
{
  page *pg, *tmp;

  /* The first page is in the same malloc'd space as the pool, so
     don't free it explicitly. */

  pg = p->firstpage->next;
   
  while (pg)
    {
      tmp = pg->next;
      free(pg);
      pg = tmp;
    }
   
  free(p);
}

char *iks_pool_strdup(ikspool *p, const char *str)
{
  guint16 n;
  char * newstr;

  if (!str)
    return NULL;

  n = strlen(str);

  newstr = iks_pool_calloc(p, n + 1, sizeof(char));
  
  if (!newstr)
    return NULL;

  strcpy(newstr, str);

  return newstr;
}

/* Utility to get some visualization of what's going on in the pool. */

void iks_pool_stats(ikspool *p)
{
  freeitem *fi;
  page *pg;

  fprintf(stderr, "FREE ITEMS:\n");
   
  for (fi = p->firstfree; fi; fi = fi->next)
    {
      fprintf(stderr, "-> %d bytes at %p\n", fi->size, (void*)fi);
    }
   
  fprintf(stderr, "PAGES:\n");
   
  for (pg = p->firstpage; pg; pg = pg->next)
    {
      fprintf(stderr, "-> at %p\n", (void*)pg);
    }
   
  return;
}

/* Actually, this makes it "best fit." */

void insert_free(ikspool *p, freeitem *newfi)
{
  freeitem *found, *fi, *prev;
   
  found = prev = NULL;
   
  for (fi = p->firstfree; fi; fi = fi->next)
    {
      if (fi->size > newfi->size)
        {
          found = fi;
          break;
        }
      prev = fi;
    }

  newfi->next = found;
	     
  if (prev)
    {
      prev->next = newfi;
    }
  else
    {
      p->firstfree = newfi;
    }
  return;
}

/* return true if fi1 and fi2 are "right next to each other". */

int abut(freeitem *fi1, freeitem *fi2)
{
  return (((void*)fi2) == (((void*)fi1) + FI_OFFSET + fi1->size));
}

/* Combine two abutting free items into one free item.
   assumes fi1 < fi2. */

static
void combine(freeitem *fi1, freeitem *fi2)
{
  fi1->size += FI_OFFSET + fi2->size;
  fi2->next = NULL;
  fi2->size = 0;
}

/* If there's more capacity in fi than needed, make the extra a new
   freeitem and add it to pool. Trim fi's size accordingly. */

void free_extra(ikspool *p, freeitem *fi, guint16 size)
{
  freeitem *newfi;

  if ((fi->size - size) > (FI_OFFSET + MIN_FREESIZE)) 
    {
      newfi = (freeitem*) (((void *)fi) + FI_OFFSET + size);
      newfi->size = (fi->size - size) - FI_OFFSET;
      fi->size = size;
      insert_free(p, newfi);
    }
}

/* Take an item out of the free linked list. */

void unlink_free(ikspool *p, freeitem *fi, freeitem *prev)
{
  if (prev)
    prev->next = fi->next;
  else 
    p->firstfree = fi->next;
}
