/*
 *  Copyright (C) 2000 by Marco G"otze.
 *
 *  This code is part of the ThoughtTracker source package, which is
 *  distributed under the terms of the GNU GPL2.
 */

#include <sys/stat.h>

#include <cerrno>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <fstream>
#include <iostream>

#include <gtk--/alignment.h>
#include <gtk--/frame.h>
#include <gtk--/label.h>
#include <gtk--/pixmap.h>
#include <gtk--/scrolledwindow.h>
#include <gtk--/style.h>
#include <gtk--/table.h>
#include <gtk--/toolbar.h>
#include <gtk--/tooltips.h>

#include "thoughttracker.h"
#include "app.h"
#include "database.h"
#include "entry.h"
#include "filedlg.h"
#include "menubar.h"
#include "querydlg.h"
#include "xpm.h"

#ifdef ENABLE_DEBUG
#undef DMSG 
#define DMSG cerr << "TTEntryWidget::" << __FUNCTION__ << "(): "
#endif  /* ENABLE_DEBUG */

using SigC::bind;
using SigC::slot;

using namespace Gtk;

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
TTEntryWidget::TTEntryWidget(TTApplication *application)
  : VBox(),
    app(application),
    fixed_font(0), normal_font(0),
    selected(-1), modified(false),
    pane_pos(-1), pane_max(-1)
{
  // configure tool bar
  {
    using namespace Toolbar_Helpers;

    // build it backwards
    Toolbar *tbar = manage(new Toolbar(GTK_ORIENTATION_HORIZONTAL,
      GTK_TOOLBAR_BOTH));
    tbar->set_button_relief(GTK_RELIEF_NONE);
    tbar->set_space_style(GTK_TOOLBAR_SPACE_LINE);
    pack_start(*tbar, false, false, false);

    tbar->tools().push_front(ButtonElem(_("Clear"),
      *manage(new Pixmap(icon_clear_xpm)),
      slot(this, &TTEntryWidget::btn_clear),
      _("clear the entire form")));

    tbar->tools().push_front(Space());
    Pixmap *icon = manage(new Pixmap(icon_unlink_xpm));
    icon->set_build_insensitive(true);
    tbar->tools().push_front(ButtonElem(_("Unlink"), *icon,
      slot(this, &TTEntryWidget::btn_unlink),
      _("delete the selected link")));
    w.b.unlink = (*tbar->tools().begin())->get_widget();

    tbar->tools().push_front(Space());
    icon = manage(new Pixmap(icon_del_xpm));
    icon->set_build_insensitive(true);
    tbar->tools().push_front(ButtonElem(_("Delete"), *icon,
      slot(this, &TTEntryWidget::btn_del),
      _("delete entry from database")));
    w.b.del = (*tbar->tools().begin())->get_widget();
    icon = manage(new Pixmap(icon_add_xpm));
    icon->set_build_insensitive(true);
    tbar->tools().push_front(ButtonElem(_("Add"), *icon,
      slot(this, &TTEntryWidget::btn_add),
      _("add entry to database")));
    w.b.add = (*tbar->tools().begin())->get_widget();
    icon = manage(new Pixmap(icon_update_xpm));
    icon->set_build_insensitive(true);
    tbar->tools().push_front(ButtonElem(_("Update"), *icon,
      slot(this, &TTEntryWidget::btn_update),
      _("update the existing entry")));
    w.b.update = (*tbar->tools().begin())->get_widget();

    tbar->tools().push_front(Space());
    tbar->tools().push_front(ButtonElem(_("Search..."),
      *(manage(new Pixmap(icon_search_xpm))),
      slot(this, &TTEntryWidget::btn_search),
      _("search for entries")));
    icon = manage(new Pixmap(icon_follow_xpm));
    icon->set_build_insensitive(true);
    tbar->tools().push_front(ButtonElem(_("Follow"), *icon,
      slot(this, &TTEntryWidget::btn_follow),
      _("follow selected link")));
    w.b.follow = (*tbar->tools().begin())->get_widget();
  }

  // configure frame around actual widget area
  Frame *frame = manage(new Frame(_("Entry")));
  frame->set_shadow_type(GTK_SHADOW_ETCHED_IN);
  frame->set_border_width(5);
  pack_start(*frame, true, true, false);

  // configure pane for input fields and link list
  w.pane = manage(new VPaned);
  w.pane->set_border_width(5);
  w.pane->set_gutter_size(12);
  frame->add(*w.pane);

  // configure table for input field layout (pane part 1)
  Table *table = manage(new Table(2, 4));
  table->set_row_spacing(0, 5);
  w.pane->pack1(*table, true, false);

  // configure summary text field plus label
  Alignment *align = manage(new Alignment(1.0, 0.0, 0.0, 0.0));
  table->attach(*align, 0, 1, 0, 1, GTK_FILL, 0, 3, 0);
  Label *label = manage(new Label(_("Summary:")));
  align->add(*label);
  w.summary = manage(new Entry);
  w.summary->changed.connect(slot(this, &TTEntryWidget::handle_modification));
  table->attach(*w.summary, 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

  // configure details text field (in scrolled window) plus label and fixed
  // width switch
  align = manage(new Alignment(1.0, 0.0, 0.0, 0.0));
  table->attach(*align, 0, 1, 1, 2, GTK_FILL, GTK_EXPAND | GTK_FILL, 3, 0);
  label = manage(new Label(_("Details:")));
  align->add(*label);
  ScrolledWindow *swin = manage(new ScrolledWindow);
  swin->set_policy(GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  table->attach(*swin, 1, 2, 1, 4, GTK_EXPAND | GTK_FILL,
    GTK_EXPAND | GTK_FILL, 0, 0);
  w.details = manage(new Text);
  w.details->set_editable(true);
  w.details->set_word_wrap(true);
  w.details->changed.connect(slot(this, &TTEntryWidget::handle_modification));
  swin->add(*w.details);

  align = manage(new Alignment(0.0, 1.0, 0.0, 0.0));
  table->attach(*align, 0, 1, 2, 3, GTK_FILL, 0, 3, 0);
  w.hub = manage(new CheckButton(_("hub"), 0, 0.5));
  manage(new Tooltips)->set_tip(*w.hub, _("designate this entry as a hub"));
  w.hub->toggled.connect(slot(this, &TTEntryWidget::handle_toggle_hub));
  align->add(*w.hub);

  align = manage(new Alignment(0.0, 1.0, 0.0, 0.0));
  table->attach(*align, 0, 1, 3, 4, GTK_FILL, 0, 3, 0);
  w.fixed = manage(new CheckButton(_("fixed"), 0, 0.5));
  manage(new Tooltips)->set_tip(*w.fixed,
    _("use fixed-width font for details"));
  w.fixed->toggled.connect(bind(slot(this,
    &TTEntryWidget::handle_toggle_fixed), true));
  align->add(*w.fixed);

  // configure search results list plus label (pane part 2)
  Box *box = manage(new VBox);
  w.pane->pack2(*box, true, false);
  align = manage(new Alignment(0.0, 1.0, 0.0, 0.0));
  box->pack_start(*align, false, false, false);
  label = manage(new Label(_("Links:")));
  align->add(*label);
  swin = manage(new ScrolledWindow);
  swin->set_policy(GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  box->pack_start(*swin, true, true, false);
  w.list = manage(new TTLinkList);
  w.list->select_row.connect(slot(this, &TTEntryWidget::handle_select_link));
  w.list->unselect_row.connect(slot(this,
    &TTEntryWidget::handle_unselect_link));
  w.list->key_press_event.connect(slot(this, &TTEntryWidget::handle_ll_return));
  swin->add(*w.list);

  // connect menu items whose functionality is implemented herein
  app->w.mbar->w.im->activate.connect(slot(this,
    &TTEntryWidget::import_details));
  app->w.mbar->w.ex->activate.connect(slot(this,
    &TTEntryWidget::export_details));
  app->w.mbar->w.revert->activate.connect(slot(this,
    &TTEntryWidget::revert));
  w.summary->focus_in_event.connect(bind(slot(app->w.mbar,
    &TTMenuBar::set_cb_widget), (Editable*) w.summary));
  w.summary->focus_out_event.connect(bind(slot(app->w.mbar,
    &TTMenuBar::set_cb_widget), (Editable*) 0));
  w.details->focus_in_event.connect(bind(slot(app->w.mbar,
    &TTMenuBar::set_cb_widget), (Editable*) w.details));
  w.details->focus_out_event.connect(bind(slot(app->w.mbar,
    &TTMenuBar::set_cb_widget), (Editable*) 0));

  // show everything but us in our entirety :-)
  show_all();
  hide();
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
void
TTEntryWidget::update_buttons()
{
  // update tool bar buttons' states
  if (app->get_active() < 0) {  // new entry being edited
    w.b.update->set_sensitive(false);
    w.b.add->set_sensitive(modified);
    w.b.del->set_sensitive(false);
    w.b.follow->set_sensitive(false);
    w.b.unlink->set_sensitive(false);
  } else {  // existing entry being edited
    w.b.update->set_sensitive(modified);
    w.b.add->set_sensitive(modified);
    w.b.del->set_sensitive(true);
    w.b.follow->set_sensitive(selected >= 0);
    w.b.unlink->set_sensitive(selected >= 0);
  }
}

//.............................................................................
void
TTEntryWidget::clear()
{
  w.summary->set_text("");
  w.details->show();
  w.details->freeze();
  w.details->set_point(0);
  w.details->forward_delete(w.details->get_length());
  w.details->thaw();
  w.hub->set_active(false);
  w.fixed->set_active(app->opts.fixeddefault);
  w.list->clear();
  modified = false;
  selected = -1;
  app->set_active(-1);
}

//.............................................................................
void
TTEntryWidget::btn_follow()
{
  if (selected < 0) return;
  dbid_t id = selected;  // changes when save_record() reloads the record
  if (!save_record(false)) return;
  load_record(id);
  prepare_for_display();
}

//.............................................................................
void
TTEntryWidget::btn_search()
{
  if (!save_record(false)) return;
  app->prev_active = -1;  // we explicitly reset prev_active on this occasion
  app->run_search();
  app->sbar_msg(_("Specify search criteria..."));
}

//.............................................................................
void
TTEntryWidget::btn_add()
{
  save_record(true);
  update_buttons();
}

//.............................................................................
void
TTEntryWidget::btn_update()
{
  save_record(false, false);
  update_buttons();
}

//.............................................................................
void
TTEntryWidget::btn_del()
{
  dbid_t active = app->get_active();
  if (active < 0) return;

  // ask for confirmation
  if (query_popup(app, _("Delete entry..."),
    _("Are you sure you want to delete this entry?"),
    TTQD_BTN_YES | TTQD_BTN_NO, TTQD_BTN_NO) != TTQD_BTN_YES)
  {
    app->sbar_msg(_("The entry has NOT been deleted."));
    return;
  }

  // delete the record
  list<dbid_t> l;  // get linked entries to update search list later on
  app->db.links(active, l);
  if (app->db.del(active) == DB_SUCCESS) {
    clear();  // also sets app's "active" variable
    w.summary->grab_focus();
    update_buttons();
    // update search result list for entries possibly linked to the deleted one
    for (list<dbid_t>::iterator i = l.begin(); i != l.end(); i++)
      app->w.search->update_list_for(*i);
    app->w.search->update_list_for(active);  // delete from search result list
    app->w.mbar->update_bookmark(active);
    app->sbar_msg(_("Entry deleted."));
  } else
    internal_error_popup(app, TTQD_ERROR_DB_WRITE);
}

//.............................................................................
void
TTEntryWidget::btn_unlink()
{
  dbid_t active = app->get_active();
  if (active < 0 || selected < 0) return;

  // delete from both the list and the DB
  if (app->db.unlink(active, selected) == DB_SUCCESS) {
    app->w.search->update_list_for(active);
    app->w.search->update_list_for(selected);
    w.list->delete_link(selected);  // changes value of selected
    update_buttons();
    app->sbar_msg(_("Link deleted."));
  } else
    internal_error_popup(app, TTQD_ERROR_DB_WRITE);
}

//.............................................................................
void
TTEntryWidget::btn_clear()
{
  if (!save_record(false)) return;
  clear();
  update_buttons();
  w.summary->grab_focus();
  app->sbar_msg(_("Enter data for a new entry..."));
}

//.............................................................................
void
TTEntryWidget::import_details()
{
  TTFileDialog dlg(app, _("Import..."));
  string filename = dlg.run();
  if (!filename.length()) return;
  ifstream im(filename.c_str(), ios::nocreate | ios::skipws);
  if (im && im.good()) {  // open succeeded
    w.details->freeze();
    if (w.details->has_selection()) w.details->delete_selection();
    while (!im.eof()) {
      // read an entire line
      char *buf;
      im.gets(&buf);
      string line = buf;
      delete[] buf;
      line += '\n';
      gtk_text_insert(GTK_TEXT(w.details->gtkobj()), 0, 0, 0, line.c_str(), -1);
    }
    w.details->thaw();
    handle_modification();
    app->sbar_msg(_("The selected file's contents have been inserted."));
  } else
    error_popup(app, _("Failed to open the specified file!"));
}

//.............................................................................
void
TTEntryWidget::export_details()
{
  TTFileDialog dlg(app, _("Export..."));
  string filename = dlg.run();
  if (!filename.length()) return;
  struct stat buf;
  if (!stat(filename.c_str(), &buf) && query_popup(app,
    _("File exists..."),
    string(_("Are you sure you want to overwrite the existing file, ")) +
    filename + '?', TTQD_BTN_YES | TTQD_BTN_NO, TTQD_BTN_NO) != TTQD_BTN_YES)
  {
    return;
  }
  ofstream ex(filename.c_str(), ios::out | ios::trunc);
  if (ex && ex.good()) {  // open succeeded
    gint a = 0, b = -1;
    if (w.details->has_selection()) {
      a = w.details->get_selection_start_pos();
      b = w.details->get_selection_end_pos();
      if (a > b) {  // swap if start > end
        gint c = a;
        a = b;
        b = c;
      }
    }
    ex << w.details->get_chars(a, b);
    if (ex.good())
      app->sbar_msg(_("The details field has been written to the file."));
    else {
      error_popup(app, string(_("An error occurred while writing to ") +
        filename + '.'));
    }
  } else
    error_popup(app, _("Failed to open the specified file!"));
}

//.............................................................................
void
TTEntryWidget::revert()
{
  if (app->get_active() < 0 || !modified) return;
  TTQDButtons btn = query_popup(app, _("Revert?"),
    string(_("Are you sure you want to revert and lose all changes made "
    "to the current entry?")),  TTQD_BTN_YES | TTQD_BTN_NO |  TTQD_BTN_CANCEL,
    TTQD_BTN_NO);
  if (btn != TTQD_BTN_YES) return;

  load_record(app->get_active());
}

//.............................................................................
bool
TTEntryWidget::save_record(bool as_new, bool ask)
{
  string summary = w.summary->get_text();
  string text = w.details->get_chars(0, -1);

  // save necessary?
  if (!modified)  // no save necessary
    // be silent when just updating, because callers usually won't check on
    // their part whether an update is necessary and just call this function
    // before e.g. quitting
    return true;

  // ask before an update (i.e., either the user explicitly pressed the update
  // button, or the program called this function to cause an implicit update
  // before e.g. switching to the search widget or quitting)
  if (!as_new) {
    if (app->get_active() < 0) {  // new entry, modified, updating
      if (ask) {
        if (query_popup(app, _("New Entry..."),
          _("Are you sure you want to discard the entered data?"),
          TTQD_BTN_YES | TTQD_BTN_NO, TTQD_BTN_NO) == TTQD_BTN_YES)
        {
          modified = false;
          update_buttons();
          return true;
        } else
          return false;
      } else {  // UPDATE NEW entry silently?!
        internal_error_popup(app, TTQD_ERROR_IMPOSSIBLE_SITUATION);
        return false;
      }
    } else if (ask) {  // existing entry, modified, updating; user to be asked
      gint res = query_popup(app, _("Update..."),
        _("Do you want to update this entry?"),
        TTQD_BTN_YES | TTQD_BTN_NO | TTQD_BTN_CANCEL, TTQD_BTN_YES);
      switch (res) {
        case TTQD_BTN_NO:
          load_record(app->get_active());  // restore record from DB
          return true;
        case TTQD_BTN_CANCEL:
          return false;
      }
    }
  }

  // sanity checks
  const char *s, *t;
  for (s = t = summary.c_str(); *s; s++)  // check if summary is non-blank
    if (*s != ' ' && *s != '\t' && *s != '\n') {
      t = s;  /* clip leading white space too */
      break;
    }
  if (!*s) {  // summary field is blank
    error_popup(app, _("The summary field is required to be non-blank.  "
      "Since this isn't the case, nothing has been written to the database."));
    return false;
  }

  // ask before saving a modified, existing entry as a new record
  if (app->get_active() >= 0 && as_new && ask && query_popup(app,
    _("Add entry..."), _("You have been editing an existing entry.\n"
    "Are you sure you want to add a new entry of these contents "
    "(instead of just updating the existing entry)?"),
    TTQD_BTN_YES | TTQD_BTN_CANCEL, TTQD_BTN_CANCEL) != TTQD_BTN_YES)
  {
     app->sbar_msg(_("The entry has NOT been added."));
     return false;
  }

  // if autolink is set, if adding a new entry and an existing entry was
  // previously shown, ask whether the two shall be linked
  dbid_t link_to = -1;
  dbf_t prev_flags;
  string prev_summary, prev_details;
  if (as_new) app->set_active(-1);  // because of prev_active
  if (app->opts.autolink && as_new && app->prev_active >= 0) {
    // first check that the record still exists, else reset app->prev_active
    // once we're at it
    if (!app->db.exists(app->prev_active))
      app->prev_active = -1;
    else {
      // now ask
      if (app->db.read(app->prev_active, prev_flags, prev_summary,
        prev_details) == DB_FAILURE)
      {
        internal_error_popup(app, TTQD_ERROR_DB_READ);
        return false;
      }
      TTQDButtons btn = query_popup(app, _("Create link?"),
        string(_("Shall the new entry and the previously shown one (\"")) +
        prev_summary + _("\") be linked?"),  TTQD_BTN_YES | TTQD_BTN_NO |
        TTQD_BTN_CANCEL, TTQD_BTN_YES);
      if (btn == TTQD_BTN_CANCEL)
        return false;
      else if (btn == TTQD_BTN_YES)
        link_to = app->prev_active;
    }
  }

  dbf_t flags = 0;
  if (w.hub->get_active())   flags |= DB_FLAG_HUB;
  if (w.fixed->get_active()) flags |= DB_FLAG_FIXED;
  int id = app->db.store(as_new ? -1 : app->get_active(), flags, t, text);
  if (id != DB_FAILURE) {
    app->set_active(id);
    // when updating, make sure a modified summary shows up correctly in the
    // search widget's result list; same goes for bookmarks menu
    if (!as_new) {
      app->w.search->update_list_for(id);
      app->w.mbar->update_bookmark(id);
    }
    app->sbar_msg(as_new ? _("The entry has been saved.") :
      _("The entry has been updated."));
  } else {
    internal_error_popup(app, TTQD_ERROR_DB_WRITE);
    return false;
  }

  // if autolink is set and user confirmed earlier, link entries
  if (link_to >= 0) {
    if (app->db.link(app->get_active(), link_to) == DB_SUCCESS)
      w.list->append_link(link_to, prev_summary, prev_flags & DB_FLAG_HUB, 1);
    else
      internal_error_popup(app, TTQD_ERROR_DB_WRITE);
  }

  modified = false;
  update_buttons();
  return true;
}

//.............................................................................
void
TTEntryWidget::load_record(dbid_t id)
{
  if (id < 0) return;

  if (!w.list->is_realized()) w.list->realize();  // prerequiste for its use

  // load data
  clear();
  dbf_t flags;
  string summary, text;
  if (app->db.read(id, flags, summary, text) == DB_FAILURE) {
    internal_error_popup(app, TTQD_ERROR_DB_READ);
    return;
  }
  w.hub->set_active(flags & DB_FLAG_HUB);
  w.fixed->set_active(flags & DB_FLAG_FIXED);
  w.summary->set_text(summary);
  w.summary->set_position(0);
  w.details->freeze();
  gtk_text_insert(GTK_TEXT(w.details->gtkobj()), 0, 0, 0, text.c_str(), -1);
  w.details->set_point(0);
  w.details->thaw();
  modified = 0;
  update_buttons();  // do it now, too, in case we return prematurely

  // load link list
  list<dbid_t> l;
  if (app->db.links(id, l) == DB_SUCCESS) {
    w.list->freeze();
    for (list<dbid_t>::iterator i = l.begin(); i != l.end(); i++) {
      dbf_t flags;
      if (app->db.read(*i, flags, summary, text) == DB_FAILURE) {
        internal_error_popup(app, TTQD_ERROR_DB_READ);
        return;
      }
      int links = app->db.linked(*i);
      w.list->append_link(*i, summary, flags & DB_FLAG_HUB,
        flags & DB_FLAG_HUB ? true : (links > 0 ? links-1 : false));
    }
    if (l.size()) w.list->gtkobj()->focus_row = 0;  // evil kludge, I know
    w.list->thaw();
  } else {  // db.links() failed
    internal_error_popup(app, TTQD_ERROR_DB_READ);
    return;
  }

  // everything went all right, so we may update the active ID
  app->set_active(id);
  update_buttons();

  app->sbar_msg(_("Showing an existing entry.  Feel free to edit it."));
}

//.............................................................................
void
TTEntryWidget::set_fixed_font(const string &name) {
  if (!normal_font) {  // remember normal font on first call
    normal_font = w.details->get_style()->get_font();
    fixed_font = normal_font;  // in case load() fails below
  }
  fixed_font.load(name);
  handle_toggle_fixed(false);  // will take care of updating the details widget
}

//:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
void
TTEntryWidget::handle_select_link(gint row, gint col, GdkEvent *event)
{
  selected = w.list->id_of_row(row);
#ifdef ENABLE_DEBUG
  DMSG << "selected id = " << selected << endl;
#endif
  update_buttons();
  if (event && event->type == GDK_2BUTTON_PRESS) btn_follow();
}

//.............................................................................
void
TTEntryWidget::handle_unselect_link(gint row, gint col, GdkEvent *event)
{
  selected = -1;
#ifdef ENABLE_DEBUG
  DMSG << "selection cleared" << endl;
#endif
  update_buttons();
}

//.............................................................................
void
TTEntryWidget::handle_modification()
{
  if (!modified) {  // we don't need a complete update_buttons() here
    if (app->get_active() >= 0) w.b.update->set_sensitive(true);
    w.b.add->set_sensitive(true);
    app->w.mbar->w.revert->set_sensitive(true);
    modified = true;
  }
}

//.............................................................................
void
TTEntryWidget::handle_toggle_fixed(bool set_modified)
{
  Style *style = w.details->get_style()->copy();
  style->set_font(w.fixed->get_active() ? fixed_font : normal_font);
  w.details->set_style(*style);
  if (set_modified && app->get_active() >= 0) handle_modification();
}

//.............................................................................
void
TTEntryWidget::handle_toggle_hub()
{
  if (app->get_active() >= 0) handle_modification();
}

