/*

    File: partxbox.c

    Copyright (C) 2005 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License along
    with this program; if not, write the Free Software Foundation, Inc., 51
    Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <ctype.h>      /* tolower */
#include "types.h"
#include "common.h"
#include "testdisk.h"
#include "fnctdsk.h"
//#include "analyse.h"
#include "lang.h"
#include "intrf.h"
#include "chgtype.h"
#include "partxbox.h"
#include "savehdr.h"
#include "fatx.h"

static int check_part_xbox(t_param_disk *disk_car, const int debug,t_partition *partition,const int saveheader);
static int get_geometry_from_xboxmbr(const unsigned char *buffer, const int debug, t_CHS *geometry);
static t_list_part *read_part_xbox(t_param_disk *disk_car, const int debug, const int saveheader);
static int write_part_xbox(t_param_disk *disk_car, const t_list_part *list_part, const int ro , const int debug, const int align);
static t_list_part *init_part_order_xbox(const t_param_disk *disk_car, t_list_part *list_part);
static t_list_part *add_partition_xbox(t_param_disk *disk_car,t_list_part *list_part, const int debug, char **current_cmd);
static void set_next_status_xbox(const t_param_disk *disk_car, t_partition *partition);
static int test_structure_xbox(t_list_part *list_part);
static int set_part_type_xbox(t_partition *partition, unsigned int part_type_xbox);
static int is_part_known_xbox(const t_partition *partition);
static void init_structure_xbox(const t_param_disk *disk_car,t_list_part *list_part, const int debug);
static const unsigned char *get_partition_name_xbox(const t_partition *partition);
static const unsigned char *get_partition_name_xbox_aux(const unsigned int part_type_xbox);
static unsigned int get_part_type_xbox(const t_partition *partition);

static const struct systypes xbox_sys_types[] = {
  { PXBOX_UNK,  "Unknown"		},
  { PXBOX_FATX, "FATX"			},
  { PXBOX_UNK,		NULL }
};

t_arch_fnct arch_xbox=
{
  .part_name="XBox",
  .part_name_option="partition_xbox",
  .msg_part_type="                P=Primary  D=Deleted",
  .read_part=read_part_xbox,
  .write_part=write_part_xbox,
  .init_part_order=init_part_order_xbox,
  .get_geometry_from_mbr=get_geometry_from_xboxmbr,
  .check_part=check_part_xbox,
  .write_MBR_code=NULL,
  .add_partition=add_partition_xbox,
  .set_prev_status=set_next_status_xbox,
  .set_next_status=set_next_status_xbox,
  .test_structure=test_structure_xbox,
  .set_part_type=set_part_type_xbox,
  .is_part_known=is_part_known_xbox,
  .init_structure=init_structure_xbox,
  .erase_list_part=NULL,
  .get_partition_name=get_partition_name_xbox,
  .get_part_type=get_part_type_xbox
};

static unsigned int get_part_type_xbox(const t_partition *partition)
{
  return partition->part_type_xbox;
}

int get_geometry_from_xboxmbr(const unsigned char *buffer, const int debug, t_CHS *geometry)
{
  if(debug>1)
  {
    ecrit_rapport("get_geometry_from_xboxmbr\n");
  }
  geometry->cylinder=0;
  geometry->head=0;
  geometry->sector=1;
  if(geometry->sector>0)
  {
    ecrit_rapport("Geometry from XBOX MBR: head=%u sector=%u\n",geometry->head+1,geometry->sector);
  }
  return 0;
}

t_list_part *read_part_xbox(t_param_disk *disk_car, const int debug, const int saveheader)
{
  unsigned char buffer[0x800];
  t_list_part *new_list_part=NULL;
  aff_buffer(BUFFER_RESET,"Q");
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, 0)!=0)
    return new_list_part;
  {
    struct xbox_partition *xboxlabel=(struct xbox_partition*)&buffer;
    if (memcmp(xboxlabel->magic,"BRFR",4))
    {
      aff_buffer(BUFFER_ADD,"\nBad XBOX partition, invalid signature\n");
      return NULL;
    }
  }
  return new_list_part;
}

static int write_part_xbox(t_param_disk *disk_car, const t_list_part *list_part, const int ro, const int debug, const int align)
{ /* TODO: Implement it */
  if(ro==0)
  {
    not_implemented("write_part_xbox");
  }
  return 0;
}

static t_list_part *init_part_order_xbox(const t_param_disk *disk_car, t_list_part *list_part)
{
  return list_part;
}

static t_list_part *add_partition_xbox(t_param_disk *disk_car,t_list_part *list_part, const int debug, char **current_cmd)
{
  t_CHS start,end;
  t_partition *new_partition=partition_new();
  int position=0;
  start.cylinder=0;
  start.head=0;
  start.sector=1;
  end.cylinder=disk_car->CHS.cylinder;
  end.head=disk_car->CHS.head;
  end.sector=disk_car->CHS.sector;
  {
    int done = FALSE;
    while (done==FALSE) {
      int command;
      static struct MenuItem menuGeometry[]=
      {
	{ 'c', "Cylinders", 	"Change starting cylinder" },
	{ 'C', "Cylinders", 	"Change ending cylinder" },
	{ 'T' ,"Type",		"Change partition type"},
	{ 'd', "Done", "" },
	{ 0, NULL, NULL }
      };
      aff_copy(stdscr);
      wmove(stdscr,4,0);
      wdoprintf(stdscr,"%s",disk_car->description(disk_car));
      new_partition->part_offset=CHS2offset(disk_car,&start);
      new_partition->part_size=CHS2offset(disk_car,&end) - new_partition->part_offset + disk_car->sector_size;
      wmove(stdscr,10, 0);
      wclrtoeol(stdscr);
      aff_part(stdscr,AFF_PART_SHORT,disk_car,new_partition);
      wmove(stdscr,COMMAND_LINE_Y, COMMAND_LINE_X);
      wclrtoeol(stdscr);
      wrefresh(stdscr);
      command=wmenuSimple(stdscr,menuGeometry, position);
      switch (command) {
	case 'c':
	  start.cylinder=ask_number(start.cylinder,0,disk_car->CHS.cylinder,"Enter the starting cylinder ");
	  position=1;
	  break;
	case 'C':
	  end.cylinder=ask_number(end.cylinder,start.cylinder,disk_car->CHS.cylinder,"Enter the ending cylinder ");
	  position=2;
	  break;
	case 'T':
	case 't':
	  change_part_type(disk_car,new_partition,current_cmd);
	  position=3;
	  break;
	case key_ESC:
	case 'd':
	case 'D':
	case 'q':
	case 'Q':
	  done = TRUE;
	  break;
      }
    }
  }
  if((CHS2offset(disk_car,&end)>new_partition->part_offset)&& new_partition->part_type_xbox>0)
  {
    t_list_part *new_list_part=insert_new_partition(list_part, new_partition);
    new_partition->status=STATUS_PRIM;
    if(test_structure_xbox(list_part)!=0)
    {
      new_partition->status=STATUS_DELETED;
    }
    return new_list_part;
  }
  free(new_partition);
  return list_part;
}

static void set_next_status_xbox(const t_param_disk *disk_car, t_partition *partition)
{
  if(partition->status==STATUS_DELETED)
    partition->status=STATUS_PRIM;
  else
    partition->status=STATUS_DELETED;
}

static int test_structure_xbox(t_list_part *list_part)
{ /* Return 1 if bad*/
  t_list_part *new_list_part=NULL;
  t_list_part *element;
  t_list_part *new_element;
  int res=0;
  /* Sort list_part in new_list_part */
  for(element=list_part;element!=NULL;element=element->next)
  {
    if(element->part->status!=STATUS_DELETED)
      new_list_part=insert_new_partition(new_list_part,element->part);
  }
  /* Test overlapping */
  for(element=new_list_part;element!=NULL;element=element->next)
  {
    if(((element->prev!=NULL) && (element->part->part_offset<=element->prev->part->part_offset+element->prev->part->part_size-1)) ||
	((element->next!=NULL) && (element->part->part_offset+element->part->part_size-1>=element->next->part->part_offset)))
    {
      res=1;
    }
  }
  for(element=new_list_part;element!=NULL;element=new_element)
  {
    new_element=element->next;
    free(element);
  }
  return res;
}

static int set_part_type_xbox(t_partition *partition, unsigned int part_type_xbox)
{
  if(part_type_xbox>0 && part_type_xbox <= 255)
  {
    partition->part_type_xbox=part_type_xbox;
    return 0;
  }
  return 1;
}

static int is_part_known_xbox(const t_partition *partition)
{
  return (partition->part_type_xbox!=PXBOX_UNK);
}

static void init_structure_xbox(const t_param_disk *disk_car,t_list_part *list_part, const int debug)
{
  t_list_part *element;
  t_list_part *new_list_part=NULL;
  /* Create new list */
  for(element=list_part;element!=NULL;element=element->next)
    element->to_be_removed=0;
  for(element=list_part;element!=NULL;element=element->next)
  {
    int to_be_removed=0;
    t_list_part *element2;
    for(element2=element->next;element2!=NULL;element2=element2->next)
      if(element->part->part_offset+element->part->part_size-1 >= element2->part->part_offset)
      {
	to_be_removed=1;
	element2->to_be_removed=1;
      }
    if(to_be_removed)
      element->to_be_removed=1;
    if(element->to_be_removed==0)
      new_list_part=insert_new_partition_aux(new_list_part,element_new(element->part),1);
  }
#ifdef DEBUG
  check_list_part(new_list_part);
#endif
    for(element=new_list_part;element!=NULL;element=element->next)
      element->part->status=STATUS_PRIM;
  if(disk_car->arch->test_structure(new_list_part))
  {
    for(element=new_list_part;element!=NULL;element=element->next)
      element->part->status=STATUS_DELETED;
  }
  {
    /* free */
    t_list_part *new_element;
    for(element=new_list_part;element!=NULL;element=new_element)
    {
      new_element=element->next;
      free(element);
    }
  }
#ifdef DEBUG
  check_list_part(list_part);
#endif
}

static int check_part_xbox(t_param_disk *disk_car,const int debug,t_partition *partition, const int saveheader)
{
  int ret=0;
  switch(partition->part_type_xbox)
  {
    case PXBOX_FATX:
      ret=check_FATX(disk_car,partition,debug);
      break;
    default:
      if(debug>0)
      {
	ecrit_rapport("check_part_xbox %u type %02X: no test\n",partition->order,partition->part_type_xbox);
      }
      break;
  }
  if(ret!=0)
  {
    ecrit_rapport("check_part_xbox failed for partition type %02X\n", partition->part_type_xbox);
    aff_part_buffer(AFF_PART_ORDER,disk_car,partition);
    if(saveheader>0)
    {
      save_header(disk_car,partition,debug);
    }
  }
  return ret;
}

static const unsigned char *get_partition_name_xbox_aux(const unsigned int part_type_xbox)
{
  int i;
  for (i=0; xbox_sys_types[i].name!=NULL; i++)
    if (xbox_sys_types[i].part_type == part_type_xbox)
      return xbox_sys_types[i].name;
  return NULL;
}

static const unsigned char *get_partition_name_xbox(const t_partition *partition)
{
  return get_partition_name_xbox_aux(partition->part_type_xbox);
}
