/*

    File: testdisk.c

    Copyright (C) 1998-2005 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#include <stdarg.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>	/* geteuid */
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <ctype.h>      /* toupper, tolower */
#ifdef HAVE_LOCALE_H
#include <locale.h>	/* setlocale */
#endif
#ifdef HAVE_SIGNAL
#include <signal.h>
#endif
#include <errno.h>
#include "types.h"
#include "common.h"
#include "testdisk.h"
#include "lang.h"
#include "intrf.h"
#include "intrface.h"
#include "godmode.h"
#include "fnctdsk.h"
extern const t_arch_fnct arch_i386;
extern const t_arch_fnct arch_sun;

static FILE* init_log(const char*filename,int argc, char**argv);

static FILE *f_rapport=NULL;
static int f_status=0;

void ecrit_rapport_string(const char *string,const int max_length)
{
  int i;
  for(i=0;(string[i]!='\0')&&(i<max_length);i++)
    ecrit_rapport("%c",string[i]);
}

int ecrit_rapport(const char *_format, ...)
{
  int res=0;
  if(f_rapport!=NULL)
  {
    va_list ap;
    fflush(stderr);
    va_start(ap,_format);
    res=vfprintf(f_rapport,_format,ap);
    va_end(ap);
    if(res<0)
    {
      f_status=1;
    }
    if(fflush(f_rapport))
    {
      f_status=1;
    }
  }
  return res;
}

static FILE* init_log(const char*filename,int argc, char**argv)
{
  FILE*f_file=fopen(filename,"a");
  if(f_file==NULL)
  {
    printf("\nWarning: Can not create testdisk.log: %s\n",strerror(errno));
    printf("Press the enter key to continue.\n");
    getchar();
  }
  else
  {
	int i;
	time_t my_time;
	my_time=time(NULL);
	fprintf(f_file,"\n\n%s",ctime(&my_time));
	fprintf(f_file,"TestDisk command line :");
	for(i=1;i<argc;i++)
	  fprintf(f_file," %s", argv[i]);
	fprintf(f_file,"\n");
	fflush(f_file);
#ifdef HAVE_DUP2
	dup2(fileno(f_file),2);
#endif
  }
  return f_file;
}

#ifdef HAVE_SIGACTION
void sighup_hdlr(int shup)
{
  ecrit_rapport("SIGHUP detected! TestDisk has been killed.\n");
  if(f_rapport!=NULL)
  {
    fclose(f_rapport);
  }
  exit(1);
}
#endif

void aff_copy(WINDOW *window)
{
  wclear(window);
  keypad(window, TRUE); /* Need it to get arrow key */
  wmove(window,0,0);
  wdoprintf(window, "TestDisk %s, Data Recovery Utility, %s",VERSION,TESTDISKDATE);
  wmove(window,1,0);
  wdoprintf(window,"Christophe GRENIER <grenier@cgsecurity.org>");
  wmove(window,2,0);
  wdoprintf(window,"http://www.cgsecurity.org");
}

int main( int argc, char **argv )
{
  int i;
  int help=0, create_log=0, debug=0, dump_ind=0;
  int do_list=0;
  int test_recovery=0;
  int write_used=0;
  int saveheader=0;
  int create_backup=0;
  t_list_disk *list_disk=NULL;
  t_list_disk *element_disk;
  const char *cmd_device=NULL;
#ifdef SOLARIS
  const t_arch_fnct *arch=&arch_sun;
#else
  const t_arch_fnct *arch=&arch_i386;
#endif
#ifdef HAVE_SIGACTION
  struct sigaction action, old_action;
#endif
#ifdef TESTING
  srand(1);
#endif
#ifdef HAVE_SIGACTION
  /* set up the signal handler for SIGHUP */
  action.sa_handler  = sighup_hdlr;
  action.sa_flags = 0;
  if(sigaction(SIGHUP, &action, &old_action)==-1)
  {
    printf("Error on SIGACTION call\n");
    return -1;
  }
#endif
#if defined(__CYGWIN__) || defined(__MINGW32__)
  /* Average Windows user doesn't know how to specify a parameter */
  if(argc==1)
  {
    create_log=1;
    debug=1;
  }
#endif
  for(i=1;i<argc;i++)
  {
    if((strcmp(argv[i],"/test_recovery")==0) ||(strcmp(argv[i],"-test_recovery")==0))
    {
      test_recovery=1;
      do_list=1;
      create_log=1;
    }
    else if((strcmp(argv[i],"/dump")==0) || (strcmp(argv[i],"-dump")==0))
      dump_ind=1;
    else if((strcmp(argv[i],"/log")==0) ||(strcmp(argv[i],"-log")==0))
    {
      create_log=1;
    }
    else if((strcmp(argv[i],"/debug")==0) || (strcmp(argv[i],"-debug")==0))
    {
      debug++;
      create_log=1;
    }
    else if((strcmp(argv[i],"/list")==0) || (strcmp(argv[i],"-list")==0))
      do_list=1;
    else if((strcmp(argv[i],"/saveheader")==0) || (strcmp(argv[i],"-saveheader")==0))
      saveheader=1;
    else if((strcmp(argv[i],"/backup")==0) || (strcmp(argv[i],"-backup")==0))
      create_backup=1;
    else if((strcmp(argv[i],"/help")==0) || (strcmp(argv[i],"-help")==0) || (strcmp(argv[i],"--help")==0) ||
      (strcmp(argv[i],"/h")==0) || (strcmp(argv[i],"-h")==0))
      help=1;
      /*
    else if(strcmp(argv[i],"/cmd")==0)
    {
      if(i+2>=argc)
	help=1
      else
      {
	cmd_device=argv[i+1];
	 TODO: handle
	  analyse
	  geometry,C,H,S
	  write
	  backup
	  restore
	  search
	  rebuildbs
	  rebuildbs,write
	  cleanfat
	  searchsb
      }
    }
       */
    else
    {
      list_disk=insert_new_disk(list_disk,file_test_availability(argv[i],debug,arch));
      if(list_disk==NULL)
      {
	help=1;
      }
    }
  }
  if(cmd_device!=NULL && list_disk!=NULL)
    help=1;
  printf("TestDisk %s, Data Recovery Utility, %s\nChristophe GRENIER <grenier@cgsecurity.org>\nhttp://www.cgsecurity.org\n",VERSION,TESTDISKDATE);
  if(help!=0)
  {
    printf("\nUsage: testdisk [/log] [/debug] [/dump] [/saveheader] [file or device]"\
		"\n       testdisk /list [/log] [/backup] [file or device]" \
	    "\n" \
	    "\n/log          : create a testdisk.log file" \
	    "\n/debug        : add debug information" \
	    "\n/dump         : dump raw sectors" \
	    "\n/list         : display current partitions" \
	    "\n/saveheader   : save unknown filesystem header" \
	    "\n/backup       : backup partition structure" \
	    "\n" \
	    "\nTestDisk checks and recovers lost partitions" \
	    "\nIt works with :" \
	    "\n- FAT12, FAT16 <32M, FAT16 >32M, FAT32" \
	    "\n- NTFS" \
	    "\n- LINUX ext2fs/ext3fs, LINUX swap" \
	    "\n- BFS (BeOS)" \
	    "\n- UFS (BSD)" \
	    "\n- ReiserFS (1 & 2)" \
	    "\n- JFS" \
	    "\n" \
	    "\nIf you have problems with TestDisk or bug reports, please contacte me.\n");
    return 0;
  }
  if(create_log==1 && f_rapport==NULL)
  {
    /*    const char *ext2fs_version=NULL; */
    f_rapport=init_log("testdisk.log",argc,argv);
    ecrit_rapport("TestDisk %s, Data Recovery Utility, %s\nChristophe GRENIER <grenier@cgsecurity.org>\nhttp://www.cgsecurity.org\n",VERSION,TESTDISKDATE);
#ifdef DJGPP
    ecrit_rapport("Dos version");
#elif defined(BSD)
    ecrit_rapport("BSD version");
#elif defined(LINUX)
    ecrit_rapport("Linux version");
#elif defined(__CYGWIN__) || defined(__MINGW32__)
    ecrit_rapport("Windows version");
#else
    ecrit_rapport("Undefined OS");
#endif
#ifdef COMPILE_BY
#ifdef COMPILE_HOST
#ifdef COMPILE_TIME
    ecrit_rapport(" (%s@%s, %s)",COMPILE_BY,COMPILE_HOST,COMPILE_TIME);
#endif
#endif
#endif
    ecrit_rapport("\n");
    /*    ext2fs_get_library_version(ext2fs_version,NULL); */
    /*    ecrit_rapport("%s\n\n",ext2fs_version); */
#ifdef DEBUG
    ecrit_rapport("Key down:      0x%x\n", KEY_DOWN);
    ecrit_rapport("Key up:        0x%x\n", KEY_UP);
    ecrit_rapport("Key left:      0x%x\n", KEY_LEFT);
    ecrit_rapport("Key right:     0x%x\n", KEY_RIGHT);
    ecrit_rapport("Key page down: 0x%x\n", KEY_NPAGE);
    ecrit_rapport("Key page up:   0x%x\n", KEY_PPAGE);
    ecrit_rapport("Key enter:     0x%x\n", KEY_ENTER);
#endif
  }
printf("Please wait...\n");
  {
    const char *locale;
    locale = setlocale (LC_ALL, "");
    if (locale==NULL) {
      locale = setlocale (LC_ALL, NULL);
      ecrit_rapport("Failed to set locale, using default '%s'.\n", locale);
    } else {
      ecrit_rapport("Using locale '%s'.\n", locale);
    }
  }
  aff_buffer(BUFFER_RESET,"Q");
  list_disk=hd_parse(list_disk,debug,arch);
  if(list_disk==NULL)
  {
    printf("No harddisk found\n");
#if defined(__CYGWIN__) || defined(__MINGW32__)
    printf("You need to be administrator to use TestDisk\n");
    printf("Under Win9x, use the DOS version instead\n");
    ecrit_rapport("You need to be administrator to use TestDisk\n");
#else
#ifndef DJGPP
#ifdef HAVE_GETEUID
    if(geteuid()!=0)
    {
      printf("You need to be root to use TestDisk\n");
      ecrit_rapport("You need to be root to use TestDisk\n");
    }
#endif
#endif
#endif
  }
#ifdef DJGPP
  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
  {
    printf("%s\n",element_disk->disk->description(element_disk->disk));
  }
#endif
  hd_parse_bis(list_disk,0,arch);
  /* save disk parameters to rapport */
  ecrit_rapport("Hard disk list\n");
  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
  {
    printf("%s, sector size=%u\n",element_disk->disk->description(element_disk->disk),element_disk->disk->sector_size);
    ecrit_rapport("%s, sector size=%u\n",element_disk->disk->description(element_disk->disk),element_disk->disk->sector_size);
  }
  printf("\n");
  ecrit_rapport("\n");
  if(do_list!=0)
  {
    for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
    {
      interface_list(element_disk->disk,debug,test_recovery,saveheader,create_backup);
      if(test_recovery!=0)
      {
	t_list_part *list_part=search_part(element_disk->disk,0,debug,0,0,0);
	if(debug!=0)
	{
	  t_list_part *element;
	  ecrit_rapport("search_part() results\n");
	  for(element=list_part;element!=NULL;element=element->next)
	    aff_part_rapport(element_disk->disk,element->part);
	}
	delete_list_part(list_part);
      }
    }
  }
  else
  {
    do_curses_testdisk(debug,dump_ind,list_disk,saveheader,&arch);
  }
  for(element_disk=list_disk;element_disk!=NULL;)
  {
    t_list_disk *element_disk_next=element_disk->next;
    write_used|=element_disk->disk->write_used;
    if(element_disk->disk->clean!=NULL)
      element_disk->disk->clean(element_disk->disk);
    FREE(element_disk->disk);
    FREE(element_disk);
    element_disk=element_disk_next;
  }
  if(f_rapport!=NULL)
  {
    ecrit_rapport("TestDisk exited normally.\n");
    if(fclose(f_rapport))
    {
      f_status=1;
    }
  }
  if(f_status!=0)
  {
    printf("TestDisk: Log file corrupted!\n");
  }
  else
  {
    printf("TestDisk exited normally.\n");
  }
  if(write_used!=0)
  {
    printf("You have to reboot for the change to take effect.\n");
  }
  return 0;
}
