/*
    File: hfsp.c, TestDisk

    Copyright (C) 2005 Christophe GRENIER <grenier@cgsecurity.org>
    Original header comes from libhfs - library for reading and writing
    Macintosh HFS volumes
    Copyright (C) 2000 Klaus Halfmann <klaus.halfmann@feri.de>
    Original work by 1996-1998 Robert Leslie <rob@mars.org>
    other work 2000 from Brad Boyer (flar@pants.nu)

    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */   
#ifndef _HFSP_H
#define _HFSP_H
#define HFSP_BLOCKSZ            512	/* A sector for Apple is always 512 bytes */
#define HFSP_BLOCKSZ_BITS       9	/* 1<<9 == 512  */
#define	HFSP_VOLHEAD_SIG	0x482B	/* 'H+'	*/

// Minimum Key size for all btrees
#define HFSP_CAT_KEY_MIN_LEN	6

// Maximum Key size for all btrees
#define HFSP_CAT_KEY_MAX_LEN	516

/* HFS+ includes POSIX permissions , although marked as reserved they will be
 * used as such. Is ignored by MacOS 8-9 but probably not by MacOS X.
 */
typedef struct {
        uint32_t         owner;
        uint32_t         group;
        uint32_t         mode;
        uint32_t         dev;
} hfsp_perm;

/* A single contiguous area (fragment) of a file */
typedef struct {
        uint32_t         start_block;
        uint32_t         block_count;
} hfsp_extent;

/* A file may contain up to 8 normale extents, all other
   are found in some extra extent area */
typedef hfsp_extent hfsp_extent_rec[8];

/* Information for a "Fork" in a file
 * Forks are the "usual" DATA and RSRC forks or special files
 * (e.g. the Volume Bitmap)
 */
typedef struct {
        uint64_t		total_size;  // logical size
        uint32_t		clump_size;  // number of bytes to preallocate
        uint32_t		total_blocks;
        hfsp_extent_rec extents;     // initial (8) extents
} hfsp_fork_raw;

/* HFS+ Volume Header
 * Always found at block 2 of the disk, a copy is stored
 * at the second to last block of the disk.
 */
typedef struct hfsp_vh {
        uint16_t         signature;   // must be HFSPLUS_VOLHEAD_SIG 'H+'
        uint16_t         version;     // currently 4, ignored 
        uint32_t         attributes;  // See bit constants below
        uint32_t         last_mount_vers;
                // Use a registered creator code here (See libhfsp.h)
		// Mac OS uses '8.10' well
        uint32_t         reserved;
 
        uint32_t         create_date; // local time !
        uint32_t         modify_date; // GMT (?)
        uint32_t         backup_date; // GMT (?)
        uint32_t         checked_date; // GMT (?) fsck ?
 
        uint32_t         file_count;
         // not including special files but including DATA and RSRC forks
        uint32_t         folder_count; // excluding the root folder
 
        uint32_t         blocksize;
         // must be multiple of HFSPLUS_SECTOR_SIZE,
         // should be a multiple of 4k for harddisk
        uint32_t         total_blocks;
        uint32_t         free_blocks;
         // The total number of unused allocation blocks on the disk.
 
        uint32_t         next_alloc;
         // hint where to search for next allocation blocks
        uint32_t         rsrc_clump_sz;
         // default clump size for rsrc forks
        uint32_t         data_clump_sz;
         // default clump size for data forks
        uint32_t	       next_cnid;
         // next unused catalog id
        uint32_t         write_count;
         // increment on every mount (and write ?)
        uint64_t        encodings_bmp;
                // for every encoding used on the disk a bit is set
                // ignored but eventually must be cared for
        char          finder_info[32];                                      
	hfsp_fork_raw   alloc_file;
         // stores bitmap of use/free blocks
        hfsp_fork_raw   ext_file;
         // stores oferflow extents
        hfsp_fork_raw   cat_file;
	 // This contains the root directory
        hfsp_fork_raw   attr_file;
        hfsp_fork_raw   start_file;
         // a special startup file may be described here (used by ?)
} hfsp_vh;

/* HFS+ volume attributes */
/* 0-6 reserved, may be used in memory only */
#define HFSPLUS_VOL_RESERVED1 0x000000FF
#define HFSPLUS_VOL_HARDLOCK  0x00000080 // Used in Memory by finder only
#define HFSPLUS_VOL_UNMNT     0x00000100
        // clear this bit when mounting, set as last step of unmounting
        // This is checked by (slower) ROM code
#define HFSPLUS_VOL_SPARE_BLK 0x00000200
#define HFSPLUS_VOL_NOCACHE   0x00000400
        // in case of RAM or ROM disk (try a HFS+ Ramdisk :)
#define HFSPLUS_VOL_INCNSTNT  0x00000800
        // Reverse meaning as of HFSPLUS_VOL_UNMNT
        // This is checked by (faster) Mac OS code
/* 12-14 reserved */
#define HFSPLUS_VOL_RESERVED2 0x00007000
#define HFSPLUS_VOL_SOFTLOCK  0x00008000
#define HFSPLUS_VOL_RESERVED3 0xFFFF0000

/* HFS+ Btree node descriptor */
typedef struct {
	uint32_t	    next;   /* pointer to next node of this kind, or 0 */
			    /* Header Node points to first MAP node */
	uint32_t	    prev;   /* pointer to previous node of this kind, or 0 */
	uint8_t	    kind;   /* see below */
	uint8_t	    height; /* root node starts with 0 */
	uint16_t	    num_rec;	/* number of records in this node */
	uint16_t	    reserved;	/* fill up to 4 byte alignment */
} btree_node_desc;

/* HFS+ Btree Node types */
#define HFSP_NODE_NDX	0x00
#define HFSP_NODE_HEAD	0x01
#define HFSP_NODE_MAP	0x02
#define HFSP_NODE_LEAF	0xFF

#define HFSP_CATALOG_MIN_NODE_SIZE  0x1000
#define HFSP_ATTRMIN_DOE_SIZE	    0x1000

/* The record offsets are found at the end of the fork
 * containing the Btree */

typedef uint16_t	btree_record_offset;

typedef struct {
        uint16_t         depth;
	    // equal to height of btree_node_desc
        uint32_t         root;
	    // root node of the hierarchy 
        uint32_t         leaf_count;
	    // number of leaf Records (not nodes)
        uint32_t         leaf_head;
	    // first leaf node
        uint32_t         leaf_tail;
	    // last leaf node
        uint16_t         node_size;
	    // node size of _all_ nodes in this fork
        uint16_t         max_key_len;
	    // maximum (or fixed) length of keys in this btree
        uint32_t         node_count;
	    // count of all (free and used) nodes in tree
        uint32_t         free_nodes;
        uint16_t         reserved1;
        uint32_t         clump_size;
         // ignored my MacOS used by ?
        uint8_t	       btree_type;
         // always 0 for HFS+
        uint8_t	       reserved2;
        uint32_t         attributes;
	 // see below 
        uint32_t         reserved3[16];
} btree_head;

/* BTree attributes */
#define HFSPLUS_BAD_CLOSE            0x01
  // Btree was not properly closed and should be checked
  // not used for HFS+ but reserved
#define HFSPLUS_TREE_BIGKEYS         0x02
  // always set for HFS+
#define HFSPLUS_TREE_VAR_NDXKEY_SIZE 0x04
  // use variable length index nodes, always set for catalog btree,
  // always cleared for extents btree.
#define HFSPLUS_TREE_RESERVED	     0xFFFFFFF8
  // Reserved bits in Attributes
 
#define HFSPLUS_TREE_UNUSED          0xFFFFFFF8

/* Some special File ID numbers */
#define HFSP_POR_CNID             1  /* Parent Of the Root */
#define HFSP_ROOT_CNID            2  /* ROOT directory */
#define HFSP_EXT_CNID             3  /* EXTents B-tree */
#define HFSP_CAT_CNID             4  /* CATalog B-tree */
#define HFSP_BAD_CNID             5  /* BAD blocks file */
#define HFSP_ALLOC_CNID           6  /* ALLOCation file */
#define HFSP_START_CNID           7  /* STARTup file */
#define HFSP_ATTR_CNID            8  /* ATTRibutes file  */
#define HFSP_EXCH_CNID           15  /* ExchangeFiles temp id */
#define HFSP_MIN_CNID		 15  /* Minimum expected value */

/* Unicode String */
typedef struct {
    uint16_t		strlen;
    uint16_t		name[255];	// unicode characters
} hfsp_unistr255;

/* HFS+ key "superclass" for follwoing keys ...*/
typedef struct {
    uint16_t		key_length;	/* excluding length */
    char		data[0];
} hfsp_key;

/* HFS+ catalog entry key */
typedef struct {
    uint16_t		key_length;	/* excluding length */
    uint32_t		parent_cnid;
    hfsp_unistr255	name;
} hfsp_cat_key;

/* HFS+ extends entry key */
typedef struct {
    uint16_t		key_length;	/* excluding length */
    uint8_t		fork_type;	/* Seee below */
    uint8_t		filler;
    uint32_t		file_id;
    uint32_t		start_block;	
} hfsp_extent_key;

#define HFSP_EXTENT_DATA    0x00
#define HFSP_EXTENT_RSRC    0xFF

/* The key is followed by a record, an index or some other data */

/* The types of these records are defined as follows */

#define HFSP_FOLDER         0x0001  // entry fo a Folder
#define HFSP_FILE           0x0002  // entry for a File
#define HFSP_FOLDER_THREAD  0x0003  
    // Like '.' in unix, identifies the folder by its id, only
#define HFSP_FILE_THREAD    0x0004  

#define HFSP_THREAD_OFFSET  0x0002  
    // add to create a thread type from a simple type

int check_HFSP(t_param_disk *disk_car,t_partition *partition,const int debug);
int test_HFSP(t_param_disk *disk_car, const struct hfsp_vh *vh,t_partition *partition,const int debug, const int dump_ind);
int recover_HFSP(t_param_disk *disk_car, const struct hfsp_vh *vh,t_partition *partition,const int debug, const int dump_ind, const int backup);

#endif
